/* NCAA Pool Administrator's program
 *
 * This is the portion that is used only by the pool administrator,
 * to configure the pool, enter game results, etc.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#define _adminp_c_
#include "ncaa.h"

extern ResultLine results;
extern int nResults;
extern short tieBreaker;
extern TeamName teamnames[kNumTeams];
extern int seeds[kNumTeams];

int doPause;

/* This creates a lock file in the document directory.  Each time the
 * main program is run, it checks for this lock file;  if it exists
 * players will not be allowed to make their picks.  Use this when
 * the tournament starts, so that players can't make picks after the
 * tournament has already started.
 */
static void TogglePickPermission(int forceLock)
{
	int doLock;

	if (FileExists(kLockFile)) {
		if (!forceLock) {
			printf("Allow players to pick? [yn]");
			if (InputChar() == 'Y') {
				doPause = 1;
				if (remove(kLockFile) < 0) {
					fprintf(stderr, "Could not remove the lock file!\n");
					perror(kLockFile);
				} else {
					printf("Done (lock removed).\n");
				}
			}
		}
	} else {
		if (forceLock)
			doLock = 1;
		else {
			printf("Prevent additional players from picking? [yn]");
			doLock = (InputChar() == 'Y');
		}
		if (doLock) {
			doPause = 1;

			/* LockFile really wasn't intended for this type of
			 * lock, but what the hell.
			 */
			if (LockFile(kLockFile) < 0) {
				fprintf(stderr, "Could not create the lock file!\n");
				perror(kLockFile);
			} else {
				if (!forceLock)
					printf("Done (lock installed).\n");
			}
		}
	}
}									   /* TogglePickPermission */



/* This is the prompt you get for each game you're supposed to enter
 * the winner of.  It just prints the teams and asks you which team
 * won, and returns the team number of the winner.
 */
static int Result(int t1, int t2)
{
	int tmp;

	/* Make it so we list the team with the higher seed (lower number)
	 * first, then the underdog.
	 */
	if (seeds[t1] > seeds[t2]) {
		tmp = t2;
		t2 = t1;
		t1 = tmp;
	}

	printf("\nA. %s (%d)  vs.   B. %s (%d)\n",
		teamnames[t1],
		seeds[t1],
		teamnames[t2],
		seeds[t2]
	);
	while (1) {
		printf("   Who won? [A or B; Q to quit] ->");
		switch (InputChar()) {
			case '1':
			case 'A':
				return t1;
			case '2':
			case 'B':
				return t2;
			case 'Q':
				printf("   Are you sure you want to quit the program? [yn]");
				if (InputChar() == 'Y')
					exit(0);
		}
	}
}									   /* Result */



/* We want to record the results of the games, so we can see the standings,
 * etc.  Naturally we don't want to re-enter the results of previous
 * rounds, so we read in the results we have, then use the previous
 * round to see what the teams are for this round.
 */
static void EnterResults(void)
{
	FILE *fp;
	int i, len, scA, scB;
	char *cp;
	char line[80];
	
	doPause = 1;
	ReadTournamentFile();
	ReadResultsFile();	/* Read what results we have already. */
	len = (int) nResults;

	switch (len) {
		case 0:
			printf("First round, 32 results to enter.\n");
			for (i = 0; i < 32; i++)
				results[i] = (char) (Result(2 * i, 2 * i + 1));
			/* Keep track of how many results we've entered in len.
			 * We also fall-through below, in case you got lazy
			 * and missed a round and now have to enter in more
			 * than one round at a time.
			 */
			len = 32;
			/* fall through */

		case 32:
			printf("\nDo you know the winners of all 16 games of round 2? [yn]");
			if (InputChar() != 'Y')
				break;
			for (i = 0; i < 16; i++)
				results[32 + i] = (char) (Result(results[2 * i], results[2 * i + 1]));
			len = 48;
			/* fall through */

		case 48:
			printf("\nDo you know the winners of all 8 games of round 3 (Sweet 16)? [yn]");
			if (InputChar() != 'Y')
				break;
			for (i = 0; i < 8; i++)
				results[48 + i] = (char) (Result(results[32 + 2 * i], results[32 + 2 * i + 1]));
			len = 56;
			/* fall through */

		case 56:
			printf("\nDo you know the winners of all 4 games of round 4? [yn]");
			if (InputChar() != 'Y')
				break;
			for (i = 0; i < 4; i++)
				results[56 + i] = (char) (Result(results[48 + 2 * i], results[48 + 2 * i + 1]));
			len = 60;
			/* fall through */

		case 60:
			printf("\nDo you know the winners of both games of round 5 (Final 4)? [yn]");
			if (InputChar() != 'Y')
				break;
			for (i = 0; i < 2; i++)
				results[60 + i] = (char) (Result(results[56 + 2 * i], results[56 + 2 * i + 1]));
			len = 62;
			/* fall through */

		case 62:
			printf("\nDo you know the score of the championship game? [yn]");
			if (InputChar() != 'Y')
				break;
			/* We just prompt for the score the game here, instead of
			 * asking for the winner.  We do this because we use the
			 * total score of the game as a tiebreaker, and just asking
			 * for the score, we can find out the tiebreaker and who
			 * won in one shot.
			 */
			printf("%s: ", teamnames[(int) results[60]]);
			/* If we got an EOF or blank line, forget about it and go home. */
			if ((FGets(line, sizeof(line), stdin) == NULL) || (line[0] == 0))
				goto abt;
			scA = atoi(line);
			printf("%s: ", teamnames[(int) results[61]]);
			if ((FGets(line, sizeof(line), stdin) == NULL) || (line[0] == 0)) {
		abt:	printf("Nevermind.\n");
				break;
			}
			scB = atoi(line);
			if (scA > scB)
				results[62] = results[60];
			else
				results[62] = results[61];
			tieBreaker = (short) (scA + scB);
			len = 63;
			break;

		case 63:
			printf(
"You have already entered the results for the entire tournament.\nIf you think you have errors, delete the file \"%s\" and start over.\n",
				   kResultsFile);
			return;

		default:
			fprintf(stderr,
			"Results file in wrong format! (line length: %d)\n", len);
			return;
	}

	/* Write the results file now.
	 *
	 * The results string is really meant to be an array of 6-bit
	 * unsigned integers, with possible values 0-63.  To make the
	 * results file readable, we treat the array elements as a
	 * string of printable characters.  We add the value of the
	 * ASCII space, since characters with value less than that are
	 * non-printing control characters.
	 */
	for (cp = results, i = 0; i < len; cp++, ++i)
		*cp += ' ' + 1;
	*cp = '\0';		/* Add a NUL terminator in position 64. */

	fp = fopen(kResultsFile, "w");
	if (fp == NULL) {
		fprintf(stderr, "Could not write out the results of the games!\n");
		perror(kResultsFile);
		return;
	}
	fprintf(fp, "%s\n", results);
	if (tieBreaker > 0)
		fprintf(fp, "%d\n", (int) tieBreaker);
	fclose(fp);
	PrivateFile(kResultsFile);
	
	/* Since you started entering in games, we'll add the lock file
	 * for you in case you forgot.
	 */
	TogglePickPermission(1);
	doPause = 0;
}									   /* EnterResults */




/* Given two teams, randomly pick one of them.  It'd be silly to just
 * flip a coin and pick, since the random player would pick a 16 seed
 * 50% of the time and the 1 seed 50% in a game between a 1 and 16
 * seeds.  So we try to pick the tougher team most of the time.
 */
static int RandPick(int t1, int t2)
{
	int roll, hs, ls;
	int higher, lower, pick;

	/* The higher seeded team is really the one with the smaller seed #. */
	if (seeds[t1] > seeds[t2]) { higher = t2;  lower = t1; }
	else { higher = t2;  lower = t1; }

	hs = seeds[higher];
	ls = seeds[lower];

	roll = rand() % (hs + ls);
	pick = (roll < hs) ? lower : higher;
	
	/* So between a game with a 2 seed and a 11 seed, we would pick the
	 * 11 seed only 2 of 13 times.
	 */
	return (pick);
}	/* RandPick */



/* I wrote this as a testing feature to see how the pool could handle
 * a lot of players, but I'm leaving it in because it's kind of interesting
 * to see how the computer fares without any knowledge of the teams
 * except their seeds.
 */
static void AddRandomPlayers(void)
{
	int i, team1, team2, pick, j, howMany;
	Picks randPicks;
	char rpname[40];
	char junk[80], rpNickname[6];
	
	printf("\nHow many random players to add? ");
	FGets(junk, sizeof(junk), stdin);
	if ((junk[0] == '\0') || ((howMany = atoi(junk)) < 1))
		return;

	for (j=0; j<howMany; j++) {
	
		/* Create a 'name' for the random player. */
		for (i=0; i<5; i++)
			rpNickname[i] = (char) ((rand() % 26) + 'A');
		rpNickname[i] = '\0';
		strcpy(rpname, "Random Guy ");
		strcat(rpname, rpNickname);

		memset(&randPicks, 0, sizeof(randPicks));
		randPicks.tieBreaker = (short) (121 + (rand() % 80));
	
		/* Round 1. */
		for (i = 0; i < 32; i++) {
			team1 = 2 * i;
			team2 = team1 + 1;
			pick = RandPick(team1, team2);
			randPicks.firstrd[i] = (char) pick;
		}
	
		/* Round 2. */
		for (i = 0; i < 16; i++) {
			team1 = (int) randPicks.firstrd[2 * i];
			team2 = (int) randPicks.firstrd[2 * i + 1];
			pick = RandPick(team1, team2);
			randPicks.secondrd[i] = (char) pick;
		}
	
		/* Round 3. */
		for (i = 0; i < 8; i++) {
			team1 = (int) randPicks.secondrd[2 * i];
			team2 = (int) randPicks.secondrd[2 * i + 1];
			pick = RandPick(team1, team2);
			randPicks.thirdrd[i] = (char) pick;
		}
	
		/* Round 4. */
		for (i = 0; i < 4; i++) {
			team1 = (int) randPicks.thirdrd[2 * i];
			team2 = (int) randPicks.thirdrd[2 * i + 1];
			pick = RandPick(team1, team2);
			randPicks.fourthrd[i] = (char) pick;
		}
	
		/* Round 5. */
		for (i = 0; i < 2; i++) {
			team1 = (int) randPicks.fourthrd[2 * i];
			team2 = (int) randPicks.fourthrd[2 * i + 1];
			pick = RandPick(team1, team2);
			randPicks.fifthrd[i] = (char) pick;
		}
	
		/* Round 6. */
		team1 = (int) randPicks.fifthrd[0];
		team2 = (int) randPicks.fifthrd[1];
		pick = RandPick(team1, team2);
		randPicks.sixthrd = (char) pick;

		/* Write this guy to the player file. */
		AddToPlayerFile(rpname, &randPicks);
	}
	
}	/* AddRandomPlayers */



/* The administrator needs to set the default options, so when the
 * players want to check the standings, the program uses the same
 * options specified.  That way, if you've set up the program in
 * such a way where the user cannot run from the command line,
 * the program uses the specified options that govern your pool.  If
 * the user can run from the command line, they can change the options,
 * to see what would happen when using a different set of options than
 * those you agreed upon.
 */
static void SetDefaultOptions(void)
{
	FILE *fp;
	int gMethod, sMethod, canRePick;

	fp = fopen(kOptionsFile, "w");
	if (fp == NULL) {
		fprintf(stderr, "Could not create the options file!\n");
		doPause = 1;
		perror(kOptionsFile);
	} else {
		ClearScreen();
		printf(
"Grading options:\n\n"
"  <0>  Don't assign points, just go by winning percentage only\n"
"  <1>  Give 1,3,5,7,9,11 pts for a correct pick in the respective round\n"
		);

  		printf(
"  <2>  Give 1 pt for each correctly predicted favorite, 2 pts for dogs\n"
"  <3>  Give 1,2,4,8,16,32 pts for a correct pick in the respective round\n\n"
		);

		do {
			printf("Choice: ");
			gMethod = InputChar() - '0';
		} while ((gMethod < 0) || (gMethod > 3));
		
		if (gMethod == 0) {
			/* It doesn't make sense to ask the next question if you
			 * don't care about the points system.
			 */
			sMethod = 2;
		} else {
			printf(
	"\nSorting options:\n\n"
	"  <1>  Rank players by points first, win%% second\n"
	"  <2>  Rank players by win%% first, points second\n\n"
			);

			do {
				printf("Choice: ");
				sMethod = InputChar() - '0';
			} while ((sMethod < 1) || (sMethod > 2));
		}
		
		printf(
"\nAllow players to re-do their picks?  You may not want to allow this\n"
"because it would allow people to change other player's picks.  So do\n"
"you trust your players?\n\n"
"Choice: [yn] "
		);
		canRePick = InputChar() == 'Y';

		fprintf(fp, "%d\n%d\n%d\n", gMethod, sMethod, canRePick);
		fclose(fp);
		PrivateFile(kOptionsFile);
	}
}	/* SetDefaultOptions */



/* Gets rid of the old tournament files generated by the program. */
static void RestartTourn(void)
{
	printf(
	"\nAre you sure you want to erase the pool data to start a new\n"
	"tournament?  This erases the players.txt, results.txt, and lock.txt\n"
	"files.  It leaves the tourney.txt file, but the players and game\n"
	"results will be deleted.\n\nChoice: [yn] "
	);
	if (InputChar() == 'Y') {
		(void) remove(kPlayerFile);
		(void) remove(kResultsFile);
		(void) remove(kLockFile);
		printf("\nYou're ready to play the pool over now.\n");
	} else {
		printf("\nNevermind.\n");
	}
	doPause = 1;
}	/* RestartTourn */




void main(int argc, char **argv)
{
	int done = 0;
	int choice;

	InitRandom();   /* Important for LockFile(). */

	/* Go to our document directory.  Then we can use just filenames
	 * to refer to the various files we need to use.
	 */
	SetDataFileDirectory();
	
	/* Let the other users look at the tournament file, but don't
	 * let them change it, because that would mess up the tournament
	 * big time.
	 */
	Chmod(kTournFile, 0644);

	/* Put up the pool admin menu.  Nothing fancy, but it's functional. */
	do {
		doPause = 0;
		ClearScreen();
		printf("\
NCAA %s.\n\
Pool Administration Menu:\n\n\
  <1>  Enter RESULTS to games played\n\
  <2>  LOCK/UNLOCK pool, so players may not/may pick\n", kVersion);
		printf("\
  <3>  Set grading, sorting, and other OPTIONS\n\
  <4>  Add RANDOM players' picks\n\
  <5>  RESTART tournament\n\
  <Q>  QUIT\n\n");
		
		printf("Choice: ");
		choice = InputChar();
		
		switch(choice) {
			case '1': EnterResults(); break;
			case '2': TogglePickPermission(0); break;
			case '3': SetDefaultOptions(); break;
			case '4': AddRandomPlayers(); break;
			case '5': RestartTourn(); break;
			case 'Q': done = 1; doPause = 0; break;
		}
		if (doPause) {
			HitAnyKey();
			ClearScreen();
		}
	} while (!done);
	exit(0);
}									   /* main */
