/* NCAA Pool main program */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#define _ncaa_c_
#include "ncaa.h"

/*
 * Scoring Method 1:
 * 
 * Modify these if you wish to change the scoring system.  For example, the
 * factory settings specify that a player gets 7 (rMult[3]) points for
 * advancing to the final 4 (round 5) from the elite 8 (round 4) if a player
 * predicted it correctly.  See also the Grade1() function below.
 */
int rMult[6] =
{
	1, 3, 5, 7, 9, 11
};

/*
 * Scoring Method 3:
 * 
 * Same as above, but using a doubling function instead
 */
int r2Mult[6] =
{
	1, 2, 4, 8, 16, 32
};

/*
 * Scoring Method 2:
 * 
 * If you use method 2, you get 1 point for every correctly predicted favorite,
 * and 2 pts for an underdog (upset specials :-)  See the Grade2() function
 * below for details.
 */
#define PtsPerPickedFave 1
#define PtsPerPickedUDog 2

PlayerName curPlayerName = "";		   /* The name of the player running the
									    * program, or the name the player
									    * wishes to supply for one of the
									    * other commands. */
int canRePick = 1;					   /* Boolean saying if the pool admin
									    * allows you to change your picks
									    * after you've already made them.
									    * The reason you may not want to
									    * allow that, is to prevent people
									    * from changing other people's picks. */
cmp_t sortproc = (cmp_t) SMETHOD;	   /* The function we're currently using
									    * to sort the standings by. */
gradeproc_t gradeproc = GMETHOD;	   /* The function we're using to grade a
									    * player's picks. */
gradeproc_t gradeprocs[] =
{									   /* A list of possible grading
									    * functions to choose from. */
	Grade0, Grade1, Grade2, Grade3
};

extern ResultLine results;
extern int nResults;
extern short tieBreaker;
extern TeamName teamnames[kNumTeams];
extern int seeds[kNumTeams];

/*
 * This is used for the Consensus.  We need to sort the teams by most "hits."
 * We go thru the list of players, and for each player's pick, say in the
 * final 4, we chalk up a hit for that team.  Then we sort the list of teams
 * by who has the most hits.
 */
static int zzcmp(struct Hits *a, struct Hits *b)
{
	return (b->hits - a->hits);
}									   /* zzcmp */

/* Prints interesting information about how the other players picked. */
static void Consensus(void)
{
	PlayerLine line;
	FILE *fp;
	FILE *pager;
	int s1, s2;
	struct Hits final4[kNumTeams];	   /* The consensus of final-4 picks. */
	struct Hits champ[kNumTeams];	   /* Consensus of the national champ. */
	int i, nPlayers = 0;
	int rd1[kNumTeams / 2];			   /* Consensus of the first round.  We
									    * only need to track half the number
									    * total teams here, or one team from
									    * each pairing in the first round.
									    * Since we'll know how many players
									    * picked, If the number of players
									    * who picked the first team the first
									    * round is x = rd1[0], the number of
									    * players who picked the team that
									    * played against the first team in
									    * the first round is nPlayers - x. */
	Player pp;
	float x;

	if ((fp = fopen(kPlayerFile, "r")) != NULL) {
		/* Initially, none are picked by anyone in the first round. */
		for (i = 0; i < kNumTeams / 2; i++)
			rd1[i] = 0;

		/*
		 * Initially, none of the teams are picked to go to the final 4 or
		 * win it all since we haven't considered any of the players yet.
		 */
		for (i = 0; i < kNumTeams; i++) {
			final4[i].hits = champ[i].hits = 0;
			final4[i].tnum = champ[i].tnum = i;
		}

		/*
		 * Since the consensus report can be quite lengthy, send all output
		 * to the pager if possible so the user can browse at the user's own
		 * speed.
		 */
		pager = OpenPager();

		fprintf(pager,
				"Here are everyone's Final-4 picks.  An asterisk denotes their champion:\n\n");

		/* Read each player and their picks. */
		while (FGets(line, sizeof(line), fp) != NULL) {
			if (LineToPlyrPtr(line, &pp) == 0) {

				/*
				 * See if they picked these teams in the first round, for our
				 * first round report below.
				 */
				for (i = 0; i < 32; i++) {
					if (pp.picks[i] == (i * 2))
						rd1[i]++;
				}

				/* Chalk up one for their champion. */
				champ[(int) pp.picks[62]].hits++;

				/*
				 * While we chalk up hits for each of this guy's final 4
				 * picks, let's go ahead and print the final 4 for each
				 * player in the report, with an asterisk next to the one
				 * they picked as the champ.
				 */
				fprintf(pager, "%s: ", pp.name);
				for (i = 56; i < 60; i++) {
					final4[(int) pp.picks[i]].hits++;
					fprintf(
							   pager,
							   "%s%s",
							   teamnames[(int) pp.picks[i]],
							   pp.picks[i] == pp.picks[62] ? "*" : ""
						);
					fprintf(pager, (i < 59) ? ", " : "\n");
				}

				/* Track how many players we did. */
				nPlayers++;
			}
		}
		fclose(fp);

		/* Now start printing the rest of the report. */
		if (nPlayers > 0) {
			fprintf(pager, "\nConsensus of first round picks:\n\n");
			for (i = 0; i < 32; i++) {
				x = PERC(rd1[i], nPlayers);
				s1 = seeds[i * 2];
				s2 = seeds[i * 2 + 1];
				if (s1 <= s2) {
					fprintf(pager,
						"#%-2d %-20s %5.1f%% vs. #%-2d %-20s %5.1f%%\n",
						s1,
						teamnames[i * 2],
						x,
						s2,
						teamnames[i * 2 + 1],
						100.0 - x
					);
				} else {
					fprintf(pager,
						"#%-2d %-20s %5.1f%% vs. #%-2d %-20s %5.1f%%\n",
						s2,
						teamnames[i * 2 + 1],
						100.0 - x,
						s1,
						teamnames[i * 2],
						x
					);
				}
			}

			/* Sort the teams by who was picked more often. */
			qsort(final4, (size_t) kNumTeams, sizeof(struct Hits), (cmp_t) zzcmp);
			qsort(champ, (size_t) kNumTeams, sizeof(struct Hits), (cmp_t) zzcmp);

			/*
			 * Print the final-4 consensus. Note that we don't bother
			 * printing any teams who weren't picked at all.  Those will have
			 * been moved to the very bottom of the list from the sort.
			 */
			fprintf(pager, "\nFinal Four picks:\n\n");
			for (i = 0; i < kNumTeams && final4[i].hits > 0; i++)
				fprintf(pager, "%-25s %2d/%2d\n", teamnames[final4[i].tnum], final4[i].hits, nPlayers);

			/* Do the same for the champ. */
			fprintf(pager, "\nNCAA Champion:\n\n");
			for (i = 0; i < kNumTeams && champ[i].hits > 0; i++)
				fprintf(pager, "%-25s %2d/%2d\n", teamnames[champ[i].tnum], champ[i].hits, nPlayers);
		}
		ClosePager(pager);
	} else {
		fprintf(stderr,
				"Could not open the player file; perhaps no players have picked yet.\n");
		perror(kPlayerFile);
	}
}									   /* Consensus */



/* Print a player's picks, using as few lines as possible. */
static void ReviewPicks(void)
{
	PlayerLine line;
	char *cp;
	TeamName team;
	Player p;
	int i;

	if (GetPlayerName(curPlayerName) < 0)
		return;
	if (FindPlayer(curPlayerName, line) < 0) {
		fprintf(stderr, "%s: No such player.\n", curPlayerName);
		return;
	}
	/* Convert what FindPlayer gave us into something more meaningful. */
	if (LineToPlyrPtr(line, &p) < 0)
		return;

	/*
	 * Print the player's first round selections.  We use a cheezy wordwrap
	 * routine so we can print as many of the picks on a line as possible.
	 * We would rather not print one pick per line, or even columns, because
	 * we want to print all the picks on the screen at once.
	 */
	strcpy(line, "First Round Picks:  ");
	for (cp = p.picks, i = 0; i < 32; i++) {
		strcpy(team, teamnames[(int) *cp++]);
		if (strlen(line) + strlen(team) + 2 >= 80) {
			printf("%s\n", line);
			strcpy(line, "  ");
		}
		strcat(line, team);
		strcat(line, (i < 31 ? ", " : ".\n"));
	}
	if (strlen(line) > 2)
		printf("%s\n", line);

	strcpy(line, "Round 2:  ");
	for (i = 0; i < 16; i++) {
		strcpy(team, teamnames[(int) *cp++]);
		if (strlen(line) + strlen(team) + 2 >= 80) {
			printf("%s\n", line);
			strcpy(line, "  ");
		}
		strcat(line, team);
		strcat(line, (i < 15 ? ", " : ".\n"));
	}
	if (strlen(line) > 2)
		printf("%s\n", line);

	strcpy(line, "Round 3:  ");
	for (i = 0; i < 8; i++) {
		strcpy(team, teamnames[(int) *cp++]);
		if (strlen(line) + strlen(team) + 2 >= 80) {
			printf("%s\n", line);
			strcpy(line, "  ");
		}
		strcat(line, team);
		strcat(line, (i < 7 ? ", " : ".\n"));
	}
	if (strlen(line) > 2)
		printf("%s\n", line);

	strcpy(line, "Round 4:  ");
	for (i = 0; i < 4; i++) {
		strcpy(team, teamnames[(int) *cp++]);
		if (strlen(line) + strlen(team) + 2 < 80) {
			strcat(line, team);
			strcat(line, (i < 3 ? ", " : ".\n"));
		} else {
			printf("%s\n", line);
			strcpy(line, "  ");
		}
	}
	if (strlen(line) > 2)
		printf("%s\n", line);

	printf("Round 5:  %s", teamnames[(int) *cp++]);
	printf(", %s.\n\n", teamnames[(int) *cp++]);
	printf("Championship Game Winner:  %s (Tie-breaker: %d)\n",
		   teamnames[(int) *cp], p.tieBreaker);
}									   /* ReviewPicks */





/*
 * This method just keeps track of wins only.  We don't want to attach any
 * significance to what round it is, if the winner was an underdog or
 * whatever.  So this method only cares about win%.
 */
void Grade0(int t1, int t2, int pick, int winner, int rd, Player * p)
{
	if (pick == winner) {
		p->scores[rd]++;
		p->roundPts[rd]++;
	}
}									   /* Grade0 */




void Grade1(int t1, int t2, int pick, int winner, int rd, Player * p)
{
	if (pick == winner) {
		p->scores[rd]++;
		p->roundPts[rd] += (short) rMult[rd];
	}
}									   /* Grade1 */




void Grade3(int t1, int t2, int pick, int winner, int rd, Player * p)
{
	if (pick == winner) {
		p->scores[rd]++;
		p->roundPts[rd] += (short) r2Mult[rd];
	}
}									   /* Grade3 */




void Grade2(int t1, int t2, int pick, int winner, int rd, Player * p)
{
	int s1, s2;						   /* seeds of team 1 and team 2. */
	int tp, sp;						   /* team picked;  seed of team picked. */
	int tn, sn;						   /* team not picked;  seed of team not
									    * picked. */

	if (pick == winner) {
		p->scores[rd]++;
		s1 = seeds[t1];
		s2 = seeds[t2];
		if (pick == t1) {
			tp = t1;
			sp = s1;
			tn = t2;
			sn = s2;
		} else {
			tp = t2;
			sp = s2;
			tn = t1;
			sn = s1;
		}

		if (sp <= sn) {
			p->roundPts[rd] += (short) PtsPerPickedFave;	/* picked a favorite or
															 * equal */
		} else {
			p->roundPts[rd] += (short) PtsPerPickedUDog;	/* picked an underdog */
		}
	}
}									   /* Grade2 */



/* Compute a player's score. */
static void Grade(Player * p)
{
	int i, t1, t2;
	char *a, *b;

	/* Grade the first round, 32 games. */
	if (nResults > 0)
		for (i = 0, a = results, b = p->picks; i < 32; i++, a++, b++) {
			t1 = i * 2;
			t2 = t1 + 1;
			(*gradeproc) (t1, t2, (int) *b, (int) *a, 0, p);
		}

	/*
	 * If the admin has entered the results for round 2 or later, go ahead
	 * and grade those.
	 */
	if (nResults > 32)
		for (i = 0; i < 16; i++, a++, b++) {
			t1 = results[i * 2 + 0];
			t2 = results[i * 2 + 1 + 0];
			(*gradeproc) (t1, t2, (int) *b, (int) *a, 1, p);
		}
	/* Grade round 3. */
	if (nResults > 48)
		for (i = 0; i < 8; i++, a++, b++) {
			t1 = results[i * 2 + 32];
			t2 = results[i * 2 + 1 + 32];
			(*gradeproc) (t1, t2, (int) *b, (int) *a, 2, p);
		}
	/* Grade round 4. */
	if (nResults > 56)
		for (i = 0; i < 4; i++, a++, b++) {
			t1 = results[i * 2 + 48];
			t2 = results[i * 2 + 1 + 48];
			(*gradeproc) (t1, t2, (int) *b, (int) *a, 3, p);
		}
	/* Grade round 5. */
	if (nResults > 60)
		for (i = 0; i < 2; i++, a++, b++) {
			t1 = results[i * 2 + 56];
			t2 = results[i * 2 + 1 + 56];
			(*gradeproc) (t1, t2, (int) *b, (int) *a, 4, p);
		}
	/* Grade round 6, the final game. */
	if (nResults > 62) {
		t1 = results[60];
		t2 = results[61];
		(*gradeproc) (t1, t2, (int) *b, (int) *a, 5, p);
	}

	/*
	 * Now sum the total number of correct picks and total points by checking
	 * each round's scores.
	 */
	for (i = 0, a = p->roundPts, b = p->scores; i < 6; i++) {
		p->nCorrectPicks += (short) (*b++);
		p->points += (short) (*a++);
	}
}									   /* Grade */




/*
 * This function prints the number of points, unless the grading function in
 * use doesn't use points.  This is just here to make things simpler below
 * when printing the scores.
 */
static char *PointStr(int pts)
{
	static char s[32];

	s[0] = 0;						   /* init to an empty string */
	if (gradeproc != Grade0)
		sprintf(s, "%3d pts", pts);
	return (s);
}									   /* PointStr */




#define zz(rd,ngames) (int) player->scores[rd], ngames, PERC(player->scores[rd],ngames), (PointStr((int)player->roundPts[rd]))

static void ShowPlayerScore(Player * player, FILE * fp)
{

	/*
	 * Print the scores, round by round.  We'll only print scores for the
	 * rounds that have actually finished.
	 */
	fprintf(fp, "\nScores for %s:\n\n\
First Round:       %2d/%2d  (%5.1f%%)   %s\n",
			player->name,
			zz(0, 32)
	);

	if (nResults > 32) {
		fprintf(fp, "\
Second Round:      %2d/%2d  (%5.1f%%)   %s\n",
				zz(1, 16)
			);
	}
	if (nResults > 48) {
		fprintf(fp, "\
Third Round:       %2d/%2d  (%5.1f%%)   %s\n",
				zz(2, 8)
			);
	}
	if (nResults > 56) {
		fprintf(fp, "\
Fourth Round:      %2d/%2d  (%5.1f%%)   %s\n",
				zz(3, 4)
			);
	}
	if (nResults > 60) {
		fprintf(fp, "\
Final Four:        %2d/%2d  (%5.1f%%)   %s\n",
				zz(4, 2)
			);
	}
	if (nResults > 62) {
		fprintf(fp, "\
Championship Game: %2d/%2d  (%5.1f%%)   %s  (Tiebreaker: %d)\n",
				zz(5, 1),
				(int) player->tieBreaker
			);
	}
	fprintf(fp, "\n\
TOTALS:            %2d/%2d  (%5.1f%%)   %s\n\n",
			(int) player->nCorrectPicks, nResults, PERC(player->nCorrectPicks, nResults), PointStr((int) player->points)
		);
}									   /* ShowPlayerScore */




/* This prints out a more detailed report of a player's picks */
static void ShowYourScore(void)
{
	Player player;
	PlayerLine line2;

	if (nResults <= 0) {
		fprintf(stderr, "Results of the games have not been entered in yet.\n");
		return;
	}
	if (GetPlayerName(curPlayerName) < 0)
		return;
	if (FindPlayer(curPlayerName, line2) < 0) {
		fprintf(stderr, "%s: No such player.\n", curPlayerName);
		return;
	}
	if (LineToPlyrPtr(line2, &player) >= 0) {
		Grade(&player);
		ShowPlayerScore(&player, stdout);
	}
}									   /* ShowYourScore */



/*
 * This is called for each game the player needs to pick.  It returns the
 * team number of the one the player chose, or -1 if the user canceled.
 */
static int Pick(int t1, int t2, int repicking)
{
	int tmp;

	/*
	 * Make it so we list the team with the higher seed (lower number) first,
	 * then the underdog.
	 */
	if (seeds[t1] > seeds[t2]) {
		tmp = t2;
		t2 = t1;
		t1 = tmp;
	}
	printf("\nA. %s (%d)  vs.   B. %s (%d)\n",
		   teamnames[t1],
		   seeds[t1],				   /* The seed is often instrumental to
									    * which team the player will pick. */
		   teamnames[t2],
		   seeds[t2]
		);
	while (1) {
		printf("   Pick [A or B; Q to quit] ->");
		switch (InputChar()) {
			case '1':
			case 'A':

				/*
				 * We accept A/B as well as 1/2, since the program originally
				 * used 1/2 as the menu choices.
				 */
				return t1;
			case '2':
			case 'B':
				return t2;
			case 'Q':
				printf("   %s\n   %s [yn] ",
					   "Are you sure you want to quit picking?",
					   (repicking) ?
					   "We'll use your old picks unless you continue." :
				   "You'll have to pick all the games over again next time!"
					);
				if (InputChar() == 'Y') {
					return (-1);
				}
		}
	}
}									   /* Pick */




static int DoPicks(Picks * yourPicks, int isRePicking)
{
	int i, team1, team2, pick;
	int picksComplete = 0;
	PlayerLine line;
	Picks picksCopy;				   /* Work with a copy, so we can use a.b
									    * instead of a->b, to save a little
									    * work. */

	/*
	 * Go through and ask them each game.  Store the picks in the array, with
	 * team numbers as the entries.
	 */
	printf("First round, 32 games to pick.\n");
	for (i = 0; i < 32; i++) {
		team1 = 2 * i;
		team2 = team1 + 1;
		if ((pick = Pick(team1, team2, isRePicking)) < 0)
			goto done;
		picksCopy.firstrd[i] = (char) pick;
	}

	printf("\nSecond round, 16 games to pick.\n");
	for (i = 0; i < 16; i++) {
		team1 = (int) picksCopy.firstrd[2 * i];
		team2 = (int) picksCopy.firstrd[2 * i + 1];
		if ((pick = Pick(team1, team2, isRePicking)) < 0)
			goto done;
		picksCopy.secondrd[i] = (char) pick;
	}

	printf("\nThird round (Sweet 16), 8 games to pick.\n");
	for (i = 0; i < 8; i++) {
		team1 = (int) picksCopy.secondrd[2 * i];
		team2 = (int) picksCopy.secondrd[2 * i + 1];
		if ((pick = Pick(team1, team2, isRePicking)) < 0)
			goto done;
		picksCopy.thirdrd[i] = (char) pick;
	}

	printf("\nFourth round, 4 games to pick.\n");
	for (i = 0; i < 4; i++) {
		team1 = (int) picksCopy.thirdrd[2 * i];
		team2 = (int) picksCopy.thirdrd[2 * i + 1];
		if ((pick = Pick(team1, team2, isRePicking)) < 0)
			goto done;
		picksCopy.fourthrd[i] = (char) pick;
	}

	printf("\nFifth round (Final 4), 2 games to pick.\n");
	for (i = 0; i < 2; i++) {
		team1 = (int) picksCopy.fourthrd[2 * i];
		team2 = (int) picksCopy.fourthrd[2 * i + 1];
		if ((pick = Pick(team1, team2, isRePicking)) < 0)
			goto done;
		picksCopy.fifthrd[i] = (char) pick;
	}

	printf("\nSixth round (National Championship Game).\n");
	team1 = (int) picksCopy.fifthrd[0];
	team2 = (int) picksCopy.fifthrd[1];
	if ((pick = Pick(team1, team2, isRePicking)) < 0)
		goto done;
	picksCopy.sixthrd = (char) pick;

	picksCopy.tieBreaker = 0;
	printf(
"\nTiebreaker:  Predict the total score of BOTH teams, and in the event of\n"
"a tie, whoever is closest to the actual total score wins.\n\n"
"Tiebreaker: ->");
	if (FGets(line, sizeof(line), stdin) != NULL)
		picksCopy.tieBreaker = (short) atoi(line);

	/* Copy them back to the original. */
	*yourPicks = picksCopy;
	picksComplete = 1;

  done:
	return (picksComplete);
}									   /* DoPicks */




static void MakeYourPicks(void)
{
	Picks yourPicks;
	PlayerLine line;
	int replacePicks = 0, err;
	FILE *fp;

	/*
	 * If the lock file is in place, the tournament has started play already,
	 * so don't let them cheat and pick games now.
	 */
	if ((fp = fopen(kLockFile, "r")) != NULL) {
		fprintf(stderr,
		"Sorry, you can't pick/re-pick because you missed the deadline.\n");
		fclose(fp);
		return;
	}
	/* See who's picking. */
	fprintf(stderr,
			"Enter your name below (i.e. Firstname Lastname).  Remember what you supply\n"
			"here, because you'll need to enter the same thing if you want to check your\n"
			"picks as the tournament progresses.\n\n"
		);

	if (GetPlayerName(curPlayerName) < 0)
		return;
	if (FindPlayer(curPlayerName, line) >= 0) {
		/* Player has already picked. */
		printf("\nThere is already a player by that name.\n");
		if (canRePick) {
			printf("Do you want to re-do your picks? [yn] ");
			if (InputChar() == 'Y')
				replacePicks = 1;
			else
				return;
		} else {
			return;
		}
	}
	if (DoPicks(&yourPicks, replacePicks) == 1) {
		if (replacePicks)
			err = ReplacePlayerPicks(curPlayerName, &yourPicks);
		else
			err = AddToPlayerFile(curPlayerName, &yourPicks);

		if (err == 0) {
			printf(
   "\nThat's it.  Don't forget you can run the program again over the course\n"
   "of the tournament!  There will be more things to do once the games get\n"
   "started, like checking the standings, etc.\n\n"
   "If you ever forget what you picked, come back and check your picks from\n"
   "the main menu.  You can also get an idea how the other players picked by\n"
   "checking the consensus (try it!).\n\n"
				);
		}
	}
}									   /* MakeYourPicks */



/*
 * If we're trying to determine which player has the better score, we can use
 * the last tiebreaker, if the whole tournament has finished.  The players
 * were told to guess the total number of points scored, so we compare that
 * with that actual total here. Whoever was closest wins.  It doesn't matter
 * if they were too high or low (this ain't the Price is Right).
 */
static int byTieBreaker(Player * a, Player * b)
{
	short aa, bb;
	int i = 0;

	if (tieBreaker != 0) {
		aa = a->tieBreaker - tieBreaker;
		if (aa < 0)
			aa = -aa;
		bb = b->tieBreaker - tieBreaker;
		if (bb < 0)
			bb = -bb;
		i = (int) (aa - bb);
	}
	return i;
}									   /* byTieBreaker */




/*
 * This decides which player did better, but using points scored in the
 * current grading system first, and if it's tied, then using total number of
 * wins second.  If still tied, use the tiebreaker, discussed above.
 */
int byPoints(Player * a, Player * b)
{
	int i;

	i = ((int) (b->points - a->points));
	if (i == 0) {
		i = byTieBreaker(a, b);
		if (i == 0)
			i = ((int) (b->nCorrectPicks - a->nCorrectPicks));
	}
	return i;
}									   /* byPoints */



/*
 * If the pool admin chose this sorting method, total number of wins is more
 * important than the points.
 */
int byWins(Player * a, Player * b)
{
	int i;

	i = ((int) (b->nCorrectPicks - a->nCorrectPicks));
	if (i == 0) {
		i = ((int) (b->points - a->points));
		if (i == 0)
			i = byTieBreaker(a, b);
	}
	return i;
}									   /* byWins */



/*
 * Print the total scores of all the players in the pool, sorted by best
 * scores if possible.
 */
static void Standings(int longMode)
{
	FILE *fp, *pager;
	PlayerLine line;
	int i, nPlayers;
	Player *players, aPlayer;

	if (nResults <= 0) {
		fprintf(stderr, "Results of the games have not been entered in yet.\n");
		return;
	}
	if ((fp = fopen(kPlayerFile, "r")) == NULL) {
		perror(kPlayerFile);
		goto aa;
	}

	/*
	 * Unfortunately we read the file twice, once just to count how many
	 * players there are.  If there were a great number of players
	 * participating, it would be silly to do this each time a player ran the
	 * program.  Instead, we could have this program or the "adminp" program
	 * generate the standings and have this program just dump a file to the
	 * screen.
	 */
	ProgressMsg("Counting players...");
	nPlayers = 0;
	while (FGets(line, sizeof(line), fp) != NULL)
		nPlayers++;
	rewind(fp);

	if (nPlayers == 0) {
		fprintf(stderr, "Empty player file.\n");
	  aa:fprintf(stderr, "Could it be that no one has picked yet?\n");
		return;
	}

	/*
	 * To make sorting the standings feasible, we have to read in the whole
	 * player file essentially.
	 */
	players = (Player *)
		calloc((size_t) nPlayers, sizeof(Player));

	if (players == (Player *) 0) {

		/*
		 * Hmmmm... couldn't read all the players into memory at once, so we
		 * won't be able to sort them I guess.
		 */

		/* Erase the last msg, because we're ready to go. */
		ProgressMsg("");

		/*
		 * Use the pager, in case there are a lot of players.  We wouldn't
		 * want the standings to scroll of screen on a dumb terminal.
		 */
		pager = OpenPager();

		i = 0;
		while (FGets(line, sizeof(line), fp) != NULL) {
			if ((LineToPlyrPtr(line, &aPlayer)) < 0)
				goto done;

			Grade(&players[i]);
			if (longMode)
				ShowPlayerScore(&players[i], pager);
			else
				fprintf(pager,
						"%-40s %2d/%2d %5.1f%%  %s\n",
						aPlayer.name,
						(int) aPlayer.nCorrectPicks,
						(int) nResults,
						PERC(aPlayer.nCorrectPicks, nResults),
						PointStr((int) aPlayer.points)
					);
		}
	} else {
		/* We can read all the players at once.  */

		ProgressMsg("Grading...");
		/* Now read each player for real this time and grade each one. */
		for (i = 0; i < nPlayers; i++) {
			if (FGets(line, sizeof(line), fp) == NULL) {

				/*
				 * Probably never going to happen, since we read it just fine
				 * last time.
				 */
				fprintf(stderr, "%s screwed up.\n", kPlayerFile);
				goto done;
			}
			if ((LineToPlyrPtr(line, &players[i])) < 0)
				goto done;
			Grade(&players[i]);
		}

		ProgressMsg("Sorting...");
		qsort(players, (size_t) nPlayers, sizeof(Player), sortproc);

		/* Erase the last msg, because we're ready to go. */
		ProgressMsg("");

		/*
		 * Use the pager, in case there are a lot of players.  We wouldn't
		 * want the standings to scroll of screen on a dumb terminal.
		 */
		pager = OpenPager();

		if (!longMode)
			fprintf(pager,
					"Rank  Player                                          Perc    %6s  TBrk\n",
					(gradeproc != Grade0) ? "Points" : ""
				);

		for (i = 0; i < nPlayers; i++) {
			if (longMode)
				ShowPlayerScore(&players[i], pager);
			else {
				fprintf(pager,
						"%3d.  %-40s %2d/%2d %5.1f%%  %-7s  ",
						i + 1,
						players[i].name,
						(int) players[i].nCorrectPicks,
						nResults,
						PERC(players[i].nCorrectPicks, nResults),
						PointStr((int) players[i].points)
					);

				/*
				 * If we know the real tiebreaker, print how far off each
				 * player was, or if we're midway through the tournament,
				 * print their guess instead.
				 */
				if (tieBreaker == 0)
					fprintf(pager, "%4d\n", (int) players[i].tieBreaker);
				else {
					fprintf(pager, " %+-2d\n",
							(int) (players[i].tieBreaker - tieBreaker)
						);
				}
			}
		}
		free(players);
	}

  done:
	fclose(fp);
	ClosePager(pager);
}									   /* Standings */



/*
 * If we ran the program with no command line switches, we want the program
 * to use the grading and sorting methods specified by the pool
 * administrator.  These options are created by the "adminp" program, and now
 * we want to read them back in here.
 */
static void ReadDefaultOptions(void)
{
	OptionsLine line;
	FILE *fp;
	int i;

	fp = fopen(kOptionsFile, "r");
	if (fp == NULL) {
		fprintf(stderr,
				"Could not open the options file!\nHas the admin configured the pool by running the adminp program?\n");
		exit(1);
	} else {
		if (FGets(line, sizeof(line), fp) != NULL) {
			i = atoi(line);

			/* Set the grading procedure, if we were given a valid index. */
			if ((i >= 0) && (i < nGradeProcs))
				gradeproc = gradeprocs[i];
			if (FGets(line, sizeof(line), fp) != NULL) {
				i = atoi(line);

				/*
				 * There are only 2 sorting procedures, so don't need to
				 * maintain a list of them.
				 */
				sortproc = (i == 1) ? ((cmp_t) byPoints) : ((cmp_t) byWins);
			}
			if (FGets(line, sizeof(line), fp) != NULL)
				canRePick = atoi(line);
		}
		fclose(fp);
	}
}									   /* ReadDefaultOptions */




static void Usage(void)
{
	fprintf(stderr,
			"ncaa by Mike Gleason, NCEMRSoft (mgleason@cse.unl.edu).\n"
			"Version %s.\n"
			"Usage: ncaa [-gN] [-pN] [-c | -r | -s | -S]\n",
			kVersion);

	fprintf(stderr,
			"    -c   :  Show a consensus of all the players' picks.\n"
			"    -gN  :  Grade by method N.\n"
			"    -pN  :  Print standings sorted by method N.\n"
			"    -r   :  Review your (or someone else's) picks.\n"
			"    -s   :  Show current tournament standings.\n"
		 	"    -S   :  Show round-by-round scores for all players (long!).\n"
		);

	fprintf(stderr,
			"\nGrading methods:\n"
			"    -g0  :  No point system; just use number of correct picks.\n"
			"    -g1  :  1,3,5,7,9,11 points per correct pick per round.\n"
			"    -g2  :  1 point for correct favorite, 2 for correct underdog.\n"
			"    -g3  :  1,2,4,8,16,32 points per correct pick per round.\n"
		);

	fprintf(stderr,
			"\nSorting methods:\n"
			"    -p1  :  Winning percentage, then points.\n"
			"    -p2  :  Points, then winning percentage.\n"
		);
}									   /* Usage */





void main(int argc, char **argv)
{
	char *cp;
	int i, c, b;
	int done;

	done = 0;
	InitRandom();					   /* Important for LockFile(). */

	/*
	 * Go to our document directory.  Then we can use just filenames to refer
	 * to the various files we need to use.
	 */
	SetDataFileDirectory();

	ReadDefaultOptions();
	ReadTournamentFile();
	ReadResultsFile();				   /* May or may not exist at this point. */

	if (argc > 1) {

		/*
		 * We make two passes through the command line switches.  The first
		 * is to see if the user wanted to change the grading/sorting
		 * options, because the other switches are commands which may use g/s
		 * options, and we assume they want to use the g/s options.
		 * 
		 * So the first pass we check to see if a switch is valid (if not they
		 * get the Usage message), or if a switch is controls the g/s
		 * options.
		 */
		for (i = 1; i < argc; i++) {
			cp = argv[i];
			if (*cp == '-')
				++cp;
			switch (*cp) {
				case 'c':
				case 'r':
				case 's':
				case 'S':
					/* We take care of these in the next pass. */
					break;

					/*
					 * We let the user change the default grading and sorting
					 * options.  That way they can see how they would do
					 * hypothetically by using the other methods.  It also
					 * offers a way for the author to test them out without
					 * running the "adminp" program to change them each time.
					 */
				case 'g':
					c = (*++cp) - '0';
					if ((c >= 0) && (c < nGradeProcs))
						gradeproc = gradeprocs[c];
					break;
				case 'p':
					c = *++cp;
					if (c != 0) {
						if (c == '1')
							sortproc = (cmp_t) byPoints;
						else
							sortproc = (cmp_t) byWins;
					}
					break;
				default:
					/* Illegal switch. */
					Usage();
					exit(1);
			}
		}

		for (i = 1; i < argc; i++) {
			cp = argv[i];
			if (*cp == '-')
				++cp;
			switch (*cp) {
				case 'c':
					Consensus();
					exit(0);
				case 'r':
					ReviewPicks();
					exit(0);
				case 's':
					Standings(0);
					exit(0);
				case 'S':
					Standings(1);
					exit(0);
			}
		}

	}
	done = 0;
	ClearScreen();

	/*
	 * Main menu.  Pretty no-frills but it works and will have to do until I
	 * right a pretty curses/termcap interface.
	 */
	do {
		printf(
				"NCAA %s\nMain Menu:\n\n"
				"  <M>  MAKE your picks\n"
				"  <R>  REVIEW your picks\n"
				"  <C>  Show a CONSENSUS of everyone's picks\n",
				  kVersion
			);
#if WANTPSCODE
		printf(
				"  <P>  PRINT your picks\n"
				"  <B>  print BLANK tournament template\n"
		);
		if (nResults > 0) {
			printf(
				"  <Z>  print CURRENT tournament results\n"
			);
		}
#endif
		if (nResults > 0) {
			printf(
				"  <K>  Show your current SCORE, round by round.\n"
				"  <S>  Show the STANDINGS with everyone's total scores\n"
			);
		}

		printf("  <Q>  QUIT\n\n");

		printf("Choice: ");
		c = InputChar();

		b = 0;
		if (nResults > 0) {
			b = 1;
			switch (c) {
				case 'K': ShowYourScore(); break;
				case 'S': Standings(0); break;
#if WANTPSCODE
				case 'Z': Print(kPrintResults); break;
#endif
				default: b = 0;
			}
		}
		if (b == 0) switch (c) {
			case 'M':
				MakeYourPicks();
				break;
			case 'R':
				ReviewPicks();
				break;
			case 'C':
				Consensus();
				break;
#if WANTPSCODE
			case 'P':
				Print(kPrintPicks);
				break;
			case 'B':
				Print(kPrintBlank);
				break;
#endif
			case 'Q':
				done = 1;
				break;
			default:
				printf("Bad selection '%c' -- please try again.\n", c);
		}
		if (c != 'Q') {
			HitAnyKey();
			ClearScreen();
		}
	} while (!done);

	ClearScreen();

	exit(0);
}									   /* main */
