/* Shared stuff between the two programs: */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#include "ncaa.h"

int nResults;						/* This is the number of games that
									 * have been entered by the admin,
									 * and is computed when you attempt
									 * to read the results file which
									 * contains the game results.
									 */
ResultLine results;					/* An array representing all the
									 * games in the tournament, with
									 * entries corresponding to team
									 * numbers.
									 */
TeamName teamnames[kNumTeams + 1];	/* Keep track of the names of the
									 * teams which are competing in
									 * the tournament.  We'll use
									 * these over and over again whenever
									 * you want to print something
									 * about the teams.  Also reserve
									 * an extra one in case we accidentally
									 * reference team number "kEOPicks."
									 */
short tieBreaker = 0;				/* If non-zero, this was the total
									 * score of both teams in the
									 * championship game.
									 */
char *Regionals[4] = {"East", "Midwest", "West", "Southeast"};

/* IMPORTANT:  The teams in the 'tourney.txt' file must be listed in the
 * order below.  For example, the #1 seed comes first, followed by the #16
 * seed, etc.
 */
int seeds[kNumTeams] =
{
	1, 16, 9, 8, 5, 12, 13, 4, 3, 14, 11, 6, 7, 10, 15, 2,
	1, 16, 9, 8, 5, 12, 13, 4, 3, 14, 11, 6, 7, 10, 15, 2,
	1, 16, 9, 8, 5, 12, 13, 4, 3, 14, 11, 6, 7, 10, 15, 2,
	1, 16, 9, 8, 5, 12, 13, 4, 3, 14, 11, 6, 7, 10, 15, 2
};


/* Used by the menus mostly.  I could have sworn that \f cleared the
 * screen, but I guess not.  We'll have to do it the stupid way
 * until a proper curses/termcap interface is installed.
 */
void ClearScreen(void)
{
#ifdef TPUT
	char cmd[128];

	sprintf(cmd, "%s clear 2> /dev/null", TPUT);
	system(cmd);
#else
	int i;
	for (i = 0; i < 25; i++) putchar('\n');
#endif
}	/* ClearScreen */



/* You can put up a little message if you expect to be doing something
 * awhile.  It's setup so subsequent messages erase the old ones.
 */
void ProgressMsg(char *msg)
{
#ifdef unix
	/* Probably don't want these messages if we're not interactive. */
	if (!isatty(1)) return;
#endif
	fprintf(stderr, "\r%-78s", msg);
	fflush(stderr);
}	/* ProgressMsg */



void InitRandom(void)
{
	srand((unsigned int) time(NULL));
}	/* InitRandom */




/* For things that may take up more than one screen, use a pager program
 * if possible, since this program could be run from a dumb terminal
 * with no way to scroll back and view older text.
 */
FILE *OpenPager(void)
{
#if	HAVE_POPEN
	FILE *fp = stdout;
	
	/* If we're running unix, try to use the pager when we can.  We
	 * don't want to use the pager if we're being redirected to a
	 * file though.  In that case, just return stdout which will
	 * be the redirected file.
	 */
#ifdef unix
	if (isatty(1)) {
#else
	{
#endif
		/* Try opening the "less" pager if you have it.  That way
		 * in case novice users are using the pool, and wouldn't know
		 * what to do at a "more" prompt, we can use less's prompt
		 * customization feature to tell them that they have to
		 * hit space.
		 */
		fp = popen(PAGER, "w");
		if (fp == NULL)
			fp = stdout;
	}
	return (fp);
#else
	return (stdout);
#endif
}	/* OpenPager */




int ClosePipe(FILE *fp)
{
	if ((fp != stdout) && (fp != NULL))
#if HAVE_POPEN
		return (pclose(fp));
#else
		return (fclose(fp));
#endif
	return (0);
}	/* ClosePipe */



/* Since the ncaa program is usually setuid, make sure they don't
 * do anything sneaky like supply "| rm -rf /" as a ``mail address''
 * or anything else.
 */
int CheckForIllegals(char *cp)
{
	int bad = 0;
	
	for (; *cp; cp++) {
		if ((*cp == '|') || (*cp == '>') || (*cp == ';'))
			bad = 1;
	}
	return bad;
}	/* CheckForIllegals */




void SetDataFileDirectory(void)
{
	/* The document directory, DFDIR, is defined in the Makefile, and
	 * passed as a command line define to the compiler.  It could be
	 * an empty string if the author is compiling this on a Macintosh
	 * for testing purposes, but other than that it must be defined
	 * on UNIX so we can find the data files.  To make things easier,
	 * we just chdir right away to this directory, so we can just open
	 * the datafiles using the filename only (and not full pathnames).
	 */
	if ((DFDIR[0] != '\0') && (chdir(DFDIR) < 0)) {
		fprintf(stderr, "Fatal error:  could not set working directory!\n");
		perror("chdir");
		exit(1);
	}
}	/* SetDataFileDirectory */



/* Just a simplification to make calling fgets more suitable for
 * our purposes.  If we're reading a file, it will skip blank
 * lines and comment lines;  it also gets rid of the trailing
 * newline which we don't want.
 */
char *FGets(char *in, size_t max, FILE * fp)
{
	char *cp;

	while (1) {
		if ((cp = fgets(in, (int) (max - 1), fp)) == NULL)
			break;
		cp[strlen(cp) - 1] = 0;		   /* no newline please. */
		if ((fp == stdin) || (*cp != kCommentChar && *cp != 0 && !iscntrl(*cp)))
			break;
	}
	return cp;
}									   /* FGets */



/* Need to read the tournament.  It must have the East regional first,
 * followed by the Midwest, West, and Southeast.  Well, I guess that's
 * arbitrary, but the year the original version was written
 * had it that way.  The important thing is that the first two
 * regionals' winners play each other, and the next two regionals'
 * winners play each other.
 *
 * Each regional must have the teams listed according to the seeds
 * array, above.  So a regional would have the top seed on one line,
 * followed by the 16 seed, the 9, 8, ..., so on, down to 15, 2.
 *
 * Each line contains the team name.  You may have the seed
 * numbers on the line also, enclosed in parentheses if you like.
 */
void ReadTournamentFile(void)
{
	FILE *fp;
	char *cp;
	TeamName tmpteamnames[kNumTeams/4];
	int i, b, regB;
	TournLine line;

	if ((fp = fopen(kTournFile, "r")) == NULL) {
		fprintf(stderr, "Fatal error:  could not find the tournament file!\n");
		perror(kTournFile);
		exit(1);
	}
	for (i = 0; i < kNumTeams; ++i) {
		if (FGets(line, sizeof(line), fp) == NULL)
			break;

		/* If the team name included a seed number, like "Nebraska (7)", 
		 * get rid of the " (7)" part.  The reason they might have those
		 * in there in the first place is because the tournament file should
		 * have them, so you can remember which line is for which seed.
		 * That way, the next year, you can just look at the official
		 * bracket and replace the old teams with the new without worrying
		 * about the correct order.
		 */
		cp = strrchr(line, '(');
		if (cp != NULL) {
			*cp = 0;
			while (isspace(*--cp))
				*cp = 0;
		}
		strncpy(teamnames[i], line, sizeof(teamnames[i]) - 1);
	}
	if (i != kNumTeams) {
		fprintf(stderr, "Only read %d teams from tournament file!\n", i);
		exit(1);
	}
	teamnames[i][0] = '\0';

	/* The Tournament file must have the regionals listed in this
	 * order: E, MW, W, SE.  But if the tournament this year is not
	 * setup so that E plays MW, and W plays SE, we'll have to switch
	 * the order of somethings so that program works correctly.  The
	 * program assumes that regional #1 plays #2, and regional #3 plays
	 * regional #4, so if this year is E/W or E/SE instead of E/MW
	 * we swap whoever E faces with MW.
	 */
	line[0] = '\0';
	(void) FGets(line, sizeof(line), fp);
	switch(toupper(line[0])) {
		case 'M':
			b = 1;
			/* Don't have to switch anything, since by default the East
			 * plays the Midwest.
			 */
			break;
		case 'W':
			/* Have to switch the Midwest and West then. */
			b = 2;
			regB = 32;
			break;
		case 'S':
			/* Have to switch the Midwest and Southeast then. */
			b = 3;
			regB = 48;
			break;
		default: fprintf(stderr,
"Incomplete tournament file;  I need to know which regional the East\n"
"plays;  the last line should have the name of this regional.\n"
			);
			exit(1);
	}

	if (b > 1) {
		/* Swap places in the regional name list. */
		cp = Regionals[1];
		Regionals[1] = Regionals[b];
		Regionals[b] = cp;
		
		/* Swap the regionals teams. */			
		memcpy(tmpteamnames, teamnames + 16, sizeof(tmpteamnames));
		memcpy(teamnames + 16, teamnames + regB, sizeof(tmpteamnames));
		memcpy(teamnames + regB, tmpteamnames, sizeof(tmpteamnames));
	}
	(void) fclose(fp);
}									   /* ReadTournamentFile */




void ClosePlayerFileOpenedForWriting(FILE *fp)
{
	fclose(fp);
	if (remove(kPlayerFileLock) < 0) {
		fprintf(stderr, "Could not remove lock on the player file!\n"),
		perror(kPlayerFileLock);
		/* This may deadlock other instances of the program! */
	}
}	/* ClosePlayerFileOpenedForWriting */



int FileExists(char *fname)
{
	int result;

#ifdef unix
	result = (access(fname, 0) == 0);
#else
	/* Check to see the lock exists the cheezy way, using ANSI C. */
	FILE *tmpfp;
	
	tmpfp = fopen(fname, "r");
	if (tmpfp != NULL) {
		result = 1;
		fclose(tmpfp);
	} else
		result = 0;
#endif
	return (result);
}	/* FileExists */




/* This creates a new lock file.  I don't want to use any of unix's
 * various locking schemes, because this may not be running on unix.
 * So we'll roll our own using ANSI C.  Note that the name of the
 * file given here is the name of the lock file, not the file
 * you're equating this with the lock!
 */
int LockFile(char *lockfname)
{
	FILE *lockfp;
	int slept = 0;

#if MultipleInstances
	while (FileExists(lockfname)) {
		if (++slept == 1)
			ProgressMsg("Please wait...");
		sleep((unsigned int) ((rand() % 3) + 1));
	}
	if (slept)
		ProgressMsg("");
#endif

	/* In theory, a race condition could occur when two or more
	 * instances of the program were polling the lock file, when
	 * the lock was removed, and simultaneously detected it,
	 * and now they both open a new lock.  In reality, the odds
	 * of that happening are probably infinitesimal.  You'd have to
	 * be running a pool with players in the thousands, and have
	 * about 100 instances of the program running at once...
	 */
	lockfp = fopen(lockfname, "w");
	if (lockfp == NULL)
		return (-1);
	fprintf(lockfp, "Just a lock file.\n");
	fclose(lockfp);
	/* Don't let other users remove the lock file. */
	PrivateFile(kLockFile);
	return (0);
}	/* LockFile */




/* To prevent corruption of the player file that could result from
 * two instances of the program trying to write a player to the
 * player file, try preventing more than one instance from having
 * write permission to the player file at a time.
 */
FILE *OpenPlayerFileForWriting(int append)
{
	FILE *pfp;

	if (LockFile(kPlayerFileLock) < 0) {
		fprintf(stderr, "Could not lock the player file!\n");
		perror(kPlayerFileLock);
	} else if ((pfp = fopen(kPlayerFile, append ? "a" : "w")) == NULL) {
		fprintf(stderr, "Could not open the player file!\n");
		perror(kPlayerFile);
	}	
	return (pfp);
}	/* OpenPlayerFileForWriting */



static int MatchedPlayerLine(char *line, char *pname)
{
	char *cp;
	int result;

	/* The player name field end with a colon. */
	cp = strchr(line, ':');
	if (cp == NULL) {
		fprintf(stderr, "%s in wrong format.\n", kPlayerFile);
		exit(1);
	}
	*cp = 0;
	result = (! STRCMP(line, pname));
	*cp = ':';
	return (result);
}	/* MatchedPlayerLine */




/* Given a player's name, see if we can find it in the players file.
 * If so, write the whole line we found back in 'line.'  This is just
 * a sequential search on an unsorted file, so this may need to be
 * revised for really large pools.
 */
int FindPlayer(PlayerName pname, PlayerLine line)
{
	FILE *fp;

	if ((fp = fopen(kPlayerFile, "r")) != NULL) {
		while (FGets(line, sizeof(PlayerLine), fp) != NULL) {
			if (MatchedPlayerLine(line, pname)) {
				fclose(fp);
				return 0;
			}
		}
		fclose(fp);
	}
	line[0] = 0;
	return -1;
}									   /* FindPlayer */



/* This writes the picks to the open player file supplied. */
static void AddPlayerPicks(char *playerName, Picks *p, FILE *fp)
{
	PlayerLine line;
	char *cp;
	int i;
	
	/* First of all, have to convert the picks, which are arrays of
	 * team numbers (0..63) into a character string with printing
	 * characters.  There are 64 ASCII characters above space which are
	 * printable, so just add the ASCII number for space to each
	 * pick.
	 */
	cp = line;
	for (i = 0; i < 32; i++)
		*cp++ = p->firstrd[i] + ' ' + 1;
	for (i = 0; i < 16; i++)
		*cp++ = p->secondrd[i] + ' ' + 1;
	for (i = 0; i < 8; i++)
		*cp++ = p->thirdrd[i] + ' ' + 1;
	for (i = 0; i < 4; i++)
		*cp++ = p->fourthrd[i] + ' ' + 1;
	for (i = 0; i < 2; i++)
		*cp++ = p->fifthrd[i] + ' ' + 1;
	*cp++ = p->sixthrd + ' ' + 1;
	*cp = 0;

	/* Write the players name, picks, and tiebreaker to the player file.
	 * Since the player's name may contain spaces, we denote the
	 * end of the name with a colon.  GetPlayerName makes sure the
	 * name doesn't have any colons in it, to prevent confusion
	 * later when reading in the picks.
	 */
	fprintf(fp, "%s: %s %d\n", playerName, line, (int) p->tieBreaker);
}	/* AddPlayerPicks */




int ReplacePlayerPicks(PlayerName pName, Picks *p)
{
	FILE *infp, *outfp;
	PlayerLine line;
	int result = -1;

	/* We definitely do not want overwrite the original now, in case
	 * something goes wrong.  We'll write to a copy, then if it worked,
	 * replace the old file with the copy.
	 */
	if ((outfp = fopen(kTmpFile, "w")) == NULL) {
		fprintf(stderr, "Could not create a temporary file!\n");
		return (result);
	}

	/* Lock the player file now, so nobody adds players while
	 * we're copying the player file.
	 */
	if (LockFile(kPlayerFileLock) < 0) {
		fprintf(stderr, "Could not lock the player file!\n");
		perror(kPlayerFileLock);
		return (result);
	}

	if ((infp = fopen(kPlayerFile, "r")) == NULL) {
		fprintf(stderr, "Could not open the player file!\n");
		perror(kPlayerFile);
	}
	
	while (FGets(line, sizeof(PlayerLine), infp) != NULL) {
		if (MatchedPlayerLine(line, pName)) {
			/* Found the player, now rewrite the new picks. */
			AddPlayerPicks(pName, p, outfp);
		} else {
			/* Just write the line back out again, since it wasn't
			 * the player we're looking for.
			 */
			fprintf(outfp, "%s\n", line);
		}
	}
	fclose(outfp);
	fclose(infp);
	
	if (remove(kPlayerFile) < 0) {
		fprintf(stderr, "Could not delete the old player file!\n");
		perror(kPlayerFile);
		goto done;
	}
	if (rename(kTmpFile, kPlayerFile) < 0) {
		fprintf(stderr, "Could not rename the new player file!\n");
		perror(kPlayerFile);
		goto done;
	}
	PrivateFile(kPlayerFile);
	result = 0;
done:
	(void) remove(kPlayerFileLock);
	return result;
}	/* DeletePlayer */




/* This just prompts for a name, and makes sure the name doesn't
 * contain any illegal characters.
 */
int GetPlayerName(PlayerName pName)
{
	char *cp;
	PlayerName oldName;
	int hadOldName;

	printf("Player name [");
	if ((hadOldName = (pName[0] != '\0'))) 
		printf("%s for \"%s\"]: ", RETURNKEY, strcpy(oldName, pName));
	else
		printf("Firstname Lastname]: ");
	FGets(pName, sizeof(PlayerName), stdin);
	if (pName[0] == '\0') {
		if (hadOldName)
			strcpy(pName, oldName);
		else {
			printf("Nevermind.\n");
			return (-1);
		}
	}

	/* Get rid of these characters which are in the format of the player
	 * file.
	 */
	for (cp = pName; *cp != '\0'; cp++)
		if (*cp == ':' || !isprint(*cp))
			*cp = '_';

	return (0);
}									   /* GetPlayerName */




int AddToPlayerFile(char *playerName, Picks *p)
{
	FILE *fp;

	fp = OpenPlayerFileForWriting(1);
	AddPlayerPicks(playerName, p, fp);
	ClosePlayerFileOpenedForWriting(fp);
	PrivateFile(kPlayerFile);
	return 0;
}									   /* AddToPlayerFile */




/* This converts a string which was written to one of the output files
 * into it's proper representation, which is an array of 6-bit integers
 * (since there are 64 teams) with each element being a team number
 * ranging from 0..63.  The string has to be composed of ASCII characters
 * numbered over 32 (space), since anything less is a control character,
 * and we wanted the output files to be normal, readable, text files.
 */
void StringToPicks(char *cp)
{
	for (; *cp != '\0'; cp++)
		*cp -= ' ' + 1;
	*cp = kEOPicks;
}	/* StringToPicks */



/* Try to read the results of the games already played.  We may
 * read no results, some results, or all the results depending
 * whether the admin has entered in the games using the "adminp"
 * program.
 */
void ReadResultsFile(void)
{
	FILE *fp;
	ResultLine line;

	/* We determine whether the results file was read by checking to
	 * see if nResults is > 0, so init it to 0 here.
	 */
	nResults = 0;
	
	/* Initialize the whole result string to zeroes. */
	memset(results, 0, sizeof(results));

	if ((fp = fopen(kResultsFile, "r")) != NULL) {
		/* Read in the results we have, which will be in the form
		 * of a printable string.  It will not have a newline at
		 * the end, so no need to worry about that when we convert
		 * the string, below.
		 */
		FGets(results, sizeof(results), fp);
		
		/* Count how many games we have results for.  We use this
		 * a lot elsewhere.
		 */
		nResults = (int) strlen(results);
		
		/* The results line contains a string of printable characters
		 * ranging from space+1 to space+64.  Since space is 32, that
		 * would be 33..96.  We want to convert this back into the
		 * team numbers, which range from 0..63, so subtract the
		 * value of space (32) and one more, which was added when
		 * we wrote out the results file.
		 */
		StringToResults(results);
		
		/* Read in the tiebreaker, the sum of the scores of the
		 * two teams in the final game, if we have all the results.
		 * There shouldn't be anything there to read if we are
		 * only partially finished with the tourney, but make sure
		 * anyway.
		 */
		if ((nResults == kTotalGames) && (FGets(line, sizeof(line), fp) != NULL))
			tieBreaker = (short) atoi(line);
		fclose(fp);
	}
}									   /* ReadResultsFile */




/* Given the player's data that we found in the player file, convert
 * it into a more convenient structure we can use.  This structure
 * also holds extra fields for scoring that we don't write or read
 * from the player file.
 */
int LineToPlyrPtr(PlayerLine line0, Player *pptr)
{
	char *cp, *cp2;
	PlayerLine line;

	strncpy(line, line0, sizeof(line) - 1);
	
	/* Clear out the whole structure. */
	memset(pptr, 0, sizeof(Player));

	cp = strchr(line, ':');
	if (cp == NULL) {
	  badFmt:
		fprintf(stderr, "%s in wrong format.\n", kPlayerFile);
		return (-1);
	}
	*cp++ = 0;
	cp2 = strchr(++cp, ' ');
	if (cp2 == NULL)
		goto badFmt;
	*cp2++ = 0;
	strncpy(pptr->name, line, sizeof(PlayerName) - 1);
	strncpy(pptr->picks, cp, sizeof(PicksLine) - 1);
	
	/* Convert the string into an array of team numbers. */
	StringToPicks(pptr->picks);
	
	pptr->tieBreaker = atoi(cp2);
	return (0);
}									   /* LineToPlyrPtr */

/* eof */
