/* t64ps: 64-Team PostScript Bracket Generator.
 * Brought to you by Mike Gleason, NCEMRSoft. (mgleason@cse.unl.edu)
 * Version 1.0 (April 2, 1994)
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#define DEBUG 0

/* Note: this is different from the declaration in the ncaa package! */
#define kNumTeams 127

typedef struct Box {
	int top;
	int left;
	int bottom;
	int right;
} Box, *BoxPtr;

/* This is the area of the page (letter size) in picas. */
Box wRect = {612,0,0,792};

/* Leave this much room on the paper for margins. */
#define kPaperMarginX 36	/* In picas. */
#define kPaperMarginY 36	/* In picas. */

/* This is the amount of room to leave in the middle for the
 * fancier championship game.
 */
#define kCenterPad 100
#define kExtenderToBox 8

/* This is supposed to be the bounding box of all our drawing, but
 * the text protrudes slightly above it.
 */
Box bBox;

/* The file we send the output to.  Probably stdout. */
FILE *gOutFP;

/* The list of names to print on each 'line' in the bracket.
 * These can be empty strings.
 */
char *gTeamNames[kNumTeams];

/* We need to keep track of the size of the font we're using,
 * usually for centering purposes.
 */
int gCurFontSize;

/* The name of the "body text" font that will get the most action,
 * since it will print all the team names, as well as the headers.
 * This is changeable by the user.
 */
char *gTeamFont = "Times-Roman";

/* Size of the team font, changeable by the user. */
int gTeamFontSize = 9;

/* Same stuff, but for the Regional and Header fonts. */
char *gRegionalFont = "Helvetica-BoldOblique";
char *gHeaderFont = "Times-Roman";
int gRegionalFontSize = 36, gHeaderFontSize = 18;

/* We let the user print some title/header lines if they like.
 * By default, we don't print any headers.
 */
char *gHeader[3] = {"", "", ""};

/* Names of the regionals, in the order you supply them in the
 * data file.  Default values could be set, but they change every year
 * anyway, and also the user may not like the way we display the
 * regional names.
 */
char *gRegionals[4] = { "", "", "", "" };




/* Just a simplification to make calling fgets more suitable for
 * our purposes.  Skip blank lines and omit #comments, as well as
 * omitting the newline.
 */
static
char *FGets(char *in, size_t max, FILE * fp)
{
	char *cp;

	while (1) {
		if ((cp = fgets(in, (int) (max - 1), fp)) == NULL)
			break;
		for (; *cp; cp++) {
			if ((*cp == '\n') || (*cp == '#')) {
				*cp = '\0';
				break;
			}
		}
		if (in[0] != '\0')
			break;
	}
	return cp;
}									   /* FGets */



/* Makes a copy of a string using dynamic memory allocation. */
static
char *NewString(char *src)
{
	char *cp = malloc(strlen(src) + 1);
	if (cp == NULL) {
		fprintf(stderr, "Out of memory!\n");
		exit(1);	
	} else
		strcpy(cp, src);
	return cp;
}	/* NewString */



/* Converts a string comma list into individual copies. */
static
void CommaList(char *src, char **dsts, int maxdsts)
{
	char *cp, *cp2;
	int i;

	for (i = 0, cp2 = src; cp2 != NULL; cp2 = cp) {
		cp = strchr(cp2, ',');
		if (cp != NULL)
			*cp = 0;
		if (i < maxdsts)
			dsts[i++] = NewString(cp2);
		else
			break;
		if (cp != NULL) {
			++cp;
			while (isspace(*cp)) ++cp;
		}
	}
	while (i < maxdsts)
		dsts[i++] = NewString("");
}



/* This reads the data file.  We expect to at least read 64 team names
 * from the file (for the first round), and possibly up to 127 if the
 * whole bracket is to be filled out.
 *
 * There may be blank and comment lines, and #comments act in the
 * usual way.  There are also option lines, which are spotted by
 * the equal sign, so the user can configure the format from the
 * data file; otherwise, any line with text on it is considered
 * to be a team name line.
 */

static
void ReadDataFile(FILE *fp)
{
	char line[256];
	char *cp, *cp2, *value;
	int teamsRead;

	teamsRead = 0;
	while (FGets(line, sizeof(line), fp) != NULL) {
		if ((cp = strchr(line, '=')) != NULL) {
			/* Option line, where option = value. */
			cp2 = cp;
			
			/* Let the first token in the line be the option
			 * name.
			 */
			while (isspace(*--cp)) ;
			*++cp = '\0';
			
			/* Let the next token, the one following the =,
			 * be the value.
			 */
			cp = cp2 + 1;
			while (isspace(*cp++)) ;
			value = cp - 1;

			/* fprintf(stderr, "option <%s> = <%s>\n", line, value); */
#define EQ(a) (!strcmp(a, line))

			if (EQ("Regionals")) {
				/* gRegionals = n1, n2, n3, n4. */
				CommaList(value, gRegionals, 4);
			} else if (EQ("RegionalFont")) {
				gRegionalFont = NewString(value);
			} else if (EQ("RegionalFontSize")) {
				gRegionalFontSize = atoi(value);
			} else if (EQ("Headers")) {
				/* Header lines: h1, h2, h3 */
				CommaList(value, gHeader, 3);
			} else if (EQ("HeaderFont")) {
				gHeaderFont = NewString(value);
			} else if (EQ("HeaderFontSize")) {
				gHeaderFontSize = atoi(value);
			} else if (EQ("Font")) {
				gTeamFont = NewString(value);
			} else if (EQ("FontSize")) {
				gTeamFontSize = atoi(value);
			} else {
				fprintf(stderr,
					"Unknown option '%s.'\n",
					line
				);
			}
		} else {
			/* Team name. */
			if (teamsRead < kNumTeams) {
				gTeamNames[teamsRead] = NewString(line);
				++teamsRead;
			}
		}
	}

	if (teamsRead == 0) {
		fprintf(stderr, "\
Did not read any teams in that file.  I expect to read at least 64 lines,\n\
with each line corresponding to a team in the tournament.  After those 64,\n\
you may include the winners of the previous rounds.\n");
		fprintf(stderr, "\n\
When a proper data file is fed to this program, it will output a PostScript\n\
file with the bracket filled in with the teams.\n");
		exit(1);
	}

	/* If we didn't read in the whole tournament, which is common
	 * if you're printing the results of the thing as you go along,
	 * then set the remaining entries to empty string, so we print
	 * "nothing."
	 */
	while (teamsRead < kNumTeams) {
		gTeamNames[teamsRead] = "";
		++teamsRead;
	}
}	/* ReadDataFile */



/* Macros to draw our PS rendering of a two-bracket with optional
 * team names.  We do each game one at a time.
 */
#define DrawBox(l,r,b1,b2,t1,t2,rd,type) \
if (rd < 4) fprintf(gOutFP, "%d %d (%s) (%s) %s\n", b2,b1,t2,t1,type); \
else fprintf(gOutFP, "%d %d %d %d (%s) (%s) %s\n", l,r,b2,b1,t2,t1,type)


#define DrawRBox(l,r,b1,b2,t1,t2,rd) DrawBox(l,r,b1,b2,t1,t2,rd,"rb")
#define DrawLBox(l,r,b1,b2,t1,t2,rd) DrawBox(l,r,b1,b2,t1,t2,rd,"lb")


/* This inserts the rest of the PS utility procedures, so we can
 * use them to make the bracket easiser.
 */
static
void InsertProcedures(void)
{
	fprintf(gOutFP,
	"90 rotate 0 -612 translate\n\n"
	"/Rct {\n"
	"	/rh exch def /rw exch def /ry exch def /rx exch def\n"
	"	gsave newpath rx ry moveto rw 0 rlineto 0 rh rlineto\n"
	"	rw neg 0 rlineto closepath\n"
	"} bind def\n\n"
	);

	fprintf(gOutFP, "%s",
	"/Font { /FSize exch def findfont FSize scalefont setfont } bind def\n\n"
	"/Cntr { dup stringwidth pop -0.5 mul FSize -0.5 mul rmoveto } bind def\n\n"
	"/Black 0 def\n"
	"/White { 1.0 Black sub } bind def\n"
	"/LtGray { 0 Black eq {0.9} {0.1} ifelse } bind def\n\n"
	);

	fprintf(gOutFP, "%s",
	"% We use this as a global variable to determine which way text is\n"
	"% justified (1 is left, 2 right, 3 center), so just init it here to be safe.\n"
	"/just 1 def\n\n"
	);

	fprintf(gOutFP, "%s",
	"/midpt { add 2 div } bind def\n\n"
	"% Show str at height y1 within a certain range [x1,x2],\n"
	"% condensing to fit if necessary.  If just is set to 1,\n"
	"% left justify, else right justify.\n"
	);

	fprintf(gOutFP, "%s",
	"/sq {\n"
	"	/str exch def\n"
	"	/y1 exch def\n"
	"	/x2 exch def\n"
	"	/x1 exch def\n"
	"	x2 x1 sub\n"
	"	/wid exch def\n"
	);

	fprintf(gOutFP, "%s",
	"	str stringwidth pop\n"
	"	/swid exch def\n"
	"	gsave\n"
	"	swid wid gt {\n"
	"		x1 y1 moveto\n"
	"		wid swid div 1 scale\n"
	);

	fprintf(gOutFP, "%s",
	"	} {\n"
	"		just 1 eq {x1} {				% left justified\n"
	"			just 2 eq {\n"
	"				wid swid sub x1 add		% right justified\n"
	"			} {\n"
	"				x1 x2 midpt				% centered\n"
	"				swid 2 div sub\n"
	"			} ifelse\n"
	"		} ifelse\n"
	"		y1 moveto\n"
	"	} ifelse\n"
	"	str show\n"
	"	grestore\n"
	"} bind def\n\n"
	);

	fprintf(gOutFP, "%s",
	"% This is the fraction of the 2-bracket that is not part of the\n"
	"% tip area, i.e. if the 2-brac has a 72 pica total width, the tip\n"
	"% part would take (1 - this_fraction) times 72 picas.\n"
	);

	fprintf(gOutFP, "%s",
	"/nonTipPerc 0.9375 def\n\n"	
	"% This is the horizontal margin on the 2-brac, so if we had 72 picas\n"
	"% to write a team name, we would inset this many picas on each side\n"
	"% before writing in the team.\n"
	"/hm2 3 def\n\n"
	);

	fprintf(gOutFP, "%s",
	"% This is the number of picas we leave as a margin between a line of\n"
	"% the 2-brac and the team name text.  (We don't want to write _on_\n"
	"% the line itself, just a little above it).\n"
	"/vm2 2 def\n\n"
	);

	fprintf(gOutFP, "%s",
	"% This writes a 2-bracket, which    -----Team 1--\n"
	"% will be in this form:            /\n"
	"%                                  \\\n"
	"%                                   -----Team 2--\n"
	);

	fprintf(gOutFP,
	"/rb {\n"
	"	gsave\n"
	"	/t2 exch def\n"
	"	/t1 exch def\n"
	"	/b2 exch def\n"
	"	/b1 exch def\n"
	"	/r exch def\n"
	);

	fprintf(gOutFP,
	"	/l exch def\n"
	"	r nonTipPerc r l sub mul sub\n"
	"	/k exch def\n"
	);

	fprintf(gOutFP,
	"	newpath k b1 moveto l b1 b2 midpt lineto k b2 lineto closepath fill\n"
	"	newpath r b1 moveto k 0.5 add dup b1 lineto b2 lineto\n"
	"	r b2 lineto stroke\n"
	);

#if (!defined(DEBUG)) || (!DEBUG)
	fprintf(gOutFP, "%s",
	"	% Draw the teamnames now.\n"
	"	k hm2 add r hm2 sub 2 copy\n"
	"	b1 vm2 add t1 sq\n"
	"	b2 vm2 add t2 sq\n"
	);
#endif

	fprintf(gOutFP, "%s",
	"	grestore\n"
	"} bind def	%end rb\n\n"
	);
	
	fprintf(gOutFP, "%s",
	"% Same as 'rb' above, but teams are on the left side.\n"
	"/lb {\n"
	"	gsave\n"
	"	/t2 exch def\n"
	"	/t1 exch def\n"
	"	/b2 exch def\n"
	"	/b1 exch def\n"
	);
	
	fprintf(gOutFP,
	"	/r exch def\n"
	"	/l exch def\n"
	"	l nonTipPerc r l sub mul add\n"
	"	/k exch def\n"
	);
	
	fprintf(gOutFP,
	"	newpath k b1 moveto r b1 b2 midpt lineto k b2 lineto closepath fill\n"
	"	newpath l b1 moveto k 0.5 sub dup b1 lineto b2 lineto\n"
	"	l b2 lineto stroke\n"
	);

#if (!defined(DEBUG)) || (!DEBUG)
	fprintf(gOutFP, "%s",
	"	% Draw the teamnames now.\n"
	"	l hm2 add k hm2 sub 2 copy\n"
	"	b1 vm2 add t1 sq\n"
	"	b2 vm2 add t2 sq\n"
	);
#endif

	fprintf(gOutFP, "%s",
	"	grestore\n"
	"} bind def	%end lb\n\n"
	);

}


/* Draws a framed rectangle. */
static
void RectStroke(BoxPtr bp)
{
	fprintf(gOutFP, "%d %d %d %d Rct stroke grestore\n",
		bp->left,
		bp->bottom,
		(bp->right - bp->left),
		(bp->top - bp->bottom)
	);
}	/* RectStroke */



/* Draws a filled (and stroked) rectangle. */
static
void RectFill(BoxPtr bp)
{
	fprintf(gOutFP, "%d %d %d %d Rct gsave fill grestore stroke grestore\n",
		bp->left,
		bp->bottom,
		(bp->right - bp->left),
		(bp->top - bp->bottom)
	);
}	/* RectFill */



/* Sets the current font and size, and remembers the size since
 * we need that sometimes.
 */
static void SetFont(char *name, int size)
{
	fprintf(gOutFP, "/%s %d Font\n", name, size);
	gCurFontSize = size;
}	/* SetFont */



/* Spacing between each group of brackets. For example, each
 * bracket is separated by 2 picas, but each regional is separated
 * by 24 so we can see them more clearly.
 */
int pads[] = {2, 4, 4, 4, 16, 0, 0};

/* Draws the whole bracket, and adds the other stuff. */
static
void DrawBracket(void)
{
	/* We actually draw the bracket in two halves, a left and right
	 * half, so leftGBox is the current bounding rect of the 2-bracket
	 * on the left, and vice versa.
	 */
	Box leftGBox, rightGBox;
	
	/* Boxes for the Championship Game, the CG's title, the Champion,
	 * and the box that holds the title "Champion."
	 */
	Box cgBox, cgtBox, cBox, ctBox;
	
	int i, j, k;
	
	/* This denotes the offset into the teams array.  We mash all
	 * the rounds together in that array, so we compute where the
	 * rounds are, and use this to offset from for each game in the
	 * round.
	 */
	int tOff;
	
	int rx[8];
	int nTeamsInAHalf, m, tnum, rd;
	int y, ht, wid, wid1;
	int x1, x2, y2;
	int padx[32];
	
	/* This is the total number of picas that will be used for white
	 * space;  then we figure out how much room we have left to actually
	 * write our stuff by subtracting this number from the total amount
	 * available.
	 */
	int totalpad;
	
	/* This important array is the basis on how we decide to place
	 * the vertical location of each 2-bracket (game).  Each additional
	 * round is based on the old round, so we need to keep this around
	 * to compute the vertical locations for the other rounds.
	 */
	int tops[32];
	
	totalpad = 0;
	for (i=1; i<=32; i++) {
		/* Compute if this game is the boundary of a set of games.
		 * For example, if this is the last game in the regional,
		 * it will be evenly divisible by 8.
		 */
		for (j=1, k= -1; (i%j)==0; k++)
			j *= 2;
		padx[i-1] = k;
		/* Add the corresponding pad, if it is a last game in a set. */
		totalpad += pads[k];
	}
	
	/* Compute the height of a two-bracket. */
	ht = (bBox.top - bBox.bottom - totalpad) / 32;
	bBox.top = bBox.bottom + totalpad + (ht * 32);

	/* ...and now the width. We want the first round brackets to be
	 * slightly longer, to accomodate more information that would
	 * only appear there, such as the seed and W-L record.
	 */
	
	/* i: amount of space we have to work with on a side;
	 * j: number of rounds we have to fit in on a side, not
	 *    including the CG.
	 * We want the first round to be 7/5 the size of the normal
	 * rounds, or m/k. (m=7, k=5)
	 */
	i = (bBox.right - bBox.left - kCenterPad) / 2;
	j = 5;
	m = 7;
	k = 5;
	wid = k*i / (k*(j-1) + m);
	wid1 = i - (j-1)*wid;
	
	/* For convenience, compute now where each round's brackets
	 * will start on the x-axis.  For example, the bracket in the
	 * first round would start at rx[0] and extend to rx[1], and
	 * the next round would be from rx[1] to rx[2].
	 */
	rx[0] = bBox.left;
	rx[1] = bBox.left + wid1;
	for (i=2; i<=j; i++)
		rx[i] = bBox.left + wid1 + wid*(i-1);

	y = bBox.top;	/* Start drawing from the top and move down. */
	
	leftGBox.left = rx[0];
	leftGBox.right = rx[1];
	rightGBox.right = bBox.right + bBox.left - leftGBox.left;
	rightGBox.left = bBox.right + bBox.left - leftGBox.right;
	
	/* Have to init the tops array for use below, since below we
	 * assume it is already computed, and we use it the array to
	 * compute the next round's tops.
	 */
	for (i=0; i<32; i++) {
		tops[i] = y;
		y -= ht + pads[padx[i]];
	}
	
	/* If we have regional names, draw 'em. */
	if (gRegionals[0][0] != '\0') {
		/* Specify the regional name by writing it in the middle of it's
		 * bracket, in the background (with lt gray).
		 */
		fprintf(gOutFP,
			"\n%%Draw regional names.\n"
			"currentgray\nLtGray setgray\n"
		);
		SetFont(gRegionalFont, gRegionalFontSize);
		x1 = rx[1] + gCurFontSize/2;
		x2 = rx[4] - (gCurFontSize/2);
		y = (tops[15] + ht + tops[0]) / 2 - gCurFontSize/2;
		y2 = (tops[31] + ht + tops[16]) / 2 - gCurFontSize/2;
		fprintf(gOutFP,
			"/just 2 def\n%d %d %d (%s) sq\n%d %d %d (%s) sq\n",
			x1, x2, y, gRegionals[0],
			x1, x2, y2, gRegionals[1]
		);
	
		x2 = bBox.right - rx[1] - (gCurFontSize/2) + bBox.left;
		x1 = bBox.right - rx[4] + (gCurFontSize/2) + bBox.left;
		fprintf(gOutFP,
			"/just 1 def\n%d %d %d (%s) sq\n%d %d %d (%s) sq\nsetgray\n\n",
			x1, x2, y, gRegionals[2],
			x1, x2, y2, gRegionals[3]
		);
	}

	/* Get ready to draw team names. */
	SetFont(gTeamFont, gTeamFontSize);
	tOff = 0;
	for (rd=1, j=16; j >= 1; m += j, j /= 2, tOff += nTeamsInAHalf, ++rd) {
		nTeamsInAHalf = 4*j;
		m = nTeamsInAHalf / 2;
		fprintf(gOutFP, "\n\n%%Round %d;  %d teams.\n/just 1 def\n",
			rd, nTeamsInAHalf);

		leftGBox.left = rx[rd-1];
		leftGBox.right = rx[rd];
		rightGBox.right = bBox.right + bBox.left - leftGBox.left;
		rightGBox.left = bBox.right + bBox.left - leftGBox.right;

		if (rd < 4)
			fprintf(gOutFP, "%d {%d %d} repeat\n", j, leftGBox.left, leftGBox.right);
		for (i=0; i<j; i++) {
			tnum = 2 * i + tOff + 0;
			DrawLBox(
				leftGBox.left,
				leftGBox.right,
				tops[2 * i],
				tops[2 * i + 1],
				gTeamNames[tnum],
				gTeamNames[tnum + 1],
				rd
			);
		}

		fprintf(gOutFP, "\n/just 2 def\n");
		if (rd < 4)
			fprintf(gOutFP, "%d {%d %d} repeat\n", j, rightGBox.left, rightGBox.right);
		for (i=0; i<j; i++) {
			tnum = 2 * i + tOff + m;
			DrawRBox(
				rightGBox.left,
				rightGBox.right,
				tops[2 * i],
				tops[2 * i + 1],
				gTeamNames[tnum],
				gTeamNames[tnum + 1],
				rd
			);

			/* Re-Compute the tops for the next round. */
			tops[i] = ((tops[2 * i] + tops[2 * i + 1]) / 2);
		}
	}

	/* Height of the Championship Game box, which must hold two lines. */
	i = gCurFontSize * 2 + 10;

	y = tops[0];	/* The vertical middle of both brackets. */
	x1 = leftGBox.right;		/* The rightmost x-coord of the left side. */
	x2 = rightGBox.left;	/* The leftmost x-coord of the right side. */

	/* Draw the Championship Game box.  First we have to compute it.
	 * we are given a certain amount to work with, but we have to
	 * leave room for the extenders, so we account for that.
	 */
	cgBox.left = x1 + kExtenderToBox;
	cgBox.right = x2 - kExtenderToBox;
	cgBox.top = y + (i/2);
	cgBox.bottom = cgBox.top - i;
	fprintf(gOutFP, "\n%% Draw the CG boxes.\n");
	RectStroke(&cgBox);
	
	/* Now draw the stuff that goes in the cgBox, the two teams
	 * that made it there.  First the winner of the left side:
	 */
	fprintf(gOutFP,
		"%% Draw CG teams in the box.\n"
		"/just 3 def\n%d %d %d (%s vs.) sq\n",
		cgBox.left, cgBox.right,
		(cgBox.top + cgBox.bottom + gCurFontSize)/2,
		gTeamNames[kNumTeams - 3]
	);

	/* ...now the winner of the right side: */
	fprintf(gOutFP, "%d %d %d (%s) sq\n",
		cgBox.left, cgBox.right,
		cgBox.bottom + gCurFontSize/2, gTeamNames[kNumTeams - 2]
	);
	
	/* Now compute and draw the title box for the cgBox.  This
	 * is done by making a black rectangle, then drawing the title
	 * in white.
	 */
	cgtBox.left = cgBox.left;
	cgtBox.right = cgBox.right;
	cgtBox.bottom = cgBox.top;
	cgtBox.top = cgtBox.bottom + gCurFontSize + 6;
	RectFill(&cgtBox);

	/* Now draw the text in white, centered within it. */
	fprintf(gOutFP,
		"currentgray White setgray %d %d %d (%s) sq setgray\n"
		"\n%% Draw the champion boxes.\n",
		cgtBox.left+3,
		cgtBox.right-3,
		(cgtBox.top + cgtBox.bottom - gCurFontSize)/2,
		"Championship Game"
	);

	/* Now draw the Champion title box, like above, but instead of
	 * for the championship game, it's for the winner of that game.
	 */
	ctBox.left = cgBox.left;
	ctBox.right = cgBox.right;
	ctBox.top = cgBox.bottom - kExtenderToBox;
	ctBox.bottom = ctBox.top - (gCurFontSize + 6);
	RectFill(&ctBox);
	
	/* And now draw the title in white centered within it. */
	fprintf(gOutFP,
		"currentgray White setgray %d %d moveto (%s) Cntr show setgray\n",
		(ctBox.left + ctBox.right)/2,
		(ctBox.top + ctBox.bottom)/2,
		"Champion"
	);
	
	/* Compute and draw the Champion box. */
	cBox = ctBox;
	cBox.top = ctBox.bottom;
	cBox.top -= (ctBox.top - ctBox.bottom);
	RectStroke(&cBox);

	/* Draw the winner. */
	fprintf(gOutFP,
		"%% Draw the champion.\n"
		"%d %d %d (%s) sq\n\n",
		cBox.left,
		cBox.right,
		(cBox.top + cBox.bottom - gCurFontSize)/2,
		gTeamNames[kNumTeams - 1]
	);

	/* Draw the little "extenders," the lines connecting the
	 * cgBox to both sides of the regular bracket.
	 */
	fprintf(gOutFP,
		"%% Draw extenders.\ngsave %d %d moveto %d %d lineto\n",
		x1, y, cgBox.left, y
	);
	fprintf(gOutFP,
		"%d %d moveto %d %d lineto\n",
		x2, y, cgBox.right, y
	);

	/* Don't forget the little extender between the championship
	 * game box and the champion title box.
	 */
	fprintf(gOutFP,
		"%d %d moveto 0 %d rlineto stroke grestore\n",
		(cgBox.left+cgBox.right)/2,
		cgBox.bottom,
		ctBox.top - cgBox.bottom
	);
	
	/* Write up to 3 header lines. */
	for (i=j=0; i<3; i++) {
		if (gHeader[i][0] != '\0') {
			if (++j == 1) {
				fprintf(gOutFP, "\n%%Draw header line(s).\n");
				SetFont(gHeaderFont, gHeaderFontSize);
			}
			fprintf(gOutFP,
				"%d %d moveto (%s) Cntr show\n",
				(bBox.left + bBox.right)/2,
				bBox.top - gHeaderFontSize/2 - 
					(i * (gHeaderFontSize * 5 / 4)),
				gHeader[i]
			);
		}
	}

#if (defined(DEBUG)) && (DEBUG)
	RectStroke(&bBox);
#endif
}	/* DrawBracket */



#define kDefaultInput "tourney.t64"
#define kDefaultOutput "tourney.ps"

void main(void)
{
#ifdef THINK_C
	FILE *fp;
	
	fp = fopen(kDefaultInput, "r");
	if (fp == NULL) {
		perror(kDefaultInput);
		exit(1);
	}
	ReadDataFile(fp);
	fclose(fp);
	gOutFP = fopen(kDefaultOutput, "w");
	if (gOutFP == NULL) {
		perror(kDefaultOutput);
		exit(1);
	}
#else	/* Normal */
	ReadDataFile(stdin);
	gOutFP = stdout;
#endif

	bBox.left = kPaperMarginX;
	bBox.right = (wRect.right - wRect.left) - kPaperMarginX;
	bBox.bottom = kPaperMarginY;
	bBox.top = (wRect.top - wRect.bottom) - kPaperMarginY;

	/* Write the Document Structure Comments. */
	fprintf(gOutFP,  "%%!PS-Adobe-3.0\n%%%%BoundingBox: %d %d %d %d\n",
		bBox.left,
		bBox.bottom,
		bBox.right,
		bBox.top + 20
	);

	fprintf(gOutFP,
		"%%%%Copyright: (Copyright 1994, by Mike Gleason, NCEMRSoft.)\n"
		"%%%%Creator: t64ps\n"
		"%%%%Title: (64 Team Bracket)\n"
	);

	fprintf(gOutFP,
		"%%%%Pages: 1\n"
		"%%%%Orientation: Landscape\n"
		"%%%%DocumentNeededResources: font %s %s %s\n"
		"%%%%EndComments\n\n",
		gTeamFont, gRegionalFont, gHeaderFont
	);

	InsertProcedures();
	DrawBracket();
	
	fprintf(gOutFP, "\nshowpage\n%%%%EOF\n");
	if (gOutFP != stdout)
		fclose(gOutFP);
	exit(0);
}									   /* main */

/* eof */
