/*
 * ************************************
 * * Program to control LORAN-C radio *
 * ************************************
 */
#include <stdio.h>
#include <ctype.h>
#include <bios.h>
#include <math.h>
#include <conio.h>

/*
 * Parameters
 */
#define PORT 0x0300				/* controller port address */

/*
 * Timing generator  definitions
 */
#define TGC PORT+0				/* stc control port (r/w) */
#define TGD PORT+1				/* stc data port (r/w) */
/*
 * Analog/digital converter
 */
#define ADC PORT+2				/* adc buffer (r)/address (w) */
#define ADCGO PORT+3			/* adc status (r)/adc start (w) */
	#define START 0x01			/* converter start bit (w) */
	#define BUSY 0x01			/* converter busy bit (r) */
	#define DONE 0x80			/* conversion done bit (r) */
/*
 * Digital/analog converter definitions
 * Note: output voltage is inverted from buffer
 */
#define DACA PORT+4				/* vco (dac a) buffer (w) */
#define DACB PORT+5				/* agc (dac b) buffer (w) */
/*
 * Code generator definitions
 * Note: bits are shifted out from the lsb first
 */
#define CODE PORT+6				/* pulse-code buffer (w) */
	#define MPCA 0xCA			/* LORAN-C master pulse code group a */
	#define MPCB 0x9F			/* LORAN-C master pulse code group b */
	#define SPCA 0xF9			/* LORAN-C slave pulse code group a */
	#define SPCB 0xAC			/* LORAN-C slave pulse code group b */
/*
 * Mode register definitions
 */
#define PAR PORT+7				/* parameter buffer (w) */
	#define GAIN 0x03			/* integrator gain mask */
	/*
	 * time constant values
	 *
	 * 0	1.000 ms
	 * 1	0.264 ms
	 * 2	0.036 ms
	 * 3	short caps
	 */
	#define GATE 0x0C			/* gate source mask */
	/*
	 * gate source values
	 *
	 * 4	always open
	 * 8	group repeition interval (GRI)
	 * c	puldse code interval (PCI)
	 * f	strobe (STB)
	 */
	#define MSB 0x10			/* load high-order dac bits */
	#define IEN 0x20			/* enable interrupt bit */
	#define EN5 0x40			/* enable counter 5 bit */
	#define ENG 0x80			/* enable gri bit */

/*
 * STC commands
 */
/* argument sssss = counter numbers 5-1 */
#define LOADDP 0x00				/* load data pointer */
	/* argument ee = element (all groups except ggg = 000 or 111) */
	#define MODEREG 0x00		/* mode register */
	#define LOADREG 0x08		/* load register */
	#define HOLDREG 0x10		/* hold register */
	#define HOLDINC 0x18		/* hold register (hold cycle increm) */
	/* argument ee = element (group ggg = 111) */
	#define ALARM1 0x07			/* alarm register 1 */
	#define ALARM2 0x0F			/* alarm register 2 */
	#define MASTER 0x17			/* master mode register */
	#define STATUS 0x1F			/* status register */
#define ARM 0x20				/* arm counters */
#define LOAD 0x40				/* load counters */
#define LOADARM 0x60			/* load and arm counters */
#define DISSAVE 0x80			/* disarm and save counters */
#define SAVE 0xA0				/* save counters */
#define DISARM 0xC0				/* disarm counters */
/* argument nnn = counter number */
#define SETTOG 0xE8				/* set toggle output HIGH for counter */
#define CLRTOG 0xE0				/* set toggle output LOW for counter */
#define STEP 0xF0				/* step counter */
/* argument eeggg, where ee = element, ggg - counter group */
/* no arguments */
#define ENABDPS 0xE0			/* enable data pointer sequencing */
#define ENABFOUT 0xE6			/* enable fout */
#define ENAB8 0xE7				/* enable 8-bit data bus */
#define DSABDPS 0xE8			/* disable data pointer sequencing */
#define ENAB16 0xEF				/* enable 16-bit data bus */
#define DSABFOUT 0xEE			/* disable fout */
#define ENABPFW 0xF8			/* enable prefetch for write */
#define DSABPFW 0xF9			/* disable prefetch for write */
#define RESET 0xFF				/* master reset */

/*
 * STC setup. Note GTI 9960 for testing and PCI 500 us.
 */
unsigned int init[] = {
	0x0162,   12,   13,			/* counter 1 (P0) */
	0x0262, 9160,  800,			/* counter 2 (GRI) */
	0x8162, 2500, 2500,			/* counter 3 (PCX) */
	0xc162,  145,   10,			/* counter 4 (STB) */
	0x8162,   25,   25 			/* counter 5 (OUT) */
	};

/*
 * Main program
 */
main() {
	int i, j, test, codesw = 1;
	double sig, sag;
	double sigi, sigq, agc, sigoffset, agcoffset;
	double agcthresh, vcothresh;
	unsigned int arg, gri, pci, mode, codea, codeb, vco;
	unsigned int status, temp;
	unsigned char kbd, key;

/*
 * Initialize
 */
	outp(TGC, RESET); outp(TGC, LOAD+0x1f); /* reset chip */
	outp(TGC, LOADDP+MASTER); outp(TGD, 0xf0); outp(TGD, 0x8a);
	printf("Alarm/Master %02x %02x %02x\n", inp(TGD),
		inp(TGD), inp(TGD));	/* readback check */
	outp(TGC, LOADDP+1);		/* element cycle */
	for (i = 0; i < 5*3; i++) {
		outp(TGD, init[i]); outp(TGD, init[i]>>8);
		}
	printf("Status %02x\n", inp(TGC));
	printf("Counters\n");		/* readback check */
	outp(TGC, LOADDP+1);		/* element cycle */
	for (i = 1; i < 6; i++) {
		printf("%2i", i);
		for (j=1; j<7; j++) printf(" %02x", inp(TGD));
		printf("\n");
		}
	outp(TGC, LOADARM+0x1f);	/* let the good times roll */

	gri = 9960; pci = 2500;
	codea = MPCA; codeb = MPCB; mode = 0x06;
	sigi = 0; sigq = 0; vco = 0; agc = 0;
	sigoffset = -154; agcoffset = -128;
	agcthresh = 48; vcothresh = 87;

	while (1) {
		printf("Test number: "); scanf("%i", &test);
		switch (test) {
		/*
		 * Test ADC
		 */
		case 1:
			printf("Read ADC channel: ");
			scanf("%i", &arg);
			outp(ADC, arg); outp(ADCGO, START);
			while ((inp(ADCGO)&DONE) == 0);
			printf("Channel %i  value %i\n", arg, inp(ADC));
			break;

		case 2:
			printf("Testing ADC\n");
			while (1) {
				printf("Channel: "); scanf("%i", &i);
				for (arg = 0;; arg++) outp(ADC, arg);
					outp(ADCGO, START);
				while ((inp(ADCGO)&DONE) == 0);
					printf("ADC channel %i = %02x\n", i, inp(ADC));
			}
			break;

		case 3:
			printf("Testing ADC\n");
			printf("Channel 0  1  2  3  4  5  6  7\n");
			while (1) {
				printf("\r       ");
				for (arg = 0; arg < 8; arg++) {
					outp(ADC, arg); outp(ADCGO, START);
					while ((inp(ADCGO)&DONE) == 0);
					printf(" %02x", inp(ADC));
				}
			}
			break;

		case 4:
			printf("Testing DACA (ramp)\n");
			while (1) for (arg = 0; arg < 4095; arg++) {
				outp(PAR, 0); outp(DACA, arg);
				outp(PAR, MSB); outp(DACA, arg >> 8);
				}
			break;

		case 5:
			printf("Testing DACB (ramp)\n");
			while (1) for (arg = 0; arg < 4095; arg++) {
				outp(PAR, 0); outp(DACB, arg);
				outp(PAR, MSB); outp(DACB, arg >> 8);
				}
			break;

		case 6:
			printf("CodeA/B (hex): ");
			scanf("%x %x", &codea, &codeb); outp(CODE, codea);
			break;

		case 7:
			printf("Parameter (hex): ");
			scanf("%x", &mode); outp(PAR, mode);
			break;

		case 8:
			printf("DACA (vco): ");
			scanf("%i", &vco);
			outp(PAR, 0); outp(DACA, vco);
			outp(PAR, MSB); outp(DACA, vco >> 8);
			break;

		case 9:
			printf("DACB (agc): ");
			scanf("%i", &vco);
			outp(PAR, 0); outp(DACB, vco);
			outp(PAR, MSB); outp(DACB, vco >> 8);
			break;

		case 10:
			printf("AGC offset (%4.1lf): ", agcoffset);
			scanf("%lf", &agcoffset);
			break;

		case 11:
			printf("VCO threshold (%4.1lf): ", vcothresh);
			scanf("%lf", &vcothresh);
			break;

		case 12:
			printf("AGC threshold (%4.1lf): ", agcthresh);
			scanf("%lf", &agcthresh);
			break;

/*
 * Main loop
 */
		case 0:

			kbd = 0; key = 0;
			printf("Generating LORAN-C master codes\n");
			while (1) {
				if (kbhit() != 0) {
					kbd = getche();
					switch (kbd) {

						case '+':
							switch (key) {
								case 't':
									vcothresh++;
									break;
								case 'c':
									agcthresh++;
									break;
								default:
									vco++;
								}
							break;

															case '-':
							switch (key) {
								case 't':
									vcothresh--;
									break;
								case 'c':
									agcthresh--;
									break;
								default:
									vco--;
								}
							break;

						case '0':
							switch (key) {
								case 't':
									vcothresh = 128;
									break;
								case 'c':
									agcthresh = 128;
									break;
								default:
									vco = 0;
								}
							break;

						case 'o':
							mode = 0x02;
							key = kbd;
							break;

						case 'a':
							mode = 0x06;
							key = kbd;
							break;

						case 't':
							mode = 0x0a;
							key = kbd;
							break;

						case 'c':
							key = kbd;

						case 'x':
							codesw = -codesw;
							break;

						case '[':
							pci += 10;
							break;

						case ']':
							pci -= 10;
							break;
						}
					}
				if (codesw > 0) {
					outp(TGC, LOADDP+0x0a);
					temp = gri-800+vco;
					outp(TGD, temp); outp(TGD, temp>>8);
					sig = sigi+sigoffset;
					sag = sigq+sigoffset;
					sag = sqrt(sig*sig+sag*sag)/2;
					outp(DACA, (int)(vcothresh+(sig-sigoffset)));
					if (kbd == 'z') {
						outp(DACB, 0);
						sig = sigi+sigq-sigoffset; sigoffset += sig/8;
						sag = agc-agcoffset; agcoffset += sag/4;
						if (sig*sig+sag*sag < 3) kbd = 0;
						}
					else {
						outp(DACB, (int)(agcthresh+(agc-agcoffset)/2));
						}
					if (kbd == 'q') break;
					sigi = 0; sigq = 0; agc = 0;
					outp(CODE, codea);
					}
				else {
					outp(TGC, LOADDP+0x0a);
					temp = gri-800;
					outp(TGD, temp); outp(TGD, temp>>8);
					outp(TGC, LOADDP+0x0b);
					outp(TGD, pci); outp(TGD, pci>>8);
					temp = 5000-pci;
					outp(TGD, temp); outp(TGD, temp>>8);
					outp(CODE, codeb);
					}
				codesw = -codesw;
				outp(PAR, ENG | mode); outp(ADC, 0); /* i */
				while ((inp(ADCGO)&DONE) == 0);
				sigi += inp(ADC)&0xff;
				outp(PAR, mode);
				outp(ADC, 1); outp(ADCGO, START); /* q */
				while ((inp(ADCGO)&DONE) == 0);
				sigq += inp(ADC)&0xff;
				outp(ADC, 2); outp(ADCGO, START); /* agc */
				while ((inp(ADCGO)&DONE) == 0);
				agc += inp(ADC)&0xff;
				}
			printf("sigi %4.1lf  sigq %4.1lf  agc %4.1lf  sig %4.1lf  sag %4.1lf\n",
				sigi, sigq, agc, sig, sag);
			printf("sigoffset %4.1lf  agcoffset %4.1lf  agcthresh %4.1lf  vcothresh %4.1lf\n",
				sigoffset, agcoffset, agcthresh, vcothresh);
			printf("Status %02x\n", inp(TGC));
			printf("Counters\n");		/* readback check */
			outp(TGC, LOADDP+1);		/* element cycle */
			for (i = 1; i < 6; i++) {
				printf("%2i", i);
				for (j=1; j<7; j++) printf(" %02x", inp(TGD));
				printf("\n");
				}
			break;
		}
	}
}

/* end program */
