/*
 * NTP simulator
 *
 * This program reads sample data from a history file and simulates the
 * behavior of the NTP filter, selection and local-clock algorithms.
 *
 * General assertions
 *
 * 1	An update from a peer with synchronization distance greater than
 *	1 s is ignored.
 *
 * 2	Only those sample aged less than MAXSEC in the clock filter are
 *	used for offset calculations. Dispersion calculations use all
 *	samples.
 *
 * 3	The system poll interval is not reset to minpoll upon source
 *	peer change.
 *
 * 4	The system clock is updated at fixed poll intervals, or at the
 *	first valid peer update after that.
 *
 * 5	All dispersions are calculated using RMS, rather than absolute
 *	differences.
 *
 * Assertions for the clock discipline algorithm
 *
 * 1	The clock discipline loop is stable over a continous update
 *	interval range from 16 s to an indefinite upper limit, but in
 *	no case less than 16384 s. For ordinary purposes, the minimum
 *	poll interval is MINPOLL and maximum is MAXPOLL s).
 *
 * 2	The minimum of the Allan variance characteristic is assumed at
 *	maxpll. A phase-lock loop (PLL) is used with update intervals
 *	less than maxpll. A frequency-lock loop (FLL) is used with
 *	update intervals greater than maxpll.
 *
 * 3	The time discipline is linear over the offset range of MAXPHASE,
 *	with step adjustments outside that range; however, once a step
 *	adjustment has been made, the discipline is linear over an
 *	indefinate range, until the offset has settled to less than
 *	MAXPHASE.
 *
 * 4	The loop has a capture range of at least MAXFREQ (500 ppm) for
 *	update intervals of 64 s and less. In cases greater than about
 *	250 ppm, a step change may occur, following which the offset may
 *	exceed MAXPHASE until the loop converges.
 *
 * 5	To allow for erratic behavior of radio clocks in the vicinity of
 *	a leap second, a time step adjustment is not performed until
 *	after a watchdog interval of MAXWATCH during which all updates
 *	are outside the linear range.
 *
 * 6	An update from a peer with dispersion above MAXEPSIL does not
 *	affect the clock frequency (dispersion clamp).
 *
 * 7	If an update is greater than SGATE times the average of the last
 *	AMAX updates, the update is ignored (spike detector).
 *
 * 8	The poll change counter is updated as a function of the latest
 *	phase error compared to the average of the most recent AMAX (4)
 *	samples. If greater than PGATE, the poll interval counter is
 *	changed as described in the comments in the localclock() routine
 *	 below.
 *
 * 9	Distance and dispersion are computed separately and combined
 *	only when the final system distance is determined.
 */
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <float.h>
#include <stdlib.h>

#define TRUE 1			/* pository */
#define FALSE 0			/* negatory */
#define PI 3.1415926535		/* definitory */
#define SQRT(x) (sqrt(x))	/* square root function */
#define SQUARE(x) ((x) * (x))	/* square function */
#define DIFF(x, y) (SQUARE((x) - (y))) /* square differences function */

/*
 * Sizes and limits of things
 */
#define NMAX 40			/* max number of clocks */
#define FMAX 8			/* max clock filter size */
#define AMAX 4			/* max stability register size */
#define EMAX 4			/* max noise register size */
#define PMAX 10			/* max source addresses to match */
#define IMAX 15			/* max poll count */
#define MAXCLOCK 20		/* max candidate clocks */
#define MINCLOCK 3		/* min survivor clocks */

/*
 * Time interval values
 */
#define DAY 86400.		/* one day of seconds */
#define MAXSEC 1600.		/* max sample age */
#define MAXWATCH 900.		/* watchdog interval  */
#define MINPOLL 6		/* log2 minimum poll interval */
#define MAXPOLL 10		/* log2 maximum poll interval */
#define MAXPLL 11		/* log2 maximum PLL interval */

/*
 * Architecture limits
 */
#define MAXPHASE .128		/* max phase error */
#define MAXFREQ 500e-6		/* max frequency error (open loop) */
#define MAXDISP 16.		/* max dispersion */
#define MAXEPSIL .128		/* max total dispersion */
#define MAXDIST 1.		/* max synch distance */
#define LIMIT 30		/* poll-adjust threshold */

/*
 * Simulation environment definitions
 */
#define RHO 40e-6		/* local clock precision */
#define PHI 15e-6		/* max frequency error (closed loop) */
#define ROOTDELAY 0		/* root delay */
#define ROOTDISP .002		/* root error bound */

/*
 * Noise gates
 */
#define PGATE 1.		/* poll-adjust gate */
#define SGATE 10.		/* spike gate */

/*
 * Weight factors
 */
#define FILTER .5		/* clock filter weight */
#define SELECT .75		/* clock select weight */

/*
 * Clock discipline states
 */
#define S_NSET	0		/* clock never set */
#define S_HOLD	1		/* disable error checks */
#define S_SYNC	2		/* clock synchronized */

/*
 * Statistics structure
 */
struct stats {
	double d0;		/* sample count */
	double d1;		/* latest sample */
	double d1sum;		/* sample integral */
	double d2sum;		/* sample integral squares */
	double d3;		/* sample absolute maximum */
	double begin;		/* time of first sample */
	double utc;		/* time of latest sample */
} stats;			/* system statistics counters */

/*
 * Peer state structure
 */
struct peer {
	char src[20];		/* IP source address */
	char dst[20];		/* IP destination address */
	int cd;			/* peer id */
	int st;			/* stratum */

	double utc;		/* last peer update time */
	double poll;		/* last poll message time */

	double filtp[FMAX];	/* filter offset samples */
	double fildp[FMAX];	/* filter delay samples */
	double filer[FMAX];	/* filter error samples */
	double tp;		/* offset */
	double dp;		/* delay */
	double er;		/* error bound */
	double ep;		/* dispersion */
	int reach;		/* reachability shift register */
	char falsetick;		/* falseticker switch */
	char used;		/* sample used switch */
	struct stats stats;	/* peer statistics counters */
};

/*
 * History register structure
 */
struct allan {
	double tp;		/* time offset sample */
	double tstamp;		/* time offset sample time */
};

/*
 * Function declarations
 */
void packet(void);		/* packet procedure */
void update(void);		/* clock update procedure */
void reachreg(void);		/* update reachability status */
void localclock(struct peer *, double, double);	/* clock discipline */
void filter(struct peer *, double, double, double); /* clock filter */
void dts(void);			/* intersection algorithm */
void select(void);		/* selection algorithm */
double combine(void);		/* combining algorithm */
double distance(struct peer *); /* compute synch distance */
void clear(struct peer *);	/* clear peer */
void clrstat(struct stats *);	/* clear statistics */
void stat(struct stats *, double); /* update statistics */
void display(void);		/* display statistics */
void reset(void);		/* reset system */
void allan_var(double, double);	/* Allan variance */
double gauss(double);		/* noise generator */

/*
 * Input variables
 */
long day;			/* modified Julian day */
long firstday;			/* first day of dataset */
long days = 30;			/* days of simulation */
double sec;			/* seconds past midnight UTC */
char src[20], dst[20];		/* IP addresses */
double offset;			/* clock offset */
double delay;			/* roundtrip delay */
double disp;			/* dispersion */

/*
 * State variables
 */
double tstamp;			/* time at current update */
double phase[AMAX];		/* phase samples */
double noise[EMAX];		/* noise samples */
double xi;			/* select dispersion */
double lclock = 0;		/* clock time */
double freq = 0;		/* clock frequency (simulated) */
double lasttime;		/* last clock update time */
double tau;			/* time constant */
double mu;			/* interval since last clock update */
double utc;			/* last peer update time */
int poll;			/* log2 poll interval (s) */
int tcnt;			/* poll interval counter */
int state;			/* clock discipline state */
struct allan allan[AMAX];	/* Allan variance sample register */
double var[AMAX];		/* Allan variance vector */

/*
 * Predicted state variables
 */
double tp = 0;			/* offset */
double dp = 0;			/* delay */
double er = 0;			/* error bound */
double ep = 0;			/* dispersion */
double rp = 0;			/* frequency */

/*
 * Local clock tuning parameters. The ratio Kf / Kg^2 must be 4.
 */
double Kf = 65536.;		/* PLL frequency weight (2^16) */
double Kg = 64.;		/* PLL phase weight (2^6) */
double G = .25;			/* frequency average (2^-3*/

/*
 * Simulation variables
 */
struct peer *peers[NMAX];	/* where the peers are */
struct peer *plist[NMAX];	/* where the truechimes are */
struct peer *source;		/* clock source */
double bot, top;		/* confidence interval limits */
double lastupd;			/* last clock update time */
int npeers;			/* number of peer clocks */
int nclock;			/* number of survivor clocks */

/*
 * Command-line options
 */
char debug = FALSE;		/* -d debug format */
FILE *fp_in = NULL;		/* -f input data file handles */
int minpoll = MINPOLL;		/* -m mimimum poll interval */
int maxpoll = MAXPOLL;		/* -M maximum poll interval */
int maxpll = MAXPLL;		/* -P maximum pll interval */

/*
 * Performance indicators
 */
int pollcnt[IMAX];		/* poll intervals, by interval */
int spikcnt;			/* spikes */
int stepcnt;			/* offset steps */
int pllcnt;			/* PLL updates */
int fllcnt;			/* FLL updates */

/*
 * Main program
 *
 * This routine reads the data files produced by the xntpd daemon to
 * simulate actual timekeeping performance.
 *
 * "rawstats" file format (produced by xntpd filegen rawstats). Time is
 * in NTP timestamp units (seconds and fractions past 1 January 1970)
 *
 *  MJD   UTC sec     Source    Destination	      T1
 *		  T2		    T3		      T4
 * ----------------------------------------------------------
 * 50290 61052.715 192.67.12.101 128.4.1.20 3047389026.431722
 *	3047389028.800547 3047389052.580259 3047389052.714514
 * 50290 61055.437 128.4.1.2 128.4.1.20 3047389055.431396
 *	3047389055.466066 3047389055.467287 3047389055.436514
 * 50290 61065.640 129.132.2.21 128.4.1.20 3047389065.431874
 *	3047389065.606788 3047389065.608274 3047389065.639677
 * 50290 61085.559 192.36.143.150 128.4.1.20 3047388991.428797
 *	3047388991.652333 3047389085.506854 3047389085.558655
 *
 * "loopstats" file format (produced by xntpd filegen rawstats). Offset
 * is in seconds, frequency is in PPM.
 *
 *  MJD  UTC sec Offset    Freq  Poll
 * ----------------------------------
 * 50430 99.695 -0.000852 39.9785 6
 * 50430 163.690 0.000223 39.9922 6
 * 50430 270.374 0.000004 39.9926 6
 * 50430 334.380 -0.000006 39.9922 6
 * 50430 355.696 0.000174 39.9957 6
 */
int
main(argc, argcv)
	int argc;		/* number of command-line arguments */
	char **argcv;		/* vector of command-line argument */
{
	double t1, t2, t3, t4;	/* file data */
	int yuckyday;		/* current yuckyday */
	double dtemp;		/* double temps */
	int eof;		/* end-of-file switch */
	unsigned long srcadr;	/* source address */
	unsigned long dstadr;	/* destination address */
	int naddr = 0;		/* number of addresses to filter */
	int a, b, c, d;		/* used to assemble IP address */
	int i;			/* int temp */

	/*
	 * Command-line options
	 */
	int insw = 0;		/* -i input file format */
	int lansw = FALSE;	/* -l ignore LAN peers */
	unsigned long restrict[50]; /* -r address filter */
	double unif = 0;	/* -p phase noise parameter */
	double walk = 0;	/* -w random-walk frequency parameter */

#ifndef _MSC_VER		/* uuuuuugly MSDOS */
	int temp;		/* int temp */
	extern char *optarg;	/* pointer to option string */
	extern int opterr;	/* error message enable */
	extern int optind;	/* argcv index of next argument */

	/*
	 * Process command-line options
	 */
	while ((temp = getopt(argc, argcv, "dD:f:F:i:lm:M:p:P:r:T:w:")) !=
		    -1) 	    {
		switch (temp) {

		/*
		 * Select debug format.
		 */
		case 'd':
			debug = TRUE;
			break;

		/*
		 * Set days of simulation run.
		 */
		case 'D':
			sscanf(optarg, "%lu", &days);
			break;

		/*
		 * Open input data file.
		 */
		case 'f':
			if ((fp_in = fopen (optarg, "r")) == NULL) {
				printf("Input file %s not found\n",
				    optarg);
				return(-1);
			}
			break;

		/*
		 * Set initial frequency offset.
		 */
		case 'F':
			sscanf(optarg, "%lf", &rp);
			break;
			
		/*
		 * Select input file format
		 *
		 * 0 use rawstats format
		 * 1 use loopstats format
		 * 2 use synthetic data
		 * 3 use synthetic data; output for Allan analysis
		 */
		case 'i':
			sscanf(optarg, "%u%", &insw);
			break;

		/*
		 * Set to ignore peers on the same LAN.
		 */
		case 'l':
			lansw = TRUE;
			break;

		/*
		 * Select minimum poll interval, default MINPOLL.
		 */
		case 'm':
			sscanf(optarg, "%u", &minpoll);
			break;

		/*
		 * Select maximum poll interval, default MAXPOLL.
		 */
		case 'M':
			sscanf(optarg, "%u", &maxpoll);
			break;

		/*
		 * Set phase noise parameter, default 0, typical value
		 * 1e-3.
		 */
		case 'p':
			sscanf(optarg, "%lf", &unif);
			break;

		/*
		 * Select maximum pll interval, default MAXPLL.
		 */
		case 'P':
			sscanf(optarg, "%u", &maxpll);
			break;

		/*
		 * Restrict host address match.
		 */
		case 'r':
			sscanf(optarg, "%u.%u.%u.%u", &a, &b, &c, &d);
			restrict[naddr] = (((a * 256) + b) * 256 + c) *
			    256 + d;
			naddr++;;
			break;

		/*
		 * Set initial time offset.
		 */
		case 'T':
			sscanf(optarg, "%lf", &lclock);
			break;

		/*
		 * Set random-walk frequency parameter, default 0,
		 * typical value 1e-8.
		 */
		case 'w':
			sscanf(optarg, "%lf", &walk);
			break;

		default:
			break;
		}
	}
#else
	int optind = 1;
#endif /* MSDOS */

	/*
	 * Process command-line arguments.
	 */
	if (argc > optind) {
		if ((fp_in = fopen (argcv[optind], "r")) == NULL) {
			printf("Input file %s not found\n",
			    argcv[optind]);
			return(-1);
		}
	}
	if (fp_in == NULL && insw < 2) {
		printf("No input data file\n");
		return(-1);
	}

	/*
	 * Main loop
	 *
	 * The program reads samples from the data file, simulates the
	 * operation of a NTP client, and writes state variables to the
	 * standard output. It steams through the data file until the
	 * end-of-file is found.
	 */
	yuckyday = 0;
	reset();
	eof = TRUE;
	while (eof) {
		switch (insw) {

			/*
			 * Rawstats - evaluate the clock filter,
			 * selection, clustering and local clock
			 * discipline response to network delay
			 * variations. It is most useful if the network
			 * paths are between stratum-1 peers.
			 */
			case 0:
			if (fscanf(fp_in,
			    "%li %lf %s %s %lf %lf %lf %lf", &day, &sec,
			    src, dst, &t1, &t2, &t3, &t4) == EOF) {
				eof = FALSE;
				continue;
			}

			/*
			 * Always discard messages to the null local
			 * address. If enabled, discard messages from
			 * peers on the same LAN and from other than
			 * selected peers. If directed, discard all
			 * peers not in the restrict list.
			 */
			sscanf(dst, "%u.%u.%u.%u", &a, &b, &c, &d);
			dstadr = (((a * 256) + b) * 256 + c) * 256 + d;
			sscanf(src, "%u.%u.%u.%u", &a, &b, &c, &d);
			srcadr = (((a * 256) + b) * 256 + c) * 256 + d;
			if (dstadr == 0)
				continue;
			if (lansw) {
				if (dstadr < 0x80000000 && ((srcadr ^
				    dstadr) & 0xff000000) == 0)
					continue;
				else if (dstadr < 0xc0000000 && ((srcadr
				    ^ dstadr) & 0xffff0000) == 0)
					continue;
				else if (((srcadr ^ dstadr) &
				    0xffffff00) == 0)
					continue;
			}
			if (naddr > 0) {
				for (i = 0; i < naddr; i++)
					if (restrict[i] == srcadr)
						break;
				if (i >= naddr)
					continue;
			}

			/*
			 * Compute clock offset, roundtrip delay and
			 * peer dispersion from rawstats data. Assume
			 * fixed frequency.
			 */
			offset = ((t2 - t1) + (t3 - t4)) / 2;
			delay = (t4 - t1) - (t3 - t2);
			disp = PHI * (t4 - t1);
			dtemp = gauss(unif);
			offset += dtemp * dtemp;
			dtemp = gauss(walk);
			freq += dtemp * dtemp;
			break;

			/*
			 * Loopstats - evaluate the local clock
			 * discipline response with respect to frequency
			 * stability.
			 */
			case 1:
			if (fscanf(fp_in, "%li %lf %lf %lf %lf", &day,
			    &sec, &t1, &t2, &t3) == EOF) {
				eof = FALSE;
				continue;
			}

			/*
			 * Compute clock offset and frequency from
			 * loopstats data. Compute dispersion from
			 * offset differences. Assume fixed roundtrip
			 * delay.
			 */
			dtemp = offset;
			offset = t1 + gauss(unif);
			disp = fabs(offset - dtemp);
			delay = ROOTDELAY;
			freq = t2 / 1e6;
			freq += gauss(walk);
			if (firstday == 0)
				rp = -freq;
			break;

			/*
			 * Yuckystats - test the response of the system
			 * to artificial noise.
			 */
			default:
			sec += 64.;
			if (sec >= 86400.) {
				yuckyday++;
				sec -= 86400.;
			}
			day = yuckyday;
			dtemp = offset;
			offset = gauss(unif);
			disp = fabs(offset - dtemp);
			delay = ROOTDELAY;
			freq += gauss(walk);
			if (insw == 2)
				break;

			/*
			 * Validation test. Run this with the following
			 * command line:
			 *
			 * ntp -i3 -D150 -p5e-6 -w1e-11
			 *
			 * to duplicate the experimentally determined
			 * Allan variance intercept of .05 ppm at 150 s
			 * for an uncompensated SPARC IPC. The phase
			 * characteristic scales directly as the phase
			 * parameter, while the frequency characteristic
			 * scales directly as the frequency parameter.
			 */
			t1 += freq * 64.;
			lclock = offset + t1;
			if (lclock >= 1.) {
				lclock -= 1.;
				sec++;
			} else if (lclock < 0.) {
				lclock += 1.;
				sec--;
			}
			printf("%6.0lf %6.0lf %5u\n", day * 86400. +
			    sec, lclock * 1e6, nclock++);
			continue;
		}

		/*
		 * Simulate poll-timer interrupt.
		 */
		if (firstday == 0)
			firstday = day;
		day -= firstday;
		if (day > days)
			break;
		dtemp = day * DAY + sec;
		if (lasttime == 0) {
			lasttime = dtemp;
			lastupd = dtemp;
			utc = dtemp;
		}
		if (dtemp - lastupd >= (1 << poll)) {
			tstamp = lastupd + (1 << poll);
			pollcnt[poll]++;
			reachreg();
			update();
			lastupd = tstamp;
		}

		/*
		 * Simulate packet arrival.
		 */
		tstamp = dtemp;
		reachreg();
		packet();
	}
	fclose(fp_in);
	if (debug)
		display();
	return(0);
}

/*
 * Packet procedure
 *
 * This procedure is called for each arriving packet. It matches the
 * packet addresses to existing associations or mobilizes a new one, if
 * necessary. Then, it updates the clock filter and associated peer
 * variables.
 */
void
packet()
{
	struct peer *pp;	/* peer structure pointer */
	int i;			/* int temp */
	double dtemp;		/* double temp */

	/*
	 * Find association; mobilize a new one if necessary.
	 */
	for (i = 0; i < npeers; i++) {
		pp = peers[i];
		if (strcmp(src, pp->src) == 0 &&
		    strcmp(dst, pp->dst) == 0)
			break;
	}
	if (i >= npeers) {
		pp = malloc(sizeof(struct peer));
		peers[i] = pp;
		clear(pp);
		clrstat(&pp->stats);
		strcpy(pp->src, src);
		strcpy(pp->dst, dst);
		pp->cd = i;
		pp->st = 1;
		pp->utc = tstamp;
		pp->poll = tstamp;
		npeers++;
	}

	/*
	 * The update is tossed if a previouis one has already been
	 * processed during the current peer update interval. This is to
	 * allow simulated poll intervals greater than the actual data
	 * intervals.
	 */
	if ((pp->reach & 0xff) == 0) {
		if (debug)
			printf("%2li %9.3lf %-15s %2d reachable %.0lf\n",
			    day, sec, pp->src, nclock, tstamp -
			    pp->utc);
		pp->utc = tstamp;
		pp->poll = tstamp;
	} else if ((pp->reach & 0x1) != 0) {
		return;
	}
	pp->reach |= 0x1;

	/*
	 * Update the local clock and current time offset, taking into
	 * account adjustments for the exponential decay of the time
	 * offset.
	 */
	dtemp = tstamp - utc;
	utc = tstamp;
	lclock += (rp + freq) * dtemp;
	dtemp = tp * (1. - pow(1. - 1. / (Kg * tau), dtemp));
	lclock += dtemp;
	tp -= dtemp;
	offset -= lclock;

	/*
	 * Update the clock filter and associated peer and statistics
	 * variables. The system clock is updated only if the timer
	 * interrupt update was suppressed and this is the first peer
	 * update to come along after that.
	 */
	filter(pp, offset, delay, disp);
	stat(&pp->stats, pp->tp);
	if (tstamp - lasttime >= (1 << poll))
		update();
}

/*
 * Clock update procedure
 *
 * This procedure is called once each poll interval to execute the
 * intersection and clustering algorithms and set the system clock. Note
 * that, under peculiar circumstances (multiple cliques, There may be no
 * intersection interval containing more than half the number of
 * sources, which results in no survivors.
 */
void
update() {
	double dtemp;		/* double temps */

	/*
	 * Update the local clock and current time offset, taking into
	 * account adjustments for the exponential decay of the time
	 * offset.
	 */
	dtemp = tstamp - utc;
	utc = tstamp;
	lclock += (rp + freq) * dtemp;
	dtemp = tp * (1. - pow(1. - 1. / (Kg * tau), dtemp));
	lclock += dtemp;
	tp -= dtemp;

	/*
	 * The NTP intersection and clustering algorithm.
	 */
	dts();
	if (nclock < 1) {
		if (debug)
			printf("%2li%10.3lf no source available\n", day,
			    sec);
		return;
	}
	select();
	if (source == 0)
		return;
	if (source->used)
		return;
	source->used = TRUE;

	/*
	 * Update the clock offset, roundtrip delay and despersion. If
	 * the distance is less than MAXDIST, update the local clock.
	 */
	dtemp = distance(source);
	if (dtemp >= MAXDIST && minpoll < maxpll) {
		if (debug)
			printf("%2li %9.3lf %-15s %2d max distance %lf %lf %lf\n",
			    day, sec, source->src, nclock, combine(),
			    source->dp, dtemp);
		return;
	}
	localclock(source, combine(), source->ep + xi + RHO * RHO);
}

/*
 * Local clock routines. They are designed to closely duplicate the
 * functions as described in the specification; however, the simulator
 * does not emulate the Unix adjtime() system call.
 */
/*
 * NTP local clock algorithm
 *
 * This routine performs the actual clock adjustment process as
 * described in the specification, including the various sanity checks
 * and nonlinear operations designed to resists spikes and noise that
 * sneak by the groom and gloom algorithms. To say these operations are
 * crafted heuristics is probably an understatement.
 *
 *	theta	clock offset
 *	epsil	total dispersion (filter + select)
 *	mu	time since last clock update
 *	lasttime time at last clock update
 *	poll	log2 poll interval
 *	tcnt	hysteresis counter
 *	tp	time prediction
 *	dp	delay prediction
 *	er	error bound prediction
 *	ep	dispersion prediction
 *	rp	frequency prediction
 *	state	clock discipline state
 */
void
localclock(pp, theta, epsil)
	struct peer *pp;	/* peer structure pointer */
	double theta;		/* clock offset */
	double epsil;		/* total dispersion */
{
	double stdphase;	/* phase estimate */
	double stdnoise;	/* noise estimate */
	double freqadj;		/* frequency adjustment */
	int i;			/* int temps */

	/*
	 * If the clock has never been set, just set it and exit.
	 */
	if (state == S_NSET) {
		tcnt = 0;
		poll = minpoll;
		tp = 0;
		lclock += theta;
		state = S_HOLD;
		if (debug)
			printf("%2li %9.3lf %-15s %2d clock set %lf\n",
			    day, sec, source->src, nclock, theta);
		return;
	}

	/*
	 * Ignore updates at intervals less than the system poll
	 * interval. Otherwise, compute PLL time constant from system
	 * poll interval.
	 */
	mu = tstamp - lasttime;
	if (mu < (1 << poll))
		return;
	tau = 1 << (poll - 2);

	/*
	 * Compute phase and noise estimates. The phase estimate is the
	 * RMS phase offset over the last AMAX updates. The noise
	 * estimate is the greater of this value and the RMS noise over
	 * the last EMAX updates.
	 */
	stdphase = stdnoise = 0;
	for (i = 0; i < AMAX; i++) {
		stdphase += phase[i];
	}
	stdphase = SQRT(stdphase / AMAX);
	for (i = 0; i < EMAX; i++)
		stdnoise += noise[i];
	stdnoise = SQRT(stdnoise / EMAX);
	if (stdphase < stdnoise)
		stdphase = stdnoise;

	/*
	 * If the offset exceeds MAXPHASE, unlock the loop and allow it
	 * to coast up to MAXWATCH with the poll interval clamped to
	 * minpoll. If the offset on a subsequent update is less than
	 * this, resume normal updates; if not, step the clock to the
	 * indicated time and transition to S_HOLD state, which will
	 * kick off error checking until the offset sublimes.
	 */
	freqadj = 0;
	if (fabs(theta) > MAXPHASE && state != S_HOLD) {
		tcnt = 0;
		poll = minpoll;
		tp = 0;
		if (mu > MAXWATCH) {
			lclock += theta;
			state = S_HOLD;
			stepcnt++;
			if (debug)
				printf("%2li %9.3lf %-15s %2d clock step %lf\n",
				    day, sec, source->src, nclock,
				    theta);
			for (i = 0; i < npeers; i++)
				clear(peers[i]);
			nclock = 0;
		} else {
			if (debug)
				printf("%2li %9.3lf %-15s %2d >128 ms %lf\n",
				    day, sec, source->src, nclock,
				    theta);
		}
		return;

	/*
	 * If the noise exceeds MAXEPSIL, just set the clock and leave
	 * the frequency alone.
	 */
	} else if (stdnoise > MAXEPSIL && state != S_HOLD) {
		if (debug)
			printf("%2li %9.3lf %-15s %2d noise %lf %lf\n",
			    day, sec, source->src, nclock, theta,
			    stdnoise);

	/*
	 * If the offset exceeds the previous average by a factor of
	 * SGATE and the interval since the last update is less than
	 * twice the system poll interval, consider the update a spike
	 * and ignore it.
	 */
	} else if (fabs(theta) > SGATE * stdphase && mu < (2 << poll) &&
	    state != S_HOLD) {
		spikcnt++;
		if (debug)
			printf("%2li %9.3lf %-15s %2d phase spike %lf %lf\n",
			    day, sec, source->src, nclock, theta,
			    stdphase);
		return;
	/*
	 * Use a phase-lock loop (PLL) at intervals < maxpll.
	 */
	} else if (mu < (1 << maxpll)) {
		freqadj = theta * mu / (tau * tau * Kf);
		pllcnt++;

	/*
	 * Otherwise, use a frequency-lock loop (FLL). The time constant
	 * is set well below the lowest poll interval used with the FLL.
	 */
	} else {
		tau = 2;
/*
		freqadj = stdrate;
*/
		freqadj = theta / mu * G;
		fllcnt++;
	}

	/*
	 * If the offset is less than MAXPHASE in any state, consider
	 * the clock synchronized. 
	 */
	if (fabs(theta) <= MAXPHASE) {
		state = S_SYNC;
		if (debug && state == S_HOLD)
			printf("%2li %9.3lf %-15s %2d synch %lf\n",
			    day, sec, source->src, nclock, theta);
	}

	/*
	 * Update the estimates and clamp the frequency to MAXFREQ.
	 */
	tp = theta;		/* offset */
	dp = pp->dp;		/* delay */
	er = pp->er;		/* error bound */
	ep = pp->ep;		/* dispersion */
	rp += freqadj;		/* frequency */
	if (rp > MAXFREQ)
		rp = MAXFREQ;
	else if (rp <= -MAXFREQ)
		rp = -MAXFREQ;
/*
	allan_var(tp, tstamp);
*/
	/*
	 * Determine when to change the the poll interval. A counter is
	 * decreased by twice the poll value if the offset exceeds
	 * PGATE times the noise estimate; otherwise it is increased by
	 * the poll value. If the counter reaches -LIMIT, poll is halved
	 * and the counter set to zero. If the couner reaches +LIMIT
	 * poll is doubled and the counter set to zero.
	 */
	if (fabs(theta) > PGATE * stdnoise) {
		tcnt -= poll << 1;
		if (tcnt < -LIMIT) {
			tcnt = -LIMIT;
			if (poll > minpoll) {
				tcnt = 0;
				poll--;
			}
		}
	} else {
		tcnt += poll;
		if (tcnt > LIMIT) {
			tcnt = LIMIT;
			if (poll < maxpoll) {
				tcnt = 0;
				poll++;
			}
		}
	}

	/*
	 * Update phase and noise registers.
	 */
	for (i = AMAX - 1; i > 0; i--) {
		phase[i] = phase[i - 1];
	}
	phase[0] = SQUARE(theta);
	for (i = EMAX - 1; i > 0; i--) {
		noise[i] = noise[i - 1];
	}
	noise[0] = epsil;
	lasttime = tstamp;
	if (debug)
		printf("%2li %9.3lf %-15s %2d %02x %5.0lf %5d %7.3lf %7.3lf %7.3lf %3d\n",
		    day, sec, pp->src, nclock, pp->reach & 0xff, mu, 1
	    << poll, lclock * 1e3, rp * 1e6, stdnoise * 1e3, tcnt);
	else
		printf("%2li %9.3lf %5.0lf %9.5lf %9.5lf %9.5lf %9.5lf %9.5lf %9.5lf\n",
		    day, sec, mu, lclock * 1e3, rp * 1e6, freq * 1e6,
		    freqadj * 1e6, stdnoise * 1e3, stdphase * 1e3);
	stat(&stats, lclock);
}

/*
 * Allan variance algorithm (not used in the actual implementation)
 *
 * This routine computes the Allan deviation of a series of time offset
 * measurements spaced at uneven intervals. While it is intended to
 * calculate only the first two terms of the deviation at one and two
 * sample intervals, this routine computes all terms up to log2(N),
 * where N is the number of samples (AMAX). Since the sample intervals
 * are not constant, the routine resamples the data at constant
 * poll and interpolates as necessary.
 *
 *	poll	log2 current poll interval
 */
void
allan_var(tp, tstamp)
	double tp;		/* sample update */
	double tstamp;		/* sample update time */
{
	double tau;		/* update interval */
	double x, y, ftmp, gtmp; /* double temps */
	int j, k, n;		/* int temps */

	/*
	 * Save new offset/timestamp samples.
	 */
	for (j = AMAX - 1; j > 0; j--) {
		allan[j].tp = allan[j - 1].tp;
		allan[j].tstamp = allan[j - 1].tstamp;
	}
	allan[0].tp = tp;
	allan[0].tstamp = tstamp;

	/*
	 * Calculate Allan deviation over last AMAX samples at the 
	 * current poll interval.
	 */
	tau = 1 << poll;
	k = 0;
	while (1) {
		var[k] = 0;
		n = 0; x = allan[0].tp; y = allan[0].tstamp;

		/*
		 * Scan all poll intervals to the end of the data.
		 */
		for (j = 1; j < AMAX && allan[j].tstamp != 0; j++) {

			/*
			 * Scan to the end of the poll interval and
			 * interpolate to find the offset and the
			 * nominal frequency.
			 */
			while (y - allan[j].tstamp >= tau) {
				ftmp = tau - (y - allan[j - 1].tstamp);
				gtmp = allan[j - 1].tstamp -
				    allan[j].tstamp;
				if (gtmp < 1.0)
					ftmp = (allan[j - 1].tp +
					    allan[j].tp) / 2.;
				else
					ftmp = allan[j - 1].tp +
					    (allan[j].tp - allan[j -
					    1].tp) * ftmp / gtmp;
				gtmp = x - ftmp;
				var[k] += gtmp * gtmp;
				n++; x = ftmp; y -= tau;
			}
		}
		if (n < 2)
			break;
		var[k] = sqrt(var[k] / (n * tau * tau));
		tau *= 2;
		k++;
	}
}

/*
 * Data grooming and glooming algorithms. They select the best data from
 * each peer and the best subset of peers alleged to be reliable and
 * accurate. The routines are intended to closely duplicate the
 * functions as described in the specification.
 */
/*
 * Clock filter algorithm
 *
 * This algorithm attempts to cast out outlyers on the basis of
 * roundtrip delay. From the most recent samples, it finds the one with
 * minimum delay and selects the associated offset as the peer variable.
 * The routine also computes the peer dispersion from the weighted
 * differences between the sample offsets.
 *
 *	tstamp	current time
 */
void
filter(pp, offset, delay, disp)
	struct peer *pp;	/* peer structure pointer */
	double offset;		/* sample offset */
	double delay;		/* sample delay */
	double disp;		/* sample dispersion */
{
	double list[FMAX];	/* synch distance array */
	int indx[FMAX];		/* index list */
	int i, j, k, n;		/* int temps */
	double x, y, phi;	/* double temps */

	/*
	 * Update the clock filter and insert the new sample, including
	 * clock offset, roundtrip delay and dispersion corrected for
	 * the frequency error integrated over the roundtrip time.
	 */
	phi = tstamp - pp->utc;
	pp->utc = tstamp;
	for (i = FMAX - 1; i > 0; i--) {
		pp->filtp[i] = pp->filtp[i - 1];
		pp->fildp[i] = pp->fildp[i - 1];
		pp->filer[i] = pp->filer[i - 1] + PHI * phi;
	}
	pp->filtp[0] = offset;
	pp->fildp[0] = delay;
	x = disp;
	if (x > MAXDISP)
		x = MAXDISP;
	pp->filer[0] = SQUARE(x + PHI * delay);

	/*
	 * Construct temp list sorted by synch distance to peer. This
	 * algorithm sorts samples less than MAXSEC old by distance to
	 * the remote peer. Older samples are left as-is. This avoids
	 * instabity when the FLL is in use.
	 */
	y = 0;
	for (n = 0; n < FMAX; n++) {
		list[n] = pp->filer[n] + pp->fildp[n] / 2;
		indx[n] = n;
		for (j = 0; j < n && y < MAXSEC && minpoll <= 10; j++) {
			if (list[j] > list[n]) {
				x = list[j];
				k = indx[j];
				list[j] = list[n];
				indx[j] = indx[n];
				list[n] = x;
				indx[n] = k;
			}
		}
		y += phi;
	}

	/*
	 * Determine the peer variables and calculate the filter
	 * dispersion, which is a weighted sum of the sample offset
	 * differences.
	 */
	i = indx[0];
	pp->tp = pp->filtp[i];
	pp->dp = pp->fildp[i];
	x = 0; y = 0;
	for (i = n - 1; i >= 0; i--) {
		x = FILTER * (x + pp->filer[indx[i]]);
		y = FILTER * (y + DIFF(pp->filtp[indx[0]],
		    pp->filtp[indx[i]]));
	}
	pp->er = x;
	pp->ep = y;
	pp->used = FALSE;
}

/*
 * Intersection algorithm
 *
 * This is a modification of the Marzullo algorithm, which determines
 * the correctness interval as the largest interval contained in the
 * intersection of the correctness intervals for all peers. The peer
 * correctness interval is defined as twice the synchronization distance
 * centered about the offset. In the modified algorithm, the midpoint of
 * each correctness interval must be included in the final intersection.
 * If this is confusing, read the SIGCOMM paper.
 *
 *	peers	peer pointer array
 *	npeers	number of peers
 *	nclock 	number of intersection peers
 *	bot	lowpoint
 *	top	highpoint
 */
void
dts()
{
	struct peer *pp;	/* peer structure pointer */
	double list[3 * NMAX];	/* temporary list */
	double dist;		/* synchronization distance */
	int indx[3 * NMAX];	/* index list */
	int f;			/* intersection ceiling */
	int end;		/* endpoint counter */
	int clk;		/* falseticker counter */
	int i, j, k, n;		/* int temps */
	double x;		/* double temps */

	/*
	 * Construct the endpoint list. Note we only keep reachable
	 * peers with synchronization distance less than MAXDISP (16 s).
	 * The spec modifies the distance by setting the high order bits
	 * to the stratum, but the simulator assumes stratum zero for
	 * all peers.
	 */
	nclock = 0;
	i = 0;
	for (n = 0; n < npeers; n++) {
		pp = peers[n];
		dist = distance(pp);
		if (dist >= MAXDISP || (pp->reach & 0xff) == 0)
			continue;
		nclock++;
		list[i] = pp->tp - (pp->st * MAXDISP + dist);
		indx[i] = -1;		/* lowpoint */
		for (j = 0; j < i; j++) {
			if ((list[j] > list[i]) || ((list[j] == list[i])
			    && (indx[j] > indx[i]))) {
				x = list[j];
				k = indx[j];
				list[j] = list[i];
				indx[j] = indx[i];
				list[i] = x;
				indx[i] = k;
				}
		}
		i = i + 1;
		list[i] = pp->tp;
		indx[i] = 0;		 /* midpoint */
		for (j = 0; j < i; j++) {
			if ((list[j] > list[i]) ||
			    ((list[j] == list[i]) && (indx[j] >
			    indx[i]))) {
				x = list[j];
				k = indx[j];
				list[j] = list[i];
				indx[j] = indx[i];
				list[i] = x;
				indx[i] = k;
			}
		}
		i = i + 1;
		list[i] = pp->tp + (pp->st * MAXDISP + dist);
		indx[i] = 1;		 /* highpoint */
		for (j = 0; j < i; j++) {
			if ((list[j] > list[i]) ||
			    ((list[j] == list[i]) && (indx[j] >
			    indx[i]))) {
				x = list[j];
				k = indx[j];
				list[j] = list[i];
				indx[j] = indx[i];
				list[i] = x;
				indx[i] = k;
			}
		}
		i = i + 1;
	}

	/*
	 * Toss out the falsetickers that do not lie in a common
	 * intersection.
	 */
	if (nclock < 1) {
		if (debug)
			printf("%2li%10.3lf no truechimers found\n",
			    day, sec);
		return;
	}
	for (f = 0; ; f++) {
		clk = 0;
		end = 0;
		for (j = 0; j < i; j++) {
			end = end - indx[j];
			bot = list[j];
			if (end >= (nclock - f))
				break;
			if (indx[j] == 0)
				clk = clk + 1;
		}
		end = 0;
		for (j = i - 1; j >= 0; j--) {
			end = end + indx[j];
			top = list[j];
			if (end >= (nclock - f))
				break;
			if (indx[j] == 0)
				clk = clk + 1;
		}
		if (clk <= f)
			break;
	}
	nclock -= clk;
}

/*
 * Selection algorithm
 *
 * This algorithm does the grunt work of selecting the best peer to
 * supply synchronization information. The idea is to order the peers by
 * synchronization distance, then compute the select dispersion between
 * each peer relative to the remaining survivors. The peer with largest
 * select dispersion is tossed; this process continues until one of
 * several termination conditions is met.
 *
 *	peers	peer pointer array
 *	npeers	number of peers
 *	plist	survivor peer pointer array
 *	nclock 	number of survivor peers
 *	bot	lowpoint
 *	top	highpoint
 *	source	system peer
 *	xi	maximum select dispersion
 */
void
select()
{
	struct peer *pp;	/* peer structure pointer */
	struct peer *pptemp;	/* another one */
	struct peer *oldsource;	/* old system peer */
	double eps;		/* min peer dispersion */
	int i, j, k, m, n;	/* int temps */
	double x;		/* double temps */
	double list[NMAX];	/* synch distance array */

	oldsource = source;

	/*
	 * Sort candidate list by synchronization distance. Non-
	 * survivors, e.g., those below bot or above top, are declared
	 * falseticker and sent to the freezer.
	 */
	m = 0;
	for (n = 0; n < npeers; n++) {
		pp = peers[n];
		if (pp->tp >= bot && pp->tp <= top) {
			pp->falsetick = FALSE;
			list[m] = distance(pp);
			plist[m] = peers[n];
			for (j = 0; j < m; j++) {
				if (list[j] > list[m]) {
					x = list[j];
					pptemp = plist[j];
					list[j] = list[m];
					plist[j] = plist[m];
					list[m] = x;
					plist[m] = pptemp;
				}
			}
			m = m + 1;
		} else if (!pp->falsetick) {
			pp->falsetick = TRUE;
			if (pp == source) {
				if (debug)
					printf("%2li %9.3lf %-15s %2d source falseticker %lf %lf %lf\n",
			 		   day, sec, pp->src, nclock,
					   pp->tp, bot, top);
				source = 0;
			}
		}
	}
	nclock = m;
	if (m == 0) {
		source = 0;
		if (debug)
			printf("%2li%10.3lf no survivors found\n", day, sec);
		return;
	}

	/*
	 * Cast out outlyers with max select dispersion less than min
	 * filter dispersion. Continue until the no further improvement
	 * can be made or the minimum number of peers is reached.
	 */
	if (m > MAXCLOCK)
		m = MAXCLOCK;
	while (1) {
		xi = 0; eps = MAXDISP;
		for (j = 0; j < m; j++) {
			x = 0;
			for (k = m - 1; k >= 0; k--)
				x = SELECT * (x + DIFF(plist[j]->tp,
				    plist[k]->tp));
			if (x > xi) {	/* max(xi) */
				xi = x;
				i = j;
			}
			x = plist[j]->ep;
			if (x < eps)
				eps = x; /* min(eps) */
		}
		if (xi <= eps || m <= MINCLOCK)
			break;
		if (plist[i] == source) {
			if (debug)
				printf("%2li %9.3lf %-15s %2d source outlyer %lf %lf %lf\n",
				    day, sec, source->src, m,
				    source->ep, xi, eps);
			source = 0;
		}
		for (j = i; j < m - 1; j++)
			plist[j] = plist[j + 1];
		m -= 1;
	}

	/*
	 * Declare the winner, but avoid clockhopping if the winner is
	 * already one of the good guys. The spec says to clamp the poll
	 * interval if the current source is booted, but we don't do
	 * that here.
	 */
	nclock = m;
	pp = plist[0];
	if (source != pp) {
		if (source == 0)
			source = pp;
		else if (pp->st < source->st)
			source = pp;
	}
	if (source != oldsource) {
		if (debug)
			printf("%2li %9.3lf %-15s %2d new clock source %lf %lf %lf\n",
			    day, sec, source->src, nclock, source->ep,
			    xi, eps);
	}
}

/*
 * Combining algorithm
 *
 * This routine computes the weighted average of the survivors of the
 * intersection and clustering algorithms. The weight is determined by
 * the synchronization distance of each peer separately.
 *
 *	plist	survivor peer pointer array
 *	nclock 	number of survivor peers
 *	returns combined time offset
 */
double
combine()
{
	struct peer *pp;	/* peer structure pointer */
	int i;			/* int temps */
	double x, y, z;		/* double temps */

	y = z = 0;
	for (i = 0; i < nclock; i++) {
		pp = plist[i];
		x = distance(pp);
		y += 1. / x;
		z += pp->tp / x;
	}
	return (z / y);
}

/*
 * Distance algorithm
 *
 * The synchronization distance includes: (1) half the root roundrip
 * delay (ROOTDELAY + fabs(pp->dp)) / 2.), (2) the root error bound
 * (ROOTDISP + pp->er + PHI * (tstamp - pp->utc)), (3) the peer
 * dispersion (SQRT(pp->ep)). In some cases the peer delay can be
 * negative due to large frequency offsets between the local and remote
 * peer.
 */
double
distance(pp)
	struct peer *pp;	/* peer structure pointer */
{
	return ((ROOTDELAY + fabs(pp->dp)) / 2. + ROOTDISP +
	    pp->er + PHI * (tstamp - pp->utc) + SQRT(pp->ep));
}

/*
 * reachreg - update reachability registers
 *
 * Update reachability status for all associations. This is not entirely
 * accurate, since the simulated poll times may not agree with the
 * actual poll times.
 */
void
reachreg() {
	struct peer *pp;	/* peer structure pointer */
	int i;			/* int temps */

	for (i = 0; i < npeers; i++) {
		pp = peers[i];
		if (tstamp - pp->poll < (1 << poll))
			continue;
		while (tstamp - pp->poll >= (1 << poll))
			pp->poll += 1 << poll;
		if ((pp->reach & 0xff) == 0)
			continue;
		pp->reach <<= 1;
		if ((pp->reach & 0xff) == 0)
			if (debug)
				printf("%2li %9.3lf %-15s %2d unreachable\n",
				    day, sec, pp->src, nclock);
			else if ((pp->reach & 0x3) == 0)
				filter(pp, 0, 0, MAXDISP);
	}
}

/*
 * Utility algorithms. These routines are specific to the simulator and
 * have no parallel in the protocol specification.
 */
/*
 * Clear peer state variables
 */
void
clear(pp)
	struct peer *pp;	/* peer structure pointer */
{
	int j;			/* int temps */

	for (j = 0; j < FMAX; j++) {
		pp->filtp[j] = pp->fildp[j] = 0;
		pp->filer[j] = MAXDISP;
	}
	pp->tp = pp->dp = pp->er = 0;
	pp->ep = MAXDISP;
	pp->reach = 0;
	pp->falsetick = pp->used = FALSE;
}

/*
 * Reset all system and peer state variables
 *
 *	tp	offset
 *	dp	delay
 *	er	error bound
 *	ep	dispersion
 *	rp	fractional frequency
 *	lclock	clock time
 *	freq	clock frequency (simulated)
 *	utc	last update time
 *	source	synch source peer
 *	phase	phase samples
 *	noise	noise samples
 *	poll	log2 poll interval
 *	stats	system statistics sructure
 *	state	clock discipline state
 *	spikcnt, stepcnt, pllcnt, fllcnt event counters
 */
void
reset()
{
	struct peer *pp;	/* peer structure pointer */
	int i;			/* int temps */

	freq = utc = 0;
	for (i = 0; i < AMAX; i++) {
		phase[i] = 0;
		allan[i].tp = 0;
		allan[i].tstamp = 0;
		var[i] = 0;
	}
	for (i = 0; i < EMAX; i++) {
		noise[i] = 0;
	}
	npeers = tcnt = nclock = 0;
	poll = minpoll;
	tau = (1 << poll);
	source = 0;
	state = S_NSET;
	clrstat(&stats);
	for (i = 0; i < npeers; i++) {
		pp = peers[i];
		clear(pp);
		clrstat(&pp->stats);
	}
	spikcnt = stepcnt = pllcnt = fllcnt = 0;
	for (i = 0; i < maxpoll; i++)
		pollcnt[i] = 0;
}

/*
 * Clear statistics
 */
void
clrstat(sp)
	struct stats *sp;	/* statistics structure pointer */
{
	sp->d0 = 0;		/* sample count */
	sp->d1 = 0;		/* latest sample */
	sp->d1sum = 0;		/* sample integral */
	sp->d2sum = 0;		/* sample integral squares */
	sp->d3 = 0;		/* sample absolute max */
	sp->begin = 0;		/* time of first sample */
	sp->utc = 0;		/* time of latest sample */
}

/*
 * Update statistics
 */
void
stat(sp, u)
	struct stats *sp;	/* statistics structure pointer */
	double u;		/* sample update */
{
	double dtemp;		/* double temps */

	if (sp->utc == 0) {
		sp->begin = tstamp;
		sp->utc = tstamp;
	}
	dtemp = tstamp - sp->utc;
	sp->utc = tstamp;
	sp->d0++;
	sp->d1sum += sp->d1 * dtemp;
	sp->d2sum += sp->d1 * sp->d1 * dtemp;
	sp->d1 = u;
	if (fabs(u) > sp->d3)
		sp->d3 = fabs(u);
}

/*
 * Display statistics. Needs sqrt() from math library.
 */
void
display()
{
	struct peer *pp;	/* peer structure pointer */
	struct stats *sp;	/* statistics structure pointer */
	int i;			/* int temps */
	double x, y, z;		/* double temps */

	printf("Poll intervals\n");
	for (i = 0; i < maxpoll + 1; i++) {
		if (pollcnt[i] > 0)
			printf("%2d %4d %u\n", i, pollcnt[i], pollcnt[i]
			    * 1 << i);
	}
	printf("PLL %u, FLL %u, spikes %u, steps %u\n", pllcnt, fllcnt,
	    spikcnt, stepcnt);
	printf("ID IP Src          IP Dst           Samples       Mean     StdDev        Max\n");
	sp = &stats;
	z = sp->utc - sp->begin;
	if (sp->d0 < 2) {
		x = 0;
		y = 0;
	} else {
		x = sp->d1sum / z;
		y = sqrt(sp->d2sum / z - x * x);
	}
	printf("   Local Clock                     %8.0f%11.3f%11.3f%11.3f\n",
		sp->d0, x * 1e3, y * 1e3, sp->d3 * 1e3);
	for (i = 0; i < npeers; i++) {
		pp = peers[i];
		sp = &pp->stats;
		z = sp->utc - sp->begin;
		if (sp->d0 < 2) {
			x = 0;
			y = 0;
		} else {
			x = sp->d1sum / z;
			y = sqrt(sp->d2sum / z - x * x);
		}
		printf("%2i %-15s %-15s %8.0f%11.3f%11.3f%11.3f\n",
		    pp->cd, pp->src, pp->dst, sp->d0, x * 1e3, y * 1e3,
		    sp->d3 * 1e3);
	}
}

/*
 * Generate a random number plucked from a Gaussian distribution with
 * mean zero and specified standard deviation. Needs cos(), log(),
 * rand(), sqrt() from math library. 
 */
double gauss(sigma)
	double sigma;		/* standard deviation */
{
 	double q1, q2;		/* double temps */

	while ((q1 = (double)rand() / 32768.) == 0);
	q2 = ((double)rand() / 32768.);
	return (sigma * sqrt(-2 * log(q1)) * cos(2 * PI * q2));
}
