/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 * Process ntp log data
 *  analyze synchronization performance
 *  predict future time
 *  simulate above prediction and analyze results
 */

#include "pd.h"

void
compute_uoffset(FILE *finfo, dataset_t *ds)
{
  int i;
  datapoint_t *dp;

  for ( i = 0; i < ds->n; i++ )
    {
      dp = &ds->dptr[i];
      dp->uoffset = dp->aoffset + dp->rsadj;
    }
}

/*
 * eliminate data points for very short runs
 * NOT and the startup transients of the rest
 */
void
trim_dat(FILE *finfo, dataset_t *ds)
{
  int first, last, npoints, currun, begtime, fvalid, i;

  /* should we trim/invalidate runs? */
  if ( CONFIG_TRIM == 0 )
    return;

  fprintf(finfo, "\n");
  /* iterate over entire data up to end of extent */
  for( first = 0; first < ds->extent_end ; first = last + 1)
    {
      currun = ds->dptr[first].run;
      begtime = ds->dptr[first].time;
      fvalid = -1;

      /* iterate over this run; count points and find first valid point */
      for (i = first, npoints = 0 ;
	   i < ds->extent_end && ds->dptr[i].run == currun ;
	   i++ )
	{

	  npoints++;
	  /* all points ok */
	  if ( fvalid == -1 )
	    fvalid = i;
	}
      last = i - 1;			/* back to last point */
      if ( last < first )
	panic("Run with 0 points");

      /* never drop last run */
      if ((fvalid == -1) ||
	  ((npoints < TRIM_POINTS ||
	    ds->dptr[last].time - ds->dptr[first].time < TRIM_TIME) &&
	   currun != RUN_MAX()))
	{
	  fprintf(finfo,
		  "TRIM: INVALIDATED RUN %d POINTS %d TIME %d (%d-%d)\n",
		  currun, npoints,
		  ds->dptr[last].time - ds->dptr[first].time,
		  first, last);
	  /* punt whole run */
	  for ( i = first; i <= last; i++ )
	    ds->dptr[i].valid = VALID_SHORT;
	}
      else
	{
	  fprintf(finfo,
		  "TRIM: ACCEPTED RUN %d POINTS %d TIME %d (%d-%d)\n",
		  currun, npoints,
		  ds->dptr[last].time - ds->dptr[first].time,
		  first, last);
	  /* XXX if trimming to fvalid, insert here, update printout */
	}
    }
}

/* used ONLY in analysis mode */
void
crunch_eval(dataset_t *ds, int backtime, int simp,
	    double *mean, double *rms, double *deriv)
{
  int firstpoint, lastpoint;

  int n, i, oldvalid;
  double acc, rsadj;
  double rsadjold, rsadjdot;
  datapoint_t *dp;

  if ( flags.predict )
    panic("evaluate in predict mode");

  *mean = *rms = *deriv = 0;

  /* find first point */
  lastpoint = ds->n - 1;
  for ( firstpoint = lastpoint;
       firstpoint >= 0 && ds->dptr[firstpoint].time + backtime > globals.last_time;
       firstpoint-- )
    ;
  firstpoint++;

  n = 0; acc = 0.0;
  oldvalid = 0;
  rsadjdot = rsadjold = 0.0;
  for ( i = firstpoint; i <= lastpoint; i++ )
    {
      dp = & ds->dptr[i];

      /* if simulating, only count points we predicted */
      if ( flags.simulate && ! dp->sim )
	continue;

      n++;
      rsadj = simp ? dp->simrsadjr[RSADJ_INT] : dp->rsadjr[RSADJ_INT];
      acc += rsadj;
      if ( oldvalid )
	{
	  double tmp = rsadjold - rsadj;
	  rsadjdot += tmp * tmp;
	}
      else
	oldvalid = 1;
      rsadjold = rsadj;
    }

  if ( n < 2 )
    return;

  *mean = acc / n;
  *deriv = sqrt( rsadjdot / (n - 1) );

  /* compute mean */
  acc = 0;
  for ( i = firstpoint; i <= lastpoint; i++ )
    {
      dp = & ds->dptr[i];
      if ( flags.simulate && ! dp->sim )
	continue;

      rsadj = simp ? dp->simrsadjr[RSADJ_INT] : dp->rsadjr[RSADJ_INT];
      rsadj -= *mean;
      acc += rsadj * rsadj;
    }

  *rms = sqrt(acc/n);
}

/*
 * print rms error and rms derivative for various time periods
 */

void
do_evaluate(FILE *finfo, int simp, dataset_t *ds)
{
  int backhours, backtime;
  double mean, rms, rmsderiv;

  /* print title */
  fprintf(finfo, "\t   ACTUAL");
  if ( simp )
    fprintf(finfo, "\t\t\tSIMULATED");
  fprintf(finfo, "\n");
  fprintf(finfo, "hours\t   mean\t    rms\t  deriv");
  if ( simp )
    fprintf(finfo,  "\t\t   mean\t    rms\t  deriv\n");
  else
    fprintf(finfo, "\n");

  for ( backtime = 3600; ; backtime *= 2 )
    {
      if ( backtime >= 2 * globals.last_time )
	break;
      if ( backtime > globals.last_time )
	backtime = globals.last_time;

      backhours = (backtime + 1799) / 3600;
      fprintf(finfo, "%d\t", backhours);

      crunch_eval(ds, backtime, 0, &mean, &rms, &rmsderiv);
      fprintf(finfo, "%7.3f %7.3f %7.3f",
	      mean * 1E3, rms * 1E3, rmsderiv * 1E3);
      if ( simp )
	{
	  crunch_eval(ds, backtime, 1, &mean, &rms, &rmsderiv);
	  fprintf(finfo, "\t\t%7.3f %7.3f %7.3f",
		  mean * 1E3, rms * 1E3, rmsderiv * 1E3);
	}
      fprintf(finfo, "\n");
    }
  fflush(finfo);
}

/*
 * set extent over which data will be examined
 * for analysis, it is entire region from globals.start to globals.stop
 * for prediction, it depends on the results from the last run
 */
void
set_extent(FILE *finfo, dataset_t *ds, int cl)
{
  int curpoint;
  time_t extent_begin = 0;
  time_t extent_extra;

  if ( flags.predict && cl != CLOCK_NTP)
    {
      if ( clock_state[cl].valid )
	{
	  extent_extra = globals.stop - Tntp2run(clock_state[cl].begin);
	  if ( extent_extra < EXTENT_MIN )
	    extent_extra = EXTENT_MIN;
	  if ( extent_extra > EXTENT_MAX )
	    extent_extra = EXTENT_MAX;

	  extent_begin = Tntp2run(clock_state[cl].begin) - extent_extra;
	  if ( extent_begin < 0 )
	    extent_begin = 0;
	}
      else
	extent_begin = 0;

      /* find first point >= begin time */
      for (curpoint = 0;
	   curpoint < ds->n && ds->dptr[curpoint].time < extent_begin;
	   curpoint++ )
	;
      ds->extent_begin = curpoint;
    }
  else
    /* analysis or ntp */
    ds->extent_begin = 0;

  /* find index of the first > globals.stop */
  for (curpoint = 0;
       curpoint < ds->n && ds->dptr[curpoint].time <= globals.stop;
       curpoint++ )
    ;
  /* index of last point is upper limit (to be used with < ) */
  ds->extent_end = curpoint;

  fprintf(finfo, "EXTENT begin %u end %u point %d %d\n",
	  Trun2unix(extent_begin),
	  Trun2unix(globals.stop),
	  ds->extent_begin, ds->extent_end);
}

struct fd_s
{
  FILE *finfntp;		/* comments for ntp fit */
  FILE *finfraw[NCLOCKS];	/* comments for raw fit */
  FILE *finfint;		/* comments for integrated fit */
} FD;

void
do_files_open(FILE *flog)
{
  int clock;
  char *cl;

  /* param file for ntp data */
  FD.finfntp = myopen2(datafiledir, "/PARAM-ntp", 1, 1);

  /* param file for raw data */
  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
    {
      cl = clock_unparse(clock);

      if ( clock_state[clock].flags & CLOCK_OMIT )
	continue;

      if ( raw[clock].n == 0 )
	{
	  fprintf(flog, "CLOCK %d (%s) has no data points!\n",
		    clock, cl);
	  continue;
	}

      FD.finfraw[clock] = myopen3(datafiledir, "/PARAM-raw.", cl, 1, 1);
    }

  /* param file for integrated data */
  FD.finfint = myopen2(datafiledir, "/PARAM-integrated", 1, 1);
}

void
do_files_close(FILE *flog)
{
  int clock;

  myclose(FD.finfntp);
  FD.finfntp = NULL;

  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
    if ( FD.finfraw[clock] != NULL )
      {
	myclose(FD.finfraw[clock]);
	FD.finfraw[clock] = NULL;
      }

  myclose(FD.finfint);
  FD.finfint = NULL;
}

/* do an iteration of basic processing */
void
do_process(FILE *flog)
{
  int clock;
  char descrip[1024];
  char *cl;

#ifdef MALLOC_DEBUG
  free(malloc(0x3011));
#endif

  if ( flags.simulate )
    sprintf(descrip, "SIMULATE %s %d %d (%s)",
	    flags.predict ? "PREDICT" : "ANALYSIS",
	    globals.start, globals.stop,
	    mytime(Trun2unix(globals.stop)));
  else
    sprintf(descrip, flags.predict ? "PREDICT" : "ANALYSIS");
  message(descrip);

  /* free and clear any stored fitsegs */
  globals_free_fitsegs();

  /*
   * set valid bits to all ok
   * reset weights
   * mark globally and per-clock not valid for integrated
   */
  ds_reinit(ntpp);
  ntpp->weight = 1.0;
  flags.intvalid = 0;
  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
    {
      ds_reinit(&raw[clock]);
      raw[clock].weight = 1.0;
      /* set clock not valid for integrated */
      clock_state[clock].flags &= ~ CLOCK_VALID_INTEGRATED;
    }

  messagen("NTP:");
  fprintf(FD.finfntp, "%s NTP\n", descrip);
  
  set_extent(FD.finfntp, ntpp, CLOCK_NTP);

  /* PROCESS NTP DATA unless simulating */
  if ( ! ( flags.predict && flags.simulate ) )
    {
      globals.fs_ntpoverall = Overall(FD.finfntp, ntpp, CLOCK_NTP);
      messagen(" overall");

      calc_rsadjr_fitseg(globals.fs_ntpoverall, RSADJ_OVERALL, 0);
      messagen(" calc_rsadjr");

      trim_dat(FD.finfntp, ntpp);
      messagen(" trimmed");

      globals.fs_ntp = Piecewise(FD.finfntp, ntpp, CLOCK_NTP, RSADJ_PQF);
      messagen(" piecewise");

      calc_rsadjr_fitseg(globals.fs_ntp, RSADJ_PQF, 0);
      messagen(" calc_rsadjr");

      fprintf(FD.finfntp, "END\n\n");
      fflush(FD.finfntp);
      message(".");
    }
  else
    message(" optimized-out.");

  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
    {
      cl = clock_unparse(clock);

      if ( FD.finfraw[clock] == NULL )
	  continue;

      messagen("RAW("); messagen(cl); messagen("):");

      fprintf(FD.finfraw[clock], "%s RAW %d (%s)\n", descrip, clock, cl);

      set_extent(FD.finfraw[clock], &raw[clock], clock);

      /* only process in simulation if needed for integrated */
      if ( flags.predict && flags.simulate &&
	! (clock_state[clock].flags & CLOCK_USE_INTEGRATED) )
	{
	  message(" optimized-out.");
	  continue;
	}
      
      if ( ! (flags.simulate && flags.predict) ) /* not in SIM PREDICT */
	{
	   /* overall fit */
	   globals.fs_rawoverall[clock] =
	     Overall(FD.finfraw[clock], &raw[clock], clock);
	 messagen(" overall");
	 
	 calc_rsadjr_fitseg(globals.fs_rawoverall[clock], RSADJ_OVERALL, 0);
	 messagen(" calc_rsadjr");
       }

      trim_dat(FD.finfraw[clock], &raw[clock]);
      messagen(" trimmed");

      select_low_delay(FD.finfraw[clock], &raw[clock], clock);
      messagen(" select");
      fflush(FD.finfraw[clock]);

      /* piecewise fit */
      globals.fs_raw[clock] =
	Piecewise(FD.finfraw[clock], &raw[clock], clock, RSADJ_PQF);
      messagen(" piecewise");

      calc_rsadjr_fitseg(globals.fs_raw[clock], RSADJ_PQF, 0);
      messagen(" calc_rsadjr");

      fprintf(FD.finfraw[clock], "END\n\n");
      fflush(FD.finfraw[clock]);
      message(".");
    }

  /* INTEGRATED */
  messagen("Integrated:");
  fprintf(FD.finfint, "%s INTEGRATED\n", descrip);
  globals.fs_integrated = Integrated(FD.finfint);
  messagen(" fit");
  fflush(FD.finfint);

  /* record simrsadj, rsadjresid, etc. as appropriate */
  if ( flags.simulate && flags.predict ) /* SIM PRED */
    {
      /* record results (or lack thereof) for later analysis */
      if ( flags.intvalid )
	{
	  calc_simrsadj_dataset(FD.finfint, ntpp, globals.fs_integrated->fp);
	  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
	    calc_simrsadj_dataset(FD.finfint, &raw[clock],
				  globals.fs_integrated->fp);
	  messagen(" calc_simrsadj");
	}
      else
	messagen(" NOTcalc_simrsadj");
    }
  else /* SIM ANAL, PRED, ANAL (all except SIM PRED) */
    {
      /* calculate residuals from integrated fit */
      if ( flags.intvalid )
	{
	  /* apply integrated fit to all clocks */
	  calc_rsadjr_dataset(ntpp, globals.fs_integrated->fp,
			      RSADJ_INT, CLOCK_NTP, 0);
	  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
	    calc_rsadjr_dataset(&raw[clock], globals.fs_integrated->fp,
				RSADJ_INT, clock, 0);
	  messagen(" calc_rsadjr");
	}
      else
	messagen(" NOTcalc_rsadjr");
    }

  fprintf(FD.finfint, "END\n\n");
  fflush(FD.finfint);
  message(".");

  /* Evaluate the results */
  if ( ! flags.predict )	/* SIM ANAL or ANAL */
    {
      messagen("EVALUATE:");
      do_evaluate(flog, flags.simulate, ntpp);
      message(".");
    }

#ifdef MALLOC_DEBUG
  free(malloc(0x3012));
#endif
}

void
main(int argc, char **argv)
{
  FILE *fin;			/* input data */
  FILE *fstate;			/* state file */
  FILE *flog;			/* random flamage */
  int clock;
  char *cp;
#ifdef GNU_MALLOC
  void mtrace();
#endif

  initjunk();
  config_init();

  flags.debug = 0;		/* DONT debug */
  flags.verbose = 0;		/* DONT do verbose output */
  flags.nooutput = 0;		/* DO output files */
  flags.noaging = 1;		/* DONT use aging */
  flags.simulate = flags.predict = 0; /* SIMULATE */
  
  if ( argc >= 3 )
    for ( cp = argv[2]; *cp; cp++ )
      switch ( *cp )
	{
	case 'd':		/* debug */
	  flags.debug++; break;
	case 'v':		/* extra output files, normally unneeded */
	  flags.verbose++; break;
	case 'o':		/* no output files */
	  flags.nooutput = 1; break;
	case 'a':		/* allow aging */
	  flags.noaging = 0; break;
	case 'A':		/* don't allow aging, but warn */
	  flags.noaging = 2; break;
	case 'p':		/* predict rather than analyze */
	  flags.predict = 1; break;
	case 's':		/* predict and simulate periodically */
	  flags.predict = flags.simulate = 1; break;
	case 'i':		/* simulate over 1 hour */
	  SIM_INCR = atoi(argv[3]);
	  if ( SIM_INCR < 10 )
	    SIM_INCR = 3600;
	  break;
	case 't':
#ifdef GNU_MALLOC
#ifdef HAVE_SETENV
	  setenv("MALLOC_TRACE", "malloc-output");
#else
#ifdef HAVE_PUTENV
	  putenv("MALLOC_TRACE=malloc-output");
#endif
#endif
	  mtrace();
#else
	  fprintf(stderr, "MALLOC_TRACE NOT SUPPORTED\n");
#endif
	  break;
	default:
	usage:
	  fprintf(stderr,
		  "Usage: pd host [dvoaps] (Debug Verbose suppress-Output Aging Predict Simulate)\n");
	  exit(-1);
	}
  else
      if ( argc != 2 )
	  goto usage;

  if ( flags.simulate )
    {
      fprintf(stdout, "SIM_INCR %d\n", SIM_INCR);
      fflush(stdout);
    }

  sprintf(datahostname, argv[1]);
  sprintf(datafiledir, "m/%s", datahostname);
  sprintf(datafilename, "%s/detailed.%s", datafiledir, datahostname);

  /* open log file */
  flog = myopen2(datafiledir, "/LOG", 1, 1);

  if ( flags.predict )
      if ( flags.simulate )
	cp = "SIMULATION MODE";
      else
	cp = "PREDICTION MODE";
  else
    cp = "ANALYSIS MODE";
  fprintf(flog, "%s\n", cp);
  message(cp);

  /* initialize data strutures */
  ds_init(ntpp);
  for ( clock = 0; clock < NCLOCKS ; clock ++ )
    ds_init(&raw[clock]);

  fin = myopen2(datafiledir, "/CONFIG", 0, 0);
  if ( fin )
    {
      config_readfile(fin, flog, stderr);
      myclose(fin);
    }

  message("INITIALIZATION COMPLETE");

  /* clear clock state information */
  bzero(clock_state, sizeof(clock_state));

  /* READ STATE INFO */
  /* open file of desired clocks and valid bits */
  fin = myopen2(datafiledir, "/CLOCK_CONFIG", 0, 0);
  if ( fin != NULL )
    {
      clock_config(fin, flog);
      fflush(flog);
      myclose(fin);
      message("CLOCK_CONFIG");
    }
  else
    message("NO CLOCK CONFIG FILE");

  /* open file of time intervals for previous fits */
  if ( flags.predict && ! flags.simulate )
    {
      fstate = myopen2(datafiledir, "/CLOCK_STATE", 0, 0);
      if ( fstate != NULL )
	{
	  clock_state_read(fstate, flog);
	  fflush(flog);
	  myclose(fstate);
	  message("CLOCK_STATE");
	}
      else
	message("NO CLOCK STATE FILE");
    }

  /* READ DATA */
  fin = myopen2(datafilename, "", 0, 1);
  input_dat(fin, flog);
  fflush(flog);
  myclose(fin);
  message("READ DATA");

  /* open output param files (needs info from reading data) */
  do_files_open(flog);

  /* make sure we have some valid clocks */
  clock_state_makevalid(flog);

  /* compute UOFFSETS */
  compute_uoffset(flog, ntpp);
  for ( clock = 0; clock <= CLOCK_MAX() ; clock ++ )
    compute_uoffset(flog, &raw[clock]);

#ifdef MALLOC_DEBUG
  free(malloc(0x3001));
#endif

  if ( ! flags.simulate )
    {
      /* either analysis or predict, but not simulate predict */
      globals.start = 0;
      globals.stop = globals.last_time;

      do_process(flog);
    }
  else
    /* SIMULATION */
    {
      /* set up time variables */
      globals.start = 0;		/* first thing seen */
      globals.stop = SIM_BEGIN;	/* XXX align on hour ?? */

      do_process(flog);

      while ( globals.stop < globals.last_time )
	{
	  globals.stop += SIM_INCR; /* advance ? hours */
	  if ( globals.stop > globals.last_time )
	    globals.stop = globals.last_time;

	  do_process(flog);
	}

      /* change to analysis mode */
      flags.predict = 0;
      /* process final time with predict = 0 BUT simulate = 1 */
      do_process(flog);
    }

  do_files_close(flog);

#ifdef MALLOC_DEBUG
  free(malloc(0x3002));
#endif

  /* OUTPUT - state information */
  if ( flags.predict )
    {
      fstate = myopen2(datafiledir, "/CLOCK_STATE", 1, 0);
      if ( fstate != NULL )
	{
	  clock_state_write(fstate, flog);
	  fflush(flog);
	  myclose(fstate);
	  message("CLOCK_STATE_WRITE");
	}
    }
  else
    message("ANALYSIS MODE - didn't write clock_state");

  /* OUTPUT - data */
  if ( flags.nooutput )
    message("NO OUTPUT.");
  else
    {
      messagen("NTP continued:");
      fprintf(flog, "NTP: %d points\n", ntpp->n);
      if ( ntpp->n > 0 )
	{
	  output_dat(CLOCK_NTP);
	  messagen(" wrote");
	}
      else
	  messagen(" NO POINTS --- DIDN'T WRITE");

      message(".");
      
      for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
	{
	  char *cl = clock_unparse(clock);
	  
	  fprintf(flog, "CLOCK %2d [%15s]: %d points\n",
		  clock, cl,  raw[clock].n);
	  
	  if ( raw[clock].n == 0 )
	    continue;
	  
	  messagen("RAW continued ("); messagen(cl); messagen("):");
	  
	  if ( ! (clock_state[clock].flags & CLOCK_OMIT)
	      && raw[clock].n >= OUTPUT_MIN_POINTS)
	    {
	      output_dat(clock);
	      messagen(" wrote");
	    }
	  else
	    messagen(" DIDN'T_WRITE");
	  
	  message(".");
	}
    }

  globals_free_fitsegs();

  ds_free(ntpp);
  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
    ds_free(&raw[clock]);

  myclose(flog);
  exit(0);
}
