/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

/* FIT_PARAM routines */

fit_param_t *
fit_param_new()
{
  fit_param_t *fp;

  fp = NEW(fit_param_t);
  if (fp == NULL)
    panic("fit_param_new malloc");
  bzero(fp, sizeof(fit_param_t));
  return fp;
}

void
fit_param_print(FILE *finfo, fit_param_t *fp)
{
  int i;

  fprintf(finfo, "FIT_PARAM: a=%f f=%f",
	  fp->aging * 86400.0 * 2.0 * KK,
	  fp->freq * KK);
  
  for ( i = 0; i <= RUN_MAX(); i++ )
    if (fp->phasevalid[i])
      fprintf(finfo, " (run %d)=%f", i, fp->phase[i]);

  for ( i = 0; i <= CLOCK_MAX(); i++ )
    if (fp->systematicvalid[i])
      fprintf(finfo, " (clock %d)=%f", i, fp->systematic[i]);

  fprintf(finfo, "\n");
}

/* return aging in ppm/day  */
double fit_param_aging(fit_param_t *fp)
{
  return fp->aging * 86400.0 * 2.0 * KK;
}

/* return frequency at t in ppm  */
double fit_param_freq(fit_param_t *fp, ntp_t t)
{
  return (fp->freq + fp->aging * t) * KK;
}

/* return phase at t in seconds */
double fit_param_phase(fit_param_t *fp, ntp_t t, int run, int clock)
{
  double phase = (t * fp->aging + fp->freq) * t;
  if ( run >= 0 )
    phase += fp->phase[run];
  if ( clock > CLOCK_NTP )
    phase += fp->systematic[clock] + clock_state[clock].offset;
  return phase;
}

/*
 * make 'good' a priori guess into fp for join of segments fit
 * by fp1, fp2.  'Average' freq, aging.  Copy phases, 'averaging'
 * those present in both.
 * The point of this is to speed up convergence of later estimation.
 */
void
fit_param_merge(fit_param_t *fp, fit_param_t *fp1, fit_param_t *fp2)
{
  int i;

  *fp = *fp1;			/* freq, aging, phase/clock info from 1 */
  /* copy info from fp2 if not in conflict with fp1 */
  for ( i = 0; i <= RUN_MAX(); i++ )
    if ( ! fp->phasevalid[i] && fp2->phasevalid[i] )
      {
	fp->phase[i] = fp2->phase[i];
	fp->phasevalid[i] = 1;
      }

  for ( i = 0; i <= CLOCK_MAX(); i++ )
    if ( ! fp->systematicvalid[i] && fp2->systematicvalid[i] )
      {
	fp->systematic[i] = fp2->systematic[i];
	fp->systematicvalid[i] = 1;
      }
}

/* given ss, calculate fit
 * return phase, and modify aging and freq in fp
 */
double
fit_param_calc(FILE *finfo, sumseg_t *ss, fit_param_t *fp,
	       int noage, int allowbogus)
{
  double c1, c2, a1, a2, d1, d2;
  double phase;

  /* start off with zero as estimate */
  fp->aging = fp->freq = 0.0;

  /* no points? */
  if ( ss->n == 0 || ss->sumweights == 0.0 )
    /* all answers are equally correct, so choose 0  */
    return 0.0;

  /* 1 point?  ==> return its value as phase */
  if ( ss->n == 1 )
    return ss->Sy / ss->sumweights;

  if ( noage )
    goto dolinear;

  /* if too short a time, don't compute aging */
  if ( ss->n < 10 || flags.noaging
      || ss->htime - ss->ltime < AGING_TIME_MIN )
    goto dolinear;

  /* attempt quadratic fit */
  c1 = ss->Sxy - (ss->Sx * ss->Sy / ss->sumweights);
  c2 = ss->Sxxy - (ss->Sxx * ss->Sy / ss->sumweights);
  a1 = (ss->Sxx * ss->Sx / ss->sumweights) - ss->Sxxx;
  a2 = (ss->Sxx * ss->Sxx / ss->sumweights) - ss->Sxxxx;
  d1 = (ss->Sx * ss->Sx / ss->sumweights) - ss->Sxx;
  d2 = (ss->Sxx * ss->Sx / ss->sumweights) - ss->Sxxx;

  /* 'aging variable' is really a * t^2 part of fit, so
     what we mean by 'aging' as in ppm/day (Hz/s really) is 2*a */
  fp->aging = (d1 * c2 - d2 * c1) / ( a1 * d2 - a2 * d1 );
  fp->freq = (a1 * c2 - a2 * c1) / ( d1 * a2 - d2 * a1 );
  phase = (ss->Sy - fp->aging * ss->Sxx - fp->freq * ss->Sx) / ss->sumweights;

  /* if aging is too large, don't allow it */
  if ( ! allowbogus && (fp->aging < - AGING_BOGUS || fp->aging > AGING_BOGUS) )
    {
      fprintf(finfo,
	      "BOGUS AGING %f ppm/day LIMIT %f ppm/day f %f ppm p %f s\n",
	      fp->aging * 86400 * 2.0 * KK,
	      AGING_BOGUS * 86400 * 2.0 * KK,
	      fp->freq * KK, phase);
#ifdef DEBUG_QUAD_FIT
      sumseg_print(finfo, ss);
#endif

    dolinear:
      fp->aging = 0;
      fp->freq = (ss->Sxy - ss->Sy * ss->Sx / ss->sumweights)
	/ (ss->Sxx - ss->Sx * ss->Sx / ss->sumweights);
      phase = ( ss->Sy - fp->freq * ss->Sx ) / ss->sumweights;

#ifdef DEBUG_QUAD_FIT
      fprintf(finfo, "NOW %f ppm/day %f ppm p %f s\n",
	      fp->aging * 86400 * 2.0 * KK, fp->freq * KK, phase);
#endif
    }

  /* if frequency bogus, set to zero */
  if ( fabs(fp->freq) >= FREQ_BOGUS )
    {
      fprintf(finfo, "FREQ_BOGUS %f ppm LIMIT %f ppm\n",
	      fp->freq * 1E6,
	      FREQ_BOGUS * 1E6);
      fp->freq = 0;
      phase = ss->Sy / ss->sumweights;
    }

  return phase;
}

/*
 * calculate poffset (residual of dp wrt fp)
 */
double
fit_param_residpoint(datapoint_t *dp, fit_param_t *fp)
{
  double predrsadj, phase, resid;
  int run;

  run = dp->run;		/* was sp->run */

  phase = fp->phase[run];
  if ( dp->clock != CLOCK_NTP )
    phase += fp->systematic[dp->clock];

  predrsadj = (fp->aging * dp->time + fp->freq) * dp->time + phase;

  resid = dp->uoffset - predrsadj;
  return resid;
}


/* FIT_QUALITY routines */
fit_quality_t *
fit_quality_new()
{
  fit_quality_t *fq;

  fq = NEW(fit_quality_t);
  if (fq == NULL)
    panic("fit_quality_new malloc");
  bzero(fq, sizeof(fit_quality_t));
  return fq;
}

void
fit_quality_print(FILE *finfo, fit_quality_t *fq)
{
  fprintf(finfo, "FIT_QUALITY: n=%d sumweight=%f sum=%f sumsq=%f offset=%f\n",
	  fq->n, fq->sumweights,
	  fq->sumresid, fq->sumsqresid, fq->offset);
}

void
fit_quality_init(fit_quality_t *fq)
{
  fq->n = 0;
  fq->sumweights = fq->sumresid = fq->sumsqresid = 0.0;
}

void
fit_quality_sum(segment_t *sp,
	    double aging, double freq, double phase,
	    fit_quality_t *fq, int includeinvalid)
{
  segment_t *spi;
  int i;
  double weight;

  /* loop over linked segments in same run */
  for ( spi = sp; spi; spi = spi->samerun )
    for ( i = spi->first; i < spi->first + spi->n; i++ )
      {
	datapoint_t *dp = &spi->ds->dptr[i];
	double resid;

	if ( dp->valid == VALID_OK || includeinvalid == 1 )
	  {
	    fq->n++;
	    /* per-point weights not yet implemented */
	    if ( spi->clock > CLOCK_NTP )
	      weight = raw[spi->clock].weight;
	    else
	      weight = 1.0;
	    fq->sumweights += weight;

	    resid = dp->uoffset -
	      ((aging * dp->time + freq) * dp->time + phase);

	    fq->sumresid += weight * resid;
	    fq->sumsqresid += weight * resid * resid;
	  }
      }
}

/* calculate residuals of fit (fp) applied to data (sp) into fq  */
void
fit_quality_calc(segment_t *sp, fit_param_t *fp, fit_quality_t *fq,
		 int crossclockruns, int doingint, int includeinvalid)
{
  segment_t *spi;
  double phase;

  fit_quality_init(fq);

  /* loop over segments, computing resid for each */
  for ( spi = sp; spi; spi = crossclockruns ? spi->next : NULL  )
    {
      phase = fp->phase[spi->run];
      if ( spi->clock != CLOCK_NTP )
	{
	  phase += fp->systematic[spi->clock];
	  if ( doingint)
	    phase += clock_state[spi->clock].offset;
	}
      fit_quality_sum(spi, fp->aging, fp->freq, phase, fq, includeinvalid);
    }

  if ( fq->sumweights != 0.0 )
    {
      fq->sdev = sqrt(fq->sumsqresid/fq->sumweights);
      fq->offset = fq->sumresid/fq->sumweights;
    }
  else
    fq->sdev = fq->offset = 0.0;
}


/* FITSEG routines */
/*
 * malloc a new fitseg, and also fp and fq inside it
 * DON'T allocate a segment
 */
fitseg_t *
fitseg_new()
{
  fitseg_t *fs = NEW(fitseg_t);
  if ( fs == NULL )
    panic("fitseg_new malloc");

  bzero(fs, sizeof(fitseg_t));

  fs->next = NULL;
  fs->generation = 0;
  fs->cand = NULL;

  fs->fp = fit_param_new();
  fs->fq = fit_quality_new();

  return fs;
}

void
fitseg_free(fitseg_t *fs)
{
  if ( fs == NULL )
    panic("fitseg_free of NULL fs");

  if ( fs->next ) fitseg_free(fs->next);
  if ( fs->sp ) segment_free(fs->sp);
  if ( fs->fp ) free(fs->fp);
  if ( fs->fq ) free(fs->fq);
  bzero(fs, sizeof(fs));
  free(fs);
}

/*
 * print a fitseg for humans
 * convey which clocks and times the segment covers
 * convey frequency/phase at begin/end of runs
 */
#define CRMAYBE		if ( needcr ) { fprintf(finfo, "\n"); needcr = 0; }
#define NEEDCR		needcr=1;
void
fitseg_print(FILE *finfo, fitseg_t *fs)
{
  int i;
  int saved_run = -1;
  segment_t *sp;
  int needcr = 0;

  fprintf(finfo, "{FITSEG:\n");

  if ( fs->sp == NULL)
    {
      fprintf(finfo, "NO segments\n");
      goto done;
    }

  /* if more than one clock, print per-clock info */
  if ( segment_count_clocks(fs->sp) > 1 )
    {
      /* for each clock, print begin and end times */
      for ( i = 0; i <= CLOCK_MAX(); i++ )
	{
	  int nsegs = 0;
	  ntp_t begin = 0;
	  ntp_t end = 0;

	  for ( sp = fs->sp; sp; sp = sp->next ) /* forall segments */
	    if ( sp->clock == i )
	      {			/* find min, max time */
		ntp_t sbegin = segment_begin_time(sp);
		ntp_t send = segment_end_time(sp, 0);
		if ( nsegs == 0 || sbegin < begin ) begin = sbegin; 
		if ( nsegs == 0 || send > end ) end = send;
		nsegs++;
	      }
	  if ( nsegs )
	    {
	      fprintf(finfo, "Clock %2d [%15s] segs=%2d offset=%8.3f ms\n",
		      i, clock_unparse(i), nsegs, fs->fp->systematic[i]*1E3);
	      if ( ! fs->fp->systematicvalid[i] )
		fprintf(finfo, " SYSTEMATIC NOT VALID!!!\n");
	      fprintf(finfo,"  BEGIN %s",
		      mytime(Trun2unix(begin)));
	      fprintf(finfo,"  END %s\n",
		      mytime(Trun2unix(end)));
	    }
	}
      NEEDCR
    }

  /* show begin, end times per run */
  for ( i = 0; i <= RUN_MAX(); i++ )
    {
	int nsegs = 0;
	int npoints = 0, nvalid = 0;
	ntp_t begin = 0, end = 0;

	for ( sp = fs->sp; sp; sp = sp->next ) /* forall segs */
	  if ( sp->run == i )
	    {
	      ntp_t sbegin = segment_begin_time(sp);
	      ntp_t send = segment_end_time(sp, 0);
	      if ( ! nsegs || sbegin < begin ) begin = sbegin;
	      if ( ! nsegs || send > end ) end = send;
	      nsegs++;
	      npoints += segment_count(sp, 0);
	      nvalid += segment_count_valid(sp, 0);
	    }

	if ( nsegs )
	  {
	    CRMAYBE
	    fprintf(finfo,
		    "Run %2d segs=%2d points=%5d valid=%5d aging=%7.3f ppm/day sdev=%7.1f us\n",
		    i, nsegs, npoints, nvalid, fit_param_aging(fs->fp),
		    fs->fq->sdev * KK);
	    
	    fprintf(finfo, "BEG %s freq=%10.6f ppm phase=%10.6f s\n",
		    mytime(Trun2unix(begin)),
		    fit_param_freq(fs->fp, begin),
		    fit_param_phase(fs->fp, begin, i, CLOCK_NTP));
	    
	    fprintf(finfo, "END %s freq=%10.6f ppm phase=%10.6f s\n",
		    mytime(Trun2unix(end)),
		    fit_param_freq(fs->fp, end),
		fit_param_phase(fs->fp, end, i, CLOCK_NTP));
	    saved_run = i;
	    NEEDCR
	  }
      }
  if ( OUTPUT_SHOWNEXTPHASE )
    if ( fs->next && fs->next->sp && fs->next->sp->run == saved_run )
      {
	ntp_t begin = segment_begin_time(fs->next->sp);
	fprintf(finfo, "NXT %s                     phase=%10.6f s\n",
		mytime(Trun2unix(begin)),
		fit_param_phase(fs->fp, begin, saved_run, CLOCK_NTP));
      }

 done:
  fprintf(finfo, "}\n");
}

/* return last fitseg in linked list */
fitseg_t *
fitseg_last(fitseg_t *fs)
{
  while ( fs->next != NULL )
    fs = fs->next;
  return fs;
}

/* return count of valid points in fitseg
 * traverse linked fitsegs if crossclockruns */
int
fitseg_count_valid(fitseg_t *fs, int crossclockruns)
{
  fitseg_t *fs1;
  int valid = 0;

  for ( fs1 = fs; fs1; fs1 = crossclockruns ? fs1->next : NULL )
    valid += segment_count_valid(fs1->sp, 1);
  return valid;
}


/* unclassified routines */

/* perform quadratic fit on this segment, but not ->next links
 * fp and fq are both output parameters
 */
void
dofit_segment_single(FILE *finfo,
	 segment_t *sp,
	 fit_param_t *fp,
	 fit_quality_t *fq)
{
  sumseg_t ss;

  /* compute sums */
  sumseg_init(&ss);
  sumseg_sum(sp, 0.0, &ss, 0, 0);

  /* calculate frequency, aging */
  bzero(fp, sizeof(fit_param_t));
  fp->phase[sp->run] = fit_param_calc(finfo, &ss, fp, 0, 0);
  fp->phasevalid[sp->run] = 1;

  /* calculate residual of fit */
  fit_quality_calc(sp, fp, fq, 0, 0, 0);
}

/*
 * Perform linear fit on segment, including next
 * Requires: only one clock
 */

void
dofit_segment_multirun( FILE *finfo,
		       segment_t *sp, 
		       fit_param_t *fp,
		       fit_quality_t *fq)
{
  int run;
  segment_t *sp1;
  sumseg_t ss[NRUNS];
  double freqtop, freqbottom;

  if ( sp == NULL )
    panic("dofit_segment_multirun: NULL segment");
  if ( segment_count_clocks(sp) != 1 )
    panic("dofit_segment_multirun: clocks != 1");

  /* compute sums PER RUN */
  for ( run = 0; run <= RUN_MAX(); run++ )
    sumseg_init(&ss[run]);
  for ( sp1 = sp; sp1; sp1 = sp1->next )
    sumseg_sum(sp1, 0.0, &ss[sp1->run], 0, 0);
  
  bzero(fp, sizeof(fit_param_t));
  fp->aging = 0;

  /* calculate freq */
  freqtop = freqbottom = 0.0;
  for ( run = 0; run <= RUN_MAX(); run++ )
    {
      if ( flags.debug > 2 )
	{
	  fprintf(finfo, "dofit_segment_multirun: run=%d\n", run);
	  sumseg_print(finfo, &ss[run]);
	}

      if ( ss[run].n > 0 && ss[run].sumweights > 0 )
	  {
	    freqtop +=
	      ss[run].Sxy - ss[run].Sy * ss[run].Sx / ss[run].sumweights;
	    freqbottom +=
	      ss[run].Sxx - ss[run].Sx * ss[run].Sx / ss[run].sumweights;
	    if ( flags.debug > 2 )
	      fprintf(finfo, "top=%f, bottom=%f\n", freqtop, freqbottom);
	  }
      }
  if ( freqbottom != 0 )
    fp->freq = freqtop / freqbottom;
  else
    {
      fprintf(finfo,
	      "segment_dofit_multirun: top %e bottom %e  ==> freq = 0\n",
	      freqtop, freqbottom);
      fp->freq = 0;
    }

  /* calculate phases */
  for ( run = 0; run <= RUN_MAX(); run++ )
    if ( ss[run].n > 0 && ss[run].sumweights > 0 )
      {
	fp->phase[run] =
	  (ss[run].Sy - fp->freq * ss[run].Sx ) / ss[run].sumweights;
	fp->phasevalid[run] = 1;
      }
  if ( flags.debug > 1 )
    fit_param_print(finfo, fp);

  /* calculate residual of fit */
  fit_quality_calc(sp, fp, fq, 1, 0, 0);
}

/*
 * perform quadratic fit on segment sp INCLUDING ->next 
 * ss, fp, fq are output parameters (for ltime/htime)
 * works by alternately estimating aging/freq and phases
 * assumes fit already done on 'similar' data, so does phases first
 * if doinging true, uses precomputed offsets for clocks
 */
void
dofit_segment_multiple(FILE *finfo, segment_t *sp, sumseg_t *ss,
       fit_param_t *fp, fit_quality_t *fq, int doingint)
{
  double phase[NRUNS];
  double phasevar[NRUNS];
  double wphase[NRUNS];
  double systematic[NCLOCKS];
  double sysvar[NCLOCKS];
  double wsystematic[NCLOCKS];

  segment_t *spi;
  int niter, i, noage;
  double cum_error;
  double off, offabs;
  double est_phase;

  /* clear valid bits; set later if data used to estimate phases */
  for ( i = 0; i <= RUN_MAX(); i++ )
    {
      fp->phase[i] = 0.0;
      fp->phasevalid[i] = 0.0;
    }
  for ( i = 0; i <= CLOCK_MAX(); i++ )
    {
      fp->systematic[i] = 0.0;
      fp->systematicvalid[i] = 0.0;
    }

  noage = 0;			/* allow aging */
  for ( niter = 0; niter < 1000; niter++ )
    {
      cum_error = 0.0;

      if ( niter == 0 )
	{
	  cum_error += 1.0;      /* do phases first first time around */
	  est_phase = 0;
	}
      else
	/* estimate frequency and aging given phases */
	{
	  /* sum data corrected for estimated phases */
	  sumseg_init(ss);		/* clear accumulators */
	  for ( spi = sp; spi; spi = spi->next )
	    {
	      est_phase = 0;
	      if ( fp->phasevalid[spi->run] )
		est_phase += fp->phase[spi->run];
	      if ( sp->clock != CLOCK_NTP )
		{
		  if ( fp->systematicvalid[spi->clock] )
		    est_phase += fp->systematic[spi->clock];
		  if ( doingint)
		    est_phase += clock_state[spi->clock].offset;
		}
	      sumseg_sum(spi, est_phase, ss, 0, 0);
	    }

	  if ( flags.debug >= 3 )
	    sumseg_print(finfo, ss);

	  /* perform quad fit */
	  est_phase = fit_param_calc(finfo, ss, fp, noage, 1);
	}
      if ( flags.debug >= 2 )
	fprintf(finfo,
		"aging %f ppm/day freq %f %f ppm phase %.1f us\n",
		fp->aging * 86400.0 * 2.0 * KK,
		(fp->freq + 2.0 * fp->aging * ss->ltime) * KK,
		(fp->freq + 2.0 * fp->aging * ss->htime) * KK,
		est_phase * KK);

      /* estimate phases of runs given aging/freq and systematic
	 phases are BEFORE systematics because if we must attribute
	 an offset to one or another, it is a phase first */
      for ( i = 0; i <= RUN_MAX(); i++ )
	phase[i] = phasevar[i] = wphase[i] = 0.0;
      for ( spi = sp; spi; spi = spi->next )
	{
	  fit_quality_calc(spi, fp, fq, 0, doingint, 0);
	  if ( flags.debug >= 2 )
	    fprintf(finfo,
		    "RUN/clock %d/%d  n=%d sumw=%.1f avg=%.1f us sdev=%.1f us\n",
		    spi->run, spi->clock, fq->n, fq->sumweights,
		    fq->offset * KK, fq->sdev * KK);
	  phase[spi->run] += fq->sumresid;
	  phasevar[spi->run] += fq->sumsqresid; 
	  wphase[spi->run] += fq->sumweights;
	}
      for ( i = 0; i <= RUN_MAX(); i++ )
	if ( wphase[i] > 0.0 )
	  {
	    off = phase[i] / wphase[i];
	    offabs = fabs(off);
	    if ( offabs > 0.0 )
	      {
		fp->phase[i] += off;
		fp->phasevalid[i] = 1;
		cum_error += offabs;
		fp->phasedev[i] = sqrt(phasevar[i] / wphase[i]);
		if ( flags.debug >= 2 )
		  fprintf(finfo,
			  "Run %d\tphase %.1f us\tincr %.1f us\tsdev %.1f us\n",
			  i, fp->phase[i] * KK, off * KK, fp->phasedev[i] * KK);
	      }
	  }

      /* estimate systematic errors given aging/freq and phases */
      for ( i = 0; i <= CLOCK_MAX(); i++ )
	systematic[i] = sysvar[i] = wsystematic[i] = 0.0;
      for ( spi = sp; spi; spi = spi->next )
	if ( spi->clock >= 0 && spi->clock <= CLOCK_MAX() )
	  {
	    fit_quality_calc(spi, fp, fq, 0, doingint, 0);
	    if ( flags.debug >= 2 )
	      fprintf(finfo,
		      "run/CLOCK %d/%d  n=%d sumw=%f avg=%.1f us sdev=%.1f us\n",
		      spi->run, spi->clock, fq->n, fq->sumweights,
		      fq->offset * KK, fq->sdev * KK);
	    systematic[spi->clock] += fq->sumresid;
	    sysvar[spi->clock] += fq->sumsqresid;
	    wsystematic[spi->clock] += fq->sumweights;
	  }
      for ( i = 0; i <= CLOCK_MAX(); i++ )
	if ( wsystematic[i] > 0.0 )
	  {
	    off = systematic[i] / wsystematic[i];
	    offabs = fabs(off);
	    if ( offabs > 0.0 )
	      {
		fp->systematic[i] += off;
		fp->sysdev[i] = sqrt(sysvar[i] / wsystematic[i]);
		fp->systematicvalid[i] = 1;
		cum_error += offabs;
		if ( flags.debug >= 2 )
		  fprintf(finfo,
			  "Clock %d\tphase %.1f us\tincr %.1f us\tsdev %.1f\t%s\n",
			  i, fp->systematic[i] * KK, off * KK, fp->sysdev[i] * KK,
			  clock_unparse(i));
	      }
	  }

      if ( flags.debug >= 1 )
	fprintf(finfo, "Iteration %2d CUM_ERROR %e noage %d\n",
		niter, cum_error, noage);

      if ( cum_error < 5E-7 )	/* 0.1 us good enough for now */
	{
	  /* if aging is bogus, disable it and try again */
	  if ( noage == 0 &&
	      (fp->aging < - AGING_BOGUS || fp->aging > AGING_BOGUS))
	    noage = 1;
	  else
	    break;
	}
    }

  /* calculate residuals for entire fit */
  fit_quality_calc(sp, fp, fq, 1, doingint, 0);

  fprintf(finfo, "dofit_segment_multiple: %d iterations %e CUM_ERROR\n",
	  niter, cum_error);
}

void
dofit_fitseg(FILE *finfo, fitseg_t *fs)
{
  sumseg_t ss;
  int nruns, nclocks;

  nruns = segment_count_runs(fs->sp);
  nclocks = segment_count_clocks(fs->sp);
  
  if ( nclocks <= 1 && nruns <= 1 )
    /* one clock and run only */
    dofit_segment_single(finfo, fs->sp, fs->fp, fs->fq);
  else
    if ( flags.noaging != 0 && nclocks <= 1 )
      /* linear fit, one clock, multiple runs */
      dofit_segment_multirun(finfo, fs->sp, fs->fp, fs->fq);
    else
      /* general case: aging or multiple clocks */
      dofit_segment_multiple(finfo, fs->sp, &ss, fs->fp, fs->fq, 0);

#ifdef DIAGNOSTIC
  if ( fs->fq->offset > 1.0E-9 || fs->fq->offset < -1.0E-9 )
    {
      segment_print_verbose(finfo, fs->sp, 0);
      fit_param_print(finfo, fs->fp);
      fit_quality_print(finfo, fs->fq);
    }
#endif
}

/* print results with aging, for comparison */
void
doaging_check_fitseg(FILE *finfo, fitseg_t *fs)
{
  fit_param_t *saved_fp, new_fp;
  fit_quality_t *saved_fq, new_fq;
  int ageflag;

  if ( flags.noaging != 2 )
    panic("doaging_check_fitseg");

  saved_fp = fs->fp;
  saved_fq = fs->fq;
  ageflag = flags.noaging;

  flags.noaging = 0;		/* allow aging */
  new_fp = *fs->fp;
  fs->fp = &new_fp;		/* start out with copy */
  bzero(&new_fq, sizeof(fit_quality_t));
  fs->fq = &new_fq;
  
  dofit_fitseg(finfo, fs);
  if ( fs->fp->aging != 0 )
    {
      fprintf(finfo, "WITH AGING:\n");
      fitseg_print(finfo, fs);
      fprintf(finfo,
	      "RESIDUAL WITH AGING: %6.1f us\n      WITHOUT AGING: %6.1f us\n",
	      fs->fq->sdev * KK, saved_fq->sdev * KK);
      if ( saved_fq->sdev > 0 )
	fprintf(finfo, "              RATIO: %.3f\n",
		fs->fq->sdev / saved_fq->sdev);
    }
  
  flags.noaging = ageflag;	/* restore */
  fs->fp = saved_fp;
  fs->fq = saved_fq;
}
