/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

FILE *
myopen3(char *name, char *suffix1, char *suffix2, int wflag, int req)
{
  FILE *fd;
  char fname[1024];
  char *buf;
  char *wmode = "w";

  if ( strlen(name) + strlen(suffix1) + strlen(suffix2) > 1000 )
    panic("filenames too long");

  sprintf(fname, "%s%s%s", name, suffix1, suffix2);
  if ( wflag == 2 )
    wmode = "a";
  fd = fopen(fname, wflag ? wmode : "r");
  if ( fd == NULL )
    if ( req )
      {
	fprintf(stderr, "Can't open %s for %s\n",
		fname, wflag ? "writing" : "reading");
	exit(-1);
      }
    else
      {
	fprintf(stderr, "WARNING: Couldn't open %s for %s\n",
		fname, wflag ? "writing" : "reading");
	return NULL;
      }

#ifdef HAVE_SETVBUF
  if ( req )
    {
      buf = (char *) malloc(32768);
#ifndef SETVBUF_REVERSED
      setvbuf(fd, buf, _IOFBF, 32768);
#else
      setvbuf(fd, _IOFBF, buf, 32768);
#endif
    }
#endif

  return fd;
}

FILE *
myopen2(char *name, char *suffix, int wflag, int req)
{
  return myopen3(name, suffix, "", wflag, req);
}

void
myclose(FILE *fd)
{
#ifdef __386BSD__NOTDEF
  unsigned char *buf;
#endif

  fflush(fd);
  fclose(fd);

#ifdef __386BSD__NOTDEF
  buf = fd->_bf._base;
  fclose(fd);
  free(buf);
#endif
}

void
input_dat(FILE *fin, FILE *finfo)
{
  dataset_t *datasetp;
  datapoint_t prev, input;
  char inputbuf[1024], clockbuf[1024];
  int i;			/* random loop indices */
  int firstpoint = 1;		/* first point of file */
  int firstpointrun = 0;	/* first point of run */
  int seenadjust = 0;		/* 1 if seen 1 ADJUST LINE */
  int run = 0;

  while ( ! feof (fin) )
    {
      /* read line and trim newline */
      {
	char *p;		/* random character pointer */
	
	inputbuf[0] = 0;
	errno = 0;
	
	if ( fgets(inputbuf, 1024, fin) == NULL )
	  {
	    if ( errno != 0 )
	      {
		perror("Reading input file");
		exit(-1);
	      }
	    continue;
	  }
	
	/* skip blank lines */
	if ( inputbuf[0] == 0 || inputbuf[0] == '\n' )
	  continue;
	
	/* trim newline */
	p = &inputbuf[strlen(inputbuf) - 1];
	if ( *p != '\n' )
	  panic("line too long");
	*p = '\0';
      }

      /* restart token? */
      if ( strncmp("RESTART", inputbuf, 7) == 0 )
	{
	  firstpointrun = 1;
	  seenadjust = 0;
	  continue;
	}

      /*  check for adjust log line */
      if ( strncmp("ADJUST", inputbuf, 6) == 0 )
	{
	  int temptype, temptau;
	  double tempaoffset, tempdrift, temprsadj;

	  i = sscanf(inputbuf,
		     "ADJUST %d %u %d %s %lf %lf %d %lf",
		     &input.pid, &input.time, &temptype, clockbuf,
		     &tempaoffset, &tempdrift, &temptau, &temprsadj);

	  if ( i != 8 )
	    {
	      fprintf(stderr, "Length %d: [%s]\n", i, inputbuf);
	      continue;
	    }

	  input.type = temptype;
	  input.tau = temptau;
	  input.aoffset = tempaoffset;
	  input.delay = tempdrift; /* DRIFT */
	  input.delay /= 1.024 * 1.024; /* convert drift to ppm */
	  input.rsadj = temprsadj;

	  seenadjust = 1;
	  input.observation = 0;
	}
      else if ( strncmp("OBSERVATION", inputbuf, 11) == 0 )
	{
	  int temppid;
	  double tempaoffset, tempdelay, temperror, temprsadj;

#ifdef NOTDEF
	  /* only use observations following at least 1 adjust */
	  if ( seenadjust == 0 )
	    {
	      fprintf(stderr, "before ADJUST %s\n", inputbuf);
	      continue;
	    }
#endif
	  i = sscanf(inputbuf,
		     "OBSERVATION %d %u %s %lf %lf %lf %lf",
		     &temppid, &input.time, clockbuf, &tempaoffset,
		     &tempdelay, &temperror, &temprsadj);

	  if ( i != 7 )
	    {
	      fprintf(stderr, "Length %d: [%s]\n", i, inputbuf);
	      continue;
	    }

	  input.observation = 1;
	  input.pid = temppid;
	  input.aoffset = tempaoffset;
	  input.delay = tempdelay;

	  /* use error as delay for directly attached refclocks */
	  if ( strncmp(clockbuf, "127.127", 7) == 0 )
	    input.delay = temperror;
#ifdef ADDERRORTODELAY
	  else
	    /* dont do this for now */
	    input.delay += temperror;
#endif
	  input.rsadj = temprsadj;

	  input.valid = VALID_OK;
	}
      else if ( strncmp("GPS", inputbuf, 3) == 0 )
	{
	  int tempoffset;
	  double temprsadj;
	
	  /* only use observations following at least 1 adjust */
	  if ( seenadjust == 0 )
	    {
	      fprintf(stderr, "before ADJUST %s\n", inputbuf);
	      continue;
	    }

	  i = sscanf(inputbuf, "%s %u %d %lf",
		     clockbuf, &input.time, &tempoffset, &temprsadj);

	  if ( i != 4 )
	    {
	      fprintf(stderr, "Length %d: [%s]\n", i, inputbuf);
	      continue;
	    }

	  if ( tempoffset > 500000 )
	    tempoffset -= 1000000;
	  input.aoffset = -1.0 * tempoffset / 1000000.0;

	  input.rsadj = temprsadj;
	  /* gps input is in unix time */
	  input.time = Tunix2ntp(input.time);
	  input.pid = prev.pid;
	  input.delay = 0.0;
	  input.observation = 1;

	  input.valid = VALID_OK;
	}
      else
	{
	  fprintf(stderr, "BOGUS INPUT: %s\n", inputbuf);
	  continue;
	}

      input.clock = clock_parse(clockbuf);
      input.valid = VALID_OK;
      input.simrsadj = input.rsadj;

      /* first point of entire file? */
      if ( firstpoint )
	/* if so, record time, to be used as local origin */
	{
	  firstpoint = 0;
	  firstpointrun = 1;
	
	  /* record time for local origin, both in ntp and unix scales */
	  globals.first_time = input.time;
	  globals.ufirst_time = Tntp2unix(globals.first_time);
	  input.time = 0;
	
	  fprintf(finfo, "RUN time origin = UNIX %u NTP %u (%s)\n",
		  globals.ufirst_time, globals.first_time,
		  mytime(globals.ufirst_time));
	}
      else
	/* not, so check for bogus time sequences */
	{
	  /* convert ntp time to local origin */
	  input.time = Tntp2run(input.time);
	
	  /* save last adjust, observation seen */
	  if ( ! input.observation )
	    globals.last_adjust = input.time;
	  if ( input.time > globals.last_time )
	    globals.last_time = input.time;

	  /* we do not store points that fail */
	  if ( input.time < prev.time )
	    {
	      fprintf(finfo,
		      "input/prev time %d/%d, input/prev rsadj %f/%f\n",
		      input.time, prev.time, input.rsadj, prev.rsadj);
	      continue;
	    }
	
	  if ( input.time == prev.time && input.rsadj != prev.rsadj )
	    {
	      /* don't complain about < 100 us differences */
	      if ( fabs(input.rsadj - prev.rsadj) > 0.000100 )
		fprintf(finfo,
			"input&prev time %d, input/prev rsadj %f/%f\n",
			input.time, input.rsadj, prev.rsadj);
	      continue;
	    }
	}

      if ( firstpointrun )
	/* first data point of a run */
	{
	  if ( (run = run_n++) >= NRUNS )
	    panic("too many runs");
	  input.run = run;
	  globals.runbeg[run] = input.time;
	  prev = input;
	  firstpointrun = 0;
	
	  fprintf(finfo,
		  "Run %d: pid %d start time=%s\n",
		  run, input.pid, mytime(Trun2unix(input.time)));
	}
      else
	/* not first point */
	input.run = run;

      if ( input.time > globals.runend[run] )
	globals.runend[run] = input.time;

      if ( input.clock < 0 || input.clock > CLOCK_MAX() )
	panic("illegal clock value at add to dataset point");

      /* add entry to raw dataset if observation */
      datasetp = input.observation == 1 ? &raw[input.clock] : ntpp;

      * (dp_new(datasetp)) = input;

      /* copy current entry to previous for reference ONLY IF ADJUSTMENT */
      if ( ! input.observation )
	prev = input;
    }

  fprintf(finfo, "Last time = RUN %u UNIX %u NTP %u (%s)\n",
	  globals.last_time,
	  Trun2unix(globals.last_time),
	  Trun2ntp(globals.last_time),
	  mytime(Trun2unix(globals.last_time)));
}

/*
 * make lines showing frequency of each fitseg
 * XXX break lines at run boundaries
 */
void
output_fitseg(FILE *fout, fitseg_t *head, int color)
{
  fitseg_t *fs;

  for ( fs = head; fs; fs = fs->next )
    {
      ntp_t begin = segment_multiple_begin_time(fs->sp);
      ntp_t end = segment_multiple_end_time(fs->sp);

      fprintf(fout, "line %u %.6f %u %.6f %d\n",
	      Trun2unix(begin),
	      fit_param_freq(fs->fp, begin),
	      Trun2unix(end),
	      fit_param_freq(fs->fp, end),
	      color);
    }
}

/*
 * make xplot files for clock
 * XXX include frequency estimates
 */
typedef struct output_control_s
{
  char *name;
  void *ptr;
  int flags;
} output_control_t;

#define PTR(x)  ((void *) (&((datapoint_t *)NULL)->x))
#ifdef __GNUC__
#define APPLYR(b, x) ( (real_t *) ((void *) (b) + ((int) ((void *) (x)))) )
#define APPLYI(b, x) ( (int *) ((void *) (b) + ((int) ((void *) (x)))) )
#else
#define APPLYR(b, x) ( (real_t *) ((char *) (b) + ((int) ((void *) (x)))) )
#define APPLYI(b, x) ( (int *) ((char *) (b) + ((int) ((void *) (x)))) )
#endif

#define FLAG_INT 0x1		/* value is int rather than real_t */
#define FLAG_BW 0x2		/* no colors */
#define FLAG_SIM 0x10		/* only if simulation */
#define FLAG_VERBOSE 0x20	/* only if simulation */
#define FLAG_NTP 0x100		/* only on ntp */
#define FLAG_RAW 0x200		/* only on raw */
#define FLAG_INTV 0x400		/* only if integrated valid */
#define FLAG_DRIFT 0x1000	/* output ntp drift lines */
#define FLAG_WEDGE 0x4000	/* wedge plot with delay on x */
#define FLAG_WEDGE_ROTATE 0x8000 /* delay going, coming on x, y respectively */

#define TFLAG_WEDGE (FLAG_RAW|FLAG_WEDGE)
#define TFLAG_WEDGE_ROTATE (TFLAG_WEDGE|FLAG_WEDGE_ROTATE)

static output_control_t output_control[] =
{
  { "rsadj", PTR(rsadj), FLAG_VERBOSE },
  { "uoffset", PTR(uoffset), FLAG_VERBOSE },

  { "aoffset", PTR(aoffset) },
  { "wedge_aoffset", PTR(aoffset), TFLAG_WEDGE },
  { "delay", PTR(delay), FLAG_RAW },

  { "frequency", PTR(delay), FLAG_NTP | FLAG_DRIFT| FLAG_BW },
  { "clock", PTR(clock), FLAG_NTP | FLAG_INT },
  { "tau", PTR(tau), FLAG_NTP | FLAG_INT },

  { "rsadjresid_piecewise", PTR(rsadjr[RSADJ_PQF]) },
  { "poffset_piecewise", PTR(poffset[RSADJ_PQF]) },
  { "wedge_poffset_piecewise", PTR(poffset[RSADJ_PQF]), TFLAG_WEDGE },
  { "wedge_rotate_poffset_piecewise", PTR(poffset[RSADJ_PQF]), TFLAG_WEDGE_ROTATE },

  { "rsadjresid_integrated", PTR(rsadjr[RSADJ_INT]), FLAG_INTV },
  { "poffset_integrated", PTR(poffset[RSADJ_INT]), FLAG_INTV },
  { "wedge_poffset_integrated", PTR(poffset[RSADJ_INT]), FLAG_INTV | TFLAG_WEDGE },

  { "rsadjresid_overall", PTR(rsadjr[RSADJ_OVERALL]) },
  { "poffset_overall", PTR(poffset[RSADJ_OVERALL]) },
  { "wedge_poffset_overall", PTR(poffset[RSADJ_OVERALL]), TFLAG_WEDGE },

  { "simaoffset", PTR(simaoffset), FLAG_SIM },
  { "wedge_simaoffset", PTR(simaoffset), FLAG_SIM|TFLAG_WEDGE },
  { "simrsadj", PTR(simrsadj), FLAG_SIM|FLAG_VERBOSE},
  { "simrsadjresid_piecewise", PTR(simrsadjr[RSADJ_PQF]), FLAG_SIM },
  { "simrsadjresid_integrated", PTR(simrsadjr[RSADJ_INT]), FLAG_SIM | FLAG_INTV },
};

#define OUTPUT_CONTROL_N ((sizeof(output_control) / sizeof(output_control_t)))

void
output_dat(int clock)
{
  char *cl = clock_unparse(clock);
  dataset_t *ds = clock2dataset(clock);
  char outdescrip[1024];
  int oc_n;
  int i;
  FILE *fout;

  if ( clock == CLOCK_NTP )
    sprintf(outdescrip, "ntp");
  else
    sprintf(outdescrip, "raw.%s", cl);

  /* loop through various output types (per datapoint) */
  for ( oc_n = 0; oc_n < OUTPUT_CONTROL_N; oc_n++ )
    {
      output_control_t *ocp = & output_control[oc_n];
      char outputname[1024];
      int fl = ocp->flags;

      if ( (fl & FLAG_VERBOSE) && ! flags.verbose )
	continue;

      if ( (fl & FLAG_SIM) && ! flags.simulate )
	continue;

      if ( (fl & FLAG_NTP) && ! (clock == CLOCK_NTP) )
	continue;

      if ( (fl & FLAG_RAW) &&  (clock == CLOCK_NTP) )
	continue;

      if ( (fl & FLAG_INTV) && (! flags.intvalid) )
	continue;

      if ( (fl & FLAG_WEDGE) && ( clock == CLOCK_NTP) )
	panic("output_dat: wedge for NTP data");

      /* skip wedge if delays equal */
      if ( (fl & FLAG_WEDGE)
	  && (clock_state[clock].low_delay == clock_state[clock].high_delay))
	continue;

      sprintf(outputname, "/xplot.%s-%s", ocp->name, outdescrip);
      fout = myopen2(datafiledir, outputname, 1, 1);
      fprintf(fout, "%s %s\ntitle\n%s:%s:%s\n",
	      fl & FLAG_WEDGE ? "dtime" : "timeval",
	      fl & FLAG_INT
	      ? "unsigned"
	      : (fl & FLAG_DRIFT ? "double" : "dtime"),
	      datahostname, cl, ocp->name);

      /* if drift, print NTP drift lines */
      if ( fl & FLAG_DRIFT )
	{
	  output_fitseg(fout, globals.fs_ntpoverall, 1);
	  output_fitseg(fout, globals.fs_ntp, 2);
	  output_fitseg(fout, globals.fs_integrated, 3);
	}

      /* if wedge, print wedge lines showing boundary of "correct" plot
       * show base, add, and mul delay lines as well */
      if ( (fl & (FLAG_RAW|FLAG_WEDGE))
	    == (FLAG_RAW|FLAG_WEDGE))
	{
	  double low = clock_state[clock].low_delay;
	  double high = clock_state[clock].high_delay;
	  double base = clock_state[clock].base_delay;
	  double add = clock_state[clock].add_delay;
	  double mul = clock_state[clock].mul_delay;
	  double offerr = (high - low)/2;

	  if ( fl & FLAG_WEDGE_ROTATE )
	    {
	      fprintf(fout, "line %.6f %.6f %.6f %.6f\n",
		      low/2, low/2, high/2, low/2);
	      fprintf(fout, "line %.6f %.6f %.6f %.6f\n",
		      low/2, low/2, low/2, high/2);
	    }
	  else
	    {
	      /* wedge */
	      fprintf(fout, "line %.6f 0 %.6f %.6f\n", low, high, offerr);
	      fprintf(fout, "line %.6f 0 %.6f %.6f\n", low, high, -offerr);
	      
	      /* delays used in calculation */
	      fprintf(fout, "line %.6f %.6f %.6f %.6f\n",
		      base, -offerr, base, offerr);
	      fprintf(fout, "line %.6f %.6f %.6f %.6f\n",
		      add, -offerr, add, offerr);
	      fprintf(fout, "line %.6f %.6f %.6f %.6f\n",
		      mul, -offerr, mul, offerr);
	    }
	}

      for ( i = 0; i < ds->n; i++ )
	{
	  datapoint_t *dp = ds_addrdp(ds, i);
	  int valid = dp->valid;
	  if (fl & FLAG_SIM)	/* used to be continue if not sim */
	    valid = dp->sim ? 0 : 2;
	  if ( fl & FLAG_INT )
	    fprintf(fout, ". %u %d %d\n",
		  Trun2unix(dp->time),
		  * APPLYI(dp, ocp->ptr),
		  fl & FLAG_BW ? 0 : valid);
	  else
	    if ( fl & FLAG_WEDGE )
	      {
		double delay = dp->delay;
		double offset = * APPLYR(dp, ocp->ptr);
		if ( fl & FLAG_WEDGE_ROTATE )
		  {
		    /* delaygoing */
		    delay = dp->delay/2 + offset;
		    /* delaycoming */
		    offset = dp->delay/2 - offset;
		  }
		fprintf(fout, ". %.6f %.6f %d\n",
			delay, offset,
			fl & FLAG_BW ? 0 : valid);
	      }
	    else
	      fprintf(fout, ". %u %.6f %d\n",
		      Trun2unix(dp->time),
		      * APPLYR(dp, ocp->ptr),
		      fl & FLAG_BW ? 0 : valid);
	}
      myclose(fout);
    }

  if ( clock == CLOCK_NTP )
    return;

  /* drift files for raw datasets
   * write lines for various fits in color
   * assume aggregation with ntp drift file if appropriate
   */
  {
    char outputname[1024];
    sprintf(outputname, "/xplot.%s-%s", "frequency", outdescrip);
    fout = myopen2(datafiledir, outputname, 1, 1);

    fprintf(fout, "timeval double\ntitle\n%s:%s:%s\n",
	    datahostname, cl, "frequency");

    output_fitseg(fout, globals.fs_rawoverall[clock], 1);
    output_fitseg(fout, globals.fs_raw[clock], 2);
    output_fitseg(fout, globals.fs_integrated, 3);

    myclose(fout);
  }
}
