/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

/* SEGMENT routines */
segment_t *
segment_new()
{
  segment_t *sp = NEW(segment_t);

  if ( sp == NULL )
    panic("segment_new");

  bzero(sp, sizeof(segment_t));

  sp->clock = 0;
  sp->run = 0;
  sp->ds = NULL;
  sp->first = 0;
  sp->n = 0;
  sp->samerun = sp->next = NULL;

  return sp;
}

void
segment_free(segment_t *sp)
{
  if ( sp == NULL )
    panic("segment_free of NULL sp");

  if ( sp->samerun ) segment_free( sp->samerun );
  if ( sp->next ) segment_free( sp->next );
  bzero(sp, sizeof(sp));
  free(sp);
}

void
segment_print(FILE *finfo, segment_t *sp, int crossclockruns)
{
  segment_t *spi, *spj;

  fprintf(finfo, "{");
  for ( spj = sp; spj; spj = crossclockruns ? spj->next : NULL)
    {
      fprintf(finfo, " [");
      for ( spi = spj; spi; spi = spi->samerun)
	fprintf(finfo, " (%d-%d)",
		spi->first, spi->first + spi->n - 1);
      fprintf(finfo, " ]");
    }
      fprintf(finfo, " }\n");
}

void
segment_print_verbose(FILE *finfo, segment_t *sp, int crossclockruns)
{
  segment_t *spi, *spj;
  int i;

  segment_print(finfo, sp, crossclockruns);

  for ( spj = sp; spj; spj = crossclockruns ? spj->next : NULL )
    for ( spi = spj; spi; spi = spi->samerun)
      for ( i = spi->first; i <= spi->first + spi->n - 1; i++)
	{
	  datapoint_t *dp = &spi->ds->dptr[i];
	  fprintf(finfo, "sp %x i %d t %u uo %f valid %d\n",
		  (unsigned int) spi, i, dp->time, dp->uoffset, dp->valid);
	}
}

ntp_t segment_multiple_begin_time(segment_t *sp)
{
  int btime, nbtime;

  btime = segment_begin_time(sp);
  for ( sp = sp->next; sp; sp = sp->next )
    {
      nbtime = segment_begin_time(sp);
      if ( nbtime < btime )
	btime = nbtime;
    }
  return btime;
}

ntp_t segment_multiple_end_time(segment_t *sp)
{
  int etime, netime;

  etime = 0;
  for ( ; sp; sp = sp->next )
    {
      netime = segment_end_time(sp, 0);
      if ( netime > etime )
	etime = netime;
    }
  return etime;
}

segment_t *
segment_copy(segment_t *sp)
{
  segment_t *spi;

  if ( sp == NULL )
    panic("Null segment_t * to segment_copy");

  if ( (spi = NEW(segment_t)) == NULL )
    panic("segment_copy malloc");

  *spi = *sp;
  if ( sp->samerun )
    spi->samerun = segment_copy(sp->samerun);
  if ( sp->next )
    sp->next = segment_copy(sp->next);

  return spi;
}

/* return last component segment
 * If 'crossclockruns' is true, follow ->next pointers */
segment_t *
segment_last(segment_t *sp, int crossclockruns)
{
  if ( crossclockruns )
    while ( sp->next )
      sp = sp->next;
  while ( sp->samerun )
    sp = sp->samerun;
  return sp;
}

/* return count of ALL points; traverse linked segments if multiple */
int
segment_count(segment_t *sp, int crossclockruns)
{
  int count;
  segment_t *spr, *spi;

  count = 0;
  for ( spr = sp; spr; spr = crossclockruns ? spr->next : NULL )
    for ( spi = spr; spi; spi = spi->samerun )
      count += spi->n;

  return count;
}

/* return count of valid points; traverse linked segments if multiple */
int
segment_count_valid(segment_t *sp, int crossclockruns)
{
  int count;
  segment_t *spr, *spi;
  int i;

  count = 0;
  for ( spr = sp; spr; spr = crossclockruns ? spr->next : NULL )
    for ( spi = spr; spi; spi = spi->samerun )
      for ( i = spi->first; i < spi->first + spi->n; i++ )
	if ( spi->ds->dptr[i].valid == VALID_OK )
	  count++;

  return count;
}

int
segment_count_clocks(segment_t *sp)
{
  int i;
  segment_t *sp1;
  int hasntp;
  int hasraw[NCLOCKS];
  int nclocks;

  hasntp = 0;
  for ( i = 0; i <= CLOCK_MAX(); i++ )
    hasraw[i] = 0;
  for ( sp1 = sp; sp1; sp1 = sp1->next )
    {
      int cl = sp1->clock;
      if ( cl == CLOCK_NTP )
	hasntp = 1;
      else
	if ( cl >= 0 && cl <= CLOCK_MAX() )
	  hasraw[cl] = 1;
	else
	  panic("segment_count_clocks: illegal clock");
    }

  nclocks = hasntp;
  for ( i = 0; i <= CLOCK_MAX(); i++ )
    nclocks += hasraw[i];
  return nclocks;
}

int
segment_count_runs(segment_t *sp)
{
  int i;
  segment_t *sp1;
  int hasrun[NRUNS];
  int nruns;

  for ( i = 0; i <= RUN_MAX(); i++ )
    hasrun[i] = 0;
  for ( sp1 = sp; sp1; sp1 = sp1->next )
    {
      int run = sp1->run;
      if ( run >= 0 && run <= RUN_MAX() )
	hasrun[run] = 1;
      else
	panic("segment_count_runs: illegal run");
    }

  for ( i = 0, nruns = 0; i <= RUN_MAX(); i++ )
    nruns += hasrun[i];
  return nruns;
}

/* append take_from-take_to from spapp to sp; return pointer to new object */
segment_t *
segment_append(FILE *finfo, segment_t *sp, segment_t *spapp,
	       int take_from, int take_to)
{
  segment_t *answer;
  segment_t *spr;		/* segment to right run */
  segment_t *sprp;		/* sprp->next should be spr */
  segment_t *spi;		/* segment to right place */
  segment_t *spip;		/* spip->samerun should be spi */
  segment_t *spk;		/* new one to insert */

  if ( sp == NULL ) panic("null 1st arg to segment_append");
  answer = sp;

  /* continguous after end? */
  spi = segment_last(sp, 1);
  if ( take_from == (spi->first + spi->n - 1) + 1 )
    {
      if ( spi->run != spapp->run )
	panic("multirun append");
      spi->n = take_to - spi->first + 1;
      return sp;
    }

  /* continguous before beginning? */
  spi = sp;
  if ( take_to == spi->first - 1 )
    {
      int newlast;

      if ( spi->run != spapp->run )
	panic("multirun append");
      newlast = spi->first + spi->n - 1;
      spi->first = take_from;
      spi->n = newlast - spi->first + 1;
      return sp;
    }

  fprintf(finfo, "warning - append not debugged\n");

  /* cdr down sp to right run, saving pointer to previous */
  for (spr = sp, sprp = NULL;
       spr->run < spapp->run && spr->next;
       sprp = spr, spr = spr->next )
    ;
  if ( spr->run != spapp->run )
    panic("segment_append: can't add new runs");

  /* cdr down to place to insert; insert before spi when done */
  for (spi = spr, spip = NULL;
       take_to < spi->first && spi->samerun;
       spip = spi, spi = spi->samerun )
    ;

  /* copy segment to insert */
  if ( (spk = (segment_t *) malloc(sizeof(segment_t))) == NULL )
    panic("segment_append: malloc");
  *spk = *spapp;
  spk->first = take_from;
  spk->n = take_to - take_from + 1;
  spk->samerun = NULL;
  spk->next = NULL;

  /* link into list */
  if ( spip != NULL )
    {
      /* in middle or end */
      spip->samerun = spk;
      spk->samerun = spi;
    }
  else
    {
      if ( spr != spi )
	panic ("append spi != spr");
      if ( sprp )
	sprp->next = spk;
      else
	answer = spk;
      spk->next = spi->next; spi->next = NULL;
      spk->samerun = spi;
    }
  return answer;
}

/* remove rem_from to rem_to from segment; one must match */
void
segment_trim(segment_t *sp, int rem_from, int rem_to)
{
  int adjust;

  if (sp->first > rem_from )
    panic("segment_trim: removing nonexistent points from bottom");
  if ( sp->first + sp->n - 1 < rem_to )
    panic("segment_trim: removing nonexistent points from top");
  if ( sp->first == rem_from )
    {
      /* begin=rem_from ..... rem_to keep ...... end */
      adjust = rem_to + 1 - rem_from;
      sp->n -= adjust;
      sp->first += adjust;
      return;
    }
  if ( sp->first + sp->n - 1 == rem_to )
    {
      /* begin ..... keep rem_from  ...... rem_to=end */
      adjust = rem_to + 1 - rem_from;
      sp->n -= adjust;
      return;
    }
  panic("segment_trim: chunk from middle!");
}

segment_t *
segment_fixup(segment_t *sp)
{
  /* XXX remove zero-length segments */
  return sp;
}

/* SUMSEG routines */
void
sumseg_print(FILE *finfo, sumseg_t *ss)
{
  fprintf(finfo,
	  "SUMSEG: n=%d first=%d sumweights=%f\nx=%f xx=%f xxx=%f xxxx=%f\ny=%f xy=%f xxy=%f\n",
	  ss->n, ss->first, ss->sumweights,
	  ss->Sx, ss->Sxx, ss->Sxxx, ss->Sxxxx,
	  ss->Sy, ss->Sxy, ss->Sxxy);
}

/* prepare ss for summing */
void
sumseg_init(sumseg_t *ss)
{
  ss->n = 0;
  ss->sumweights = 0.0;
  ss->Sx = ss->Sxx = ss->Sxxx = ss->Sxxxx = 0;
  ss->Sy = ss->Sxy = ss->Sxxy = 0;
  ss->first = 1;		/* indicates htime, ltime invalid */
}

/* sum points in a segment into ss, including ->samerun */
void
sumseg_sum(segment_t *sp, double offset, sumseg_t *ss,
	   int crossclockrun, int includeinvalid)
{
  segment_t *spr, *spi;
  int i;
  double weight;
  double x, y, tmp;

  /* iterate over the points */
  for ( spr = sp; spr; spr = crossclockrun ? spr->next : NULL )
    for ( spi = spr; spi; spi = spi->samerun )
      for ( i = spi->first; i < spi->first + spi->n; i++ )
	{
	  datapoint_t *dp = &spi->ds->dptr[i];
	
	  if ( includeinvalid == 0 && dp->valid != VALID_OK )
	    continue;
	
	  if ( ss->first )
	    {
	      ss->ltime = ss->htime = dp->time;
	      ss->first = 0;
	    }
	
	  if ( dp->time < ss->ltime ) ss->ltime = dp->time;
	  if ( dp->time > ss->htime ) ss->htime = dp->time;
	
	  ss->n++;

	  /* get per-clock weight */
	  if ( spi->clock > CLOCK_NTP )
	    weight = raw[spi->clock].weight;
	  else
	    weight = 1.0;

	  /* XXX per-point weights not yet implemented */

	  x = dp->time;
	  y = dp->uoffset;
	  y -= offset;		/* remove pre-estimated offset */
	
	  ss->sumweights += weight;
#ifndef DONT_OPTIMIZE_SUMSEG
	  tmp = weight * x; ss->Sx += tmp;
	  tmp *= x; ss->Sxx += tmp;
	  tmp *= x; ss->Sxxx += tmp;
	  tmp *= x; ss->Sxxxx += tmp;
	
	  tmp = weight * y; ss->Sy += tmp;
	  tmp *= x; ss->Sxy += tmp;
	  tmp *= x; ss->Sxxy += tmp;
#else
	  ss->Sx += weight * x;
	  ss->Sxx += weight * x * x;
	  ss->Sxxx += weight * x * x * x;
	  ss->Sxxxx += weight * x * x * x * x;
	
	  ss->Sy += weight * y;
	  ss->Sxy += weight * x * y;
	  ss->Sxxy += weight * x * x * y;
#endif
	}
}
