#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>
#include "anrep.h"
#include "metric.h"
#include "partree.h"
#include "symbol.h"
#include "score.h"
#include "motif.h"

/* "Line-at-a-time" io buffer and error recovery engine */

static jmp_buf errenv;

static int shell_argc;
static char **shell_argv;

static FILE *source;
static char *buffer, *scan;
static  int  echo, eof;
static  int  okeof, errsig;
static char *raw, *bufpt, *parpt;
static  int  slash, atsgn, atnum;

#define IO_ENGINE		\
FILE *isrc;			\
char *ibuf, *iscan;		\
int   iecho, ieof;		\
char *iraw, *ibufpt, *iparpt;	\
int   islash, iatsgn, iatnum;

#define PUSH_STRM(f,e)	\
isrc  = source;		\
ibuf  = buffer;		\
iscan = scan;		\
iecho = echo;		\
ieof  = eof;		\
iraw  = raw;		\
ibufpt = bufpt;		\
iparpt = parpt;		\
islash = slash;		\
iatsgn = atsgn;		\
iatnum = atnum;		\
initio(f,e);

#define POP_STRM	\
free(raw);		\
free(buffer);		\
atnum  = iatnum;	\
atsgn  = iatsgn;	\
slash  = islash;	\
parpt  = iparpt;	\
bufpt  = ibufpt;	\
raw    = iraw;		\
eof    = ieof;		\
echo   = iecho;		\
scan   = iscan;		\
buffer = ibuf;		\
source = isrc;


initio(src,prnt) FILE *src; int prnt;
{ source = src;
  if ((buffer = (char *) malloc((unsigned) (sizeof(char)*BUFMAX))) == 0)
    fatal("out of memory");
  *buffer = '\0';
  scan   = buffer-1;
  echo   = prnt;
  eof    = 0;
  if ((raw = (char *) malloc((unsigned) (sizeof(char)*BUFMAX))) == 0)
    fatal("out of memory");
  bufpt  = 0;
  parpt  = 0;
  slash  = 0;
  atsgn  = 0;
  atnum  = -1;
}

int fillbuf()
{ char *ender;
  int i;

  if (bufpt == 0)
    { if (fgets(raw,BUFMAX,source) == NULL)
        return (1);
      bufpt = raw;
    }
  scan = buffer;
  ender = buffer + (BUFMAX-1);
  if (parpt != 0)
    { while (*parpt != '\0')
        { if (scan >= ender) goto xferd;
          *scan++ = *parpt++;
        }
      parpt = 0;
    }
  while (*bufpt != '\0')
    { if (scan >= ender) goto xferd;
      if (atsgn)
        { atsgn = 0;
          if (isdigit(*bufpt))
            atnum = *bufpt++ - '0';
          else
            *scan++ = '@';
        }
      else if (atnum >= 0)
        if (isdigit(*bufpt))
          atnum = 10*atnum + (*bufpt++ - '0');
        else if (atnum < shell_argc)
          { parpt = shell_argv[atnum];
            atnum = -1;
            while (*parpt != '\0')
              { if (scan >= ender) goto xferd;
                *scan++ = *parpt++;
              }
            parpt = 0;
          }
        else
          atnum = -1;
      else if (!slash && *bufpt == '@')
        { atsgn = 1;
          bufpt += 1;
        }
      else
        { slash = (*bufpt == '\\');
          *scan++ = *bufpt++;
        }
    }
  bufpt = 0;
xferd:
  *scan = '\0';
  scan = buffer;
  return (0);
}

char *bump()
{ if (*++scan == '\0')
    { if (fillbuf())
        { eof = 1;
          if (okeof)
            { okeof = 0;
              longjmp(errenv,1);
            }
          else
            error("premature end-of-file",scan);
        }
      if (echo)
        { fputs(buffer,stdout);
          if (buffer[strlen(buffer)-1] != '\n')
            fputc('\n',stdout);
        }
      scan = buffer;
    }
  return (scan);
}

error(msg,pos) char *msg, *pos;
{ register int i;
  register char *s;

  errsig = 1;
  if (echo)
    { if (pos != 0)
        { i = 0;
          for (s = buffer; s < pos; s++)
            if (*s == '\t')
              i += (8 - i%8);
            else
              i += 1;
          fprintf(stdout,"%*s^\n",i,"");
        }
      else
        fprintf(stdout,"\n");
      fprintf(stdout,"*** error: %s\n\n",msg);
      okeof = 1;
      if (!eof && *scan != PCHAR)
        while (*bump() != PCHAR);
      okeof = 0;
    }
  else
    { fprintf(stdout,"\n*** error in redirected file\n\n");
      eof = 1;
    }
  freetree();
  longjmp(errenv,1);
}

warning(msg) char *msg;
{ fprintf(stdout,"\n*** warning: %s\n\n",msg);
  return (1);
}

fatal(msg) char *msg;
{ fprintf(stdout,"\n*** fatal: %s\n\n",msg);
  exit(1);
}

/* Follow set table for concatenation */

#define Ufol 0
#define Lfol 1
#define Pfol 2
#define Nfol 3

static int fset[SIGMA];

initfset()
{ register int i;

  for (i = 0; i < SIGMA; i++)
    fset[i] = Ufol;

  fset[ ']' ] = Lfol;
  fset[ ':' ] = Lfol;

  fset[ '"' ] = Pfol;

  fset[ '}' ] = Nfol;
  fset[ ')' ] = Nfol;
  fset[ '|' ] = Nfol;
  fset[ ',' ] = Nfol;
  fset[PCHAR] = Nfol;
}


/* Whitespace and keywords */

WHITE()
{ register int s;
  s = *scan;
  while (isspace(s))
    s = *bump();
}

ADVANCE()
{ register int s;
  s = *bump();
  while (isspace(s))
    s = *bump();
}

#define Kscore   0
#define Knumber  1
#define Kalpha   2
#define Kmotif   3
#define Knet     4
#define Ksearch  5
#define Kfor     6
#define Kshow    7
#define Kinclude 8
#define Kexec    9

static char *keyerrpt;

int KEYWORD()
{ register int s, key;

  keyerrpt = scan;
  s = *scan;
  if (s == 'a')
    { if (*bump() != 'l') goto fail;
      if (*bump() != 'p') goto fail;
      if (*bump() != 'h') goto fail;
      if (*bump() != 'a') goto fail;
      if (*bump() != 'b') goto fail;
      if (*bump() != 'e') goto fail;
      if (*bump() != 't') goto fail;
      key = Kalpha;
      goto found;
    }
  if (s == 'e')
    { if (*bump() != 'x') goto fail;
      if (*bump() != 'e') goto fail;
      if (*bump() != 'c') goto fail;
      if (*bump() != 'u') goto fail;
      if (*bump() != 't') goto fail;
      if (*bump() != 'e') goto fail;
      key = Kexec;
      goto found;
    }
  if (s == 'f')
    { if (*bump() != 'o') goto fail;
      if (*bump() != 'r') goto fail;
      key = Kfor;
      goto found;
    }
  if (s == 'i')
    { if (*bump() != 'n') goto fail;
      if (*bump() != 'c') goto fail;
      if (*bump() != 'l') goto fail;
      if (*bump() != 'u') goto fail;
      if (*bump() != 'd') goto fail;
      if (*bump() != 'e') goto fail;
      key = Kinclude;
      goto found;
    }
  if (s == 'm')
    { if (*bump() != 'o') goto fail;
      if (*bump() != 't') goto fail;
      if (*bump() != 'i') goto fail;
      if (*bump() != 'f') goto fail;
      key = Kmotif;
      goto found;
    }
  if (s == 'n')
    { s = *bump();
      if (s == 'e')
        { if (*bump() != 't') goto fail;
          key = Knet;
          goto found;
        }
      if (s != 'u') goto fail;
      if (*bump() != 'm') goto fail;
      if (*bump() != 'b') goto fail;
      if (*bump() != 'e') goto fail;
      if (*bump() != 'r') goto fail;
      key = Knumber;
      goto found;
    }
  if (s == 's')
    { s = *bump();
      if (s == 'c')
        { if (*bump() != 'o') goto fail;
          if (*bump() != 'r') goto fail;
          if (*bump() != 'e') goto fail;
          key = Kscore;
          goto found;
        }
      if (s == 'e')
        { if (*bump() != 'a') goto fail;
          if (*bump() != 'r') goto fail;
          if (*bump() != 'c') goto fail;
          if (*bump() != 'h') goto fail;
          key = Ksearch;
          goto found;
        }
      if (s != 'h') goto fail;
      if (*bump() != 'o') goto fail;
      if (*bump() != 'w') goto fail;
      key = Kshow;
      goto found;
    }
fail:
  error("keyword expected",keyerrpt);
found:
  return (key);
}


/* Identifiers, descriptors, and type checking */

static char *iderrpt;

FILE *FILEID()
{ register char *loc, buf[BUFMAX];
  register int s;
  FILE *unit;

  iderrpt = scan;
  loc = buf;
  s = *scan;
  do
    { *loc++ = s;
      s = *bump();
    }
  while (!isspace(s)  &&  s != ';');
  *loc = '\0';
  if ((unit = fopen(buf,"r")) == NULL)
    error("can't open file",iderrpt);
  WHITE();
  return (unit);
}

struct entry *ID()
{ register char *loc, buf[BUFMAX];
  register int len, s;

  iderrpt = scan;
  loc = buf;
  s = *scan;
  if ( ! isalpha(s) && s != '_')
    error("identifier expected",scan);
  *loc++ = s;
  s = *bump();
  while (isalnum(s) || s == '_')
    { *loc++ = s;
      s = *bump();
    }
  len = loc-buf;
  return (add(buf,len));
}

struct node *ISTYPE(e,t) register struct entry *e; register int t;
{ if (e->param)
    { if (e->ptype == Tnil || (e->ptype == Tflp && t == Tint))
        e->ptype = t;
      else if (e->ptype != t && (e->ptype != Tint || t != Tflp))
        error("formal param is inconsistently used",iderrpt);
      WHITE();
      return (ileaf(Lfp,e->pnum));
    }
  if (e->type != t && (e->type != Tint || t != Tflp))
    { if (e->type == Tnil)
        error("id is undefined",iderrpt);
      error("id is not of correct type",iderrpt);
    }
  WHITE();
  return (e->tree);
}

DEFVAR(e,ty,nf,prm,p)
struct entry *e;
int ty, nf;
struct entry **prm;
struct node *p;
{ p = newref(p);
  if (e->type != Tnil)
    { decref(e->tree);
      if (e->nform > 0) free((char *) (e->tform));
    }
  e->type = ty;
  e->tree = p;
  keeptree();
  e->nform = nf;
  if (nf > 0)
    if ((e->tform = (int *) malloc((unsigned) (sizeof(int)*nf))) == 0)
      fatal("out of memory");
  while (nf > 0)
    { nf -= 1;
      e->tform[nf] = prm[nf]->ptype;
      prm[nf]->param = 0;
    }
}


/* LL(1) scanners for "basic" phrases */

char *STRING(term) char term;
{ register char *loc;
  register int s, t;

  static char *buf = 0, *bufend, *tmp;
  static int buflen;

  if (buf == 0)
    { buflen = 20;
      buf = (char *) malloc(sizeof(char)*buflen);
      if (buf == NULL)
        fatal("out of memory");
      bufend = buf + (buflen-10);
    }

  t = term;
  s = *bump();
  loc = buf;
  while (s != t)
    { if (loc > bufend)
        { buflen = 1.5*buflen;
          tmp = (char *) malloc(sizeof(char)*buflen);
          if (tmp == NULL)
            fatal("out of memory");
          *loc = '\0';
          strcpy(tmp,buf);
          loc = tmp + (loc-buf);
          free((char *) buf);
          buf = tmp;
          bufend = buf + (buflen-10);
        }
      *loc++ = s;
      if (s == '\\')
        { s = *bump();
          *loc++ = s;
        }
      s = *bump();
    }
  *loc = '\0';
  ADVANCE();
  return (buf);
}

struct node *SYMB()
{ register int s;

  s = *scan;
  if (s == '\\')
    { s = *bump();
      if (s == PCHAR)
        error("premature end of statement",scan);
    }
  else if ( ! isalpha(s))
    error("symbol expected",scan);
  ADVANCE();
  return (ileaf(Lsymb,s));
}

struct node *IFCON(t) register int t;
{ register int sign, num, frc, s;
  float rat;
  char *errpt;

  errpt = scan;		/* Scan sign */
  s = *scan;
  sign = 0;
  if (s == '+')
    sign = 1;
  else if (s == '-')
    sign = -1;
  if (sign != 0)
    s = *bump();

  num = -1;		/* Scan integer part */
  if (s != '.')
    { if ( ! isdigit(s))
        if (sign != 0)
          { WHITE();
            return (ileaf(Lint,sign));
          }
        else
          error("number expected",errpt);
      num = s-'0';
      for (s = *bump(); isdigit(s); s = *bump())
        num = 10*num + (s-'0');
      if (s != '.' && s != 'E' && s != 'e')
        { WHITE();
          if (sign < 0)
            num = -num;
          return (ileaf(Lint,num));
        }
    }
  if (t == Tint)
    error("integer expected",errpt);

  if (s == '.')		/* Scan fractional part */
    { frc = 0;
      rat = 1.;
      for (s = *bump(); isdigit(s); s = *bump())
        { frc  = 10*frc + (s-'0');
          rat *= 10.;
        }
      if (num < 0)
        if (rat < 10.)
          error("number expected",errpt);
        else
          num = 0;
      rat = num + frc/rat;
    }
  else
    rat = num;
  if (sign < 0)
    rat = -rat;

  if (s == 'E' || s == 'e')	/* Scan exponent */
    { s = *bump();
      if (s == '+')
        { sign =  1; s = *bump(); }
      else if (s == '-')
        { sign = -1; s = *bump(); }
      else
        sign = 1;
      if (!isdigit(s))
        error("exponent missing",scan);
      num = s-'0';
      for (s = *bump(); isdigit(s); s = *bump())
        num = 10*num + (s-'0');
      if (sign < 0)
        while (0 < num--)
          rat /= 10.;
      else
        while (0 < num--)
          rat *= 10.;
    }

  WHITE();
  return (fleaf(Lflp,rat));
}

struct node *ACON()
{ register struct node *p;
  if (*scan != '[')
    error("alphabet constant expected",scan);
  ADVANCE();
  p = 0;
  do { p = binary(Acon,p,SYMB()); } while (*scan != ']');
  ADVANCE();
  return (p);
}

struct node *SET()
{ register int s;
  register struct node *p;
  p = 0;
  s = *scan;
  if (s == '[')
    { ADVANCE();
      do
        { if (*scan == ECHAR)
            { ADVANCE();
              p = binary(Acon,p,ileaf(Ldel,0));
            }
          else
            p = binary(Acon,p,SYMB());
        }
      while (*scan != ']');
      ADVANCE();
      return (p);
    }
  if (s == ECHAR)
    { ADVANCE();
      return (binary(Acon,p,ileaf(Ldel,0)));
    }
  if (isalpha(s) || s == '\\')
    return (binary(Acon,p,SYMB()));
  error("set constant expected",scan);
}

struct node *ALP()
{ register int s;
  s = *scan;
  if (s == '[')
    return (ACON());
  if (isalpha(s) || s == '_')
    return (ISTYPE(ID(),Talp));
  error("alphabet expected",scan);
}

struct node *NUM(t) register int t;
{ register int s;
  s = *scan;
  if (s == '+' || s == '-' || s == '.' || isdigit(s))
    return (IFCON(t));
  if (s == '_' || isalpha(s))
    return (ISTYPE(ID(),t));
  error("integer expected",scan);
}

struct node *REF(k) int k;
{ register struct node *p, *q;
  register struct entry *e;
  register int cnt, j;
  struct node *NET(), *MOT();

  e = ID();
  p = ISTYPE(e,k);
  q = 0;
  if (*scan == '{')
    { if (p->label == Lfp)
        error("referent cannot be a parameter",scan);
      cnt = 0;
      do
        { if (e->nform <= cnt)
            error("too many arguments",scan);
          ADVANCE();
          j = e->tform[cnt];
          if (j == Tnet)
            q = binary(j,q,NET());
          else if (j == Tmot)
            q = binary(j,q,MOT());
          else if (j == Tint || j == Tflp)
            q = binary(j,q,NUM(j));
          else  /* j == Tmet */
            q = binary(j,q,ISTYPE(ID(),j));
          cnt += 1;
        }
      while (*scan == ',');
      if (*scan != '}')
        error("} expected",scan);
      if (e->nform > cnt)
        error("too few arguments",scan);
      ADVANCE();
    }
  else if (e->nform > 0)
    error("too few arguments",scan);
  if (k == Tmot)
    return (binary(Mref,q,p));
  else
    return (binary(Nref,q,p));
}


/* LL(1) parsing routines for ATOMS */

struct node *LTERM()
{ register int s;
  register struct node *p;
  s = *scan;
  if (s == ICHAR)
    { ADVANCE();
      return (ileaf(Lins,0));
    }
  if (s == DCHAR)
    { ADVANCE();
      return (ileaf(Ldel,0));
    }
  if (s == SCHAR)
    { ADVANCE();
      return (ileaf(Lsub,0));
    }
  if (isalpha(s) || s == '\\')
    return (SYMB());
  error("literal expected",scan);
}

struct node *LCON()
{ register int s;
  register struct node *p;
  p = LTERM();
  s = *scan;
  while (fset[s] < Lfol)
    { p = binary(Lcon,p,LTERM());
      s = *scan;
    }
  return (p);
}

struct node *LWGT()
{ register struct node *p;
  p = LCON();
  if (*scan == ':')
    { ADVANCE();
      p = binary(Lwgt,p,NUM(Tflp));
    }
  return (p);
}

struct node *LIT()
{ register struct node *p;
  p = LWGT();
  while (*scan != ']')
    p = binary(Lcon,p,LWGT());
  return (p);
}


/* LL(1) parsing routines for MOTIFS */

struct node *PTERM()
{ register struct node *p, *q;
  struct node *PAT();
  register int k;
  switch (*scan)
  { case '(':
      ADVANCE();
      p = PAT();
      if (*scan != ')')
        error("unbalanced parens in motif",scan);
      ADVANCE();
      return (p);
    case '.':
      ADVANCE();
      return (ileaf(Ldot,0));
    case '[':
      p = q = 0;
      ADVANCE();
      if (*scan == '=')
        { ADVANCE();
          k = Ldef;
        }
      else if (*scan == '~')
        { ADVANCE();
          k = Lavg;
        }
      else if (*scan == '<')
        { ADVANCE();
          k = Lmin;
        }
      else if (*scan == '>')
        { ADVANCE();
          k = Lmax;
        }
      else
        { k = Lmax;
          q = binary(Lcon,ileaf(Lins,0),ileaf(Ldel,0));
        }
      if (q == 0)
        p = binary(k,(struct node *) 0,LIT());
      else
        p = binary(k,(struct node *) 0,binary(Lcon,LIT(),q));
      if (*scan != ']')
        error("] expected",scan);
      ADVANCE();
      return (p);
    default:
      if (isalpha(*scan) || *scan == '\\')
        return (SYMB());
      error("motif operand expected",scan);
  }
}

struct node *POPT()
{ register struct node *p;
  p = PTERM();
  while (*scan == ':'  ||  *scan == '!'  ||  *scan == '?')
    if (*scan == ':')
      { ADVANCE();
        p = binary(Mwgt,p,NUM(Tflp));
      }
    else if (*scan == '!')
      { ADVANCE();
        p = binary(Mmet,p,ISTYPE(ID(),Tmet));
      }
    else
      { ADVANCE();
        p = binary(Mopt,p,(struct node *) 0);
      }
  return (p);
}

struct node *PCON()
{ register int s;
  register struct node *p;
  p = POPT();
  s = *scan;
  while (fset[s] < Pfol)
    { p = binary(Mcon,p,POPT());
      s = *scan;
    }
  return (p);
}

struct node *PAT()
{ register struct node *p;
  p = PCON();
  while (*scan == '|')
    { ADVANCE();
      p = binary(Mor,p,PCON());
    }
  return (p);
}

struct node *MTERM()
{ register struct node *p;
  struct node *MOT();
  switch (*scan)
  { case '(':
      ADVANCE();
      p = MOT();
      if (*scan != ')')
        error("unbalanced parens in motif",scan);
      ADVANCE();
      return (p);
    case '"':
      ADVANCE();
      p = PAT();
      if (*scan != '"')
        error("\" expected",scan);
      ADVANCE();
      return (p);
    default:
      return (REF(Tmot));
  }
}

struct node *MOPT()
{ register struct node *p;
  p = MTERM();
  while (*scan == ':'  ||  *scan == '!'  ||  *scan == '?')
    if (*scan == ':')
      { ADVANCE();
        p = binary(Mwgt,p,NUM(Tflp));
      }
    else if (*scan == '!')
      { ADVANCE();
        p = binary(Mmet,p,ISTYPE(ID(),Tmet));
      }
    else
      { ADVANCE();
        p = binary(Mopt,p,(struct node *) 0);
      }
  return (p);
}

struct node *MCON()
{ register int s;
  register struct node *p;
  p = MOPT();
  s = *scan;
  while (fset[s] < Nfol)
    { p = binary(Mcon,p,MOPT());
      s = *scan;
    }
  return (p);
}

struct node *MOT()
{ register struct node *p;
  p = MCON();
  while (*scan == '|')
    { ADVANCE();
      p = binary(Mor,p,MCON());
    }
  return (p);
}


/* LL(1) parsing routines for NETWORKS */

struct node *NTERM()
{ register struct node *p;
  struct node *NET();
  switch (*scan)
  { case '(':
      ADVANCE();
      p = NET();
      if (*scan != ')')
        error("unbalanced parens in network",scan);
      ADVANCE();
      return (p);
    case '<':
      ADVANCE();
      p = NUM(Tint);
      if (*scan == ',')
        { ADVANCE();
          p = binary(Nspacer,p,NUM(Tint));
        }
      else
        p = binary(Nspacer,p,p);
      if (*scan != '>')
        error("> expected",scan);
      ADVANCE();
      return (p);
    case '{':
      ADVANCE();
      p = MOT();
      if (*scan == ',')
        ADVANCE();
      else
        error(", expected",scan);
      p = binary(Nmot,p,NUM(Tflp));
      if (*scan != '}')
        error("} expected",scan);
      ADVANCE();
      return (p);
    default:
      if (isalpha(*scan) || *scan == '_')
        return (REF(Tnet));
      error("network operand expected",scan);
  }
}

struct node *NOPT()
{ register struct node *p;
  p = NTERM();
  while (*scan == '?')
    { ADVANCE();
      p = binary(Nopt,p,(struct node *) 0);
    }
  return (p);
}

struct node *NCON()
{ register int s;
  register struct node *p;
  p = NOPT();
  s = *scan;
  while (fset[s] < Nfol)
    { p = binary(Ncon,p,NOPT());
      s = *scan;
    }
  return (p);
}

struct node *NET()
{ register struct node *p;
  p = NCON();
  while (*scan == '|')
    { ADVANCE();
      p = binary(Nor,p,NCON());
    }
  return (p);
}


/* Special parses for metric DECLARATIONS */

C_EXPR()
{ register int s;
  s = *scan;
  while (s != PCHAR && s != '#')
    { if (s == '\'')
        { DEPOSIT(s);
          s = *bump();
          if (s == '$')
            { DEPOSIT('\\');
              DEPOSIT('0');
              s = *bump();
            }
          while (s != '\'')
            { DEPOSIT(s);
              s = *bump();
            }
        }
      DEPOSIT(s);
      s = *bump();
    }
}

DECL()
{ register struct node *p, *q, *r;
  char o;
  if (*scan == '<')
    { ADVANCE();
      p = SET();
      if (*scan == '>')
        { ADVANCE();
          o = ',';
          q = NULL;
        }
      else
        { if (*scan == '.')
            { ADVANCE();
              o = '.';
              q = SET();
            }
          else if (*scan == ',')
            { ADVANCE();
              o = ',';
              q = SET();
            }
          else
            error(", or . expected",scan);
          if (*scan != '>')
            error ("> expected",scan);
          ADVANCE();
        }
      INIT_DECL(p,o,q);
      if (*scan != '#')
        error("# expected",scan);
      ADVANCE();
      r = IFCON(Tflp);
      if (*scan != PCHAR)
        error("terminator expected",scan);
      FINI_DECL(r);
    }
  else
    { INIT_DECL(NULL,' ',NULL);
      C_EXPR();
      GLUE_DECL();
      if (*scan != '#')
        error("# expected",scan);
      ADVANCE();
      C_EXPR();
      if (*scan != PCHAR)
        error("terminator expected",scan);
      FINI_DECL(NULL);
    }
  freetree();
}


/* LL(1) parser for STATEMENTS */

metric cmet;

static int enform;
static char pmessage[100];
static struct entry *prm[PARMAX];

STMT()
{ register struct node *p;
  register struct entry *e, *f;
  register int key, i;
  FILE *ifile;
  char *cmd, *fmt;
  int scoerr;

  if (setjmp(errenv)) return;
  okeof = 1;
  ADVANCE();
  while (*scan == '#')
    { while (*bump() != '\n')
        continue;
      WHITE();
    }
  okeof = 0;
  key = KEYWORD();
  if (key == Kfor)
    error("statement keyword expected",keyerrpt);
  ADVANCE();
  switch (key)
  { case Kscore:
      e = ID();
      WHITE();
      if (*scan == PCHAR)
        { if (e->type == Tnil)
            error("id is undefined",iderrpt);
          if (e->type != Tmet)
            error("id is not of type metric",iderrpt);
        }
      else
        { if (*scan != '=')
            error("= expected",scan);
          ADVANCE();
          p = newref(ALP());
          keeptree();
          if (*scan != '{')
            error("{ expected",scan);
          INIT_MET(p);
          scoerr = 0;
          if (setjmp(errenv))
            { if (!echo || eof)
                { CLEAN_MET(1);
                  decref(p);
                  return;
                }
              scoerr = 1;
            }
          ADVANCE();
          while (*scan != '}')
            { DECL();
              ADVANCE();
            }
          decref(p);
          if (setjmp(errenv))
            { CLEAN_MET(1);
              return;
            }
          ADVANCE();
          if (*scan != PCHAR)
            error("terminator expected",scan);
          if (scoerr)
            { CLEAN_MET(1);
              return;
            }
          DEFVAR(e,Tmet,0,prm,mleaf(Lmet,FINI_MET()));
        }
      cmet = MVAL(e->tree);
      return;
    case Knumber:
    case Kalpha:
    case Kmotif:
    case Knet:
      e = ID();
      WHITE();
      enform = 0;
      if (setjmp(errenv))
        { while (enform > 0)
            { enform -= 1;
              prm[enform]->param = 0;
            }
          return;
        }
      if (*scan == '=')
        ADVANCE();
      else
        { if (key == Knumber || key == Kalpha)
            error("= expected",scan);
          if (*scan != '{')
            error("{ or = expected",scan);
          do
            { ADVANCE();
              f = ID();
              if (f->param)
                error("duplicate parameter name",iderrpt);
              if (enform >= PARMAX)
                error("too many parameters",iderrpt);
              WHITE();
              f->param = 1;
              f->ptype = Tnil;
              f->pnum  = enform;
              prm[enform++] = f;
            }
          while (*scan == ',');
          if (*scan != '}')
            error("} expected",scan);
          ADVANCE();
          if (*scan != '=')
            error("= expected",scan);
          ADVANCE();
        }
      switch (key)
      { case Knumber:
          p = NUM(Tflp);
          if (p->label == Lint)
            key = Tint;
          else
            key = Tflp;
          break;
        case Kalpha:
          p = ALP();
          key = Talp;
          break;
        case Kmotif:
          p = MOT();
          key = Tmot;
          break;
        case Knet:
          p = NET();
          key = Tnet;
      }
      if (*scan != PCHAR)
        error("terminator expected",scan);
      for (i = 0; i < enform; i++)
        if (prm[i]->ptype == Tnil)
          { sprintf(pmessage,"parameter `%.10s' never used",prm[i]->name);
            error(pmessage,scan);
          }
      DEFVAR(e,key,enform,prm,p);
      return;
    case Ksearch:
      if (*scan == '(')
        fmt = STRING(')');
      else
        fmt = "";
      ifile = FILEID();
      if (KEYWORD() != Kfor)
        error("for expected",keyerrpt);
      ADVANCE();
      p = NET();
      if (*scan != PCHAR)
        error("terminator expected",scan);
      if (Start_scan(fmt,ifile))
        { fclose(ifile);
          error("unsupported file format",0);
        }
      if (setjmp(errenv))
        { fclose(ifile);
          CleanSearch();
          return;
        }
      Search(p);
      fclose(ifile);
      decref(p);
      return;
    case Kshow:
      e = ID();
      if (e->type == Tnil)
        error("id undefined",iderrpt);
      WHITE();
      if (*scan != PCHAR)
        error("terminator expected",scan);
      showvar(e);
      return;
    case Kinclude:
      ifile = FILEID();
      goto redirect;
    case Kexec:
      if (*scan != '\"')
        error("quoted string expected",scan);
      cmd = STRING('\"');
      strcpy(cmd+strlen(cmd)," >..x");
      if (system(cmd))
        { unlink("..x");
          error("command failed",0);
        }
      if ((ifile = fopen("..x","r")) == NULL)
        fatal("cannot open system file");
    redirect:
      if (setjmp(errenv))
        { fclose(ifile);
          if (key == Kexec) unlink("..x");
          return;
        }
      if (*scan != PCHAR)
        error("terminator expected",scan);
      { IO_ENGINE
        PUSH_STRM(ifile,0)
        do STMT(); while ( ! (eof || errsig));
        POP_STRM
      }
      fclose(ifile);
      if (key == Kexec) unlink("..x");
      return;
  }
}

main(argc,argv) int argc; char *argv[];
{ printf(VERSION);

  shell_argc = argc;
  shell_argv = argv;
  inithash();
  initfset();
  initio(stdin,1);
  cmet = init_met();
  strcpy(pmessage,INIT_MATRIX);
  DEFVAR(add(pmessage,strlen(INIT_MATRIX)),Tmet,0,0,mleaf(Lmet,cmet));
  INIT_CONTEXT();
  do 
    { errsig = 0;
      STMT();
    }
  while (!eof);
}
