#include <stdio.h>
#include <string.h>
#include "anrep.h"
#include "metric.h"
#include "partree.h"

#define RAND_MAX 017777777777L


/* Context stack for references and formal parameters */

struct context { struct context *cspred, *cntxt;
                 struct node   **cparam;
               };

static struct context bottom, *topcon;

INIT_CONTEXT()
{ bottom.cspred = bottom.cntxt = 0;
  bottom.cparam = 0;
  topcon = &bottom;
}

struct node *PUSH_CONTEXT(p) register struct node *p;
{ register struct node *q;
  register struct context *c;
  register int n;

  if ((c = (struct context *) malloc((unsigned)sizeof(struct context))) == 0)
    fatal("evaluation stack out of memory");

  if (p->label != Lfp && p->rgt->label != Lfp)

    { c->cspred = c->cntxt = topcon;
      n = 0;
      for (q = p->lft; q != 0; q = q->lft)
        n += 1;
      if (n > 0)
        { if ((c->cparam = (struct node **) malloc((unsigned)(sizeof(struct node *)*n))) == 0)
            fatal("evaluation stack out of memory");
          for (q = p->lft; q != 0; q = q->lft)
            c->cparam[--n] = q->rgt;
        }
      else
        c->cparam = 0;
      p = p->rgt;
    }

  else
    { if (p->label != Lfp)
        p = p->rgt;
      if (topcon->cntxt == 0)
        fatal("system bug - contact supplier(M1)");
      c->cspred = topcon;
      c->cntxt  = topcon->cntxt->cntxt;
      c->cparam = topcon->cntxt->cparam;
      p = topcon->cparam[IVAL(p)];
    }

  topcon = c;
  return (p);
}


POP_CONTEXT()
{ register struct context *c;
  if (topcon == &bottom)
    fatal("system bug - contact supplier(M2)");
  c = topcon;
  topcon = c->cspred;
  if (c->cntxt == c->cspred && c->cparam != 0)
    free((char *)(c->cparam));
  free((char *)c);
}

CLEAN_CONTEXT()
{ while (topcon != &bottom)
    POP_CONTEXT();
}


/* Number, integer, and metric evaluations */

float NUMEVAL(p) register struct node *p;
{ register float n;
  if (p->label == Lfp)
    { p = PUSH_CONTEXT(p);
      n = NUMEVAL(p);
      POP_CONTEXT();
    }
  else if (p->label == Lint)
    n = IVAL(p);
  else if (p->label == Lflp)
    n = FVAL(p);
  else
    fatal("system bug - contact supplier(M3)");
  return (n);
}

int INTEVAL(p) register struct node *p;
{ register int n;
  if (p->label == Lfp)
    { p = PUSH_CONTEXT(p);
      n = INTEVAL(p);
      POP_CONTEXT();
    }
  else if (p->label == Lint)
    n = IVAL(p);
  else
    fatal("system bug - contact supplier(M4)");
  return (n);
}

metric METEVAL(p) register struct node *p;
{ metric n;
  if (p->label == Lfp)
    { p = PUSH_CONTEXT(p);
      n = METEVAL(p);
      POP_CONTEXT();
    }
  else if (p->label == Lmet)
    n = MVAL(p);
  else
    fatal("system bug - contact supplier(M5)");
  return (n);
}


/* Character class evaluation */

static int    Alphabet[SIGMA];
static int    Qsym[SIGMA], Qdel, Qins, Qsub;

static float  thr;
static float  Vdotsub[SIGMA+1], Vdotins[SIGMA+1];
static float *Vins, *Vsub;
static float  Wsym[SIGMA], Wdel, Wins, Wsub, wgt_attr;

static metric met_attr;

static munch_class(p,w) register struct node *p; register float w;
{ register int c;

  switch (p->label)
  { case Lcon:
      munch_class(p->lft,w);
      munch_class(p->rgt,w);
      break;
    case Lwgt:
      munch_class(p->lft,w*NUMEVAL(p->rgt));
      break;
    case Lsymb:
      c = MAP(IVAL(p));
      if (c == 0)
        { char buf[100];
          sprintf(buf,"the motif symbol '%c' is not in the alphabet",IVAL(p));
          error(buf,0);
        }
      else
        { Qsym[c] = 1;
          Wsym[c] = w;
        }
      break;
    case Lsub:
      Qsub = 1;
      Wsub = w;
      break;
    case Lins:
      Qins = 1;
      Wins = w;
      break;
    case Ldel:
      Qdel = 1;
      Wdel = w;
      break;
  }
}

static make_scores(p) struct node *p;
{ register int i, j;
  register float *w, *v, x, y;
  register struct node *q;
  float wgtsum;
  int   hasem;

  for (q = p->lft; q != 0; q = q->lft)
    if (MVAL(q) == met_attr)
      { Vins = VINS(q);
        Vsub = VSUB(q);
        q->rcnt += 1;
        return;
      }

  if ((Vins = (float *) malloc((unsigned)(sizeof(float)*(ALPHA+1)))) == 0)
    fatal("out of memory");
  if ((Vsub = (float *) malloc((unsigned)(sizeof(float)*(ALPHA+1)))) == 0)
    fatal("out of memory");
  q = sleaf(Lvec,met_attr,Vins,Vsub);
  q->lft  = p->lft;
  p->lft  = q;
  q->rcnt = 1;

  for (i = 0; i < SIGMA; i++) Qsym[i] = 0;
  Qdel = Qins = Qsub = 0;
  munch_class(p->rgt,1.0);

  switch (p->label)
  { case Lmax:
      for (i = 0; i <= ALPHA; i++) Vsub[i] = Vins[i] = NONE;
      break;
    case Lavg:
      for (i = 0; i <= ALPHA; i++) Vsub[i] = Vins[i] = 0.0;
      break;
    case Lmin:
      for (i = 0; i <= ALPHA; i++) Vsub[i] = Vins[i] = -NONE;
      break;
    case Ldef:
      for (i = 0; i <= ALPHA; i++) Vsub[i] = Vins[i] = NONE;
      break;
  }

  wgtsum = 0.;
  hasem = (p->label == Ldef);
  v = &(WEIGHT(0,0));
  for (i = 1; i <= ALPHA; i++)
    { if (Qsym[i])
        x = Wsym[i];
      else if (Qsub)
        x = Wsub;
      else
        continue;
      w = &(WEIGHT(i,0));
      switch (p->label)
      { case Lmax:
          for (j = 0; j <= ALPHA; j++)
            { y = w[j]*x;
              if (Vsub[j] < y) Vsub[j] = y;
              y = v[j]*x;
              if (Vins[j] < y) Vins[j] = y;
            }
          hasem = 1;
          break;
        case Lavg:
          for (j = 0; j <= ALPHA; j++)
            { Vsub[j] += w[j]*x;
              Vins[j] += v[j]*x;
            }
          hasem = 1;
          wgtsum += x;
	  break;
        case Lmin:
          for (j = 0; j <= ALPHA; j++)
            { y = w[j]*x;
              if (Vsub[j] > y) Vsub[j] = y;
              y = v[j]*x;
              if (Vins[j] > y) Vins[j] = y;
            }
          hasem = 1;
          break;
        case Ldef:
          Vsub[i] = x;
          break;
      }
    }
  if (!hasem)
    error("Consensus symbol is empty",0);

  if (p->label == Lavg)
    { if (wgtsum == 0.)
        error("Sum of weights in average consensus symbol is 0",0);
      for (i = 0; i <= ALPHA; i++)
        { Vsub[i] /= wgtsum;
          Vins[i] /= wgtsum;
        }
    }

  if (!Qdel)
    Vsub[0] = NONE;
  else if (p->label == Ldef)
    Vsub[0] = Wdel;
  else
    Vsub[0] *= Wdel;

  if (!Qins)
    for (j = 1; j <= ALPHA; j++)
      Vins[j] = NONE;
  else if (p->label == Ldef)
    for (j = 1; j <= ALPHA; j++)
      Vins[j] = Wins;
  else
    for (j = 1; j <= ALPHA; j++)
      Vins[j] *= Wins;
}


/* Motif evaluation */

#define SPLIT 0
#define CNCAT 1
#define LMERG 2
#define RMERG 3
#define FINAL 4

struct vtx { int          ptype, stype, *map, old, LC, LP;
             struct vtx   *rpred, *lpred, *rsucc, *lsucc, *split, *merge;
             struct node  *symbl;
             float        wgt, bfc, brc, *ins, *sub, C, P;
             metric       metric;
             struct vtx   *lnk, *dom;
#ifdef DEBUG
             int          alpha;
#endif
           };

struct mach { struct vtx *beg, *end;
              float      thr, val, D, E, F;
              int        mlen;
            };

static struct vtx  *Z;

static int VtxCount (p) register struct node *p;
{ switch (p->label)
  { case Mor:
      return (VtxCount(p->lft)+VtxCount(p->rgt)+2);
    case Mcon:
      return (VtxCount(p->lft)+VtxCount(p->rgt));
    case Mopt:
      return (VtxCount(p->lft)+3);
    case Mwgt:
    case Mmet:
      return (VtxCount(p->lft));
    case Mref:
      { register int i;
        i = VtxCount(PUSH_CONTEXT(p));
        POP_CONTEXT();
        return (i);
      }
    case Lmax:
    case Lavg:
    case Lmin:
    case Ldef:
    case Lsymb:
    case Ldot:
      return(1);
    default:
      fatal("system bug - contact supplier(M6)");
  }
}

float maxscore (wgt,sub,alpha)
register float wgt, *sub; register int alpha;
{ register float max;
  register int i;

  if (sub == 0 || sub == Vdotsub)
    return (0.);
  max = NONE;
  for (i = 0; i <= alpha; i++)
    if (max < wgt*sub[i]-thr)
      max = wgt*sub[i] - thr;
  return(max);
}

static struct vtx *ATOM(p,i,s) struct node *p; float *i, *s;
{ register struct vtx *v;
  register int c;

  v = Z++;
  v->ptype = v->stype = 0;
  v->rpred = v->lpred = v->rsucc = v->lsucc = v->split = v->merge = 0;
  v->symbl = p;
  v->ins   = i;
  v->sub   = s;
  v->wgt   = wgt_attr;
  v->bfc   = maxscore(wgt_attr,s,ALPHA);
  v->brc   = v->bfc;
  v->map   = &(MAP(0));
  for (c = 0; c < SIGMA; c++)
    Alphabet[c] = Alphabet[c] && MAP(c);
#ifdef DEBUG
  v->alpha = ALPHA;
#endif
  return (v);
}

static MOTEVAL(p,s0,f0)
register struct node *p; register struct vtx **s0, **f0;
{ switch (p->label)
  { case Mor:
    case Mopt:
      { struct vtx *s1, *f1, *s2, *f2;
        *s0 = ATOM(0,0,0);
        MOTEVAL(p->lft,&s1,&f1);
        if (p->label == Mor)
          MOTEVAL(p->rgt,&s2,&f2);
        else
          s2 = f2 = ATOM(0,0,0);
        *f0 = ATOM(0,0,0);
        (*s0)->rsucc = s1;
        (*s0)->lsucc = s2;
        s1->rpred = s1->lpred = s2->rpred = s2->lpred = (*f0)->split = *s0;
        f1->rsucc = f1->lsucc = f2->rsucc = f2->lsucc = (*s0)->merge = *f0;
        (*f0)->rpred = f1;
        (*f0)->lpred = f2;
        (*s0)->stype = (*f0)->ptype = SPLIT;
        s1->ptype = f1->stype = RMERG;
        s2->ptype = f2->stype = LMERG;
      }
      break;
    case Mcon:
      { struct vtx *f1, *s2;
        MOTEVAL(p->lft,s0,&f1);
        MOTEVAL(p->rgt,&s2,f0);
        f1->rsucc = f1->lsucc = s2;
        s2->rpred = s2->lpred = f1;
        f1->stype = s2->ptype = CNCAT;
      }
      break;
    case Mwgt:
      { float saved;
        saved     = wgt_attr;
        wgt_attr *= NUMEVAL(p->rgt);
        MOTEVAL(p->lft,s0,f0);
        wgt_attr  = saved;
      }
      break;
    case Mmet:
      { metric saved;
        saved    = met_attr;
        met_attr = METEVAL(p->rgt);
        set_met(met_attr);
        MOTEVAL(p->lft,s0,f0);
        met_attr = saved;
        set_met(met_attr);
      }
      break;
    case Mref:
      p = PUSH_CONTEXT(p);
      MOTEVAL(p,s0,f0);
      POP_CONTEXT();
      break;
    case Lmax:
    case Lavg:
    case Lmin:
    case Ldef:
      make_scores(p);
      *s0 = *f0 = ATOM(p,Vins,Vsub);
      break;
    case Lsymb:
      if (MAP(IVAL(p)) == 0)
        { char buf[64];
          sprintf(buf,"the symbol '%c' is not in the alphabet",IVAL(p));
          error(buf,0);
        }
      *s0 = *f0 = ATOM(p,&(WEIGHT(0,0)),&(WEIGHT(MAP(IVAL(p)),0)));
      break;
    case Ldot:
      *s0 = *f0 = ATOM(p,Vdotins,Vdotsub);
      break;
    default:
      fatal("system bug - contact supplier(M7)");
    }
}

#ifdef DEBUG
static show_vtx (m) struct mach *m;
{ register struct vtx *v;
  register int i;

  fprintf(stdout,"\n\n\n");
  for (v = m->beg; v <= m->end; v++)
    { if (v->rpred != v->lpred)
        fprintf(stdout,"%3d <--",v->rpred-m->beg);
      else
        fprintf(stdout,"       ");
      fprintf(stdout,"[%06.3f,%06.3f]",v->brc,v->bfc);
      if (v->rsucc != v->lsucc)
        fprintf(stdout,"--> %-3d",v->rsucc-m->beg);
      else
        fprintf(stdout,"       ");
      if (v->ins != 0)
        { fprintf(stdout,"  [");
          for (i = 0; i <= v->alpha; i++)
	    fprintf(stdout," %g",v->ins[i]);
          fprintf(stdout," ]");
        }
      fprintf(stdout,"\n");
      if (v->rpred == 0  &&  v->lpred == 0)
        fprintf(stdout,"      <");
      else if (v->rpred == v->lpred)
        fprintf(stdout,"%3d <--",v->rpred-m->beg);
      else
        fprintf(stdout,"       ");
      fprintf(stdout,"|%6d,%6.3f|",v-m->beg,v->wgt);
      if (v->rsucc == 0  &&  v->lsucc == 0)
        fprintf(stdout,">      ");
      else if (v->rsucc == v->lsucc)
        fprintf(stdout,"--> %-3d",v->rsucc-m->beg);
      else
        fprintf(stdout,"       ");
      fprintf(stdout,"\n");
      if (v->rpred != v->lpred)
        fprintf(stdout,"%3d <--",v->lpred-m->beg);
      else
        fprintf(stdout,"       ");
      fprintf(stdout,"[%06d,%06d]",(int)v->ins,(int)v->sub);
      if (v->rsucc != v->lsucc)
        fprintf(stdout,"--> %-3d",v->lsucc-m->beg);
      else
        fprintf(stdout,"       ");
      if (v->sub != 0)
        { fprintf(stdout,"  [");
          for (i = 0; i <= v->alpha; i++)
	    fprintf(stdout," %g",v->sub[i]);
          fprintf(stdout," ]");
        }
      fprintf(stdout,"\n\n\n");
  }
}
#endif

struct mach *build(p,thresh) register struct node *p; float thresh;
{ register int i;
  register struct mach *m;
  struct vtx *s, *t;

  thr = thresh;
  Vdotsub[0] = NONE;
  for (i = 1; i <= SIGMA; i++)
    { Vdotsub[i] = 0.0;
      Vdotins[i] = NONE;
      Alphabet[i-1] = 1;
    }
  wgt_attr = 1.0;
  met_attr = cmet;
  if ((m = (struct mach *) malloc((unsigned)sizeof(struct mach))) == 0)
    fatal("out of memory");
  Z = (struct vtx *) malloc((unsigned)(sizeof(struct vtx)*(VtxCount(p)+2)));
  if (Z == 0)
    fatal("out of memory");
  m->beg = ATOM(0,&(WEIGHT(0,0)),0);
  m->beg->brc = 0.0;
  MOTEVAL(p,&s,&t);
  m->end = ATOM(0,0,0);
  m->end->bfc = 0.0;
  m->thr = thresh;
  m->val = 0.0;
  m->beg->rsucc = m->beg->lsucc = s;
  s->rpred = s->lpred = m->beg;
  t->rsucc = t->lsucc = m->end;
  m->end->rpred = m->end->lpred = t;
  m->beg->stype = s->ptype = m->end->ptype = t->stype = CNCAT;
  m->beg->ptype = m->end->stype = FINAL;
  for (s = m->end-1; s >= m->beg; s--)
    if (s->rsucc->bfc + s->rsucc->brc > s->lsucc->bfc + s->lsucc->brc)
      s->bfc = s->rsucc->bfc + s->rsucc->brc;
    else
      s->bfc = s->lsucc->bfc + s->lsucc->brc;
  for (s = m->beg+1; s <= m->end; s++)
    if (s->rpred->brc > s->lpred->brc)
      s->brc += s->rpred->brc;
    else
      s->brc += s->lpred->brc;
  inject(m);
#ifdef DEBUG
  fprintf(stdout,"N = %d T = %g Avg. match = %g #Verts/Row = %g Success freq. = %g\n",verts(m),m->thr,m->D,m->E,m->F);
  show_vtx(m);
#endif
  return (m);
}

float init (m) struct mach *m; { return (m->D); }

float cost (m) struct mach *m; { return (m->E); }

float freq (m) struct mach *m; { return (m->F); }

int  verts (m) struct mach *m; { return (m->end - m->beg + 1); }

typedef struct relevant { struct vtx *lnk, *dom;
                          float       C;
                          int         old;
                        };

static int maxmot = -1;
static struct relevant *savarr;
static float machval;

savestate(m) struct mach *m;
{ register struct vtx *f;
  register struct relevant *s;

  if (verts(m) > maxmot)
    { if (maxmot >= 0) free(savarr);
      maxmot = verts(m) + 10;
      savarr = (struct relevant *) malloc(sizeof(struct relevant)*maxmot);
    }
  s = savarr;
  for (f = m->beg; f <= m->end; f++)
    { s->C   = f->C;
      s->dom = f->dom;
      s->lnk = f->lnk;
      (s++)->old = f->old;
    }
  machval = m->val;
}

restorestate(m) struct mach *m;
{ register struct vtx *f;
  register struct relevant *s;

  s = savarr;
  for (f = m->beg; f <= m->end; f++)
    { f->C = f->P = s->C;
      f->dom = s->dom;
      f->lnk = s->lnk;
      f->old = (s++)->old;
    }
  m->val = machval;
}

int movefwd (m,a,i) struct mach *m; int a, i;
{ register struct vtx *r, *f, *x, *y, *z;
  register float e, d, t;
  register int b;

  t = m->thr;
  r = m->beg;
  if (a == 0)
    { r->C = 0.0;
      r->old = 1;
      r->lnk = 0;
      for (r++; r <= m->end; r++)
        { e = r->rpred->C;
          d = r->lpred->C;
          if (e < d) e = d;
	  if (r->sub != 0)
            if (r->sub == Vdotsub)
	      e += r->sub[0]*r->wgt;
            else
	      e += r->sub[0]*r->wgt - t;
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          r->C = e;
          r->old = 1;
          r->lnk = r-1;
        }
      r -= 1;
      f  = 0;
    }
  else
    { if (i)
        { if ( ! r->old) r->lnk = 0;
          e = 0.0;
        }
      else
        { if ( ! r->old) return (-1);
          b = r->map[a];
          if (b == 0)
            e = NONE;
          else
            e = r->P + r->ins[b]*r->wgt;
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
        }
      r->C = e;
      x = r->rsucc;
      if ((r->old || e+r->bfc >= 0.0) && r->lnk != x)
        { x->lnk = r->lnk;
          r->lnk = x;
        }
      r->old = 1;
      f = r->lnk;
      r->lnk = 0;
    
      while (f != 0)
        { if (f->ptype == SPLIT)
	    { e = f->rpred->C;
	      d = f->lpred->C;
	      if (e < d) e = d;
	    }
	  else if (f->sub == 0)
	    { e = f->rpred->C;
	    }
	  else
	    { b = f->map[a];
              if (b == 0)
                e = NONE;
              else
                { e = f->P + f->ins[b]*f->wgt;
                  if (f->sub == Vdotsub)
                    d = f->rpred->P + f->sub[b]*f->wgt;
                  else
                    d = f->rpred->P + f->sub[b]*f->wgt - t;
                  if (e < d) e = d;
                  if (f->sub == Vdotsub)
                    d = f->rpred->C + f->sub[0]*f->wgt;
                  else
                    d = f->rpred->C + f->sub[0]*f->wgt - t;
                  if (e < d) e = d;
                }
	    }
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          f->C = e;
          if (f->old || e+f->bfc >= 0.0)
            { x = f->rsucc;
              switch (f->stype)
              { case SPLIT:
                  y = f->lsucc;
                  if (!y->old)
                    { if (x->old)
                        z = x->dom;
                      else
                        z = f;
                      y->lnk = z->lnk;
                      z->lnk = y;
                    }
                case CNCAT:
                case LMERG:
                  if (f->lnk != x)
                    { x->lnk = f->lnk;
                      f->lnk = x;
                    }
                  break;
                case RMERG:
                  if (!x->old)
                    { z = x->split->lsucc;
                      if (z->old) z = z->dom;
                      x->lnk = z->lnk;
                      z->lnk = x;
                    }
                case FINAL:
                  break;
              }
            }
          f->old = 1;
          z = f;
          f = f->lnk;
          z->lnk = r;
          r = z;
        }
    }

  while (r != 0)
    { r->P = r->C;
      if ( ! (r->old = (r->C+r->bfc >= 0.0)) )
        switch (r->stype)
        { case SPLIT:
            r->old = r->rsucc->old || r->lsucc->old;
            break;
          case CNCAT:
            r->old = r->rsucc->old;
            break;
          case RMERG:
            r->old = r->rsucc->old && ! r->rsucc->lpred->old;
            break;
          case LMERG:
            r->old = r->rsucc->old && ! r->rsucc->rpred->old;
          case FINAL:
            break;
        }
      if (r->old)
        { switch (r->stype)
          { case SPLIT:
              if (r->merge->old)
                r->dom = r->merge->dom;
              else if (r->lsucc->old)
                r->dom = r->lsucc->dom;
              else if (r->rsucc->old)
                r->dom = r->rsucc->dom;
              else
                r->dom = r;
              break;
            case CNCAT:
              if (r->rsucc->old)
                r->dom = r->rsucc->dom;
              else
                r->dom = r;
              break;
            default:
              r->dom = r;
          }
          z = r;
          r = r->lnk;
          z->lnk = f;
          f = z;
        }
      else
        r = r->lnk;
    }
  m->val = m->end->C;
#ifdef TRACE
  printf(">%c:",a ? a : '*');
  for (r=m->beg; r<=m->end; r++)
      printf(" %g",r->C);
  printf(" %d\n",(f==0)?(-1):(m->val>=0.0));
#endif
  if (f == 0)
    return (-1);
  else
    return (m->val >= 0.0);
}

int moverev(m,a,i) struct mach *m; int a, i;
{ register struct vtx *r, *f, *x, *y, *z;
  register float e, d, t;
  register int b;

  t = m->thr;
  r = m->end;
  if (a == 0)
    { r->C = 0.0;
      r->old = 1;
      r->lnk = 0;
      for (r--; r >= m->beg; r--)
        { z = r->rsucc;
          if (z->sub == 0)
            e = z->C;
          else if (z->sub == Vdotsub)
	    e = z->C + z->sub[0]*z->wgt;
          else
	    e = z->C + z->sub[0]*z->wgt - t;
          if (r->stype == SPLIT)
            { z = r->lsucc;
              if (z->sub == 0)
                d = z->C;
              else if (z->sub == Vdotsub)
	        d = z->C + z->sub[0]*z->wgt;
              else
	        d = z->C + z->sub[0]*z->wgt - t;
              if (e < d) e = d;
            }
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          r->C = e;
          r->old = 1;
          r->lnk = r+1;
        }
      r += 1;
      f  = 0;
    }
  else
    { if (i)
        { if ( ! r->old) r->lnk = 0;
          e = 0.0;
        }
      else
        { if ( ! r->old) return (-1);
          e =  NONE;
        }
      r->C = e;
      x = r->rpred;
      if ((r->old || e+r->brc >= 0.0) && r->lnk != x)
        { x->lnk = r->lnk;
          r->lnk = x;
        }
      r->old = 1;
      f = r->lnk;
      r->lnk = 0;
    
      while (f != 0)
	{ b = f->map[a];
          if (b == 0 || f->ins == 0)
            e = NONE;
          else
            e = f->P + f->ins[b]*f->wgt;
          z = f->rsucc;
          if (z->sub == 0)
            { if (e < z->C) e = z->C; }
          else
            { b = z->map[a];
              if (b != 0)
                { if (z->sub == Vdotsub)
                    d = z->P + z->sub[b]*z->wgt;
                  else
                    d = z->P + z->sub[b]*z->wgt - t;
                  if (e < d) e = d;
                  if (z->sub == Vdotsub)
                    d = z->C + z->sub[0]*z->wgt;
                  else
                    d = z->C + z->sub[0]*z->wgt - t;
                  if (e < d) e = d;
                }
	    }
          if (f->stype == SPLIT)
            { z = f->lsucc;
              if (z->sub == 0)
                { if (e < z->C) e = z->C; }
              else
                { b = z->map[a];
                  if (b != 0)
                    { if (z->sub == Vdotsub)
                        d = z->P + z->sub[b]*z->wgt;
                      else
                        d = z->P + z->sub[b]*z->wgt - t;
                      if (e < d) e = d;
                      if (z->sub == Vdotsub)
                        d = z->C + z->sub[0]*z->wgt;
                      else
                        d = z->C + z->sub[0]*z->wgt - t;
                      if (e < d) e = d;
                    }
	        }
            }
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          f->C = e;
          if (f->old || e+f->brc >= 0.0)
            { x = f->lpred;
              switch (f->ptype)
              { case SPLIT:
                  y = f->rpred;
                  if (!y->old)
                    { if (x->old)
                        z = x->dom;
                      else
                        z = f;
                      y->lnk = z->lnk;
                      z->lnk = y;
                    }
                case CNCAT:
                case RMERG:
                  if (f->lnk != x)
                    { x->lnk = f->lnk;
                      f->lnk = x;
                    }
                  break;
                case LMERG:
                  if (!x->old)
                    { z = x->merge->rpred;
                      if (z->old) z = z->dom;
                      x->lnk = z->lnk;
                      z->lnk = x;
                    }
                case FINAL:
                  break;
              }
            }
          f->old = 1;
          z = f;
          f = f->lnk;
          z->lnk = r;
          r = z;
        }
    }

  while (r != 0)
    { r->P = r->C;
      if ( ! (r->old = (r->C+r->brc >= 0.0)) )
        switch (r->ptype)
        { case SPLIT:
            r->old = r->rpred->old || r->lpred->old;
            break;
          case CNCAT:
            r->old = r->rpred->old;
            break;
          case RMERG:
            r->old = r->rpred->old && ! r->rpred->lsucc->old;
            break;
          case LMERG:
            r->old = r->rpred->old && ! r->rpred->rsucc->old;
          case FINAL:
            break;
        }
      if (r->old)
        { switch (r->ptype)
          { case SPLIT:
              if (r->split->old)
                r->dom = r->split->dom;
              else if (r->rpred->old)
                r->dom = r->rpred->dom;
              else if (r->lpred->old)
                r->dom = r->lpred->dom;
              else
                r->dom = r;
              break;
            case CNCAT:
              if (r->rpred->old)
                r->dom = r->rpred->dom;
              else
                r->dom = r;
              break;
            default:
              r->dom = r;
          }
          z = r;
          r = r->lnk;
          z->lnk = f;
          f = z;
        }
      else
        r = r->lnk;
    }
  m->val = m->beg->C;
#ifdef TRACE
  printf(">%c:",a ? a : '*');
  for (r=m->beg; r<=m->end; r++)
      printf(" %g",r->C);
  printf(" %d\n",(f==0)?(-1):(m->val>=0.0));
#endif
  if (f == 0)
    return (-1);
  else
    return (m->val >= 0.0);
}

int bestfwd (m,a,i) struct mach *m; int a, i;
{ register struct vtx *r, *f, *x, *y, *z;
  register float e, d, t;
  register int b, le, ld;

  t = m->thr;
  r = m->beg;
  if (a == 0)
    { r->C = 0.0;
      r->LC = 0;
      r->old = 1;
      r->lnk = 0;
      for (r++; r <= m->end; r++)
        { e = r->rpred->C;
          le = r->rpred->LC;
          d = r->lpred->C;
          ld = r->lpred->LC;
          if (e < d)
            { e = d; le = ld; }
          else if (e == d)
            { if (ld < le) le = ld; }
	  if (r->sub != 0)
            if (r->sub == Vdotsub)
	      e += r->sub[0]*r->wgt;
            else
	      { e += r->sub[0]*r->wgt - t;
                le += 1;
              }
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          r->C = e;
          r->LC = le;
          r->old = 1;
          r->lnk = r-1;
        }
      r -= 1;
      f  = 0;
    }
  else
    { if (i)
        { if ( ! r->old) r->lnk = 0;
          e = 0.0;
        }
      else
        { if ( ! r->old) return (-1);
          b = r->map[a];
          if (b == 0)
            e = NONE;
          else
            e = r->P + r->ins[b]*r->wgt;
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
        }
      r->C = e;
      x = r->rsucc;
      if ((r->old || e+r->bfc >= 0.0) && r->lnk != x)
        { x->lnk = r->lnk;
          r->lnk = x;
        }
      r->old = 1;
      f = r->lnk;
      r->lnk = 0;
    
      while (f != 0)
        { if (f->ptype == SPLIT)
            { e = f->rpred->C;
              le = f->rpred->LC;
              d = f->lpred->C;
              ld = f->lpred->LC;
              if (e < d)
                { e = d; le = ld; }
              else if (e == d)
                { if (ld < le) le = ld; }
	    }
	  else if (f->sub == 0)
	    { e = f->rpred->C;
              le = f->rpred->LC;
	    }
	  else
	    { b = f->map[a];
              if (b == 0)
                { e = NONE;
                  le = f->LP;
                }
              else
                { e = f->P + f->ins[b]*f->wgt;
                  le = f->LP;
                  if (f->sub == Vdotsub)
                    { d = f->rpred->P + f->sub[b]*f->wgt;
                      ld = f->rpred->LP;
                    }
                  else
                    { d = f->rpred->P + f->sub[b]*f->wgt - t;
                      ld = f->rpred->LP + 1;
                    }
                  if (e < d)
                    { e = d; le = ld; }
                  else if (e == d)
                    { if (ld < le) le = ld; }
                  if (f->sub == Vdotsub)
                    { d = f->rpred->C + f->sub[0]*f->wgt;
                      ld = f->rpred->LC;
                    }
                  else
                    { d = f->rpred->C + f->sub[0]*f->wgt - t;
                      ld = f->rpred->LC + 1;
                    }
                  if (e < d)
                    { e = d; le = ld; }
                  else if (e == d)
                    { if (ld < le) le = ld; }
                }
	    }
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          f->C = e;
          f->LC = le;
          if (f->old || e+f->bfc >= 0.0)
            { x = f->rsucc;
              switch (f->stype)
              { case SPLIT:
                  y = f->lsucc;
                  if (!y->old)
                    { if (x->old)
                        z = x->dom;
                      else
                        z = f;
                      y->lnk = z->lnk;
                      z->lnk = y;
                    }
                case CNCAT:
                case LMERG:
                  if (f->lnk != x)
                    { x->lnk = f->lnk;
                      f->lnk = x;
                    }
                  break;
                case RMERG:
                  if (!x->old)
                    { z = x->split->lsucc;
                      if (z->old) z = z->dom;
                      x->lnk = z->lnk;
                      z->lnk = x;
                    }
                case FINAL:
                  break;
              }
            }
          f->old = 1;
          z = f;
          f = f->lnk;
          z->lnk = r;
          r = z;
        }
    }

  while (r != 0)
    { r->P = r->C;
      r->LP = r->LC;
      if ( ! (r->old = (r->C+r->bfc >= 0.0)) )
        switch (r->stype)
        { case SPLIT:
            r->old = r->rsucc->old || r->lsucc->old;
            break;
          case CNCAT:
            r->old = r->rsucc->old;
            break;
          case RMERG:
            r->old = r->rsucc->old && ! r->rsucc->lpred->old;
            break;
          case LMERG:
            r->old = r->rsucc->old && ! r->rsucc->rpred->old;
          case FINAL:
            break;
        }
      if (r->old)
        { switch (r->stype)
          { case SPLIT:
              if (r->merge->old)
                r->dom = r->merge->dom;
              else if (r->lsucc->old)
                r->dom = r->lsucc->dom;
              else if (r->rsucc->old)
                r->dom = r->rsucc->dom;
              else
                r->dom = r;
              break;
            case CNCAT:
              if (r->rsucc->old)
                r->dom = r->rsucc->dom;
              else
                r->dom = r;
              break;
            default:
              r->dom = r;
          }
          z = r;
          r = r->lnk;
          z->lnk = f;
          f = z;
        }
      else
        r = r->lnk;
    }
  m->val = m->end->C;
  m->mlen = m->end->LC;
#ifdef TRACE
  printf("> %c:",a ? a : '*');
  for (r=m->beg; r<=m->end; r++)
      printf(" % .1e",r->C);
  printf(" % d\n",(f==0)?(-1):(m->val>=0.0));
#endif
  if (f == 0)
    return (-1);
  else
    return (m->val >= 0.0);
}

int bestrev(m,a,i) struct mach *m; int a, i;
{ register struct vtx *r, *f, *x, *y, *z;
  register float e, d, t;
  register int b, le, ld;

  t = m->thr;
  r = m->end;
  if (a == 0)
    { r->C = 0.0;
      r->LC = 0;
      r->old = 1;
      r->lnk = 0;
      for (r--; r >= m->beg; r--)
        { z = r->rsucc;
          if (z->sub == 0)
            { e = z->C; le = z->LC; }
          else if (z->sub == Vdotsub)
	    { e = z->C + z->sub[0]*z->wgt; le = z->LC; }
          else
	    { e = z->C + z->sub[0]*z->wgt - t; le = z->LC + 1; }
          if (r->stype == SPLIT)
            { z = r->lsucc;
              if (z->sub == 0)
                { d = z->C; ld = z->LC; }
              else if (z->sub == Vdotsub)
	        { d = z->C + z->sub[0]*z->wgt; ld = z->LC; }
              else
	        { d = z->C + z->sub[0]*z->wgt - t; ld = z->LC + 1; }
              if (e < d)
                { e = d; le = ld; }
              else if (e == d)
                { if (ld < le) le = ld; }
            }
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          r->C = e;
          r->LC = le;
          r->old = 1;
          r->lnk = r+1;
        }
      r += 1;
      f  = 0;
    }
  else
    { if (i)
        { if ( ! r->old) r->lnk = 0;
          e = 0.0;
        }
      else
        { if ( ! r->old) return (-1);
          e =  NONE;
        }
      r->C = e;
      x = r->rpred;
      if ((r->old || e+r->brc >= 0.0) && r->lnk != x)
        { x->lnk = r->lnk;
          r->lnk = x;
        }
      r->old = 1;
      f = r->lnk;
      r->lnk = 0;
    
      while (f != 0)
	{ b = f->map[a];
          if (b == 0 || f->ins == 0)
            { e = NONE; le = f->LP; }
          else
            { e = f->P + f->ins[b]*f->wgt; le = f->LP; }
          z = f->rsucc;
          if (z->sub == 0)
            { if (e < z->C)
                { e = z->C; le = z->LC; }
              else if (e == z->C)
                { if (z->LC < le) le = z->LC; }
            }
          else
            { b = z->map[a];
              if (b != 0)
                { if (z->sub == Vdotsub)
                    { d = z->P + z->sub[b]*z->wgt; ld = z->LP; }
                  else
                    { d = z->P + z->sub[b]*z->wgt - t; ld = z->LP + 1; }
                  if (e < d)
                    { e = d; le = ld; }
                  else if (e == d)
                    { if (ld > le) le = ld; }
                  if (z->sub == Vdotsub)
                    { d = z->C + z->sub[0]*z->wgt; ld = z->LC; }
                  else
                    { d = z->C + z->sub[0]*z->wgt - t; ld = z->LC + 1; }
                  if (e < d)
                    { e = d; le = ld; }
                  else if (e == d)
                    { if (ld > le) le = ld; }
                }
	    }
          if (f->stype == SPLIT)
            { z = f->lsucc;
              if (z->sub == 0)
                { if (e < z->C)
                    { e = z->C; le = z->LC; }
                  else if (e == z->C)
                    { if (z->LC < le) le = z->LC; }
                }
              else
                { b = z->map[a];
                  if (b != 0)
                    { if (z->sub == Vdotsub)
                        { d = z->P + z->sub[b]*z->wgt; ld = z->LP; }
                      else
                        { d = z->P + z->sub[b]*z->wgt - t; ld = z->LP + 1; }
                      if (e < d)
                        { e = d; le = ld; }
                      else if (e == d)
                        { if (ld > le) le = ld; }
                      if (z->sub == Vdotsub)
                        { d = z->C + z->sub[0]*z->wgt; ld = z->LC; }
                      else
                        { d = z->C + z->sub[0]*z->wgt - t; ld = z->LC + 1; }
                      if (e < d)
                        { e = d; le = ld; }
                      else if (e == d)
                        { if (ld > le) le = ld; }
                    }
	        }
            }
          if (e <  NONE) e =  NONE;
          if (e > -NONE) e = -NONE;
          f->C = e;
          f->LC = le;
          if (f->old || e+f->brc >= 0.0)
            { x = f->lpred;
              switch (f->ptype)
              { case SPLIT:
                  y = f->rpred;
                  if (!y->old)
                    { if (x->old)
                        z = x->dom;
                      else
                        z = f;
                      y->lnk = z->lnk;
                      z->lnk = y;
                    }
                case CNCAT:
                case RMERG:
                  if (f->lnk != x)
                    { x->lnk = f->lnk;
                      f->lnk = x;
                    }
                  break;
                case LMERG:
                  if (!x->old)
                    { z = x->merge->rpred;
                      if (z->old) z = z->dom;
                      x->lnk = z->lnk;
                      z->lnk = x;
                    }
                case FINAL:
                  break;
              }
            }
          f->old = 1;
          z = f;
          f = f->lnk;
          z->lnk = r;
          r = z;
        }
    }

  while (r != 0)
    { r->P = r->C;
      r->LP = r->LC;
      if ( ! (r->old = (r->C+r->brc >= 0.0)) )
        switch (r->ptype)
        { case SPLIT:
            r->old = r->rpred->old || r->lpred->old;
            break;
          case CNCAT:
            r->old = r->rpred->old;
            break;
          case RMERG:
            r->old = r->rpred->old && ! r->rpred->lsucc->old;
            break;
          case LMERG:
            r->old = r->rpred->old && ! r->rpred->rsucc->old;
          case FINAL:
            break;
        }
      if (r->old)
        { switch (r->ptype)
          { case SPLIT:
              if (r->split->old)
                r->dom = r->split->dom;
              else if (r->rpred->old)
                r->dom = r->rpred->dom;
              else if (r->lpred->old)
                r->dom = r->lpred->dom;
              else
                r->dom = r;
              break;
            case CNCAT:
              if (r->rpred->old)
                r->dom = r->rpred->dom;
              else
                r->dom = r;
              break;
            default:
              r->dom = r;
          }
          z = r;
          r = r->lnk;
          z->lnk = f;
          f = z;
        }
      else
        r = r->lnk;
    }
  m->val = m->beg->C;
  m->mlen = m->beg->LC;
#ifdef TRACE
  printf("< %c:",a ? a : '*');
  for (r=m->beg; r<=m->end; r++)
      printf(" % .1e",r->C);
  printf(" % d\n",(f==0)?(-1):(m->val>=0.0));
#endif
  if (f == 0)
    return (-1);
  else
    return (m->val >= 0.0);
}

int csize (m) struct mach *m;
{ register struct vtx *f;
  register int cnt;
  f = m->beg;
  if ( ! f->old)
    { f = m->end;
      if ( ! f->old)
        return (0);
    }
  for (cnt = 0; f != 0; cnt++)
    f = f->lnk;
  return (cnt);
}

float cvalue (m) register struct mach *m; { return (m->val); }

int clength (m) register struct mach *m; { return (m->mlen); }

float thresh (m) register struct mach *m; { return (m->thr); }

destroy (m) struct mach *m;
{ register struct vtx *v;
  register struct node *p, *q;

  for (v = m->beg; v <= m->end; v++)
    if (v->symbl != 0 && v->symbl->label != Lsymb && v->symbl->label != Ldot)
      for (p = v->symbl; p->lnk != 0; p = p->lnk)
        { q = p->lnk;
          if (v->ins == VINS(q) && v->sub == VSUB(q) && --(q->rcnt) == 0)
            { p->lnk = q->lnk;
              free((char *)VINS(q));
              free((char *)VSUB(q));
              free((char *)q);
              break;
            }
        }
  free((char *)m->beg);
  free((char *)m);
}

#define SAMP   3
#define PRECISION .001

inject (m) struct mach *m;
{ long Dbar, Ebar, Fbar;
  int post, size;
  float Dsample[SAMP], Esample[SAMP], Fsample[SAMP];
  register int f, l, i, b;
  register float erel, min, max;

  movefwd(m,0,0);

  post = SAMP-1;
  Dbar = Ebar = Fbar = 0L;
  for (size = i = 0; i < SIGMA; i++)
    if (Alphabet[i] != 0)
      Alphabet[size++] = i;
  for (i = 0; i < SAMP; i++)
    Dsample[i] = Esample[i] = Fsample[i] = 0.0;
  erel = 1.0;

  for (f = 0, l = 1000; erel > PRECISION; f = l, l += 0.2*l)
    { for (i = f+1; i <= l; i++)
        { b = Alphabet[(int)(((float)rand()/(float)RAND_MAX)*size)];
          if (movefwd(m,b,1) == 1)
            { Fbar += 1L;
              Ebar += (long)csize(m);
              do
                { Dbar += 1L;
                  b = Alphabet[(int)(((float)rand()/(float)RAND_MAX)*size)];
                } while (movefwd(m,b,0) == 1 && Dbar < l);
            }
          else
            Ebar += (long)csize(m);
        }

      post = (post+1) % SAMP;
      Dsample[post] = Fbar == 0L ? 0.0 : Dbar/((float)Fbar);
      Esample[post] = Ebar/((float)l);
      Fsample[post] = Fbar/((float)l);
      min = max = Esample[0];
      for (i = 1; i < SAMP; i++)
        { if (Esample[i] < min) min = Esample[i];
          if (Esample[i] > max) max = Esample[i];
        }
      erel = ((max == 0.0) ? 0.0 : (max-min)/max);
    }

  for (i = 1; i < SAMP; i++)
    { *Dsample += Dsample[i];
      *Esample += Esample[i];
      *Fsample += Fsample[i];
    }
  m->D = *Dsample/SAMP;
  m->E = *Esample/SAMP;
  m->F = *Fsample/SAMP;
}
