#include	<stdio.h>
#include	<string.h>
#include	"anrep.h"
#include	"metric.h"
#include	"partree.h"
#include	"motif.h"

#define		TRUE	1
#define		FALSE	0

#define		CNCAT	0
#define		SPLIT	1
#define		LMERG	2
#define		RMERG	3
#define		FINAL	4

#define		EMPTY	0
#define		MOTIF	1
#define		RANGE	2

#define		FAILURE	(-1)
#define		UNKNOWN	0
#define		SUCCESS	1

#define		LEFT	(-1)
#define		DONE	0
#define		RIGHT	1

#define		BSIZE	(3*MAXMATCH)

typedef	struct	vtx {
	int		type, ptype, stype;
        struct  vtx     *data, *path, *rpred, *lpred, *rsucc, *lsucc, *split, *merge;
}	vtx;

typedef	struct	net {
	vtx		*beg, *end;
}	net;

typedef	struct	end {
	char		mark, *pos;
	struct	end	*pred, *succ;
}	end;

typedef	struct	match {
	char		mark, *oend, *delta;
	end		*last, *iend;
	int		length;
	float		score;
	struct	match	*best, *pred, *succ;
}	match;

typedef	struct	mtf {
	mach		mach;
	float		bval;
        long            last;
	match		*LFM, *LLM, *RFM, *RLM;
}	mtf;

typedef	struct	rng {
	int		beg, end;
}	rng;

typedef struct  front {
        int		ord, cut;
        vtx             *path;
        struct  front   *pred, *succ;
}       front;

typedef struct	result {
	long		beg, end;
	char		*string;
	float		*values, val;
}	result;

static	char	_S[BSIZE+2];	/* string buffer */
static  char    _I[BSIZE+2];    /* interval buffer */

static  char    *H;             /* header pointer */
static  char    *S = _S + 1;    /* string pointer */
static  long    O;              /* string offset */

static  result  _B[2];          /* result buffer */

static 	net 	*N;		/* network */

static 	vtx 	*A;             /* network vertices */
static  vtx     *Z;             /* network vertices */

static  float   *I;             /* initialization costs */
static  float   *C;             /* per-column costs */
static  float   *F;             /* frequencies */
static  float   *D;             /* distances */
static  char   **AP;            /* match pointers */

static  float   *Q;             /* recurrence matrix */
static  char    *T;             /* recurrence traces */

static  front   *X;             /* minimum cut */

static  mtf     **M;            /* seed motifs */

static int VtxCount (p) register node *p; {
    register int i;

    switch (p->label) {
    case Nor:
        return(VtxCount(p->lft)+VtxCount(p->rgt)+2);
    case Ncon:
        return(VtxCount(p->lft)+VtxCount(p->rgt));
    case Nopt:
        return(VtxCount(p->lft)+3);
    case Nref:
        i = VtxCount(PUSH_CONTEXT(p));
        POP_CONTEXT();
        return(i);
    case Nmot:
    case Nspacer:
        return(1);
    default:
        fatal("system bug - contact supplier(S1).");
        return(0);
    }
}

static vtx *NewVtx (t,d) register int t; register vtx *d; {
    register vtx *v;

    v = Z++;
    v->type  = t;
    v->data  = d;
    v->path  = NULL;
    v->ptype = v->stype = 0;
    v->rpred = v->lpred = v->rsucc = v->lsucc = v->split = v->merge = NULL;
    return(v);
}

static vtx *BuildMotif (p,f) register node *p; float f; {
    register mtf *m;

    if ((m = (mtf *)malloc(sizeof(mtf))) == NULL) {
        fatal("out of memory");
    }
    m->mach = build(p,f);
    m->bval = -1.0;
    m->last = 0L;
    m->LFM  = NULL;
    m->LLM  = NULL;
    m->RFM  = NULL;
    m->RLM  = NULL;
    return((vtx *)m);
}

static vtx *BuildSpacer (i,j) register int i, j; {
    register rng *r;

    if ((r = (rng *)malloc(sizeof(rng))) == NULL) {
        fatal("out of memory");
    }
    if (i < j) {
        r->beg = i;
        r->end = j;
    } else {
        r->beg = j;
        r->end = i;
    }
    return((vtx *)r);
}

static int BuildNetwork (p,s0,f0) register node *p; register vtx **s0, **f0; {
    register int i;
    vtx *s1, *f1, *s2, *f2;

    switch (p->label) {
    case Nor:
    case Nopt:
        *s0 = NewVtx(EMPTY,(vtx *)NULL);
        i = BuildNetwork(p->lft,&s1,&f1);
        if (p->label == Nor) {
          i += BuildNetwork(p->rgt,&s2,&f2);
        } else {
          s2 = f2 = NewVtx(EMPTY,(vtx *)NULL);
        }
        *f0 = NewVtx(EMPTY,(vtx *)NULL);
        (*s0)->lsucc = (*s0)->path = s1;
        (*s0)->rsucc = s2;
        s1->rpred = s1->lpred = s2->rpred = s2->lpred = (*f0)->split = *s0;
        f1->rsucc = f1->lsucc = f2->rsucc = f2->lsucc = (*s0)->merge = *f0;
        (*f0)->rpred = f2;
        (*f0)->lpred = (*f0)->path = f1;
        (*s0)->stype = (*f0)->ptype = SPLIT;
        s1->ptype = f1->stype = LMERG;
        s2->ptype = f2->stype = RMERG;
        break;
    case Ncon:
        i  = BuildNetwork(p->lft,s0,&f1);
        i += BuildNetwork(p->rgt,&s2,f0);
        f1->rsucc = f1->lsucc = s2;
        s2->rpred = s2->lpred = f1;
        f1->stype = s2->ptype = CNCAT;
        break;
    case Nref:
        p = PUSH_CONTEXT(p);
        i = BuildNetwork(p,s0,f0);
        POP_CONTEXT();
        break;
    case Nmot:
        i = 1;
        *s0 = *f0 = NewVtx(MOTIF,BuildMotif(p->lft,NUMEVAL(p->rgt)));
        break;
    case Nspacer:
        i = 0;
        *s0 = *f0 = NewVtx(RANGE,BuildSpacer(INTEVAL(p->lft),INTEVAL(p->rgt)));
        break;
    default:
        fatal("system bug - contact supplier(S2)");
    }
    return(i);
}

#ifdef DEBUG
static ShowNet (n) register net *n; {
    register vtx *v;

    for (v = n->beg; v <= n->end; v++) {
        fprintf(stderr,"\n");
        if (v->lpred != v->rpred) {
            fprintf(stderr,"%3d <--",v->lpred-n->beg);
        } else {
            fprintf(stderr,"       ");
        }
        if (v->type == MOTIF) {
            fprintf(stderr,"[%013d]",((mtf *)v->data)->mach);
        } else if (v->type == RANGE) {
            fprintf(stderr,"[%013d]",((rng *)v->data)->beg);
        } else {
            fprintf(stderr,"[-------------]");
        }
        if (v->lsucc != v->rsucc) {
            fprintf(stderr,"--> %-3d",v->lsucc-n->beg);
        }
        fprintf(stderr,"\n");
        if (v->lpred == 0  &&  v->rpred == 0) {
            fprintf(stderr,"      <");
        } else if (v->lpred == v->rpred) {
            fprintf(stderr,"%3d <--",v->lpred-n->beg);
        } else {
            fprintf(stderr,"       ");
        }
        fprintf(stderr,"|%7d      |",v-n->beg);
        if (v->lsucc == 0  &&  v->rsucc == 0) {
            fprintf(stderr,">      ");
        } else if (v->lsucc == v->rsucc) {
            fprintf(stderr,"--> %-3d",v->lsucc-n->beg);
        }
        fprintf(stderr,"\n");
        if (v->lpred != v->rpred) {
            fprintf(stderr,"%3d <--",v->rpred-n->beg);
        } else {
            fprintf(stderr,"       ");
        }
        if (v->type == MOTIF) {
            fprintf(stderr,"[%013.3f]",((mtf *)v->data)->bval);
        } else if (v->type == RANGE) {
            fprintf(stderr,"[%013d]",((rng *)v->data)->end);
        } else {
            fprintf(stderr,"[-------------]");
        }
        if (v->lsucc != v->rsucc) {
            fprintf(stderr,"--> %-3d",v->rsucc-n->beg);
        }
        fprintf(stderr,"\n");
    }
}
#endif

static rng Spacer (v,w) register vtx *v, *w; {
    rng r;

    r.beg = 1;
    r.end = 2;
    for (v = v->lsucc; v->type != MOTIF; v = (v->stype==SPLIT ? v->path:v->lsucc)) {
        if (v->type == RANGE) {
            r.beg += ((rng *)v->data)->beg;
            r.end += ((rng *)v->data)->end;
        }
    }
    if (v != w) {
       r.beg = r.end = 0;
    }
    return(r);
}

static front *LeftFront (f,t) front *f; vtx *t; {
    register vtx *u, *v, *w, *x, *y;
    int cut, level, found, xxx;
    vtx p;

    y = Z;
    cut = TRUE;
    p.path = NULL;
    for (u = f->path; u != NULL; u = u->path) {
        level = 0;
        found = FALSE;
        v = u->rpred;
        while ((level > 0  ||  !found)  &&  v->ptype != FINAL) {
            if (v->type == MOTIF  &&  !found) {
                if (p.path == NULL) {
                    w = &p;
                    p.path = v;
                } else if (w->path > v) {
                    if (y > v) {
                        y = v;
                    }
                    cut = FALSE;
                } else if (w->path < v) {
                    for (x = v; x > w->path; x = x->ptype!=SPLIT ? x->rpred:x->split);
                    if (x == w->path) {
                        if (y > w->path) {
                            y = w->path;
                        }
                        w->path = v;
                        cut = FALSE;
                    } else {
                        w = w->path;
                        w->path = v;
                    }
                }
                found = TRUE;
            }
            switch (v->ptype) {
            case CNCAT:
                v = v->rpred;
                break;
            case SPLIT:
                if (found) {
                    v = v->split;
                } else {
                    level += 1;
                    v = v->lpred;
                }
                break;
            case LMERG:
                if (level > 0) {
                    xxx = found;
                    found = FALSE;
                    v = v->rpred->merge->rpred;
                } else {
                    v = v->rpred;
                }
                break;
            case RMERG:
                if (level > 0) {
                    if (cut) {
                        cut = found == xxx;
                    }
                    found = found || xxx;
                    level -= 1;
                }
                v = v->rpred;
                break;
            default:
                fatal("system bug - contact supplier(S3)");
                break;
            }
        }
    }
    if (p.path != NULL) {
        if ((f = (front *)malloc(sizeof(front))) == NULL) {
            fatal("out of memory");
        }
        f->ord = 0;
        if (cut  &&  t != Z) {
            f->cut = FALSE;
            for (x = p.path; x != NULL; x = x->path) {
                if (x == t) {
                    f->cut = TRUE;
                    break;
                }
            }
        } else {
            f->cut = cut;
        }
        f->path = p.path;
        if (!f->cut) {
            if (t > y) {
                t = y;
            }
        } else {
            t = Z;
        }
        if ((f->pred = LeftFront(f,t)) != NULL) {
            f->pred->succ = f;
        }
        return(f);
    } else {
        return(NULL);
    }
}

static front *RightFront (f,t) front *f; vtx *t; {
    register vtx *u, *v, *w, *x, *y;
    int cut, level, found, xxx;
    vtx p;

    y = A;
    cut = TRUE;
    p.path = NULL;
    for (u = f->path; u != NULL; u = u->path) {
        level = 0;
        found = FALSE;
        v = u->lsucc;
        while ((level > 0  ||  !found)  &&  v->stype != FINAL) {
            if (v->type == MOTIF  &&  !found) {
                if (p.path == NULL) {
                    w = &p;
                    p.path = v;
                } else if (w->path > v) {
                    if (y < w->path) {
                        y = w->path;
                    }
                    w->path = v;
                    cut = FALSE;
                } else if (w->path < v) {
                    for (x = w->path; x < v; x = x->stype!=SPLIT ? x->lsucc:x->merge);
                    if (x == v) {
                        if (y < v) {
                            y = v;
                        }
                        cut = FALSE;
                    } else {
                        w = w->path;
                        w->path = v;
                    }
                }
                found = TRUE;
            }
            switch (v->stype) {
            case CNCAT:
                v = v->lsucc;
                break;
            case SPLIT:
                if (found) {
                    v = v->merge;
                } else {
                    level += 1;
                    v = v->lsucc;
                }
                break;
            case LMERG:
                if (level > 0) {
                    xxx = found;
                    found = FALSE;
                    v = v->rsucc->split->rsucc;
                } else {
                    v = v->lsucc;
                }
                break;
            case RMERG:
                if (level > 0) {
                    if (cut) {
                        cut = found == xxx;
                    }
                    found = found || xxx;
                    level -= 1;
                }
                v = v->lsucc;
                break;
            default:
                fatal("system bug - contact supplier(S4)");
                break;
            }
        }
    }
    if (p.path != NULL) {
        if ((f = (front *)malloc(sizeof(front))) == NULL) {
            fatal("out of memory");
        }
        f->ord = 0;
        if (cut  &&  t != A) {
            f->cut = FALSE;
            for (x = p.path; x != NULL; x = x->path) {
                if (x == t) {
                    f->cut = TRUE;
                    break;
                }
            }
        } else {
            f->cut = cut;
        }
        f->path = p.path;
        if (!f->cut) {
            if (t < y) {
                t = y;
            }
        } else {
            t = A;
        }
        if ((f->succ = RightFront(f,t)) != NULL) {
            f->succ->pred = f;
        }
        return(f);
    } else {
        return(NULL);
    }
}

static Recurrence (n) register int n; {
    register int i, j;
    float a, b;

    /* compute recurrence */
    Q[n-1] = 0.0;
    T[n-1] = '*';
    for (j = n-1; j > 0; j--) {
	Q[j-1] = (Q[j]*F[j] + C[j])*D[j-1] + I[j];
        T[j-1] = '>';
    }
    Q[0] = Q[0]*F[0] + C[0];
    for (i = 1; i < n; i++) {
	Q[(i+1)*n-1] = (Q[i*n-1]*F[i] + C[i])*D[i-1] + I[i];
        T[(i+1)*n-1] = '^';
	for (j = n-1; j > i; j--) {
	    a = (Q[(i-1)*n+j-1]*F[i] + C[i])*D[i-1] + I[i];
	    b = (Q[i*n+j]*F[j] + C[j])*D[j-1] + I[j];
	    if (a < b) {
		Q[i*n+j-1] = a;
                T[i*n+j-1] = '^';
            } else {
		Q[i*n+j-1] = b;
                T[i*n+j-1] = '>';
	    }
	}
	Q[i*(n+1)] = Q[i*(n+1)]*F[i] + C[i];
    }
#ifdef DEBUG
    fprintf(stderr,"\n");
    for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
            fprintf(stderr,i<=j?"%7.1E%c":"        ",Q[i*n+j],T[i*n+j]);
        }
        fprintf(stderr,"\n");
    }
#endif
}

static int SetPath (v,w) register vtx *v, *w; {

    while (v < w) {
        switch (v->stype) {
        case CNCAT:
            v = v->lsucc;
            break;
        case SPLIT:
            if (w < v->rsucc) {
                v = v->path = v->lsucc;
            } else if (w < v->merge) {
                v = v->path = v->rsucc;
            } else {
                v->path = v->lsucc;
                SetPath(v->lsucc,v->merge->lpred);
                v = v->merge;
                SetPath(v->split->rsucc,v->rpred);
                v->path = v->lpred;
            }
            break;
        case LMERG:
            v->lsucc->path = v;
            v = v->lsucc;
            break;
        case RMERG:
            v->rsucc->path = v;
            v = v->rsucc;
            break;
        }
    }
    return(v == w);
}

static int MorePaths (v,w) register vtx *v, *w; {
    register vtx *x;

    x = w;
    while (v < x) {
        if (x->stype == SPLIT  &&  x->merge < w) {
            if (x->path == x->rsucc) {
                x->path = x->lsucc;
                x->merge->path = x->merge->lpred;
            } else {
                x->path = x->rsucc;
                x->merge->path = x->merge->rpred;
                return(TRUE);
            }
        }
        x = (x->ptype == SPLIT  ?  x->path : x->lpred);
    }
    return(FALSE);
}

static NetworkCost (x,i,j,f) vtx *x; int *i, *j; register front *f; {
    register front *g, *h;
    register vtx *v, *w;
    rng r, s;

    f->ord  = 0;
    f->cut  = TRUE;
    f->path = x;
    if ((f->pred = LeftFront(f,Z)) != NULL) {
        f->pred->succ = f;
    }
    if ((f->succ = RightFront(f,A)) != NULL) {
        f->succ->pred = f;
    }
    for (g = f; g->pred != NULL; g = g->pred);
    for (*i = 0; g != f; g = g->succ) {
        for (v = g->path; v != NULL; v = v->path, (*i)++) {
            I[*i] = init(((mtf *)v->data)->mach);
            C[*i] = cost(((mtf *)v->data)->mach);
            F[*i] = g->cut ? freq(((mtf *)v->data)->mach) : 1.0;
            D[*i] = 0.0;
            h = g;
            do {
                h = h->succ;
                for (w = h->path; w != NULL; w = w->path) {
                    if (SetPath(v,w)) {
                        r.beg = r.end = 1;
                        do {
                            s = Spacer(v,w);
                            if (s.beg != s.end) {
                                if (r.beg > s.beg) {
                                    r.beg = s.beg;
                                }
                                if (r.end < s.end) {
                                    r.end = s.end;
                                }
                            }
                        } while (MorePaths(v,w));
                        D[*i] += (float)(r.end - r.beg);
                    }
                }
            } while (!h->cut);
        }
    }
    v = f->path;
    I[*i] = init(((mtf *)v->data)->mach);
    C[*i] = cost(((mtf *)v->data)->mach);
    F[*i] = freq(((mtf *)v->data)->mach);
    for (*j = (*i)++, g = g->succ; g != NULL; g = g->succ) {
        for (v = g->path; v != NULL; v = v->path, (*i)++) {
            I[*i] = init(((mtf *)v->data)->mach);
            C[*i] = cost(((mtf *)v->data)->mach);
            F[*i] = g->cut ? freq(((mtf *)v->data)->mach) : 1.0;
            D[*i-1] = 0.0;
            h = g;
            do {
                h = h->pred;
                for (w = h->path; w != NULL; w = w->path) {
                    if (SetPath(w,v)) {
                        r.beg = r.end = 1;
                        do {
                            s = Spacer(w,v);
                            if (s.beg != s.end) {
                                if (r.beg > s.beg) {
                                    r.beg = s.beg;
                                }
                                if (r.end < s.end) {
                                    r.end = s.end;
                                }
                            }
                        } while (MorePaths(w,v));
                        D[*i-1] += (float)(r.end - r.beg);
                    }
                }
            } while (!h->cut);
        }
    }
    if (*i == 0) {
        error("network matches empty string",0);
    } else {
        Recurrence(*i);
    }
}

static vtx *MinimumCut (v,w) register vtx *v, *w; {
    register vtx *x, *y, *z;

    if (v->stype != SPLIT) {
        x = v->type  == MOTIF  ?  v : NULL;
        y = v->stype == CNCAT  ?  MinimumCut(v->lsucc,w) : NULL;
    } else {
        x = MinimumCut(v->lsucc,v->merge->lpred);
        y = MinimumCut(v->rsucc,v->merge->rpred);
        if (x != NULL  &&  y != NULL) {
            for (z = x; z->merge != NULL; z = z->merge);
            z->merge = y;
            ((mtf *)x->data)->bval += ((mtf *)y->data)->bval;
        } else {
            x = NULL;
        }
        y = MinimumCut(v->merge->lsucc,w);
    }
    if (x != NULL) {
        if (y != NULL) {
            return(((mtf *)x->data)->bval <= ((mtf *)y->data)->bval  ?  x : y);
        } else {
            return(x);
        }
    } else {
        if (y != NULL) {
            return(y);
        } else {
            return(NULL);
        }
    }
}

static FindOrder (x) vtx *x; {
    register front *f, *g, *h;
    register vtx *v, *w;
    int i, j, k;

    for (f = X; x != NULL; x = x->merge, f++) {
	NetworkCost(x,&i,&j,f);
        g  = h = f;
        v  = w = f->path;
        j *= i + 1;
        k  = 1;
        while (j != i-1) {
            if (T[j] == '^') {
                if (v->path != NULL) {
                    v = v->path;
                } else {
                    g = g->pred;
                    v = g->path;
                }
                if (v->path == NULL) {
                    g->ord = k++;
                }
                j -= i;
            } else if (T[j] == '>') {
                if (w->path != NULL) {
                    w = w->path;
                } else {
                    h = h->succ;
                    w = h->path;
                }
                if (w->path == NULL) {
                    h->ord = k++;
                }
                j += 1;
            } else {
                fatal("system bug - contact supplier(S5)");
            }
        }
#ifdef DEBUG
        for (g = f; g->pred != NULL; g = g->pred);
        fprintf(stderr,"\n");
        for (i = 0; g != f; g = g->succ) {
            fprintf(stderr,"%c (%03d):",g->cut ? 'X':'O',g->ord);
            for (v = g->path; v != NULL; v = v->path, i++) {
                fprintf(stderr," (%03d %5.3f %5.3f %5.3f) %d",v-N->beg,I[i],C[i],F[i],(int)D[i]);
            }
            fprintf(stderr,"\n");
        }
        fprintf(stderr,"* (%03d): (%03d %5.3f %5.3f %5.3f)\n",g->ord,g->path-N->beg,I[i],C[i],F[i]);
        for (i++, g = g->succ; g != NULL; g = g->succ) {
            fprintf(stderr,"%c (%03d):",g->cut ? 'X':'O',g->ord);
            for (v = g->path; v != NULL; v = v->path, i++) {
                fprintf(stderr," %d (%03d %5.3f %5.3f %5.3f)",(int)D[i-1],v-N->beg,I[i],C[i],F[i]);
            }
            fprintf(stderr,"\n");
        }
#endif
    }
    f->path = NULL;
}

static AddMatch (f,l,p) register match **f, **l; register char *p; {
    register match *m;

    /* append new match */
    if ((m = (match *)malloc((unsigned)(sizeof(match)))) == NULL) {
	fatal("out of memory");
    }
    m->mark   = FALSE;
    m->last   = NULL;
    m->iend   = NULL;
    m->oend   = p;
    m->delta  = NULL;
    m->length = 0;
    m->score  = 0.0;
    m->best   = NULL;
    m->pred   = *l;
    m->succ   = NULL;
    if (*f == NULL) {
	*f = m;
    } else {
	(*l)->succ = m;
    }
    *l = m;
}

static DeleteMatch (f,l,m) register match **f, **l, *m; {

    /* remove match from list */
    if (*f == m) {
	*f = m->succ;
    } else {
	m->pred->succ = m->succ;
    }
    if (*l == m) {
	*l = m->pred;
    } else {
	m->succ->pred = m->pred;
    }
    free((char *)m);
}

static int NextFront (f,g) register front *f, *g; {

    /* determine next front to be processed */
    if (f->pred != NULL  &&  g->succ != NULL) {
        return(f->pred->ord < g->succ->ord ? LEFT : RIGHT);
    } else if (f->pred != NULL) {
        return(LEFT);
    } else if (g->succ != NULL) {
        return(RIGHT);
    } else {
        return(DONE);
    }
}

static AddEnd (f,m,p) register match *f; register int m; register char *p; {
    register end *e;

    /* append new end */
    if ((e = (end *)malloc((unsigned)(sizeof(end)))) == NULL) {
	fatal("out of memory");
    }
    e->mark = m;
    e->pos  = p;
    e->pred = f->last;
    e->succ = NULL;
    if (f->iend == NULL) {
	f->iend = e;
    } else {
	f->last->succ = e;
    }
    f->last = e;
}

static DeleteEnd (f,e) register match *f; register end *e; {

    /* remove end from list */
    if (e == f->iend) {
	f->iend = e->succ;
    } else {
	e->pred->succ = e->succ;
    }
    if (e == f->last) {
	f->last = e->pred;
    } else {
	e->succ->pred = e->pred;
    }
    free((char *)e);
}

static int MatchLeft (x,y) front *x, *y; {
    register match *f, *g;
    register char *s, *i;
    register vtx *v, *w;
    char *j, *t, *I;
    int success, found, MatchLeft();
    float value;
    front *z;
    mtf *m, *n;
    rng r;

    for (I = _I+1, found = FALSE, v = x->path; v != NULL; v = v->path) {

        /* construct search interval */
        s = S;
        t = S + BSIZE;
        z = x;
        do {
            z = z->succ;
            for (w = z->path; w != NULL; w = w->path) {
                if (((mtf *)w->data)->LFM != NULL  &&  SetPath(v,w)) {
                    do {
                        r = Spacer(v,w);
                        if (r.beg != r.end) {
                            for (f = ((mtf *)w->data)->LFM; f != NULL; f = f->succ) {
                                if (S+BSIZE-1 < f->oend-r.beg) {
	                            warning("potential match exceeds buffer capacity(W1)");
                                    i = S + BSIZE - 1;
                                } else {
                                    i = f->oend - r.beg;
                                }
                                if (s < i) {
                                    s = i;
                                }
                                if (S-1 > f->oend-r.end  &&  O > 0L) {
	                            warning("potential match exceeds buffer capacity(W2)");
                                    j = S - 1;
                                } else {
                                    j = f->oend - r.end;
                                }
                                if (t > j) {
                                    t = j;
                                }
                                while (i > j) {
                                    I[i---S] = 0x01;
                                }
                            }
                        }
                    } while (MorePaths(v,w));
                }
            }
        } while (!z->cut);

        /* find left ends of all possible matches for motif m */
        if (s > t) {
            m = (mtf *)v->data;
            success = moverev(m->mach,0,0);
            for (i = I+(s-S); *s != '\0'   &&   (s > t  ||  success != FAILURE); s--) {
	        if ((success=moverev(m->mach,*s,(int)*i)) == SUCCESS) {
	            AddMatch(&m->LFM,&m->LLM,s);
	        }
                *i-- = 0x00;
            }
            if (*s == '\0'  &&  s == _S  &&  O > 0L) {
	        warning("potential match exceeds buffer capacity(W3)");
            }
            if (m->LFM != NULL) {
                found = TRUE;
            }
        }

    }

    /* match next front */
    if (found  ||  !x->cut) {
	switch (NextFront(x,y)) {
	case LEFT:
	    if (MatchLeft(x->pred,y) == FAILURE) {
		return(FAILURE);
	    }
	    break;
	case RIGHT:
	    if (MatchRight(x,y->succ) == FAILURE) {
		return(FAILURE);
	    }
	    break;
	}
    } else {
	return(FAILURE);
    }

    /* for all matches found, find corresponding right ends */
    for (found = FALSE, v = x->path; v != NULL; v = v->path) {
        m = (mtf *)v->data;
        for (f = m->LFM; f != NULL; f = f->succ) {
	    success = bestfwd(m->mach,0,0);
	    for (s = f->oend; *s != '\0'  &&  success != FAILURE; s++) {
	        if ((success=bestfwd(m->mach,*s,0)) == SUCCESS) {
                    z = x;
                    do {
                        z = z->succ;
                        for (w = z->path; w != NULL; w = w->path) {
                            if (SetPath(v,w)) {
                                do {
                                    r = Spacer(v,w);
                                    if (r.beg != r.end) {
                                        n = (mtf *)w->data;
		                        for (g = n->LFM; g != NULL; g = g->succ) {
		                            if (g->oend-r.end < s  &&  s <= g->oend-r.beg) {
			                        AddEnd(f,FALSE,s);
                                                value = cvalue(m->mach)/clength(m->mach) + thresh(m->mach);
			                        if (g->best == NULL   ||   f->score+value > g->score   ||   f->score+value == g->score  &&  f->length+s-f->oend+1 > g->length) {
			                            g->delta  = s;
			                            g->length = f->length + s - f->oend + 1;
			                            g->score  = f->score  + value;
			                            g->best   = f;
			                        }
		                            }
		                        }
                                    }
                                } while (MorePaths(v,w));
                            }
                        }
                    } while (!z->cut);
	        }
	    }
        }
        if (m->LFM != NULL) {
            found = TRUE;
        }
    }

    /* remove dead ends */
    for (v = x->succ->path; v != NULL; v = v->path) {
        m = (mtf *)v->data;
        f = m->LFM;
        while (f != NULL) {
	    g = f;
	    f = f->succ;
	    if (g->best == NULL) {
	        DeleteMatch(&m->LFM,&m->LLM,g);
	    }
        }
    }

    return(found ? SUCCESS : (x->cut ? FAILURE : UNKNOWN));
}

static int MatchRight (x,y) front *x, *y; {
    register match *f, *g;
    register char *s, *i;
    register vtx *v, *w;
    char *j, *t, *I;
    int success, found, MatchLeft();
    float value;
    front *z;
    mtf *m, *n;
    rng r;

    for (I = _I+1, found = FALSE, v = y->path; v != NULL; v = v->path) {

        /* construct search interval */
        s = S + BSIZE;
        t = S;
        z = y;
        do {
            z = z->pred;
            for (w = z->path; w != NULL; w = w->path) {
                if (((mtf *)w->data)->RFM != NULL  &&  SetPath(w,v)) {
                    do {
                        r = Spacer(w,v);
                        if (r.beg != r.end) {
                            for (f = ((mtf *)w->data)->RFM; f != NULL; f = f->succ) {
                                if (S > f->oend+r.beg  &&  O > 0L) {
	                            warning("potential match exceeds buffer capacity(W4)");
                                    i = S;
                                } else {
                                    i = f->oend + r.beg;
                                }
                                if (s > i) {
                                    s = i;
                                }
                                if (S+BSIZE < f->oend+r.end) {
	                            warning("potential match exceeds buffer capacity(W5)");
                                    j = S + BSIZE;
                                } else {
                                    j = f->oend + r.end;
                                }
                                if (t < j) {
                                    t = j;
                                }
                                while (i < j) {
                                    I[i++-S] = 0x01;
                                }
                            }
                        }
                    } while (MorePaths(w,v));
                }
            }
        } while (!z->cut);

        /* find right ends of all possible matches for motif m */
        if (s < t) {
            m = (mtf *)v->data;
            success = movefwd(m->mach,0,0);
            for (i = I+(s-S); *s != '\0'   &&   (s < t  ||  success != FAILURE); s++) {
	        if ((success=movefwd(m->mach,*s,(int)*i)) == SUCCESS) {
	            AddMatch(&m->RFM,&m->RLM,s);
	        }
                *i++ = 0x00;
            }
            if (*s == '\0'  &&  s == S+BSIZE) {
	        warning("potential match exceeds buffer capacity(W6)");
            }
            if (m->RFM != NULL) {
                found = TRUE;
            }
        }

    }

    /* match next front */
    if (found  ||  !y->cut) {
        switch (NextFront(x,y)) {
        case LEFT:
            if (MatchLeft(x->pred,y) == FAILURE) {
                return(FAILURE);
            }
            break;
        case RIGHT:
            if (MatchRight(x,y->succ) == FAILURE) {
                return(FAILURE);
            }
            break;
        }
    } else {
	return(FAILURE);
    }

    /* for all matches found, find corresponding left ends */
    for (found = FALSE, v = y->path; v != NULL; v = v->path) {
        m = (mtf *)v->data;
        for (f = m->RFM; f != NULL; f = f->succ) {
	    success = bestrev(m->mach,0,0);
	    for (s = f->oend; *s != '\0'  &&  success != FAILURE; s--) {
	        if ((success=bestrev(m->mach,*s,0)) == SUCCESS) {
                    z = y;
                    do {
                        z = z->pred;
                        for (w = z->path; w != NULL; w = w->path) {
                            if (SetPath(w,v)) {
                                do {
                                    r = Spacer(w,v);
                                    if (r.beg != r.end) {
                                        n = (mtf *)w->data;
		                        for (g = n->RFM; g != NULL; g = g->succ) {
		                            if (g->oend+r.beg <= s  &&  s < g->oend+r.end) {
			                        AddEnd(f,FALSE,s);
                                                value = cvalue(m->mach)/clength(m->mach) + thresh(m->mach);
			                        if (g->best == NULL   ||   f->score+value > g->score   ||   f->score+value == g->score  &&  f->length+f->oend-s+1 > g->length) {
			                            g->delta  = s;
			                            g->length = f->length + f->oend - s + 1;
			                            g->score  = f->score  + value;
			                            g->best   = f;
			                        }
		                            }
		                        }
                                    }
                                } while (MorePaths(w,v));
                            }
                        }
                    } while (!z->cut);
	        }
	    }
        }
        if (m->RFM != NULL) {
            found = TRUE;
        }
    }

    /* remove dead ends */
    for (v = y->pred->path; v != NULL; v = v->path) {
        m = (mtf *)v->data;
        f = m->RFM;
        while (f != NULL) {
	    g = f;
	    f = f->succ;
	    if (g->best == NULL) {
	        DeleteMatch(&m->RFM,&m->RLM,g);
	    }
        }
    }

    return(found ? SUCCESS : (y->cut ? FAILURE : UNKNOWN));
}

static char *Advance (s,i) register char *s; int i;
{ register char *t;
  void Get_sequence();

  O += i;
  s = strcpy(S,S+i) + BSIZE - i;
  Get_sequence(s,i);
  return(s);
}

static int CombineSeed (m,b) register mtf *m; match *b; {
    register match *f, *g;
    register char *s;
    int success;
    float value;

    b->pred = b->succ = NULL;
    for (f = m->LFM; f != NULL; f = f->succ) {
        success = bestfwd(m->mach,0,0);
        for (s = f->oend; *s != '\0'  &&  success != FAILURE; s++) {
            if ((success=bestfwd(m->mach,*s,0)) == SUCCESS) {
                for (g = m->RFM; g != NULL; g = g->succ) {
                    if (s == g->oend) {
                        value = cvalue(m->mach)/clength(m->mach) + thresh(m->mach);
		        if (b->pred == NULL   ||   value+f->score+g->score > b->score   ||   value+f->score+g->score == b->score  &&  f->length+g->length+s-f->oend+1 > b->length) {
		            b->length = f->length + g->length + s - f->oend + 1;
		            b->score  = value + f->score + g->score;
		            b->pred   = f;
		            b->succ   = g;
		        }
		    }
	        }
            }
        }
    }
    if (b->pred != NULL) {
        f = m->LFM;
        while (f != NULL) {
	    success = movefwd(m->mach,0,0);
	    for (s = f->oend; *s != '\0'  &&  success != FAILURE; s++) {
	        if ((success=movefwd(m->mach,*s,0)) == SUCCESS) {
		    for (g = m->RFM; g != NULL; g = g->succ) {
		        if (s == g->oend) {
			    AddEnd(f,TRUE,s);
		        }
		    }
	        }
	    }
	    g = f;
	    f = f->succ;
	    if (g->iend == NULL) {
	        DeleteMatch(&m->LFM,&m->LLM,g);
	    }
        }
        f = m->RFM;
        while (f != NULL) {
	    success = moverev(m->mach,0,0);
	    for (s = f->oend; *s != '\0'  &&  success != FAILURE; s--) {
	        if ((success=moverev(m->mach,*s,0)) == SUCCESS) {
		    for (g = m->LFM; g != NULL; g = g->succ) {
		        if (s == g->oend) {
			    AddEnd(f,TRUE,s);
		        }
		    }
	        }
	    }
	    g = f;
	    f = f->succ;
	    if (g->iend == NULL) {
	        DeleteMatch(&m->RFM,&m->RLM,g);
	    }
        }
        return(TRUE);
    } else {
        return(FALSE);
    }
}

static FindRanges (x) front *x; {
    register end *d, *e;
    register match *f, *g;
    register vtx *v, *w;
    front *y, *z;
    mtf *m;
    rng r;


    /* remove invalid right ends */
    y = x;
    while (y->pred != NULL) {
        for (v = y->pred->path; v != NULL; v = v->path) {
            m = (mtf *)v->data;
	    f = m->LFM;
	    while (f != NULL) {
	        d = f->iend;
	        while (d != NULL) {
                    z = y->pred;
                    do {
                        z = z->succ;
                        for (w = z->path; w != NULL; w = w->path) {
                            if (SetPath(v,w)) {
                                do {
                                    r = Spacer(v,w);
                                    if (r.beg != r.end) {
		                        for (g = ((mtf *)w->data)->LFM; g != NULL; g = g->succ) {
		                            if (d->pos+r.beg <= g->oend  &&  g->oend < d->pos+r.end) {
			                        d->mark = TRUE;
                                                goto L1;
		                            }
		                        }
                                    }
                                } while (MorePaths(v,w));
                            }
                        }
                    } while (!z->cut);
                    L1:   e = d;
		    d = d->succ;
		    if (e->mark == FALSE) {
		        DeleteEnd(f,e);
		    }
	        }
	        g = f;
	        f = f->succ;
	        if (g->iend == NULL) {
		    DeleteMatch(&m->LFM,&m->LLM,g);
	        }
	    }
        }
        y = y->pred;
    }

    /* remove invalid left ends */
    while (y != x) {
        for (v = y->succ->path; v != NULL; v = v->path) {
            m = (mtf *)v->data;
	    f = m->LFM;
	    while (f != NULL) {
                z = y->succ;
                do {
                    z = z->pred;
                    for (w = z->path; w != NULL; w = w->path) {
                        if (SetPath(w,v)) {
                            do {
                                r = Spacer(w,v);
                                if (r.beg != r.end) {
	                            for (g = ((mtf *)w->data)->LFM; g != NULL; g = g->succ) {
		                        for (d = g->iend; d != NULL; d = d->succ) {
		                            if (d->pos+r.beg <= f->oend  &&  f->oend < d->pos+r.end) {
			                        f->mark = TRUE;
                                                goto L2;
		                            }
		                        }
	                            }
                                }
                            } while (MorePaths(w,v));
                        }
                    }
                } while (!z->cut);
	        L2:   g = f;
	        f = f->succ;
	        if (g->mark == FALSE) {
		    while (g->iend != NULL) {
		        DeleteEnd(g,g->iend);
		    }
		    DeleteMatch(&m->LFM,&m->LLM,g);
	        }
	    }
        }
        y = y->succ;
    }

    /* remove invalid left ends */
    while (y->succ != NULL) {
        for (v = y->succ->path; v != NULL; v = v->path) {
            m = (mtf *)v->data;
	    f = m->RFM;
	    while (f != NULL) {
	        d = f->iend;
	        while (d != NULL) {
                    z = y->succ;
                    do {
                        z = z->pred;
                        for (w = z->path; w != NULL; w = w->path) {
                            if (SetPath(w,v)) {
                                do {
                                    r = Spacer(w,v);
                                    if (r.beg != r.end) {
		                        for (g = ((mtf *)w->data)->RFM; g != NULL; g = g->succ) {
		                            if (d->pos-r.end < g->oend  &&  g->oend <= d->pos-r.beg) {
			                        d->mark = TRUE;
                                                goto L3;
		                            }
		                        }
                                    }
                                } while (MorePaths(w,v));
                            }
                        }
                    } while (!z->cut);
		    L3:   e = d;
		    d = d->succ;
		    if (e->mark == FALSE) {
		        DeleteEnd(f,e);
		    }
	        }
	        g = f;
	        f = f->succ;
	        if (g->iend == NULL) {
		    DeleteMatch(&m->RFM,&m->RLM,g);
	        }
	    }
        }
        y = y->succ;
    }

    /* remove invalid right ends */
    while (y != x) {
        for (v = y->pred->path; v != NULL; v = v->path) {
            m = (mtf *)v->data;
	    f = m->RFM;
	    while (f != NULL) {
                z = y->pred;
                do {
                    z = z->succ;
                    for (w = z->path; w != NULL; w = w->path) {
                        if (SetPath(v,w)) {
                            do {
                                r = Spacer(v,w);
                                if (r.beg != r.end) {
	                            for (g = ((mtf *)w->data)->RFM; g != NULL; g = g->succ) {
		                        for (d = g->iend; d != NULL; d = d->succ) {
		                            if (d->pos-r.end < f->oend  &&  f->oend <= d->pos-r.beg) {
			                        f->mark = TRUE;
                                                goto L4;
		                            }
		                        }
	                            }
                                }
                            } while (MorePaths(v,w));
                        }
                    }
                } while (!z->cut);
	        L4:   g = f;
	        f = f->succ;
	        if (g->mark == FALSE) {
		    while (g->iend != NULL) {
		        DeleteEnd(g,g->iend);
		    }
		    DeleteMatch(&m->RFM,&m->RLM,g);
	        }
	    }
        }
        y = y->pred;
    }
}

static BufferMatch (x,b,r) front *x; match *b; result *r; {
    char *s, *t;
    int i, j, k;
    match *f, *g;
    vtx *v;
    front *y, *z;
    end *e;

    r->beg = ((mtf *)x->path->data)->LLM->oend - _S + O;
    r->end = ((mtf *)x->path->data)->RLM->oend - _S + O;

    i = 1;
    z = x;
    s = ((mtf *)x->path->data)->LLM->oend;
    for (y = x->pred; y != NULL; y = y->pred) {
        for (v = y->path; v != NULL; v = v->path) {
            if (((mtf *)v->data)->LFM != NULL) {
                for (f = ((mtf *)v->data)->LFM; f != NULL; f = f->succ) {
                    if (r->beg > f->oend-_S+O) {
                        r->beg = f->oend - _S + O;
                        s = f->oend;
                    }
                    for (e = f->iend; e != NULL; e = e->succ) {
                        if (r->end < e->pos-_S+O) {
                            r->end = e->pos - _S + O;
                        }
                    }
                }
                i += 1;
            }
        }
        if (y->pred == NULL) {
            z = y;
        }
    }
    for (y = x->succ; y != NULL; y = y->succ) {
        for (v = y->path; v != NULL; v = v->path) {
            if (((mtf *)v->data)->RFM != NULL) {
                for (f = ((mtf *)v->data)->RFM; f != NULL; f = f->succ) {
                    if (r->end < f->oend-_S+O) {
                        r->end = f->oend - _S + O;
                    }
                    for (e = f->iend; e != NULL; e = e->succ) {
                        if (r->beg > e->pos-_S+O) {
                            r->beg = e->pos - _S + O;
                            s = e->pos;
                        }
                    }
                }
                i += 1;
            }
        }
    }
    if ((r->string = t = (char *)malloc((unsigned)(((i+1)*(r->end-r->beg+2)+1)*sizeof(char)))) == NULL) {
	fatal("out of memory");
    }
    if ((r->values = (float *)malloc((unsigned)(i*sizeof(float)))) == NULL) {
	fatal("out of memory");
    }
    t = strncpy(t,s,(int)(r->end-r->beg)+1);
    t[r->end-r->beg+1] = '\0';
    t += strlen(t) + 1;
    for (i = 0, y = z; y != x; y = y->succ) {
        for (v = y->path; v != NULL; v = v->path) {
            if (((mtf *)v->data)->LFM != NULL) {
                for (j = 0; j <= r->end-r->beg; j++) {
                    t[j] = ' ';
                }
                r->values[i] = -1.0;
	        for (f = ((mtf *)v->data)->LFM; f != NULL; f = f->succ) {
	            for (j = f->oend-s; j <= f->last->pos-s; j++) {
		        t[j] = '.';
                    }
	        }
                t[r->end-r->beg+1] = '\0';
                t += strlen(t) + 1;
                i += 1;
            }
        }
    }
    for (k = i; y != NULL; y = y->succ) {
        for (v = y->path; v != NULL; v = v->path) {
            if (((mtf *)v->data)->RFM != NULL) {
                for (j = 0; j <= r->end-r->beg; j++) {
                    t[j] = ' ';
                }
                r->values[i] = -1.0;
	        for (f = ((mtf *)v->data)->RFM; f != NULL; f = f->succ) {
	            for (j = f->oend-s; j >= f->last->pos-s; j--) {
		        t[j] = '.';
                    }
	        }
                t[r->end-r->beg+1] = '\0';
                t += strlen(t) + 1;
                i += 1;
            }
        }
    }
    t[0] = '\0';
    r->val = b->score;
    r->values[k] = b->score - b->pred->score - b->succ->score;
    t = r->string + (k+1)*(r->end-r->beg+2);
    for (j = b->pred->oend-s; j <= b->succ->oend-s; j++) {
	t[j] = '^';
    }
    i = k;
    f = b->pred;
    for (y = x->pred; f->best != NULL  &&  y != NULL; y = y->pred) {
        for (v = y->path; f->best != NULL  &&  v != NULL; v = v->path) {
            if (((mtf *)v->data)->LFM != NULL) {
                i -= 1;
                for (g = ((mtf *)v->data)->LFM; f->best != NULL  &&  g != NULL; g = g->succ) {
                    if (f->best == g) {
                        r->values[i] = f->score - g->score;
                        t = r->string + (i+1)*(r->end-r->beg+2);
	                for (j = f->delta-s; j >= f->best->oend-s; j--) {
	                    t[j] = '^';
	                }
                        f = f->best;
                    }
                }
            }
        }
    }
    i = k;
    f = b->succ;
    for (y = x->succ; f->best != NULL  &&  y != NULL; y = y->succ) {
        for (v = y->path; f->best != NULL  &&  v != NULL; v = v->path) {
            if (((mtf *)v->data)->RFM != NULL) {
                i += 1;
                for (g = ((mtf *)v->data)->RFM; f->best != NULL  &&  g != NULL; g = g->succ) {
                    if (f->best == g) {
                        r->values[i] = f->score - g->score;
                        t = r->string + (i+1)*(r->end-r->beg+2);
	                for (j = f->delta-s; j <= f->best->oend-s; j++) {
	                    t[j] = '^';
	                }
                        f = f->best;
                    }
                }
            }
        }
    }
}

static ReportMatch(r) register result *r; {
    register char *s, *t, *a, *b;
    register int j;
    char *send;
    int   rbeg, len,

    hw = 0;
    for (s = r->string; *s != '\0'; s += strlen(s)+1)
      { AP[hw] = s;
        hw += 1;
      }
    AP[hw] = s;
    rbeg = r->beg;
    send = AP[1]-1;
    printf("\n");
    for (s = AP[0]; s < send; s += len)
      { len = send-s;
        if (len > 50) len = 50;
        printf("%10ld> ",rbeg);
        for (t = s; t < s+len; t++)
          if (' ' <= *t && *t <= '~')
            printf("%c",*t);
          else if (*t == '\n')
            printf("\\n");
          else if (*t == '\t')
            printf("\\t");
          else
            printf("\\%03o",*t);
        printf(" <%-10ld\n",rbeg+(len-1));
        rbeg += len;
        for (j = 1; j < hw; j++)
          { a = AP[j];
            b = AP[j]+len;
            for (t = a; t < b; t++)
              if (*t != ' ') break;
            if (t < b)
              { printf("%10.4f: ",r->values[j-1]);
                for (t = s; t < s+len; t++, a++)
                  if (' ' <= *t && *t <= '~')
                    printf("%c",*a);
                  else if (*t == '\n' || *t == '\t')
                    printf("%c%c",*a,*a);
                  else
                    printf("%c%c%c%c",*a,*a,*a,*a);
                if (b[-1] != ' ' && *b != ' ' && *b != '\0')
                  printf("->");
                printf("\n");
              }
            AP[j] = b;
          }
        printf("\n");
      }
    printf("\n");

}

static int MatchSeed () {
    register char *s, *t;
    register mtf **m, *n;
    int i, j, success, matches;
    front *x;
    match b;
    vtx *v;
    void Get_sequence();

    /* initialize */
    i = 0;
    s = _S;
    matches = 0;
    for (m = M, x = X; x->path != NULL; x++, m++) {
        *m = (mtf *)x->path->data;
	movefwd((*m)->mach,0,0);
    }
    *m = NULL;

    /* scan */
    O = 0L;
    Get_sequence(S,BSIZE);
    do {

	/* find a maximal interval for seed motif */
        do {
            if (*++s == '\0'  &&  s == S+BSIZE) {
                s = Advance(s,2*MAXMATCH);
            }
            success = FALSE;
            for (m = M; *m != NULL; m++) {
                if ((*m)->last > 0L) {
                    (*m)->last -= 1L;
                } else if (movefwd((*m)->mach,*s,1) == SUCCESS) {
                    (*m)->last = 1L;
                    success = TRUE;
                }
            }
	} while (*s != '\0'  &&  !success);

        if (*s != '\0') {

            /* shift string in buffer */
            if (s >= S + 2*MAXMATCH) {
                t = Advance(t,j=(s-S)-MAXMATCH);
                s -= j;
            }

            for (x = X, m = M; *m != NULL; m++, x++) {
                if ((*m)->last > 0L) {

                    /* find right and left ends */
	            t = s;
	            do {
		        AddMatch(&(*m)->RFM,&(*m)->RLM,t);
	            } while (*++t != '\0'  &&  movefwd((*m)->mach,*t,1) == SUCCESS);
	            if (*t == '\0'  &&  t == S+BSIZE) {
		        warning("potential match exceeds buffer capacity(W7)");
	            }

                    (*m)->last = t - s;
		    savestate((*m)->mach);
	            moverev((*m)->mach,0,0);

		    t--;
		    while (t > s) {
			if (moverev((*m)->mach,*t,1) == SUCCESS)
			    AddMatch(&(*m)->LFM,&(*m)->LLM,t);
			t--;
	            }
		    do {
			success = moverev((*m)->mach,*t,0);
			if (success == SUCCESS)
		            AddMatch(&(*m)->LFM,&(*m)->LLM,t);
		    } while (success != FAILURE && *--t != '\0');
	            if (*t == '\0'  &&  t == _S  &&  O > 0L) {
		        warning("potential match exceeds buffer capacity(W8)");
	            }

	            /* match remaining motifs */
                    switch (NextFront(x,x)) {
                    case LEFT:
                        success = MatchLeft(x->pred,x);
                        break;
                    case RIGHT:
                        success = MatchRight(x,x->succ);
                        break;
                    case DONE:
                        success = SUCCESS;
                        break;
                    }

	            if (success != FAILURE  &&  CombineSeed(*m,&b)) {

	                /* compute ranges and buffer result */
	                FindRanges(x);
	                BufferMatch(x,&b,_B+i);

	                /* report a match */
	                if (matches++) {
		            if ((float)(_B[(i+1)%2].end-_B[i].beg+1)/(float)(_B[i].end-_B[(i+1)%2].beg+1) < OVERLAP) {
		                i = (i+1) % 2;
		                ReportMatch(_B+i);
		            } else if (_B[(i+1)%2].val < _B[i].val  ||  _B[(i+1)%2].val == _B[i].val  &&  _B[(i+1)%2].end-_B[(i+1)%2].beg < _B[i].end-_B[i].beg) {
		                i = (i+1) % 2;
		            }
		            free((char *)_B[i].string), _B[i].string = NULL;
		            free((char *)_B[i].values), _B[i].values = NULL;
	                } else {
                            printf("%s",H);
		            i = (i+1) % 2;
	                }

	            }

	            /* clean up */
                    for (v = A; v < Z; v++) {
                        if (v->type == MOTIF) {
                            n = (mtf *)v->data;
	                    while (n->LFM != NULL) {
		                while (n->LFM->iend != NULL) {
		                    DeleteEnd(n->LFM,n->LFM->iend);
		                }
		                DeleteMatch(&n->LFM,&n->LLM,n->LFM);
                            }
	                    while (n->RFM != NULL) {
		                while (n->RFM->iend != NULL) {
		                    DeleteEnd(n->RFM,n->RFM->iend);
		                }
		                DeleteMatch(&n->RFM,&n->RLM,n->RFM);
	                    }
                        }
	            }

		    restorestate((*m)->mach);

                }
            }

	}

    } while (*s != '\0');

    /* report last match */
    if (matches) {
	i = (i+1) % 2;
	ReportMatch(_B+i);
	free((char *)_B[i].string), _B[i].string = NULL;
	free((char *)_B[i].values), _B[i].values = NULL;
    }

    return(matches);
}

Search (p) node *p; {
    register vtx *x, *y;
    front f, *g, *h;
    vtx *v, *w;
    int i, j;
    char *Next_header();

    /* build network */
    i = VtxCount(p) + 2;
    if ((N = (net *)malloc((unsigned)sizeof(net))) == NULL) {
        fatal("out of memory");
    }
    if ((A = Z = (vtx *)malloc((unsigned)(sizeof(vtx)*i))) == NULL) {
        fatal("out of memory");
    }
    N->beg = NewVtx(EMPTY,(vtx *)NULL);
    i = BuildNetwork(p,&v,&w);
    N->end = NewVtx(EMPTY,(vtx *)NULL);
    N->beg->lsucc = N->beg->rsucc = v;
    v->lpred = v->rpred = N->beg;
    w->lsucc = w->rsucc = N->end;
    N->end->lpred = N->end->rpred = w;
    N->beg->stype = v->ptype = w->stype = N->end->ptype = CNCAT;
    N->beg->ptype = N->end->stype = FINAL;
#ifdef DEBUG
    ShowNet(N);
#endif

    /* find minimum cut */
    if ((I = (float *)malloc((unsigned)(sizeof(float)*i))) == NULL) {
        fatal("out of memory");
    }
    if ((C = (float *)malloc((unsigned)(sizeof(float)*i))) == NULL) {
        fatal("out of memory");
    }
    if ((F = (float *)malloc((unsigned)(sizeof(float)*i))) == NULL) {
        fatal("out of memory");
    }
    if ((D = (float *)malloc((unsigned)(sizeof(float)*i))) == NULL) {
	fatal("out of memory");
    }
    if ((AP = (char **)malloc((unsigned)(sizeof(char *)*(i+2)))) == NULL) {
	fatal("out of memory");
    }
    if ((Q = (float *)malloc((unsigned)(sizeof(float)*i*i))) == NULL) {
        fatal("out of memory");
    }
    if ((T = (char  *)malloc((unsigned)(sizeof(char )*i*i))) == NULL) {
        fatal("out of memory");
    }
    for (x = N->beg; x <= N->end; x++) {
	if (x->type == MOTIF) {
	    NetworkCost(x,&i,&j,&f);
	    ((mtf *)x->data)->bval = Q[j*i+j];
	    for (g = &f; g != NULL; ) {
		h = g;
		g = g->pred;
		free((char *)h);
	    }
            for (g = &f; g != NULL; ) {
		h = g;
		g = g->succ;
		free((char *)h);
	    }
	    for (y = N->beg; y <= N->end; y++) {
		if (y->type == MOTIF) {
		    y->path = NULL;
		}
	    }
        }
    }
    x = MinimumCut(N->beg->lsucc,N->end->lpred);
#ifdef DEBUG
    fprintf(stderr,"\n");
    for (y = x; y != NULL; y = y->merge) {
        fprintf(stderr,"%d   ",y-N->beg);
    }
    fprintf(stderr,"\n");
#endif

    /* find order */
    for (i = 0, y = x; y != NULL; y = y->merge, i++);
    if ((X = (front *)malloc((unsigned)(sizeof(front)*(i+1)))) == NULL) {
	fatal("out of memory");
    }
    FindOrder(x);

    /* scan the string */
    if ((M = (mtf **)malloc((unsigned)(sizeof(mtf *)*(i+1)))) == NULL) {
	fatal("out of memory");
    }
    printf("\n");
    i = FALSE;
    for (j = 0; (H = Next_header()) != 0; j++)
      { if (MatchSeed())
          i = TRUE;
      }
    if (!i) {
        printf("*** no match\n\n");
    }

    /* clean up */
    free((char *)M), M = NULL;
    for (g = X; g->path != NULL; g++) {
        h = g;
        while (h->pred != NULL) {
            h = h->pred;
        }
        while (h != g) {
            h = h->succ;
            free((char *)h->pred);
        }
        while (h->succ != NULL) {
            h = h->succ;
        }
        while (h != g) {
            h = h->pred;
            free((char *)h->succ);
        }
    }
    free((char *)X), X = NULL;
    free((char *)T), T = NULL;
    free((char *)Q), Q = NULL;
    free((char *)D), D = NULL;
    free((char **)AP), AP = NULL;
    free((char *)F), F = NULL;
    free((char *)C), C = NULL;
    free((char *)I), I = NULL;
    for (x = A; x < Z; x++) {
        if (x->type == MOTIF) {
            destroy(((mtf *)x->data)->mach);
            free((char *)x->data);
        } else if (x->type == RANGE) {
            free((char *)x->data);
        }
    }
    free((char *)A), A = NULL;
    free((char *)N), N = NULL;
}

CleanSearch () {
    register front *f, *g;
    register vtx *v;
    register mtf *m;

    if (_B[0].string != NULL) {
        free((char *)_B[0].string), _B[0].string = NULL;
    }
    if (_B[0].values != NULL) {
        free((char *)_B[0].values), _B[0].values = NULL;
    }
    if (_B[1].string != NULL) {
        free((char *)_B[1].string), _B[1].string = NULL;
    }
    if (_B[1].values != NULL) {
        free((char *)_B[1].values), _B[1].values = NULL;
    }
    if (M != NULL) {
        free((char *)M), M = NULL;
    }
    if (X != NULL) {
        for (f = X; f->path != NULL; f++) {
            g = f;
            while (g->pred != NULL) {
                g = g->pred;
            }
            while (g != f) {
                g = g->succ;
                free((char *)g->pred);
            }
            while (g->succ != NULL) {
                g = g->succ;
            }
            while (g != f) {
                g = g->pred;
                free((char *)g->succ);
            }
        }
        free((char *)X), X = NULL;
    }
    if (Q != NULL) {
        free((char *)Q), Q = NULL;
    }
    if (T != NULL) {
        free((char *)T), T = NULL;
    }
    if (D != NULL) {
        free((char *)D), D = NULL;
    }
    if (F != NULL) {
        free((char *)F), F = NULL;
    }
    if (C != NULL) {
        free((char *)C), C = NULL;
    }
    if (I != NULL) {
        free((char *)I), I = NULL;
    }
    if (A != NULL) {
        for (v = A; v < Z; v++) {
            if (v->type == MOTIF) {
                m = (mtf *)v->data;
	        while (m->LFM != NULL) {
		    while (m->LFM->iend != NULL) {
		        DeleteEnd(m->LFM,m->LFM->iend);
		    }
		    DeleteMatch(&m->LFM,&m->LLM,m->LFM);
                }
	        while (m->RFM != NULL) {
		    while (m->RFM->iend != NULL) {
		        DeleteEnd(m->RFM,m->RFM->iend);
		    }
		    DeleteMatch(&m->RFM,&m->RLM,m->RFM);
	        }
                destroy(m->mach);
                free((char *)v->data);
            } else if (v->type == RANGE) {
                free((char *)v->data);
            }
        }
        free((char *)A), A = NULL;
    }
    if (N != NULL) {
        free((char *)N), N = NULL;
    }
}
