/*
    This file is part of GNU APL, a free implementation of the
    ISO/IEC Standard 13751, "Programming Language APL, Extended"

    Copyright (C) 2008-2015  Dr. Jürgen Sauermann

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <errno.h>
#include <fenv.h>
#include <math.h>

#include "Workspace.hh"
#include "Value.icc"
#include "Error.hh"
#include "ComplexCell.hh"
#include "FloatCell.hh"
#include "IntCell.hh"

//-----------------------------------------------------------------------------
ErrorCode
NumericCell::bif_not(Cell * Z) const
{
   if (!is_near_bool())   return E_DOMAIN_ERROR;

   if (get_near_bool())   new (Z) IntCell(0);
   else                   new (Z) IntCell(1);
   return E_NO_ERROR;
}
//-----------------------------------------------------------------------------

const uint64_t N_choose_8[] = { (887), 1, 9, 45, 165, 495, 1287, 3003, 6435,
 12870, 24310, 43758, 75582, 0x1EC12, 0x31AE2, 0x4E11A, 0x77B4A, 0xB38EF,
 0x1080E7, 0x17D6A3, 0x21E02B, 0x2F6D09, 0x417E31, 0x594EFD, 0x785F55,
 0xA07F1C, 0xD3DAFC, 0x1150AAC, 0x16720CC, 0x1CDBC74, 0x24D1C94, 0x2EA3544,
 0x3AAC724, 0x49578ED, 0x5B1F355, 0x708FF69, 0x8A4A781, 0xA905AF3, 0xCD914AB,
 0xF8D84CF, 0x12BE3D97, 0x167DE382, 0x1AE16112, 0x20001E3A, 0x25F43BAA,
 0x2CDAD226, 0x34D43616, 0x3E043F7E, 0x4892968E, 0x54AB04FB, 0x627DCC53,
 0x7240017F, 0x842BEDA7, 0x988174AD, 0xAF868175, 0xC9877831, 0xE6D7AEE9,
 0x107D1EC78ULL, 0x12CD8EE38ULL, 0x15657F498ULL, 0x184C356D8ULL,
 0x1B8991E28ULL, 0x1F261A868ULL, 0x232B052C8ULL, 0x27A242C88ULL,
 0x2C968B219ULL, 0x3213690E1ULL, 0x3825473E5ULL, 0x3ED97D99DULL,
 0x463E5F337ULL, 0x4E6348D8FULL, 0x5758B0423ULL, 0x613033E4BULL,
 0x6BFCAB6FEULL, 0x77D238F6EULL, 0x84C65ACC6ULL, 0x92EFFE156ULL,
 0xA2679217AULL, 0xB3471C48AULL, 0xC5AA4D222ULL, 0xD9AE95C12ULL,
 0xEF733E547ULL, 0x107197D5FFULL, 0x120C48FD9BULL, 0x13C99D2263ULL,
 0x15AC0D9F91ULL, 0x17B63911F9ULL, 0x19EAE513A5ULL, 0x1C4D0006BDULL,
 0x1EDFA2F014ULL, 0x21A61361B4ULL, 0x24A3C575C4ULL, 0x27DC5DDA24ULL,
 0x2B53B3ED1CULL, 0x2F0DD3EB7CULL, 0x330F01308CULL, 0x375BB8882CULL,
 0x3BF8B29385ULL, 0x40EAE640ADULL, 0x46378B55A1ULL, 0x4BE41D0EF9ULL,
 0x51F65CD2BBULL, 0x587454F7B3ULL, 0x5F645BA1B7ULL, 0x66CD15B33FULL,
 0x6EB579D4BAULL, 0x7724D3920AULL, 0x8022C68E92ULL, 0x89B751D042ULL,
 0x93EAD3220EULL, 0x9EC60A8E3EULL, 0xAA521DF106ULL, 0xB6989CA3D6ULL,
 0xC3A38341D3ULL, 0xD17D3F85EBULL, 0xE030B442F7ULL, 0xEFC93D765FULL,
 0x10052B475B5ULL, 0x111D97437BDULL, 0x1246A5DB959ULL, 0x13812DC7ED1ULL,
 0x14CE0EB31F0ULL, 0x162E3185D70ULL, 0x17A288B4630ULL, 0x192C108E2B0ULL,
 0x1ACBCF8F150ULL, 0x1C82D6B2DD0ULL, 0x1E5241CA690ULL, 0x203B37D3210ULL,
 0x223EEB50531ULL, 0x245E9AA6AB9ULL, 0x269B9079C9DULL, 0x28F7240BF95ULL,
 0x2B72B9A017FULL, 0x2E0FC2DDB17ULL, 0x30CFBF3758BULL, 0x33B43C53473ULL,
 0x36BED6764B6ULL, 0x39F138F10E6ULL, 0x3D4D1E8FB9EULL, 0x40D4520C06EULL,
 0x4488AE81BE2ULL, 0x486C1FE5B32ULL, 0x4C80A37F42AULL, 0x50C848645DAULL,
 0x55452FF829FULL, 0x59F98E6C417ULL, 0x5EE7AB44993ULL, 0x6411E1DE19BULL,
 0x697AA1F7F19ULL, 0x6F24703FAC1ULL, 0x7511E6E014DULL, 0x7B45B612F25ULL,
 0x81C2A4B5A0CULL, 0x888B90E096CULL, 0x8FA37081DDCULL, 0x970D51FA87CULL,
 0x9ECC5CBF2C4ULL, 0xA6E3D1FB764ULL, 0xAF570D38CD4ULL, 0xB8298508234ULL,
 0xC15ECBAEF1DULL, 0xCAFA8FD7705ULL, 0xD5009D440D9ULL, 0xDF74DD86371ULL,
 0xEA5B58B8783ULL, 0xF5B8363BFBBULL, 0x1018FBD7979FULL, 0x10DE656A59E7ULL,
 0x11AC08B88EF2ULL, 0x12823084B402ULL, 0x136129C42BEAULL, 0x144943AC69DAULL,
 0x153ACFC056F6ULL, 0x163621DDF366ULL, 0x173B904C338EULL, 0x184B73C91A1EULL,
 0x196627980FABULL, 0x1A8C09907883ULL, 0x1BBD7A2C896FULL, 0x1CFADC985C17ULL,
 0x1E4496C143BDULL, 0x1F9B11656305ULL, 0x20FEB8238381ULL, 0x226FF98B2FB9ULL,
 0x23EF472D1068ULL, 0x257D15AB8DA8ULL, 0x2719DCCBB4C8ULL, 0x28C617866388ULL,
 0x2A824419B978ULL, 0x2C4EE41AD038ULL, 0x2E2C7C87BB58ULL, 0x301B95D9D098ULL,
 0x321CBC183949ULL, 0x34307EEACD91ULL, 0x365771AD3A55ULL, 0x38922B82728DULL,
 0x3AE147686CC7ULL, 0x3D45644C2D9FULL, 0x3FBF251E1FF3ULL, 0x424F30E6BB9BULL,
 0x44F632DB7B6EULL, 0x47B4DA74235EULL, 0x4A8BDB805776ULL, 0x4D7BEE3D8486ULL,
 0x5085CF6D1B4AULL, 0x53AA406B1EDAULL, 0x56EA07450732ULL, 0x5A45EED0F8A2ULL,
 0x5DBEC6C550F7ULL, 0x615563D08B2FULL, 0x650A9FB17A8BULL, 0x68DF594FDDD3ULL,
 0x6CD474D54BA1ULL, 0x70EADBC67889ULL, 0x75237D1CD7F5ULL, 0x797F4D60988DULL,
 0x7DFF46C2FD04ULL, 0x82A469391224ULL, 0x876FBA96C2F4ULL, 0x8C6246AA4BD4ULL,
 0x917D1F580D6CULL, 0x96C15CB6C04CULL, 0x9C301D2C0A1CULL, 0xA1CA8589753CULL,
 0xA791C129CBB5ULL, 0xAD87020ED65DULL, 0xB3AB80FF8111ULL, 0xBA007DA664E9ULL,
 0xC0873EB0B94BULL, 0xC74111EDACC3ULL, 0xCE2F4C6E2687ULL, 0xD5534AA4F18FULL,
 0xDCAE7087522AULL, 0xE44229AE06FAULL, 0xEC0FE976B642ULL, 0xF4192B25C872ULL,
 0xFC5F7208B0DEULL, 0x104E44998A58EULL, 0x10DA9459DC716ULL,
 0x116B00252B966ULL, 0x11FFA2488AE83ULL, 0x1298959CBE41BULL,
 0x1335F588894E7ULL, 0x13D7DE0305ECFULL, 0x147E6B9601EC5ULL,
 0x1529BB606434DULL, 0x15D9EB18996A9ULL, 0x168F190F081A1ULL,
 0x174964308C7E0ULL, 0x1808EC08FBEE0ULL, 0x18CDD0C5B0060ULL,
 0x1998333819960ULL, 0x1A6834D85B6A0ULL, 0x1B3DF7C7ECFA0ULL,
 0x1C199ED445120ULL, 0x1CFB4D798C820ULL, 0x1DE327E558E61ULL,
 0x1ED152F96F969ULL, 0x1FC5F44E90D0DULL, 0x20C132374B285ULL,
 0x21C333C2D750FULL, 0x22CC20BFFC527ULL, 0x23DC21BFFC35BULL,
 0x24F36019893C3ULL, 0x261205EBC3B26ULL, 0x27383E21406D6ULL,
 0x286634731804EULL, 0x299C156BFED9EULL, 0x2ADA0E6B65FB2ULL,
 0x2C204DA8A4F82ULL, 0x2D6F02362CB3AULL, 0x2EC65C04C346AULL,
 0x30268BE6C904FULL, 0x318FC39386B47ULL, 0x330235AA85083ULL,
 0x347E15B6ED70BULL, 0x36039832F4529ULL, 0x3792F28B4CB51ULL,
 0x392C5B22A579DULL, 0x3AD00955302F5ULL, 0x3C7E357C318FCULL,
 0x3E3718F19BBDCULL, 0x3FFAEE13B250CULL, 0x41C9F048B842CULL,
 0x43A45C02A7D14ULL, 0x458A6EC2F4634ULL, 0x477C671E56864ULL,
 0x497A84C0A2144ULL, 0x4B850870A694DULL, 0x4D9C341419EB5ULL,
 0x4FC04AB38D649ULL, 0x51F1907E6D361ULL, 0x54304ACF0A813ULL,
 0x567CC02EAFECBULL, 0x58D73859C0E6FULL, 0x5B3FFC43E3A37ULL,
 0x5DB7561C35E62ULL, 0x603D91518CAF2ULL, 0x62D2FA96BED9AULL,
 0x6577DFE6FAC0AULL, 0x682C908A26FC6ULL, 0x6AF15D194E4B6ULL,
 0x6DC6978316B9EULL, 0x70AC9310441AEULL, 0x73A3A46845E5BULL,
 0x76AC2195D08B3ULL, 0x79C6620B8255FULL, 0x7CF2BEA893E87ULL,
 0x803191BD946CDULL, 0x8383371131895ULL, 0x86E80BE50B2D1ULL,
 0x8A606EFA93489ULL, 0x8DECC097F9858ULL, 0x918D628D23118ULL,
 0x9542B838AE8F8ULL, 0x990D268D04438ULL, 0x9CED1415728C8ULL,
 0xA0E2E8FB56C08ULL, 0xA4EF0F0B527E8ULL, 0xA911F1BA8D8A8ULL,
 0xAD4BFE2C04479ULL, 0xB19DA335E2E41ULL, 0xB6075166ED4C5ULL,
 0xBA897B0BF3F7DULL, 0xBF24943555A57ULL, 0xC3D912BC8E1AFULL,
 0xC8A76E49D1FC3ULL, 0xCD902059B7CEBULL, 0xD293A442EE3DEULL,
 0xD7B2773BFFB4EULL, 0xDCED186123626ULL, 0xE24408BA1BBB6ULL,
 0xE7B7CB402291AULL, 0xED48E4E3E2D2AULL, 0xF2F7DC9380042ULL,
 0xF8C53B40AB932ULL, 0xFEB18BE6C80A7ULL, 0x104BD5B911A45FULL,
 0x10AE9396108B7BULL, 0x11135B69468D43ULL, 0x117A3668BDABB1ULL,
 0x11E32DED133319ULL, 0x124E4B71DF4045ULL, 0x12BB98961D2E5DULL,
 0x132B1F1C94EDF4ULL, 0x139CE8EC454694ULL, 0x14110010CF0424ULL,
 0x14876EBAE11184ULL, 0x15003F40A581BCULL, 0x157B7C1E2F891CULL,
 0x15F92FF5EA67ACULL, 0x1679659109464CULL, 0x16FC27DFF807E5ULL,
 0x178181FACD100DULL, 0x18097F21BC0081ULL, 0x18942ABD896ED9ULL,
 0x1921905FFF93DBULL, 0x19B1BBC463F5D3ULL, 0x1A44B8CFEE0F57ULL,
 0x1ADA93923EF3DFULL, 0x1B735845D9F39AULL, 0x1C0F13509E3FEAULL,
 0x1CADD1444191F2ULL, 0x1D4F9EDECBD4A2ULL, 0x1DF4890B13D3AEULL,
 0x1E9C9CE13CF0DEULL, 0x1F47E7A735E126ULL, 0x1FF676D13872F6ULL,
 0x20A858024A5F33ULL, 0x215D990CBF264BULL, 0x221647F2BAFAD7ULL,
 0x22D272E6B6BB3FULL, 0x2392284C04FBD5ULL, 0x245576B75822DDULL,
 0x251C6CEF4997F9ULL, 0x25E719ECE20871ULL, 0x26B58CDC22C1D0ULL,
 0x2787D51C902450ULL, 0x285E0241BD2E90ULL, 0x29382413D82410ULL,
 0x2A164A90384FF0ULL, 0x2AF885E9ECE570ULL, 0x2BDEE68A4CFFB0ULL,
 0x2CC97D1188C230ULL, 0x2DB85A573B9B91ULL, 0x2EAB8F6AFFAC19ULL,
 0x2FA32D9502517DULL, 0x309F465699D975ULL, 0x319FEB6ADC5C9FULL,
 0x32A52EC737C337ULL, 0x33AF229C0AF52BULL, 0x34BDD955403713ULL,
 0x35D1659AE8B596ULL, 0x36E9DA51D940C6ULL, 0x38074A9C4838FEULL,
 0x3929C9DA6CAECEULL, 0x3A516BAB1EB782ULL, 0x3B7E43EC78F7D2ULL,
 0x3CB066BC7B664AULL, 0x3DE7E879AF46FAULL, 0x3F24DDC3CC61FFULL,
 0x40675B7C5F7677ULL, 0x41AF76C771EB73ULL, 0x42FD450C32C07BULL,
 0x4450DBF5A0BF39ULL, 0x45AA517335EFE1ULL, 0x4709BBB99451EDULL,
 0x486F314333DAC5ULL, 0x49DAC8D111BBECULL, 0x4B4C996B60F24CULL,
 0x4CC4BA623C203CULL, 0x4E43434E58B3DCULL, 0x4FC84C11BB5B64ULL,
 0x5153ECD86DC904ULL, 0x52E63E1935C7F4ULL, 0x547F58964DA454ULL,
 0x561F555E1DE77DULL, 0x57C64DCBF86A65ULL, 0x59745B88D4BFB9ULL,
 0x5B29988C0DF751ULL, 0x5CE61F1C21BCA3ULL, 0x5EAA09CF70D1DBULL,
 0x6075738D00E93FULL, 0x6248778D3FDE87ULL, 0x6423315AC851D2ULL,
 0x6605BCD327A5E2ULL, 0x67F03627A5634AULL, 0x69E2B9DE0C023AULL,
 0x6BDD64D1731C96ULL, 0x6DE054330B0A06ULL, 0x6FEBA58AE9E7AEULL,
 0x71FF76B8DA0D3EULL, 0x741BE5F529F10BULL, 0x764111D17D7CE3ULL,
 0x786F1939A0D54FULL, 0x7AA61B745C94F7ULL, 0x7CE638244B7DDDULL,
 0x7F2F8F48B1A225ULL, 0x8182413E550621ULL, 0x83DE6EC057BD59ULL,
 0x864438E9138448ULL, 0x88B3C132F6D888ULL, 0x8B2D2979639128ULL,
 0x8DB093F98EF8E8ULL, 0x903E2353636C18ULL, 0x92D5FA8A637BD8ULL,
 0x95783D068E9878ULL, 0x98250E954744B8ULL, 0x9ADC936A3AD3A9ULL,
 0x9D9EF0204AB2F1ULL, 0xA06C49BA774335ULL, 0xA344C5A4CC406DULL,
 0xA62889B54EBBE7ULL, 0xA917BC2CECA9BFULL, 0xAC1283B86E0393ULL,
 0xAF19077167823BULL, 0xB22B6EDF2EF04EULL, 0xB549E1F7D1173EULL,
 0xB87489210948D6ULL, 0xBBAB8D313A86E6ULL, 0xBEEF17706A4AEAULL,
 0xC23F51993CEF7AULL, 0xC59C65D9F3BD52ULL, 0xC9067ED56C9DC2ULL,
 0xCC7DC7A4237457ULL, 0xD0026BD535218FULL, 0xD394976F64306BULL,
 0xD73476F21F30B3ULL, 0xDAE2375688BFC1ULL, 0xDE9E06108141A9ULL,
 0xE268110FB24C95ULL, 0xE64086C09BC82DULL, 0xEA27960DA2C2E4ULL,
 0xEE1D6E6021FF04ULL, 0xF2223FA17C3954ULL, 0xF6363A3C302B34ULL,
 0xFA598F1CEE4A0CULL, 0xFE8C6FB3B045ECULL, 0x102CF0DF4D2493CULL,
 0x107219C5A2DFB5CULL, 0x10B844DE4374815ULL, 0x10FF7561B1AEDBDULL,
 0x1147AE90FDED3F1ULL, 0x1190F3B5D842CC9ULL, 0x11DB4822A2B626BULL,
 0x1226AF32839D2E3ULL, 0x12732C497815C27ULL, 0x12C0C2D4669BA2FULL,
 0x130F764931BB90AULL, 0x135F4A26CAE3CDAULL, 0x13B041F545521A2ULL,
 0x14026145E91F4D2ULL, 0x1455ABB34668A7EULL, 0x14AA24E1489702EULL,
 0x14FFD07D49C3F36ULL, 0x1556B23E263D086ULL, 0x15AECDE450253E3ULL,
 0x16082739E334C7BULL, 0x1662C212B8974C7ULL, 0x16BEA24C7AE8BAFULL,
 0x171BCBCEBA50CE5ULL, 0x177A428B00BD66DULL, 0x17DA0A7CE63BD49ULL,
 0x183B27AA2571341ULL, 0x189D9E22B031FC0ULL, 0x19017200C438DC0ULL,
 0x1966A768FFFD0C0ULL, 0x19CD428A77A82C0ULL, 0x1A35479ECA2BD40ULL,
 0x1A9EBAEA3676F40ULL, 0x1B09A0BBB0CB240ULL, 0x1B75FD6CF832040ULL,
 0x1BE3D562AC12CC1ULL, 0x1C532D0C61E82C9ULL, 0x1CC408E4BB169EDULL,
 0x1D366D717AE3465ULL, 0x1DAA5F439C8B82FULL, 0x1E1FE2F7697D547ULL,
 0x1E96FD348FB0AFBULL, 0x1F0FB2AE3821E63ULL, 0x1F8A08231D6D406ULL,
 0x2006025DA28BEB6ULL, 0x2083A633E9B25AEULL, 0x2102F887EB503FEULL,
 0x2183FE478D32352ULL, 0x2206BC6CB9C5422ULL, 0x228B37FD777C55AULL,
 0x2311760C0057D8AULL, 0x23997BB6D98F7AFULL, 0x24234E28EB5E5A7ULL,
 0x24AEF29998F1A63ULL, 0x253C6E4CD879DEBULL, 0x25CBC6934B5ED49ULL,
 0x265D00CA5696871ULL, 0x26F0225C3B1F03DULL, 0x278530C02E9B695ULL,
 0x281C317A74142DCULL, 0x28B52A1C74DACBCULL, 0x29502044D990F6CULL,
 0x29ED199FA35378CULL, 0x2A8C1BE64508DB4ULL, 0x2B2D2CDFBCD3FD4ULL,
 0x2BD05260ADAAB84ULL, 0x2C75924B7910B64ULL, 0x2D1CF29058F69ADULL,
 0x2DC6792D79BDA15ULL, 0x2E722C2F145FD29ULL, 0x2F2011AF88BCF41ULL,
 0x2FD02FD7780C533ULL, 0x30828CDDDF738EBULL, 0x31372F0832C280FULL,
 0x31EE1CAA77546D7ULL, 0x32A75C275F16942ULL, 0x3362F3F063B44D2ULL,
 0x3420EA85E1E8CFAULL, 0x34E1467734F6C6AULL, 0x35A40E62D245D66ULL,
 0x366948F66526356ULL, 0x3730FCEEEABA7BEULL, 0x37FB3118CE07CCEULL,
 0x38C7EC50042C7BBULL, 0x3997358028BD513ULL, 0x3A6913A49A4993FULL,
 0x3B3D8DC89705F67ULL, 0x3C14AB07599E8EDULL, 0x3CEE728C362FFB5ULL,
 0x3DCAEB92B767D71ULL, 0x3EAA1D66BBCCA29ULL, 0x3F8C0F64932D438ULL,
 0x4070C8F91C383F8ULL, 0x415851A1E23AD58ULL, 0x4242B0ED3B08198ULL,
 0x432FEE7A6508368ULL, 0x442011F9A56FFA8ULL, 0x4513232C66A0D08ULL,
 0x460929E556B14C8ULL, 0x47022E08861E6D9ULL, 0x47FE378B86A5BA1ULL,
 0x48FD4E758A485A5ULL, 0x49FF7ADF827755DULL, 0x4B04C4F43F69177ULL,
 0x4C0D34F08F985CFULL, 0x4D18D3235F6CB63ULL, 0x4E27A7EDD90CB8BULL,
 0x4F39BBC3845A0BEULL, 0x504F172A671772EULL, 0x5167C2BB2538F86ULL,
 0x5283C721215E616ULL, 0x53A32D1A9D780BAULL, 0x54C5FD78DB965CAULL,
 0x55EC41203EE3E62ULL, 0x571601086CCA652ULL, 0x5843463C6E42C07ULL,
 0x597419DAD1502BFULL, 0x5AA88515CAA695BULL, 0x5BE09133577C823ULL,
 0x5D1C478D5F887D1ULL, 0x5E5BB191D72A439ULL, 0x5F9ED8C2E1BFCE5ULL,
 0x60E5C6B6F4265FDULL, 0x62308518F767BD4ULL, 0x637F1DA86B93B74ULL,
 0x64D19A398AC6284ULL, 0x662804B56C598E4ULL, 0x6782671A284665CULL,
 0x68E0CB7AFAAF6BCULL, 0x6A433C00679AECCULL, 0x6BA9C2E85ED946CULL,
 0x6D146A866018C45ULL, 0x6E833D439F26F6DULL, 0x6FF6459F285FB61ULL,
 0x716D8E2E0549EB9ULL, 0x72E9219B61624FBULL, 0x74690AA8AF143F3ULL,
 0x75ED542DCCE0CF7ULL, 0x777609192AB447FULL, 0x7903346FEF6A27AULL,
 0x7A94E14E1E7FDCAULL, 0x7C2B1AE6BDF6552ULL, 0x7DC5EC83FC62902ULL,
 0x7F656187572D54EULL, 0x81098569C10237EULL, 0x82B263BBC86E146ULL,
 0x84600825BEAD216ULL, 0x86127E67DEA8C93ULL, 0x87C9D25A74256ABULL,
 0x89860FEE03202B7ULL, 0x8B47432B6F5D01FULL, 0x8D0D783424251F5ULL,
 0x8ED8BB423C35DFDULL, 0x90A918A8A9E0699ULL, 0x927E9CD35F5A211ULL,
 0x94595447773E1B0ULL, 0x96394BA35D3FB30ULL, 0x981E8F9EF70E6F0ULL,
 0x9A092D0BCD6B570ULL, 0x9BF930D5356FE90ULL, 0x9DEEA8007A06D10ULL,
 0x9FE99FAD05968D0ULL, 0xA1EA25148BDE250ULL, 0xA3F0458B34041F1ULL,
 0xA5FC0E7FC2D7D79ULL, 0xA80D8D7BC54565DULL, 0xAA24D023BAFC355ULL,
 0xAC41E43741487BFULL, 0xAE64D7913E1FB57ULL, 0xB08DB8280B604CBULL,
 0xB2BC940DA2449B3ULL, 0xB4F1796FC709676ULL, 0xB72C769834C80A6ULL,
 0xB96D99ECC98465EULL, 0xBBB4F1EFB26ED2EULL, 0xBE028D3F985A322ULL,
 0xC0567A97CC66472ULL, 0xC2B0C8D074DE86AULL, 0xC51186DEBA4D81AULL,
 0xC778C3D4F4C515FULL, 0xC9E68EE2D95B8D7ULL, 0xCC5AF755A7DDD53ULL,
 0xCED60C9858B6F5BULL, 0xD157DE33CB0CF59ULL, 0xD3E07BCEF313501ULL,
 0xD66FF52F089328DULL, 0xD9065A37B5A9665ULL, 0xDBA3BAEB45BADCCULL,
 0xDE48276AD49EB2CULL, 0xE0F3AFF67DFF29CULL, 0xE3A664ED8CF0F3CULL,
 0xE66056CEABC1404ULL, 0xE921963813FAAA4ULL, 0xEBEA33E7BEA1314ULL,
 0xEEBA40BB94A5674ULL, 0xF191CDB19F8EFDDULL, 0xF470EBE83A5EDC5ULL,
 0xF757AC9E42A8E99ULL, 0xFA46213349E5B31ULL, 0xFD3C5B27C6FC1C3ULL,
 0x1003A6C1D48033FBULL, 0x1034065D6A43CADFULL, 0x1064E5A382E47327ULL,
 0x109645B47E68A4B2ULL, 0x10C827B2DADDA7C2ULL, 0x10FA8CC337856AAAULL,
 0x112D760C5807EA9AULL, 0x1160E4B727A83236ULL, 0x1194D9EEBC7CF0A6ULL,
 0x11C956E05AACABCEULL, 0x11FE5CBB77AD905EULL, 0x1233ECB1BD88E26BULL,
 0x126A07F70E221143ULL, 0x12A0AFC18681712FULL, 0x12D7E54982229DD7ULL,
 0x130FA9C99E4687FDULL, 0x1347FE7EBD493145ULL, 0x1380E4A809FB18C1ULL,
 0x13BA5D86FAFE5AF9ULL, 0x13F46A5F56278828ULL, 0x142F0C7733E23368ULL,
 0x146A451702993D88ULL, 0x14A615898A22DE48ULL, 0x14E27F1BEF306EB8ULL,
 0x151F831DB6C1F778ULL, 0x155D22E0C99D8598ULL, 0x159B5FB977CA48D8ULL,
 0x15DA3AFE7C0F7E09ULL, 0x1619B608FF772851ULL, 0x1659D2349CD49C15ULL,
 0x169A90DF644EDE4DULL, 0x16DBF369DEEEDB07ULL, 0x171DFB37123175DFULL,
 0x1760A9AC839D7733ULL, 0x17A400323C5D58DBULL, 0x17E80032CCDCF52EULL,
 0x182CAB1B506B1B1EULL, 0x1872025B70DF0A36ULL, 0x18B807656A41D946ULL,
 0x18FEBBAE0E7BCA8AULL, 0x194620ACC905901AULL, 0x198E37DBA29D8372ULL,
 0x19D702B74500D2E2ULL, 0x1A2082BEFEA8A7B7ULL, 0x1A6AB974C68B47EFULL,
 0x1AB5A85D3FE1364BULL, 0x1B0150FFBDEE5393ULL, 0x1B4DB4E647CF03E1ULL,
 0x1B9AD59D9C495AC9ULL, 0x1BE8B4B535A25135ULL, 0x1C3753BF4D7707CDULL,
 0x1C86B450E09A18C4ULL, 0x1CD6D801B2F4FBE4ULL, 0x1D27C06C536D7FB4ULL,
 0x1D796F2E1FCF5A94ULL, 0x1DCBE5E748B9D6ACULL, 0x1E1F263AD5919B8CULL,
 0x1E7331CEA876985CULL, 0x1EC80A4B823E117CULL, 0x1F1DB15D0670D475ULL,
 0x1F7428B1BF4D951DULL, 0x1FCB71FB21CF76D1ULL, 0x20238EED91B8C4A9ULL,
 0x207C814065A1DB8BULL, 0x20D64AADEB0C4903ULL, 0x2130ECF36A7A21C7ULL,
 0x218C69D12B8992CFULL, 0x21E8C30A7914AFEAULL, 0x2245FA65A55582BAULL,
 0x22A411AC0E0E5D02ULL, 0x23030AAA20B67132ULL, 0x2362E72F5EAAB41EULL,
 0x23C3A90E61630ACEULL, 0x2425521CDEABC756ULL, 0x2487E433ACE377A6ULL,
 0x24EB612EC73D0943ULL, 0x254FCAED520644DBULL, 0x25B523519EF2A4A7ULL,
 0x261B6C41316A888FULL, 0x2682A7A4C2DECB05ULL, 0x26EAD7684720B98DULL,
 0x2753FD7AF0BE73E9ULL, 0x27BE1BCF3563B4E1ULL, 0x2829345AD23F07A0ULL,
 0x28954916D06B6CA0ULL, 0x29025BFF895E7120ULL, 0x29706F14AB5ABC20ULL,
 0x29DF84593DE713E0ULL, 0x2A4F9DD3A649DEE0ULL, 0x2AC0BD8DAC092360ULL,
 0x2B32E5947D6F0860ULL, 0x2BA617F8B412DB21ULL, 0x2C1A56CE59669C29ULL,
 0x2C8FA42CEB4916CDULL, 0x2D06022F609C8645ULL, 0x2D7D72F42DE1CB4FULL,
 0x2DF5F89D49D83567ULL, 0x2E6F95503221E29BULL, 0x2EEA4B35EFECB903ULL,
 0x2F661C7B1C9FFCE6ULL, 0x2FE30B4FE68E8696ULL, 0x306119E815AD9B0EULL,
 0x30E04A7B10506A5EULL, 0x31609F43DFE836F2ULL, 0x31E21A8135C928C2ULL,
 0x3264BE756FF3CF7AULL, 0x32E88D669DE356AAULL, 0x336D899E85606F0FULL,
 0x33F3B56AA758F007ULL, 0x347B131C44BC3443ULL, 0x3503A508635C34CBULL,
 0x358D6D87D2D36569ULL, 0x36186EF7316F5591ULL, 0x36A4ABB6F12018DDULL,
 0x3732262B5C6C7A35ULL, 0x37C0E0BC9B6AFCBCULL, 0x3850DDD6B8BFAD9CULL,
 0x38E21FE9A69EC9CCULL, 0x3974A96943D43AECULL, 0x3A087CCD60CFEE54ULL,
 0x3A9D9C91C4B70974ULL, 0x3B340B363279FEA4ULL, 0x3BCBCB3E6DEF8584ULL,
 0x3C64DF3240F47A0DULL, 0x3CFF499D8090A575ULL, 0x3D9B0D1012207409ULL,
 0x3E382C1DF0839B21ULL, 0x3ED6A95F3150B253ULL, 0x3F7687700A0DC30BULL,
 0x4017C8F0D56DD1AFULL, 0x40BA708618936377ULL, 0x415E80D888580422ULL,
 0x4203FC950E98CEB2ULL, 0x42AAE66CCF87FC5AULL, 0x435341152F037CCAULL,
 0x43FD0F47D5F09B06ULL, 0x44A853C2B79CB1F6ULL, 0x455511481722F3DEULL,
 0x46034A9E8CD747EEULL, 0x46B302910BB6411BULL, 0x47643BEEE6DA3173ULL,
 0x4816F98BD6F55D1FULL, 0x48CB3E3FFFD15047ULL, 0x49810CE7F5D35B0DULL,
 0x4A386864C38636D5ULL, 0x4AF1539BEF28D811ULL, 0x4BABD17780426FC9ULL,
 0x4C67E4E6053BA018ULL, 0x4D2590DA98FCE6D8ULL, 0x4DE4D84CE89241B8ULL,
 0x4EA5BE3938D40EF8ULL, 0x4F6845A06C152E08ULL, 0x502C718807D66348ULL,
 0x50F244FA3A7F0228ULL, 0x51B9C305E11AE0E8ULL, 0x5282EEBE8D1D9939ULL,
 0x534DCB3C8A2B1901ULL, 0x541A5B9CE3E58685ULL, 0x54E8A3016BC07B3DULL,
 0x55B8A490BED99897ULL, 0x568A63764BD679EFULL, 0x575DE2E258C80703ULL,
 0x5833260A09132A2BULL, 0x590A3027635EED9EULL, 0x59E304795788030EULL,
 0x5ABDA643C499B8E6ULL, 0x5B9A18CF7ECC6076ULL, 0x5C785F6A5589285AULL,
 0x5D587D6719736E6AULL, 0x5E3A761DA2778C82ULL, 0x5F1E4CEAD5DF2372ULL,
 0x60040530AC6AE767ULL, 0x60EBA2563871F11FULL, 0x61D527C7AC06973BULL,
 0x62C098F65F20D303ULL, 0x63ADF958D5CE33F1ULL, 0x649D4C6AC6676559ULL,
 0x658E95AD1FCB4985ULL, 0x6681D8A60F9FAD9DULL, 0x677718E1089799B4ULL,
 0x686E59EEC8BF4054ULL, 0x69679F655FCD90E4ULL, 0x6A62ECE0357B7044ULL,
 0x6B6046000FE09AFCULL, 0x6C5FAE6B19D6345CULL, 0x6D6129CCE95F05ECULL,
 0x6E64BBD68615728CULL, 0x6F6A683E6F9F20A5ULL, 0x707232C0A4265ECDULL,
 0x717C1F1EA6D94641ULL, 0x7288311F866E9E99ULL, 0x73966C8FE3B0861BULL,
 0x74A6D541F80CE213ULL, 0x75B96F0D9C2B9A97ULL, 0x76CE3DD04E8AA51FULL,
 0x77E5456D3A1FE15AULL, 0x78FE89CD3D00CBAAULL, 0x7A1A0EDEEF1008B2ULL,
 0x7B37D896A8B0CD62ULL, 0x7C57EAEE898026EEULL, 0x7D7A49E67F14261EULL,
 0x7E9EF9844BC0F166ULL, 0x7FC5FDD38D63C136ULL,
};
const uint64_t N_choose_9[] = { (534), 1, 10, 55, 220, 715, 2002, 5005,
 11440, 24310, 48620, 92378, 0x29018, 0x47C2A, 0x7970C, 0xC7826, 0x13F370,
 0x1F2C5F, 0x2FAD46, 0x4783E9, 0x696414, 0x98D11D, 0xDA4F4E, 0x1339E4B,
 0x1ABFDA0, 0x24C7CBC, 0x32057B8, 0x4356264, 0x59C8330, 0x76A3FA4, 0x9B75C38,
 0xCA1917C, 0x104C58A0, 0x14E1D18D, 0x1A93C4E2, 0x219CC44B, 0x2A416BCC,
 0x34D1C6BF, 0x41AADB6A, 0x51386039, 0x63F69DD0, 0x7A748152, 0x9555E264,
 0xB556009E, 0xDB4A3C48, 0x108250E6EULL, 0x13CF94484ULL, 0x17AFD8402ULL,
 0x1C3901A90ULL, 0x2183B1F8BULL, 0x27AB8EBDEULL, 0x2ECF8ED5DULL,
 0x37124DB04ULL, 0x409A64FB1ULL, 0x4B92CD126ULL, 0x582B44957ULL,
 0x6698BF840ULL, 0x7715DE4B8ULL, 0x89E36D2F0ULL, 0x9F48EC788ULL,
 0xB79521E60ULL, 0xD31EB3C88ULL, 0xF244CE4F0ULL, 0x1156FD37B8ULL,
 0x13D1216440ULL, 0x169A8A1659ULL, 0x19BBC0A73AULL, 0x1D3E151B1FULL,
 0x212BACF4BCULL, 0x258F92E7F3ULL, 0x2A75C77582ULL, 0x2FEB5279A5ULL,
 0x35FE55B7F0ULL, 0x3CBE206EEEULL, 0x443B43FE5CULL, 0x4C87A9AB22ULL,
 0x55B6A98C78ULL, 0x5FDD22ADF2ULL, 0x6B1194727CULL, 0x776C39449EULL,
 0x850722A0B0ULL, 0x93FE5685F7ULL, 0xA46FEE5BF6ULL, 0xB67C375991ULL,
 0xCA45D47BF4ULL, 0xDFF1E21B85ULL, 0xF7A81B2D7EULL, 0x11193004123ULL,
 0x12DE00047E0ULL, 0x14CBFA337F4ULL, 0x16E65B699A8ULL, 0x193097C0F6CULL,
 0x1BAE5D9E990ULL, 0x1E6398DD6ACULL, 0x2154761C228ULL, 0x2485662F2B4ULL,
 0x27FB21B7AE0ULL, 0x2BBAACE0E65ULL, 0x2FC95B44F12ULL, 0x342CD3FA4B3ULL,
 0x38EB15CB3ACULL, 0x3E0A7B98667ULL, 0x4391C0E7E1AULL, 0x498806A1FD1ULL,
 0x4FF4D7FD310ULL, 0x56E02F9A7CAULL, 0x5E527CD39D4ULL, 0x6654A93C866ULL,
 0x6EF01E598A8ULL, 0x782ECB8BAB6ULL, 0x821B2C348F4ULL, 0x8CC04E139FAULL,
 0x9829D7DDDD0ULL, 0xA4641011FA3ULL, 0xB17BE40A58EULL, 0xBF7EEF4E885ULL,
 0xCE7B8325EE4ULL, 0xDE80AE6D499ULL, 0xEF9E45B0C56ULL, 0x101E4EB8C5AFULL,
 0x115661954480ULL, 0x12A342807670ULL, 0x14062598D3E0ULL, 0x15804E241A10ULL,
 0x17130F2CFCC0ULL, 0x18BFCC25EE10ULL, 0x1A87F9911BE0ULL, 0x1C6D1DADC270ULL,
 0x1E70D12AF480ULL, 0x2094BFDFF9B1ULL, 0x22DAA98A646AULL, 0x254462920107ULL,
 0x27D3D4D2C09CULL, 0x2A8B006CC21BULL, 0x2D6BFC9A9D32ULL, 0x3078F88E12BDULL,
 0x33B43C534730ULL, 0x372029BAABE6ULL, 0x3ABF3D49BCCCULL, 0x3E940F32B86AULL,
 0x42A1545378D8ULL, 0x46E9DF3B94BAULL, 0x4B70A139EFECULL, 0x5038AB71E416ULL,
 0x55452FF829F0ULL, 0x5A9982F7AC8FULL, 0x60391BDE70A6ULL, 0x66279692BA39ULL,
 0x6C68B4B09BD4ULL, 0x73005ED01AEDULL, 0x79F2A5D415AEULL, 0x8143C44216FBULL,
 0x88F81FA34620ULL, 0x911449EEA02CULL, 0x999D02FCA998ULL, 0xA2973A04C774ULL,
 0xAC080F246FF0ULL, 0xB5F4D4F062B4ULL, 0xC06312101A18ULL, 0xCB5882E3A6ECULL,
 0xD6DB1B342920ULL, 0xE2F107EF183DULL, 0xEFA0B0EC8F42ULL, 0xFCF0BAC0D01BULL,
 0x10AE80899338CULL, 0x1198DBE24BB0FULL, 0x128E941887ACAULL,
 0x139023D601269ULL, 0x149E0A2CA6C50ULL, 0x15B8CAB82FB42ULL,
 0x16E0EDC07AF44ULL, 0x1817005CBDB2EULL, 0x195B949784508ULL,
 0x1AAF419389BFEULL, 0x1C12A3B168F64ULL, 0x1D865CB62C2F2ULL,
 0x1F0B13F2BDD10ULL, 0x20A1766C3ECBBULL, 0x224A37054653EULL,
 0x24060EA80EEADULL, 0x25D5BC7194AC4ULL, 0x27BA05DDA8E81ULL,
 0x29B3B6F3FF186ULL, 0x2BC3A27637507ULL, 0x2DEAA20EEA4C0ULL,
 0x30299681BB528ULL, 0x328167DC742D0ULL, 0x34F305A92F798ULL,
 0x377F672195B20ULL, 0x3A278B6331498ULL, 0x3CEC79A4DE4D0ULL,
 0x3FCF416D5A028ULL, 0x42D0FACAF70C0ULL, 0x45F2C68C7AA09ULL,
 0x4935CE7B2779AULL, 0x4C9B4595FB1EFULL, 0x5024684E2247CULL,
 0x53D27CC4A9143ULL, 0x57A6D3096BEE2ULL, 0x5BA2C55B4DED5ULL,
 0x5FC7B869B9A70ULL, 0x64171B97715DEULL, 0x6892693EB393CULL,
 0x6D3B26F6B90B2ULL, 0x7212E5DA91538ULL, 0x771B42D163082ULL,
 0x7C55E6D814F5CULL, 0x81C4874C6568EULL, 0x8768E63974F30ULL,
 0x8D44D2A5CA027ULL, 0x935A28E2D2B56ULL, 0x99AAD2DDEA5E1ULL,
 0xA038C872E83B4ULL, 0xA7060FC03CF55ULL, 0xAE14BD7CA47DEULL,
 0xB566F54E71FD3ULL, 0xBCFEEA247B860ULL, 0xC4DEDE90AB564ULL,
 0xCD0925243C788ULL, 0xD58020CDA8A7CULL, 0xDE4645384D650ULL,
 0xE75E172DCE3BCULL, 0xF0CA2CF93A408ULL, 0xFA8D2ECBFAE24ULL,
 0x104A9D72492360ULL, 0x10F22F3372EF15ULL, 0x119FB63581C572ULL,
 0x125361B6814683ULL, 0x130D623427AB6CULL, 0x13CDE972D864B7ULL,
 0x14952A84C6117AULL, 0x156359D1343801ULL, 0x1638AD1BD92990ULL,
 0x17155B8C607BBAULL, 0x17F99DB60E82B4ULL, 0x18E5AD9F8538F6ULL,
 0x19D9C6CAAB0168ULL, 0x1AD6263CB3B246ULL, 0x1BDB0A864C57D4ULL,
 0x1CE8B3CBEA1EEAULL, 0x1DFF63CE3CD850ULL, 0x1F1F5DF2C586D3ULL,
 0x2048E74C916AEEULL, 0x217C46A519FFD5ULL, 0x22B9C4854A5EA4ULL,
 0x2401AB3EAA7D69ULL, 0x255446F4B0C0B6ULL, 0x26B1E5A63A575FULL,
 0x281AD7372AD900ULL, 0x298F6D7A33A0E0ULL, 0x2B0FFC3AC35FC0ULL,
 0x2C9CD9471E6020ULL, 0x2E365C7A9FF980ULL, 0x2FDCDFC825B020ULL,
 0x3190BF44A47FC0ULL, 0x33525931E8D0E0ULL, 0x35220E09819900ULL,
 0x37004087D72761ULL, 0x38ED55B76E20CAULL, 0x3AE9B4FC572DD7ULL,
 0x3CF5C81FCBE05CULL, 0x3F11FB5BF9556BULL, 0x413EBD67F91A92ULL,
 0x437C7F83F8DDEDULL, 0x45CBB5859171B0ULL, 0x482CD5E44DACD6ULL,
 0x4AA059C661B3ACULL, 0x4D26BD0D9333FAULL, 0x4FC07E64532198ULL,
 0x526E1F4B09814AULL, 0x5530242593D0CCULL, 0x58071448F69C06ULL,
 0x5AF37A0942D070ULL, 0x5DF5E2C7AF60BFULL, 0x610EDF00E7CC06ULL,
 0x643F025B901C89ULL, 0x6786E3B6FEF394ULL, 0x6AE71D3A2E38BDULL,
 0x6E604C62E3040EULL, 0x71F312150D5BABULL, 0x75A012AA605EA0ULL,
 0x7967F60223779CULL, 0x7D4B67913D3378ULL, 0x814B1672785884ULL,
 0x8567B57703DCB0ULL, 0x89A1FB372E59C4ULL, 0x8DFAA2235D9FF8ULL,
 0x9272689543085CULL, 0x970A10E14D29A0ULL, 0x9BC261685792EDULL,
 0xA09C24A99931A2ULL, 0xA5982954D207EBULL, 0xAAB7425CB8DB4CULL,
 0xAFFA4709A9835FULL, 0xB562130C94822AULL, 0xBAEF8692309099ULL,
 0xC0A386566ECAD0ULL, 0xC67EFBB8322932ULL, 0xCC82D4CD4AF424ULL,
 0xD2B00476B6E1BEULL, 0xD9078275268DC8ULL, 0xDF8A4B7DC8FD8EULL,
 0xE639614F5DE244ULL, 0xED15CAC78F4DE2ULL, 0xF42093F8938F90ULL,
 0xFB5ACE3F17EDEBULL, 0x102C5905874F69EULL, 0x10A61F6792D1BFDULL,
 0x112312263B65A84ULL, 0x11A343B7F8FA151ULL, 0x1226C6EF0A2B9E6ULL,
 0x12ADAEFAEF36CB7ULL, 0x13380F69E9CA140ULL, 0x13C5FC2A81C3998ULL,
 0x1457898D0EE6AB0ULL, 0x14ECCC4547953A8ULL, 0x1585D96BD4997E0ULL,
 0x1622C67FEA0C0A8ULL, 0x16C3A968E562CB0ULL, 0x17689877F0B5498ULL,
 0x1811AA69AB42D40ULL, 0x18BEF667D7471B9ULL, 0x1970940B0D29FFAULL,
 0x1A269B5C74174BFULL, 0x1AE124D7800B43CULL, 0x1BA0496BB560E93ULL,
 0x1C64227E71EF042ULL, 0x1D2CC9ECBBC1005ULL, 0x1DFA5A0D1578CF0ULL,
 0x1ECCEDB158670CEULL, 0x1FA4A0289466C1CULL, 0x20818D40F58A242ULL,
 0x2163D149AFA5DF8ULL, 0x224B8914EFC8712ULL, 0x2338D1F9D3AB43CULL,
 0x242BC9D6672B47EULL, 0x25248F11A7D6DB0ULL, 0x2623409D8E9EE57ULL,
 0x2727FDF91FB92B6ULL, 0x2832E73280C1E31ULL, 0x29441CE9152AB74ULL,
 0x2A5BC04FA105725ULL, 0x2B79F32E7238A3EULL, 0x2C9ED7E5902CA83ULL,
 0x2DCA916EF1FF8E0ULL, 0x2EFD4360BB4E6D4ULL, 0x303711EF7FA2D68ULL,
 0x317821F08C9318CULL, 0x32C098DC3AA4310ULL, 0x34109CD044FC4CCULL,
 0x3568549227F4DE8ULL, 0x36C7E791869B594ULL, 0x382F7DEA972FBE0ULL,
 0x399F406896B03C5ULL, 0x3B17588843813D2ULL, 0x3C97F07A5F41453ULL,
 0x3E21332637D832CULL, 0x3FB34C2C37D1707ULL, 0x414E67E87E10CDAULL,
 0x42F2B3757CF1C31ULL, 0x44A05CAEA0E1010ULL, 0x46579232FE803AAULL,
 0x481883680864394ULL, 0x49E3607C4C7D586ULL, 0x4BB85A6A393AA28ULL,
 0x4D97A2FAEA77DD6ULL, 0x4F816CC8FE46EB4ULL, 0x5175EB4371A4FDAULL,
 0x537552B0852C2D0ULL, 0x557FD830A9D2203ULL, 0x5795B1C175C484EULL,
 0x59B71640A174325ULL, 0x5BE43D6F0CDFE64ULL, 0x5E1D5FF3CD2FA39ULL,
 0x6062B75F42B1D16ULL, 0x62B47E2E374B50FULL, 0x6512EFCD056BD80ULL,
 0x677E489AC797F50ULL, 0x69F6C5EC909A3A0ULL, 0x6C7CA610AC6D230ULL,
 0x6F102851E9EF640ULL, 0x71B18CFAED74630ULL, 0x746115598C42BA0ULL,
 0x771F03C23112B50ULL, 0x79EB9B93499ED80ULL, 0x7CC72138BD58911ULL,
 0x7FB1DA2F6D5352AULL, 0x82AC0D08BD786A7ULL, 0x85B6016E271601CULL,
 0x88D00024D4DBCBBULL, 0x8BFA53114857FF2ULL, 0x8F35453B090751DULL,
 0x928122D05D0AC30ULL, 0x95DE392A0B961C6ULL, 0x994CD6CF292A28CULL,
 0x9CCD4B78EDADB8AULL, 0xA05FE8169478A58ULL, 0xA404FED146641DAULL,
 0xA7BCE3100DF39ACULL, 0xAB87E97BD5A9FF6ULL, 0xAF666803709E6F0ULL,
 0xB358B5DFAD648EFULL, 0xB75F2B97735BF66ULL, 0xBB7A2303EA7AAD9ULL,
 0xBFA9F754ADA6B54ULL, 0xC3EF051407B2A8DULL, 0xC849AA2B3B11A6EULL,
 0xCCBA45E6D456C5BULL, 0xD14138FB0794720ULL, 0xD5DEE58818B030CULL,
 0xDA93AF1ECEBF558ULL, 0xDF5FFAC4F281594ULL, 0xE4442EF9D80C970ULL,
 0xE940B3BAF3C24D4ULL, 0xEE55F2887A9EDD8ULL, 0xF384566A0DFB5CCULL,
 0xF8CC4BF372D5A20ULL, 0xFE2E414954B419DULL, 0x103AAA626143AC02ULL,
 0x10941EBDEA186BBBULL, 0x10EF48567626630CULL, 0x114C2E7592481FAFULL,
 0x11AAD87F61B8F18AULL, 0x120B4DF2EEB9DAC9ULL, 0x126D966A7BF9B950ULL,
 0x12D1B99BD6C20B22ULL, 0x1337BF58A9E9B104ULL, 0x139FAF8ED18F144EULL,
 0x14099248AF9B1688ULL, 0x14756FAD810E331EULL, 0x14E35001B4193D24ULL,
 0x15533BA73F0324D2ULL, 0x15C53B1DF7DD3210ULL, 0x16395703ED07231BULL,
 0x16AF9815BE849FFEULL, 0x1728072EF825754DULL, 0x17A2AD4A6C820A44ULL,
 0x181F938290CD8821ULL, 0x189EC311D97F2A46ULL, 0x1920455317D43067ULL,
 0x19A423C1D82BEDC0ULL, 0x1A2A67FAC13F7208ULL, 0x1AB31BBBF4364A90ULL,
 0x1B3E48E56D99DBB8ULL, 0x1BCBF9796728D4A0ULL, 0x1C5C379CBA8C40B8ULL,
 0x1CEF0D9744EFBC90ULL, 0x1D8485D44B7E5508ULL, 0x1E1CAAE2E0C599C0ULL,
 0x1EB787764B006D69ULL, 0x1F5526666B4B205AULL, 0x1FF592B025C2638FULL,
 0x2098D775CA8EA3FCULL, 0x213EFFFF7FDD5FE3ULL, 0x21E817BBACCA09A2ULL,
 0x22942A3F65380D35ULL, 0x23434346D69F8F70ULL, 0x23F56EB5B5CE7FBEULL,
 0x24AAB897AD9F96FCULL, 0x25632D20CEA8DFD2ULL, 0x261ED8ADFFE366B8ULL,
 0x26DDC7C5704DB1A2ULL, 0x27A00717098AA11CULL, 0x2865A37CE37E5E6EULL,
 0x292EA9FBB8EAFC30ULL, 0x29FB27C35D0E7087ULL, 0x2ACB2A2F32439216ULL,
 0x2B9EBEC6A1A7C281ULL, 0x2C75F33D93C6F334ULL, 0x2D50D574EA4FB2F5ULL,
 0x2E2F737AFAD0F49EULL, 0x2F11DB8C0A834133ULL, 0x2FF81C12CB1F0960ULL,
 0x30E243A8D8C1CC44ULL, 0x31D0611738E3CB48ULL, 0x32C28356DA60049CULL,
 0x33B8B99116902FD0ULL, 0x34B31320337E79DCULL, 0x35B19F8FE72EBFC8ULL,
 0x36B46E9DDC010904ULL, 0x37BB903A362F0460ULL, 0x38C714881A664C75ULL,
 0x39D70BDE35813A32ULL, 0x3AEB86C745600E23ULL, 0x3C049602A2E43AECULL,
 0x3D224A84CD0F9D57ULL, 0x3E44B577F549703AULL, 0x3F6BE83C8CCACC61ULL,
 0x4097F469D3348690ULL, 0x41C8EBCE66503F9AULL, 0x42FEE070D2FE7C74ULL,
 0x4439E49027539E16ULL, 0x457A0AA485E592E8ULL, 0x46BF655FBA4C1D66ULL,
 0x480A07ADCED58D94ULL, 0x495A04B5A371CCCAULL, 0x4AAF6FD985D59D50ULL,
 0x4C0A5CB7CAD7F133ULL, 0x4D6ADF2B690B3DAEULL, 0x4ED10B4C9494B275ULL,
 0x503CF5715C433E24ULL, 0x51AEB22E47E84B09ULL, 0x53265656F7F42176ULL,
 0x54A3F6FEC657DEBFULL, 0x5627A97968AEF200ULL, 0x57B1835B93B211C0ULL,
 0x59419A7B9FF59F80ULL, 0x5AD804F22FF57040ULL, 0x5C74D91AD76FF300ULL,
 0x5E182D94C412B040ULL, 0x5FC21943677A1F80ULL, 0x6172B34F2286D1C0ULL,
 0x632A1325F209F200ULL, 0x64E8507C1CCB1EC1ULL, 0x66AD834CE2E9A18AULL,
 0x6879C3DB2E9B0B77ULL, 0x6A4D2AB246493FDCULL, 0x6C27D0A68011F80BULL,
 0x6E09CED5F6A9CD52ULL, 0x6FF33EA93FA4D84DULL, 0x71E439D42326F6B0ULL,
 0x73DCDA5654FDCAB6ULL, 0x75DD3A7C2F26896CULL, 0x77E574DF6DC1AF1AULL,
 0x79F5A467EC76B318ULL, 0x7C0DE44C6549D66AULL, 0x7E2E501330E62A8CULL,
};
const uint64_t N_choose_10[] = { (361), 1, 11, 66, 286, 1001, 3003, 8008,
 19448, 43758, 92378, 0x2D1B4, 0x561CC, 0x9DDF6, 0x117502, 0x1DED28,
 0x31E098, 0x510CF7, 0x80BA3D, 0xC83E26, 0x131A23A, 0x1CA7357, 0x2A4C2A5,
 0x3D860F0, 0x5845E90, 0x7D0DB4C, 0xAF13304, 0xF269568, 0x14C31898,
 0x1C2D583C, 0x25E4B474, 0x328645F0, 0x42D29E90, 0x57B4701D, 0x724834FF,
 0x93E4F94A, 0xBE266516, 0xF2F82BD5, 0x134A3073FULL, 0x185DB6778ULL,
 0x1E9D20548ULL, 0x26446869AULL, 0x2F99C68FEULL, 0x3AEF2699CULL,
 0x48A3CA5E4ULL, 0x59261B452ULL, 0x6CF5AF8D6ULL, 0x84A587CD8ULL,
 0xA0DE89768ULL, 0xC2623B6F3ULL, 0xEA0DCA2D1ULL, 0x118DD5902EULL,
 0x14FEFA6B32ULL, 0x1908A0BAE3ULL, 0x1DC1CD8C09ULL, 0x234481D560ULL,
 0x29AE0DCDA0ULL, 0x311F6BB258ULL, 0x39BDA28548ULL, 0x43B2314CD0ULL,
 0x4F2B836B30ULL, 0x5C5D6EA7B8ULL, 0x6B81BB8CA8ULL, 0x7CD8B8C460ULL,
 0x90A9DA28A0ULL, 0xA744643EF9ULL, 0xC10024E633ULL, 0xDE3E3A0152ULL,
 0xFF69E6F60EULL, 0x124F979DE01ULL, 0x14F6F415383ULL, 0x17F5A93CD28ULL,
 0x1B558E98518ULL, 0x1F21709F406ULL, 0x236524DF262ULL, 0x282D9F79D84ULL,
 0x2D890A129FCULL, 0x3386DC3D7EEULL, 0x3A37F584A6AULL, 0x41AEB918F08ULL,
 0x49FF2B42FB8ULL, 0x533F10AB5AFULL, 0x5D860F911A5ULL, 0x68EDD306B36ULL,
 0x7592304E72AULL, 0x83914E702AFULL, 0x930BD02302DULL, 0xA4250027150ULL,
 0xB703002B930ULL, 0xCBCEFA5F124ULL, 0xE2B555C8ACCULL, 0xFBE5ED89A38ULL,
 0x117944B283C8ULL, 0x135F7E405A74ULL, 0x1574C5A21C9CULL, 0x17BD1C050F50ULL,
 0x1A3CCE208A30ULL, 0x1CF878EE9895ULL, 0x1FF50EA2E7A7ULL, 0x2337DBE28C5AULL,
 0x26C68D3F4006ULL, 0x2AA734F8C66DULL, 0x2EE051074487ULL, 0x3378D1716458ULL,
 0x38781EF13768ULL, 0x3DE621EADF32ULL, 0x43CB49B81906ULL, 0x4A30944BE16CULL,
 0x511F96317A14ULL, 0x58A282EA34CAULL, 0x60C435AD7DBEULL, 0x69903A8EB7B8ULL,
 0x7312D80C9588ULL, 0x7D59190DB52BULL, 0x8870D74E5AB9ULL, 0x9468C643433EULL,
 0xA1507E75A222ULL, 0xAF38895C76BBULL, 0xBE326DB78311ULL, 0xCE50BC7048C0ULL,
 0xDFA71E058D40ULL, 0xF24A608603B0ULL, 0x10650861ED790ULL,
 0x11BD0D442F1A0ULL, 0x132E3E36FEE60ULL, 0x14BA3AF95DC70ULL,
 0x1662BA926F850ULL, 0x18298C6D4BAC0ULL, 0x1A10997FFAF40ULL,
 0x1C19E57DFA8F1ULL, 0x1E479016A0D5BULL, 0x209BD63FC0E62ULL,
 0x2319138CECEFEULL, 0x25C1C393B9119ULL, 0x2898835D62E4BULL,
 0x2BA012E644108ULL, 0x2EDB56AB78838ULL, 0x324D59472341EULL,
 0x35F94D1BBF0EAULL, 0x39E28E0EEA954ULL, 0x3E0CA3542222CULL,
 0x427B4147DB6E6ULL, 0x47324B5B7A6D2ULL, 0x4C35D61298AE8ULL,
 0x518A29121B4D8ULL, 0x5733C14196167ULL, 0x5D3752FF7D20DULL,
 0x6399CC68A8C46ULL, 0x6A6057B3B281AULL, 0x71905DA0B4307ULL,
 0x792F87FDF58B5ULL, 0x8143C44216FB0ULL, 0x89D3463C4B5D0ULL,
 0x92E48ADB355FCULL, 0x9C7E5B0AFFF94ULL, 0xA6A7CEAB4C708ULL,
 0xB1684F9D936F8ULL, 0xBCC79CEC999ACULL, 0xC8CDCE0D9B3C4ULL,
 0xD583563BD5AB0ULL, 0xE2F107EF183D0ULL, 0xF120186E09C0DULL,
 0x1001A237CD2B4FULL, 0x10FE92F28DFB6AULL, 0x12097AFB272EF6ULL,
 0x132308B94BEA05ULL, 0x144BF1FAD464CFULL, 0x1584F438347738ULL,
 0x16CED4DAFEE388ULL, 0x182A618681DECAULL, 0x19987062898E0EULL,
 0x1B19E06855693CULL, 0x1CAF99B1CDAE44ULL, 0x1E5A8DCB064A42ULL,
 0x201BB8061CD9A6ULL, 0x21F41DD17F9C98ULL, 0x23E4CF10AB79A8ULL,
 0x25EEE6776F6663ULL, 0x281389E7C3CBA1ULL, 0x2A53EAD244BA4EULL,
 0x2CB146995E0512ULL, 0x2F2CE6F7389393ULL, 0x31C82266788519ULL,
 0x34845C8DDBFA20ULL, 0x376306AECA9EE0ULL, 0x3A65A016E65408ULL,
 0x3D8DB694AD96D8ULL, 0x40DCE6EF408E70ULL, 0x4454DD6159E990ULL,
 0x47F756178CFE28ULL, 0x4BC61DB1DAE2F8ULL, 0x4FC311C8B08320ULL,
 0x53F021755FF3E0ULL, 0x584F4DDE279DE9ULL, 0x5CE2AAC5DA1583ULL,
 0x61AC5F1F39C772ULL, 0x66AEA5A41BEBEEULL, 0x6BEBCD70667D31ULL,
 0x71663AA0FD3C13ULL, 0x772066F6B21AE8ULL, 0x7D1CE27D4DB558ULL,
 0x835E5436C4CB36ULL, 0x89E77ACAB00472ULL, 0x90BB2D3A1B9524ULL,
 0x97DC5B97C4AA5CULL, 0x9F4E0FC4DADADEULL, 0xA7136E325C2A3AULL,
 0xAF2FB6A72280C8ULL, 0xB7A6450AB9CFF8ULL, 0xC07A923516701FULL,
 0xC9B034C3439B75ULL, 0xD34AE1F1224156ULL, 0xDD4E6E7850C50AULL,
 0xE7BECF7454945FULL, 0xF2A01B4C1EDC3DULL, 0xFDF68AA105FC10ULL,
 0x109C679434DB470ULL, 0x11614672C5869D4ULL, 0x122E4F97E9C315CULL,
 0x1303CFB8B76BBD8ULL, 0x13E215FDEFB9228ULL, 0x14C974151D875E4ULL,
 0x15BA3E4216C19ECULL, 0x16B4CB70E2BC810ULL, 0x17B97548074EB70ULL,
 0x18C8983B3E7DA85ULL, 0x19E2939E9699FF7ULL, 0x1B07C9B9FEAE67AULL,
 0x1C389FDD41291E6ULL, 0x1D757E746EAF69DULL, 0x1EBED11CBB10817ULL,
 0x201506B9CE54018ULL, 0x2178918B8BE69A8ULL, 0x22E9E74451EE562ULL,
 0x2469811FB2D6816ULL, 0x25F7DBF9AB2A10CULL, 0x2795786655DA274ULL,
 0x2942DACA21154BAULL, 0x2B008B7285DAC8EULL, 0x2CCF16AF447CB78ULL,
 0x2EAF0CEC284A3C8ULL, 0x30A102CB54A2A9BULL, 0x32A591401DB9589ULL,
 0x34BD55AA6F5955EULL, 0x36E8F1F2C3FF402ULL, 0x39290CA6AEA716BULL,
 0x3B7E5115F9B3221ULL, 0x3DE96F705D58980ULL, 0x406B1CE3D006280ULL,
 0x430413BB7340360ULL, 0x45B5137F1F76320ULL, 0x487EE113915C340ULL,
 0x4B6246DB3B5BCC0ULL, 0x4E6014D7BDB6CE0ULL, 0x517920CC07FECA0ULL,
 0x54AE465F268BD80ULL, 0x5800673FBEA5680ULL, 0x5B706B483C17DE1ULL,
 0x5EFF40A3B2F9EABULL, 0x62ADDBF3786CC82ULL, 0x667D3875752ACDEULL,
 0x6A6E582B34C0249ULL, 0x6E824401B451CDBULL, 0x72BA0BF9F3DFAC8ULL,
 0x7716C7524CF6C78ULL, 0x7B9994B091D194EULL, 0x80439A4CF7ECCFAULL,
 0x8516061DD1200F4ULL, 0x8A120E04165228CULL, 0x8F38EFF8C6EA3D6ULL,
 0x948BF23B20274A2ULL, 0x9A0C637FAF910A8ULL, 0x9FBB9B2043BE118ULL,
 0xA59AF94CBEB41D7ULL, 0xABABE73CCD30DDDULL, 0xB1EFD7628632A66ULL,
 0xB868459DF621DFAULL, 0xBF16B77199056B7ULL, 0xC5FCBC37C735AC5ULL,
 0xCD1BED59180B670ULL, 0xD475EE83BE11510ULL, 0xDC0C6DE3E048CACULL,
 0xE3E1245CF41C024ULL, 0xEBF5D5C41BA18A8ULL, 0xF44C511B8BDF558ULL,
 0xFCE670CEFEC4F1CULL, 0x105C61AF1349EF14ULL, 0x10EED417A88CF770ULL,
 0x1185DE2889DA2110ULL, 0x1221A089F231B3FDULL, 0x12C23CAE9BCAE59FULL,
 0x1367D4D7F09CED8AULL, 0x14128C1A4D55C8D6ULL, 0x14C2866156FF4C35ULL,
 0x1577E8746393CE5FULL, 0x1632D7FAF5C45EF8ULL, 0x16F37B814C3329C8ULL,
 0x17B9FA7D046552FAULL, 0x18867D51D1B0471EULL, 0x19592D56486728DCULL,
 0x1A3234D8BD8DB6A4ULL, 0x1B11BF243B56B432ULL, 0x1BF7F8858AB49676ULL,
 0x1CE50E505243E458ULL, 0x1DD92EE44AD773E8ULL, 0x1ED489B289EF61D3ULL,
 0x1FD74F42E2645871ULL, 0x20E1B1395B91746EULL, 0x21F3E25BBF47CEF2ULL,
 0x230E16973ED77043ULL, 0x243083062F7A2A29ULL, 0x255B5DF5DE6D96E0ULL,
 0x268EDEEC7D0A3820ULL, 0x27CB3EAF252671B8ULL, 0x2910B747F614DC68ULL,
 0x2A5F840C4A8E3010ULL, 0x2BB7E1A307D7C7F0ULL, 0x2D1A0E0B06788898ULL,
 0x2E8648A194CEB548ULL, 0x2FFCD22913DA09E0ULL, 0x317DECCFAE8E3720ULL,
 0x3309DC362C02A8D9ULL, 0x34A0E576DCD548D3ULL, 0x36434F2CA416BD92ULL,
 0x37F1617A1C1771CEULL, 0x39AB6610D76D8061ULL, 0x3B71A838BE8C70A3ULL,
 0x3D4474D78A4880A8ULL, 0x3F241A785BA00D98ULL, 0x4110E95371267E66ULL,
 0x430B3355FA6CEA82ULL, 0x45134C2A09C58CC4ULL, 0x4729893EA4BFEABCULL,
 0x494E41CFF3BC71CEULL, 0x4B81CEEF90F7260AULL, 0x4DC48B8CF769DA88ULL,
 0x5016D47E11E74838ULL, 0x52790887EAD1368FULL, 0x54EB88677CCCC945ULL,
 0x576EB6DAA4D8E776ULL, 0x5A02F8A9362B92EAULL, 0x5CA8B4AE303BEA0FULL,
 0x5F6053E1175F744DULL, 0x622A415F70623ED0ULL, 0x6506EA765F8237B0ULL,
 0x67F6BEAC6B371E84ULL, 0x6AFA2FCB63314BECULL, 0x6E11B1EA6BFA7D78ULL,
 0x713DBB782FA4C088ULL, 0x747EC54533F48554ULL, 0x77D54A8E5673D33CULL,
 0x7B41C9076EDD88D0ULL, 0x7EC4C0E6185084B0ULL,
};
const uint64_t N_choose_11[] = { (265), 1, 12, 78, 364, 1365, 4368, 12376,
 31824, 75582, 0x29018, 0x561CC, 0xAC398, 0x14A18E, 0x261690, 0x4403B8,
 0x75E450, 0xC6F147, 0x147AB84, 0x20FE9AA, 0x3418BE4, 0x50BFF3B, 0x7B0C1E0,
 0xB8922D0, 0x110D8160, 0x18DE5CAC, 0x23CF8FB0, 0x32F62518, 0x47B93DB0,
 0x63E695EC, 0x89CB4A60, 0xBC519050, 0xFF242EE0, 0x156D89EFDULL,
 0x1C920D3FCULL, 0x25D05CD46ULL, 0x31B2C325CULL, 0x40E245E31ULL,
 0x542C76570ULL, 0x6C8A2CCE8ULL, 0x8B274D230ULL, 0xB16BB58CAULL,
 0xE1057C1C8ULL, 0x11BF4A2B64ULL, 0x164986D148ULL, 0x1BDBE8859AULL,
 0x22AB437E70ULL, 0x2AF59BFB48ULL, 0x35038492B0ULL, 0x4129A849A3ULL,
 0x4FCA84EC74ULL, 0x61585A7CA2ULL, 0x765754E7D4ULL, 0x8F5FF5A2B7ULL,
 0xAD21C32EC0ULL, 0xD066450420ULL, 0xFA1452D1C0ULL, 0x12B33BE8418ULL,
 0x164F1610960ULL, 0x1A8A3925630ULL, 0x1F7CF15C160ULL, 0x2542C846918ULL,
 0x2BFAE3FF5C0ULL, 0x33C86F8BA20ULL, 0x3CD30D2E2C0ULL, 0x474753721B9ULL,
 0x535755C07ECULL, 0x613B396093EULL, 0x7131D7CFF4CULL, 0x83816F6DD4DULL,
 0x987863830D0ULL, 0xB06E0CBFDF8ULL, 0xCBC39B58310ULL, 0xEAE50BF7716ULL,
 0x10E4A30D6978ULL, 0x13677D0506FCULL, 0x16400DA630F8ULL, 0x19787B6A08E6ULL,
 0x1D1BFAC25350ULL, 0x2136E653E258ULL, 0x25D6D9081210ULL, 0x2B0ACA12C7BFULL,
 0x30E32B0BD964ULL, 0x3772083C449AULL, 0x3ECB2B412BC4ULL, 0x470440282E73ULL,
 0x5034FD2A5EA0ULL, 0x5A774D2CCFF0ULL, 0x65E77D2F8920ULL, 0x72A46CD57A44ULL,
 0x80CFC2320510ULL, 0x908E210A9F48ULL, 0xA20765BD2310ULL, 0xB566E3FD7D84ULL,
 0xCADBA99F9A20ULL, 0xE298C5A4A970ULL, 0xFCD593C533A0ULL, 0x119CE0CB3CC35ULL,
 0x139C31B56B3DCULL, 0x15CFAF7394036ULL, 0x183C18478803CULL,
 0x1AE68B97146A9ULL, 0x1DD490A788B30ULL, 0x210C1DBE9EF88ULL,
 0x24939FADB26F0ULL, 0x287201CC60622ULL, 0x2CAEB667E1F28ULL,
 0x3151BFACA0094ULL, 0x3663B90FB7AA8ULL, 0x3BEDE13E5AF72ULL,
 0x41FA249932D30ULL, 0x489328421E4E8ULL, 0x4FC455C2E7A70ULL,
 0x5799E753C2F9BULL, 0x6020F4C8A8A54ULL, 0x6967812CDCD92ULL,
 0x737C891436FB4ULL, 0x7E7011A9FE66FULL, 0x8A53388576980ULL,
 0x9738444C7B240ULL, 0xA532B62CD3F80ULL, 0xB4575C3534330ULL,
 0xC4BC649721AC0ULL, 0xD67971DB50C60ULL, 0xE9A7B0124FAC0ULL,
 0xFE61EB0BAD730ULL, 0x114C4A59E1CF80ULL, 0x12CEE320B68A40ULL,
 0x146FECB8B63980ULL, 0x16318B1095E271ULL, 0x18160411FFEFCCULL,
 0x1A1FC175FBFE2EULL, 0x1C5152AECACD2CULL, 0x1EAD6EE8065E45ULL,
 0x2136F71DDC8C90ULL, 0x23F0F84C40CD98ULL, 0x26DEADB6F855D0ULL,
 0x2A03834B6A89EEULL, 0x2D63181D267AD8ULL, 0x310140FE15242CULL,
 0x34E20B33574658ULL, 0x3909BF47D4FD3EULL, 0x3D7CE3FD8CA410ULL,
 0x4240415EB62EF8ULL, 0x4758E3EFD7E3D0ULL, 0x4CCC2003F14537ULL,
 0x529F9533E91744ULL, 0x58D931FA73A38AULL, 0x5F7F3775AECBA4ULL,
 0x66983D4FBA0EABULL, 0x6E2B35CF996760ULL, 0x763F7213BAD710ULL,
 0x7EDCA6777F8CE0ULL, 0x880AEF2532E2DCULL, 0x91D2D4D5E2E270ULL,
 0x9C3D51C097A978ULL, 0xA753D6BA70E070ULL, 0xB32050893A7A1CULL,
 0xBFAD2D6A142DE0ULL, 0xCD0562CDD18890ULL, 0xDB34734CC30C60ULL,
 0xEA4674D3A3A86DULL, 0xFA48170B70D3BCULL, 0x10B46A9FDFECF26ULL,
 0x11D5024F925FE1CULL, 0x130732DB271E821ULL, 0x144BF1FAD464CF0ULL,
 0x15A4413E57AC428ULL, 0x17112E8C079A7B0ULL, 0x1893D4A46FB867AULL,
 0x1A2D5BAA9851488ULL, 0x1BDEF9B11DA7DC4ULL, 0x1DA9F34C3A82C08ULL,
 0x1F8F9C28EAE764AULL, 0x219157A94CB4FF0ULL, 0x23B0998664AEC88ULL,
 0x25EEE6776F66630ULL, 0x284DD4DEE65CC93ULL, 0x2ACF0D7D6299834ULL,
 0x2D744C2A86E5282ULL, 0x303F60941CC5794ULL, 0x33322F03904EB27ULL,
 0x364EB129F7D7040ULL, 0x3996F6F2D596A60ULL, 0x3D0D275DC240940ULL,
 0x40B3815F30A5D48ULL, 0x448C5CC87B7F420ULL, 0x489A2B376F88290ULL,
 0x4CDF790D8526C20ULL, 0x515EEE6EFDF6A48ULL, 0x561B504A1BA4D40ULL,
 0x5B178166A6AD060ULL, 0x6056837DFCAC440ULL, 0x65DB785BDF26229ULL,
 0x6BA9A3083CC77ACULL, 0x71C468FA3063F1EULL, 0x782F53547222B0CULL,
 0x7EEE102B788A83DULL, 0x860473D5885E450ULL, 0x8D767A44F37FF38ULL,
 0x9548486CC85B490ULL, 0x9D7E2DB034A7FC6ULL, 0xA61CA55CDFA8438ULL,
 0xAF285830816195CULL, 0xB8A61DE9FDAC3B8ULL, 0xC29AFEE64B59E96ULL,
 0xCD0C35C9711C8D0ULL, 0xD7FF3133E344998ULL, 0xE37995848EE1990ULL,
 0xEF813EA7E0489AFULL, 0xFC1C41F41482524ULL, 0x10950F01326A667AULL,
 0x11725D6FAABB2B84ULL, 0x125A1C3F1F0FBFE3ULL, 0x134CBC5A6B2E9C20ULL,
 0x144AB2E50C349830ULL, 0x1554795E4F824CA0ULL, 0x166A8DC57BDAB674ULL,
 0x178D72BEFA76E7D0ULL, 0x18BDAFBA85EDA3A8ULL, 0x19FBD11A64E935D0ULL,
 0x1B48685BB6C1ABB4ULL, 0x1CA40C3FD82DC5A0ULL, 0x1E0F58F6E6598DB0ULL,
 0x1F8AF04B66CE7920ULL, 0x211779CF1AB653A5ULL, 0x22B5A309041FF39CULL,
 0x24661FA4A40ADA16ULL, 0x2629A9A2781D6BFCULL, 0x28010189BF086299ULL,
 0x29ECEE9B8AB96AB0ULL, 0x2BEE3F07279EAAC8ULL, 0x2E05C81FE05D1470ULL,
 0x30346694257BF9D2ULL, 0x327AFEA620A961E8ULL, 0x34DA7C65BB5C02F4ULL,
 0x3753D3EC20B9A568ULL, 0x39E80198C2CAFA22ULL, 0x3C980A4FEB28A6B0ULL,
 0x3F64FBBADF707228ULL, 0x424FEC89A1F515F0ULL, 0x4559FCB6573F408BULL,
 0x488455CA591AD614ULL, 0x4BD02B2500106B72ULL, 0x4F3EBA442C505F74ULL,
 0x52D14B0E973AD0DFULL, 0x5689301FF6D60300ULL, 0x5A67C716FCAB8C80ULL,
 0x5E6E78E539ABEF00ULL, 0x629EBA20F0DFF260ULL, 0x66FA0B58E2D75580ULL,
 0x6B81F96A1BED18C0ULL, 0x70381DD7CFA2D580ULL, 0x751E1F254B7E4260ULL,
 0x7A35B1320BFE2F00ULL, 0x7F809597FE66EC80ULL,
};
const uint64_t N_choose_12[] = { (206), 1, 13, 91, 455, 1820, 6188, 18564,
 50388, 0x1EC12, 0x47C2A, 0x9DDF6, 0x14A18E, 0x29431C, 0x4F59AC, 0x935D64,
 0x10941B4, 0x1D032FB, 0x317DE7F, 0x527C829, 0x869540D, 0xD755348,
 0x15261528, 0x20AF37F8, 0x31BCB958, 0x4A9B1604, 0x6E6AA5B4, 0xA160CACC,
 0xE91A087C, 0x14D009E68ULL, 0x1D6CBE8C8ULL, 0x2931D7918ULL, 0x39241A7F8ULL,
 0x4E91A46F5ULL, 0x6B23B1AF1ULL, 0x90F40E837ULL, 0xC2A6D1A93ULL,
 0x10389178C4ULL, 0x157B58DE34ULL, 0x1C43FBAB1CULL, 0x24F6707D4CULL,
 0x300D2BD616ULL, 0x3E1D8397DEULL, 0x4FDCCDC342ULL, 0x662654948AULL,
 0x82023D1A24ULL, 0xA4AD809894ULL, 0xCFA31C93DCULL, 0x104A6A1268CULL,
 0x145D049702FULL, 0x1959ACE5CA3ULL, 0x1F6F328D945ULL, 0x26D4A7DC119ULL,
 0x2FCAA7363D0ULL, 0x3A9CC369290ULL, 0x47A327B96B0ULL, 0x57446CE6870ULL,
 0x69F7A8CEC88ULL, 0x8046BEDF5E8ULL, 0x9AD0F804C18ULL, 0xBA4DE960D78ULL,
 0xDF90B1A7690ULL, 0x10B8B95A6C50ULL, 0x13F540532670ULL, 0x17C271260930ULL,
 0x1C36E65D2AE9ULL, 0x216C5BB932D5ULL, 0x27800F4F3C13ULL, 0x2E932CCC3B5FULL,
 0x36CB43C318ACULL, 0x4052C9FB497CULL, 0x4B59AAC74774ULL, 0x5815E47CCA84ULL,
 0x66C4353C419AULL, 0x77A8D849AB12ULL, 0x8B10554EB20EULL, 0xA15062F4E306ULL,
 0xBAC8DE5EEBECULL, 0xD7E4D9213F3CULL, 0xF91BBF752194ULL, 0x11EF2987D33A4ULL,
 0x149FD628FFB63ULL, 0x17AE08D9BD4C7ULL, 0x1B25295D81961ULL,
 0x1F11DC1194525ULL, 0x2382201417398ULL, 0x28856FE6BD238ULL,
 0x2E2CE4B98A228ULL, 0x348B5C8C82B48ULL, 0x3BB5A359DA58CULL,
 0x43C29F7CFAA9CULL, 0x4CCB818DA49E4ULL, 0x56EBF7E976CF4ULL,
 0x624266294EA78ULL, 0x6EF020C348498ULL, 0x7D19AD1D92E08ULL,
 0x8CE70659E61A8ULL, 0x9E83E72522DDDULL, 0xB22018DA8E1B9ULL,
 0xC7EFC84E221EFULL, 0xE02BE095AA22BULL, 0xFB126C2CBE8D4ULL,
 0x118E6FCD447404ULL, 0x139F31A92E638CULL, 0x15E86BA4098A7CULL,
 0x186F8BC0CF909EULL, 0x1B3A77274DAFC6ULL, 0x1E4F932217B05AULL,
 0x21B5CEB3132B02ULL, 0x2574ACC6F8DA74ULL, 0x29944F108C07A4ULL,
 0x2E1D8194ADEC8CULL, 0x3319C6F0DC66FCULL, 0x38936566189697ULL,
 0x3E9574B2A320EBULL, 0x452BECC570EE7DULL, 0x4C63B556B45E31ULL,
 0x544AB6715444A0ULL, 0x5CEFE9F9ABAE20ULL, 0x66636E3E736060ULL,
 0x70B699A1409FE0ULL, 0x7BFC0F6493E310ULL, 0x8847D5AE05FDD0ULL,
 0x95AF6CCBBB0A30ULL, 0xA449E7CCE004F0ULL, 0xB430067D9ADC20ULL,
 0xC57C50D77CABA0ULL, 0xD84B33F83335E0ULL, 0xECBB20B0E96F60ULL,
 0x102ECABC17F51D1ULL, 0x11B02AFD37F419DULL, 0x1352271497B3FCBULL,
 0x15173C3F8460CF7ULL, 0x1702132E04C6B3CULL, 0x1915829FE28F7CCULL,
 0x1B549224A69C564ULL, 0x1DC27D001621B34ULL, 0x2062B534CCCA522ULL,
 0x2338E6B69F31FFAULL, 0x2648FAC68084426ULL, 0x29971B79B5F8A7EULL,
 0x2D27B76E33487BCULL, 0x30FF85AE0C12BCCULL, 0x352389C3F775AC4ULL,
 0x39991802F4F3E94ULL, 0x3E65DA0334083CBULL, 0x438FD3567299B0FULL,
 0x491D667619D3E99ULL, 0x4F1559ED74C0A3DULL, 0x557EDDC270618E8ULL,
 0x5C61911F69F8048ULL, 0x63C58840A5A5758ULL, 0x6BB352A81D9E438ULL,
 0x7434019A70CC714ULL, 0x7D512EE7CEFA984ULL, 0x87150403D8752FCULL,
 0x918A416F7F8336CULL, 0x9CBC4678132AD88ULL, 0xA8B7194EB46DB68ULL,
 0xB5876F7B91863F8ULL, 0xC33AB6B05DB7058ULL, 0xD1DF1DFD97F18C5ULL,
 0xE1839F6E4EFEC81ULL, 0xF2380A0E2EEBBA7ULL, 0x1040D0C5DC14B9C3ULL,
 0x117143F38E86A1E4ULL, 0x12B603133BCCEED4ULL, 0x141047272147B2FCULL,
 0x15815A0FE1C15AACULL, 0x170A975A28BCE126ULL, 0x18AD6D14D241F5AEULL,
 0x1A6B5CAFE41C7372ULL, 0x1C45FBE4A7C49F7AULL, 0x1E3EF5A7367315C4ULL,
 0x20580B21CB3E65B4ULL, 0x229314BA3189523CULL, 0x24F20321A87FB86CULL,
 0x2776E06F96E584FFULL, 0x2A23D1476D0F1D33ULL, 0x2CFB160A157D6FB5ULL,
 0x2FFF0C135749C749ULL, 0x33322F03904EB270ULL, 0x36971A162FCC22B0ULL,
 0x3A3089855D258D10ULL, 0x3E015BFB39499650ULL, 0x420C94112C53F398ULL,
 0x465559DDB40BE7B8ULL, 0x4ADEFC912B046A48ULL, 0x4FACF4220356D668ULL,
 0x54C2E308F33640B0ULL, 0x5A24980D94F08DF0ULL, 0x5FD61023FF5B5E50ULL,
 0x65DB785BDF262290ULL, 0x6C392FE19D1884B9ULL, 0x72F3CA1220E4FC65ULL,
 0x7A1010A1C3EB3B83ULL,
};
const uint64_t N_choose_13[] = { (169), 1, 14, 105, 560, 2380, 8568, 27132,
 77520, 0x31AE2, 0x7970C, 0x117502, 0x261690, 0x4F59AC, 0x9EB358, 0x13210BC,
 0x23B5270, 0x40B856B, 0x72363EA, 0xC4B2C13, 0x14B48020, 0x2229D368,
 0x374FE890, 0x57FF2088, 0x89BBD9E0, 0xD456EFE4, 0x142C19598ULL,
 0x1E4226064ULL, 0x2CD3C68E0ULL, 0x41A3D0748ULL, 0x5F108F010ULL,
 0x884266928ULL, 0xC16681120ULL, 0x10FF825815ULL, 0x17B1BD7306ULL,
 0x20C0FE5B3DULL, 0x2CEB6B75D0ULL, 0x3D23FCEE94ULL, 0x529F55CCC8ULL,
 0x6EE35177E4ULL, 0x93D9C1F530ULL, 0xC3E6EDCB46ULL, 0x10204716324ULL,
 0x151E13F2666ULL, 0x1B80793BAF0ULL, 0x23A09D0D514ULL, 0x2DEB7516DA8ULL,
 0x3AE5A6E0184ULL, 0x4B3010F2810ULL, 0x5F8D158983FULL, 0x78E6C26F4E2ULL,
 0x9855F4FCE27ULL, 0xBF2A9CD8F40ULL, 0xEEF5440F310ULL, 0x1299207785A0ULL,
 0x171352F31C50ULL, 0x1C8799C184C0ULL, 0x2327144E7148ULL, 0x2B2B803C6730ULL,
 0x34D88FBCB348ULL, 0x407D6E52C0C0ULL, 0x4E76796D3750ULL, 0x5F2F32C7A3A0ULL,
 0x7324731ACA10ULL, 0x8AE6E440D340ULL, 0xA71DCA9DFE29ULL, 0xC88A265730FEULL,
 0xF00A35A66D11ULL, 0x11E9D6272A870ULL, 0x15568A635C11CULL,
 0x195BB70310A98ULL, 0x1E1151AF8520CULL, 0x2392AFF751C90ULL,
 0x29FEF34B15E2AULL, 0x317980CFB093CULL, 0x3A2A86249BB4AULL,
 0x443F8C53E9E50ULL, 0x4FEC1A39D8A3CULL, 0x5D6A67CBEC978ULL,
 0x6CFC23C33EB0CULL, 0x7EEB4D4B11EB0ULL, 0x938B237411A13ULL,
 0xAB392C4DCEEDAULL, 0xC65E55AB5083BULL, 0xE57031BCE4D60ULL,
 0x108F251D0FC0F8ULL, 0x13177C1B7B9330ULL, 0x15FA4A67143558ULL,
 0x1943002FDC60A0ULL, 0x1CFE5A657A062CULL, 0x213A845D49B0C8ULL,
 0x26073C7623FAACULL, 0x2B75FBF4BB67A0ULL, 0x319A2257505218ULL,
 0x3889246384D6B0ULL, 0x405ABF355E04B8ULL, 0x49292F9AFC6660ULL,
 0x53116E0D4E943DULL, 0x5E336F9AF775F6ULL, 0x6AB26C1FD997E5ULL,
 0x78B52A29343A10ULL, 0x886650EC0022E4ULL, 0x99F4C0B94496E8ULL,
 0xAD93F26272FA74ULL, 0xC37C5E067C84F0ULL, 0xDBEBE9C74C158EULL,
 0xF72660EE99C554ULL, 0x11575F410B175AEULL, 0x1372BC2C3C4A0B0ULL,
 0x15CA06F8ABD7B24ULL, 0x18634BE9B4982C8ULL, 0x1B452402FF76F54ULL,
 0x1E76C0720D3D650ULL, 0x21FFF6C86EC6CE7ULL, 0x25E94E1398F8DD2ULL,
 0x2A3C0CDFF007C4FULL, 0x2F0248355B4DA80ULL, 0x3446F39C7091F20ULL,
 0x3A15F23C0B4CD40ULL, 0x407C291FF282DA0ULL, 0x478792BA068CD80ULL,
 0x4F4753B04FCB090ULL, 0x57CBD10B302AE60ULL, 0x6126C7D7EBDB890ULL,
 0x6B6B6654B9DBD80ULL, 0x76AE66BC93899A0ULL, 0x83062BCA0B54540ULL,
 0x908ADF098E87B20ULL, 0x9F5691149D1EA80ULL, 0xAF855BD0B513C51ULL,
 0xC13586CDED07DEEULL, 0xD487ADE284BBDB9ULL, 0xE99EEA22091CAB0ULL,
 0x100A0FD500DE35ECULL, 0x119B67FEFF072DB8ULL, 0x1350B1214970F31CULL,
 0x152CD8F14AD30E50ULL, 0x17330444979FB372ULL, 0x196692B00192D36CULL,
 0x1BCB225C699B1792ULL, 0x1E64941404FAA210ULL, 0x21370F8AE82F29CCULL,
 0x244707E5C8F05598ULL, 0x279940820867B05CULL, 0x2B32D20237B6EEF0ULL,
 0x2F192FA26AF772BBULL, 0x33522CD7D2210DCAULL, 0x37E4033F33BE4C63ULL,
 0x3CD558DE0B0A56A0ULL, 0x422D46BA32106F88ULL, 0x47F35FCC28AFEFD0ULL,
 0x4E2FB850330A4728ULL, 0x54EAED7AB4E42B60ULL, 0x5C2E2D945BF0F274ULL,
 0x64034082D8E09BF8ULL, 0x6C7490C31667EEF4ULL, 0x758D34DA0E602260ULL,
 0x7F58F9418F92CFE8ULL,
};
const uint64_t N_choose_14[] = { (143), 1, 15, 120, 680, 3060, 11628, 38760,
 0x1C638, 0x4E11A, 0xC7826, 0x1DED28, 0x4403B8, 0x935D64, 0x13210BC,
 0x2642178, 0x49F73E8, 0x8AAF953, 0xFCE5D3D, 0x1C198950, 0x30CE0970,
 0x52F7DCD8, 0x8A47C568, 0xE246E5F0, 0x16C02BFD0ULL, 0x24059AFB4ULL,
 0x3831B454CULL, 0x5673DA5B0ULL, 0x8347A0E90ULL, 0xC4EB715D8ULL,
 0x123FC005E8ULL, 0x1AC3E66F10ULL, 0x26DA4E8030ULL, 0x37D9D0D845ULL,
 0x4F8B8E4B4BULL, 0x704C8CA688ULL, 0x9D37F81C58ULL, 0xDA5BF50AECULL,
 0x12CFB4AD7B4ULL, 0x19BDE9C4F98ULL, 0x22FB85E44C8ULL, 0x2F39F4C100EULL,
 0x3F5A3BD7332ULL, 0x54784FC9998ULL, 0x6FF8C905488ULL, 0x9399661299CULL,
 0xC184DB29744ULL, 0xFC6A82098C8ULL, 0x1479A92FC0D8ULL, 0x1A727A885917ULL,
 0x2200E6AF4DF9ULL, 0x2B8645FF1C20ULL, 0x3778EFCCAB60ULL, 0x4668440D9E70ULL,
 0x590164852410ULL, 0x7014B7784060ULL, 0x8C9C5139C520ULL, 0xAFC365883668ULL,
 0xDAEEE5C49D98ULL, 0x10FC7758150E0ULL, 0x15044E3D411A0ULL,
 0x19EBB5D4148F0ULL, 0x1FDEA9008EC90ULL, 0x2710F0323B6A0ULL,
 0x2FBF5E76489E0ULL, 0x3A313B2028809ULL, 0x46B9DD859B907ULL,
 0x55BA80E002618ULL, 0x67A457072CE88ULL, 0x7CFAE16A88FA4ULL,
 0x9656986D99A3CULL, 0xB467EA1D1EC48ULL, 0xD7FA9A14708D8ULL,
 0x101F98D5F86702ULL, 0x133730E2F3703EULL, 0x16D9D9453D2B88ULL,
 0x1B1DD20A7BC9D8ULL, 0x201C93AE195414ULL, 0x25F33A2AD81D8CULL,
 0x2CC2FC670C0898ULL, 0x34B1B13BBD2748ULL, 0x3DEA6372FE415BULL,
 0x489DF637DB3035ULL, 0x5503DB92903870ULL, 0x635ADEAE5E85D0ULL,
 0x73EA03CB6E46C8ULL, 0x87017FE6E9D9F8ULL, 0x9CFBCA4DFE0F50ULL,
 0xB63ECA7DDA6FF0ULL, 0xD33D24E354761CULL, 0xF477A9409E26E4ULL,
 0x11A7EE5B6C22190ULL, 0x145F4E1AB7D8930ULL, 0x1778F0402CDDB48ULL,
 0x1B018286652B1F8ULL, 0x1F072E79BB0B6B0ULL, 0x2399C1736AD1D10ULL,
 0x28CAD8543FBB14DULL, 0x2EAE0F4DEF32743ULL, 0x3559360FECCBF28ULL,
 0x3CE488B2800F938ULL, 0x456AEDC14011C1CULL, 0x4F0A39CCD45B304ULL,
 0x59E378F2FB8AD78ULL, 0x661B3ED36353268ULL, 0x73D9FD6FD8147F6ULL,
 0x834C637EC1B0D4AULL, 0x94A3C2BFCCC82F8ULL, 0xA8167EEC09123A8ULL,
 0xBDE085E4B4E9ECCULL, 0xD643D1CE6982194ULL, 0xF188F5D168F90E8ULL,
 0x10FFFB6437636738ULL, 0x131FFAD0BE4FD41FULL, 0x157E8FB1F7DF61F1ULL,
 0x1822507FF6DFDE40ULL, 0x1B1275034C94B8C0ULL, 0x1E56E43D139DD7E0ULL,
 0x21F84360D452A520ULL, 0x260005F2D37AD2C0ULL, 0x2A787F1E73E3A040ULL,
 0x2F6CF45978E050D0ULL, 0x34E9B16A2BE2FF30ULL, 0x3AFC1DE7AAA0B7C0ULL,
 0x41B2D44CF63E7540ULL, 0x491DBAB8BF770EE0ULL, 0x514E1D75602C5420ULL,
 0x5A56CB65F914CF40ULL, 0x644C347742E6B9C0ULL, 0x6F448A344E37F611ULL,
 0x7B57E2A12D0873FFULL,
};
const uint64_t N_choose_15[] = { (125), 1, 16, 136, 816, 3876, 15504, 54264,
 0x29A30, 0x77B4A, 0x13F370, 0x31E098, 0x75E450, 0x10941B4, 0x23B5270,
 0x49F73E8, 0x93EE7D0, 0x11E9E123, 0x21B83E60, 0x3DD1C7B0, 0x6E9FD120,
 0xC197ADF8, 0x14BDF7360ULL, 0x22E265950ULL, 0x39A291920ULL, 0x5DA82C8D4ULL,
 0x95D9E0E20ULL, 0xEC4DBB3D0ULL, 0x16F955C260ULL, 0x23480CD838ULL,
 0x3587CCDE20ULL, 0x504BB34D30ULL, 0x772601CD60ULL, 0xAEFFD2A5A5ULL,
 0xFE8B60F0F0ULL, 0x16ED7ED9778ULL, 0x20C0FE5B3D0ULL, 0x2E66BDABEBCULL,
 0x41367259670ULL, 0x5AF45C1E608ULL, 0x7DEFE202AD0ULL, 0xAD29D6C3ADEULL,
 0xEC84129AE10ULL, 0x140FC62647A8ULL, 0x1B0F52B69C30ULL, 0x2448E917C5CCULL,
 0x306136CA5D10ULL, 0x4027DEEAF5D8ULL, 0x54A1881AB6B0ULL, 0x6F1402A30FC7ULL,
 0x9114E9525DC0ULL, 0xBC9B2F5179E0ULL, 0xF4141F1E2540ULL, 0x13A7C632BC3B0ULL,
 0x1937DC7B0E7C0ULL, 0x203927F292820ULL, 0x2902ED062ED40ULL,
 0x33FF235EB23A8ULL, 0x41AE11BAFC140ULL, 0x52AA891311220ULL,
 0x67AED750523C0ULL, 0x819A8D2466CB0ULL, 0xA1793624F5940ULL,
 0xC88A265730FE0ULL, 0xF84984CD799C0ULL, 0x1327ABFEDA21C9ULL,
 0x179349D733DAD0ULL, 0x1CEEF1E53400E8ULL, 0x23693755A6CF70ULL,
 0x2B38E56C4F5F14ULL, 0x349E4EF328F950ULL, 0x3FE4CD94FAE598ULL,
 0x4D64773641EE70ULL, 0x5D84100C3A5572ULL, 0x70BB40EF2DC5B0ULL,
 0x87951A346AF138ULL, 0xA2B2EC3EE6BB10ULL, 0xC2CF7FED000F24ULL,
 0xE8C2BA17D82CB0ULL, 0x11585B67EE43548ULL, 0x14A3767BAA15C90ULL,
 0x18821CB2D9F9DEBULL, 0x1D0BFC1657ACE20ULL, 0x225C39CF80B0690ULL,
 0x2891E7BA6698C60ULL, 0x2FD087F71D7D328ULL, 0x38409FF58C1AD20ULL,
 0x42105C9A6BFBC70ULL, 0x4D74494249A2C60ULL, 0x5AA81B907EEA27CULL,
 0x69EF962488CC960ULL, 0x7B97847FF4EEAF0ULL, 0x8FF6D29AACC7420ULL,
 0xA76FC2DAD9A4F68ULL, 0xC27145613ED0160ULL, 0xE17873DAF9DB810ULL,
 0x10512354E64AD520ULL, 0x12DDD0DA2A46866DULL, 0x15C8B1CF0939ADB0ULL,
 0x191E453008066CD8ULL, 0x1CEC8DBB30076610ULL, 0x21433C974408822CULL,
 0x2633E034114E3530ULL, 0x2BD217C34106E2A8ULL, 0x3233CBB0773C1510ULL,
 0x39716B8774BD5D06ULL, 0x41A631BF60D86A50ULL, 0x4AF06DEB5DA4ED48ULL,
 0x5571D5DA1E3610F0ULL, 0x614FDE386984AFBCULL, 0x6EB41B55501CD150ULL,
 0x7DCCAAB266AC6238ULL,
};
const uint64_t N_choose_16[] = { (111), 1, 17, 153, 969, 4845, 20349, 74613,
 0x3BDA5, 0xB38EF, 0x1F2C5F, 0x510CF7, 0xC6F147, 0x1D032FB, 0x40B856B,
 0x8AAF953, 0x11E9E123, 0x23D3C246, 0x458C00A6, 0x835DC856, 0xF1FD9976,
 0x1B395476EULL, 0x2FF74BACEULL, 0x52D9B141EULL, 0x8C7C42D3EULL,
 0xEA246F612ULL, 0x17FFE50432ULL, 0x26C4C0B802ULL, 0x3DBE167A62ULL,
 0x610623529AULL, 0x968DF030BAULL, 0xE6D9A37DEAULL, 0x15DFFA54B4AULL,
 0x20CFF77F0EFULL, 0x30B8AD8E1DFULL, 0x47A62C67957ULL, 0x68672AC2D27ULL,
 0x96CDE86EBE3ULL, 0xD8045AC8253ULL, 0x132F8B6E685BULL, 0x1B0E898E932BULL,
 0x25E126FACE09ULL, 0x34A968247C19ULL, 0x48B92E4AC3C1ULL, 0x63C881015FF1ULL,
 0x88116A1925BDULL, 0xB872A0E382CDULL, 0xF89A7FCE78A5ULL, 0x14D3C07E92F55ULL,
 0x1BC500A8C3F1CULL, 0x24D64F3DE9CDCULL, 0x30A00233016BCULL,
 0x3FE14424E3BFCULL, 0x53890A579FFACULL, 0x6CC0E6D2AE76CULL,
 0x8CFA0EC540F8CULL, 0xB5FCFBCB6FCCCULL, 0xE9FC1F2A22074ULL,
 0x12BAA30E51E1B4ULL, 0x17E54B9F82F3D4ULL, 0x1E603914881794ULL,
 0x2679E1E6CE8444ULL, 0x309175491DDD84ULL, 0x3D1A17AE90ED64ULL,
 0x4C9EAFFB688724ULL, 0x5FC65BFA42A8EDULL, 0x7759A5D17683BDULL,
 0x944897B6AA84A5ULL, 0xB7B1CF0C515415ULL, 0xE2EAB478A0B329ULL,
 0x11789036BC9AC79ULL, 0x1576DD100C49211ULL, 0x1A4D24837068081ULL,
 0x20256584340D5F3ULL, 0x2731199326E9BA3ULL, 0x2FAA6B366D98CDBULL,
 0x39D599FA5C047EBULL, 0x460291F92C0570FULL, 0x548EBD9AA9883BFULL,
 0x65E7190297CB907ULL, 0x7A8A8F7E41E1597ULL, 0x930CAC311BDB382ULL,
 0xB018A84773881A2ULL, 0xD274E216F438832ULL, 0xFB06C9D15AD1492ULL,
 0x12AD751C8784E7BAULL, 0x16317F1BE04694DAULL, 0x1A5284E58706514AULL,
 0x1F29C979ABA07DAAULL, 0x24D44B32B38F2026ULL, 0x2B734494FC1BE986ULL,
 0x332CBCDCFB6AD476ULL, 0x3C2C2A06A6374896ULL, 0x46A3263453D197FEULL,
 0x52CA3A8A67BE995EULL, 0x60E1C1C8175C516EULL, 0x7132E51CFDA7268EULL,
};
const uint64_t N_choose_17[] = { (101), 1, 18, 171, 1140, 5985, 26334,
 0x18A53, 0x547F8, 0x1080E7, 0x2FAD46, 0x80BA3D, 0x147AB84, 0x317DE7F,
 0x72363EA, 0xFCE5D3D, 0x21B83E60, 0x458C00A6, 0x8B18014C, 0x10E75C9A2ULL,
 0x200736318ULL, 0x3B408AA86ULL, 0x6B37D6554ULL, 0xBE1187972ULL,
 0x14A8DCA6B0ULL, 0x234B239CC2ULL, 0x3B4B08A0F4ULL, 0x620FC958F6ULL,
 0x9FCDDFD358ULL, 0x100D40325F2ULL, 0x19761F356ACULL, 0x27E3B96D496ULL,
 0x3DC3B3C1FE0ULL, 0x5E93AB410CFULL, 0x8F4C58CF2AEULL, 0xD6F28536C05ULL,
 0x13F59AFF992CULL, 0x1D627986850FULL, 0x2AE2BF330762ULL, 0x3E124AA16FBDULL,
 0x5920D43002E8ULL, 0x7F01FB2AD0F1ULL, 0xB3AB634F4D0AULL, 0xFC64919A10CBULL,
 0x1602D129B70BCULL, 0x1E83E7CB49679ULL, 0x2A0B11D981946ULL,
 0x3994B9D6691EBULL, 0x4E687A54FC140ULL, 0x6A2D7AFDC005CULL,
 0x8F03CA3BA9D38ULL, 0xBFA3CC6EAB3F4ULL, 0xFF8510938EFF0ULL,
 0x1530E1AEB2EF9CULL, 0x1BFCF01BDDD708ULL, 0x24CC910831E694ULL,
 0x302C60C4E8E360ULL, 0x3ECC22B78B03D4ULL, 0x5186C5C5DCE588ULL,
 0x696C11655FD95CULL, 0x87CC4A79E7F0F0ULL, 0xAE462C60B67534ULL,
 0xDED7A1A9D452B8ULL, 0x11BF1B95865401CULL, 0x168906953CDC740ULL,
 0x1C856C54E10702DULL, 0x23FB06B1F86F3EAULL, 0x2D3F902D631788FULL,
 0x38BAAD1E282CCA4ULL, 0x46E95865B237FCDULL, 0x5861E89C6ED2C46ULL,
 0x6DD8C5AC7B1BE57ULL, 0x8825EA2FEB83ED8ULL, 0xA84B4FB41F914CBULL,
 0xCF7C6947467B06EULL, 0xFF26D47DB413D49ULL, 0x138FC6E781018534ULL,
 0x17EFF00713C1DC43ULL, 0x1D38DBE0BE5A6002ULL, 0x23974D70E7D71909ULL,
 0x2B3FF668CBF52EA0ULL, 0x3470C12BDDB2E222ULL, 0x3F724BB054EB63C4ULL,
 0x4C9999D1C42EEBF6ULL, 0x5C4A066ED9DC0088ULL, 0x6EF77B8B6160E842ULL,
};
const uint64_t N_choose_18[] = { (94), 1, 19, 190, 1330, 7315, 33649,
 0x20DC4, 0x755BC, 0x17D6A3, 0x4783E9, 0xC83E26, 0x20FE9AA, 0x527C829,
 0xC4B2C13, 0x1C198950, 0x3DD1C7B0, 0x835DC856, 0x10E75C9A2ULL,
 0x21CEB9344ULL, 0x41D5EF65CULL, 0x7D167A0E2ULL, 0xE84E50636ULL,
 0x1A65FD7FA8ULL, 0x2F0EDA2658ULL, 0x5259FDC31AULL, 0x8DA506640EULL,
 0xEFB4CFBD04ULL, 0x18F82AF905CULL, 0x29056B2B64EULL, 0x427B8A60CFAULL,
 0x6A5F43CE190ULL, 0xA822F790170ULL, 0x106B6A2D123FULL, 0x19602FBA04EDULL,
 0x26CF580D70F2ULL, 0x3AC4F30D0A1EULL, 0x58276C938F2DULL, 0x830A2BC6968FULL,
 0xC11C7668064CULL, 0x11A3D4A980934ULL, 0x1993F45C2DA25ULL,
 0x24CEAA912272FULL, 0x3494F3AAC37FAULL, 0x4A97C4D47A8B6ULL,
 0x691BAC9FC3F2FULL, 0x9326BE7945875ULL, 0xCCBB784FAEA60ULL,
 0x11B23F2A4AABA0ULL, 0x185516DA26ABFCULL, 0x2145537DE14934ULL,
 0x2D3F9044CBFD28ULL, 0x3D37E14E04ED18ULL, 0x5268C2FCB7DCB4ULL,
 0x6E65B31895B3BCULL, 0x93324420C79A50ULL, 0xC35EA4E5B07DB0ULL,
 0x1022AC79D3B8184ULL, 0x153B18D6318670CULL, 0x1BD1D9EC8784068ULL,
 0x244E9E942603158ULL, 0x2F33015A316A68CULL, 0x3D207B74CEAF944ULL,
 0x4EDF970A5503960ULL, 0x65689D9F91E00A0ULL, 0x81EE09F472E70CDULL,
 0xA5E910A66B564B7ULL, 0xD328A0D3CE6DD46ULL, 0x10BE34DF1F69A9EAULL,
 0x152CCA657A8D29B7ULL, 0x1AB2E8EF417A55FDULL, 0x2190754A092C1454ULL,
 0x2A12D3ED07E4532CULL, 0x349788E849DD67F7ULL, 0x418F4F7CBE451865ULL,
 0x5181BCC4998655AEULL, 0x651183AC1A87DAE2ULL, 0x7D0173B32E49B725ULL,
};
const uint64_t N_choose_19[] = { (88), 1, 20, 210, 1540, 8855, 42504,
 0x2B3CC, 0xA0988, 0x21E02B, 0x696414, 0x131A23A, 0x3418BE4, 0x869540D,
 0x14B48020, 0x30CE0970, 0x6E9FD120, 0xF1FD9976, 0x200736318ULL,
 0x41D5EF65CULL, 0x83ABDECB8ULL, 0x100C258D9AULL, 0x1E910A93D0ULL,
 0x38F7081378ULL, 0x6805E239D0ULL, 0xBA5FDFFCEAULL, 0x14804E660F8ULL,
 0x237B9B61DFCULL, 0x3C73C65AE58ULL, 0x657931864A6ULL, 0xA7F4BBE71A0ULL,
 0x11253FFB5330ULL, 0x1BA76F7454A0ULL, 0x2C12D9A166DFULL, 0x4573095B6BCCULL,
 0x6C426168DCBEULL, 0xA7075475E6DCULL, 0xFF2EC1097609ULL, 0x18238ECD00C98ULL,
 0x24355633812E4ULL, 0x35D92ADD01C18ULL, 0x4F6D1F392F63DULL,
 0x743BC9CA51D6CULL, 0xA8D0BD7515566ULL, 0xF36882498FE1CULL,
 0x15C842EE953D4BULL, 0x1EFAAED62995C0ULL, 0x2BC6665B248020ULL,
 0x3D78A5856F2BC0ULL, 0x55CDBC5F95D7BCULL, 0x77130FDD7720F0ULL,
 0xA452A022431E18ULL, 0xE18A8170480B30ULL, 0x133F3446CFFE7E4ULL,
 0x1A258F785959BA0ULL, 0x2358B3BA65D35F0ULL, 0x2F8E9E08C0DB3A0ULL,
 0x3FB14A829493524ULL, 0x54EC6358C619C30ULL, 0x70BE3D454D9DC98ULL,
 0x950CDBD973A0DF0ULL, 0xC43FDD33A50B47CULL, 0x1016058A873BADC0ULL,
 0x1503FEFB2C8BE720ULL, 0x1B5A88D525A9E7C0ULL, 0x237969746CD8588DULL,
 0x2DD7FA7ED38DBD44ULL, 0x3B0A848C10749A8AULL, 0x4BC8B96B2FDE4474ULL,
 0x60F583D0AA6B6E2BULL, 0x7BA86CBFEBE5C428ULL,
};
const uint64_t N_choose_20[] = { (83), 1, 21, 231, 1771, 10626, 53130,
 0x38356, 0xD8CDE, 0x2F6D09, 0x98D11D, 0x1CA7357, 0x50BFF3B, 0xD755348,
 0x2229D368, 0x52F7DCD8, 0xC197ADF8, 0x1B395476EULL, 0x3B408AA86ULL,
 0x7D167A0E2ULL, 0x100C258D9AULL, 0x20184B1B34ULL, 0x3EA955AF04ULL,
 0x77A05DC27CULL, 0xDFA63FFC4CULL, 0x19A061FF936ULL, 0x2E20B065A2EULL,
 0x519C4BC782AULL, 0x8E101222682ULL, 0xF38943A8B28ULL, 0x19B7DFF8FCC8ULL,
 0x2ADD1FF44FF8ULL, 0x46848F68A498ULL, 0x7297690A0B77ULL, 0xB80A72657743ULL,
 0x1244CD3CE5401ULL, 0x1CB5428443ADDULL, 0x2CA82E94DB0E6ULL,
 0x44CBBD61DBD7EULL, 0x690113955D062ULL, 0x9EDA3E725EC7AULL,
 0xEE475DAB8E2B7ULL, 0x162832775E0023ULL, 0x20B53E4EAF5589ULL,
 0x2FEBC6734853A5ULL, 0x45B40961DD90F0ULL, 0x64AEB8380726B0ULL,
 0x90751E932BA6D0ULL, 0xCDEDC4189AD290ULL, 0x123BB807830AA4CULL,
 0x19ACE9055A7CB3CULL, 0x23F213077EAE954ULL, 0x320ABB1E832F484ULL,
 0x4549EF65532DC68ULL, 0x5F6F7EDDAC87808ULL, 0x82C83298125ADF8ULL,
 0xB256D0A0D336198ULL, 0xF2081B2367C96BCULL, 0x146F47E7C2DE32ECULL,
 0x1B7B2BBC17B80F84ULL, 0x24CBF979AEF21D74ULL, 0x310FF74CE942D1F0ULL,
 0x4125FCD7707E7FB0ULL, 0x5629FBD29D0A66D0ULL, 0x718484A7C2B44E90ULL,
};
const uint64_t N_choose_21[] = { (79), 1, 22, 253, 2024, 12650, 65780,
 0x4844A, 0x121128, 0x417E31, 0xDA4F4E, 0x2A4C2A5, 0x7B0C1E0, 0x15261528,
 0x374FE890, 0x8A47C568, 0x14BDF7360ULL, 0x2FF74BACEULL, 0x6B37D6554ULL,
 0xE84E50636ULL, 0x1E910A93D0ULL, 0x3EA955AF04ULL, 0x7D52AB5E08ULL,
 0xF4F3092084ULL, 0x1D499491CD0ULL, 0x36E9F691606ULL, 0x650AA6F7034ULL,
 0xB6A6F2BE85EULL, 0x144B704E0EE0ULL, 0x238404889A08ULL, 0x3D3BE48196D0ULL,
 0x68190475E6C8ULL, 0xAE9D93DE8B60ULL, 0x12134FCE896D7ULL,
 0x1D93F6F4E0E1AULL, 0x2FD8C431C621BULL, 0x4C8E06B609CF8ULL,
 0x7936354AE4DDEULL, 0xBE01F2ACC0B5CULL, 0x1270306421DBBEULL,
 0x1C5DD44B47C838ULL, 0x2B424A2600AAEFULL, 0x416A7C9D5EAB12ULL,
 0x621FBAEC0E009BULL, 0x920B815F565440ULL, 0xD7BF8AC133E530ULL,
 0x13C6E42F93B0BE0ULL, 0x1CCE3618C66B2B0ULL, 0x29AD125A5018540ULL,
 0x3BE8CA61D322F8CULL, 0x5595B3672D9FAC8ULL, 0x7987C66EAC4E41CULL,
 0xAB92818D2F7D8A0ULL, 0xF0DC70F282AB508ULL, 0x1504BEFD02F32D10ULL,
 0x1D3142268418DB08ULL, 0x2856AF30914C3CA0ULL, 0x377730E2C7C8D35CULL,
 0x4BE678CA8AA70648ULL, 0x6761A486A25F15CCULL,
};
const uint64_t N_choose_22[] = { (76), 1, 23, 276, 2300, 14950, 80730,
 0x5BFA4, 0x17D0CC, 0x594EFD, 0x1339E4B, 0x3D860F0, 0xB8922D0, 0x20AF37F8,
 0x57FF2088, 0xE246E5F0, 0x22E265950ULL, 0x52D9B141EULL, 0xBE1187972ULL,
 0x1A65FD7FA8ULL, 0x38F7081378ULL, 0x77A05DC27CULL, 0xF4F3092084ULL,
 0x1E9E6124108ULL, 0x3BE7F5B5DD8ULL, 0x72D1EC473DEULL, 0xD7DC933E412ULL,
 0x18E8385FCC70ULL, 0x2D33A8ADDB50ULL, 0x50B7AD367558ULL, 0x8DF391B80C28ULL,
 0xF60C962DF2F0ULL, 0x1A4AA2A0C7E50ULL, 0x2C5DF26F51527ULL,
 0x49F1E96432341ULL, 0x79CAAD95F855CULL, 0xC658B44C02254ULL,
 0x13F8EE996E7032ULL, 0x1FD90DC43A7B8EULL, 0x32493E285C574CULL,
 0x4EA71273A41F84ULL, 0x79E95C99A4CA73ULL, 0xBB53D937037585ULL,
 0x11D739423117620ULL, 0x1AF7F158267CA60ULL, 0x2873EA0439BAF90ULL,
 0x3C3ACE33CD6BB70ULL, 0x5909044C93D6E20ULL, 0x82B616A6E3EF360ULL,
 0xBE9EE108B7122ECULL, 0x11434946FE4B1DB4ULL, 0x18DBC5ADE91001D0ULL,
 0x2394EDC6BC07DA70ULL, 0x32A2B4D5E4328F78ULL, 0x47A773D2E725BC88ULL,
 0x64D8B5F96B3E9790ULL,
};
const uint64_t N_choose_23[] = { (74), 1, 24, 300, 2600, 17550, 98280,
 0x73F8C, 0x1F1058, 0x785F55, 0x1ABFDA0, 0x5845E90, 0x110D8160, 0x31BCB958,
 0x89BBD9E0, 0x16C02BFD0ULL, 0x39A291920ULL, 0x8C7C42D3EULL, 0x14A8DCA6B0ULL,
 0x2F0EDA2658ULL, 0x6805E239D0ULL, 0xDFA63FFC4CULL, 0x1D499491CD0ULL,
 0x3BE7F5B5DD8ULL, 0x77CFEB6BBB0ULL, 0xEAA1D7B2F8EULL, 0x1C27E6AF13A0ULL,
 0x35101F0EE010ULL, 0x6243C7BCBB60ULL, 0xB2FB74F330B8ULL, 0x140EF06AB3CE0ULL,
 0x236FB9CD92FD0ULL, 0x3DBA5C6E5AE20ULL, 0x6A184EDDAC347ULL,
 0xB40A3841DE688ULL, 0x12DD4E5D7D6BE4ULL, 0x1F42D9A23D8E38ULL,
 0x333BC83BABFE6AULL, 0x5314D5FFE679F8ULL, 0x855E142842D144ULL,
 0xD405269BE6F0C8ULL, 0x14DEE83358BBB3BULL, 0x209425C6C8F30C0ULL,
 0x326B5F08FA0A6E0ULL, 0x4D6350612087140ULL, 0x75D73A655A420D0ULL,
 0xB212089927ADC40ULL, 0x10B1B0CE5BB84A60ULL, 0x18DD1238C9F73DC0ULL,
 0x24C70049556860ACULL, 0x360A499053B37E60ULL, 0x4EE60F3E3CC38030ULL,
 0x727AFD04F8CB5AA0ULL,
};
const uint64_t N_choose_24[] = { (72), 1, 25, 325, 2925, 20475, 0x1CFE3,
 0x90F6F, 0x281FC7, 0xA07F1C, 0x24C7CBC, 0x7D0DB4C, 0x18DE5CAC, 0x4A9B1604,
 0xD456EFE4, 0x24059AFB4ULL, 0x5DA82C8D4ULL, 0xEA246F612ULL, 0x234B239CC2ULL,
 0x5259FDC31AULL, 0xBA5FDFFCEAULL, 0x19A061FF936ULL, 0x36E9F691606ULL,
 0x72D1EC473DEULL, 0xEAA1D7B2F8EULL, 0x1D543AF65F1CULL, 0x397C21A572BCULL,
 0x6E8C40B452CCULL, 0xD0D008710E2CULL, 0x183CB7D643EE4ULL,
 0x2C4BA840F7BC4ULL, 0x4FBB620E8AB94ULL, 0x8D75BE7CE59B4ULL,
 0xF78E0D5A91CFBULL, 0x1AB98459C70383ULL, 0x2D96D2B7446F67ULL,
 0x4CD9AC5981FD9FULL, 0x801574952DFC09ULL, 0xD32A4A95147601ULL,
 0x158885EBD574745ULL, 0x22C8D85593E380DULL, 0x37A7C088EC9F348ULL,
 0x583BE64FB592408ULL, 0x8AA74558AF9CAE8ULL, 0xD80A95B9D023C28ULL,
 0x14DE1D01F2A65CF8ULL, 0x1FFF3D8B85213938ULL, 0x30B0EE59E0D98398ULL,
 0x498E0092AAD0C158ULL, 0x6E5500DC00392204ULL,
};
const uint64_t N_choose_25[] = { (70), 1, 26, 351, 3276, 23751, 0x22CAA,
 0xB3C19, 0x335BE0, 0xD3DAFC, 0x32057B8, 0xAF13304, 0x23CF8FB0, 0x6E6AA5B4,
 0x142C19598ULL, 0x3831B454CULL, 0x95D9E0E20ULL, 0x17FFE50432ULL,
 0x3B4B08A0F4ULL, 0x8DA506640EULL, 0x14804E660F8ULL, 0x2E20B065A2EULL,
 0x650AA6F7034ULL, 0xD7DC933E412ULL, 0x1C27E6AF13A0ULL, 0x397C21A572BCULL,
 0x72F8434AE578ULL, 0xE18483FF3844ULL, 0x1B2548C704670ULL,
 0x3362009D48554ULL, 0x5FADA8DE40118ULL, 0xAF690AECCACACULL,
 0x13CDEC969B0660ULL, 0x2346CD6C44235BULL, 0x3E0051C60B26DEULL,
 0x6B97247D4F9645ULL, 0xB870D0D6D193E4ULL, 0x13886456BFF8FEDULL,
 0x20BB090011405EEULL, 0x36438EEBE6B4D33ULL, 0x590C67417A98540ULL,
 0x90B427CA6737888ULL, 0xE8F00E1A1CC9C90ULL, 0x173975372CC66778ULL,
 0x24BA1E92C9C8A3A0ULL, 0x39983B94BC6F0098ULL, 0x59977920419039D0ULL,
};
const uint64_t N_choose_26[] = { (69), 1, 27, 378, 3654, 27405, 0x297B7,
 0xDD3D0, 0x412FB0, 0x1150AAC, 0x4356264, 0xF269568, 0x32F62518, 0xA160CACC,
 0x1E4226064ULL, 0x5673DA5B0ULL, 0xEC4DBB3D0ULL, 0x26C4C0B802ULL,
 0x620FC958F6ULL, 0xEFB4CFBD04ULL, 0x237B9B61DFCULL, 0x519C4BC782AULL,
 0xB6A6F2BE85EULL, 0x18E8385FCC70ULL, 0x35101F0EE010ULL, 0x6E8C40B452CCULL,
 0xE18483FF3844ULL, 0x1C30907FE7088ULL, 0x3755D946EB6F8ULL,
 0x6AB7D9E433C4CULL, 0xCA6582C273D64ULL, 0x179CE8DAF3EA10ULL,
 0x2B6AD5718EF070ULL, 0x4EB1A2DDD313CBULL, 0x8CB1F4A3DE3AA9ULL,
 0xF84919212DD0EEULL, 0x1B0B9E9F7FF64D2ULL, 0x2E9402F63FEF4BFULL,
 0x4F4F0BF6512FAADULL, 0x85929AE237E47E0ULL, 0xDE9F0223B27CD20ULL,
 0x16F5329EE19B45A8ULL, 0x258433808367E238ULL, 0x3CBDA8B7B02E49B0ULL,
 0x6177C74A79F6ED50ULL,
};
const uint64_t N_choose_27[] = { (68), 1, 28, 406, 4060, 31465, 0x312A0,
 0x10E670, 0x521620, 0x16720CC, 0x59C8330, 0x14C31898, 0x47B93DB0,
 0xE91A087C, 0x2CD3C68E0ULL, 0x8347A0E90ULL, 0x16F955C260ULL,
 0x3DBE167A62ULL, 0x9FCDDFD358ULL, 0x18F82AF905CULL, 0x3C73C65AE58ULL,
 0x8E101222682ULL, 0x144B704E0EE0ULL, 0x2D33A8ADDB50ULL, 0x6243C7BCBB60ULL,
 0xD0D008710E2CULL, 0x1B2548C704670ULL, 0x3755D946EB6F8ULL,
 0x6EABB28DD6DF0ULL, 0xD9638C720AA3CULL, 0x1A3C90F347E7A0ULL,
 0x31D979CE3BD1B0ULL, 0x5D444F3FCAC220ULL, 0xABF5F21D9DD5EBULL,
 0x138A7E6C17C1094ULL, 0x230F0FFE2A9E182ULL, 0x3E1AAE9DAA94654ULL,
 0x6CAEB193EA83B13ULL, 0xBBFDBD8A3BB35C0ULL, 0x14190586C7397DA0ULL,
 0x2202F5A902614AC0ULL, 0x38F82847E3FC9068ULL, 0x5E7C5BC8676472A0ULL,
};
const uint64_t N_choose_28[] = { (67), 1, 29, 435, 4495, 35960, 0x39F18,
 0x148588, 0x669BA8, 0x1CDBC74, 0x76A3FA4, 0x1C2D583C, 0x63E695EC,
 0x14D009E68ULL, 0x41A3D0748ULL, 0xC4EB715D8ULL, 0x23480CD838ULL,
 0x610623529AULL, 0x100D40325F2ULL, 0x29056B2B64EULL, 0x657931864A6ULL,
 0xF38943A8B28ULL, 0x238404889A08ULL, 0x50B7AD367558ULL, 0xB2FB74F330B8ULL,
 0x183CB7D643EE4ULL, 0x3362009D48554ULL, 0x6AB7D9E433C4CULL,
 0xD9638C720AA3CULL, 0x1B2C718E415478ULL, 0x35690281893C18ULL,
 0x67427C4FC50DC8ULL, 0xC486CB8F8FCFE8ULL, 0x1707CBDAD2DA5D3ULL,
 0x2A924A46EA9B667ULL, 0x4DA15A4515397E9ULL, 0x8BBC08E2BFCDE3DULL,
 0xF86ABA76AA51950ULL, 0x1B4687800E604F10ULL, 0x2F5F8D06D599CCB0ULL,
 0x516282AFD7FB1770ULL,
};
const uint64_t N_choose_29[] = { (67), 1, 30, 465, 4960, 40920, 0x43EF0,
 0x18C478, 0x7F6020, 0x24D1C94, 0x9B75C38, 0x25E4B474, 0x89CB4A60,
 0x1D6CBE8C8ULL, 0x5F108F010ULL, 0x123FC005E8ULL, 0x3587CCDE20ULL,
 0x968DF030BAULL, 0x19761F356ACULL, 0x427B8A60CFAULL, 0xA7F4BBE71A0ULL,
 0x19B7DFF8FCC8ULL, 0x3D3BE48196D0ULL, 0x8DF391B80C28ULL, 0x140EF06AB3CE0ULL,
 0x2C4BA840F7BC4ULL, 0x5FADA8DE40118ULL, 0xCA6582C273D64ULL,
 0x1A3C90F347E7A0ULL, 0x35690281893C18ULL, 0x6AD20503127830ULL,
 0xD2148152D785F8ULL, 0x1969B4CE26755E0ULL, 0x307180A8F94FBB3ULL,
 0x5B03CAEFE3EB21AULL, 0xA8A52534F924A03ULL, 0x134612E17B8F2840ULL,
 0x22CCBE88E6344190ULL, 0x3E134608F49490A0ULL, 0x6D72D30FCA2E5D50ULL,
};
const uint64_t N_choose_30[] = { (66), 1, 31, 496, 5456, 46376, 0x4F418,
 0x1DB890, 0x9D18B0, 0x2EA3544, 0xCA1917C, 0x328645F0, 0xBC519050,
 0x2931D7918ULL, 0x884266928ULL, 0x1AC3E66F10ULL, 0x504BB34D30ULL,
 0xE6D9A37DEAULL, 0x27E3B96D496ULL, 0x6A5F43CE190ULL, 0x11253FFB5330ULL,
 0x2ADD1FF44FF8ULL, 0x68190475E6C8ULL, 0xF60C962DF2F0ULL, 0x236FB9CD92FD0ULL,
 0x4FBB620E8AB94ULL, 0xAF690AECCACACULL, 0x179CE8DAF3EA10ULL,
 0x31D979CE3BD1B0ULL, 0x67427C4FC50DC8ULL, 0xD2148152D785F8ULL,
 0x1A42902A5AF0BF0ULL, 0x33AC44F881661D0ULL, 0x641DC5A17AB5D83ULL,
 0xBF2190915EA0F9DULL, 0x167C6B5C657C59A0ULL, 0x29C27E3DE10B81E0ULL,
 0x4C8F3CC6C73FC370ULL,
};
const uint64_t N_choose_31[] = { (66), 1, 32, 528, 5984, 52360, 0x5C0A0,
 0x237930, 0xC091E0, 0x3AAC724, 0x104C58A0, 0x42D29E90, 0xFF242EE0,
 0x39241A7F8ULL, 0xC16681120ULL, 0x26DA4E8030ULL, 0x772601CD60ULL,
 0x15DFFA54B4AULL, 0x3DC3B3C1FE0ULL, 0xA822F790170ULL, 0x1BA76F7454A0ULL,
 0x46848F68A498ULL, 0xAE9D93DE8B60ULL, 0x1A4AA2A0C7E50ULL,
 0x3DBA5C6E5AE20ULL, 0x8D75BE7CE59B4ULL, 0x13CDEC969B0660ULL,
 0x2B6AD5718EF070ULL, 0x5D444F3FCAC220ULL, 0xC486CB8F8FCFE8ULL,
 0x1969B4CE26755E0ULL, 0x33AC44F881661D0ULL, 0x675889F102CC3A0ULL,
 0xCB764F927D82123ULL, 0x18A97E023DC230C0ULL, 0x2F25E95EA33E8A60ULL,
 0x58E8679C844A0C40ULL,
};
const uint64_t N_choose_32[] = { (66), 1, 33, 561, 6545, 58905, 0x6A6B9,
 0x2A1FE9, 0xEAB1C9, 0x49578ED, 0x14E1D18D, 0x57B4701D, 0x156D89EFDULL,
 0x4E91A46F5ULL, 0x10FF825815ULL, 0x37D9D0D845ULL, 0xAEFFD2A5A5ULL,
 0x20CFF77F0EFULL, 0x5E93AB410CFULL, 0x106B6A2D123FULL, 0x2C12D9A166DFULL,
 0x7297690A0B77ULL, 0x12134FCE896D7ULL, 0x2C5DF26F51527ULL,
 0x6A184EDDAC347ULL, 0xF78E0D5A91CFBULL, 0x2346CD6C44235BULL,
 0x4EB1A2DDD313CBULL, 0xABF5F21D9DD5EBULL, 0x1707CBDAD2DA5D3ULL,
 0x307180A8F94FBB3ULL, 0x641DC5A17AB5D83ULL, 0xCB764F927D82123ULL,
 0x196EC9F24FB04246ULL, 0x321847F48D727306ULL, 0x613E315330B0FD66ULL,
};
const uint64_t N_choose_33[] = { (66), 1, 34, 595, 7140, 66045, 0x7A8B6,
 0x31C89F, 0x11C7A68, 0x5B1F355, 0x1A93C4E2, 0x724834FF, 0x1C920D3FCULL,
 0x6B23B1AF1ULL, 0x17B1BD7306ULL, 0x4F8B8E4B4BULL, 0xFE8B60F0F0ULL,
 0x30B8AD8E1DFULL, 0x8F4C58CF2AEULL, 0x19602FBA04EDULL, 0x4573095B6BCCULL,
 0xB80A72657743ULL, 0x1D93F6F4E0E1AULL, 0x49F1E96432341ULL,
 0xB40A3841DE688ULL, 0x1AB98459C70383ULL, 0x3E0051C60B26DEULL,
 0x8CB1F4A3DE3AA9ULL, 0x138A7E6C17C1094ULL, 0x2A924A46EA9B667ULL,
 0x5B03CAEFE3EB21AULL, 0xBF2190915EA0F9DULL, 0x18A97E023DC230C0ULL,
 0x321847F48D727306ULL, 0x64308FE91AE4E60CULL,
};

static const uint64_t * N_choose_K[] = { 0, 0, 0, 0, 0, 0, 0, 0,
 N_choose_8,  N_choose_9,  N_choose_10, N_choose_11, N_choose_12, N_choose_13, 
 N_choose_14, N_choose_15, N_choose_16, N_choose_17, N_choose_18, N_choose_19, 
 N_choose_20, N_choose_21, N_choose_22, N_choose_23, N_choose_24, N_choose_25, 
 N_choose_26, N_choose_27, N_choose_28, N_choose_29, N_choose_30, N_choose_31, 
 N_choose_32, N_choose_33, 
};
void
NumericCell::do_binomial(Cell * Z, APL_Integer K, APL_Integer N, bool negate)
{
   Assert(N >= K);
   Assert(K >= 0);

   // make K < N/2 (or: K < N-K)
   //
   if (K > (N - K))   K = N - K;

   // for small K, the max N gets very large. We handle these cases
   // separately in order to have a small max_N for integer results.
   //
   if (K <= 7)
      {
        APL_Integer z = 0;
        if (K == 0)
           {
             z = 1;
           }
        else if (K == 1)
           {
             z = N;
           }
        else if (K == 2)   // K == 2 or N-2
           {
             if (N > 4294967296)   goto non_integer;   // K!N too large

             // either N or N-1 must be even
             //
             if (N & 1)   // N-1 even: divide N-1
                {
                  z = N * ((N-1)/2);
                }
             else         // N even: divide N-1
                {
                  z = (N - 1) * (N/2);
                }
           }
        else if (K == 3)   // K == 3 or N-3
           {
             if (N > 3810779)   goto non_integer;   // K!N too large

             APL_Integer N0 = N;
             APL_Integer N1 = N - 1;
             APL_Integer N2 = N - 2;

             if (N0 & 1)   N1 = N1/2;
             else          N0 = N0/2;

             if      ((N0 % 3) == 0)   N0 = N0/3;
             else if ((N1 % 3) == 0)   N1 = N1/3;
             else                      N2 = N2/3;
             z = N0 * N1 * N2;
           }
        else if (K == 4)   // K == 4 or N-4
           {
             if (N > 121977)   goto non_integer;   // K!N too large

             APL_Integer N0 = N;
             APL_Integer N1 = N - 1;
             APL_Integer N2 = N - 2;
             APL_Integer N3 = N - 3;

             for (int j = 0; j < 3; ++j)
                 {
                   if      ((N0 & 1) == 0)   N0 = N0/2;
                   else if ((N1 & 1) == 0)   N1 = N1/2;
                   else if ((N2 & 1) == 0)   N2 = N2/2;
                   else                      N3 = N3/2;
                 }

             if      ((N0 % 3) == 0)   N0 = N0/3;
             else if ((N1 % 3) == 0)   N1 = N1/3;
             else                      N2 = N2/3;

             z = N0 * N1 * N2 * N3;
           }
        else if (K == 5)   // K == 5 or N-5
           {
             if (N > 16175)   goto non_integer;   // K!N too large

             APL_Integer N0 = N;
             APL_Integer N1 = N - 1;
             APL_Integer N2 = N - 2;
             APL_Integer N3 = N - 3;
             APL_Integer N4 = N - 4;

             for (int j = 0; j < 3; ++j)
                 {
                   if      ((N0 & 1) == 0)   N0 = N0/2;
                   else if ((N1 & 1) == 0)   N1 = N1/2;
                   else if ((N2 & 1) == 0)   N2 = N2/2;
                   else if ((N3 & 1) == 0)   N3 = N3/2;
                   else                      N4 = N4/2;
                 }

             if      ((N0 % 3) == 0)   N0 = N0/3;
             else if ((N1 % 3) == 0)   N1 = N1/3;
             else                      N2 = N2/3;

             if      ((N0 % 5) == 0)   N0 = N0/5;
             else if ((N1 % 5) == 0)   N1 = N1/5;
             else if ((N2 % 5) == 0)   N2 = N2/5;
             else if ((N3 % 5) == 0)   N3 = N3/5;
             else                      N4 = N4/5;

             z = N0 * N1 * N2 * N3 * N4;
           }
        else if (K == 6)   // K == 6 or N-6
           {
             if (N > 4337)   goto non_integer;   // K!N too large

             APL_Integer N0 = N;
             APL_Integer N1 = N - 1;
             APL_Integer N2 = N - 2;
             APL_Integer N3 = N - 3;
             APL_Integer N4 = N - 4;
             APL_Integer N5 = N - 5;

             for (int j = 0; j < 4; ++j)
                 {
                   if      ((N0 & 1) == 0)   N0 = N0/2;
                   else if ((N1 & 1) == 0)   N1 = N1/2;
                   else if ((N2 & 1) == 0)   N2 = N2/2;
                   else if ((N3 & 1) == 0)   N3 = N3/2;
                   else if ((N4 & 1) == 0)   N4 = N4/2;
                   else                      N5 = N5/2;
                 }

             for (int j = 0; j < 2; ++j)
                 {
                   if      ((N0 % 3) == 0)   N0 = N0/3;
                   else if ((N1 % 3) == 0)   N1 = N1/3;
                   else if ((N2 % 3) == 0)   N2 = N2/3;
                   else if ((N3 % 3) == 0)   N3 = N3/3;
                   else if ((N4 % 3) == 0)   N4 = N4/3;
                   else                      N5 = N5/3;
                 }

             if      ((N0 % 5) == 0)   N0 = N0/5;
             else if ((N1 % 5) == 0)   N1 = N1/5;
             else if ((N2 % 5) == 0)   N2 = N2/5;
             else if ((N3 % 5) == 0)   N3 = N3/5;
             else if ((N4 % 5) == 0)   N4 = N4/5;
             else                      N5 = N5/5;

             z = N0 * N1 * N2 * N3 * N4 * N5;
           }
        else if (K == 7)   // K == 7 or N-7
           {
             if (N > 1733)   goto non_integer;   // K!N too large

             APL_Integer N0 = N;
             APL_Integer N1 = N - 1;
             APL_Integer N2 = N - 2;
             APL_Integer N3 = N - 3;
             APL_Integer N4 = N - 4;
             APL_Integer N5 = N - 5;
             APL_Integer N6 = N - 6;

             for (int j = 0; j < 4; ++j)
                 {
                   if      ((N0 & 1) == 0)   N0 = N0/2;
                   else if ((N1 & 1) == 0)   N1 = N1/2;
                   else if ((N2 & 1) == 0)   N2 = N2/2;
                   else if ((N3 & 1) == 0)   N3 = N3/2;
                   else if ((N4 & 1) == 0)   N4 = N4/2;
                   else if ((N5 & 1) == 0)   N5 = N5/2;
                   else                      N6 = N6/2;
                 }

             for (int j = 0; j < 2; ++j)
                 {
                   if      ((N0 % 3) == 0)   N0 = N0/3;
                   else if ((N1 % 3) == 0)   N1 = N1/3;
                   else if ((N2 % 3) == 0)   N2 = N2/3;
                   else if ((N3 % 3) == 0)   N3 = N3/3;
                   else if ((N4 % 3) == 0)   N4 = N4/3;
                   else if ((N5 % 3) == 0)   N5 = N5/3;
                   else                      N6 = N6/3;
                 }

             if      ((N0 % 5) == 0)   N0 = N0/5;
             else if ((N1 % 5) == 0)   N1 = N1/5;
             else if ((N2 % 5) == 0)   N2 = N2/5;
             else if ((N3 % 5) == 0)   N3 = N3/5;
             else if ((N4 % 5) == 0)   N4 = N4/5;
             else if ((N5 % 5) == 0)   N5 = N5/5;
             else                      N6 = N6/5;

             if      ((N0 % 7) == 0)   N0 = N0/7;
             else if ((N1 % 7) == 0)   N1 = N1/7;
             else if ((N2 % 7) == 0)   N2 = N2/7;
             else if ((N3 % 7) == 0)   N3 = N3/7;
             else if ((N4 % 7) == 0)   N4 = N4/7;
             else if ((N5 % 7) == 0)   N5 = N5/7;
             else                      N6 = N6/7;

             z = N0 * N1 * N2 * N3 * N4 * N5 *N6;
           }

        if (negate)   new (Z) IntCell(-z);
        else          new (Z) IntCell( z);
        return;
      }

   // at this point, K >= 8. If K ≤ 33 then we try a table lookup.
   // That may for small N (and give a more precise interger result if so).
   //
   if (K <= 33)
      {
        const uint64_t * table_K = N_choose_K[K];
        Assert(table_K);

        // the first element in the table is the max. N (including) for K
        //
        if (N <= table_K[0])
           {
             const APL_Integer z = table_K[1 + (N - K)];
             if (negate)   new (Z) IntCell(-z);
             else          new (Z) IntCell( z);
             return;
           }
      }

non_integer:

   // at this point, the result is too large for an APL_Integer
   // N! fits into double for N ≤ 170
   //
   if (N > 170)   DOMAIN_ERROR;

double z  = tgamma(N + 1);          // N!
       z /= tgamma(K + 1);          // ÷ K!
       z /= tgamma((N - K) + 1);    // ÷ (N-K)!

   new (Z) FloatCell(z);
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::bif_binomial(Cell * Z, const Cell * A) const
{
   if (!is_near_real() || !A->is_near_real())   // complex result
      {
        const APL_Float r_1_a    = A->get_real_value();
        const APL_Float r_1_b    = get_real_value();
        const APL_Float r_1_b__a = r_1_b - A->get_real_value();

        const APL_Float i_a   = A->get_imag_value();
        const APL_Float i_b   = get_imag_value();
        const APL_Float i_b_a = i_b - i_a;

        const APL_Complex gam_1_a    = ComplexCell::gamma(r_1_a,    i_a);
        const APL_Complex gam_1_b    = ComplexCell::gamma(r_1_b,    i_b);
        const APL_Complex gam_1_b__a = ComplexCell::gamma(r_1_b__a, i_b_a);

        new (Z) ComplexCell(gam_1_b / (gam_1_a * gam_1_b__a));
        return E_NO_ERROR;
      }

   if (!is_near_int() || !A->is_near_int())   // non-integer result
      {
        const APL_Float r_1_a    = 1.0 + A->get_real_value();
        const APL_Float r_1_b    = 1.0 + get_real_value();
        const APL_Float r_1_b__a = r_1_b - A->get_real_value();

        if (r_1_a < 0    && is_near_int(r_1_a))      return E_DOMAIN_ERROR;
        if (r_1_b < 0    && is_near_int(r_1_b))      return E_DOMAIN_ERROR;
        if (r_1_b__a < 0 && is_near_int(r_1_b__a))   return E_DOMAIN_ERROR;

        new (Z) FloatCell(  tgamma(r_1_b) / (tgamma(r_1_a) * tgamma(r_1_b__a)));
        return E_NO_ERROR;
      }

const APL_Integer a = A->get_near_int();
const APL_Integer b = get_near_int();

int how = 0;
   if (a < 0)    how |= 4;
   if (b < 0)    how |= 2;
   if (b < a)    how |= 1;

   switch(how)
      {
        case 0: do_binomial(Z, a, b, false);                       break;
        case 1: new (Z) IntCell(0);                                break;
        case 2: Assert(0 && "Impossible case 2");                  break;
        case 3: do_binomial(Z, a, a - (b + 1), a & 1);             break;
        case 4: new (Z) IntCell(0);                                break;
        case 5: Assert(0 && "Impossible case 5");                  break;
        case 6: do_binomial(Z, -(b + 1), -(a + 1), (b - a) & 1);   break;
        case 7: new (Z) IntCell(0);                                break;
      }

   return E_NO_ERROR;
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::bif_and(Cell * Z, const Cell * A) const
{
const APL_Float qct = Workspace::get_CT();

   // if either value is 0 then return 0
   //
   if (A->is_near_zero() || is_near_zero())
      {
        new (Z) IntCell(0);
        return E_NO_ERROR;
      }

   // if both args are 1 then return the classical A ∧ B
   //
   if (A->is_near_one() && is_near_one())
      {
        new (Z) IntCell(1);
        return E_NO_ERROR;
      }

   // handle the complex case before the int case because
   // is_near_int() can be true for complex numbers.
   //
   if (A->is_complex_cell() || is_complex_cell())   // complex
      {
        // a or b is complex; we assume (require) Gaussian integers.
        //
        APL_Complex gcd;
        const ErrorCode err = cpx_gcd(gcd, A->get_complex_value(),
                                           get_complex_value(), qct);
        if (err)   return err;

        new (Z) ComplexCell(A->get_complex_value() * (get_complex_value()/gcd));
        return E_NO_ERROR;
      }

   // if both args are int then return the least common multiple of them
   //
   if (A->is_near_int() && is_near_int())
      {
        const APL_Integer a = A->get_near_int();
        const APL_Integer b =    get_near_int();
        APL_Integer gcd;
        const ErrorCode err = int_gcd(gcd, a, b);
        if (err)   return err;
        new (Z) IntCell(a * (b / gcd));
        return E_NO_ERROR;
      }

   // if both args are real then return the (real) least common multiple of them
   //
   if (A->is_near_real() && is_near_real())
      {
        const APL_Float a = A->get_real_value();
        const APL_Float b =    get_real_value();
        APL_Float gcd;
        const ErrorCode err = flt_gcd(gcd, a, b, qct);
        if (err)   return err;
        new (Z) FloatCell(a * (b / gcd));
        return E_NO_ERROR;
      }

   return E_DOMAIN_ERROR;   // char ?
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::bif_nand(Cell * Z, const Cell * A) const
{
   if (A->get_near_bool() && get_near_bool())   new (Z) IntCell(0);
   else                                         new (Z) IntCell(1);
   return E_NO_ERROR;
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::bif_nor(Cell * Z, const Cell * A) const
{
   if (A->get_near_bool() || get_near_bool())   new (Z) IntCell(0);
   else                                         new (Z) IntCell(1);
   return E_NO_ERROR;
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::bif_or(Cell * Z, const Cell * A) const
{
const APL_Float qct = Workspace::get_CT();

   // if both args are boolean then return the classical A ∨ B
   //
   if (A->is_near_bool() && is_near_bool())
      {
        if ( A->get_near_bool() || get_near_bool())   new (Z) IntCell(1);
        else                                          new (Z) IntCell(0);
        return E_NO_ERROR;
      }

   if (!A->is_near_real() || !is_near_real())   // complex
      {
        if (A->is_near_zero())
           {
             new (Z) ComplexCell(get_complex_value());
             return E_NO_ERROR;
           }

        if (is_near_zero())
           {
             new (Z) ComplexCell(A->get_complex_value());
             return E_NO_ERROR;
           }

        // a or b is complex; we assume (require) Gaussian integers.
        //
        APL_Complex gcd;
        const ErrorCode err = cpx_gcd(gcd, A->get_complex_value(),
                                           get_complex_value(), qct);
        if (err)   return err;

        new (Z) ComplexCell(gcd);
        return E_NO_ERROR;
      }

   // if both args are int then return the greatest common divisor of them
   //
   if (A->is_near_int() && is_near_int())
      {
        const APL_Integer a = A->get_near_int();
        const APL_Integer b =    get_near_int();
        APL_Integer gcd;
        const ErrorCode err = int_gcd(gcd, a, b);
        if (err)   return err;
        new (Z) IntCell(gcd);
        return E_NO_ERROR;
      }

   // if both args are real then return the (real) greatest common divisor
   //
   if (A->is_near_real() && is_near_real())
      {
        const APL_Float a = A->get_real_value();
        const APL_Float b =    get_real_value();
        APL_Float gcd;
        const ErrorCode err = flt_gcd(gcd, a, b, qct);
        if (err)   return err;
        new (Z) FloatCell(gcd);
        return E_NO_ERROR;
      }

   return E_DOMAIN_ERROR;   // char ?
}
//-----------------------------------------------------------------------------
APL_Complex
NumericCell::cpx_max_real(APL_Complex a)
{
const double r_abs = a.real() < 0 ? -a.real() : a.real();
const double i_abs = a.imag() < 0 ? -a.imag() : a.imag();
APL_Complex z;

   if (r_abs < i_abs)   // multiply by i in order to exchange real and imag
      {
        z = APL_Complex(- a.imag(), a.real());
     }
   else
      {
        z = a;
     }

   if (z.real() < 0)   // multiply by -1 in order to make z.real positive
      {
        z = APL_Complex(- z.real(), - z.imag());
      }

   return z;
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::int_gcd(APL_Integer & z, APL_Integer a, APL_Integer b)
{
   if (a == 0x8000000000000000ULL)   return E_DOMAIN_ERROR;
   if (b == 0x8000000000000000ULL)   return E_DOMAIN_ERROR;

   if (a < 0)   a = - a;
   if (b < 0)   b = - b;
   if (b < a)
      {
         const APL_Integer _b = b;
         b = a;
         a = _b;
      }

   // at this point 0 ≤ a ≤ b
   //
   for (;;)
       {
         if (a == 0)   { z = b;   return E_NO_ERROR; }
         const APL_Integer r = b%a;
         b = a;
         a = r;
       }
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::flt_gcd(APL_Float & z, APL_Float a, APL_Float b, APL_Float qct)
{
   if (a < 0)   a = - a;
   if (b < 0)   b = - b;
   if (b < a)
      {
         const APL_Float _b = b;
         b = a;
         a = _b;
      }

   // at this point 0 ≤ a ≤ b
   //
   for (;;)
       {
         if (is_near_zero(a))   { z = b;   return E_NO_ERROR; }
         const APL_Float r = fmod(b, a);
         b = a;
         a = r;
       }
}
//-----------------------------------------------------------------------------
ErrorCode
NumericCell::cpx_gcd(APL_Complex & z, APL_Complex a, APL_Complex b,
                     APL_Float qct)
{
   if (!is_near_int(a.real()))   return E_DOMAIN_ERROR;
   if (!is_near_int(a.imag()))   return E_DOMAIN_ERROR;
   if (!is_near_int(b.real()))   return E_DOMAIN_ERROR;
   if (!is_near_int(b.imag()))   return E_DOMAIN_ERROR;

   // make a and b true integers
   //
   a = APL_Complex(round(a.real()), a.imag());
   b = APL_Complex(round(b.real()), b.imag());

   for (;;)
       {
         if (abs(a) > abs(b))   // make ∣b∣ > ∣a∣
            {
               const APL_Complex _b = b;
               b = a;
               a = _b;
            }

         if (abs(a) < 0.2)   { z = b;   return E_NO_ERROR; }

         const APL_Complex xy = b/a;
         const APL_Complex q(round(xy.real()), round(xy.imag()));
         const APL_Complex r(b - q*a);

         b = a;
         a = r;
       }
}
//-----------------------------------------------------------------------------
