#  Copyright (C) 2013-2014 Free Software Foundation, Inc.
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 3, or (at your option) any
# later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING3.  If not see
# <http://www.gnu.org/licenses/>.

# This Awk script takes passes.def and writes pass-instances.def,
# counting the instances of each kind of pass, adding an instance number
# to everywhere that NEXT_PASS is used.
#
# For example, the single-instanced pass:
#     NEXT_PASS (pass_warn_unused_result);
# becomes this in the output:
#     NEXT_PASS (pass_warn_unused_result, 1);
#
# The various instances of
#   NEXT_PASS (pass_copy_prop);
# become:
#   NEXT_PASS (pass_copy_prop, 1);
# through:
#   NEXT_PASS (pass_copy_prop, 8);
# (currently there are 8 instances of that pass)

# Usage: awk -f gen-pass-instances.awk passes.def

BEGIN {
	print "/* This file is auto-generated by gen-pass-instances.awk";
	print "   from passes.def.  */";
}

function handle_line()
{
	line = $0;
	where = match(line, /NEXT_PASS \((.+)\)/)
	if (where != 0)
	{
		len_of_start = length("NEXT_PASS (")
		len_of_end = length(")")
		len_of_pass_name = RLENGTH - (len_of_start + len_of_end)
		line_length = length(line)
		pass_starts_at = where + len_of_start
		pass_name = substr(line, pass_starts_at, len_of_pass_name)
		if (pass_name in pass_counts)
			pass_counts[pass_name]++;
		else
			pass_counts[pass_name] = 1;
		printf "%s, %s%s\n",
			substr(line, 1, pass_starts_at + len_of_pass_name - 1),
			pass_counts[pass_name],
			substr(line, pass_starts_at + len_of_pass_name);
	} else {
		print line;
	}
}

{ handle_line() }
