/* SPDX-License-Identifier: GPL-2.0-or-later */

#include <baseboard/gpio.h>
#include <baseboard/variants.h>
#include <commonlib/helpers.h>
#include <soc/gpio.h>
#include <fw_config.h>

/* Pad configuration in ramstage for Pujjo */
static const struct pad_config override_gpio_table[] = {
	/* A8  : WWAN_RF_DISABLE_ODL */
	PAD_CFG_GPO(GPP_A8, 1, DEEP),
	/* D3  : WCAM_RST_L ==> NC */
	PAD_NC_LOCK(GPP_D3, NONE, LOCK_CONFIG),
	/* D6  : SRCCLKREQ1# ==> WWAN_EN */
	PAD_CFG_GPO(GPP_D6, 1, DEEP),
	/* D15  : EN_PP2800_WCAM_X ==> NC */
	PAD_NC_LOCK(GPP_D15, NONE, LOCK_CONFIG),
	/* D16  : EN_PP1800_PP1200_WCAM_X ==> NC */
	PAD_NC_LOCK(GPP_D16, NONE, LOCK_CONFIG),
	/* D17 : NC ==> SD_WAKE_N */
	PAD_CFG_GPI_LOCK(GPP_D17, NONE, LOCK_CONFIG),
	/* F12 : WWAN_RST_L */
	PAD_CFG_GPO_LOCK(GPP_F12, 1, LOCK_CONFIG),
	/* H19 : SOC_I2C_SUB_INT_ODL */
	PAD_CFG_GPI_APIC(GPP_H19, NONE, PLTRST, LEVEL, NONE),
	/* H22  : WCAM_MCLK_R ==> NC */
	PAD_NC(GPP_H22, NONE),
	/* H23 : WWAN_SAR_DETECT_ODL ==> NC */
	PAD_NC_LOCK(GPP_H23, NONE, LOCK_CONFIG),
};

/* Pad configuration in ramstage for Pujjoteen5 */
static const struct pad_config override_5g_gpio_table[] = {
	/* A8  : WWAN_RF_DISABLE_ODL */
	PAD_CFG_GPO(GPP_A8, 1, DEEP),
	/* A12 : WWAN_PCIE_WAKE_ODL */
	PAD_CFG_GPI_LOCK(GPP_A12, NONE, LOCK_CONFIG),
	/* D3  : WCAM_RST_L ==> NC */
	PAD_NC_LOCK(GPP_D3, NONE, LOCK_CONFIG),
	/* D5  : SRCCLKREQ0# ==> WWAN_CLKREQ_ODL */
	PAD_CFG_NF(GPP_D5, NONE, DEEP, NF1),
	/* D6  : SRCCLKREQ1# ==> WWAN_EN */
	PAD_CFG_GPO_LOCK(GPP_D6, 1, LOCK_CONFIG),
	/* D15  : EN_PP2800_WCAM_X ==> NC */
	PAD_NC_LOCK(GPP_D15, NONE, LOCK_CONFIG),
	/* D16  : EN_PP1800_PP1200_WCAM_X ==> NC */
	PAD_NC_LOCK(GPP_D16, NONE, LOCK_CONFIG),
	/* D17 : NC ==> SD_WAKE_N */
	PAD_CFG_GPI_LOCK(GPP_D17, NONE, LOCK_CONFIG),
	/* F12 : WWAN_RST_L */
	PAD_CFG_GPO(GPP_F12, 0, DEEP),
	/* H19 : SOC_I2C_SUB_INT_ODL */
	PAD_CFG_GPI_APIC(GPP_H19, NONE, PLTRST, LEVEL, NONE),
	/* H21  : WCAM_MCLK_R ==> WWAN_PERST_L */
	PAD_CFG_GPO(GPP_H21, 0, DEEP),
	/* H22  : WCAM_MCLK_R ==> NC */
	PAD_NC(GPP_H22, NONE),
	/* H23 : WWAN_SAR_DETECT_ODL ==> NC */
	PAD_NC_LOCK(GPP_H23, NONE, LOCK_CONFIG),
};

/* Pad configuration in ramstage for Pujjo LTE EM060 */
static const struct pad_config override_em060_gpio_table[] = {
	/* A8  : WWAN_RF_DISABLE_ODL */
	PAD_CFG_GPO(GPP_A8, 1, DEEP),
	/* D3  : WCAM_RST_L ==> NC */
	PAD_NC_LOCK(GPP_D3, NONE, LOCK_CONFIG),
	/* D6  : SRCCLKREQ1# ==> WWAN_EN */
	PAD_CFG_GPO(GPP_D6, 1, DEEP),
	/* D15  : EN_PP2800_WCAM_X ==> NC */
	PAD_NC_LOCK(GPP_D15, NONE, LOCK_CONFIG),
	/* D16  : EN_PP1800_PP1200_WCAM_X ==> NC */
	PAD_NC_LOCK(GPP_D16, NONE, LOCK_CONFIG),
	/* D17 : NC ==> SD_WAKE_N */
	PAD_CFG_GPI_LOCK(GPP_D17, NONE, LOCK_CONFIG),
	/* F12 : WWAN_RST_L */
	PAD_CFG_GPO(GPP_F12, 1, DEEP),
	/* H19 : SOC_I2C_SUB_INT_ODL */
	PAD_CFG_GPI_APIC(GPP_H19, NONE, PLTRST, LEVEL, NONE),
	/* H22  : WCAM_MCLK_R ==> NC */
	PAD_NC(GPP_H22, NONE),
	/* H23 : WWAN_SAR_DETECT_ODL ==> NC */
	PAD_NC_LOCK(GPP_H23, NONE, LOCK_CONFIG),
};

/* Early pad configuration in bootblock */
static const struct pad_config early_gpio_table[] = {
	/* A13 : GPP_A13 ==> GSC_SOC_INT_ODL */
	PAD_CFG_GPI_APIC(GPP_A13, NONE, PLTRST, LEVEL, INVERT),
	/*
	* WWAN_EN is asserted in ramstage to meet the 500 ms warm reset toff
	* requirement. WWAN_EN must be asserted before WWAN_RST_L is released
	* (with min delay 0 ms), so this works as long as the pin used for
	* WWAN_EN comes before the pin used for WWAN_RST_L.
	*/
	/* D6  : SRCCLKREQ1# ==> WWAN_EN */
	PAD_CFG_GPO(GPP_D6, 0, DEEP),
	/* E12 : THC0_SPI1_IO1 ==> SOC_WP_OD */
	PAD_CFG_GPI_GPIO_DRIVER(GPP_E12, NONE, DEEP),
	/* F12 : WWAN_RST_L */
	PAD_CFG_GPO(GPP_F12, 0, DEEP),
	/* F18 : THC1_SPI2_INT# ==> EC_IN_RW_OD */
	PAD_CFG_GPI(GPP_F18, NONE, DEEP),
	/* H4  : I2C0_SDA ==> SOC_I2C_GSC_SDA */
	PAD_CFG_NF(GPP_H4, NONE, DEEP, NF1),
	/* H5  : I2C0_SCL ==> SOC_I2C_GSC_SCL */
	PAD_CFG_NF(GPP_H5, NONE, DEEP, NF1),
	/* H10 : UART0_RXD ==> UART_SOC_RX_DBG_TX */
	PAD_CFG_NF(GPP_H10, NONE, DEEP, NF2),
	/* H11 : UART0_TXD ==> UART_SOC_TX_DBG_RX */
	PAD_CFG_NF(GPP_H11, NONE, DEEP, NF2),
	/* H12 : UART0_RTS# ==> SD_PERST_L */
	PAD_CFG_GPO(GPP_H12, 0, DEEP),
	/* H13 : UART0_CTS# ==> EN_PP3300_SD_X */
	PAD_CFG_GPO(GPP_H13, 1, DEEP),
};


/* Early pad configuration in bootblock for Pujjoteen5 */
static const struct pad_config early_5g_gpio_table[] = {
	/* A13 : GPP_A13 ==> GSC_SOC_INT_ODL */
	PAD_CFG_GPI_APIC(GPP_A13, NONE, PLTRST, LEVEL, INVERT),
	/*
	* WWAN_EN is asserted in ramstage to meet the 500 ms warm reset toff
	* requirement. WWAN_EN must be asserted before WWAN_RST_L is released
	* (with min delay 0 ms), so this works as long as the pin used for
	* WWAN_EN comes before the pin used for WWAN_RST_L.
	*/
	/* D6  : SRCCLKREQ1# ==> WWAN_EN */
	PAD_CFG_GPO(GPP_D6, 0, DEEP),
	/* E12 : THC0_SPI1_IO1 ==> SOC_WP_OD */
	PAD_CFG_GPI_GPIO_DRIVER(GPP_E12, NONE, DEEP),
	/* F12 : WWAN_RST_L */
	PAD_CFG_GPO(GPP_F12, 0, DEEP),
	/* F18 : THC1_SPI2_INT# ==> EC_IN_RW_OD */
	PAD_CFG_GPI(GPP_F18, NONE, DEEP),
	/* H4  : I2C0_SDA ==> SOC_I2C_GSC_SDA */
	PAD_CFG_NF(GPP_H4, NONE, DEEP, NF1),
	/* H5  : I2C0_SCL ==> SOC_I2C_GSC_SCL */
	PAD_CFG_NF(GPP_H5, NONE, DEEP, NF1),
	/* H10 : UART0_RXD ==> UART_SOC_RX_DBG_TX */
	PAD_CFG_NF(GPP_H10, NONE, DEEP, NF2),
	/* H11 : UART0_TXD ==> UART_SOC_TX_DBG_RX */
	PAD_CFG_NF(GPP_H11, NONE, DEEP, NF2),
	/* H12 : UART0_RTS# ==> SD_PERST_L */
	PAD_CFG_GPO(GPP_H12, 0, DEEP),
	/* H13 : UART0_CTS# ==> EN_PP3300_SD_X */
	PAD_CFG_GPO(GPP_H13, 1, DEEP),
	/* H21  : WCAM_MCLK_R ==> WWAN_PERST_L */
	PAD_CFG_GPO(GPP_H21, 0, DEEP),
};

/* Pad configuration in romstage for Pujjo */
static const struct pad_config romstage_gpio_table[] = {
	/* Enable touchscreen, hold in reset */
	/* C0  : SMBCLK ==> EN_PP3300_TCHSCR */
	PAD_CFG_GPO(GPP_C0, 1, DEEP),
	/* C1  : SMBDATA ==> USI_RST_L */
	PAD_CFG_GPO(GPP_C1, 0, DEEP),

	/* H12 : UART0_RTS# ==> SD_PERST_L */
	PAD_CFG_GPO(GPP_H12, 1, DEEP),
};

/* Pad configuration in romstage for Pujjo WWAN EM060 */
static const struct pad_config romstage_em060_gpio_table[] = {
	/* H12 : UART0_RTS# ==> SD_PERST_L */
	PAD_CFG_GPO(GPP_H12, 1, DEEP),
	/* F12 : WWAN_RST_L */
	PAD_CFG_GPO(GPP_F12, 1, DEEP),
};

const struct pad_config *variant_gpio_override_table(size_t *num)
{
	if (fw_config_probe(FW_CONFIG(WWAN_5G, WWAN_5G_PRESENT))) {
		*num = ARRAY_SIZE(override_5g_gpio_table);
		return override_5g_gpio_table;
	} else if (fw_config_probe(FW_CONFIG(LTE_EM060, LTE_EM060_PRESENT))) {
		*num = ARRAY_SIZE(override_em060_gpio_table);
		return override_em060_gpio_table;
	} else {
		*num = ARRAY_SIZE(override_gpio_table);
		return override_gpio_table;
	}
}

const struct pad_config *variant_early_gpio_table(size_t *num)
{
	if (fw_config_probe(FW_CONFIG(WWAN_5G, WWAN_5G_PRESENT))) {
		*num = ARRAY_SIZE(early_5g_gpio_table);
		return early_5g_gpio_table;
	} else {
		*num = ARRAY_SIZE(early_gpio_table);
		return early_gpio_table;
	}
}

const struct pad_config *variant_romstage_gpio_table(size_t *num)
{
	if (fw_config_probe(FW_CONFIG(LTE_EM060, LTE_EM060_PRESENT))) {
		*num = ARRAY_SIZE(romstage_em060_gpio_table);
		return romstage_em060_gpio_table;
	} else {
		*num = ARRAY_SIZE(romstage_gpio_table);
		return romstage_gpio_table;
	}
}
