/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRAMMOD_H_
#define _VRAMMOD_H_

#include <VrInterpolatingSigProc.h>

#define TAP_RANGE            1024.0 //  = 2^TAP_RANGE_EXP 
#define TAP_RANGE_EXP          10

template<class iType,class oType> 
class VrAMMod : public VrInterpolatingSigProc<iType,oType> {
protected:
  float center_freq, amplitude, mod_index;                     
  int in1, in2, in3, in4, in5, in6, total_bits, mod_offset;
  int *filter_init_ptr;
  int buffer_length, buffer_bits;            
  int max_input_amplitude;
  unsigned int index, mod_mask;
  oType* waveform;
  virtual void initialize();
public:   
  virtual void work(int n);  
  VrAMMod(int, float, float, float);
};

template<class iType,class oType> void
VrAMMod<iType,oType>::work(int n)
{
  int *filter_ptr, res1, res2, res3, res4, res5, res6, res, temp;
  int i,j;
 
  for ( i=0; i < n/interp; i++,incReadPtr(1)) {

    in2 = in1; 
    in3 = in2;
    in4 = in3;
    in5 = in4;
    in6 = in5;
    in1 = inputRead();
    filter_ptr = filter_init_ptr;

    for ( j=0; j<interp; j++){
      
      res1 = in1 * *filter_ptr++;
      res2 = in2 * *filter_ptr++;
      res3 = in3 * *filter_ptr++;
      res4 = in4 * *filter_ptr++;
      res5 = in5 * *filter_ptr++;
      res6 = in6 * *filter_ptr++;


      res =  res1 + res2 + res3 + res4 + res5 + res6;
      //        cout << res << endl;
      temp = ((res  + 524288) * waveform[(index++) & mod_mask])>> 20 ;
      //   cout << (mod_offset ) << endl;
      outputWrite(temp);
    }
  }
  return;
}

template<class iType,class oType> void
VrAMMod<iType,oType>::initialize() {

  int M, cycles_per_buffer, mod_offset, mod_scale_bits, total_bits;
  float sf_out;

  buffer_length = 4096;
  mod_mask = buffer_length-1; 
  sf_out = (float) getOutputSamplingFrequency();
  cycles_per_buffer = buffer_length / (int) (sf_out / center_freq);

  /* Modify center_freq to make integral number of cycles in buffer */

  center_freq =  sf_out / (float) (buffer_length) *  cycles_per_buffer; 
  cout << "Setting Center Frequency to " << center_freq << " Hz" << endl; 

  mod_offset = (int)((float) (max_input_amplitude*TAP_RANGE) / mod_index);
  mod_scale_bits = int(log(mod_offset)/log(2.0))+1;
  total_bits = mod_scale_bits + TAP_RANGE_EXP; 

  /* allocate and create waveform */

  waveform = new oType[buffer_length];

  for (int i=0; i<buffer_length; i++) {
    waveform[i] = (oType)(amplitude*sin(2*M_PI*center_freq/sf_out*i));
  }

  /* allocate and create interpolation filter */
  /* note that filter coefs are reorderd for easier use later in work()*/

  filter_init_ptr = new int[6*interp];
  M = 6*interp-1;

  for (int i=0; i< interp; i++) {

    filter_init_ptr[6*i]   = (int)(TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i)/(float)M)));
    filter_init_ptr[6*i+1] = (int)(TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+interp)/(float)M)));
    filter_init_ptr[6*i+2] = (int)(TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+2*interp)/(float)M)));
    filter_init_ptr[6*i+3] = (int)(TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+3*interp)/(float)M)));
    filter_init_ptr[6*i+2] = (int)(TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+4*interp)/(float)M)));
    filter_init_ptr[6*i+3] = (int)(TAP_RANGE*
				   (0.54-0.46*cos(2*M_PI*(float)(i+5*interp)/(float)M)));

  }

  in1 = 0;
  in2 = 0;
  in3 = 0;
  in4 = 0;
  in5 = 0;
  in6 = 0;
  index = 0;
}



template<class iType,class oType> 
VrAMMod<iType,oType>::VrAMMod(int i, float fc, float a, float m)
  :VrInterpolatingSigProc<iType, oType>(1, i), center_freq(fc), amplitude(a),mod_index(m)
{
  max_input_amplitude = 127;
  setOutputSize (interp);
  setOutputSizeExact (1); //so we can use fast pointers
}

#endif







