## Copyright (C) 2016-2017 Kai T. Ohlhus <k.ohlhus@gmail.com>
##
## This file is part of Octave.
##
## Octave is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or (at
## your option) any later version.
##
## Octave is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with Octave; see the file COPYING.  If not, see
## <http://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @deftypefn  {} {} grabcode (@var{url})
## @deftypefnx {} {@var{code_str} =} grabcode (@var{url})
##
## Grab by the @code{publish} function generated HTML reports from Octave
## script files.
##
## The input parameter @var{url} must point to a local or remote HTML file
## with extension @samp{.htm} or @samp{.html} which was generated by the
## @code{publish} function.  With any other HTML file this will not work!
##
## If no return value is given, the grabbed code is saved to a temporary
## file and opened in the default editor.
##
## NOTE: You have to save the file at another location with arbitrary name,
## otherwise any grabbed code will be lost!
##
## With a return value given, the grabbed code will be returned as string
## @var{code_str}.
##
## An example:
##
## @example
## @group
## publish ("my_script.m");
## grabcode ("html/my_script.html");
## @end group
## @end example
##
## The example above publishes @samp{my_script.m} by default to
## @samp{html/my_script.html}.  Afterwards this published Octave script
## is grabbed to edit its content in a new temporary file.
##
## @seealso{publish}
## @end deftypefn

function varargout = grabcode (url)
  narginchk (1, 1);
  nargoutchk (0, 1);

  [~,~,ext] = fileparts (url);
  if (! strncmp (ext, ".htm", 4))
    error ("grabcode: URL should point to a published \".html\"-file");
  endif

  ## If url is a local file
  if (exist (url) == 2)
    oct_code = fileread (url);
  ## Otherwise try to read a url
  else
    [oct_code, success, message] = urlread (url);
    if (! success)
      error (["grabcode: ", message]);
    endif
  endif

  ## Extract relevant part
  start_str = "##### SOURCE BEGIN #####";
  end_str = "##### SOURCE END #####";
  oct_code = oct_code(strfind (oct_code, start_str) + length(start_str) + 1: ...
    strfind (oct_code, end_str)-1);

  ## Return Octave code string ...
  if (nargout == 1)
    varargout{1} = oct_code;
  ## ... or open temporary file in editor
  else
    fname = [tempname(), ".m"];
    fid = fopen (fname, "w");
    if (fid < 0)
      error ("grabcode: cannot open temporary file");
    endif
    fprintf (fid, "%s", oct_code);
    fclose (fid);
    edit (fname);
    warndlg (["grabcode: Make sure to save the temporary file\n\n\t", ...
      fname, "\n\nto a location of your choice. ", ...
      "Otherwise all grabbed code will be lost!"]);
  endif
endfunction

## Bad function calls

%!error grabcode ()
%!error grabcode (1)
%!error grabcode ("html/test_script.html", "pdf")
%!error [str1, str2] = grabcode ("html/test_script.html")