/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  ConfirmPayResultType,
  Duration,
  MerchantApiClient,
  PreparePayResultType,
  narrowOpSuccessOrThrow,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/helpers.js";

/**
 * Test for taler://payment-template/ URIs
 */
export async function runPaymentTemplateTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV3(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const mySummary = "hello, I'm a summary";

  const createTemplateRes = await merchantClient.createTemplate({
    template_id: "template1",
    template_description: "my test template",
    template_contract: {
      minimum_age: 0,
      pay_duration: Duration.toTalerProtocolDuration(
        Duration.fromSpec({
          minutes: 2,
        }),
      ),
      summary: mySummary,
    },
    editable_defaults: {
      amount: "TESTKUDOS:1" as AmountString,
    },
  });
  narrowOpSuccessOrThrow("createTemplate", createTemplateRes);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });
  await wres.withdrawalFinishedCond;

  const talerPayTemplateUri = `taler+http://pay-template/localhost:${merchant.port}/template1`;

  const checkPayTemplateResult = await walletClient.call(
    WalletApiOperation.CheckPayForTemplate,
    {
      talerPayTemplateUri,
    },
  );

  t.assertDeepEqual(
    checkPayTemplateResult.templateDetails.template_contract.summary,
    mySummary,
  );

  // Request a template payment

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForTemplate,
    {
      talerPayTemplateUri,
      templateParams: {
        amount: "TESTKUDOS:1",
      },
    },
  );

  console.log(preparePayResult);

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  // Pay for it

  const r2 = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: preparePayResult.transactionId,
  });

  t.assertTrue(r2.type === ConfirmPayResultType.Done);

  // Check if payment was successful.

  const orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: preparePayResult.contractTerms.order_id,
    instance: "default",
  });

  t.assertTrue(orderStatus.order_status === "paid");
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runPaymentTemplateTest.suites = ["wallet"];
