/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  NotificationType,
  TalerCorebankApiClient,
  TransactionMajorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import { createSimpleTestkudosEnvironmentV3 } from "../harness/helpers.js";

/**
 * Test behavior when an order is deleted while the wallet is paying for it.
 */
export async function runWalletBalanceNotificationsTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, walletService } =
    await createSimpleTestkudosEnvironmentV3(t);

  const amount = "TESTKUDOS:20";

  const user = await bankClient.createRandomBankUser();
  bankClient.setAuth({
    username: user.username,
    password: user.password,
  });

  const wop = await bankClient.createWithdrawalOperation(user.username, amount);

  // Hand it to the wallet

  await walletClient.call(WalletApiOperation.GetWithdrawalDetailsForUri, {
    talerWithdrawUri: wop.taler_withdraw_uri,
  });

  // Withdraw (AKA select)

  const balanceChangedNotif1 = walletClient.waitForNotificationCond(
    (x) => x.type === NotificationType.BalanceChange,
  );

  const acceptRes = await walletClient.call(
    WalletApiOperation.AcceptBankIntegratedWithdrawal,
    {
      exchangeBaseUrl: exchange.baseUrl,
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  t.logStep("wait-balance-notif-1");
  await balanceChangedNotif1;
  t.logStep("done-wait-balance-notif-1");

  const withdrawalFinishedCond = walletClient.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.newTxState.major === TransactionMajorState.Done &&
      x.transactionId === acceptRes.transactionId,
  );

  // Confirm it

  await bankClient.confirmWithdrawalOperation(user.username, {
    withdrawalOperationId: wop.withdrawal_id,
  });

  await withdrawalFinishedCond;

  // Second withdrawal!
  {
    const wop2 = await bankClient.createWithdrawalOperation(
      user.username,
      "TESTKUDOS:5",
    );

    await walletClient.call(WalletApiOperation.GetWithdrawalDetailsForUri, {
      talerWithdrawUri: wop2.taler_withdraw_uri,
    });

    const acceptRes = await walletClient.call(
      WalletApiOperation.AcceptBankIntegratedWithdrawal,
      {
        exchangeBaseUrl: exchange.baseUrl,
        talerWithdrawUri: wop2.taler_withdraw_uri,
      },
    );

    const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
    t.assertAmountEquals(bal.balances[0].available, "TESTKUDOS:19.53");
    t.assertAmountEquals(bal.balances[0].pendingIncoming, "TESTKUDOS:4.85");

    await walletService.stop();
  }
}

runWalletBalanceNotificationsTest.suites = ["wallet"];
