/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TalerCorebankApiClient,
  TransactionIdStr,
  TransactionMajorState,
  TransactionMinorState,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
} from "../harness/helpers.js";

/**
 * Test handing over a withdrawal to another wallet.
 */
export async function runWithdrawalHandoverTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t);

  // Do one normal withdrawal with the new split API
  {
    // Create a withdrawal operation

    const user = await bankClient.createRandomBankUser();
    const userBankClient = new TalerCorebankApiClient(bankClient.baseUrl);
    userBankClient.setAuth(user);
    const amount = "TESTKUDOS:10";
    const wop = await userBankClient.createWithdrawalOperation(
      user.username,
      amount,
    );

    const checkResp = await walletClient.call(
      WalletApiOperation.GetWithdrawalDetailsForUri,
      {
        talerWithdrawUri: wop.taler_withdraw_uri,
      },
    );

    t.assertTrue(!!checkResp.defaultExchangeBaseUrl);

    const prepareResp = await walletClient.call(
      WalletApiOperation.PrepareBankIntegratedWithdrawal,
      {
        // exchangeBaseUrl: checkResp.defaultExchangeBaseUrl,
        talerWithdrawUri: wop.taler_withdraw_uri,
      },
    );

    console.log(`prepareResp: ${j2s(prepareResp)}`);

    t.assertTrue(!!prepareResp.transactionId);

    const txns1 = await walletClient.call(WalletApiOperation.GetTransactions, {
      sort: "stable-ascending",
    });
    console.log(j2s(txns1));

    await walletClient.call(WalletApiOperation.ConfirmWithdrawal, {
      transactionId: prepareResp.transactionId,
      amount,
      exchangeBaseUrl: checkResp.defaultExchangeBaseUrl,
    });

    await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: prepareResp.transactionId as TransactionIdStr,
      txState: {
        major: TransactionMajorState.Pending,
        minor: TransactionMinorState.BankConfirmTransfer,
      },
    });

    await userBankClient.confirmWithdrawalOperation(user.username, {
      withdrawalOperationId: wop.withdrawal_id,
    });

    await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: prepareResp.transactionId as TransactionIdStr,
      txState: {
        major: TransactionMajorState.Done,
      },
    });
  }

  // Do one another withdrawal with handover.
  {
    t.logStep("start-subtest-handover");

    const w2 = await createWalletDaemonWithClient(t, {
      name: "w2",
    });

    // Create a withdrawal operation

    const user = await bankClient.createRandomBankUser();
    const userBankClient = new TalerCorebankApiClient(bankClient.baseUrl);
    userBankClient.setAuth(user);
    const amount = "TESTKUDOS:10";

    const wop = await userBankClient.createWithdrawalOperation(
      user.username,
      amount,
    );

    const checkResp = await walletClient.call(
      WalletApiOperation.GetWithdrawalDetailsForUri,
      {
        talerWithdrawUri: wop.taler_withdraw_uri,
      },
    );

    t.assertTrue(!!checkResp.defaultExchangeBaseUrl);

    const prepareRespW1 = await walletClient.call(
      WalletApiOperation.PrepareBankIntegratedWithdrawal,
      {
        // exchangeBaseUrl: checkResp.defaultExchangeBaseUrl,
        talerWithdrawUri: wop.taler_withdraw_uri,
      },
    );

    const prepareRespW2 = await w2.walletClient.call(
      WalletApiOperation.PrepareBankIntegratedWithdrawal,
      {
        // exchangeBaseUrl: checkResp.defaultExchangeBaseUrl,
        talerWithdrawUri: wop.taler_withdraw_uri,
      },
    );

    t.assertTrue(!!prepareRespW2.transactionId);

    await w2.walletClient.call(WalletApiOperation.ConfirmWithdrawal, {
      transactionId: prepareRespW2.transactionId,
      amount,
      exchangeBaseUrl: checkResp.defaultExchangeBaseUrl,
    });

    await w2.walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: prepareRespW2.transactionId as TransactionIdStr,
      txState: {
        major: TransactionMajorState.Pending,
        minor: TransactionMinorState.BankConfirmTransfer,
      },
    });

    await userBankClient.confirmWithdrawalOperation(user.username, {
      withdrawalOperationId: wop.withdrawal_id,
    });

    console.log(`wopid is ${wop.withdrawal_id}`);

    t.logStep("start-wait-w2-done");
    await w2.walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: prepareRespW2.transactionId as TransactionIdStr,
      txState: {
        major: TransactionMajorState.Done,
      },
    });
    t.logStep("done-wait-w2-done");

    t.logStep("start-wait-w1-done");

    await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: prepareRespW1.transactionId as TransactionIdStr,
      txState: {
        major: TransactionMajorState.Aborted,
        minor: TransactionMinorState.CompletedByOtherWallet,
      },
    });

    t.logStep("done-wait-w1-done");
  }
}

runWithdrawalHandoverTest.suites = ["wallet"];
