;;  -*-  indent-tabs-mode:nil; coding: utf-8 -*-
;;  Copyright (C) 2013,2014,2015
;;      "Mu Lei" known as "NalaGinrut" <NalaGinrut@gmail.com>
;;  Artanis is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License and GNU
;;  Lesser General Public License published by the Free Software
;;  Foundation, either version 3 of the License, or (at your option)
;;  any later version.

;;  Artanis is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License and GNU Lesser General Public License
;;  for more details.

;;  You should have received a copy of the GNU General Public License
;;  and GNU Lesser General Public License along with this program.
;;  If not, see <http://www.gnu.org/licenses/>.

(define-module (artanis utils)
  #:use-module (artanis crypto md5)
  #:use-module (artanis crypto sha-1)
  #:use-module (artanis crypto base64)
  #:use-module (artanis tpl sxml)
  #:use-module (artanis config)
  #:use-module (artanis irregex)
  #:use-module (artanis env)
  #:use-module (artanis mime)
  #:use-module (system foreign)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 regex)
  #:use-module (ice-9 match)
  #:use-module (ice-9 format)
  #:use-module (ice-9 ftw)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-19)
  #:use-module (ice-9 local-eval)
  #:use-module (ice-9 receive)
  #:use-module (ice-9 q)
  #:use-module (web http)
  #:use-module (web request)
  #:use-module ((rnrs)
                #:select (get-bytevector-all utf8->string put-bytevector
                          bytevector-u8-ref string->utf8 bytevector-length
                          make-bytevector bytevector-s32-native-ref))
  #:export (regexp-split hash-keys cat bv-cat get-global-time
            get-local-time string->md5 unsafe-random string-substitute
            get-file-ext get-global-date get-local-date uri-decode
            nfx static-filename remote-info seconds-now local-time-stamp
            parse-date write-date make-expires export-all-from-module!
            alist->hashtable expires->time-utc local-eval-string
            time-expired? valid-method? mmap munmap get-random-from-dev
            string->byteslist string->sha-1 list-slice bv-slice uni-basename
            checkout-the-path make-string-template guess-mime prepare-headers
            new-stack new-queue stack-slots queue-slots stack-pop! stack-push!
            stack-top stack-empty? queue-out! queue-in! queue-head queue-tail
            queue-empty? list->stack list->queue stack-remove! queue-remove!
            queue->list stack->list queue-length stack-length
            plist->alist make-db-string-template non-list?
            keyword->string range oah->handler oah->opts string->keyword
            alist->klist alist->kblist is-hash-table-empty?
            symbol-downcase symbol-upcase normalize-column run-before-run!
            sxml->xml-string run-after-request! run-before-response!
            make-pipeline HTML-entities-replace eliminate-evil-HTML-entities
            generate-kv-from-post-qstr handle-proper-owner
            generate-data-url verify-ENTRY
            draw-expander remove-ext scan-app-components cache-this-route!
            dump-route-from-cache generate-modify-time delete-directory
            handle-existing-file check-drawing-method
            subbv->string subbv=? bv-read-line bv-read-delimited put-bv
            bv-u8-index bv-u8-index-right build-bv-lookup-table filesize
            plist-remove gen-migrate-module-name try-to-load-migrate-cache
            flush-to-migration-cache gen-local-conf-file with-dbd errno
            c/struct-sizeof)
  #:re-export (the-environment))

;; There's a famous rumor that 'urandom' is safer, so we pick it.
(define* (get-random-from-dev #:key (length 8) (uppercase #f))
  (call-with-input-file "/dev/urandom" 
    (lambda (port)  
      (let* ((bv ((@ (rnrs) get-bytevector-n) port length))
             (str (format #f "~{~2,'0x~}" ((@ (rnrs) bytevector->u8-list) bv))))
        (if uppercase
            (string-upcase str)
            str)))))

(define uri-decode (@ (web uri) uri-decode))
(define parse-date (@@ (web http) parse-date))
(define write-date (@@ (web http) write-date))

(define-syntax-rule (local-eval-string str e)
  (local-eval 
   (call-with-input-string (format #f "(begin ~a)" str) read)
   e))

(define (alist->hashtable al)
  (let ((ht (make-hash-table)))
    (for-each (lambda (x)
                (hash-set! ht (car x) (cadr x)))
              al)
    ht))

(eval-when (eval load compile)
           (define (export-all-from-module! module-name)
             (let ((mod (resolve-module module-name)))
               (module-for-each (lambda (s m) 
                                  (module-add! (current-module) s m)) mod))))

(define (time-expired? expires)
  (if expires
      (let ((t (expires->time-utc expires)))
        (time>? (current-time) t))
      #t)) ;; no expired, means session cookie, which is always expired

(define (expires->time-utc str)
  (date->time-utc (parse-date str)))

(define (make-expires sec)
  (get-local-time (+ (seconds-now) sec)))

(define (seconds-now)
  ((@ (guile) current-time)))

;; This function only used for local logger
(define (local-time-stamp)
  (strftime "%F %T" (localtime (seconds-now))))

;; default time is #f, get current time
(define* (get-global-time #:optional (time #f) (nsec 0))
  (call-with-output-string 
   (lambda (port)
     ;; NOTE: (time-utc->data t 0) to get global time.
     (write-date 
      (time-utc->date 
       (if time (make-time 'time-utc nsec time) (current-time))
       0)
      port))))

;; default time is #f, get current time
(define* (get-local-time #:optional (time #f) (nsec 0))
  (call-with-output-string 
   (lambda (port)
     ;; NOTE: (time-utc->data t) to get local time.
     (write-date 
      (time-utc->date 
       (if time (make-time 'time-utc nsec time) (current-time)))
      port))))

(define* (regexp-split regex str #:optional (flags 0))
  (let ((ret (fold-matches 
              regex str (list '() 0 str)
              (lambda (m prev)
                (let* ((ll (car prev))
                       (start (cadr prev))
                       (tail (match:suffix m))
                       (end (match:start m))
                       (s (substring/shared str start end))
                       (groups (map (lambda (n) (match:substring m n))
                                    (iota (1- (match:count m)) 1))))
                  (list `(,@ll ,s ,@groups) (match:end m) tail)))
              flags)))
    `(,@(car ret) ,(caddr ret))))

(define (hash-keys ht)
  (hash-map->list (lambda (k v) k) ht))

;; WARN: besure that you've already checked the file exists before!!!
(define* (cat file/port #:optional (out (current-output-port)))
  (define get-string-all (@ (rnrs io ports) get-string-all))
  (let ((str (if (port? file/port)
                 (get-string-all file/port)
                 (call-with-input-file file/port get-string-all))))
    (if out
        (display str out)
        str)))

;; WARN: besure that you've already checked the file exists before!!!
(define* (bv-cat file/port #:optional (out (current-output-port)))
  (define get-bytevector-all (@ (rnrs io ports) get-bytevector-all))
  (let ((bv (if (port? file/port)
                (get-bytevector-all file/port)
                (call-with-input-file file/port get-bytevector-all))))
    (if out
        (display bv out)
        bv)))

(define (string->md5 str)
  (call-with-input-string str md5))

;; 35147 is the length of GPLv3 in bytes
(define* (unsafe-random #:optional (n 35147))
  (random n (random-state-from-platform)))

(define (string-substitute str re what)
  (regexp-substitute/global #f re str 'pre what 'post))

(define-syntax get-file-ext               
  (syntax-rules ()
    ((_ filename)
     (substring/shared filename
                       (1+ (string-index-right filename #\.))))))

(define* (get-global-date #:optional (time #f))
  (parse-header 'date 
                (if time
                    (get-global-time (car time) (cdr time)) 
                    (get-global-time))))

(define* (get-local-date #:optional (time #f))
  (parse-header 'date 
                (if time
                    (get-local-time (car time) (cdr time)) 
                    (get-local-time))))

(define (nfx exp)   
  (let lp((rest exp) (result '()) (cur #f))
    (cond 
     ((null? rest) result)
     ((null? result)
      (let ((e (list (cadr rest) (car rest) (caddr rest)))) 
        (lp (cdddr rest) e (car rest))))
     (else
      (let ((e (list cur result (cadr rest)))) 
        (lp (cddr rest) e #f))))))

(define-syntax-rule (static-filename path)
  (substring/shared path 1))

(define-syntax-rule (request-ip req)
  ;; TODO: support AF_INET6 in the future
  (if (port-filename (request-port req))
      ;; Valid socket port
      (inet-ntop AF_INET (sockaddr:addr (getpeername (request-port req))))
      "localtest")) ; fake hostname for testing

(define-syntax-rule (remote-info req)
  (if (get-conf '(server nginx))
      (assoc-ref (request-headers req) 'x-real-ip)
      (request-ip req)))

(define *methods-list* '(HEAD GET POST PUT PATCH DELETE))
(define (allowed-method? method)
  ;; TODO: check allowed method from config
  #t)
(define (valid-method? method)
  (if (and (member method *methods-list*) (allowed-method? method))
      method
      (throw 'artanis-err 405 "invalid HTTP method" method)))

;; -------------- mmap ---------------------
(define-public ACCESS_COPY              #x3)
(define-public ACCESS_READ              #x1)
(define-public ACCESS_WRITE             #x2)
(define-public ALLOCATIONGRANULARITY #x1000)

(define-public PROT_READ       #x1)       
(define-public PROT_WRITE      #x2)       
(define-public PROT_EXEC       #x4)       
(define-public PROT_SEM        #x8)       
(define-public PROT_NONE       #x0)       
(define-public PROT_GROWSDOWN  #x01000000)
(define-public PROT_GROWSUP    #x02000000)

(define-public PAGESIZE       #x1000)
(define-public MAP_ANON         #x20)
(define-public MAP_DENYWRITE   #x800)
(define-public MAP_EXECUTABLE #x1000)
(define-public MAP_SHARED       #x01)
(define-public MAP_PRIVATE      #x02)
(define-public MAP_TYPE         #x0f)
(define-public MAP_FIXED        #x10)
(define-public MAP_ANONYMOUS    #x20)
(define-public MAP_UNINITIALIZED 0) ;; don't support map uninitialized

(define *libc-ffi* (dynamic-link))
(define %mmap
  (pointer->procedure '*
                      (dynamic-func "mmap" *libc-ffi*)
                      (list '* size_t int int int size_t)))
(define %munmap
  (pointer->procedure int
                      (dynamic-func "munmap" *libc-ffi*)
                      (list '* size_t)))
(define* (mmap size #:key (addr %null-pointer) (fd -1) (prot MAP_SHARED) 
               (flags PROT_READ) (offset 0))
  (pointer->bytevector (%mmap addr size prot flags fd offset) size))
(define (munmap bv size)
  (%munmap (bytevector->pointer bv size) size))

;; FIXME: what if len is not even?
(define (string->byteslist str step base)
  (define len (string-length str))
  (let lp((ret '()) (i 0)) 
    (cond 
     ((>= i len) (reverse ret))
     ((zero? (modulo i step)) 
      (lp (cons (string->number (substring/shared str i (+ i step)) base) ret) (1+ i))) 
     (else (lp ret (1+ i))))))

(define (string->sha-1 str/bv)
  (let ((in (cond
             ((string? str/bv)
              ((@ (rnrs) string->utf8) str/bv))
             (((@ (rnrs) bytevector?) str/bv)
              str/bv)
             (else (error "need string or bytevector!" str/bv)))))
    (sha-1->string (sha-1 in))))

(define-syntax list-slice
  (syntax-rules (:)
    ((_ ll lo : hi)
     (let ((len (length ll)))
       (and (<= lo len) (>= len hi)
	    (let lp((rest ll) (result '()) (cnt 1))
	      (cond
	       ((null? rest) (error "no"))
	       ((<= cnt lo) (lp (cdr rest) result (1+ cnt)))
	       ((> cnt hi) (reverse result))
	       (else (lp (cdr rest) (cons (car rest) result) (1+ cnt))))))))
    ((_ ll lo :)
     (drop ll lo))
    ((_ ll : hi)
     (take ll hi))))

;; TODO: 
;; 1. (> hi (bytevector-length bv))
;; 2. (< lo 0) wrap reference
(define (%bv-slice bv lo hi) 
  (let* ((len (- hi lo)) 
         (slice ((@ (rnrs) make-bytevector) len)))
    ((@ (rnrs) bytevector-copy!) bv lo slice 0 len) slice))

;; NOT SAFE %bytevector-slice for GC, need 
;;(define (%bytevector-slice bv lo hi)
;;  (and (< hi lo) (error %bytevector-slice "wrong range" lo hi))
;;  (let* ((ptr (bytevector->pointer bv))
;;         (addr (pointer-address ptr))
;;        (la (+ addr lo))
;;         (len (- hi lo)))
;;    (pointer->bytevector (make-pointer la) len)))

(define-syntax bv-slice
  (syntax-rules (:)
    ((_ bv lo : hi)
     (%bv-slice bv lo hi))
    ((_ bv lo :)
     (%bv-slice bv lo ((@ (rnrs bytevectors) bytevector-length) bv)))
    ((_ bv : hi)
     (%bv-slice bv 0 hi))))

;; get the unified basename both POSIX and WINDOWS
(define (uni-basename filename)
  (substring filename
             (1+ 
              (string-index-right filename 
                                  (lambda (c) (or (char=? c #\\) (char=? c #\/)))))))

;; FIXME: checkout-the-path only support POSIX file path
;; FIXME: what's the proper default mode for the dir?
(define* (checkout-the-path path #:optional (mode #o775))
  (define (->path p)
    (let ((pp (irregex-split "/" p)))
      (if (char=? (string-ref p 0) #\/)
          (cons (string-append "/" (car pp)) (cdr pp))
          pp)))
  (let ((paths (->path path)))
    (let lp((next paths) (last ""))
      (cond
       ((null? next) #t)
       ((string-null? (car next)) (lp (cdr next) last))
       (else 
        (let ((now-path (string-append last (car next) "/")))
          (cond
           ((file-exists? now-path)
            (lp (cdr next) now-path))
           (else
            (mkdir now-path mode)
            (lp (cdr next) now-path)))))))))

;; NOTE: This my original verion of make-string-template

;; (define *stpl-SRE* '(or (=> tilde "~")
;;                         (=> dollar "$$")
;;                         (: "${" (=> name (+ (~ #\}))) "}")))

;; (define* (make-string-template str #:optional (mode #f) . opts)
;;   (define ll '()) ; list for all keywords
;;   (define lv '()) ; list for default value
;;   (define template
;;     (irregex-replace/all 
;;      ;;"(\\$\\{([^$])+\\})"
;;      *stpl-SRE* str
;;      (lambda (m) 
;;        (cond
;;         ((irregex-match-substring m 'dollar) "$")
;;         ((irregex-match-substring m 'tilde) "~~")
;;         (else
;;          (let* ((var (irregex-match-substring m 1))
;;                 (key (symbol->keyword (string->symbol 
;;                                        (irregex-match-substring m 'name))))
;;                 (v (kw-arg-ref opts key)))
;;            (and v (set! lv (cons (cons key v) lv))) ; default value
;;            (set! ll (cons key ll))
;;            (set! lk (cons var lk))
;;            "~a"))))))
;;   (lambda args
;;     (let ((vals (map (lambda (x) 
;;                        (or (kw-arg-ref args x) (assoc-ref lv x)
;;                            (if mode (assoc-ref lk x) "NONE"))) ll)))
;;     (format #f "~?" template (reverse vals)))))

;; NOTE: This is mark_weaver version for efficiency, Thanks mark!
(define (%make-string-template mode template . defaults)
  (define irx (sre->irregex '(or (=> dollar "$$")
                                 (: "${" (=> var (+ (~ #\}))) "}"))))
  (define (->string obj)
    (if (string? obj) obj (object->string obj)))
  (define (get-the-val lst key)
    (let ((str (kw-arg-ref lst key)))
      (case mode
        ((normal) str)
        ((db) (string-concatenate (list "\"" (->string str) "\"")))
        (else (throw 'artanis-err 500 "%make-string-template: invalid mode" mode)))))
  (define (optimize rev-items tail)
    (cond ((null? rev-items) tail)
          ((not (string? (car rev-items)))
           (optimize (cdr rev-items)
                     (cons (car rev-items) tail)))
          (else (receive (strings rest) (span string? rev-items)
                         (let ((s (string-concatenate-reverse strings)))
                           (if (string-null? s)
                               (optimize rest tail)
                               (optimize rest (cons s tail))))))))
  (define (match->item m)
    (or (and (irregex-match-substring m 'dollar) "$")
        (let* ((name (irregex-match-substring m 'var))
               (key (symbol->keyword (string->symbol name))))
          (cons key (kw-arg-ref defaults key)))))
  (let* ((rev-items (irregex-fold
                     irx
                     (lambda (idx m tail)
                       (cons* (match->item m)
                              (substring template
                                         idx
                                         (irregex-match-start-index m 0))
                              tail))
                     '()
                     template
                     (lambda (idx tail)
                       (cons (substring template idx) tail))))
         (items (optimize rev-items '())))
    (lambda keyword-args
      (define (item->string item)
        (if (string? item)
            item
            (or (and=> (get-the-val keyword-args (car item)) ->string)
                (cdr item) ; default value
                (throw 'artanis-err 500
                       "(utils)item->string: Missing keyword" (car item)))))
      (string-concatenate (map item->string items)))))

;; the normal mode, no double quotes for vals
(define (make-string-template tpl . vals)
  (apply %make-string-template 'normal tpl vals))

;; DB str tpl will treat all values with double quotes, for SQL
(define (make-db-string-template tpl . vals)
  (apply %make-string-template 'db tpl vals))

(define (guess-mime filename)
  (mime-guess (get-file-ext filename)))

(define (bytevector-null? bv)
  ((@ (rnrs bytevectors) bytevector=?) bv #u8()))

(define (generate-modify-time t)
  (get-local-date (cons (time-second t) (time-nanosecond t))))

(define (prepare-headers headers)
  (define *default-headers*
    `((content-type . (text/html (charset . ,(get-conf '(server charset)))))
      (date . ,(get-global-date))))
  (lset-union (lambda (x y) (eq? (car x) (car y)))
              (assq-remove! headers 'last-modified) *default-headers*))

(define new-stack make-q)
(define new-queue make-q)
(define stack-slots car)
(define queue-slots car)

(define (%q-remove-with-key! q key)
  (assoc-remove! (car q) key)
  (sync-q! q))

(define stack-pop! q-pop!)
(define stack-push! q-push!)
(define stack-top q-front)
(define stack-remove! %q-remove-with-key!)
(define stack-empty? q-empty?) 
(define stack-length q-length)
(define (stack->list stk) (list-copy (stack-slots stk)))

(define queue-out! q-pop!)
(define queue-in! enq!)
(define queue-head q-front)
(define queue-tail q-rear)
(define queue-remove! %q-remove-with-key!)
(define queue-empty? q-empty?)
(define queue-length q-length)
(define (queue->list q) (list-copy (queue-slots q)))

(define* (list->stack lst #:optional (stk (new-stack))) ; NOTE: make-stack exists in Guile
  (for-each (lambda (x) (stack-push! stk x)) lst)
  stk)

(define* (list->queue lst #:optional (queue (new-queue)))
  (for-each (lambda (x) (queue-in! queue x)) lst)
  queue)

;; NOTE: keyword could be the value, so this version is correct.
(define (plist->alist lst)
  (let lp((next lst) (ret '()))
    (match next
      (() (reverse ret))
      ((k v . rest) (lp (cddr next) (acons (keyword->symbol k) v ret))))))

(define-syntax-rule (non-list? x) (not (list? x)))
(define* (keyword->string x #:optional (proc identity))
  (proc (symbol->string (keyword->symbol x))))

(define* (range from to #:optional (step 1))
  (iota (- to from) from step))

;; NOTE: handler must be the last element of the list, it's should be error
;;       if it's not so.
(define (oah->handler opts-and-handler)
  (let ((handler (and (list? opts-and-handler) (last opts-and-handler))))
    (if (or (procedure? handler) (string? handler))
        handler
        (error oah->handler "You have to specify a handler for this rule!"))))

;; get all kw-args from the middle of args
(define (oah->opts opts-and-handler)
  (if (procedure? opts-and-handler)
      '() ; there's no opts
      (let lp((next opts-and-handler) (kl '()) (vl '()))
        (match next
          (((? keyword? k) v rest ...)
           (lp rest (cons k kl) (cons v vl)))
          ((or (? null?) (? procedure?))
           ;; no opts left, return the result
           (list kl vl))
          (else (lp (cdr next) kl vl))))))

(define (string->keyword str)
  (symbol->keyword (string->symbol str)))

(define (alist->klist al)
  (let lp((next al) (ret '()))
    (cond
     ((null? next) ret)
     (else
      (let ((k (symbol->keyword (car (car next))))
            (v (cdr (car next))))
        (lp (cdr next) (cons k (cons v ret))))))))

(define (alist->kblist al)
  (let lp((next al) (ret '()))
    (cond
     ((null? next) ret)
     (else
      (let ((k (string->keyword (string-append ":" (car (car next)))))
            (v (cdr (car next))))
        (lp (cdr next) (cons k (cons v ret))))))))

(define (is-hash-table-empty? ht)
  (zero? (hash-count values ht)))

(define (symbol-strop proc sym)
  (string->symbol (proc (symbol->string sym))))

(define (symbol-downcase sym)
  (symbol-strop string-downcase sym))

(define (symbol-upcase sym)
  (symbol-strop string-upcase sym))

(define* (normalize-column col #:optional (ci? #f))
  (define-syntax-rule (-> c p) (if ci? (p col) col))
  (cond
   ((string? col) (string->symbol (-> c string-downcase)))
   ((symbol? col) (-> col symbol-downcase))
   ((keyword? col) (normalize-column (keyword->string col) ci?))
   (else (throw 'artanis-err 500 "normalize-column: Invalid type of column" col))))

(define* (sxml->xml-string sxml #:key (escape? #f))
  (call-with-output-string
   (lambda (port)
     (sxml->xml sxml port escape?))))

(define (run-after-request! proc)
  (add-hook! *after-request-hook* proc))

(define (run-before-response! proc)
  (add-hook! *before-response-hook* proc))

(define (run-before-run! proc)
  (add-hook! *before-run-hook* proc))

;; NOTE: For `pipeline' methodology, please read my post:
;; http://nalaginrut.com/archives/2014/04/25/oba-pipeline-style%21
(define (make-pipeline . procs)
  (lambda (x) (fold (lambda (y p) (y p)) x procs)))

(define (HTML-entities-replace set content)
  (define in (open-input-string content))
  (define (hit? c/str) (assoc-ref set c/str))
  (define (get-estr port)
    (let lp((n 0) (ret '()))
      (cond
       ((= n 3) (list->string (reverse! ret)))
       (else (lp (1+ n) (cons (read-char port) ret))))))
  (call-with-output-string
   (lambda (out)
     (let lp((c (peek-char in)))
       (cond
        ((eof-object? c) #t)
        ((hit? c)
         => (lambda (str)
              (display str out)
              (read-char in)
              (lp (peek-char in))))
        ((char=? c #\%)
         (let* ((s (get-estr in))
                (e (hit? s)))
           (if e
               (display e out)
               (display s out))
           (lp (peek-char in))))
        (else
         (display (read-char in) out)
         (lp (peek-char in))))))))

(define *terrible-HTML-entities*
  '((#\< . "&lt;") (#\> . "&gt;") (#\& . "&amp;") (#\" . "&quot;")
    ("%3C" . "&lt;") ("%3E" . "&gt;") ("%26" . "&amp;") ("%22" . "&quot;")))
;; NOTE: cooked for anti-XSS.
(define (eliminate-evil-HTML-entities content)
  (HTML-entities-replace *terrible-HTML-entities* content))

(define* (generate-kv-from-post-qstr body #:key (no-evil? #f)
                                     (key-converter identity))
  (define cook (if no-evil? eliminate-evil-HTML-entities identity))
  (define (%convert lst)
    (match lst
      ((k v) (list (key-converter k) v))
      (else (throw 'artanis-err 500 "generate-kv-from-post-qstr: Fatal! Can't be here!" lst))))
  (define (-> x)
    (string-trim-both x (lambda (c) (member c '(#\sp #\: #\return)))))
  (map (lambda (x)
         (%convert (map -> (string-split (cook x) #\=))))
       (string-split (utf8->string body) #\&)))

;; NOTE: We accept warnings, which means if warnings occurred, it'll be 200(OK) anyway, but
;;       Artanis will throw warnings in the server-side log.
;; NOTE: We *DO NOT* accept errors, which means if errors occurred, Artanis will throw 500.
(define (handle-proper-owner file uid gid)
  (define-syntax-rule (print-the-warning exe reason)
    (format (current-error-port) "[WARNING] '~a' encountered system error: ~s~%" exe reason))
  (define-syntax-rule (->err-reason exe reason)
    (format #f "'~a' encoutered system error: ~s" exe reason))
  (catch 'system-error
         (lambda ()
           (chown file (or uid (getuid)) (or gid (getgid))))
         (lambda (k . e)
           (let ((exe (car e))
                 (reason (caaddr e)))
             (match (cons k reason)
               ('(system-error . "Operation not permitted")
                (print-the-warning exe reason)
                (display
                 "Maybe you run Artanis as unprivileged user? (say, not as root)\n"
                 (current-error-port)))
               ('(system-error . "No such file or directory")
                (throw 'artanis-err 500 (->err-reason exe reason) file))
               (else (apply throw k e)))))))

;; According to wiki, here's the standard format of data_url_scheme:
;; data:[<MIME-type>][;charset=<encoding>][;base64],<data>
(define* (generate-data-url bv/str #:key (mime 'application/octet-stream)
                            (crypto 'base64) (charset 'utf-8))
  (define-syntax-rule (->crypto)
    (match crypto
      ((or 'base64 "base64") ";base64")
      (else "")))
  (define-syntax-rule (->charset)
    (if (or (string? charset) (symbol? charset))
        (format #f ",charset=~a" charset)
        ""))
  (define-syntax-rule (->mime)
    (match mime
      (`(guess ,ext) (or (mime-guess ext) 'application/octet-stream))
      ((or (? symbol?) (? string?))
       (or (and (get-conf 'debug-mode) (mime-check mime) (format #f "~a" mime))
           (format #f "~a" mime)))
      (else (throw 'artanis-err 500
                   "generate-data-url: Invalid MIME! Should be symbol or string"
                   mime))))
  (let ((b64 (base64-encode bv/str)))
    (string-concatenate (list "data:" (->mime) (->crypto) (->charset) "," b64))))

(define (verify-ENTRY entry)
  (cond
   ((not (file-exists? entry)) #f)
   (else
    (let* ((line (call-with-input-file entry read-line))
           (m (string-match "Artanis top-level: (.*)" line)))
      (and m (string=? (match:substring m 1) (dirname entry)))))))

(define-syntax draw-expander
  (syntax-rules (rule options method)
    ((_ (options options* ...) rest rest* ...)
     `(,@(list options* ...) ,@(draw-expander rest rest* ...)))
    ((_ (method method*) rest rest* ...)
     `((method ,'method*) ,@(draw-expander rest rest* ...)))
    ((_ (rule url) rest rest* ...)
     `((rule ,url) ,@(draw-expander rest rest* ...)))
    ((_ handler) (list handler))))

(define (remove-ext str)
  (let ((i (string-contains str ".")))
    (substring str 0 i)))

(define (scan-app-components component)
  (let ((toplevel (current-toplevel)))
    (map (lambda (f) (string->symbol (remove-ext f)))
         (scandir (format #f "~a/app/~a/" toplevel component)
                  (lambda (f)
                    (not (or (string=? f ".")
                             (string=? f ".."))))))))

(define (cache-this-route! url meta)
  (define (write-header port)
    (format port ";; Do not touch anything!!!~%")
    (format port ";; All things here should be automatically handled properly!!!~%"))
  (define route-cache (string-append (current-toplevel) "/tmp/cache/route.cache"))
  (when (or (not (file-exists? route-cache))
            (and (not url) (not meta))) ; for regenerating route cache
    (format (artanis-current-output) "Route cache is missing, regenerating...~%")
    (call-with-output-file route-cache
      (lambda (port) (write-header port) (write '() port))))
  (when (and url meta)
    (let ((rl (call-with-input-file route-cache read)))
      (delete-file route-cache)
      (call-with-output-file route-cache
        (lambda (port)
          (flock port LOCK_EX)
          (write-header port)
          (if (eof-object? rl)
              (write '() port)
              (write (assoc-set! rl url (drop-right meta 1)) port))
          (flock port LOCK_UN))))))

(define (dump-route-from-cache)
  (define toplevel (current-toplevel))
  (define route-cache (string-append toplevel "/tmp/cache/route.cache"))
  (define route (string-append toplevel "/.route"))
  (define (load-customized-router)
    (let ((croute (string-append toplevel "conf/route")))
      (cond
       ((not (file-exists? croute)) #t) ; No customized route
       (else
        (use-modules (artanis mvc route)) ; black magic to make Guile happy
        (load croute)))))
  (when (file-exists? route) (delete-file route))
  (when (not (file-exists? route-cache))
        (cache-this-route! #f #f)
        (dump-route-from-cache))
  (let ((rl (call-with-input-file route-cache read)))
    (cond
     ((eof-object? rl)
      (cache-this-route! #f #f)
      (dump-route-from-cache))
     (else
      (call-with-output-file route
        (lambda (port)
          (for-each (lambda (r)
                      (let* ((meta (cdr r))
                             (rule (assq-ref meta 'rule))
                             (method (assq-ref meta 'method)))
                        (format port "~2t(~a ~s)~%"
                                (if method method 'get)
                                (if rule rule (car r)))))
                    rl)))
      ;; load customized router
      (load-customized-router)))))

(define* (delete-directory dir #:optional (checkonly? #f))
  (cond
   ((and (file-is-directory? dir) (file-exists? dir))
    (system (format #f "rm -f ~a" dir)))
   (else
    (and (not checkonly?)
         (error delete-directory "Not a directory or doesn't exist " dir)))))

;; TODO: handle it more elegantly
(define* (handle-existing-file path #:optional (dir? #f))
  (let* ((pp (if dir? (dirname path) path))
         (component (basename (dirname pp)))
         (name (car (string-split (basename pp) #\.))))
    (cond
     ((draw:is-force?)
      (if (file-is-directory? path)
          (delete-directory path)
          (delete-file path)))
     ((draw:is-skip?)
      (format (artanis-current-output) "skip ~10t app/~a/~a~%" component name))
   (else
    (format (artanis-current-output)
            "~a `~a' exists! (Use --force/-f to overwrite or --skip/-s to ignore)~%"
            (string-capitalize component) name)
    (exit 1)))))

;; Check if all methods are valid
(define (check-drawing-method lst)
  (define errstr "Invalid drawing method, shouldn't contain '/' ")
  (for-each (lambda (name)
              (when (not (irregex-match "[^/]+" name))
                    (error check-drawing-method errstr name)))
            lst)
  lst)

(define (subbv->string bv encoding start end)
  (call-with-output-string
   (lambda (port)
     (set-port-encoding! port encoding)
     (put-bytevector port bv start (- end start)))))

(define* (bv-u8-index bv u8 #:optional (time 1))
  (let ((len (bytevector-length bv)))
    (let lp((i 0) (t 1))
      (cond
       ((>= i len) #f)
       ((= (bytevector-u8-ref bv i) u8)
        (if (= t time) i (lp (1+ i) (1+ t))))
       (else (lp (1+ i) t))))))

(define* (bv-u8-index-right bv u8 #:optional (time 1))
  (let ((len (bytevector-length bv)))
    (let lp((i (1- len)) (t 1))
    (cond
     ((< i 0) #f)
     ((= (bytevector-u8-ref bv i) u8)
      (if (= t time) i (lp (1- i) (1+ t))))
     (else (lp (1- i) t))))))

(define* (subbv=? bv bv2 #:optional (start 0) (end (1- (bytevector-length bv))))
  (and (<= (bytevector-length bv2) (bytevector-length bv))
       (let lp((i end) (j (1- (bytevector-length bv2))))
         (cond
          ((< i start) #t)
          ((= (bytevector-u8-ref bv i) (bytevector-u8-ref bv2 j))
           (lp (1- i) (1- j)))
          (else #f)))))

;; return position after delim
(define* (bv-read-delimited bv delim #:optional (start 0) (end (bytevector-length bv)))
  (define len (- end start -1))
  (let lp((i start))
    (cond
     ((> i end) #f)
     ((= (bytevector-u8-ref bv i) delim) i)
     (else (lp (1+ i))))))

;; return position after newline
(define* (bv-read-line bv #:optional (start 0) (end (bytevector-length bv)))
  (bv-read-delimited bv 10 start end))

(define (put-bv port bv from to)
  (put-bytevector port bv from (- to from 2)))

;; TODO: build a char occurence indexing table
(define (build-bv-lookup-table bv)
  (let ((ht (make-hash-table)))
    (for-each (lambda (i)
                (hash-set! ht (bytevector-u8-ref bv i) #t))
              (iota (bytevector-length bv)))
    ht))

(define Gbytes (ash 1 30))
(define Mbytes (ash 1 20))
(define Kbytes (ash 1 10))
(define (filesize size) 
  (cond
   ((>= size Gbytes)
    (format #f "~,1fGiB" (/ size Gbytes)))
   ((>= size Mbytes)
    (format #f "~,1fMiB" (/ size Mbytes)))
   ((>= size Kbytes)
    (format #f "~,1fKiB" (/ size Kbytes)))
   (else (format #f "~a Bytes" size))))

(define* (plist-remove lst k #:optional (no-value? #f))
  (let lp((next lst) (kk '__) (ret '()))
    (cond
     ((null? next) (values (reverse ret) kk))
     ((eq? (car next) k)
      (if no-value?
          (lp (cdr next) (car next) ret)
          (lp (cddr next) (list (car next) (cadr next)) ret)))
     (else (lp (cdr next) kk (cons (car next) ret))))))

(define *name-re* (string->sre "([^.]+)\\.scm"))
(define (gen-migrate-module-name f)
  (cond
   ((irregex-search *name-re* (basename f))
    => (lambda (m) (irregex-match-substring m 1)))
   (else (throw 'artanis-err 500
                "Migrate: wrong parsing of module name, shouldn't be here!" f))))

(define (try-to-load-migrate-cache name)
  (let ((file (format #f "~a/tmp/cache/migration/~a.scm" (current-toplevel) name)))
    (cond
     ((file-exists? file) (load file))
     (else
      (format (artanis-current-output)
              "[WARN] No cache for migration of `~a'~%" name)
      (format (artanis-current-output)
              "Run `art migrate up ~a', then try again!~%" name)))))

(define (flush-to-migration-cache name fl)
  (let ((file (format #f "~a/tmp/cache/migration/~a.scm" (current-toplevel) name)))
    (when (file-exists? file) (delete-file file))
    (call-with-output-file file
      (lambda (port)
        (format port "(define-~a~%" name)
        (for-each
         (lambda (ft) (format port "~2t~a~%" ft))
         fl)
        (format port "~2t)~%")))))

(define (gen-local-conf-file)
  (format #f "~a/conf/artanis.conf" (current-toplevel)))

(define-syntax-rule (with-dbd dbd0 body ...)
  (let ((dbd1 (get-conf '(db dbd))))
    (cond
     ((or (and (list? dbd0) (memq dbd1 dbd0)) (eq? dbd1 dbd0)) body ...)
     (else
      (throw 'artanis-err 500
             (format #f "This is only supported by `~a', but the current dbd is `~a'"
                     dbd0 dbd1)
             'body ...)))))

(define %libc-errno-pointer
  ;; Glibc's 'errno' pointer.
  (let ((errno-loc (dynamic-func "__errno_location" (dynamic-link))))
    (and errno-loc
         (let ((proc (pointer->procedure '* errno-loc '())))
           (proc)))))

(define errno
  (if %libc-errno-pointer
      (let ((bv (pointer->bytevector %libc-errno-pointer (sizeof int))))
        (lambda ()
          "Return the current errno."
          ;; XXX: We assume that nothing changes 'errno' while we're doing all this.
          ;; In particular, that means that no async must be running here.

          ;; Use one of the fixed-size native-ref procedures because they are
          ;; optimized down to a single VM instruction, which reduces the risk
          ;; that we fiddle with 'errno' (needed on Guile 2.0.5, libc 2.11.)
          (let-syntax ((ref (lambda (s)
                              (syntax-case s ()
                                ((_ bv)
                                 (case (sizeof int)
                                   ((4)
                                    #'(bytevector-s32-native-ref bv 0))
                                   ((8)
                                    #'(bytevector-s64-native-ref bv 0))
                                   (else
                                    (error "unsupported 'int' size"
                                           (sizeof int)))))))))
            (ref bv))))
      (lambda () 0)))

(define (c/struct-sizeof meta)
  (apply +
         (map (lambda (m) (if (list? m) (c/struct-sizeof m) (sizeof m)))
              meta)))
