;;; buffer-tests.el --- tests for buffer.c functions -*- lexical-binding: t -*-

;; Copyright (C) 2015-2023 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.

;;; Code:

(require 'ert)
(require 'seq)
(require 'ert-x)
(require 'cl-lib)
(require 'let-alist)

(defun overlay-tests-start-recording-modification-hooks (overlay)
  "Start recording modification hooks on OVERLAY.

Always overwrites the `insert-in-front-hooks',
`modification-hooks' and `insert-behind-hooks' properties.  Any
recorded history from a previous call is erased.

The history is stored in a property on the overlay itself.  Call
`overlay-tests-get-recorded-modification-hooks' to retrieve the
recorded calls conveniently."
  (dolist (hooks-property '(insert-in-front-hooks
                            modification-hooks
                            insert-behind-hooks))
    (overlay-put
     overlay
     hooks-property
     (list (lambda (ov &rest args)
             (should inhibit-modification-hooks)
             (should (eq ov overlay))
             (push (list hooks-property args)
                   (overlay-get overlay
                                'recorded-modification-hook-calls)))))
    (overlay-put overlay 'recorded-modification-hook-calls nil)))

(defun overlay-tests-get-recorded-modification-hooks (overlay)
  "Extract the recorded calls made to modification hooks on OVERLAY.

Must be preceded by a call to
`overlay-tests-start-recording-modification-hooks' on OVERLAY.

Returns a list.  Each element of the list represents a recorded
call to a particular modification hook.

Each call is itself a sub-list where the first element is a
symbol matching the modification hook property (one of
`insert-in-front-hooks', `modification-hooks' or
`insert-behind-hooks') and the second element is the list of
arguments passed to the hook.  The first hook argument, the
overlay itself, is omitted to make test result verification
easier."
  (reverse (overlay-get overlay
                        'recorded-modification-hook-calls)))

(ert-deftest overlay-modification-hooks ()
  "Test the basic functionality of overlay modification hooks.

This exercises hooks registered on the `insert-in-front-hooks',
`modification-hooks' and `insert-behind-hooks' overlay
properties."
    ;; This is a data driven test loop.  Each test case is described
    ;; by an alist.  The test loop initializes a new temporary buffer
    ;; for each case, creates an overlay, registers modification hooks
    ;; on the overlay, modifies the buffer, and then verifies which
    ;; modification hooks (if any) were called for the overlay, as
    ;; well as which arguments were passed to the hooks.
    ;;
    ;; The following keys are available in the alist:
    ;;
    ;; `buffer-text': the initial buffer text of the temporary buffer.
    ;; Defaults to "1234".
    ;;
    ;; `overlay-beg' and `overlay-end': the begin and end positions of
    ;; the overlay under test.  Defaults to 2 and 4 respectively.
    ;;
    ;; `insert-at': move to the given position and insert the string
    ;; "x" into the test case's buffer.
    ;;
    ;; `replace': replace the first occurrence of the given string in
    ;; the test case's buffer with "x".  The test will fail if the
    ;; string is not found.
    ;;
    ;; `expected-calls': a description of the expected buffer
    ;; modification hooks.  See
    ;; `overlay-tests-get-recorded-modification-hooks' for the format.
    ;; May be omitted, in which case the test will insist that no
    ;; modification hooks are called.
    ;;
    ;; The test will fail itself in the degenerate case where no
    ;; buffer modifications are requested.
    (dolist (test-case
             '(
               ;; Remember that the default buffer text is "1234" and
               ;; the default overlay begins at position 2 and ends at
               ;; position 4.  Most of the test cases below assume
               ;; this.

               ;; TODO: (info "(elisp) Special Properties") says this
               ;; about `modification-hooks': "Furthermore, insertion
               ;; will not modify any existing character, so this hook
               ;; will only be run when removing some characters,
               ;; replacing them with others, or changing their
               ;; text-properties."  So, why are modification-hooks
               ;; being called when inserting at position 3 below?
               ((insert-at . 1))
               ((insert-at . 2)
                (expected-calls . ((insert-in-front-hooks (nil 2 2))
                                   (insert-in-front-hooks (t 2 3 0)))))
               ((insert-at . 3)
                (expected-calls . ((modification-hooks (nil 3 3))
                                   (modification-hooks (t 3 4 0)))))
               ((insert-at . 4)
                (expected-calls . ((insert-behind-hooks (nil 4 4))
                                   (insert-behind-hooks (t 4 5 0)))))
               ((insert-at . 5))

               ;; Replacing text never calls `insert-in-front-hooks'
               ;; or `insert-behind-hooks'.  It calls
               ;; `modification-hooks' if the overlay covers any text
               ;; that has changed.
               ((replace . "1"))
               ((replace . "2")
                (expected-calls . ((modification-hooks (nil 2 3))
                                   (modification-hooks (t 2 3 1)))))
               ((replace . "3")
                (expected-calls . ((modification-hooks (nil 3 4))
                                   (modification-hooks (t 3 4 1)))))
               ((replace . "4"))
               ((replace . "4") (overlay-beg . 4)) ;bug#65929
               ((replace . "12")
                (expected-calls . ((modification-hooks (nil 1 3))
                                   (modification-hooks (t 1 2 2)))))
               ((replace . "23")
                (expected-calls . ((modification-hooks (nil 2 4))
                                   (modification-hooks (t 2 3 2)))))
               ((replace . "34")
                (expected-calls . ((modification-hooks (nil 3 5))
                                   (modification-hooks (t 3 4 2)))))
               ((replace . "123")
                (expected-calls . ((modification-hooks (nil 1 4))
                                   (modification-hooks (t 1 2 3)))))
               ((replace . "234")
                (expected-calls . ((modification-hooks (nil 2 5))
                                   (modification-hooks (t 2 3 3)))))
               ((replace . "1234")
                (expected-calls . ((modification-hooks (nil 1 5))
                                   (modification-hooks (t 1 2 4)))))

               ;; Inserting at the position of a zero-length overlay
               ;; calls both `insert-in-front-hooks' and
               ;; `insert-behind-hooks'.
               ((buffer-text . "") (overlay-beg . 1) (overlay-end . 1)
                (insert-at . 1)
                (expected-calls . ((insert-in-front-hooks
                                    (nil 1 1))
                                   (insert-behind-hooks
                                    (nil 1 1))
                                   (insert-in-front-hooks
                                    (t 1 2 0))
                                   (insert-behind-hooks
                                    (t 1 2 0)))))))
      (ert-info ((format "test-case: %S" test-case))
        ;; All three hooks ignore the overlay's `front-advance' and
        ;; `rear-advance' option, so test both ways while expecting the same
        ;; result.
        (dolist (advance '(nil t))
          (ert-info ((format "advance is %S" advance))
            (let-alist test-case
              (with-temp-buffer
                ;; Set up the temporary buffer and overlay as specified by
                ;; the test case.
                (insert (or .buffer-text "1234"))
                (let ((overlay (make-overlay
                                (or .overlay-beg 2)
                                (or .overlay-end 4)
                                nil
                                advance advance)))
                  (overlay-tests-start-recording-modification-hooks overlay)

                  ;; Modify the buffer, possibly inducing calls to the
                  ;; overlay's modification hooks.
                  (should (or .insert-at .replace))
                  (when .insert-at
                    (goto-char .insert-at)
                    (insert "x"))
                  (when .replace
                    (goto-char (point-min))
                    (search-forward .replace)
                    (replace-match "x"))

                  ;; Verify that the expected and actual modification hook
                  ;; calls match.
                  (should (equal
                           .expected-calls
                           (overlay-tests-get-recorded-modification-hooks
                            overlay)))))))))))

(ert-deftest overlay-modification-hooks-message-other-buf ()
  "Test for bug#21824.
After a modification-hook has been run and there is an overlay in
the *Messages* buffer, the message coalescing [2 times] wrongly
runs the modification-hook of the overlay in the 1st buffer, but
with parameters from the *Messages* buffer modification."
  (let ((buf nil)
        (msg-ov nil))
    (with-temp-buffer
      (insert "123")
      (overlay-put (make-overlay 1 3)
                   'modification-hooks
                   (list (lambda (&rest _)
                           (setq buf (current-buffer)))))
      (goto-char 2)
      (insert "x")
      (unwind-protect
          (progn
            (setq msg-ov (make-overlay 1 1 (get-buffer-create "*Messages*")))
            (message "a message")
            (message "a message")
            (should (eq buf (current-buffer))))
        (when msg-ov (delete-overlay msg-ov))))))

(ert-deftest overlay-modification-hooks-deleted-overlay ()
  "Test for bug#30823."
  (let ((check-point nil)
	(ov-delete nil)
	(ov-set nil))
    (with-temp-buffer
      (insert "abc")
      (setq ov-set (make-overlay 1 3))
      (overlay-put ov-set 'modification-hooks
		   (list (lambda (_o after &rest _args)
			   (and after (setq check-point t)))))
      (setq ov-delete (make-overlay 1 3))
      (overlay-put ov-delete 'modification-hooks
		   (list (lambda (o after &rest _args)
			   (and (not after) (delete-overlay o)))))
      (goto-char 2)
      (insert "1")
      (should (eq check-point t)))))

(ert-deftest test-generate-new-buffer-name-bug27966 ()
  (should-not (string-equal "nil"
                            (progn (get-buffer-create "nil")
                                   (generate-new-buffer-name "nil")))))

(ert-deftest test-buffer-base-buffer-indirect ()
  (with-temp-buffer
    (let* ((ind-buf-name (generate-new-buffer-name "indbuf"))
           (ind-buf (make-indirect-buffer (current-buffer) ind-buf-name)))
      (should (eq (buffer-base-buffer ind-buf) (current-buffer))))))

(ert-deftest test-buffer-base-buffer-non-indirect ()
  (with-temp-buffer
    (should (eq (buffer-base-buffer (current-buffer)) nil))))

(ert-deftest buffer-tests--overlays-indirect-bug58928 ()
  (with-temp-buffer
    (insert "hello world")
    (let* ((base (current-buffer))
           (ol1 (make-overlay (+ 2 (point-min)) (+ 8 (point-min))))
           (ib (make-indirect-buffer
                base (generate-new-buffer-name "bug58928")))
           (ol2 (with-current-buffer ib
                  (make-overlay (+ 2 (point-min)) (+ 8 (point-min))))))
      (should (equal (overlay-start ol1) (overlay-start ol2)))
      (should (equal (overlay-end ol1) (overlay-end ol2)))
      (goto-char (+ 3 (point-min)))
      (insert "a") (delete-char 2)
      (should (equal (overlay-start ol1) (overlay-start ol2)))
      (should (equal (overlay-end ol1) (overlay-end ol2)))
      (with-current-buffer ib
        (goto-char (+ 4 (point-min)))
        (insert "a") (delete-char 2))
      (should (equal (overlay-start ol1) (overlay-start ol2)))
      (should (equal (overlay-end ol1) (overlay-end ol2))))))

(ert-deftest buffer-tests--overlays-indirect-evaporate ()
  "Verify that deleting text evaporates overlays in every related buffer.

Deleting characters from either a base or an indirect buffer
should evaporate overlays in both."
  ;; Loop twice, erasing from the base buffer the first time and the
  ;; indirect buffer the second.
  (dolist (erase-where '(base indirect))
    (ert-info ((format "erase-where %S" erase-where))
      (with-temp-buffer
        (insert "xxx")
        (let* ((beg 2)
               (end 3)
               (base (current-buffer))
               (base-overlay (make-overlay beg end base))
               (indirect (make-indirect-buffer
                          base
                          (generate-new-buffer-name
                           (concat (buffer-name base) "-indirect"))))
               (indirect-overlay (make-overlay beg end indirect)))
          (overlay-put base-overlay 'evaporate t)
          (overlay-put indirect-overlay 'evaporate t)
          (with-current-buffer (pcase-exhaustive erase-where
                                 (`base base)
                                 (`indirect indirect))
            (delete-region beg end))
          (ert-info ((prin1-to-string
                      `(,base ,base-overlay ,indirect ,indirect-overlay)))
            (should (not (buffer-live-p (overlay-buffer base-overlay))))
            (should (not (buffer-live-p (overlay-buffer indirect-overlay))))
            (should (equal nil (with-current-buffer base
                                 (overlays-in (point-min) (point-max)))))
            (should (equal nil (with-current-buffer indirect
                                 (overlays-in (point-min) (point-max)))))))))))

(ert-deftest overlay-evaporation-after-killed-buffer ()
  (let* ((ols (with-temp-buffer
                (insert "toto")
                (list
                 (make-overlay (point-min) (point-max))
                 (make-overlay (point-min) (point-max))
                 (make-overlay (point-min) (point-max)))))
         (ol (nth 1 ols)))
    (overlay-put ol 'evaporate t)
    ;; Evaporation within move-overlay of an overlay that was deleted because
    ;; of a kill-buffer, triggered an assertion failure in unchain_both.
    (with-temp-buffer
      (insert "toto")
      (move-overlay ol (point-min) (point-min)))))


;; +==========================================================================+
;; | Overlay test setup
;; +==========================================================================+

(eval-and-compile
  (defun buffer-tests--make-test-name (fn x y)
    (intern (format "buffer-tests--%s-%s-%s" fn x y))))

(defun buffer-tests--unmake-test-name (symbol)
  (let ((name (if (stringp symbol) symbol (symbol-name symbol))))
    (when (string-match "\\`buffer-tests--\\(.*\\)-\\(.*\\)-\\(.*\\)\\'" name)
      (list (match-string 1 name)
            (match-string 2 name)
            (match-string 3 name)))))

(defmacro deftest-make-overlay-1 (id args)
  (declare (indent 1))
  `(ert-deftest ,(buffer-tests--make-test-name 'make-overlay 1 id) ()
     (with-temp-buffer
       (should ,(cons 'make-overlay args)))))

(defmacro deftest-make-overlay-2 (id args condition)
  (declare (indent 1))
  `(ert-deftest ,(buffer-tests--make-test-name 'make-overlay 2 id) ()
     (with-temp-buffer
       (should-error
        ,(cons 'make-overlay args)
        :type ',condition
        :exclude-subtypes t))))

(defmacro deftest-overlay-start/end-1 (id start-end-args start-end-should)
  (declare (indent 1))
  (cl-destructuring-bind (start end sstart send)
      (append start-end-args start-end-should)
    `(ert-deftest ,(buffer-tests--make-test-name 'overlay-start/end 1 id) ()
       (with-temp-buffer
         (insert (make-string 9 ?\n))
         (let ((ov (make-overlay ,start ,end)))
           (should (equal ,sstart (overlay-start ov)))
           (should (equal ,send (overlay-end ov))))))))

(defmacro deftest-overlay-buffer-1 (id arg-expr should-expr)
  (declare (indent 1))
  `(ert-deftest ,(buffer-tests--make-test-name 'overlay-buffer 1 id) ()
     (with-temp-buffer
       (should (equal (overlay-buffer (make-overlay 1 1 ,arg-expr))
                      ,should-expr)))))

(defmacro deftest-overlayp-1 (id arg-expr should-expr)
  (declare (indent 1))
  `(ert-deftest ,(buffer-tests--make-test-name 'overlayp 1 id) ()
     (with-temp-buffer
       (should (equal ,should-expr (overlayp ,arg-expr))))))

(defmacro deftest-next-overlay-change-1 (id pos result &rest ov-tuple)
  `(ert-deftest ,(buffer-tests--make-test-name 'next-overlay-change 1 id) ()
     (let ((tuple (copy-sequence ',ov-tuple)))
       (with-temp-buffer
         (insert (make-string (max 100 (if tuple
                                           (apply #'max
                                                  (mapcar
                                                   (lambda (m) (apply #'max m))
                                                   tuple))
                                         0))
                              ?\n))
         (dolist (tup tuple)
           (make-overlay (car tup) (cadr tup)))
         (should (equal (next-overlay-change ,pos)
                        ,result))))))

(defmacro deftest-previous-overlay-change-1 (id pos result &rest ov-tuple)
  `(ert-deftest ,(buffer-tests--make-test-name 'previous-overlay-change 1 id) ()
     (let ((tuple ',ov-tuple))
       (with-temp-buffer
         (insert (make-string (max 100 (if tuple
                                           (apply #'max
                                                  (mapcar
                                                   (lambda (m) (apply #'max m))
                                                   tuple))
                                         0))
                              ?\n))
         (dolist (tup tuple)
           (make-overlay (car tup) (cadr tup)))
         (should (equal (previous-overlay-change ,pos)
                        ,result))))))

(defmacro deftest-overlays-at-1 (id pos result &rest ov-triple)
  `(ert-deftest ,(buffer-tests--make-test-name 'overlays-at 1 id) ()
     (let ((pos* ,pos))
       (with-temp-buffer
         (insert (make-string 100 ?\s))
         (should-not (memq nil ',result))
         (dolist (v ',ov-triple)
           (cl-destructuring-bind (tag start end)
               v
             (overlay-put (make-overlay start end) 'tag tag)))
         (let ((ovl (overlays-at pos*)))
           (should (equal (length ovl) (length ',result)))
           (dolist (ov ovl)
             (should (memq (overlay-get ov 'tag) ',result))))))))

(defmacro deftest-overlays-in-1 (id beg end result &rest ov-triple)
  `(ert-deftest ,(buffer-tests--make-test-name 'overlays-in 1 id) ()
     (let ((beg* ,beg)
           (end* ,end))
       (with-temp-buffer
         (insert (make-string 100 ?\s))
         (should-not (memq nil ',result))
         (dolist (v ',ov-triple)
           (cl-destructuring-bind (tag start end)
               v
             (overlay-put (make-overlay start end) 'tag tag)))
         (let ((ovl (overlays-in beg* end*)))
           (should (equal (length ovl) (length ',result)))
           (dolist (ov ovl)
             (should (memq (overlay-get ov 'tag) ',result))))))))

(defmacro test-with-overlay-in-buffer (symbol-beg-end-fa-ra &rest body)
  (declare (indent 1))
  (cl-destructuring-bind (symbol beg end &optional fa ra)
      symbol-beg-end-fa-ra
    `(with-temp-buffer
       (insert (make-string (max 1000 (1- ,end)) ?\s))
       (goto-char 1)
       (let ((,symbol (make-overlay ,beg ,end nil ,fa ,ra)))
         ,@body))))

(defmacro deftest-overlays-equal-1 (id result ov1-args ov2-args)
  `(ert-deftest ,(buffer-tests--make-test-name 'overlays-equal 1 id) ()
     (cl-flet ((create-overlay (args)
                 (cl-destructuring-bind (start end &optional fa ra
                                                   &rest properties)
                     args
                   (let ((ov (make-overlay start end nil fa ra)))
                     (while properties
                       (overlay-put ov (pop properties) (pop properties)))
                     ov))))
       (with-temp-buffer
         (insert (make-string 1024 ?\s))
         (should (,(if result 'identity 'not)
                  (equal (create-overlay ',ov1-args)
                         (create-overlay ',ov2-args))))))))


(defun buffer-tests--find-ert-test (name)
  (let ((test (buffer-tests--unmake-test-name name)))
    (or (and test
             (cl-destructuring-bind (fn x y)
                 test
               (let ((regexp (format "deftest-%s-%s +%s" fn x y)))
                 (re-search-forward regexp nil t))))
        (let ((find-function-regexp-alist
               (cl-remove #'buffer-tests--find-ert-test
                          find-function-regexp-alist :key #'cdr)))
          (find-function-do-it name 'ert-deftest
                               #'switch-to-buffer-other-window)))))

(add-to-list 'find-function-regexp-alist
             `(ert-deftest . ,#'buffer-tests--find-ert-test))


;; +==========================================================================+
;; | make-overlay
;; +==========================================================================+

;; Test if making an overlay succeeds.
(deftest-make-overlay-1 A (1 1))
(deftest-make-overlay-1 B (7 26))
(deftest-make-overlay-1 C (29 7))
(deftest-make-overlay-1 D (most-positive-fixnum 1))
(deftest-make-overlay-1 E (most-negative-fixnum 1))
(deftest-make-overlay-1 F (1 most-positive-fixnum))
(deftest-make-overlay-1 G (1 most-negative-fixnum))
(deftest-make-overlay-1 H (1 1 nil t))
(deftest-make-overlay-1 I (1 1 nil nil))
(deftest-make-overlay-1 J (1 1 nil nil nil))
(deftest-make-overlay-1 K (1 1 nil nil t))
(deftest-make-overlay-1 L (1 1 nil t t))
(deftest-make-overlay-1 M (1 1 nil "yes" "yes"))

;; Test if trying to make an overlay signals conditions.
(deftest-make-overlay-2 A ()            wrong-number-of-arguments)
(deftest-make-overlay-2 B (1)           wrong-number-of-arguments)
(deftest-make-overlay-2 C (1 2 3 4 5 6) wrong-number-of-arguments)
(deftest-make-overlay-2 D ("1")         wrong-number-of-arguments)
(deftest-make-overlay-2 E ("1" "2")     wrong-type-argument)
(deftest-make-overlay-2 F (1 2 "b")     wrong-type-argument)
(deftest-make-overlay-2 G (1 2 3.14)    wrong-type-argument)
(deftest-make-overlay-2 H (3.14 3)      wrong-type-argument)
(deftest-make-overlay-2 I (1 [1])       wrong-type-argument)
(deftest-make-overlay-2 J (1 1 (with-temp-buffer
                                 (current-buffer)))
  error)


;; +==========================================================================+
;; | overlay-start/end
;; +==========================================================================+

;; Test if the overlays return proper positions.  point-max of the
;; buffer will equal 10.        ARG RESULT
(deftest-overlay-start/end-1 A (1 1) (1 1))
(deftest-overlay-start/end-1 B (2 7) (2 7))
(deftest-overlay-start/end-1 C (7 2) (2 7))
(deftest-overlay-start/end-1 D (1 10) (1 10))
(deftest-overlay-start/end-1 E (1 11) (1 10))
(deftest-overlay-start/end-1 F (1 most-positive-fixnum) (1 10))
(deftest-overlay-start/end-1 G (most-positive-fixnum 1) (1 10))
(deftest-overlay-start/end-1 H (most-positive-fixnum most-positive-fixnum)
                               (10 10))
(deftest-overlay-start/end-1 I (100 11) (10 10))
(deftest-overlay-start/end-1 J (11 100) (10 10))
(deftest-overlay-start/end-1 K (0 1) (1 1))
(deftest-overlay-start/end-1 L (1 0) (1 1))
(deftest-overlay-start/end-1 M (0 0) (1 1))

(ert-deftest test-overlay-start/end-2 ()
  (should-not (overlay-start (with-temp-buffer (make-overlay 1 1))))
  (should-not (overlay-end (with-temp-buffer (make-overlay 1 1)))))


;; +==========================================================================+
;; | overlay-buffer
;; +==========================================================================+

;; Test if overlay-buffer returns appropriate values.
(deftest-overlay-buffer-1 A (current-buffer) (current-buffer))
(deftest-overlay-buffer-1 B nil (current-buffer))
(ert-deftest test-overlay-buffer-1-C ()
  (should-error (make-overlay
                 1 1 (with-temp-buffer (current-buffer)))))


;; +==========================================================================+
;; | overlayp
;; +==========================================================================+

;; Check the overlay predicate.
(deftest-overlayp-1 A (make-overlay 1 1) t)
(deftest-overlayp-1 B (with-temp-buffer (make-overlay 1 1)) t)
(deftest-overlayp-1 C nil                            nil)
(deftest-overlayp-1 D 'symbol                        nil)
(deftest-overlayp-1 E "string"                       nil)
(deftest-overlayp-1 F 42                             nil)
(deftest-overlayp-1 G [1 2]                          nil)
(deftest-overlayp-1 H (symbol-function 'car)         nil)
(deftest-overlayp-1 I float-pi                       nil)
(deftest-overlayp-1 J (cons 1 2)                     nil)
(deftest-overlayp-1 K (make-hash-table)              nil)
(deftest-overlayp-1 L (symbol-function 'ert-deftest) nil)
(deftest-overlayp-1 M (current-buffer)               nil)
(deftest-overlayp-1 N (selected-window)              nil)
(deftest-overlayp-1 O (selected-frame)               nil)


;; +==========================================================================+
;; | overlay equality
;; +==========================================================================+

(deftest-overlays-equal-1 A t (1 1) (1 1))
(deftest-overlays-equal-1 B t (5 10) (5 10))
(deftest-overlays-equal-1 C nil (5 11) (5 10))
(deftest-overlays-equal-1 D t (10 20 t) (10 20))
(deftest-overlays-equal-1 E t (10 20 nil t) (10 20))
(deftest-overlays-equal-1 F t (10 20 t t) (10 20 nil t))
(deftest-overlays-equal-1 G t (10 20 t t) (10 20 t nil))
(deftest-overlays-equal-1 H t (10 20 nil nil foo 42) (10 20 nil nil foo 42))
(deftest-overlays-equal-1 I nil (10 20 nil nil foo 42) (10 20 nil nil foo 43))


;; +==========================================================================+
;; | overlay-lists
;; +==========================================================================+

;; Check whether overlay-lists returns something sensible.
(ert-deftest test-overlay-lists-1 ()
  (with-temp-buffer
    (should (equal (cons nil nil) (overlay-lists)))
    (dotimes (i 10) (make-overlay 1 i))
    (should (listp (car (overlay-lists))))
    (should (listp (cdr (overlay-lists))))
    (let ((list (append (car (overlay-lists))
                        (cdr (overlay-lists)))))
      (should (= 10 (length list)))
      (should (seq-every-p #'overlayp list)))))


;; +==========================================================================+
;; | overlay-put/get/properties
;; +==========================================================================+

;; Test if overlay-put properties can be retrieved by overlay-get and
;; overlay-properties.
(ert-deftest test-overlay-props-1 ()
  (with-temp-buffer
    (let* ((keys '(:k1 :k2 :k3))
           (values '(1 "v2" v3))
           (ov (make-overlay 1 1))
           (n (length keys)))
      (should (equal (length keys) (length values)))
      (should (null (overlay-properties ov)))
      ;; Insert keys and values.
      (dotimes (i n)
        (should (equal (overlay-put ov (nth i keys) (nth i values))
                       (nth i values))))
      ;; Compare with what overlay-get says.
      (dotimes (i n)
        (should (equal (overlay-get ov (nth i keys))
                       (nth i values))))
      ;; Test if overlay-properties is a superset.
      (dotimes (i n)
        (should (equal (plist-get (overlay-properties ov)
                                  (nth i keys))
                       (nth i values))))
      ;; Check if overlay-properties is a subset.
      (should (= (length (overlay-properties ov)) (* n 2))))))


;; +==========================================================================+
;; | next-overlay-change
;; +==========================================================================+

;; Test if next-overlay-change returns RESULT if called with POS in a
;; buffer with overlays corresponding to OVS and point-max >= 100.
;;                               (POS RESULT &rest OVS)
;; 0 overlays
(deftest-next-overlay-change-1 A (point-min) (point-max))
(deftest-next-overlay-change-1 B (point-max) (point-max))
;; 1 non-empty overlay
(deftest-next-overlay-change-1 C 1 10 (10 20))
(deftest-next-overlay-change-1 D 10 20 (10 20))
(deftest-next-overlay-change-1 E 15 20 (10 20))
(deftest-next-overlay-change-1 F 20 (point-max) (10 20))
(deftest-next-overlay-change-1 G 30 (point-max) (10 20))
;; 1 empty overlay
(deftest-next-overlay-change-1 H 1 10 (10 10))
(deftest-next-overlay-change-1 I 10 (point-max) (10 10))
(deftest-next-overlay-change-1 J 20 (point-max) (10 10))
;; 2 non-empty, non-intersecting
(deftest-next-overlay-change-1 D2 10 20 (20 30) (40 50))
(deftest-next-overlay-change-1 E2 35 40 (20 30) (40 50))
(deftest-next-overlay-change-1 F2 60 (point-max) (20 30) (40 50))
(deftest-next-overlay-change-1 G2 30 40 (20 30) (40 50))
(deftest-next-overlay-change-1 H2 50 (point-max) (20 30) (40 50))
;; 2 non-empty, intersecting
(deftest-next-overlay-change-1 I2 10 20 (20 30) (25 35))
(deftest-next-overlay-change-1 J2 20 25 (20 30) (25 35))
(deftest-next-overlay-change-1 K 23 25 (20 30) (25 35))
(deftest-next-overlay-change-1 L 25 30 (20 30) (25 35))
(deftest-next-overlay-change-1 M 28 30 (20 30) (25 35))
(deftest-next-overlay-change-1 N 30 35 (20 30) (25 35))
(deftest-next-overlay-change-1 O 35 (point-max) (20 30) (25 35))
(deftest-next-overlay-change-1 P 50 (point-max) (20 30) (25 35))
;; 2 non-empty, continuous
(deftest-next-overlay-change-1 Q 10 20 (20 30) (30 40))
(deftest-next-overlay-change-1 R 20 30 (20 30) (30 40))
(deftest-next-overlay-change-1 S 25 30 (20 30) (30 40))
(deftest-next-overlay-change-1 T 30 40 (20 30) (30 40))
(deftest-next-overlay-change-1 U 35 40 (20 30) (30 40))
(deftest-next-overlay-change-1 V 40 (point-max) (20 30) (30 40))
(deftest-next-overlay-change-1 W 50 (point-max) (20 30) (30 40))
;; 1 empty, 1 non-empty, non-in
(deftest-next-overlay-change-1 a 10 20 (20 20) (30 40))
(deftest-next-overlay-change-1 b 20 30 (20 30) (30 40))
(deftest-next-overlay-change-1 c 25 30 (20 30) (30 40))
(deftest-next-overlay-change-1 d 30 40 (20 30) (30 40))
(deftest-next-overlay-change-1 e 35 40 (20 30) (30 40))
(deftest-next-overlay-change-1 f 40 (point-max) (20 30) (30 40))
(deftest-next-overlay-change-1 g 50 (point-max) (20 30) (30 40))
;; 1 empty, 1 non-empty, intersecting at begin
(deftest-next-overlay-change-1 h 10 20 (20 20) (20 30))
(deftest-next-overlay-change-1 i 20 30 (20 20) (20 30))
(deftest-next-overlay-change-1 j 25 30 (20 20) (20 30))
(deftest-next-overlay-change-1 k 30 (point-max) (20 20) (20 30))
(deftest-next-overlay-change-1 l 40 (point-max) (20 20) (20 30))
;; 1 empty, 1 non-empty, intersecting at end
(deftest-next-overlay-change-1 h2 10 20 (30 30) (20 30))
(deftest-next-overlay-change-1 i2 20 30 (30 30) (20 30))
(deftest-next-overlay-change-1 j2 25 30 (30 30) (20 30))
(deftest-next-overlay-change-1 k2 30 (point-max) (20 20) (20 30))
(deftest-next-overlay-change-1 l2 40 (point-max) (20 20) (20 30))
;; 1 empty, 1 non-empty, intersecting in the middle
(deftest-next-overlay-change-1 m 10 20 (25 25) (20 30))
(deftest-next-overlay-change-1 n 20 25 (25 25) (20 30))
(deftest-next-overlay-change-1 o 25 30 (25 25) (20 30))
(deftest-next-overlay-change-1 p 30 (point-max) (25 25) (20 30))
(deftest-next-overlay-change-1 q 40 (point-max) (25 25) (20 30))
;; 2 empty, intersecting
(deftest-next-overlay-change-1 r 10 20 (20 20) (20 20))
(deftest-next-overlay-change-1 s 20 (point-max) (20 20) (20 20))
(deftest-next-overlay-change-1 t 30 (point-max) (20 20) (20 20))
;; 2 empty, non-intersecting
(deftest-next-overlay-change-1 u 10 20 (20 20) (30 30))
(deftest-next-overlay-change-1 v 20 30 (20 20) (30 30))
(deftest-next-overlay-change-1 w 25 30 (20 20) (30 30))
(deftest-next-overlay-change-1 x 30 (point-max) (20 20) (30 30))
(deftest-next-overlay-change-1 y 50 (point-max) (20 20) (30 30))
;; 10 random
(deftest-next-overlay-change-1 aa 1 5
  (58 66) (41 10) (9 67) (28 88) (27 43)
  (24 27) (48 36) (5 90) (61 9))
(deftest-next-overlay-change-1 ab (point-max) (point-max)
  (58 66) (41 10) (9 67) (28 88) (27 43)
  (24 27) (48 36) (5 90) (61 9))
(deftest-next-overlay-change-1 ac 67 88
  (58 66) (41 10) (9 67) (28 88) (27 43)
  (24 27) (48 36) (5 90) (61 9))


;; +==========================================================================+
;; | previous-overlay-change.
;; +==========================================================================+

;; Same for previous-overlay-change.
;; 1 non-empty overlay
(deftest-previous-overlay-change-1 A (point-max) 1)
(deftest-previous-overlay-change-1 B 1 1)
(deftest-previous-overlay-change-1 C 1 1 (10 20))
(deftest-previous-overlay-change-1 D 10 1 (10 20))
(deftest-previous-overlay-change-1 E 15 10 (10 20))
(deftest-previous-overlay-change-1 F 20 10 (10 20))
(deftest-previous-overlay-change-1 G 30 20 (10 20))
;; 1 empty overlay
(deftest-previous-overlay-change-1 H 1 1 (10 10))
(deftest-previous-overlay-change-1 I 10 1 (10 10))
(deftest-previous-overlay-change-1 J 20 10 (10 10))
;; 2 non-empty, non-intersecting
(deftest-previous-overlay-change-1 D2 10 1 (20 30) (40 50))
(deftest-previous-overlay-change-1 E2 35 30 (20 30) (40 50))
(deftest-previous-overlay-change-1 F2 60 50 (20 30) (40 50))
(deftest-previous-overlay-change-1 G2 30 20 (20 30) (40 50))
(deftest-previous-overlay-change-1 H2 50 40 (20 30) (40 50))
;; 2 non-empty, intersecting
(deftest-previous-overlay-change-1 I2 10 1 (20 30) (25 35))
(deftest-previous-overlay-change-1 J2 20 1 (20 30) (25 35))
(deftest-previous-overlay-change-1 K 23 20 (20 30) (25 35))
(deftest-previous-overlay-change-1 L 25 20 (20 30) (25 35))
(deftest-previous-overlay-change-1 M 28 25 (20 30) (25 35))
(deftest-previous-overlay-change-1 N 30 25 (20 30) (25 35))
(deftest-previous-overlay-change-1 O 35 30 (20 30) (25 35))
(deftest-previous-overlay-change-1 P 50 35 (20 30) (25 35))
;; 2 non-empty, continuous
(deftest-previous-overlay-change-1 Q 10 1 (20 30) (30 40))
(deftest-previous-overlay-change-1 R 20 1 (20 30) (30 40))
(deftest-previous-overlay-change-1 S 25 20 (20 30) (30 40))
(deftest-previous-overlay-change-1 T 30 20 (20 30) (30 40))
(deftest-previous-overlay-change-1 U 35 30 (20 30) (30 40))
(deftest-previous-overlay-change-1 V 40 30 (20 30) (30 40))
(deftest-previous-overlay-change-1 W 50 40 (20 30) (30 40))
;; 1 empty, 1 non-empty, non-intersecting
(deftest-previous-overlay-change-1 a 10 1 (20 20) (30 40))
(deftest-previous-overlay-change-1 b 20 1 (20 30) (30 40))
(deftest-previous-overlay-change-1 c 25 20 (20 30) (30 40))
(deftest-previous-overlay-change-1 d 30 20 (20 30) (30 40))
(deftest-previous-overlay-change-1 e 35 30 (20 30) (30 40))
(deftest-previous-overlay-change-1 f 40 30 (20 30) (30 40))
(deftest-previous-overlay-change-1 g 50 40 (20 30) (30 40))
;; 1 empty, 1 non-empty, intersecting at begin
(deftest-previous-overlay-change-1 h 10 1 (20 20) (20 30))
(deftest-previous-overlay-change-1 i 20 1 (20 20) (20 30))
(deftest-previous-overlay-change-1 j 25 20 (20 20) (20 30))
(deftest-previous-overlay-change-1 k 30 20 (20 20) (20 30))
(deftest-previous-overlay-change-1 l 40 30 (20 20) (20 30))
;; 1 empty, 1 non-empty, intersecting at end
(deftest-previous-overlay-change-1 m 10 1 (30 30) (20 30))
(deftest-previous-overlay-change-1 n 20 1 (30 30) (20 30))
(deftest-previous-overlay-change-1 o 25 20 (30 30) (20 30))
(deftest-previous-overlay-change-1 p 30 20 (20 20) (20 30))
(deftest-previous-overlay-change-1 q 40 30 (20 20) (20 30))
;; 1 empty, 1 non-empty, intersecting in the middle
(deftest-previous-overlay-change-1 r 10 1 (25 25) (20 30))
(deftest-previous-overlay-change-1 s 20 1 (25 25) (20 30))
(deftest-previous-overlay-change-1 t 25 20 (25 25) (20 30))
(deftest-previous-overlay-change-1 u 30 25 (25 25) (20 30))
(deftest-previous-overlay-change-1 v 40 30 (25 25) (20 30))
;; 2 empty, intersecting
(deftest-previous-overlay-change-1 w 10 1 (20 20) (20 20))
(deftest-previous-overlay-change-1 x 20 1 (20 20) (20 20))
(deftest-previous-overlay-change-1 y 30 20 (20 20) (20 20))
;; 2 empty, non-intersecting
(deftest-previous-overlay-change-1 z 10 1 (20 20) (30 30))
(deftest-previous-overlay-change-1 aa 20 1 (20 20) (30 30))
(deftest-previous-overlay-change-1 ab 25 20 (20 20) (30 30))
(deftest-previous-overlay-change-1 ac 30 20 (20 20) (30 30))
(deftest-previous-overlay-change-1 ad 50 30 (20 20) (30 30))
;; 10 random
(deftest-previous-overlay-change-1 ae 100 90
  (58 66) (41 10) (9 67) (28 88) (27 43)
  (24 27) (48 36) (5 90) (61 9))
(deftest-previous-overlay-change-1 af (point-min) (point-min)
  (58 66) (41 10) (9 67) (28 88) (27 43)
  (24 27) (48 36) (5 90) (61 9))
(deftest-previous-overlay-change-1 ag 29 28
  (58 66) (41 10) (9 67) (28 88) (27 43)
  (24 27) (48 36) (5 90) (61 9))


;; +==========================================================================+
;; | overlays-at
;; +==========================================================================+


;; Test whether overlay-at returns RESULT at POS after overlays OVL were
;; created in a buffer.         POS RES OVL
(deftest-overlays-at-1 A 1 ())
;; 1 overlay
(deftest-overlays-at-1 B 10 (a) (a 10 20))
(deftest-overlays-at-1 C 15 (a) (a 10 20))
(deftest-overlays-at-1 D 19 (a) (a 10 20))
(deftest-overlays-at-1 E 20 ()  (a 10 20))
(deftest-overlays-at-1 F 1 () (a 10 20))

;; 2 non-empty overlays non-intersecting
(deftest-overlays-at-1 G 1 () (a 10 20) (b 30 40))
(deftest-overlays-at-1 H 10 (a) (a 10 20) (b 30 40))
(deftest-overlays-at-1 I 15 (a) (a 10 20) (b 30 40))
(deftest-overlays-at-1 K 20 () (a 10 20) (b 30 40))
(deftest-overlays-at-1 L 25 () (a 10 20) (b 30 40))
(deftest-overlays-at-1 M 30 (b) (a 10 20) (b 30 40))
(deftest-overlays-at-1 N 35 (b) (a 10 20) (b 30 40))
(deftest-overlays-at-1 O 40 () (a 10 20) (b 30 40))
(deftest-overlays-at-1 P 50 () (a 10 20) (b 30 40))

;; 2 non-empty overlays intersecting
(deftest-overlays-at-1 G2 1 () (a 10 30) (b 20 40))
(deftest-overlays-at-1 H2 10 (a) (a 10 30) (b 20 40))
(deftest-overlays-at-1 I2 15 (a) (a 10 30) (b 20 40))
(deftest-overlays-at-1 K2 20 (a b) (a 10 30) (b 20 40))
(deftest-overlays-at-1 L2 25 (a b) (a 10 30) (b 20 40))
(deftest-overlays-at-1 M2 30 (b) (a 10 30) (b 20 40))
(deftest-overlays-at-1 N2 35 (b) (a 10 30) (b 20 40))
(deftest-overlays-at-1 O2 40 () (a 10 30) (b 20 40))
(deftest-overlays-at-1 P2 50 () (a 10 30) (b 20 40))

;; 2 non-empty overlays continuous
(deftest-overlays-at-1 G3 1 () (a 10 20) (b 20 30))
(deftest-overlays-at-1 H3 10 (a) (a 10 20) (b 20 30))
(deftest-overlays-at-1 I3 15 (a) (a 10 20) (b 20 30))
(deftest-overlays-at-1 K3 20 (b) (a 10 20) (b 20 30))
(deftest-overlays-at-1 L3 25 (b) (a 10 20) (b 20 30))
(deftest-overlays-at-1 M3 30 () (a 10 20) (b 20 30))

;; overlays-at never returns empty overlays.
(deftest-overlays-at-1 N3 1 (a) (a 1 60) (c 1 1) (b 30 30) (d 50 50))
(deftest-overlays-at-1 O3 20 (a) (a 1 60) (c 1 1) (b 30 30) (d 50 50))
(deftest-overlays-at-1 P3 30 (a) (a 1 60) (c 1 1) (b 30 30) (d 50 50))
(deftest-overlays-at-1 Q 40 (a) (a 1 60) (c 1 1) (b 30 30) (d 50 50))
(deftest-overlays-at-1 R 50 (a) (a 1 60) (c 1 1) (b 30 30) (d 50 50))
(deftest-overlays-at-1 S 60 () (a 1 60) (c 1 1) (b 30 30) (d 50 50))

;; behavior at point-min and point-max
(ert-deftest test-overlays-at-2 ()
  (cl-macrolet ((should-length (n list)
                               `(should (= ,n (length ,list)))))
    (with-temp-buffer
      (insert (make-string 100 ?\s))
      (make-overlay 1 (point-max))
      (make-overlay 10 10)
      (make-overlay 20 20)
      (make-overlay (point-max) (point-max))
      (should-length 1 (overlays-at 1))
      (should-length 1 (overlays-at 10))
      (should-length 1 (overlays-at 20))
      (should-length 0 (overlays-at (point-max)))
      (narrow-to-region 10 20)
      (should-length 1 (overlays-at (point-min)))
      (should-length 1 (overlays-at 15))
      (should-length 1 (overlays-at (point-max))))))

(defun sorted-overlays (overlays)
  (sort
   (mapcar (lambda (overlay)
             (list (overlay-start overlay)
                   (overlay-end overlay)))
           overlays)
   (lambda (first second)
     (cl-loop for a in first
              for b in second
              thereis (< a b)
              until (> a b)))))

(defun sorted-overlays-at (pos)
  (sorted-overlays (overlays-at pos)))

(defun sorted-overlays-in (beg end)
  (sorted-overlays (overlays-in beg end)))

(ert-deftest test-overlays-at-narrow-to-region-end ()
  ;; See bug#58703.
  (with-temp-buffer
   (insert (make-string 30 ?x))
   (make-overlay 10 11)
   (narrow-to-region 10 10)
   (should (equal
            '((10 11))
            (sorted-overlays-at 10)))))

;; +==========================================================================+
;; | overlay-in
;; +==========================================================================+


;; Test whether overlays-in returns RES in BEG,END after overlays OVL were
;; created in a buffer.

(deftest-overlays-in-1 A 1 (point-max) ());;POS RES OVL
;; 1 overlay
(deftest-overlays-in-1 B 1 10 () (a 10 20))
(deftest-overlays-in-1 C 5 10 () (a 10 20))
(deftest-overlays-in-1 D 5 15 (a) (a 10 20))
(deftest-overlays-in-1 E 10 15 (a)  (a 10 20))
(deftest-overlays-in-1 F 10 20 (a) (a 10 20))
(deftest-overlays-in-1 G 15 20 (a) (a 10 20))
(deftest-overlays-in-1 H 15 25 (a) (a 10 20))
(deftest-overlays-in-1 I 20 25 () (a 10 20))
(deftest-overlays-in-1 J 30 50 () (a 10 20))

;; 2 non-empty overlays non-intersecting
(deftest-overlays-in-1 K 1 5 () (a 10 20) (b 30 40))
(deftest-overlays-in-1 L 5 10 () (a 10 20) (b 30 40))
(deftest-overlays-in-1 M 5 15 (a) (a 10 20) (b 30 40))
(deftest-overlays-in-1 N 10 15 (a) (a 10 20) (b 30 40))
(deftest-overlays-in-1 O 15 20 (a) (a 10 20) (b 30 40))
(deftest-overlays-in-1 P 15 25 (a) (a 10 20) (b 30 40))
(deftest-overlays-in-1 Q 20 25 () (a 10 20) (b 30 40))
(deftest-overlays-in-1 R 20 30 () (a 10 20) (b 30 40))
(deftest-overlays-in-1 S 25 30 () (a 10 20) (b 30 40))
(deftest-overlays-in-1 T 25 35 (b) (a 10 20) (b 30 40))
(deftest-overlays-in-1 U 30 35 (b) (a 10 20) (b 30 40))
(deftest-overlays-in-1 V 40 50  () (a 10 20) (b 30 40))
(deftest-overlays-in-1 W 50 60  () (a 10 20) (b 30 40))
(deftest-overlays-in-1 X 1 50  (a b) (a 10 20) (b 30 40))
(deftest-overlays-in-1 Y 10 40  (a b) (a 10 20) (b 30 40))
(deftest-overlays-in-1 Z 10 41  (a b) (a 10 20) (b 30 40))

;; 2 non-empty overlays intersecting
(deftest-overlays-in-1 a 1 5 () (a 10 30) (b 20 40))
(deftest-overlays-in-1 b 5 10 () (a 10 30) (b 20 40))
(deftest-overlays-in-1 c 5 15 (a) (a 10 30) (b 20 40))
(deftest-overlays-in-1 d 10 15 (a) (a 10 30) (b 20 40))
(deftest-overlays-in-1 e 10 20 (a) (a 10 30) (b 20 40))
(deftest-overlays-in-1 f 15 20 (a) (a 10 30) (b 20 40))
(deftest-overlays-in-1 g 20 30 (a b) (a 10 30) (b 20 40))
(deftest-overlays-in-1 h 20 40 (a b) (a 10 30) (b 20 40))
(deftest-overlays-in-1 i 25 30 (a b) (a 10 30) (b 20 40))
(deftest-overlays-in-1 j 30 30 (b) (a 10 30) (b 20 40))
(deftest-overlays-in-1 k 30 35 (b) (a 10 30) (b 20 40))
(deftest-overlays-in-1 l 35 40 (b) (a 10 30) (b 20 40))
(deftest-overlays-in-1 m 40 45 () (a 10 30) (b 20 40))
(deftest-overlays-in-1 n 41 45 () (a 10 30) (b 20 40))
(deftest-overlays-in-1 o 50 60 () (a 10 30) (b 20 40))

;; 2 non-empty overlays continuous
(deftest-overlays-in-1 p 1 5 () (a 10 20) (b 20 30))
(deftest-overlays-in-1 q 5 10 () (a 10 20) (b 20 30))
(deftest-overlays-in-1 r 15 20 (a) (a 10 20) (b 20 30))
(deftest-overlays-in-1 s 15 25 (a b) (a 10 20) (b 20 30))
(deftest-overlays-in-1 t 20 25 (b) (a 10 20) (b 20 30))
(deftest-overlays-in-1 u 25 30 (b) (a 10 20) (b 20 30))
(deftest-overlays-in-1 v 29 35 (b) (a 10 20) (b 20 30))
(deftest-overlays-in-1 w 30 35 () (a 10 20) (b 20 30))
(deftest-overlays-in-1 x 35 50 () (a 10 20) (b 20 30))
(deftest-overlays-in-1 y 1 50 (a b) (a 10 20) (b 20 30))
(deftest-overlays-in-1 z 15 50 (a b) (a 10 20) (b 20 30))
(deftest-overlays-in-1 aa 1 25 (a b) (a 10 20) (b 20 30))

;; 1 empty overlay
(deftest-overlays-in-1 ab 1 10 () (a 10 10))
(deftest-overlays-in-1 ac 10 10 (a) (a 10 10))
(deftest-overlays-in-1 ad 9 10 () (a 10 10))
(deftest-overlays-in-1 ae 9 11 (a) (a 10 10))
(deftest-overlays-in-1 af 10 11 (a) (a 10 10))

;; behavior for empty range
(ert-deftest test-overlays-in-empty-range ()
    (with-temp-buffer
      (insert (make-string 4 ?x))
      (cl-loop for start from (point-min) to (point-max)
               do (cl-loop for end from start to (point-max)
                           do (when (<= start end)
                                (make-overlay start end))))

      (cl-loop for pos from (point-min) to (point-max)
               do (ert-info ((format "after (overlay-recenter %d)" pos))
                    (overlay-recenter pos)
                    (should (equal
                             '((1 1))
                             (sorted-overlays-in (point-min) (point-min))))
                    (should (equal
                             '((1 3) (1 4) (1 5) (2 2))
                             (sorted-overlays-in 2 2)))
                    (should (equal
                             '((1 4) (1 5) (2 4) (2 5) (3 3))
                             (sorted-overlays-in 3 3)))
                    (should (equal
                             '((1 5) (2 5) (3 5) (4 4))
                             (sorted-overlays-in 4 4)))
                    (should (equal
                             '((5 5))
                             (sorted-overlays-in (point-max) (point-max))))))))

(ert-deftest test-overlays-in-empty-range-bug58672 ()
  (with-temp-buffer
    (insert (make-string 10 ?=))
    (make-overlay 5 7 nil nil t)
    (should (equal nil (overlays-in 5 5)))))

;; behavior at point-max
(ert-deftest test-overlays-in-2 ()
  (cl-macrolet ((should-length (n list)
                               `(should (= ,n (length ,list)))))
    (with-temp-buffer
      (insert (make-string 100 ?\s))
      (make-overlay (point-max) (point-max))
      (make-overlay 50 50)
      (should-length 1 (overlays-in 50 50))
      (should-length 2 (overlays-in 1 (point-max)))
      (should-length 1 (overlays-in (point-max) (point-max)))
      (narrow-to-region 1 50)
      (should-length 1 (overlays-in 1 (point-max)))
      (should-length 1 (overlays-in (point-max) (point-max))))))


;; +==========================================================================+
;; | overlay-recenter
;; +==========================================================================+

;; This function is a noop in the overlay tree branch.
(ert-deftest test-overlay-recenter ()
  (with-temp-buffer
    (should-not (overlay-recenter 1))
    (insert (make-string 100 ?\s))
    (dotimes (i 10)
      (make-overlay i (1+ i))
      (should-not (overlay-recenter i)))))


;; +==========================================================================+
;; | move-overlay
;; +==========================================================================+

;; buffer nil with live overlay
(ert-deftest test-move-overlay-1 ()
  (test-with-overlay-in-buffer (ov 1 100)
    (move-overlay ov 50 60)
    (should (= 50 (overlay-start ov)))
    (should (= 60 (overlay-end ov)))
    (should (eq (current-buffer) (overlay-buffer ov)))))

;; buffer nil, dead overlay
(ert-deftest test-move-overlay-2 ()
  (with-temp-buffer
    (let ((ov (test-with-overlay-in-buffer (ov 1 100) ov)))
      (insert (make-string 100 ?\s))
      (move-overlay ov 50 60)
      (should (= 50 (overlay-start ov)))
      (should (= 60 (overlay-end ov)))
      (should (eq (current-buffer) (overlay-buffer ov))))))

;; buffer non-nil, live overlay
(ert-deftest test-move-overlay-3 ()
  (test-with-overlay-in-buffer (ov 10 100)
    (with-temp-buffer
      (move-overlay ov 1 1 (current-buffer))
      (should (= 1 (overlay-start ov)))
      (should (= 1 (overlay-end ov)))
      (should (eq (current-buffer) (overlay-buffer ov))))
    (should-not (overlay-start ov))
    (should-not (overlay-end ov))
    (should-not (overlay-buffer ov))))

;; buffer non-nil, dead overlay
(ert-deftest test-move-overlay-4 ()
  (let ((ov (test-with-overlay-in-buffer (ov 1 1) ov)))
    (with-temp-buffer
      (move-overlay ov 1 1 (current-buffer))
      (should (= 1 (overlay-start ov)))
      (should (= 1 (overlay-end ov)))
      (should (eq (current-buffer) (overlay-buffer ov))))
    (should-not (overlay-start ov))
    (should-not (overlay-end ov))
    (should-not (overlay-buffer ov))))

;; +==========================================================================+
;; | delete-(all-)overlay
;; +==========================================================================+

;; delete live overlay
(ert-deftest test-delete-overlay-1 ()
  (test-with-overlay-in-buffer (ov 10 100)
    (should (buffer-live-p (overlay-buffer ov)))
    (delete-overlay ov)
    (should-not (overlay-start ov))
    (should-not (overlay-end ov))
    (should-not (overlay-buffer ov))))

;; delete dead overlay
(ert-deftest test-delete-overlay-2 ()
  (let ((ov (test-with-overlay-in-buffer (ov 10 100) ov)))
    (should-not (overlay-start ov))
    (should-not (overlay-end ov))
    (should-not (overlay-buffer ov))
    (should-not (delete-overlay ov))
    (should-not (overlay-start ov))
    (should-not (overlay-end ov))
    (should-not (overlay-buffer ov))))

(ert-deftest test-delete-all-overlay-1 ()
  (with-temp-buffer
    (should-not (delete-all-overlays))
    (should-not (delete-all-overlays (current-buffer)))
    (insert (make-string 100 ?\s))
    (dotimes (i 10) (make-overlay i (1+ i)))
    (should-not (delete-all-overlays (current-buffer)))
    (should-not (delete-all-overlays))))


;; +==========================================================================+
;; | get-pos-property
;; +==========================================================================+

(ert-deftest get-pos-property-overlay-beg ()
  "Test `get-pos-property' at the beginning of an overlay.
Regression test for bug#58706."
  (with-temp-buffer
    (insert (make-string 10000 ?x))
    (let ((overlay (make-overlay 9999 10001)))
      (overlay-put overlay 'forty-two 42))
    (should (equal 42 (get-pos-property 9999 'forty-two)))))

(ert-deftest get-pos-property-overlay-empty-rear-advance ()
  "Test `get-pos-property' at the end of an empty rear-advance overlay.
Regression test for bug#58706."
  (with-temp-buffer
    (insert (make-string 10000 ?x))
    (let ((overlay (make-overlay 9999 9999 nil nil t)))
      (overlay-put overlay 'forty-two 42))
    (should (equal 42 (get-pos-property 9999 'forty-two)))))

(ert-deftest get-pos-property-overlay-past-rear-advance ()
  "Test `get-pos-property' past the end of an empty rear-advance overlay.
Regression test for bug#58706."
  (with-temp-buffer
    (insert (make-string 10000 ?x))
    (let ((overlay (make-overlay 9998 9998 nil nil t)))
      (overlay-put overlay 'forty-two 42))
    (should (equal nil (get-pos-property 9999 'forty-two)))))

(ert-deftest get-pos-property-overlay-at-narrowed-end ()
  "Test `get-pos-property' at the end of a narrowed region.
Regression test for bug#58706."
  (with-temp-buffer
    (insert (make-string 11000 ?x))
    (narrow-to-region 9998 10000)
    (let ((overlay (make-overlay 10000 10000 nil t nil)))
      (overlay-put overlay 'forty-two 42))
    (should (equal nil (get-pos-property 9999 'forty-two)))))

;; FIXME: add more `get-pos-property' tests

;; +==========================================================================+
;; | get-char-property(-and-overlay)
;; +==========================================================================+

;; FIXME: TBD


;; +==========================================================================+
;; | Moving by insertions
;; +==========================================================================+

(defmacro deftest-moving-insert-1 (id beg-end insert sbeg-send fa ra)
  (cl-destructuring-bind (beg end ipos ilen sbeg send fa ra)
      (append beg-end insert sbeg-send (list fa ra) nil)
    `(ert-deftest ,(buffer-tests--make-test-name 'moving-insert 1 id) ()
       (test-with-overlay-in-buffer (ov ,beg ,end ,fa ,ra)
         (should (= ,beg (overlay-start ov)))
         (should (= ,end (overlay-end ov)))
         (goto-char ,ipos)
         (insert (make-string ,ilen ?x))
         (should (= ,sbeg (overlay-start ov)))
         (should (= ,send (overlay-end ov)))))))

;; non-empty, no fa, no ra
;; --------------------  OV      INS    RESULT
(deftest-moving-insert-1 A (10 20) (15 3) (10 23) nil nil)
(deftest-moving-insert-1 B (10 20) (20 4) (10 20) nil nil)
(deftest-moving-insert-1 C (10 20) (5 5) (15 25) nil nil)
(deftest-moving-insert-1 D (10 20) (10 3) (10 23) nil nil)
(deftest-moving-insert-1 E (10 20) (20 4) (10 20) nil nil)

;; non-empty no fa, ra
(deftest-moving-insert-1 F (10 20) (15 3) (10 23) nil t)
(deftest-moving-insert-1 G (10 20) (20 4) (10 24) nil t)
(deftest-moving-insert-1 H (10 20) (5 5) (15 25) nil t)
(deftest-moving-insert-1 I (10 20) (10 3) (10 23) nil t)
(deftest-moving-insert-1 J (10 20) (20 4) (10 24) nil t)

;; non-empty, fa, no r
(deftest-moving-insert-1 K (10 20) (15 3) (10 23) t nil)
(deftest-moving-insert-1 L (10 20) (20 4) (10 20) t nil)
(deftest-moving-insert-1 M (10 20) (5 5) (15 25) t nil)
(deftest-moving-insert-1 N (10 20) (10 3) (13 23) t nil)
(deftest-moving-insert-1 O (10 20) (20 4) (10 20) t nil)

;; This used to fail.
(ert-deftest test-moving-insert-2-a ()
  (with-temp-buffer
    (insert (make-string 1 ?.))
    (let ((ov (make-overlay 1 1 nil t nil)))
      (insert "()")
      (should (= 1 (overlay-end ov))))))

;; non-empty, fa, ra
(deftest-moving-insert-1 P (10 20) (15 3) (10 23) t t)
(deftest-moving-insert-1 Q (10 20) (20 4) (10 24) t t)
(deftest-moving-insert-1 R (10 20) (5 5) (15 25) t t)
(deftest-moving-insert-1 S (10 20) (10 3) (13 23) t t)
(deftest-moving-insert-1 T (10 20) (20 4) (10 24) t t)

;; empty, no fa, no ra
(deftest-moving-insert-1 U (15 15) (20 4) (15 15) nil nil)
(deftest-moving-insert-1 V (15 15) (5 5) (20 20) nil nil)
(deftest-moving-insert-1 W (15 15) (15 3) (15 15) nil nil)

;; empty no fa, ra
(deftest-moving-insert-1 X (15 15) (20 4) (15 15) nil t)
(deftest-moving-insert-1 Y (15 15) (5 5) (20 20) nil t)
(deftest-moving-insert-1 Z (15 15) (15 3) (15 18) nil t)

;; empty, fa, no ra
(deftest-moving-insert-1 a (15 15) (20 4) (15 15) t nil)
(deftest-moving-insert-1 b (15 15) (5 5) (20 20) t nil)
(deftest-moving-insert-1 c (15 15) (15 3) (15 15) t nil)

;; empty, fa, ra
(deftest-moving-insert-1 d (15 15) (20 4) (15 15) t t)
(deftest-moving-insert-1 e (15 15) (5 5) (20 20) t t)
(deftest-moving-insert-1 f (15 15) (15 3) (18 18) t t)

;; Try to trigger a pathological case where the tree could become
;; unordered due to an insert operation.

(ert-deftest test-moving-insert-2 ()
  (with-temp-buffer
    (insert (make-string 1000 ?x))
    (let ((root (make-overlay 50 75 nil    nil             'rear-advance))
          (left (make-overlay 25 50 nil    'front-advance  'rear-advance))
          (right (make-overlay 75 100 nil  nil             nil)))
      ;;     [50] <--- start
      ;;    /    \
      ;; (25)    (75)
      (delete-region 25 75)
      ;;     [25]
      ;;    /    \
      ;; (25)    (25)
      (should (= 25 (overlay-start root)))
      (should (= 25 (overlay-end root)))
      (should (= 25 (overlay-start left)))
      (should (= 25 (overlay-end left)))
      (should (= 25 (overlay-start right)))
      (should (= 50 (overlay-end right)))
      ;; Inserting at start should make left advance while right and
      ;; root stay, thus we would have left > right .
      (goto-char 25)
      (insert (make-string 25 ?x))
      ;;     [25]
      ;;    /    \
      ;; (50)    (25)
      (should (= 25 (overlay-start root)))
      (should (= 50 (overlay-end root)))
      (should (= 50 (overlay-start left)))
      (should (= 50 (overlay-end left)))
      (should (= 25 (overlay-start right)))
      (should (= 75 (overlay-end right)))
      ;; Try to detect the error, by removing left.  The should fail
      ;; an eassert, since it won't be found by a regular tree
      ;; traversal - in theory.
      (delete-overlay left)
      (should (= 2 (length (overlays-in 1 (point-max))))))))

;; +==========================================================================+
;; | Moving overlays with insert-before-markers
;; +==========================================================================+

(ert-deftest test-overlay-insert-before-markers-at-start ()
  "`insert-before-markers' always advances an overlay's start.
Test both front-advance and non-front-advance overlays."
  (dolist (front-advance '(nil t))
    (ert-info ((format "front-advance %S" front-advance))
      (with-temp-buffer
        (insert "1234")
        (let* ((beg (1+ (point-min)))
               (end (1+ beg))
               (overlay (make-overlay beg end nil front-advance nil)))
          (goto-char beg)
          (insert-before-markers "x")
          (should (equal (1+ beg) (overlay-start overlay)))
          (should (equal (1+ end) (overlay-end overlay))))))))

(ert-deftest test-overlay-insert-before-markers-at-end ()
  "`insert-before-markers' always advances an overlay's end.
Test both rear-advance and non-rear-advance overlays."
  (dolist (rear-advance '(nil t))
    (ert-info ((format "rear-advance %S" rear-advance))
      (with-temp-buffer
        (insert "1234")
        (let* ((beg (1+ (point-min)))
               (end (1+ beg))
               (overlay (make-overlay beg end nil nil rear-advance)))
          (goto-char end)
          (insert-before-markers "x")
          (should (equal beg (overlay-start overlay)))
          (should (equal (1+ end) (overlay-end overlay))))))))

(ert-deftest test-overlay-insert-before-markers-empty ()
  (dolist (advance-args '((nil nil) (t nil) (nil t) (t t)))
    (ert-info ((format "advance args %S" advance-args))
      (with-temp-buffer
        (insert "1234")
        (let* ((pos (1+ (point-min)))
               (overlay (apply #'make-overlay pos pos nil advance-args)))
          (goto-char pos)
          (insert-before-markers "x")
          (should (equal (1+ pos) (overlay-start overlay)))
          (should (equal (1+ pos) (overlay-end overlay))))))))

;; +==========================================================================+
;; | Moving by deletions
;; +==========================================================================+

(defmacro deftest-moving-delete-1 (id beg-end delete sbeg-send fa ra)
  (cl-destructuring-bind (beg end dpos dlen sbeg send fa ra)
      (append beg-end delete sbeg-send (list fa ra) nil)
    `(ert-deftest ,(buffer-tests--make-test-name 'moving-delete 1 id) ()
       (test-with-overlay-in-buffer (ov ,beg ,end ,fa ,ra)
         (should (= ,beg (overlay-start ov)))
         (should (= ,end (overlay-end ov)))
         (delete-region ,dpos (+ ,dpos ,dlen))
         (should (= ,sbeg (overlay-start ov)))
         (should (= ,send (overlay-end ov)))))))

;; non-empty, no fa, no ra
;; --------------------  OV      DEL    RESULT
(deftest-moving-delete-1 A (10 20) (15 3) (10 17) nil nil)
(deftest-moving-delete-1 B (10 20) (20 4) (10 20) nil nil)
(deftest-moving-delete-1 C (10 20) (5 5) (5 15) nil nil)
(deftest-moving-delete-1 D (10 20) (10 3) (10 17) nil nil)
(deftest-moving-delete-1 E (10 20) (20 4) (10 20) nil nil)

;; non-empty no fa, ra
(deftest-moving-delete-1 F (10 20) (15 3) (10 17) nil t)
(deftest-moving-delete-1 G (10 20) (20 4) (10 20) nil t)
(deftest-moving-delete-1 H (10 20) (5 5) (5 15) nil t)
(deftest-moving-delete-1 I (10 20) (10 3) (10 17) nil t)
(deftest-moving-delete-1 J (10 20) (20 4) (10 20) nil t)

;; non-empty, fa, no ra
(deftest-moving-delete-1 K (10 20) (15 3) (10 17) t nil)
(deftest-moving-delete-1 L (10 20) (20 4) (10 20) t nil)
(deftest-moving-delete-1 M (10 20) (5 5) (5 15) t nil)
(deftest-moving-delete-1 N (10 20) (10 3) (10 17) t nil)
(deftest-moving-delete-1 O (10 20) (20 4) (10 20) t nil)

;; non-empty, fa, ra
(deftest-moving-delete-1 P (10 20) (15 3) (10 17) t t)
(deftest-moving-delete-1 Q (10 20) (20 4) (10 20) t t)
(deftest-moving-delete-1 R (10 20) (5 5) (5 15) t t)
(deftest-moving-delete-1 S (10 20) (10 3) (10 17) t t)
(deftest-moving-delete-1 T (10 20) (20 4) (10 20) t t)

;; empty, no fa, no ra
(deftest-moving-delete-1 U (15 15) (20 4) (15 15) nil nil)
(deftest-moving-delete-1 V (15 15) (5 5) (10 10) nil nil)
(deftest-moving-delete-1 W (15 15) (15 3) (15 15) nil nil)

;; empty no fa, ra
(deftest-moving-delete-1 X (15 15) (20 4) (15 15) nil t)
(deftest-moving-delete-1 Y (15 15) (5 5) (10 10) nil t)
(deftest-moving-delete-1 Z (15 15) (15 3) (15 15) nil t)

;; empty, fa, no ra
(deftest-moving-delete-1 a (15 15) (20 4) (15 15) t nil)
(deftest-moving-delete-1 b (15 15) (5 5) (10 10) t nil)
(deftest-moving-delete-1 c (15 15) (15 3) (15 15) t nil)

;; empty, fa, ra
(deftest-moving-delete-1 d (15 15) (20 4) (15 15) t t)
(deftest-moving-delete-1 e (15 15) (5 5) (10 10) t t)
(deftest-moving-delete-1 f (15 15) (15 3) (15 15) t t)


;; +==========================================================================+
;; | make-indirect-buffer
;; +==========================================================================+

;; Check if overlays are cloned/separate from indirect buffer.
(ert-deftest test-make-indirect-buffer-1 ()
  (with-temp-buffer
    (dotimes (_ 10) (make-overlay 1 1))
    (let (indirect clone)
      (unwind-protect
          (progn
            (setq indirect (make-indirect-buffer
                            (current-buffer) "indirect"))
            (with-current-buffer indirect
              (should-not (overlays-in (point-min) (point-max)))
              (dotimes (_ 20) (make-overlay 1 1))
              (should (= 20 (length (overlays-in (point-min) (point-max)))))
              (delete-all-overlays)
              (should-not (overlays-in (point-min) (point-max))))
            (should (= 10 (length (overlays-in (point-min) (point-max)))))
            (setq clone (make-indirect-buffer
                         (current-buffer) "clone" 'clone))
            (with-current-buffer clone
              (should (= 10 (length (overlays-in (point-min) (point-max)))))
              (dotimes (_ 30) (make-overlay 1 1))
              (should (= 40 (length (overlays-in (point-min) (point-max))))))
            ;; back in temp buffer
            (should (= 10 (length (overlays-in (point-min) (point-max)))))
            (with-current-buffer clone
              (mapc #'delete-overlay
                    (seq-take (overlays-in (point-min) (point-max)) 10))
              (should (= 30 (length (overlays-in (point-min) (point-max))))))
            (should (= 10 (length (overlays-in (point-min) (point-max)))))
            (delete-all-overlays)
            (with-current-buffer clone
              (should (= 30 (length (overlays-in (point-min) (point-max)))))))
        (when (buffer-live-p clone)
          (kill-buffer clone))
        (when (buffer-live-p indirect)
          (kill-buffer indirect))))))



;; +==========================================================================+
;; | buffer-swap-text
;; +==========================================================================+

(defmacro buffer-tests--with-temp-buffers (vars &rest body)
  (declare (indent 1) (debug (sexp &rest form)))
  (if (null vars)
      `(progn ,@body)
    `(with-temp-buffer
       (let ((,(car vars) (current-buffer)))
         (buffer-tests--with-temp-buffers ,(cdr vars) ,@body)))))

;; basic
(ert-deftest test-buffer-swap-text-1 ()
  (buffer-tests--with-temp-buffers (buffer other)
    (with-current-buffer buffer
      (let ((ov (make-overlay 1 1)))
        (buffer-swap-text other)
        (should-not (overlays-in 1 1))
        (with-current-buffer other
          (should (overlays-in 1 1))
          (should (eq ov (car (overlays-in 1 1)))))))))

;; properties
(ert-deftest test-buffer-swap-text-2 ()
  (buffer-tests--with-temp-buffers (buffer other)
    (with-current-buffer other
      (overlay-put (make-overlay 1 1) 'buffer 'other))
    (with-current-buffer buffer
      (overlay-put (make-overlay 1 1) 'buffer 'buffer)
      (buffer-swap-text other)
      (should (= 1 (length (overlays-in 1 1))))
      (should (eq (overlay-get (car (overlays-in 1 1)) 'buffer) 'other)))
    (with-current-buffer other
      (should (= 1 (length (overlays-in 1 1))))
      (should (eq (overlay-get (car (overlays-in 1 1)) 'buffer) 'buffer)))))


;; +==========================================================================+
;; | priorities
;; +==========================================================================+

(ert-deftest test-overlay-priorities-1 ()
  (with-temp-buffer
    (insert " ")
    (dotimes (i 10)
      (let ((ov (make-overlay 1 2)))
        (overlay-put ov 'priority i)
        (overlay-put ov 'value i)))
    (should (eq 9 (get-char-property 1 'value)))))

(ert-deftest test-overlay-priorities-2 ()
  (with-temp-buffer
    (insert " ")
    (dotimes (j 10)
      (let* ((i (- 9 j))
             (ov (make-overlay 1 2)))
        (overlay-put ov 'priority i)
        (overlay-put ov 'value i)))
    (should (eq 9 (get-char-property 1 'value)))))

(ert-deftest buffer-tests--overlay-bug58479 ()
  (with-temp-buffer
    (insert "ab")
    (let* ((pos (+ (point-min) 1))
           (ol (make-overlay pos pos)))
      (overlay-put ol 'my-prop 'set)
      (should (null (get-char-property pos 'my-prop))))))


;; +==========================================================================+
;; | Other
;; +==========================================================================+

(defun test-overlay-regions ()
  (sort (mapcar (lambda (ov)
                  (cons (overlay-start ov)
                        (overlay-end ov)))
                (overlays-in (point-min)
                             (point-max)))
        (lambda (o1 o2)
          (or (< (car o1) (car o2))
              (and (= (car o1) (car o2))
                   (< (cdr o1) (cdr o2)))))))

;; This test used to fail.
(ert-deftest overlay-complex-delete-with-offset ()
  (with-temp-buffer
    (let (todelete)
      (insert (make-string 1000 ?\s))
      (make-overlay 1 2 nil t nil)
      (make-overlay 2 3 nil t nil)
      (make-overlay 3 4 nil t nil)
      (make-overlay 4 5 nil t nil)
      (setq todelete (make-overlay 280 287 nil t nil))
      (make-overlay 265 275 nil t nil)
      (make-overlay 329 386 nil t nil)
      (make-overlay 386 390 nil t nil)
      (goto-char 50)
      (delete-char 50)
      (goto-char 1)
      (delete-char 2)
      (delete-overlay todelete)
      (should (equal (test-overlay-regions)
                     '((1 . 1) (1 . 1) (1 . 2) (2 . 3) (213 . 223) (277 . 334) (334 . 338)))))))

;; This test used to fail.
(ert-deftest overlay-complex-insert-1 ()
  (with-temp-buffer
    (insert "          ")
    (make-overlay 8 11 nil nil t)
    (make-overlay 2 7 nil nil nil)
    (make-overlay 2 4 nil t nil)
    (goto-char 1)
    (insert "     ")
    (should (equal (test-overlay-regions)
                   '((7 . 9)
                     (7 . 12)
                     (13 . 16))))))

;; This test used to fail.
(ert-deftest overlay-complex-insert-2 ()
  (with-temp-buffer
    (insert (make-string 100 ?\s))
    (make-overlay 77 7 nil nil t)
    (make-overlay 21 53 nil t t)
    (make-overlay 84 14 nil nil nil)
    (make-overlay 38 69 nil t nil)
    (make-overlay 93 15 nil nil t)
    (make-overlay 73 48 nil t t)
    (make-overlay 96 51 nil t t)
    (make-overlay 6 43 nil t t)
    (make-overlay 15 100 nil t t)
    (make-overlay 22 17 nil nil nil)
    (make-overlay 72 45 nil t nil)
    (make-overlay 2 74 nil nil t)
    (make-overlay 15 29 nil t t)
    (make-overlay 17 34 nil t t)
    (make-overlay 101 66 nil t nil)
    (make-overlay 94 24 nil nil nil)
    (goto-char 78)
    (insert "           ")
    (narrow-to-region 47 19)
    (goto-char 46)
    (widen)
    (narrow-to-region 13 3)
    (goto-char 9)
    (delete-char 0)
    (goto-char 11)
    (insert "           ")
    (goto-char 3)
    (insert "          ")
    (goto-char 8)
    (insert "       ")
    (goto-char 26)
    (insert "  ")
    (goto-char 14)
    (widen)
    (narrow-to-region 71 35)
    (should
     (equal (test-overlay-regions)
            '((2 . 104)
              (23 . 73)
              (24 . 107)
              (44 . 125)
              (45 . 59)
              (45 . 134)
              (45 . 141)
              (47 . 52)
              (47 . 64)
              (51 . 83)
              (54 . 135)
              (68 . 99))))))

(ert-deftest test-overlay-multibyte-transition-1 ()
  (with-temp-buffer
    (set-buffer-multibyte t)
    (insert "ääää")
    ;; aeaeaeae
    ;; 1 2 3 4 5
    ;; 123456789
    (let ((nonempty-bob (make-overlay 1 2))
          (empty-bob    (make-overlay 1 1))
          (empty        (make-overlay 2 2))
          (nonempty     (make-overlay 2 4))
          (nonempty-eob (make-overlay 4 5))
          (empty-eob    (make-overlay 5 5)))
      (set-buffer-multibyte nil)
      (cl-macrolet
          ((ovshould (ov begin end)
             `(should (equal (list (overlay-start ,ov) (overlay-end ,ov))
                             (list ,begin ,end)))))
        (ovshould nonempty-bob 1 3)
        (ovshould empty-bob    1 1)
        (ovshould empty        3 3)
        (ovshould nonempty     3 7)
        (ovshould nonempty-eob 7 9)
        (ovshould empty-eob    9 9)))))

(ert-deftest test-overlay-multibyte-transition-2 ()
  (with-temp-buffer
    (set-buffer-multibyte t)
    (insert "ääää")
    (set-buffer-multibyte nil)
    ;; aeaeaeae
    ;; 1 2 3 4 5
    ;; 123456789
    (let ((nonempty-bob-end (make-overlay 1 2))
          (nonempty-bob-beg (make-overlay 1 3))
          (empty-bob        (make-overlay 1 1))
          (empty-beg        (make-overlay 3 3))
          (empty-end        (make-overlay 2 2))
          (nonempty-beg-beg (make-overlay 3 7))
          (nonempty-beg-end (make-overlay 3 8))
          (nonempty-end-beg (make-overlay 4 7))
          (nonempty-end-end (make-overlay 4 8))
          (nonempty-eob-beg (make-overlay 5 9))
          (nonempty-eob-end (make-overlay 6 9))
          (empty-eob        (make-overlay 9 9)))
      (set-buffer-multibyte t)
      (cl-macrolet
          ((ovshould (ov begin end)
             `(should (equal (list (overlay-start ,ov) (overlay-end ,ov))
                             (list ,begin ,end)))))
        (ovshould nonempty-bob-end 1 2)
        (ovshould nonempty-bob-beg 1 2)
        (ovshould empty-bob        1 1)
        (ovshould empty-beg        2 2)
        (ovshould empty-end        2 2)
        (ovshould nonempty-beg-beg 2 4)
        (ovshould nonempty-beg-end 2 5)
        (ovshould nonempty-end-beg 3 4)
        (ovshould nonempty-end-end 3 5)
        (ovshould nonempty-eob-beg 3 5)
        (ovshould nonempty-eob-end 4 5)
        (ovshould empty-eob        5 5)))))

(ert-deftest test-overlay-randomly ()
  "Exercise overlay code, but perform few assertions.

This test works best when Emacs is configured with
--enable-checking=yes.  This is a little bit like fuzz testing,
except this test has no way to reduce to a minimal failing test
case.  Regardless, by exercising many corner cases bugs can be
found using Emacs' internal consistency assertions."
  (let* (
         ;; The size and slack for the test buffer size.
         (buffer-size-target 1000)
         (buffer-size-slack 200)

         ;; Use up to 100 overlays.  We need not use more to observe
         ;; reasonable variation in the overlay data structures.
         (overlay-count-limit 100)

         ;; This test maintains a vector of overlays.  Each iteration
         ;; may append or erase one overlay.
         (overlays (make-vector overlay-count-limit nil))
         (overlay-count 0)

         ;; The test is either slowly growing or shrinking the overlay
         ;; count.  Deletions still occur while growing, and additions
         ;; still occur while shrinking.  The GROWING variable only
         ;; controls the relative probability of doing one or the
         ;; other.
         (growing t)

         ;; Loop up to 1M times.
         (iteration-count 0)
         (iteration-target 100000))
    (with-temp-buffer
      (while (< (buffer-size) buffer-size-target)
        (insert "Sed ut perspiciatis, unde omnis iste natus error sit voluptatem
accusantium doloremque laudantium, totam rem aperiam eaque ipsa,
quae ab illo inventore veritatis et quasi architecto beatae vitae
dicta sunt, explicabo.  "))

      (while (< iteration-count iteration-target)
        (cl-incf iteration-count)

        ;; Toggle GROWING if we've reached a size boundary.  The idea
        ;; is to initially steadily increase the overlay count, then
        ;; steadily decrease it, then repeat.
        (when (and growing (= overlay-count overlay-count-limit))
          (setq growing nil))
        (when (and (not growing) (= overlay-count 0))
          (setq growing t))

        ;; Create or delete a random overlay according to a
        ;; probability chosen by GROWING.
        (let ((create-overlay (>= (random 100) (if growing 40 60))))
          (cond
           ;; Possibly create a new overlay in a random place in the
           ;; buffer.  We have two easy choices.  We can choose the
           ;; overlay BEGIN randomly, then choose its END among the
           ;; valid remaining buffer posiitions.  Or we could choose
           ;; the overlay width randomly, then choose a valid BEGIN.
           ;; We take the former approach, because the overlay data
           ;; structure is ordered primarily by BEGIN.
           ((and create-overlay (< overlay-count overlay-count-limit))
            (let* ((begin (random (buffer-size)))
                   (end (+ begin (random (- (buffer-size) begin))))
                   (ov (make-overlay begin end nil
                                     (= 0 (random 2)) (= 0 (random 2)))))
              (aset overlays overlay-count ov)
              (cl-incf overlay-count)))
           ((and (not create-overlay) (> overlay-count 0))

            ;; Possibly delete a random overlay.
            (let* ((last-index (1- overlay-count))
                   (index (random overlay-count))
                   (ov (aref overlays index)))
              (when (< index last-index)
                (aset overlays index (aref overlays last-index)))
              (aset overlays last-index nil)
              (cl-decf overlay-count)
              (delete-overlay ov)))))

        ;; Modify the buffer on occasion, which exercises the
        ;; insert/remove gap logic in the overlay implementation.
        (when (and (< (buffer-size) (+ buffer-size-target buffer-size-slack))
                   (zerop (random 10)))
          (goto-char (1+ (random (buffer-size))))
          (insert (+ ?a (random 26))))
        (when (and (> (buffer-size) (- buffer-size-target buffer-size-slack))
                   (zerop (random 10)))
          (goto-char (1+ (random (buffer-size))))
          (delete-char 1))))))



;; +===================================================================================+
;; | Autogenerated insert/delete/narrow tests
;; +===================================================================================+

(when nil ;; Let's comment these out for now.

;; (defun test-overlay-generate-test (name)
;;   (interactive)
;;   (with-temp-buffer
;;     (let ((forms nil)
;;           (buffer-size 64)
;;           (noverlays 16)
;;           (nforms 32)
;;           (dist '(0.5 0.4 0.1)))
;;       (cl-labels ((brand ()
;;                     (+ (point-min)
;;                        (random (1+ (- (point-max) (point-min)))))))
;;         (cl-macrolet ((push-eval (form)
;;                         `(cl-destructuring-bind (&rest args)
;;                              (list ,@(cdr form))
;;                            (push (cons ',(car form) args) forms)
;;                            (apply #',(car form) args))))
;;           (push-eval (insert (make-string buffer-size ?.)))
;;           (dotimes (_ noverlays)
;;             (push-eval (make-overlay (brand) (brand)
;;                                      nil
;;                                      (= 0 (random 2))
;;                                      (= 0 (random 2)))))
;;           (dotimes (_ nforms)
;;             (push-eval (goto-char (brand)))
;;             (pcase (/ (random 100) 100.0)
;;               ((and x (guard (< x (nth 0 dist))))
;;                (push-eval (insert (make-string (random 16) ?.))))
;;               ((and x (guard (< x (+ (nth 0 dist) (nth 1 dist)))))
;;                (push-eval (delete-char (random (1+ (- (point-max) (point)))))))
;;               (_
;;                (push-eval (widen))
;;                (push-eval (narrow-to-region (brand) (brand))))))
;;           `(ert-deftest ,name ()
;;              (with-temp-buffer
;;                ,@(nreverse forms)
;;                (should (equal (test-overlay-regions)
;;                               ',(test-overlay-regions))))))))))

;; (defun test-overlay-generate-tests (n)
;;   (let ((namefmt "overlay-autogenerated-test-%d")
;;         (standard-output (current-buffer))
;;         (print-length nil)
;;         (print-level nil)
;;         (print-quoted t))
;;     (dotimes (i n)
;;       (pp (test-overlay-generate-test (intern (format namefmt i))))
;;       (terpri))))

;; (progn (random "4711") (test-overlay-generate-tests 64))

(ert-deftest overlay-autogenerated-test-0 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 63 7 nil t t)
    (make-overlay 47 9 nil nil nil)
    (make-overlay 50 43 nil nil nil)
    (make-overlay 20 53 nil nil t)
    (make-overlay 62 4 nil nil t)
    (make-overlay 40 27 nil t t)
    (make-overlay 58 44 nil t t)
    (make-overlay 46 38 nil nil nil)
    (make-overlay 51 28 nil t nil)
    (make-overlay 12 53 nil t t)
    (make-overlay 52 60 nil nil nil)
    (make-overlay 13 47 nil nil nil)
    (make-overlay 16 31 nil nil nil)
    (make-overlay 9 48 nil t t)
    (make-overlay 43 29 nil nil t)
    (make-overlay 48 13 nil t nil)
    (goto-char 44)
    (delete-char 15)
    (goto-char 19)
    (widen)
    (narrow-to-region 20 8)
    (goto-char 9)
    (delete-char 3)
    (goto-char 16)
    (insert "..............")
    (goto-char 12)
    (delete-char 15)
    (goto-char 12)
    (delete-char 4)
    (goto-char 12)
    (delete-char 0)
    (goto-char 12)
    (insert "......")
    (goto-char 13)
    (delete-char 5)
    (goto-char 8)
    (insert "...")
    (goto-char 10)
    (insert ".............")
    (goto-char 14)
    (insert ".......")
    (goto-char 25)
    (delete-char 4)
    (goto-char 26)
    (insert "...............")
    (goto-char 27)
    (insert "...")
    (goto-char 29)
    (delete-char 7)
    (goto-char 24)
    (insert "...")
    (goto-char 30)
    (insert "..........")
    (goto-char 29)
    (widen)
    (narrow-to-region 34 41)
    (goto-char 40)
    (delete-char 0)
    (goto-char 35)
    (delete-char 4)
    (goto-char 36)
    (widen)
    (narrow-to-region 80 66)
    (goto-char 74)
    (delete-char 5)
    (goto-char 69)
    (delete-char 5)
    (goto-char 70)
    (widen)
    (narrow-to-region 50 71)
    (goto-char 66)
    (insert "...............")
    (goto-char 54)
    (insert "...............")
    (goto-char 84)
    (insert "....")
    (goto-char 72)
    (insert "...........")
    (goto-char 84)
    (insert "..........")
    (goto-char 102)
    (insert "")
    (goto-char 80)
    (delete-char 25)
    (should
     (equal
      (test-overlay-regions)
      '((4 . 99)
        (7 . 100)
        (48 . 99)
        (48 . 99)
        (48 . 99)
        (49 . 99)
        (49 . 99)
        (51 . 80)
        (51 . 99)
        (80 . 99)
        (80 . 99)
        (80 . 99)
        (99 . 99)
        (99 . 99)
        (99 . 99)
        (99 . 99))))))

(ert-deftest overlay-autogenerated-test-1 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 17 27 nil nil nil)
    (make-overlay 13 28 nil nil t)
    (make-overlay 8 56 nil nil nil)
    (make-overlay 34 64 nil nil nil)
    (make-overlay 51 4 nil t t)
    (make-overlay 1 19 nil nil nil)
    (make-overlay 53 59 nil nil t)
    (make-overlay 25 13 nil nil nil)
    (make-overlay 19 28 nil t nil)
    (make-overlay 33 23 nil t nil)
    (make-overlay 10 46 nil t t)
    (make-overlay 18 39 nil nil nil)
    (make-overlay 1 49 nil t nil)
    (make-overlay 57 21 nil t t)
    (make-overlay 10 58 nil t t)
    (make-overlay 39 49 nil nil t)
    (goto-char 37)
    (delete-char 9)
    (goto-char 3)
    (insert "......")
    (goto-char 38)
    (delete-char 14)
    (goto-char 18)
    (insert "..........")
    (goto-char 53)
    (insert "....")
    (goto-char 49)
    (delete-char 10)
    (goto-char 11)
    (delete-char 12)
    (goto-char 17)
    (delete-char 22)
    (goto-char 8)
    (insert ".")
    (goto-char 16)
    (insert "........")
    (goto-char 16)
    (delete-char 5)
    (goto-char 11)
    (delete-char 0)
    (goto-char 22)
    (insert ".......")
    (goto-char 18)
    (delete-char 11)
    (goto-char 16)
    (delete-char 0)
    (goto-char 9)
    (insert "...........")
    (goto-char 7)
    (insert "...............")
    (goto-char 2)
    (insert ".......")
    (goto-char 21)
    (delete-char 11)
    (goto-char 13)
    (insert "..............")
    (goto-char 17)
    (delete-char 3)
    (goto-char 21)
    (insert "......")
    (goto-char 15)
    (delete-char 32)
    (goto-char 10)
    (insert "........")
    (goto-char 25)
    (widen)
    (narrow-to-region 15 20)
    (goto-char 17)
    (insert ".............")
    (goto-char 22)
    (insert "............")
    (goto-char 21)
    (delete-char 8)
    (goto-char 36)
    (delete-char 1)
    (goto-char 32)
    (delete-char 2)
    (goto-char 21)
    (insert ".....")
    (goto-char 31)
    (insert "......")
    (should
     (equal
      (test-overlay-regions)
      '((1 . 58)
        (1 . 58))))))

(ert-deftest overlay-autogenerated-test-2 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 15 59 nil t t)
    (make-overlay 56 16 nil nil nil)
    (make-overlay 65 51 nil t nil)
    (make-overlay 14 24 nil t nil)
    (make-overlay 28 9 nil t nil)
    (make-overlay 58 50 nil t t)
    (make-overlay 13 32 nil t t)
    (make-overlay 12 21 nil t nil)
    (make-overlay 60 23 nil t nil)
    (make-overlay 39 38 nil nil t)
    (make-overlay 15 64 nil t nil)
    (make-overlay 17 21 nil nil t)
    (make-overlay 46 23 nil t t)
    (make-overlay 19 40 nil t nil)
    (make-overlay 13 48 nil nil t)
    (make-overlay 35 11 nil t nil)
    (goto-char 41)
    (delete-char 19)
    (goto-char 45)
    (insert "......")
    (goto-char 3)
    (delete-char 32)
    (goto-char 19)
    (insert "")
    (goto-char 16)
    (insert "...............")
    (goto-char 2)
    (insert "")
    (goto-char 30)
    (delete-char 0)
    (goto-char 18)
    (delete-char 17)
    (goto-char 2)
    (insert "...............")
    (goto-char 12)
    (insert "...")
    (goto-char 2)
    (insert ".............")
    (goto-char 16)
    (insert ".......")
    (goto-char 15)
    (insert ".......")
    (goto-char 43)
    (insert "......")
    (goto-char 22)
    (insert ".........")
    (goto-char 25)
    (delete-char 1)
    (goto-char 38)
    (insert "...............")
    (goto-char 76)
    (delete-char 3)
    (goto-char 12)
    (delete-char 5)
    (goto-char 70)
    (delete-char 9)
    (goto-char 36)
    (delete-char 4)
    (goto-char 18)
    (insert "...............")
    (goto-char 52)
    (delete-char 14)
    (goto-char 23)
    (insert "..........")
    (goto-char 64)
    (insert "...........")
    (goto-char 68)
    (delete-char 21)
    (goto-char 71)
    (insert "........")
    (goto-char 28)
    (delete-char 43)
    (goto-char 25)
    (insert "....")
    (goto-char 2)
    (insert "...............")
    (goto-char 40)
    (insert "....")
    (goto-char 56)
    (delete-char 2)
    (should
     (equal
      (test-overlay-regions)
      '((51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 51)
        (51 . 58))))))

(ert-deftest overlay-autogenerated-test-3 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 53 38 nil t nil)
    (make-overlay 17 40 nil t t)
    (make-overlay 64 26 nil t t)
    (make-overlay 48 24 nil t nil)
    (make-overlay 21 18 nil nil nil)
    (make-overlay 2 20 nil nil t)
    (make-overlay 43 26 nil t t)
    (make-overlay 56 28 nil t nil)
    (make-overlay 19 51 nil nil nil)
    (make-overlay 39 61 nil t nil)
    (make-overlay 59 12 nil t nil)
    (make-overlay 65 7 nil t nil)
    (make-overlay 41 7 nil t nil)
    (make-overlay 62 50 nil t nil)
    (make-overlay 7 10 nil t t)
    (make-overlay 45 28 nil t nil)
    (goto-char 13)
    (insert "...")
    (goto-char 37)
    (widen)
    (narrow-to-region 2 10)
    (goto-char 8)
    (delete-char 1)
    (goto-char 3)
    (delete-char 6)
    (goto-char 2)
    (insert "...........")
    (goto-char 5)
    (widen)
    (narrow-to-region 55 70)
    (goto-char 55)
    (insert "......")
    (goto-char 64)
    (delete-char 12)
    (goto-char 61)
    (insert ".....")
    (goto-char 64)
    (insert "..............")
    (goto-char 72)
    (delete-char 6)
    (goto-char 63)
    (delete-char 12)
    (goto-char 63)
    (delete-char 2)
    (goto-char 57)
    (insert "..............")
    (goto-char 68)
    (insert "........")
    (goto-char 77)
    (delete-char 6)
    (goto-char 77)
    (insert ".............")
    (goto-char 67)
    (delete-char 0)
    (goto-char 84)
    (insert "........")
    (goto-char 74)
    (delete-char 12)
    (goto-char 78)
    (insert "...")
    (goto-char 80)
    (insert "............")
    (goto-char 69)
    (insert "......")
    (goto-char 89)
    (insert ".")
    (goto-char 56)
    (insert "....")
    (goto-char 100)
    (insert ".............")
    (goto-char 114)
    (delete-char 0)
    (goto-char 61)
    (widen)
    (narrow-to-region 94 50)
    (goto-char 55)
    (insert "............")
    (goto-char 53)
    (insert ".............")
    (goto-char 116)
    (delete-char 3)
    (goto-char 81)
    (insert "...............")
    (should
     (equal
      (test-overlay-regions)
      '((14 . 166)
        (16 . 164)
        (26 . 164)
        (31 . 68)
        (33 . 165)
        (35 . 52)
        (35 . 164)
        (45 . 164)
        (46 . 164))))))

(ert-deftest overlay-autogenerated-test-4 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 25 15 nil nil t)
    (make-overlay 8 13 nil nil nil)
    (make-overlay 45 49 nil t t)
    (make-overlay 22 13 nil t t)
    (make-overlay 34 17 nil nil t)
    (make-overlay 42 15 nil nil t)
    (make-overlay 43 28 nil t t)
    (make-overlay 3 28 nil t nil)
    (make-overlay 32 61 nil nil t)
    (make-overlay 30 64 nil t t)
    (make-overlay 21 39 nil nil t)
    (make-overlay 32 62 nil t nil)
    (make-overlay 25 29 nil t nil)
    (make-overlay 34 43 nil t nil)
    (make-overlay 9 11 nil t nil)
    (make-overlay 21 65 nil nil t)
    (goto-char 21)
    (delete-char 4)
    (goto-char 25)
    (insert "..")
    (goto-char 53)
    (insert "..")
    (goto-char 2)
    (insert "...............")
    (goto-char 42)
    (delete-char 36)
    (goto-char 23)
    (delete-char 12)
    (goto-char 22)
    (widen)
    (narrow-to-region 30 32)
    (goto-char 30)
    (delete-char 0)
    (goto-char 31)
    (delete-char 1)
    (goto-char 31)
    (widen)
    (narrow-to-region 28 27)
    (goto-char 27)
    (delete-char 1)
    (goto-char 27)
    (delete-char 0)
    (goto-char 27)
    (delete-char 0)
    (goto-char 27)
    (insert ".")
    (goto-char 28)
    (insert "......")
    (goto-char 34)
    (delete-char 0)
    (goto-char 27)
    (delete-char 5)
    (goto-char 27)
    (delete-char 1)
    (goto-char 27)
    (insert ".............")
    (goto-char 30)
    (insert "..............")
    (goto-char 37)
    (delete-char 15)
    (goto-char 32)
    (delete-char 2)
    (goto-char 36)
    (delete-char 1)
    (goto-char 34)
    (delete-char 0)
    (goto-char 34)
    (delete-char 1)
    (goto-char 32)
    (widen)
    (narrow-to-region 24 19)
    (goto-char 21)
    (delete-char 1)
    (goto-char 21)
    (widen)
    (narrow-to-region 11 38)
    (goto-char 27)
    (widen)
    (narrow-to-region 20 22)
    (goto-char 20)
    (delete-char 1)
    (goto-char 20)
    (widen)
    (narrow-to-region 36 4)
    (goto-char 26)
    (delete-char 9)
    (should
     (equal
      (test-overlay-regions)
      '((18 . 25)
        (21 . 21)
        (21 . 21)
        (21 . 22)
        (21 . 22)
        (21 . 27)
        (21 . 27)
        (22 . 25)
        (22 . 27)
        (22 . 28)
        (26 . 27))))))

(ert-deftest overlay-autogenerated-test-5 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 64 1 nil nil nil)
    (make-overlay 38 43 nil nil nil)
    (make-overlay 42 19 nil t nil)
    (make-overlay 22 12 nil nil nil)
    (make-overlay 12 30 nil t t)
    (make-overlay 38 46 nil nil nil)
    (make-overlay 18 23 nil nil nil)
    (make-overlay 58 65 nil nil t)
    (make-overlay 52 41 nil nil nil)
    (make-overlay 12 26 nil nil nil)
    (make-overlay 39 4 nil nil nil)
    (make-overlay 20 1 nil nil t)
    (make-overlay 36 60 nil nil nil)
    (make-overlay 24 18 nil t nil)
    (make-overlay 9 50 nil nil nil)
    (make-overlay 19 17 nil t nil)
    (goto-char 40)
    (insert "")
    (goto-char 64)
    (insert ".............")
    (goto-char 32)
    (delete-char 40)
    (goto-char 25)
    (insert "...")
    (goto-char 31)
    (delete-char 1)
    (goto-char 8)
    (delete-char 14)
    (goto-char 20)
    (delete-char 5)
    (goto-char 20)
    (insert "...........")
    (goto-char 20)
    (insert ".........")
    (goto-char 17)
    (widen)
    (narrow-to-region 11 21)
    (goto-char 14)
    (widen)
    (narrow-to-region 9 24)
    (goto-char 24)
    (insert ".............")
    (goto-char 30)
    (widen)
    (narrow-to-region 47 45)
    (goto-char 47)
    (insert ".")
    (goto-char 46)
    (widen)
    (narrow-to-region 30 42)
    (goto-char 32)
    (delete-char 0)
    (goto-char 34)
    (insert ".......")
    (goto-char 42)
    (delete-char 4)
    (goto-char 39)
    (delete-char 6)
    (goto-char 31)
    (delete-char 6)
    (goto-char 31)
    (insert "............")
    (goto-char 30)
    (insert "......")
    (goto-char 50)
    (delete-char 0)
    (goto-char 30)
    (insert "....")
    (goto-char 53)
    (insert "............")
    (goto-char 41)
    (delete-char 12)
    (goto-char 52)
    (insert ".......")
    (goto-char 56)
    (insert "...........")
    (goto-char 68)
    (insert ".......")
    (goto-char 52)
    (insert "......")
    (goto-char 71)
    (delete-char 10)
    (goto-char 47)
    (insert "")
    (should
     (equal
      (test-overlay-regions)
      '((20 . 89))))))

(ert-deftest overlay-autogenerated-test-6 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 28 59 nil nil nil)
    (make-overlay 36 21 nil t t)
    (make-overlay 60 19 nil t nil)
    (make-overlay 26 30 nil t nil)
    (make-overlay 47 27 nil nil t)
    (make-overlay 8 25 nil t t)
    (make-overlay 57 43 nil t t)
    (make-overlay 28 61 nil nil t)
    (make-overlay 42 31 nil nil t)
    (make-overlay 15 44 nil t nil)
    (make-overlay 56 38 nil nil nil)
    (make-overlay 39 44 nil nil t)
    (make-overlay 50 6 nil t nil)
    (make-overlay 6 19 nil t nil)
    (make-overlay 50 44 nil t t)
    (make-overlay 34 60 nil nil t)
    (goto-char 27)
    (insert "...............")
    (goto-char 23)
    (insert "..............")
    (goto-char 50)
    (widen)
    (narrow-to-region 53 67)
    (goto-char 60)
    (delete-char 0)
    (goto-char 54)
    (insert "......")
    (goto-char 64)
    (delete-char 1)
    (goto-char 66)
    (delete-char 3)
    (goto-char 58)
    (insert ".............")
    (goto-char 58)
    (insert ".........")
    (goto-char 76)
    (insert "...........")
    (goto-char 57)
    (insert "....")
    (goto-char 106)
    (widen)
    (narrow-to-region 5 45)
    (goto-char 31)
    (delete-char 8)
    (goto-char 36)
    (insert "...")
    (goto-char 6)
    (insert "........")
    (goto-char 33)
    (insert ".............")
    (goto-char 38)
    (delete-char 3)
    (goto-char 28)
    (delete-char 6)
    (goto-char 42)
    (widen)
    (narrow-to-region 17 25)
    (goto-char 19)
    (insert "..............")
    (goto-char 37)
    (delete-char 1)
    (goto-char 22)
    (delete-char 9)
    (goto-char 28)
    (insert "..............")
    (goto-char 37)
    (delete-char 3)
    (goto-char 18)
    (insert "...............")
    (goto-char 30)
    (widen)
    (narrow-to-region 68 25)
    (goto-char 38)
    (delete-char 22)
    (goto-char 43)
    (widen)
    (narrow-to-region 47 96)
    (goto-char 86)
    (insert ".")
    (goto-char 63)
    (insert "......")
    (goto-char 78)
    (widen)
    (narrow-to-region 61 27)
    (goto-char 43)
    (delete-char 8)
    (should
     (equal
      (test-overlay-regions)
      '((14 . 38)
        (14 . 132)
        (16 . 43)
        (38 . 118)
        (38 . 126)
        (38 . 142)
        (44 . 115)
        (45 . 129))))))

(ert-deftest overlay-autogenerated-test-7 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 13 50 nil t nil)
    (make-overlay 28 44 nil nil t)
    (make-overlay 56 27 nil t nil)
    (make-overlay 8 34 nil nil nil)
    (make-overlay 22 8 nil nil t)
    (make-overlay 8 28 nil t nil)
    (make-overlay 65 31 nil nil t)
    (make-overlay 44 8 nil nil nil)
    (make-overlay 52 64 nil nil t)
    (make-overlay 52 27 nil t t)
    (make-overlay 47 32 nil nil nil)
    (make-overlay 18 62 nil nil nil)
    (make-overlay 18 24 nil t t)
    (make-overlay 33 46 nil nil t)
    (make-overlay 20 8 nil t nil)
    (make-overlay 51 51 nil t nil)
    (goto-char 2)
    (delete-char 46)
    (goto-char 12)
    (delete-char 5)
    (goto-char 2)
    (delete-char 12)
    (goto-char 2)
    (insert "..")
    (goto-char 2)
    (widen)
    (narrow-to-region 2 4)
    (goto-char 4)
    (insert "......")
    (goto-char 4)
    (widen)
    (narrow-to-region 4 6)
    (goto-char 5)
    (insert "")
    (goto-char 6)
    (insert "...............")
    (goto-char 9)
    (insert "...")
    (goto-char 7)
    (delete-char 13)
    (goto-char 8)
    (delete-char 1)
    (goto-char 9)
    (insert "...............")
    (goto-char 24)
    (delete-char 1)
    (goto-char 15)
    (insert "...............")
    (goto-char 16)
    (insert "............")
    (goto-char 17)
    (delete-char 8)
    (goto-char 36)
    (widen)
    (narrow-to-region 47 38)
    (goto-char 43)
    (delete-char 0)
    (goto-char 46)
    (delete-char 0)
    (goto-char 40)
    (delete-char 4)
    (goto-char 39)
    (insert ".......")
    (goto-char 50)
    (delete-char 0)
    (goto-char 47)
    (insert "...........")
    (goto-char 45)
    (insert ".....")
    (goto-char 38)
    (delete-char 3)
    (goto-char 59)
    (delete-char 1)
    (goto-char 42)
    (insert "...............")
    (goto-char 65)
    (insert "...........")
    (goto-char 73)
    (delete-char 13)
    (goto-char 72)
    (insert "....")
    (goto-char 47)
    (insert "..")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 81)
        (2 . 81)
        (2 . 81)
        (2 . 81)
        (2 . 81)
        (81 . 81)
        (81 . 81))))))

(ert-deftest overlay-autogenerated-test-8 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 20 6 nil t nil)
    (make-overlay 48 13 nil t nil)
    (make-overlay 58 65 nil nil t)
    (make-overlay 63 65 nil nil nil)
    (make-overlay 42 40 nil t t)
    (make-overlay 40 6 nil nil t)
    (make-overlay 37 46 nil t nil)
    (make-overlay 4 14 nil nil nil)
    (make-overlay 58 44 nil t t)
    (make-overlay 14 16 nil nil t)
    (make-overlay 31 61 nil t nil)
    (make-overlay 34 3 nil nil nil)
    (make-overlay 11 16 nil t nil)
    (make-overlay 19 42 nil nil t)
    (make-overlay 30 9 nil nil t)
    (make-overlay 63 52 nil t t)
    (goto-char 57)
    (delete-char 2)
    (goto-char 8)
    (insert "........")
    (goto-char 30)
    (insert "...........")
    (goto-char 35)
    (insert "...........")
    (goto-char 66)
    (insert "...............")
    (goto-char 53)
    (delete-char 15)
    (goto-char 75)
    (delete-char 10)
    (goto-char 62)
    (delete-char 21)
    (goto-char 52)
    (delete-char 10)
    (goto-char 10)
    (insert "............")
    (goto-char 42)
    (insert "...........")
    (goto-char 68)
    (insert ".............")
    (goto-char 12)
    (insert "........")
    (goto-char 1)
    (insert "...............")
    (goto-char 89)
    (insert "")
    (goto-char 94)
    (insert ".............")
    (goto-char 57)
    (insert "...........")
    (goto-char 130)
    (insert "...")
    (goto-char 69)
    (insert "..")
    (goto-char 101)
    (insert "......")
    (goto-char 128)
    (delete-char 13)
    (goto-char 19)
    (delete-char 100)
    (goto-char 22)
    (insert "..")
    (goto-char 13)
    (widen)
    (narrow-to-region 30 16)
    (goto-char 19)
    (insert "..........")
    (goto-char 22)
    (delete-char 3)
    (goto-char 19)
    (insert ".........")
    (goto-char 17)
    (insert "..")
    (goto-char 16)
    (insert "............")
    (goto-char 47)
    (insert ".")
    (goto-char 50)
    (insert "..........")
    (goto-char 70)
    (delete-char 1)
    (should
     (equal
      (test-overlay-regions)
      '((32 . 75)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 60)
        (33 . 75)
        (33 . 75)
        (33 . 75)
        (60 . 75))))))

(ert-deftest overlay-autogenerated-test-9 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 58 13 nil nil nil)
    (make-overlay 29 4 nil nil t)
    (make-overlay 3 53 nil nil nil)
    (make-overlay 31 9 nil t t)
    (make-overlay 48 30 nil nil nil)
    (make-overlay 43 50 nil nil nil)
    (make-overlay 7 27 nil nil t)
    (make-overlay 30 59 nil nil nil)
    (make-overlay 42 25 nil nil t)
    (make-overlay 15 13 nil t t)
    (make-overlay 39 11 nil t t)
    (make-overlay 21 62 nil t t)
    (make-overlay 35 2 nil t nil)
    (make-overlay 60 53 nil nil t)
    (make-overlay 64 8 nil nil t)
    (make-overlay 58 59 nil t t)
    (goto-char 28)
    (insert ".............")
    (goto-char 28)
    (insert "...............")
    (goto-char 71)
    (insert ".......")
    (goto-char 65)
    (insert "......")
    (goto-char 3)
    (delete-char 12)
    (goto-char 79)
    (delete-char 11)
    (goto-char 65)
    (widen)
    (narrow-to-region 12 53)
    (goto-char 38)
    (insert ".......")
    (goto-char 20)
    (insert ".........")
    (goto-char 27)
    (insert "...........")
    (goto-char 75)
    (insert "........")
    (goto-char 85)
    (insert "............")
    (goto-char 52)
    (insert "..........")
    (goto-char 16)
    (delete-char 8)
    (goto-char 15)
    (insert "...............")
    (goto-char 112)
    (insert "")
    (goto-char 61)
    (insert "..")
    (goto-char 29)
    (delete-char 34)
    (goto-char 52)
    (delete-char 32)
    (goto-char 43)
    (insert "........")
    (goto-char 45)
    (insert "..")
    (goto-char 35)
    (insert "...........")
    (goto-char 29)
    (insert ".......")
    (goto-char 75)
    (widen)
    (narrow-to-region 69 55)
    (goto-char 67)
    (delete-char 2)
    (goto-char 66)
    (delete-char 0)
    (goto-char 62)
    (delete-char 1)
    (goto-char 61)
    (delete-char 3)
    (goto-char 63)
    (insert ".")
    (goto-char 56)
    (insert ".....")
    (goto-char 67)
    (insert ".............")
    (goto-char 76)
    (delete-char 3)
    (should
     (equal
      (test-overlay-regions)
      '((2 . 90)
        (3 . 90)
        (3 . 90)
        (3 . 99)
        (3 . 117)
        (3 . 117)
        (3 . 120)
        (9 . 118)
        (13 . 102))))))

(ert-deftest overlay-autogenerated-test-10 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 16 60 nil nil nil)
    (make-overlay 36 53 nil nil nil)
    (make-overlay 44 39 nil t t)
    (make-overlay 61 47 nil t t)
    (make-overlay 58 39 nil nil t)
    (make-overlay 23 54 nil nil t)
    (make-overlay 65 59 nil t t)
    (make-overlay 13 57 nil nil t)
    (make-overlay 22 64 nil nil t)
    (make-overlay 16 19 nil nil nil)
    (make-overlay 16 1 nil nil t)
    (make-overlay 28 21 nil t t)
    (make-overlay 10 62 nil nil nil)
    (make-overlay 12 18 nil nil t)
    (make-overlay 15 5 nil nil t)
    (make-overlay 36 31 nil nil t)
    (goto-char 42)
    (insert "...")
    (goto-char 25)
    (delete-char 28)
    (goto-char 30)
    (delete-char 10)
    (goto-char 8)
    (delete-char 9)
    (goto-char 5)
    (insert "........")
    (goto-char 6)
    (delete-char 2)
    (goto-char 4)
    (insert "")
    (goto-char 21)
    (insert ".............")
    (goto-char 6)
    (delete-char 33)
    (goto-char 1)
    (delete-char 1)
    (goto-char 6)
    (insert "..........")
    (goto-char 8)
    (insert "...........")
    (goto-char 21)
    (insert "........")
    (goto-char 16)
    (delete-char 18)
    (goto-char 5)
    (insert "...")
    (goto-char 5)
    (delete-char 8)
    (goto-char 11)
    (insert ".")
    (goto-char 1)
    (insert ".......")
    (goto-char 9)
    (delete-char 9)
    (goto-char 5)
    (insert "")
    (goto-char 8)
    (delete-char 0)
    (goto-char 11)
    (insert "..............")
    (goto-char 12)
    (insert "")
    (goto-char 11)
    (delete-char 8)
    (goto-char 7)
    (delete-char 3)
    (goto-char 5)
    (delete-char 3)
    (goto-char 1)
    (delete-char 8)
    (goto-char 1)
    (insert "....")
    (goto-char 1)
    (insert "..")
    (goto-char 7)
    (insert "...")
    (goto-char 8)
    (widen)
    (narrow-to-region 9 11)
    (goto-char 11)
    (delete-char 0)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 10)
        (1 . 10)
        (1 . 10)
        (1 . 10)
        (1 . 10)
        (1 . 12)
        (1 . 12)
        (1 . 12)
        (10 . 10)
        (10 . 10)
        (10 . 12))))))

(ert-deftest overlay-autogenerated-test-11 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 33 18 nil nil nil)
    (make-overlay 56 38 nil t nil)
    (make-overlay 2 45 nil nil t)
    (make-overlay 19 55 nil nil t)
    (make-overlay 28 42 nil t t)
    (make-overlay 50 29 nil t nil)
    (make-overlay 40 63 nil nil nil)
    (make-overlay 13 2 nil nil t)
    (make-overlay 26 7 nil t t)
    (make-overlay 22 25 nil nil nil)
    (make-overlay 14 14 nil t nil)
    (make-overlay 15 39 nil t t)
    (make-overlay 51 22 nil t t)
    (make-overlay 58 5 nil t nil)
    (make-overlay 16 10 nil nil nil)
    (make-overlay 32 33 nil t nil)
    (goto-char 40)
    (delete-char 20)
    (goto-char 45)
    (delete-char 0)
    (goto-char 6)
    (insert "..")
    (goto-char 45)
    (insert "...")
    (goto-char 26)
    (insert "...............")
    (goto-char 27)
    (insert "...........")
    (goto-char 38)
    (insert "......")
    (goto-char 62)
    (insert "...............")
    (goto-char 18)
    (insert "...........")
    (goto-char 99)
    (widen)
    (narrow-to-region 37 17)
    (goto-char 29)
    (delete-char 2)
    (goto-char 28)
    (delete-char 2)
    (goto-char 17)
    (insert ".....")
    (goto-char 21)
    (widen)
    (narrow-to-region 34 96)
    (goto-char 44)
    (delete-char 22)
    (goto-char 39)
    (insert "..")
    (goto-char 53)
    (insert "...............")
    (goto-char 58)
    (insert ".............")
    (goto-char 93)
    (insert ".........")
    (goto-char 78)
    (widen)
    (narrow-to-region 27 104)
    (goto-char 93)
    (delete-char 11)
    (goto-char 59)
    (insert "....")
    (goto-char 59)
    (insert "..............")
    (goto-char 74)
    (delete-char 5)
    (goto-char 70)
    (insert ".")
    (goto-char 37)
    (insert "...........")
    (goto-char 34)
    (delete-char 46)
    (goto-char 49)
    (insert "......")
    (goto-char 55)
    (insert "...")
    (goto-char 42)
    (insert "...")
    (goto-char 70)
    (delete-char 8)
    (goto-char 48)
    (delete-char 28)
    (should
     (equal
      (test-overlay-regions)
      '((2 . 62)
        (5 . 62)
        (9 . 34)
        (22 . 61)
        (33 . 55)
        (33 . 62)
        (34 . 34)
        (34 . 62))))))

(ert-deftest overlay-autogenerated-test-12 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 18 50 nil nil nil)
    (make-overlay 63 3 nil nil t)
    (make-overlay 44 20 nil t t)
    (make-overlay 58 38 nil nil t)
    (make-overlay 3 17 nil t nil)
    (make-overlay 31 62 nil t nil)
    (make-overlay 12 17 nil t nil)
    (make-overlay 17 52 nil nil nil)
    (make-overlay 9 35 nil nil nil)
    (make-overlay 17 38 nil nil nil)
    (make-overlay 53 54 nil nil t)
    (make-overlay 65 34 nil t nil)
    (make-overlay 12 33 nil t nil)
    (make-overlay 54 58 nil nil nil)
    (make-overlay 42 26 nil t nil)
    (make-overlay 2 4 nil t nil)
    (goto-char 4)
    (delete-char 26)
    (goto-char 39)
    (insert ".")
    (goto-char 2)
    (delete-char 14)
    (goto-char 16)
    (widen)
    (narrow-to-region 19 1)
    (goto-char 7)
    (delete-char 9)
    (goto-char 6)
    (insert ".........")
    (goto-char 6)
    (insert "..........")
    (goto-char 16)
    (insert ".............")
    (goto-char 36)
    (delete-char 1)
    (goto-char 4)
    (insert "..........")
    (goto-char 49)
    (delete-char 2)
    (goto-char 16)
    (insert "............")
    (goto-char 52)
    (widen)
    (narrow-to-region 36 38)
    (goto-char 37)
    (delete-char 1)
    (goto-char 37)
    (insert ".............")
    (goto-char 46)
    (insert ".")
    (goto-char 40)
    (delete-char 5)
    (goto-char 45)
    (delete-char 0)
    (goto-char 46)
    (delete-char 0)
    (goto-char 40)
    (insert "..........")
    (goto-char 39)
    (delete-char 4)
    (goto-char 39)
    (delete-char 3)
    (goto-char 40)
    (widen)
    (narrow-to-region 8 9)
    (goto-char 8)
    (delete-char 1)
    (goto-char 8)
    (delete-char 0)
    (goto-char 8)
    (widen)
    (narrow-to-region 45 15)
    (goto-char 40)
    (insert "...............")
    (goto-char 29)
    (delete-char 7)
    (goto-char 30)
    (delete-char 6)
    (goto-char 21)
    (delete-char 9)
    (goto-char 22)
    (insert "...............")
    (goto-char 51)
    (insert "..............")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 92)
        (2 . 92)
        (2 . 93)
        (2 . 96)
        (2 . 97)
        (2 . 99))))))

(ert-deftest overlay-autogenerated-test-13 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 18 30 nil t t)
    (make-overlay 54 37 nil nil t)
    (make-overlay 16 61 nil nil t)
    (make-overlay 58 7 nil nil t)
    (make-overlay 27 39 nil nil t)
    (make-overlay 39 31 nil nil t)
    (make-overlay 11 47 nil nil nil)
    (make-overlay 47 40 nil t t)
    (make-overlay 27 18 nil nil nil)
    (make-overlay 33 26 nil nil t)
    (make-overlay 55 4 nil t t)
    (make-overlay 62 50 nil t t)
    (make-overlay 47 65 nil t t)
    (make-overlay 17 23 nil nil t)
    (make-overlay 30 31 nil t nil)
    (make-overlay 10 37 nil t nil)
    (goto-char 8)
    (delete-char 6)
    (goto-char 56)
    (delete-char 0)
    (goto-char 28)
    (insert ".........")
    (goto-char 19)
    (insert "..............")
    (goto-char 4)
    (delete-char 28)
    (goto-char 49)
    (delete-char 4)
    (goto-char 2)
    (insert "............")
    (goto-char 10)
    (delete-char 37)
    (goto-char 19)
    (delete-char 2)
    (goto-char 20)
    (delete-char 0)
    (goto-char 16)
    (insert "..")
    (goto-char 8)
    (widen)
    (narrow-to-region 12 3)
    (goto-char 10)
    (delete-char 2)
    (goto-char 9)
    (insert "..")
    (goto-char 12)
    (insert "...............")
    (goto-char 25)
    (insert ".....")
    (goto-char 10)
    (widen)
    (narrow-to-region 42 18)
    (goto-char 20)
    (insert ".......")
    (goto-char 18)
    (insert ".........")
    (goto-char 55)
    (delete-char 3)
    (goto-char 48)
    (insert ".......")
    (goto-char 52)
    (delete-char 6)
    (goto-char 45)
    (delete-char 11)
    (goto-char 27)
    (delete-char 13)
    (goto-char 22)
    (insert "...........")
    (goto-char 19)
    (delete-char 15)
    (goto-char 20)
    (delete-char 0)
    (goto-char 23)
    (widen)
    (narrow-to-region 12 25)
    (goto-char 16)
    (insert "..........")
    (goto-char 25)
    (widen)
    (narrow-to-region 2 38)
    (goto-char 34)
    (delete-char 0)
    (goto-char 31)
    (insert "...............")
    (should
     (equal
      (test-overlay-regions)
      '((12 . 12)
        (12 . 12)
        (12 . 12)
        (12 . 12)
        (12 . 53)
        (12 . 53)
        (12 . 53)
        (12 . 53)
        (12 . 53)
        (12 . 53)
        (12 . 55))))))

(ert-deftest overlay-autogenerated-test-14 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 29 37 nil t nil)
    (make-overlay 15 44 nil nil nil)
    (make-overlay 31 34 nil nil t)
    (make-overlay 35 33 nil t t)
    (make-overlay 4 27 nil t t)
    (make-overlay 37 5 nil nil t)
    (make-overlay 58 19 nil nil t)
    (make-overlay 57 47 nil nil t)
    (make-overlay 49 5 nil t t)
    (make-overlay 21 59 nil t t)
    (make-overlay 42 33 nil t nil)
    (make-overlay 22 16 nil t t)
    (make-overlay 9 51 nil t nil)
    (make-overlay 20 24 nil nil t)
    (make-overlay 21 7 nil t t)
    (make-overlay 58 52 nil t t)
    (goto-char 39)
    (widen)
    (narrow-to-region 55 54)
    (goto-char 54)
    (insert ".............")
    (goto-char 55)
    (insert "............")
    (goto-char 66)
    (delete-char 10)
    (goto-char 62)
    (insert "...............")
    (goto-char 82)
    (delete-char 2)
    (goto-char 82)
    (delete-char 0)
    (goto-char 76)
    (insert "..............")
    (goto-char 60)
    (insert ".............")
    (goto-char 71)
    (insert "...............")
    (goto-char 122)
    (delete-char 0)
    (goto-char 93)
    (delete-char 3)
    (goto-char 108)
    (delete-char 1)
    (goto-char 121)
    (insert "........")
    (goto-char 92)
    (insert "")
    (goto-char 103)
    (insert "..........")
    (goto-char 85)
    (delete-char 13)
    (goto-char 116)
    (delete-char 7)
    (goto-char 103)
    (widen)
    (narrow-to-region 60 27)
    (goto-char 28)
    (delete-char 16)
    (goto-char 35)
    (insert ".......")
    (goto-char 47)
    (insert "........")
    (goto-char 38)
    (delete-char 1)
    (goto-char 43)
    (insert "..........")
    (goto-char 59)
    (insert "........")
    (goto-char 57)
    (insert "........")
    (goto-char 36)
    (insert "...........")
    (goto-char 82)
    (delete-char 11)
    (goto-char 67)
    (insert "..........")
    (goto-char 46)
    (delete-char 1)
    (goto-char 47)
    (insert "......")
    (goto-char 69)
    (delete-char 7)
    (should
     (equal
      (test-overlay-regions)
      '((5 . 28)
        (5 . 33)
        (9 . 35)
        (15 . 28)
        (19 . 154)
        (21 . 155)
        (28 . 28)
        (28 . 28)
        (28 . 28)
        (28 . 28)
        (31 . 153)
        (58 . 154))))))

(ert-deftest overlay-autogenerated-test-15 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 3 19 nil t t)
    (make-overlay 11 18 nil t nil)
    (make-overlay 28 51 nil nil t)
    (make-overlay 29 15 nil t t)
    (make-overlay 46 57 nil t t)
    (make-overlay 26 24 nil nil nil)
    (make-overlay 29 43 nil nil nil)
    (make-overlay 54 29 nil nil nil)
    (make-overlay 34 52 nil t nil)
    (make-overlay 10 32 nil nil nil)
    (make-overlay 28 34 nil nil t)
    (make-overlay 11 43 nil nil nil)
    (make-overlay 18 50 nil t t)
    (make-overlay 28 39 nil nil nil)
    (make-overlay 62 62 nil t t)
    (make-overlay 30 62 nil t nil)
    (goto-char 30)
    (widen)
    (narrow-to-region 6 22)
    (goto-char 9)
    (insert "..")
    (goto-char 12)
    (insert ".............")
    (goto-char 29)
    (insert "..............")
    (goto-char 47)
    (insert "........")
    (goto-char 46)
    (insert ".............")
    (goto-char 55)
    (insert "..........")
    (goto-char 62)
    (insert "...............")
    (goto-char 47)
    (delete-char 49)
    (goto-char 11)
    (insert "...........")
    (goto-char 40)
    (delete-char 1)
    (goto-char 27)
    (insert "..............")
    (goto-char 51)
    (insert "......")
    (goto-char 60)
    (delete-char 10)
    (goto-char 37)
    (insert ".........")
    (goto-char 69)
    (insert ".")
    (goto-char 36)
    (insert "............")
    (goto-char 75)
    (insert ".............")
    (goto-char 21)
    (widen)
    (narrow-to-region 44 21)
    (goto-char 37)
    (insert ".............")
    (goto-char 55)
    (widen)
    (narrow-to-region 84 28)
    (goto-char 58)
    (widen)
    (narrow-to-region 96 49)
    (goto-char 62)
    (delete-char 0)
    (goto-char 72)
    (delete-char 24)
    (goto-char 61)
    (widen)
    (narrow-to-region 105 83)
    (goto-char 96)
    (widen)
    (narrow-to-region 109 46)
    (goto-char 95)
    (delete-char 4)
    (goto-char 81)
    (insert ".")
    (goto-char 51)
    (delete-char 8)
    (goto-char 52)
    (insert ".")
    (goto-char 60)
    (delete-char 10)
    (goto-char 50)
    (insert "......")
    (should
     (equal
      (test-overlay-regions)
      '((3 . 81)
        (23 . 88)
        (66 . 99)
        (69 . 81)
        (78 . 85)
        (81 . 106)
        (84 . 85)
        (85 . 90)
        (85 . 95)
        (85 . 99)
        (85 . 107)
        (85 . 110)
        (86 . 118)
        (90 . 108))))))

(ert-deftest overlay-autogenerated-test-16 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 3 55 nil t nil)
    (make-overlay 45 47 nil nil nil)
    (make-overlay 23 57 nil t t)
    (make-overlay 64 55 nil nil nil)
    (make-overlay 37 26 nil t t)
    (make-overlay 29 38 nil nil t)
    (make-overlay 33 3 nil t t)
    (make-overlay 49 16 nil t nil)
    (make-overlay 35 56 nil t t)
    (make-overlay 9 39 nil nil nil)
    (make-overlay 2 61 nil nil nil)
    (make-overlay 59 26 nil nil t)
    (make-overlay 5 50 nil t t)
    (make-overlay 19 19 nil nil t)
    (make-overlay 64 21 nil t nil)
    (make-overlay 21 8 nil nil t)
    (goto-char 17)
    (insert ".....")
    (goto-char 29)
    (insert "............")
    (goto-char 42)
    (delete-char 38)
    (goto-char 24)
    (insert "")
    (goto-char 9)
    (delete-char 2)
    (goto-char 20)
    (insert "..")
    (goto-char 27)
    (delete-char 8)
    (goto-char 25)
    (delete-char 6)
    (goto-char 8)
    (delete-char 21)
    (goto-char 9)
    (insert "..............")
    (goto-char 3)
    (insert "....")
    (goto-char 8)
    (delete-char 18)
    (goto-char 6)
    (widen)
    (narrow-to-region 5 8)
    (goto-char 5)
    (delete-char 3)
    (goto-char 5)
    (insert "...")
    (goto-char 8)
    (insert "..........")
    (goto-char 5)
    (insert "")
    (goto-char 7)
    (delete-char 8)
    (goto-char 8)
    (widen)
    (narrow-to-region 2 2)
    (goto-char 2)
    (delete-char 0)
    (goto-char 2)
    (delete-char 0)
    (goto-char 2)
    (delete-char 0)
    (goto-char 2)
    (delete-char 0)
    (goto-char 2)
    (widen)
    (narrow-to-region 10 3)
    (goto-char 8)
    (delete-char 2)
    (goto-char 7)
    (insert ".......")
    (goto-char 8)
    (delete-char 3)
    (goto-char 12)
    (insert "..")
    (goto-char 9)
    (delete-char 2)
    (goto-char 7)
    (insert "......")
    (goto-char 15)
    (insert "..........")
    (goto-char 4)
    (insert "........")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 13)
        (13 . 13)
        (13 . 13)
        (13 . 13)
        (13 . 13)
        (13 . 13)
        (13 . 13)
        (13 . 36)
        (13 . 36)
        (13 . 36)
        (13 . 36))))))

(ert-deftest overlay-autogenerated-test-17 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 15 37 nil t nil)
    (make-overlay 40 3 nil t t)
    (make-overlay 61 19 nil t t)
    (make-overlay 46 9 nil nil t)
    (make-overlay 64 39 nil nil t)
    (make-overlay 50 58 nil nil t)
    (make-overlay 21 30 nil t nil)
    (make-overlay 44 54 nil t nil)
    (make-overlay 32 2 nil t nil)
    (make-overlay 14 9 nil t t)
    (make-overlay 41 40 nil t nil)
    (make-overlay 17 26 nil t nil)
    (make-overlay 57 50 nil t t)
    (make-overlay 16 65 nil nil t)
    (make-overlay 13 61 nil t t)
    (make-overlay 39 64 nil nil t)
    (goto-char 37)
    (widen)
    (narrow-to-region 12 1)
    (goto-char 12)
    (insert "......")
    (goto-char 8)
    (delete-char 4)
    (goto-char 11)
    (delete-char 3)
    (goto-char 6)
    (insert ".....")
    (goto-char 6)
    (widen)
    (narrow-to-region 53 48)
    (goto-char 48)
    (delete-char 5)
    (goto-char 48)
    (widen)
    (narrow-to-region 59 58)
    (goto-char 59)
    (delete-char 0)
    (goto-char 58)
    (insert "...")
    (goto-char 60)
    (insert "...............")
    (goto-char 58)
    (insert ".............")
    (goto-char 67)
    (insert ".....")
    (goto-char 73)
    (insert "")
    (goto-char 68)
    (insert ".....")
    (goto-char 64)
    (insert "....")
    (goto-char 62)
    (insert "..")
    (goto-char 91)
    (insert "..........")
    (goto-char 80)
    (insert "............")
    (goto-char 100)
    (delete-char 21)
    (goto-char 74)
    (insert "...")
    (goto-char 60)
    (delete-char 30)
    (goto-char 64)
    (widen)
    (narrow-to-region 71 23)
    (goto-char 53)
    (delete-char 11)
    (goto-char 23)
    (delete-char 21)
    (goto-char 39)
    (delete-char 0)
    (goto-char 35)
    (insert "")
    (goto-char 35)
    (insert ".........")
    (goto-char 30)
    (insert "...........")
    (goto-char 35)
    (insert "..")
    (goto-char 37)
    (delete-char 1)
    (goto-char 28)
    (delete-char 3)
    (should
     (equal
      (test-overlay-regions)
      '((13 . 27)
        (17 . 67)
        (20 . 71)
        (23 . 23)
        (23 . 24)
        (23 . 67)
        (23 . 70)
        (23 . 70)
        (27 . 41)
        (28 . 41)
        (28 . 41))))))

(ert-deftest overlay-autogenerated-test-18 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 43 52 nil nil t)
    (make-overlay 27 29 nil nil t)
    (make-overlay 24 18 nil nil nil)
    (make-overlay 39 52 nil nil nil)
    (make-overlay 33 62 nil t t)
    (make-overlay 16 7 nil t nil)
    (make-overlay 47 39 nil nil t)
    (make-overlay 59 41 nil nil nil)
    (make-overlay 22 55 nil nil t)
    (make-overlay 60 16 nil t t)
    (make-overlay 55 20 nil nil t)
    (make-overlay 25 12 nil nil t)
    (make-overlay 26 2 nil nil t)
    (make-overlay 17 35 nil nil t)
    (make-overlay 46 41 nil t nil)
    (make-overlay 57 53 nil t t)
    (goto-char 52)
    (insert "")
    (goto-char 4)
    (delete-char 21)
    (goto-char 17)
    (insert "")
    (goto-char 35)
    (insert "...............")
    (goto-char 8)
    (insert "...............")
    (goto-char 9)
    (insert "........")
    (goto-char 73)
    (delete-char 9)
    (goto-char 62)
    (insert "...............")
    (goto-char 27)
    (widen)
    (narrow-to-region 34 84)
    (goto-char 81)
    (insert "...........")
    (goto-char 48)
    (insert "...")
    (goto-char 74)
    (insert ".......")
    (goto-char 41)
    (widen)
    (narrow-to-region 37 105)
    (goto-char 75)
    (insert "...............")
    (goto-char 47)
    (insert "..........")
    (goto-char 99)
    (delete-char 13)
    (goto-char 105)
    (delete-char 4)
    (goto-char 94)
    (delete-char 5)
    (goto-char 96)
    (insert "..............")
    (goto-char 74)
    (insert "")
    (goto-char 121)
    (insert "...")
    (goto-char 102)
    (insert "...")
    (goto-char 64)
    (insert "......")
    (goto-char 67)
    (insert "...")
    (goto-char 95)
    (delete-char 19)
    (goto-char 37)
    (insert "..........")
    (goto-char 50)
    (widen)
    (narrow-to-region 67 96)
    (goto-char 88)
    (insert "..........")
    (goto-char 91)
    (insert ".............")
    (goto-char 70)
    (delete-char 8)
    (goto-char 111)
    (widen)
    (narrow-to-region 72 103)
    (goto-char 101)
    (insert "...............")
    (should
     (equal
      (test-overlay-regions)
      '((4 . 119)
        (4 . 119)
        (4 . 162)
        (35 . 162)
        (51 . 78)
        (53 . 162)
        (55 . 78)
        (79 . 162))))))

(ert-deftest overlay-autogenerated-test-19 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 19 31 nil t t)
    (make-overlay 40 5 nil nil nil)
    (make-overlay 13 41 nil t t)
    (make-overlay 41 43 nil nil t)
    (make-overlay 7 60 nil t nil)
    (make-overlay 40 23 nil t nil)
    (make-overlay 32 15 nil t t)
    (make-overlay 12 45 nil nil nil)
    (make-overlay 18 1 nil nil nil)
    (make-overlay 58 32 nil t t)
    (make-overlay 30 3 nil t t)
    (make-overlay 43 61 nil t nil)
    (make-overlay 54 57 nil nil t)
    (make-overlay 34 14 nil t t)
    (make-overlay 26 49 nil nil t)
    (make-overlay 54 49 nil nil t)
    (goto-char 28)
    (insert "........")
    (goto-char 32)
    (insert "...........")
    (goto-char 78)
    (delete-char 6)
    (goto-char 37)
    (delete-char 0)
    (goto-char 49)
    (insert ".........")
    (goto-char 40)
    (widen)
    (narrow-to-region 8 30)
    (goto-char 20)
    (delete-char 4)
    (goto-char 23)
    (delete-char 1)
    (goto-char 10)
    (insert ".")
    (goto-char 22)
    (delete-char 2)
    (goto-char 22)
    (insert "......")
    (goto-char 17)
    (insert "..........")
    (goto-char 34)
    (delete-char 0)
    (goto-char 21)
    (insert "............")
    (goto-char 45)
    (delete-char 7)
    (goto-char 39)
    (insert "...............")
    (goto-char 29)
    (insert "........")
    (goto-char 9)
    (delete-char 3)
    (goto-char 63)
    (delete-char 1)
    (goto-char 33)
    (insert "........")
    (goto-char 16)
    (delete-char 36)
    (goto-char 20)
    (delete-char 2)
    (goto-char 28)
    (delete-char 0)
    (goto-char 24)
    (insert "...........")
    (goto-char 43)
    (insert "..........")
    (goto-char 30)
    (delete-char 1)
    (goto-char 40)
    (delete-char 13)
    (goto-char 22)
    (delete-char 19)
    (goto-char 10)
    (delete-char 8)
    (goto-char 14)
    (delete-char 0)
    (goto-char 12)
    (delete-char 2)
    (goto-char 11)
    (delete-char 0)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 12)
        (3 . 40)
        (5 . 50)
        (7 . 69)
        (10 . 42)
        (10 . 44)
        (10 . 51)
        (10 . 55))))))

(ert-deftest overlay-autogenerated-test-20 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 44 42 nil t t)
    (make-overlay 47 1 nil nil nil)
    (make-overlay 24 48 nil nil nil)
    (make-overlay 62 50 nil nil t)
    (make-overlay 54 38 nil nil nil)
    (make-overlay 3 9 nil nil nil)
    (make-overlay 61 28 nil t nil)
    (make-overlay 33 33 nil nil t)
    (make-overlay 37 37 nil t nil)
    (make-overlay 20 13 nil nil t)
    (make-overlay 54 36 nil t nil)
    (make-overlay 18 58 nil nil t)
    (make-overlay 55 3 nil nil t)
    (make-overlay 23 21 nil t t)
    (make-overlay 47 55 nil t t)
    (make-overlay 50 12 nil nil nil)
    (goto-char 11)
    (delete-char 46)
    (goto-char 7)
    (delete-char 3)
    (goto-char 14)
    (delete-char 1)
    (goto-char 14)
    (insert "......")
    (goto-char 14)
    (delete-char 4)
    (goto-char 12)
    (widen)
    (narrow-to-region 11 12)
    (goto-char 11)
    (insert "...")
    (goto-char 13)
    (delete-char 1)
    (goto-char 14)
    (insert ".")
    (goto-char 13)
    (delete-char 2)
    (goto-char 11)
    (delete-char 2)
    (goto-char 11)
    (insert "")
    (goto-char 11)
    (delete-char 0)
    (goto-char 11)
    (delete-char 0)
    (goto-char 11)
    (delete-char 0)
    (goto-char 11)
    (insert ".")
    (goto-char 11)
    (insert ".")
    (goto-char 12)
    (insert "......")
    (goto-char 14)
    (delete-char 2)
    (goto-char 11)
    (delete-char 2)
    (goto-char 14)
    (insert "............")
    (goto-char 19)
    (insert "..............")
    (goto-char 29)
    (insert ".....")
    (goto-char 42)
    (delete-char 1)
    (goto-char 22)
    (insert ".....")
    (goto-char 19)
    (insert "..............")
    (goto-char 42)
    (insert ".....")
    (goto-char 63)
    (widen)
    (narrow-to-region 26 42)
    (goto-char 36)
    (insert "..........")
    (goto-char 40)
    (delete-char 11)
    (goto-char 26)
    (delete-char 13)
    (goto-char 28)
    (delete-char 0)
    (should
     (equal
      (test-overlay-regions)
      '((8 . 56))))))

(ert-deftest overlay-autogenerated-test-21 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 65 15 nil nil nil)
    (make-overlay 52 31 nil nil nil)
    (make-overlay 12 51 nil t t)
    (make-overlay 42 20 nil nil t)
    (make-overlay 51 48 nil nil nil)
    (make-overlay 59 28 nil t t)
    (make-overlay 51 53 nil t nil)
    (make-overlay 50 59 nil nil t)
    (make-overlay 24 40 nil t nil)
    (make-overlay 51 61 nil nil nil)
    (make-overlay 12 58 nil nil t)
    (make-overlay 64 17 nil t t)
    (make-overlay 26 38 nil t t)
    (make-overlay 23 36 nil nil nil)
    (make-overlay 57 50 nil nil nil)
    (make-overlay 42 15 nil nil t)
    (goto-char 14)
    (insert "............")
    (goto-char 37)
    (insert ".")
    (goto-char 73)
    (insert "..........")
    (goto-char 17)
    (delete-char 31)
    (goto-char 21)
    (delete-char 35)
    (goto-char 9)
    (delete-char 0)
    (goto-char 7)
    (delete-char 2)
    (goto-char 1)
    (insert "")
    (goto-char 5)
    (insert ".......")
    (goto-char 8)
    (insert "....")
    (goto-char 27)
    (delete-char 0)
    (goto-char 10)
    (insert ".............")
    (goto-char 24)
    (delete-char 16)
    (goto-char 14)
    (insert ".............")
    (goto-char 25)
    (delete-char 11)
    (goto-char 3)
    (insert "........")
    (goto-char 38)
    (insert "............")
    (goto-char 41)
    (insert "..............")
    (goto-char 56)
    (delete-char 3)
    (goto-char 15)
    (widen)
    (narrow-to-region 16 53)
    (goto-char 19)
    (widen)
    (narrow-to-region 18 33)
    (goto-char 32)
    (insert "......")
    (goto-char 38)
    (delete-char 1)
    (goto-char 19)
    (widen)
    (narrow-to-region 11 11)
    (goto-char 11)
    (insert ".........")
    (goto-char 11)
    (insert ".........")
    (goto-char 20)
    (widen)
    (narrow-to-region 22 69)
    (goto-char 49)
    (insert ".........")
    (goto-char 54)
    (delete-char 22)
    (goto-char 44)
    (insert "........")
    (goto-char 40)
    (delete-char 7)
    (goto-char 29)
    (delete-char 22)
    (should
     (equal
      (test-overlay-regions)
      '((33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33)
        (33 . 33))))))

(ert-deftest overlay-autogenerated-test-22 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 12 14 nil nil t)
    (make-overlay 54 7 nil nil t)
    (make-overlay 8 3 nil nil nil)
    (make-overlay 42 32 nil nil nil)
    (make-overlay 10 27 nil t t)
    (make-overlay 50 28 nil t t)
    (make-overlay 39 35 nil nil nil)
    (make-overlay 12 4 nil t t)
    (make-overlay 29 54 nil nil nil)
    (make-overlay 14 52 nil t t)
    (make-overlay 9 15 nil t nil)
    (make-overlay 44 11 nil nil nil)
    (make-overlay 46 29 nil t t)
    (make-overlay 40 58 nil t t)
    (make-overlay 40 61 nil t nil)
    (make-overlay 13 59 nil nil t)
    (goto-char 32)
    (insert ".............")
    (goto-char 25)
    (delete-char 10)
    (goto-char 3)
    (insert ".............")
    (goto-char 33)
    (delete-char 32)
    (goto-char 39)
    (widen)
    (narrow-to-region 41 46)
    (goto-char 43)
    (delete-char 2)
    (goto-char 42)
    (delete-char 2)
    (goto-char 42)
    (insert "...")
    (goto-char 43)
    (delete-char 1)
    (goto-char 42)
    (widen)
    (narrow-to-region 8 46)
    (goto-char 25)
    (delete-char 7)
    (goto-char 12)
    (delete-char 10)
    (goto-char 23)
    (insert "...............")
    (goto-char 41)
    (delete-char 3)
    (goto-char 17)
    (insert ".........")
    (goto-char 37)
    (insert "...............")
    (goto-char 53)
    (delete-char 7)
    (goto-char 53)
    (delete-char 0)
    (goto-char 42)
    (widen)
    (narrow-to-region 20 54)
    (goto-char 20)
    (delete-char 28)
    (goto-char 23)
    (insert "..........")
    (goto-char 30)
    (insert "......")
    (goto-char 26)
    (delete-char 1)
    (goto-char 27)
    (widen)
    (narrow-to-region 40 37)
    (goto-char 37)
    (insert ".....")
    (goto-char 41)
    (widen)
    (narrow-to-region 13 37)
    (goto-char 29)
    (insert "...........")
    (goto-char 33)
    (delete-char 7)
    (goto-char 33)
    (delete-char 8)
    (goto-char 20)
    (insert "")
    (goto-char 23)
    (delete-char 7)
    (goto-char 14)
    (widen)
    (narrow-to-region 33 33)
    (should
     (equal
      (test-overlay-regions)
      '((15 . 39)
        (16 . 38)
        (16 . 39))))))

(ert-deftest overlay-autogenerated-test-23 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 51 32 nil t t)
    (make-overlay 13 61 nil t nil)
    (make-overlay 47 19 nil nil t)
    (make-overlay 11 30 nil nil nil)
    (make-overlay 50 26 nil t t)
    (make-overlay 64 13 nil t t)
    (make-overlay 29 8 nil t t)
    (make-overlay 25 42 nil t t)
    (make-overlay 33 28 nil t t)
    (make-overlay 54 7 nil nil nil)
    (make-overlay 30 59 nil nil nil)
    (make-overlay 65 50 nil t t)
    (make-overlay 64 15 nil t nil)
    (make-overlay 16 35 nil nil nil)
    (make-overlay 40 36 nil nil t)
    (make-overlay 31 35 nil t nil)
    (goto-char 61)
    (insert "......")
    (goto-char 55)
    (delete-char 2)
    (goto-char 20)
    (insert "..............")
    (goto-char 56)
    (insert "............")
    (goto-char 48)
    (delete-char 6)
    (goto-char 9)
    (delete-char 54)
    (goto-char 20)
    (delete-char 2)
    (goto-char 16)
    (delete-char 12)
    (goto-char 18)
    (insert ".............")
    (goto-char 24)
    (delete-char 7)
    (goto-char 5)
    (delete-char 2)
    (goto-char 1)
    (insert ".......")
    (goto-char 1)
    (insert ".......")
    (goto-char 33)
    (insert "")
    (goto-char 4)
    (insert "..")
    (goto-char 5)
    (widen)
    (narrow-to-region 17 4)
    (goto-char 13)
    (insert ".")
    (goto-char 8)
    (insert "............")
    (goto-char 9)
    (delete-char 3)
    (goto-char 4)
    (widen)
    (narrow-to-region 32 32)
    (goto-char 32)
    (delete-char 0)
    (goto-char 32)
    (delete-char 0)
    (goto-char 32)
    (delete-char 0)
    (goto-char 32)
    (insert "...............")
    (goto-char 43)
    (delete-char 4)
    (goto-char 32)
    (delete-char 1)
    (goto-char 40)
    (widen)
    (narrow-to-region 33 19)
    (goto-char 27)
    (insert "........")
    (goto-char 38)
    (delete-char 2)
    (goto-char 26)
    (insert "")
    (goto-char 33)
    (delete-char 1)
    (goto-char 27)
    (insert ".")
    (should
     (equal
      (test-overlay-regions)
      '((38 . 56))))))

(ert-deftest overlay-autogenerated-test-24 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 63 8 nil t t)
    (make-overlay 10 13 nil nil t)
    (make-overlay 40 38 nil nil nil)
    (make-overlay 21 34 nil t t)
    (make-overlay 55 29 nil nil nil)
    (make-overlay 36 65 nil t t)
    (make-overlay 29 12 nil t nil)
    (make-overlay 41 3 nil nil t)
    (make-overlay 20 9 nil t t)
    (make-overlay 52 42 nil t t)
    (make-overlay 21 56 nil nil t)
    (make-overlay 25 65 nil nil nil)
    (make-overlay 38 4 nil t t)
    (make-overlay 48 23 nil t t)
    (make-overlay 52 9 nil nil t)
    (make-overlay 48 19 nil nil nil)
    (goto-char 43)
    (delete-char 8)
    (goto-char 30)
    (delete-char 16)
    (goto-char 7)
    (insert "...")
    (goto-char 14)
    (delete-char 5)
    (goto-char 36)
    (delete-char 0)
    (goto-char 9)
    (insert "...............")
    (goto-char 13)
    (delete-char 17)
    (goto-char 16)
    (delete-char 2)
    (goto-char 9)
    (insert "")
    (goto-char 11)
    (delete-char 5)
    (goto-char 18)
    (insert "........")
    (goto-char 15)
    (insert "....")
    (goto-char 16)
    (delete-char 14)
    (goto-char 20)
    (insert ".")
    (goto-char 25)
    (delete-char 1)
    (goto-char 14)
    (delete-char 14)
    (goto-char 3)
    (delete-char 7)
    (goto-char 3)
    (delete-char 4)
    (goto-char 1)
    (insert "...........")
    (goto-char 9)
    (insert ".......")
    (goto-char 5)
    (delete-char 7)
    (goto-char 12)
    (insert ".........")
    (goto-char 2)
    (delete-char 4)
    (goto-char 3)
    (widen)
    (narrow-to-region 14 6)
    (goto-char 9)
    (insert "..........")
    (goto-char 13)
    (delete-char 8)
    (goto-char 7)
    (delete-char 7)
    (goto-char 7)
    (insert "..")
    (goto-char 9)
    (insert ".............")
    (goto-char 9)
    (insert "..........")
    (goto-char 21)
    (insert "...............")
    (goto-char 42)
    (insert ".........")
    (should
     (equal
      (test-overlay-regions)
      'nil))))

(ert-deftest overlay-autogenerated-test-25 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 24 8 nil nil t)
    (make-overlay 41 16 nil t nil)
    (make-overlay 3 16 nil nil nil)
    (make-overlay 26 42 nil nil nil)
    (make-overlay 32 45 nil nil t)
    (make-overlay 34 19 nil nil nil)
    (make-overlay 37 54 nil nil t)
    (make-overlay 44 34 nil t nil)
    (make-overlay 49 40 nil t t)
    (make-overlay 29 34 nil t nil)
    (make-overlay 54 16 nil t t)
    (make-overlay 29 4 nil t nil)
    (make-overlay 44 57 nil nil nil)
    (make-overlay 5 32 nil nil nil)
    (make-overlay 12 33 nil nil t)
    (make-overlay 38 29 nil t nil)
    (goto-char 12)
    (delete-char 53)
    (goto-char 1)
    (delete-char 6)
    (goto-char 5)
    (widen)
    (narrow-to-region 6 1)
    (goto-char 6)
    (insert "......")
    (goto-char 10)
    (insert "...............")
    (goto-char 17)
    (delete-char 5)
    (goto-char 7)
    (insert ".....")
    (goto-char 8)
    (insert "...............")
    (goto-char 4)
    (insert ".....")
    (goto-char 44)
    (widen)
    (narrow-to-region 18 11)
    (goto-char 15)
    (delete-char 1)
    (goto-char 17)
    (delete-char 0)
    (goto-char 13)
    (delete-char 3)
    (goto-char 14)
    (insert "..")
    (goto-char 16)
    (insert "..")
    (goto-char 15)
    (delete-char 3)
    (goto-char 13)
    (delete-char 0)
    (goto-char 14)
    (insert "..........")
    (goto-char 19)
    (insert ".")
    (goto-char 23)
    (delete-char 1)
    (goto-char 12)
    (widen)
    (narrow-to-region 23 40)
    (goto-char 35)
    (insert "....")
    (goto-char 33)
    (insert "..........")
    (goto-char 37)
    (delete-char 16)
    (goto-char 37)
    (delete-char 0)
    (goto-char 23)
    (widen)
    (narrow-to-region 30 8)
    (goto-char 29)
    (delete-char 0)
    (goto-char 15)
    (delete-char 15)
    (goto-char 9)
    (insert "...........")
    (goto-char 9)
    (delete-char 1)
    (goto-char 22)
    (delete-char 3)
    (goto-char 10)
    (insert ".........")
    (should
     (equal
      (test-overlay-regions)
      '((1 . 30)
        (1 . 30)
        (1 . 30)
        (2 . 53)
        (30 . 30)
        (30 . 30)
        (30 . 30)
        (30 . 30)
        (30 . 30)
        (30 . 30)
        (30 . 30)
        (30 . 53)
        (30 . 53)
        (30 . 53))))))

(ert-deftest overlay-autogenerated-test-26 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 60 59 nil t nil)
    (make-overlay 18 11 nil nil t)
    (make-overlay 4 44 nil nil nil)
    (make-overlay 7 22 nil nil nil)
    (make-overlay 54 50 nil t nil)
    (make-overlay 59 28 nil nil nil)
    (make-overlay 49 23 nil nil t)
    (make-overlay 21 5 nil t nil)
    (make-overlay 17 39 nil t nil)
    (make-overlay 16 14 nil nil nil)
    (make-overlay 50 26 nil nil nil)
    (make-overlay 37 14 nil nil nil)
    (make-overlay 6 59 nil nil t)
    (make-overlay 30 17 nil nil t)
    (make-overlay 17 34 nil nil t)
    (make-overlay 7 22 nil t nil)
    (goto-char 35)
    (delete-char 25)
    (goto-char 30)
    (delete-char 7)
    (goto-char 25)
    (widen)
    (narrow-to-region 3 19)
    (goto-char 6)
    (insert ".........")
    (goto-char 21)
    (insert "...............")
    (goto-char 12)
    (insert ".............")
    (goto-char 34)
    (widen)
    (narrow-to-region 64 37)
    (goto-char 62)
    (insert ".............")
    (goto-char 50)
    (widen)
    (narrow-to-region 72 38)
    (goto-char 66)
    (insert "")
    (goto-char 54)
    (insert "...")
    (goto-char 70)
    (delete-char 4)
    (goto-char 49)
    (delete-char 13)
    (goto-char 38)
    (insert "....")
    (goto-char 46)
    (insert ".")
    (goto-char 43)
    (widen)
    (narrow-to-region 74 53)
    (goto-char 60)
    (delete-char 10)
    (goto-char 53)
    (insert "..............")
    (goto-char 72)
    (insert "............")
    (goto-char 87)
    (delete-char 2)
    (goto-char 73)
    (insert "............")
    (goto-char 81)
    (insert "........")
    (goto-char 106)
    (insert "...")
    (goto-char 95)
    (widen)
    (narrow-to-region 77 39)
    (goto-char 43)
    (insert "..........")
    (goto-char 40)
    (insert "...............")
    (goto-char 101)
    (insert "")
    (goto-char 53)
    (insert "....")
    (goto-char 79)
    (delete-char 21)
    (goto-char 85)
    (insert "........")
    (goto-char 52)
    (delete-char 41)
    (goto-char 43)
    (insert ".....")
    (should
     (equal
      (test-overlay-regions)
      '((4 . 90)
        (5 . 57)
        (6 . 90)
        (29 . 57)
        (29 . 57)
        (33 . 57))))))

(ert-deftest overlay-autogenerated-test-27 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 20 12 nil t nil)
    (make-overlay 3 10 nil t t)
    (make-overlay 11 53 nil t nil)
    (make-overlay 59 3 nil t nil)
    (make-overlay 28 19 nil t t)
    (make-overlay 16 30 nil t t)
    (make-overlay 39 19 nil t t)
    (make-overlay 33 50 nil t nil)
    (make-overlay 36 54 nil nil nil)
    (make-overlay 42 59 nil nil nil)
    (make-overlay 30 48 nil t nil)
    (make-overlay 20 13 nil nil t)
    (make-overlay 63 48 nil t nil)
    (make-overlay 48 12 nil t t)
    (make-overlay 64 50 nil nil nil)
    (make-overlay 7 7 nil nil nil)
    (goto-char 20)
    (widen)
    (narrow-to-region 21 54)
    (goto-char 40)
    (insert "..........")
    (goto-char 21)
    (delete-char 2)
    (goto-char 35)
    (widen)
    (narrow-to-region 70 11)
    (goto-char 45)
    (insert "...............")
    (goto-char 74)
    (insert ".")
    (goto-char 28)
    (widen)
    (narrow-to-region 77 67)
    (goto-char 72)
    (insert "..........")
    (goto-char 85)
    (delete-char 1)
    (goto-char 82)
    (widen)
    (narrow-to-region 83 86)
    (goto-char 83)
    (delete-char 0)
    (goto-char 86)
    (delete-char 0)
    (goto-char 86)
    (insert "...........")
    (goto-char 97)
    (insert ".......")
    (goto-char 103)
    (widen)
    (narrow-to-region 44 68)
    (goto-char 49)
    (insert "..")
    (goto-char 65)
    (insert ".............")
    (goto-char 59)
    (delete-char 0)
    (goto-char 57)
    (insert "........")
    (goto-char 55)
    (delete-char 30)
    (goto-char 45)
    (insert "...............")
    (goto-char 44)
    (insert "")
    (goto-char 62)
    (insert "............")
    (goto-char 63)
    (widen)
    (narrow-to-region 12 5)
    (goto-char 8)
    (delete-char 4)
    (goto-char 6)
    (delete-char 0)
    (goto-char 7)
    (insert "..........")
    (goto-char 15)
    (delete-char 0)
    (goto-char 16)
    (insert "............")
    (goto-char 20)
    (insert ".........")
    (goto-char 13)
    (insert "..")
    (goto-char 32)
    (insert "..............")
    (should
     (equal
      (test-overlay-regions)
      '((3 . 55)
        (3 . 173)
        (7 . 7))))))

(ert-deftest overlay-autogenerated-test-28 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 59 48 nil t nil)
    (make-overlay 59 4 nil nil t)
    (make-overlay 45 35 nil t nil)
    (make-overlay 13 18 nil t t)
    (make-overlay 10 7 nil t t)
    (make-overlay 9 8 nil nil nil)
    (make-overlay 33 47 nil nil t)
    (make-overlay 1 57 nil t nil)
    (make-overlay 16 59 nil nil t)
    (make-overlay 43 58 nil nil t)
    (make-overlay 6 11 nil nil nil)
    (make-overlay 59 7 nil t nil)
    (make-overlay 3 57 nil t t)
    (make-overlay 61 35 nil nil nil)
    (make-overlay 57 8 nil nil nil)
    (make-overlay 5 32 nil t nil)
    (goto-char 18)
    (insert "............")
    (goto-char 43)
    (delete-char 2)
    (goto-char 38)
    (delete-char 26)
    (goto-char 42)
    (insert ".....")
    (goto-char 52)
    (insert "..........")
    (goto-char 45)
    (delete-char 11)
    (goto-char 33)
    (insert "....")
    (goto-char 23)
    (delete-char 14)
    (goto-char 33)
    (widen)
    (narrow-to-region 30 33)
    (goto-char 30)
    (delete-char 0)
    (goto-char 30)
    (insert "...........")
    (goto-char 30)
    (delete-char 7)
    (goto-char 30)
    (insert ".")
    (goto-char 32)
    (delete-char 4)
    (goto-char 34)
    (delete-char 0)
    (goto-char 34)
    (delete-char 0)
    (goto-char 32)
    (insert "...............")
    (goto-char 46)
    (insert ".........")
    (goto-char 45)
    (delete-char 3)
    (goto-char 49)
    (delete-char 2)
    (goto-char 42)
    (delete-char 2)
    (goto-char 32)
    (insert "..........")
    (goto-char 47)
    (insert "....")
    (goto-char 59)
    (insert ".......")
    (goto-char 35)
    (insert ".")
    (goto-char 45)
    (insert "..............")
    (goto-char 37)
    (insert "..")
    (goto-char 80)
    (insert ".....")
    (goto-char 30)
    (insert ".............")
    (goto-char 102)
    (insert "............")
    (goto-char 113)
    (insert "")
    (goto-char 66)
    (widen)
    (narrow-to-region 47 38)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 45)
        (3 . 117)
        (4 . 121)
        (7 . 121)
        (8 . 45)
        (16 . 121)
        (28 . 121)
        (28 . 121)
        (28 . 121))))))

(ert-deftest overlay-autogenerated-test-29 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 5 63 nil nil t)
    (make-overlay 20 28 nil t t)
    (make-overlay 58 53 nil t nil)
    (make-overlay 4 57 nil t t)
    (make-overlay 4 16 nil nil nil)
    (make-overlay 33 26 nil t nil)
    (make-overlay 9 32 nil t t)
    (make-overlay 11 8 nil nil nil)
    (make-overlay 59 35 nil nil t)
    (make-overlay 15 25 nil t t)
    (make-overlay 36 16 nil nil nil)
    (make-overlay 8 37 nil nil nil)
    (make-overlay 65 63 nil nil t)
    (make-overlay 3 20 nil nil t)
    (make-overlay 44 55 nil t t)
    (make-overlay 45 25 nil t nil)
    (goto-char 39)
    (insert "...")
    (goto-char 22)
    (insert "........")
    (goto-char 60)
    (insert ".........")
    (goto-char 17)
    (insert "............")
    (goto-char 13)
    (widen)
    (narrow-to-region 79 16)
    (goto-char 19)
    (delete-char 11)
    (goto-char 25)
    (insert "........")
    (goto-char 61)
    (insert "....")
    (goto-char 45)
    (widen)
    (narrow-to-region 73 66)
    (goto-char 71)
    (insert "............")
    (goto-char 81)
    (delete-char 2)
    (goto-char 73)
    (insert "..........")
    (goto-char 74)
    (insert "............")
    (goto-char 82)
    (delete-char 7)
    (goto-char 78)
    (delete-char 18)
    (goto-char 75)
    (insert ".........")
    (goto-char 66)
    (insert ".........")
    (goto-char 86)
    (delete-char 12)
    (goto-char 77)
    (widen)
    (narrow-to-region 23 55)
    (goto-char 43)
    (insert ".")
    (goto-char 50)
    (insert "..")
    (goto-char 25)
    (delete-char 18)
    (goto-char 33)
    (delete-char 7)
    (goto-char 26)
    (insert "........")
    (goto-char 29)
    (insert "...........")
    (goto-char 33)
    (insert "...")
    (goto-char 40)
    (insert "..........")
    (goto-char 26)
    (insert "")
    (goto-char 35)
    (insert ".")
    (goto-char 59)
    (insert ".")
    (goto-char 51)
    (insert "..")
    (goto-char 59)
    (insert ".............")
    (should
     (equal
      (test-overlay-regions)
      '((4 . 130)
        (5 . 136)
        (8 . 82)
        (9 . 82)
        (15 . 25)
        (16 . 82)
        (21 . 77)
        (25 . 105)
        (75 . 82))))))

(ert-deftest overlay-autogenerated-test-30 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 27 65 nil t t)
    (make-overlay 39 51 nil t t)
    (make-overlay 53 2 nil nil nil)
    (make-overlay 3 17 nil nil t)
    (make-overlay 35 4 nil nil t)
    (make-overlay 65 53 nil t nil)
    (make-overlay 8 21 nil t t)
    (make-overlay 18 62 nil t t)
    (make-overlay 42 59 nil nil t)
    (make-overlay 12 37 nil t t)
    (make-overlay 64 31 nil t nil)
    (make-overlay 39 54 nil nil t)
    (make-overlay 41 24 nil t nil)
    (make-overlay 10 21 nil nil t)
    (make-overlay 49 15 nil t nil)
    (make-overlay 49 63 nil nil t)
    (goto-char 43)
    (insert "..........")
    (goto-char 44)
    (delete-char 29)
    (goto-char 32)
    (insert "..")
    (goto-char 13)
    (insert ".")
    (goto-char 42)
    (insert ".........")
    (goto-char 39)
    (insert "..........")
    (goto-char 15)
    (insert "............")
    (goto-char 58)
    (delete-char 9)
    (goto-char 63)
    (insert ".........")
    (goto-char 49)
    (insert ".")
    (goto-char 28)
    (delete-char 51)
    (goto-char 12)
    (delete-char 6)
    (goto-char 20)
    (delete-char 2)
    (goto-char 7)
    (widen)
    (narrow-to-region 2 9)
    (goto-char 5)
    (insert "...............")
    (goto-char 18)
    (delete-char 1)
    (goto-char 4)
    (insert ".............")
    (goto-char 13)
    (delete-char 22)
    (goto-char 12)
    (insert "")
    (goto-char 3)
    (insert ".............")
    (goto-char 22)
    (insert "...............")
    (goto-char 9)
    (insert "....")
    (goto-char 8)
    (insert "...........")
    (goto-char 6)
    (delete-char 34)
    (goto-char 21)
    (insert "....")
    (goto-char 14)
    (insert ".....")
    (goto-char 20)
    (insert ".......")
    (goto-char 34)
    (widen)
    (narrow-to-region 3 2)
    (goto-char 3)
    (delete-char 0)
    (goto-char 2)
    (insert "..............")
    (goto-char 15)
    (delete-char 2)
    (goto-char 11)
    (insert "......")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 68))))))

(ert-deftest overlay-autogenerated-test-31 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 54 64 nil nil nil)
    (make-overlay 49 12 nil nil t)
    (make-overlay 40 12 nil t nil)
    (make-overlay 17 38 nil nil nil)
    (make-overlay 21 36 nil t t)
    (make-overlay 8 38 nil t nil)
    (make-overlay 50 22 nil t nil)
    (make-overlay 65 15 nil nil t)
    (make-overlay 57 60 nil t t)
    (make-overlay 35 11 nil nil t)
    (make-overlay 49 44 nil nil t)
    (make-overlay 45 31 nil nil t)
    (make-overlay 51 24 nil t t)
    (make-overlay 20 14 nil nil nil)
    (make-overlay 6 18 nil t t)
    (make-overlay 25 3 nil nil nil)
    (goto-char 18)
    (delete-char 10)
    (goto-char 36)
    (delete-char 13)
    (goto-char 8)
    (delete-char 4)
    (goto-char 2)
    (delete-char 8)
    (goto-char 12)
    (delete-char 10)
    (goto-char 15)
    (delete-char 4)
    (goto-char 16)
    (insert ".........")
    (goto-char 17)
    (insert "...............")
    (goto-char 33)
    (delete-char 0)
    (goto-char 38)
    (delete-char 0)
    (goto-char 11)
    (insert "...........")
    (goto-char 8)
    (delete-char 14)
    (goto-char 32)
    (insert "........")
    (goto-char 40)
    (widen)
    (narrow-to-region 14 6)
    (goto-char 10)
    (delete-char 1)
    (goto-char 7)
    (widen)
    (narrow-to-region 18 39)
    (goto-char 36)
    (delete-char 1)
    (goto-char 34)
    (widen)
    (narrow-to-region 39 14)
    (goto-char 22)
    (widen)
    (narrow-to-region 25 21)
    (goto-char 23)
    (delete-char 2)
    (goto-char 23)
    (delete-char 0)
    (goto-char 23)
    (insert ".........")
    (goto-char 32)
    (delete-char 0)
    (goto-char 31)
    (insert ".........")
    (goto-char 32)
    (insert "...")
    (goto-char 30)
    (widen)
    (narrow-to-region 10 56)
    (goto-char 10)
    (insert ".........")
    (goto-char 38)
    (insert ".........")
    (goto-char 19)
    (insert "..")
    (goto-char 11)
    (insert "..............")
    (goto-char 66)
    (insert "...............")
    (goto-char 13)
    (insert "......")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 41)
        (3 . 117)
        (6 . 41)
        (8 . 41)
        (9 . 41)
        (10 . 42)
        (41 . 42))))))

(ert-deftest overlay-autogenerated-test-32 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 35 60 nil nil t)
    (make-overlay 45 46 nil nil nil)
    (make-overlay 47 11 nil nil t)
    (make-overlay 12 51 nil t nil)
    (make-overlay 61 17 nil t nil)
    (make-overlay 7 24 nil t nil)
    (make-overlay 36 37 nil nil t)
    (make-overlay 5 39 nil t t)
    (make-overlay 5 40 nil nil t)
    (make-overlay 38 40 nil t t)
    (make-overlay 47 45 nil t nil)
    (make-overlay 61 48 nil nil nil)
    (make-overlay 23 39 nil t t)
    (make-overlay 11 52 nil nil nil)
    (make-overlay 37 35 nil nil nil)
    (make-overlay 19 20 nil t nil)
    (goto-char 43)
    (insert "........")
    (goto-char 7)
    (insert "")
    (goto-char 28)
    (delete-char 41)
    (goto-char 3)
    (delete-char 17)
    (goto-char 2)
    (insert ".")
    (goto-char 7)
    (insert ".........")
    (goto-char 21)
    (delete-char 4)
    (goto-char 13)
    (delete-char 1)
    (goto-char 2)
    (insert "...............")
    (goto-char 7)
    (insert "")
    (goto-char 14)
    (insert ".....")
    (goto-char 16)
    (insert ".")
    (goto-char 10)
    (insert "..............")
    (goto-char 16)
    (delete-char 18)
    (goto-char 1)
    (delete-char 36)
    (goto-char 1)
    (delete-char 0)
    (goto-char 1)
    (delete-char 0)
    (goto-char 1)
    (insert ".............")
    (goto-char 9)
    (insert ".")
    (goto-char 14)
    (insert ".....")
    (goto-char 9)
    (delete-char 0)
    (goto-char 15)
    (delete-char 0)
    (goto-char 6)
    (delete-char 4)
    (goto-char 11)
    (delete-char 5)
    (goto-char 5)
    (insert "....")
    (goto-char 5)
    (insert ".....")
    (goto-char 12)
    (insert "")
    (goto-char 13)
    (insert ".......")
    (goto-char 14)
    (insert "......")
    (goto-char 9)
    (delete-char 3)
    (goto-char 17)
    (delete-char 0)
    (goto-char 7)
    (delete-char 12)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 18)
        (1 . 18)
        (1 . 18)
        (1 . 18)
        (18 . 18)
        (18 . 18)
        (18 . 18))))))

(ert-deftest overlay-autogenerated-test-33 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 65 33 nil t nil)
    (make-overlay 45 54 nil t t)
    (make-overlay 17 38 nil t nil)
    (make-overlay 58 46 nil nil t)
    (make-overlay 21 36 nil t t)
    (make-overlay 31 63 nil nil t)
    (make-overlay 37 64 nil t t)
    (make-overlay 42 19 nil nil nil)
    (make-overlay 51 60 nil t nil)
    (make-overlay 47 15 nil t t)
    (make-overlay 57 47 nil nil nil)
    (make-overlay 40 45 nil nil nil)
    (make-overlay 44 47 nil t nil)
    (make-overlay 42 35 nil t nil)
    (make-overlay 1 65 nil nil t)
    (make-overlay 29 63 nil t nil)
    (goto-char 33)
    (insert "...........")
    (goto-char 56)
    (insert ".........")
    (goto-char 67)
    (insert "....")
    (goto-char 28)
    (delete-char 35)
    (goto-char 9)
    (insert "......")
    (goto-char 43)
    (delete-char 17)
    (goto-char 29)
    (insert ".......")
    (goto-char 20)
    (insert "....")
    (goto-char 53)
    (insert ".......")
    (goto-char 14)
    (widen)
    (narrow-to-region 38 57)
    (goto-char 51)
    (insert "")
    (goto-char 57)
    (insert ".......")
    (goto-char 64)
    (insert ".....")
    (goto-char 59)
    (delete-char 3)
    (goto-char 45)
    (delete-char 12)
    (goto-char 43)
    (insert "......")
    (goto-char 48)
    (insert "......")
    (goto-char 52)
    (insert "........")
    (goto-char 57)
    (delete-char 16)
    (goto-char 43)
    (delete-char 9)
    (goto-char 40)
    (insert "")
    (goto-char 39)
    (insert "..........")
    (goto-char 50)
    (widen)
    (narrow-to-region 31 27)
    (goto-char 27)
    (insert "..........")
    (goto-char 33)
    (delete-char 0)
    (goto-char 37)
    (insert "..")
    (goto-char 38)
    (delete-char 4)
    (goto-char 38)
    (insert "..........")
    (goto-char 45)
    (insert ".....")
    (goto-char 53)
    (insert "...")
    (goto-char 51)
    (insert ".")
    (goto-char 28)
    (insert "...")
    (should
     (equal
      (test-overlay-regions)
      '((1 . 93)
        (25 . 92)
        (41 . 88)
        (60 . 88))))))

(ert-deftest overlay-autogenerated-test-34 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 2 63 nil nil t)
    (make-overlay 54 30 nil t nil)
    (make-overlay 21 57 nil t nil)
    (make-overlay 61 19 nil nil nil)
    (make-overlay 55 8 nil nil t)
    (make-overlay 14 51 nil nil nil)
    (make-overlay 33 13 nil t t)
    (make-overlay 36 25 nil t t)
    (make-overlay 22 21 nil nil t)
    (make-overlay 21 48 nil nil t)
    (make-overlay 36 7 nil nil t)
    (make-overlay 2 40 nil nil nil)
    (make-overlay 21 27 nil nil t)
    (make-overlay 26 2 nil nil nil)
    (make-overlay 60 43 nil nil nil)
    (make-overlay 12 50 nil t t)
    (goto-char 44)
    (delete-char 6)
    (goto-char 5)
    (insert "..")
    (goto-char 17)
    (insert "........")
    (goto-char 48)
    (insert "..")
    (goto-char 27)
    (delete-char 29)
    (goto-char 10)
    (delete-char 2)
    (goto-char 35)
    (insert ".............")
    (goto-char 20)
    (delete-char 0)
    (goto-char 6)
    (insert ".")
    (goto-char 9)
    (delete-char 6)
    (goto-char 38)
    (insert ".........")
    (goto-char 5)
    (insert ".........")
    (goto-char 10)
    (delete-char 20)
    (goto-char 6)
    (delete-char 6)
    (goto-char 14)
    (insert ".............")
    (goto-char 31)
    (delete-char 10)
    (goto-char 20)
    (widen)
    (narrow-to-region 27 39)
    (goto-char 34)
    (delete-char 5)
    (goto-char 32)
    (delete-char 1)
    (goto-char 27)
    (insert "..")
    (goto-char 28)
    (insert "........")
    (goto-char 39)
    (insert "........")
    (goto-char 38)
    (delete-char 7)
    (goto-char 44)
    (delete-char 0)
    (goto-char 30)
    (insert "...............")
    (goto-char 43)
    (insert "............")
    (goto-char 56)
    (delete-char 1)
    (goto-char 65)
    (delete-char 3)
    (goto-char 36)
    (insert ".........")
    (goto-char 74)
    (insert ".....")
    (goto-char 67)
    (delete-char 5)
    (goto-char 38)
    (insert "..")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 80)
        (6 . 78))))))

(ert-deftest overlay-autogenerated-test-35 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 38 16 nil nil nil)
    (make-overlay 19 22 nil t nil)
    (make-overlay 16 43 nil nil t)
    (make-overlay 27 5 nil nil nil)
    (make-overlay 43 34 nil t nil)
    (make-overlay 47 4 nil nil t)
    (make-overlay 1 47 nil nil t)
    (make-overlay 27 35 nil t nil)
    (make-overlay 41 41 nil nil t)
    (make-overlay 21 19 nil nil nil)
    (make-overlay 16 38 nil nil t)
    (make-overlay 33 39 nil t nil)
    (make-overlay 34 51 nil nil t)
    (make-overlay 45 36 nil t nil)
    (make-overlay 42 18 nil t t)
    (make-overlay 12 30 nil nil nil)
    (goto-char 18)
    (insert "")
    (goto-char 58)
    (delete-char 3)
    (goto-char 58)
    (delete-char 0)
    (goto-char 1)
    (insert ".......")
    (goto-char 48)
    (delete-char 17)
    (goto-char 39)
    (delete-char 6)
    (goto-char 33)
    (widen)
    (narrow-to-region 45 46)
    (goto-char 46)
    (insert "")
    (goto-char 46)
    (delete-char 0)
    (goto-char 46)
    (insert ".....")
    (goto-char 51)
    (widen)
    (narrow-to-region 17 26)
    (goto-char 25)
    (widen)
    (narrow-to-region 50 41)
    (goto-char 45)
    (insert "..............")
    (goto-char 59)
    (insert "...........")
    (goto-char 47)
    (delete-char 9)
    (goto-char 59)
    (insert "")
    (goto-char 46)
    (insert "")
    (goto-char 54)
    (delete-char 5)
    (goto-char 57)
    (widen)
    (narrow-to-region 57 31)
    (goto-char 42)
    (delete-char 2)
    (goto-char 52)
    (insert "....")
    (goto-char 44)
    (insert "..")
    (goto-char 44)
    (insert "...............")
    (goto-char 72)
    (delete-char 1)
    (goto-char 66)
    (delete-char 6)
    (goto-char 64)
    (delete-char 5)
    (goto-char 49)
    (delete-char 12)
    (goto-char 32)
    (insert "......")
    (goto-char 44)
    (delete-char 2)
    (goto-char 39)
    (delete-char 12)
    (goto-char 42)
    (insert "......")
    (goto-char 36)
    (widen)
    (narrow-to-region 14 47)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 39)
        (11 . 39)
        (12 . 39)
        (19 . 39)
        (23 . 39)
        (23 . 39)
        (23 . 39)
        (25 . 39)
        (26 . 28)
        (26 . 29)
        (39 . 39)
        (39 . 39)
        (39 . 39)
        (39 . 39)
        (39 . 39)
        (39 . 39))))))

(ert-deftest overlay-autogenerated-test-36 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 1 38 nil t t)
    (make-overlay 58 34 nil t nil)
    (make-overlay 6 33 nil nil t)
    (make-overlay 63 54 nil nil t)
    (make-overlay 54 54 nil t t)
    (make-overlay 21 61 nil nil nil)
    (make-overlay 64 55 nil nil t)
    (make-overlay 28 65 nil nil t)
    (make-overlay 32 51 nil t nil)
    (make-overlay 36 38 nil nil nil)
    (make-overlay 35 21 nil nil nil)
    (make-overlay 65 48 nil nil nil)
    (make-overlay 32 27 nil nil t)
    (make-overlay 27 55 nil t t)
    (make-overlay 30 22 nil t nil)
    (make-overlay 14 58 nil t nil)
    (goto-char 40)
    (delete-char 7)
    (goto-char 42)
    (insert "......")
    (goto-char 11)
    (widen)
    (narrow-to-region 64 9)
    (goto-char 21)
    (delete-char 23)
    (goto-char 24)
    (insert "...")
    (goto-char 13)
    (insert "..........")
    (goto-char 12)
    (delete-char 5)
    (goto-char 10)
    (delete-char 0)
    (goto-char 21)
    (widen)
    (narrow-to-region 9 5)
    (goto-char 6)
    (delete-char 0)
    (goto-char 9)
    (delete-char 0)
    (goto-char 9)
    (delete-char 0)
    (goto-char 7)
    (insert "............")
    (goto-char 9)
    (insert "...")
    (goto-char 18)
    (insert ".")
    (goto-char 23)
    (delete-char 1)
    (goto-char 9)
    (insert "....")
    (goto-char 6)
    (insert ".....")
    (goto-char 23)
    (widen)
    (narrow-to-region 28 1)
    (goto-char 6)
    (insert "...........")
    (goto-char 30)
    (delete-char 8)
    (goto-char 2)
    (insert ".")
    (goto-char 18)
    (insert "......")
    (goto-char 5)
    (delete-char 9)
    (goto-char 5)
    (delete-char 20)
    (goto-char 4)
    (delete-char 3)
    (goto-char 3)
    (delete-char 2)
    (goto-char 3)
    (delete-char 0)
    (goto-char 1)
    (insert "......")
    (goto-char 8)
    (widen)
    (narrow-to-region 39 2)
    (goto-char 13)
    (delete-char 12)
    (goto-char 24)
    (delete-char 0)
    (should
     (equal
      (test-overlay-regions)
      '((7 . 20)
        (9 . 20)
        (13 . 36)
        (20 . 20)
        (20 . 20)
        (20 . 20)
        (20 . 20)
        (20 . 29)
        (20 . 33)
        (20 . 36)
        (20 . 39)
        (20 . 43)
        (20 . 43))))))

(ert-deftest overlay-autogenerated-test-37 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 26 30 nil nil nil)
    (make-overlay 55 50 nil nil t)
    (make-overlay 43 54 nil nil t)
    (make-overlay 53 48 nil nil nil)
    (make-overlay 37 51 nil nil t)
    (make-overlay 15 30 nil nil nil)
    (make-overlay 2 24 nil t t)
    (make-overlay 56 61 nil t nil)
    (make-overlay 65 46 nil t nil)
    (make-overlay 28 47 nil t nil)
    (make-overlay 21 24 nil t t)
    (make-overlay 17 13 nil t t)
    (make-overlay 7 44 nil t nil)
    (make-overlay 28 63 nil nil nil)
    (make-overlay 22 16 nil t t)
    (make-overlay 26 44 nil t t)
    (goto-char 57)
    (delete-char 6)
    (goto-char 42)
    (insert ".....")
    (goto-char 63)
    (insert ".............")
    (goto-char 17)
    (insert "")
    (goto-char 57)
    (insert "...........")
    (goto-char 3)
    (delete-char 47)
    (goto-char 15)
    (insert ".............")
    (goto-char 28)
    (insert "")
    (goto-char 17)
    (delete-char 31)
    (goto-char 7)
    (delete-char 16)
    (goto-char 2)
    (insert "...........")
    (goto-char 2)
    (insert "..")
    (goto-char 18)
    (widen)
    (narrow-to-region 20 8)
    (goto-char 13)
    (widen)
    (narrow-to-region 12 10)
    (goto-char 10)
    (delete-char 1)
    (goto-char 11)
    (delete-char 0)
    (goto-char 10)
    (insert "...")
    (goto-char 11)
    (delete-char 0)
    (goto-char 13)
    (insert "..")
    (goto-char 16)
    (delete-char 0)
    (goto-char 10)
    (delete-char 2)
    (goto-char 11)
    (insert ".....")
    (goto-char 16)
    (widen)
    (narrow-to-region 6 13)
    (goto-char 10)
    (insert "..")
    (goto-char 6)
    (delete-char 6)
    (goto-char 8)
    (insert "...............")
    (goto-char 21)
    (delete-char 0)
    (goto-char 21)
    (widen)
    (narrow-to-region 36 11)
    (goto-char 12)
    (insert "...............")
    (goto-char 19)
    (insert ".......")
    (goto-char 56)
    (delete-char 2)
    (goto-char 42)
    (delete-char 11)
    (should
     (equal
      (test-overlay-regions)
      '((44 . 45))))))

(ert-deftest overlay-autogenerated-test-38 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 29 13 nil t t)
    (make-overlay 19 28 nil nil t)
    (make-overlay 47 33 nil nil nil)
    (make-overlay 8 44 nil t nil)
    (make-overlay 48 4 nil t nil)
    (make-overlay 8 20 nil t t)
    (make-overlay 38 31 nil nil t)
    (make-overlay 17 65 nil nil t)
    (make-overlay 49 31 nil nil nil)
    (make-overlay 39 19 nil nil t)
    (make-overlay 40 49 nil t t)
    (make-overlay 24 16 nil t t)
    (make-overlay 4 41 nil t nil)
    (make-overlay 61 42 nil t nil)
    (make-overlay 46 11 nil nil nil)
    (make-overlay 1 43 nil nil t)
    (goto-char 62)
    (delete-char 2)
    (goto-char 25)
    (widen)
    (narrow-to-region 30 38)
    (goto-char 37)
    (delete-char 1)
    (goto-char 37)
    (insert "...........")
    (goto-char 41)
    (delete-char 3)
    (goto-char 39)
    (delete-char 5)
    (goto-char 39)
    (widen)
    (narrow-to-region 31 9)
    (goto-char 11)
    (insert "..............")
    (goto-char 9)
    (widen)
    (narrow-to-region 62 30)
    (goto-char 32)
    (widen)
    (narrow-to-region 17 48)
    (goto-char 39)
    (delete-char 7)
    (goto-char 24)
    (delete-char 8)
    (goto-char 19)
    (insert "")
    (goto-char 25)
    (delete-char 5)
    (goto-char 28)
    (delete-char 0)
    (goto-char 22)
    (widen)
    (narrow-to-region 52 35)
    (goto-char 49)
    (delete-char 0)
    (goto-char 49)
    (delete-char 3)
    (goto-char 48)
    (insert "...........")
    (goto-char 37)
    (delete-char 23)
    (goto-char 36)
    (delete-char 0)
    (goto-char 35)
    (insert "....")
    (goto-char 35)
    (insert "..")
    (goto-char 39)
    (delete-char 4)
    (goto-char 39)
    (delete-char 0)
    (goto-char 36)
    (delete-char 3)
    (goto-char 36)
    (delete-char 0)
    (goto-char 36)
    (delete-char 0)
    (goto-char 36)
    (delete-char 0)
    (goto-char 36)
    (insert ".....")
    (goto-char 38)
    (delete-char 1)
    (goto-char 35)
    (delete-char 3)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 37)
        (24 . 44)
        (25 . 37))))))

(ert-deftest overlay-autogenerated-test-39 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 15 49 nil t t)
    (make-overlay 27 20 nil t nil)
    (make-overlay 55 50 nil t nil)
    (make-overlay 17 5 nil t t)
    (make-overlay 26 56 nil nil t)
    (make-overlay 42 11 nil t t)
    (make-overlay 24 35 nil nil t)
    (make-overlay 47 45 nil t t)
    (make-overlay 37 12 nil nil t)
    (make-overlay 17 25 nil t nil)
    (make-overlay 32 53 nil nil nil)
    (make-overlay 20 34 nil nil t)
    (make-overlay 56 58 nil nil t)
    (make-overlay 42 31 nil nil t)
    (make-overlay 22 55 nil t t)
    (make-overlay 55 11 nil t nil)
    (goto-char 16)
    (insert ".............")
    (goto-char 30)
    (insert ".")
    (goto-char 12)
    (delete-char 56)
    (goto-char 9)
    (insert ".............")
    (goto-char 6)
    (insert "....")
    (goto-char 19)
    (delete-char 19)
    (goto-char 19)
    (insert "...............")
    (goto-char 13)
    (delete-char 21)
    (goto-char 7)
    (delete-char 0)
    (goto-char 14)
    (widen)
    (narrow-to-region 5 6)
    (goto-char 5)
    (delete-char 0)
    (goto-char 6)
    (insert "......")
    (goto-char 10)
    (delete-char 0)
    (goto-char 7)
    (widen)
    (narrow-to-region 2 6)
    (goto-char 2)
    (insert "..........")
    (goto-char 2)
    (delete-char 9)
    (goto-char 7)
    (insert "...")
    (goto-char 9)
    (insert "...")
    (goto-char 10)
    (insert "......")
    (goto-char 4)
    (delete-char 14)
    (goto-char 4)
    (insert ".")
    (goto-char 5)
    (insert "..............")
    (goto-char 13)
    (insert "......")
    (goto-char 10)
    (insert "......")
    (goto-char 20)
    (insert "............")
    (goto-char 16)
    (widen)
    (narrow-to-region 3 32)
    (goto-char 18)
    (insert "..")
    (goto-char 6)
    (insert "......")
    (goto-char 38)
    (delete-char 0)
    (goto-char 31)
    (insert "............")
    (goto-char 28)
    (insert "")
    (goto-char 9)
    (delete-char 23)
    (should
     (equal
      (test-overlay-regions)
      'nil))))

(ert-deftest overlay-autogenerated-test-40 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 52 3 nil t nil)
    (make-overlay 35 41 nil t t)
    (make-overlay 4 2 nil t nil)
    (make-overlay 51 48 nil nil t)
    (make-overlay 44 57 nil t t)
    (make-overlay 13 32 nil nil nil)
    (make-overlay 46 29 nil t nil)
    (make-overlay 28 13 nil t nil)
    (make-overlay 10 65 nil t t)
    (make-overlay 41 48 nil nil t)
    (make-overlay 36 44 nil nil t)
    (make-overlay 29 61 nil t nil)
    (make-overlay 25 24 nil nil t)
    (make-overlay 22 45 nil nil t)
    (make-overlay 37 55 nil nil t)
    (make-overlay 36 39 nil nil nil)
    (goto-char 16)
    (delete-char 48)
    (goto-char 17)
    (delete-char 0)
    (goto-char 7)
    (insert "..............")
    (goto-char 30)
    (insert "........")
    (goto-char 11)
    (insert "..........")
    (goto-char 5)
    (delete-char 14)
    (goto-char 19)
    (insert ".")
    (goto-char 27)
    (insert "..")
    (goto-char 35)
    (delete-char 1)
    (goto-char 29)
    (delete-char 0)
    (goto-char 33)
    (delete-char 2)
    (goto-char 33)
    (insert "..")
    (goto-char 28)
    (insert ".........")
    (goto-char 30)
    (delete-char 4)
    (goto-char 40)
    (delete-char 1)
    (goto-char 15)
    (widen)
    (narrow-to-region 40 8)
    (goto-char 10)
    (delete-char 13)
    (goto-char 11)
    (delete-char 5)
    (goto-char 15)
    (insert "........")
    (goto-char 26)
    (delete-char 4)
    (goto-char 11)
    (delete-char 1)
    (goto-char 14)
    (insert "............")
    (goto-char 33)
    (insert ".")
    (goto-char 10)
    (insert "...")
    (goto-char 30)
    (widen)
    (narrow-to-region 28 9)
    (goto-char 27)
    (delete-char 0)
    (goto-char 27)
    (delete-char 1)
    (goto-char 26)
    (insert "..")
    (goto-char 27)
    (insert "..")
    (goto-char 20)
    (delete-char 5)
    (goto-char 12)
    (widen)
    (narrow-to-region 40 30)
    (goto-char 37)
    (delete-char 3)
    (should
     (equal
      (test-overlay-regions)
      '((13 . 37)
        (14 . 37)
        (14 . 37)
        (14 . 37)
        (14 . 37)
        (14 . 37)
        (14 . 37)
        (37 . 37)
        (37 . 37))))))

(ert-deftest overlay-autogenerated-test-41 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 28 48 nil nil t)
    (make-overlay 30 11 nil nil t)
    (make-overlay 7 12 nil t nil)
    (make-overlay 65 35 nil t nil)
    (make-overlay 22 61 nil t nil)
    (make-overlay 37 42 nil nil nil)
    (make-overlay 33 38 nil nil t)
    (make-overlay 48 45 nil t t)
    (make-overlay 45 62 nil t nil)
    (make-overlay 63 7 nil nil t)
    (make-overlay 23 42 nil t nil)
    (make-overlay 21 4 nil t nil)
    (make-overlay 64 41 nil t nil)
    (make-overlay 20 33 nil t t)
    (make-overlay 41 26 nil t nil)
    (make-overlay 43 31 nil t t)
    (goto-char 55)
    (delete-char 3)
    (goto-char 12)
    (insert "..")
    (goto-char 62)
    (insert "")
    (goto-char 24)
    (delete-char 2)
    (goto-char 41)
    (insert "............")
    (goto-char 2)
    (insert ".")
    (goto-char 55)
    (insert "........")
    (goto-char 67)
    (delete-char 6)
    (goto-char 58)
    (delete-char 10)
    (goto-char 29)
    (insert "")
    (goto-char 6)
    (widen)
    (narrow-to-region 44 45)
    (goto-char 44)
    (delete-char 1)
    (goto-char 44)
    (widen)
    (narrow-to-region 24 37)
    (goto-char 30)
    (delete-char 7)
    (goto-char 27)
    (insert "......")
    (goto-char 35)
    (delete-char 0)
    (goto-char 32)
    (insert "...............")
    (goto-char 37)
    (delete-char 9)
    (goto-char 40)
    (insert "..........")
    (goto-char 35)
    (insert "......")
    (goto-char 25)
    (delete-char 7)
    (goto-char 40)
    (delete-char 4)
    (goto-char 25)
    (delete-char 14)
    (goto-char 28)
    (insert "")
    (goto-char 28)
    (widen)
    (narrow-to-region 17 43)
    (goto-char 20)
    (insert "..........")
    (goto-char 22)
    (delete-char 2)
    (goto-char 48)
    (insert "............")
    (goto-char 47)
    (insert ".........")
    (goto-char 69)
    (widen)
    (narrow-to-region 52 25)
    (goto-char 26)
    (insert "......")
    (goto-char 53)
    (insert "..")
    (should
     (equal
      (test-overlay-regions)
      '((5 . 38)
        (8 . 97)
        (12 . 47)
        (37 . 47)
        (39 . 52)
        (39 . 87)
        (39 . 95)
        (46 . 90)
        (47 . 49)
        (47 . 90)
        (47 . 99)
        (48 . 87))))))

(ert-deftest overlay-autogenerated-test-42 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 20 23 nil nil nil)
    (make-overlay 45 51 nil t nil)
    (make-overlay 34 58 nil t nil)
    (make-overlay 27 11 nil nil nil)
    (make-overlay 14 8 nil t t)
    (make-overlay 64 43 nil t nil)
    (make-overlay 61 56 nil nil t)
    (make-overlay 28 14 nil t nil)
    (make-overlay 21 46 nil t t)
    (make-overlay 30 34 nil t t)
    (make-overlay 47 40 nil nil nil)
    (make-overlay 5 44 nil t t)
    (make-overlay 11 45 nil nil nil)
    (make-overlay 65 8 nil nil t)
    (make-overlay 47 54 nil t t)
    (make-overlay 37 57 nil t nil)
    (goto-char 11)
    (insert "....")
    (goto-char 65)
    (delete-char 0)
    (goto-char 56)
    (delete-char 4)
    (goto-char 11)
    (delete-char 2)
    (goto-char 23)
    (insert ".............")
    (goto-char 2)
    (insert "............")
    (goto-char 84)
    (delete-char 1)
    (goto-char 10)
    (insert "..............")
    (goto-char 19)
    (insert "............")
    (goto-char 69)
    (delete-char 6)
    (goto-char 15)
    (insert "........")
    (goto-char 104)
    (insert "")
    (goto-char 94)
    (delete-char 11)
    (goto-char 66)
    (insert ".....")
    (goto-char 67)
    (insert "")
    (goto-char 53)
    (delete-char 22)
    (goto-char 42)
    (insert ".")
    (goto-char 38)
    (delete-char 13)
    (goto-char 27)
    (insert "......")
    (goto-char 16)
    (insert "............")
    (goto-char 71)
    (widen)
    (narrow-to-region 59 15)
    (goto-char 46)
    (insert "..")
    (goto-char 20)
    (widen)
    (narrow-to-region 95 93)
    (goto-char 94)
    (insert ".............")
    (goto-char 103)
    (widen)
    (narrow-to-region 97 7)
    (goto-char 93)
    (insert "....")
    (goto-char 85)
    (insert "...........")
    (goto-char 69)
    (delete-char 24)
    (goto-char 87)
    (insert ".............")
    (goto-char 7)
    (delete-char 28)
    (goto-char 65)
    (delete-char 8)
    (goto-char 48)
    (insert "......")
    (should
     (equal
      (test-overlay-regions)
      '((31 . 44)
        (33 . 33)
        (33 . 41)
        (33 . 41)
        (33 . 41)
        (33 . 41)
        (33 . 82)
        (40 . 44)
        (41 . 41)
        (41 . 41)
        (41 . 47)
        (41 . 48)
        (44 . 45)
        (44 . 46)
        (44 . 63)
        (46 . 57))))))

(ert-deftest overlay-autogenerated-test-43 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 8 53 nil t nil)
    (make-overlay 11 50 nil t nil)
    (make-overlay 1 30 nil nil nil)
    (make-overlay 54 15 nil t t)
    (make-overlay 22 30 nil nil nil)
    (make-overlay 1 33 nil nil nil)
    (make-overlay 18 15 nil t nil)
    (make-overlay 43 39 nil nil t)
    (make-overlay 43 17 nil t nil)
    (make-overlay 2 29 nil t nil)
    (make-overlay 57 42 nil t nil)
    (make-overlay 40 1 nil nil nil)
    (make-overlay 8 64 nil nil nil)
    (make-overlay 64 15 nil nil nil)
    (make-overlay 9 11 nil nil t)
    (make-overlay 40 21 nil t nil)
    (goto-char 5)
    (delete-char 37)
    (goto-char 25)
    (delete-char 2)
    (goto-char 17)
    (insert "...........")
    (goto-char 19)
    (widen)
    (narrow-to-region 20 20)
    (goto-char 20)
    (delete-char 0)
    (goto-char 20)
    (insert "..........")
    (goto-char 24)
    (delete-char 5)
    (goto-char 24)
    (insert "...")
    (goto-char 28)
    (widen)
    (narrow-to-region 20 36)
    (goto-char 26)
    (delete-char 2)
    (goto-char 31)
    (insert ".............")
    (goto-char 22)
    (insert ".....")
    (goto-char 38)
    (delete-char 0)
    (goto-char 31)
    (delete-char 4)
    (goto-char 27)
    (insert "...")
    (goto-char 23)
    (widen)
    (narrow-to-region 37 20)
    (goto-char 22)
    (insert ".............")
    (goto-char 33)
    (insert "......")
    (goto-char 43)
    (insert "............")
    (goto-char 59)
    (insert ".......")
    (goto-char 25)
    (delete-char 26)
    (goto-char 49)
    (insert ".........")
    (goto-char 50)
    (insert ".......")
    (goto-char 39)
    (widen)
    (narrow-to-region 54 86)
    (goto-char 64)
    (insert "...............")
    (goto-char 83)
    (insert "............")
    (goto-char 70)
    (insert "........")
    (goto-char 58)
    (insert "..............")
    (goto-char 83)
    (insert "............")
    (goto-char 83)
    (insert "..........")
    (goto-char 69)
    (delete-char 75)
    (goto-char 75)
    (delete-char 3)
    (should
     (equal
      (test-overlay-regions)
      '((5 . 75)
        (5 . 75)
        (5 . 80)
        (5 . 80))))))

(ert-deftest overlay-autogenerated-test-44 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 8 48 nil nil t)
    (make-overlay 52 38 nil nil nil)
    (make-overlay 3 63 nil nil nil)
    (make-overlay 44 15 nil nil t)
    (make-overlay 27 44 nil nil t)
    (make-overlay 43 9 nil nil t)
    (make-overlay 11 27 nil t nil)
    (make-overlay 36 41 nil nil t)
    (make-overlay 23 25 nil t t)
    (make-overlay 19 60 nil t t)
    (make-overlay 11 55 nil t nil)
    (make-overlay 59 2 nil t nil)
    (make-overlay 32 64 nil t nil)
    (make-overlay 15 8 nil nil nil)
    (make-overlay 61 15 nil nil nil)
    (make-overlay 64 30 nil t t)
    (goto-char 42)
    (delete-char 20)
    (goto-char 44)
    (delete-char 1)
    (goto-char 43)
    (insert "...........")
    (goto-char 43)
    (delete-char 1)
    (goto-char 28)
    (delete-char 8)
    (goto-char 37)
    (delete-char 9)
    (goto-char 4)
    (delete-char 30)
    (goto-char 6)
    (delete-char 0)
    (goto-char 7)
    (delete-char 0)
    (goto-char 2)
    (delete-char 2)
    (goto-char 5)
    (delete-char 0)
    (goto-char 5)
    (delete-char 0)
    (goto-char 2)
    (insert ".....")
    (goto-char 10)
    (insert "...........")
    (goto-char 21)
    (insert "...")
    (goto-char 10)
    (delete-char 13)
    (goto-char 9)
    (insert "..........")
    (goto-char 16)
    (delete-char 1)
    (goto-char 16)
    (delete-char 4)
    (goto-char 16)
    (delete-char 0)
    (goto-char 14)
    (delete-char 1)
    (goto-char 3)
    (widen)
    (narrow-to-region 2 9)
    (goto-char 2)
    (insert "")
    (goto-char 2)
    (insert ".............")
    (goto-char 17)
    (insert "....")
    (goto-char 12)
    (insert "........")
    (goto-char 8)
    (widen)
    (narrow-to-region 32 23)
    (goto-char 29)
    (insert ".....")
    (goto-char 35)
    (delete-char 2)
    (goto-char 27)
    (delete-char 7)
    (goto-char 23)
    (widen)
    (narrow-to-region 4 14)
    (goto-char 8)
    (insert "...............")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 43)
        (2 . 43)
        (2 . 43)
        (2 . 43)
        (2 . 43)
        (2 . 44))))))

(ert-deftest overlay-autogenerated-test-45 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 15 48 nil nil nil)
    (make-overlay 1 47 nil t nil)
    (make-overlay 43 4 nil t t)
    (make-overlay 9 45 nil t t)
    (make-overlay 1 25 nil t t)
    (make-overlay 5 46 nil t t)
    (make-overlay 7 14 nil t nil)
    (make-overlay 1 53 nil nil t)
    (make-overlay 13 41 nil t nil)
    (make-overlay 5 31 nil t t)
    (make-overlay 26 10 nil nil nil)
    (make-overlay 56 37 nil nil nil)
    (make-overlay 23 15 nil t nil)
    (make-overlay 62 30 nil t t)
    (make-overlay 2 35 nil t t)
    (make-overlay 46 41 nil nil nil)
    (goto-char 65)
    (delete-char 0)
    (goto-char 55)
    (insert "...........")
    (goto-char 22)
    (insert "")
    (goto-char 73)
    (delete-char 3)
    (goto-char 43)
    (widen)
    (narrow-to-region 54 63)
    (goto-char 56)
    (insert "......")
    (goto-char 61)
    (delete-char 3)
    (goto-char 65)
    (insert "......")
    (goto-char 66)
    (insert ".....")
    (goto-char 62)
    (insert ".")
    (goto-char 74)
    (insert ".........")
    (goto-char 76)
    (delete-char 4)
    (goto-char 56)
    (widen)
    (narrow-to-region 2 46)
    (goto-char 43)
    (insert "...........")
    (goto-char 20)
    (delete-char 4)
    (goto-char 38)
    (delete-char 7)
    (goto-char 25)
    (delete-char 21)
    (goto-char 12)
    (insert ".........")
    (goto-char 19)
    (widen)
    (narrow-to-region 72 61)
    (goto-char 63)
    (insert "")
    (goto-char 65)
    (delete-char 4)
    (goto-char 61)
    (delete-char 5)
    (goto-char 63)
    (delete-char 0)
    (goto-char 63)
    (delete-char 0)
    (goto-char 62)
    (delete-char 0)
    (goto-char 61)
    (insert "............")
    (goto-char 72)
    (insert "..............")
    (goto-char 62)
    (delete-char 7)
    (goto-char 71)
    (delete-char 5)
    (goto-char 75)
    (widen)
    (narrow-to-region 29 8)
    (goto-char 17)
    (delete-char 2)
    (goto-char 27)
    (insert "........")
    (should
     (equal
      (test-overlay-regions)
      '((1 . 36)
        (1 . 41)
        (1 . 47)
        (2 . 40)
        (4 . 40)
        (5 . 40)
        (5 . 40)
        (7 . 21)
        (9 . 40)
        (10 . 37)
        (20 . 40)
        (22 . 27)
        (22 . 42))))))

(ert-deftest overlay-autogenerated-test-46 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 2 43 nil nil t)
    (make-overlay 44 40 nil nil t)
    (make-overlay 49 14 nil nil t)
    (make-overlay 6 55 nil nil nil)
    (make-overlay 13 52 nil t t)
    (make-overlay 40 54 nil t nil)
    (make-overlay 51 41 nil nil t)
    (make-overlay 7 28 nil nil t)
    (make-overlay 10 47 nil nil t)
    (make-overlay 63 21 nil t nil)
    (make-overlay 4 55 nil nil nil)
    (make-overlay 52 58 nil t nil)
    (make-overlay 62 11 nil t t)
    (make-overlay 22 49 nil t nil)
    (make-overlay 23 65 nil nil nil)
    (make-overlay 50 33 nil nil t)
    (goto-char 22)
    (insert "..............")
    (goto-char 12)
    (insert "....")
    (goto-char 25)
    (delete-char 16)
    (goto-char 14)
    (delete-char 53)
    (goto-char 2)
    (insert "............")
    (goto-char 20)
    (delete-char 5)
    (goto-char 11)
    (delete-char 7)
    (goto-char 9)
    (widen)
    (narrow-to-region 11 7)
    (goto-char 8)
    (insert "...............")
    (goto-char 12)
    (delete-char 4)
    (goto-char 21)
    (insert "...")
    (goto-char 20)
    (delete-char 5)
    (goto-char 7)
    (delete-char 3)
    (goto-char 16)
    (delete-char 0)
    (goto-char 12)
    (delete-char 1)
    (goto-char 15)
    (delete-char 0)
    (goto-char 7)
    (insert "..............")
    (goto-char 17)
    (insert "...........")
    (goto-char 15)
    (insert "............")
    (goto-char 20)
    (delete-char 5)
    (goto-char 7)
    (insert "....")
    (goto-char 37)
    (delete-char 7)
    (goto-char 8)
    (insert "..........")
    (goto-char 47)
    (insert ".............")
    (goto-char 65)
    (insert ".......")
    (goto-char 39)
    (delete-char 26)
    (goto-char 14)
    (delete-char 2)
    (goto-char 27)
    (insert ".............")
    (goto-char 17)
    (widen)
    (narrow-to-region 54 32)
    (goto-char 40)
    (widen)
    (narrow-to-region 10 3)
    (goto-char 7)
    (insert "........")
    (goto-char 13)
    (insert "..............")
    (should
     (equal
      (test-overlay-regions)
      '((2 . 85))))))

(ert-deftest overlay-autogenerated-test-47 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 58 62 nil t nil)
    (make-overlay 14 38 nil nil nil)
    (make-overlay 63 44 nil t t)
    (make-overlay 41 41 nil nil t)
    (make-overlay 19 39 nil nil nil)
    (make-overlay 10 49 nil t t)
    (make-overlay 56 38 nil t t)
    (make-overlay 23 38 nil nil t)
    (make-overlay 1 64 nil nil t)
    (make-overlay 21 3 nil t nil)
    (make-overlay 1 1 nil nil t)
    (make-overlay 27 61 nil nil nil)
    (make-overlay 29 59 nil nil nil)
    (make-overlay 37 30 nil t nil)
    (make-overlay 47 21 nil nil t)
    (make-overlay 34 26 nil t nil)
    (goto-char 6)
    (delete-char 44)
    (goto-char 8)
    (delete-char 0)
    (goto-char 8)
    (insert "....")
    (goto-char 17)
    (delete-char 2)
    (goto-char 12)
    (insert "...")
    (goto-char 20)
    (insert "")
    (goto-char 2)
    (delete-char 20)
    (goto-char 1)
    (insert ".........")
    (goto-char 7)
    (insert ".............")
    (goto-char 27)
    (delete-char 0)
    (goto-char 15)
    (insert "..........")
    (goto-char 36)
    (insert "..............")
    (goto-char 26)
    (insert "..............")
    (goto-char 63)
    (insert "...........")
    (goto-char 9)
    (insert "............")
    (goto-char 71)
    (delete-char 17)
    (goto-char 36)
    (insert "....")
    (goto-char 45)
    (delete-char 31)
    (goto-char 28)
    (delete-char 8)
    (goto-char 10)
    (delete-char 16)
    (goto-char 14)
    (delete-char 4)
    (goto-char 16)
    (delete-char 0)
    (goto-char 15)
    (insert "")
    (goto-char 14)
    (delete-char 1)
    (goto-char 10)
    (delete-char 2)
    (goto-char 6)
    (delete-char 0)
    (goto-char 1)
    (insert ".........")
    (goto-char 23)
    (insert "......")
    (goto-char 25)
    (insert "..........")
    (goto-char 25)
    (widen)
    (narrow-to-region 10 30)
    (goto-char 21)
    (delete-char 1)
    (goto-char 17)
    (insert "..........")
    (should
     (equal
      (test-overlay-regions)
      '((1 . 48)
        (1 . 48)
        (32 . 32)
        (32 . 32)
        (32 . 32)
        (32 . 32)
        (32 . 32)
        (32 . 32)
        (32 . 32)
        (32 . 32)
        (32 . 48)
        (32 . 48)
        (32 . 48))))))

(ert-deftest overlay-autogenerated-test-48 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 1 11 nil nil nil)
    (make-overlay 35 29 nil t t)
    (make-overlay 24 46 nil nil t)
    (make-overlay 15 43 nil nil t)
    (make-overlay 51 49 nil t t)
    (make-overlay 25 43 nil t nil)
    (make-overlay 23 59 nil nil nil)
    (make-overlay 10 4 nil t nil)
    (make-overlay 40 45 nil nil nil)
    (make-overlay 42 43 nil nil t)
    (make-overlay 20 38 nil t nil)
    (make-overlay 17 49 nil nil nil)
    (make-overlay 9 25 nil nil t)
    (make-overlay 13 19 nil nil nil)
    (make-overlay 44 31 nil t nil)
    (make-overlay 12 65 nil nil t)
    (goto-char 59)
    (widen)
    (narrow-to-region 28 14)
    (goto-char 26)
    (insert "...")
    (goto-char 30)
    (delete-char 1)
    (goto-char 23)
    (insert "...")
    (goto-char 27)
    (widen)
    (narrow-to-region 45 67)
    (goto-char 50)
    (insert "...............")
    (goto-char 59)
    (insert "..............")
    (goto-char 55)
    (insert ".............")
    (goto-char 106)
    (delete-char 0)
    (goto-char 97)
    (delete-char 10)
    (goto-char 67)
    (delete-char 16)
    (goto-char 76)
    (insert "..............")
    (goto-char 71)
    (insert ".............")
    (goto-char 110)
    (delete-char 0)
    (goto-char 56)
    (delete-char 38)
    (goto-char 61)
    (delete-char 10)
    (goto-char 56)
    (delete-char 5)
    (goto-char 49)
    (insert ".......")
    (goto-char 62)
    (insert "...")
    (goto-char 54)
    (insert "..........")
    (goto-char 47)
    (delete-char 10)
    (goto-char 47)
    (delete-char 20)
    (goto-char 46)
    (insert ".............")
    (goto-char 56)
    (insert "...........")
    (goto-char 70)
    (delete-char 1)
    (goto-char 62)
    (widen)
    (narrow-to-region 50 64)
    (goto-char 60)
    (insert "..")
    (goto-char 55)
    (delete-char 6)
    (goto-char 60)
    (insert ".............")
    (goto-char 61)
    (delete-char 9)
    (goto-char 64)
    (delete-char 0)
    (goto-char 53)
    (widen)
    (narrow-to-region 15 62)
    (should
     (equal
      (test-overlay-regions)
      '((9 . 28)
        (12 . 73)
        (13 . 19)
        (15 . 70)
        (17 . 70)
        (20 . 43)
        (23 . 70)
        (27 . 70)
        (28 . 70)
        (34 . 40)
        (36 . 70)
        (45 . 70))))))

(ert-deftest overlay-autogenerated-test-49 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 24 10 nil nil t)
    (make-overlay 53 23 nil t nil)
    (make-overlay 53 9 nil nil t)
    (make-overlay 65 64 nil t t)
    (make-overlay 48 2 nil nil t)
    (make-overlay 12 58 nil nil t)
    (make-overlay 64 64 nil nil nil)
    (make-overlay 26 13 nil t t)
    (make-overlay 46 26 nil nil t)
    (make-overlay 28 59 nil t t)
    (make-overlay 33 52 nil nil nil)
    (make-overlay 39 8 nil t t)
    (make-overlay 9 59 nil t t)
    (make-overlay 50 45 nil nil t)
    (make-overlay 41 53 nil nil t)
    (make-overlay 51 51 nil t nil)
    (goto-char 61)
    (insert "..............")
    (goto-char 19)
    (widen)
    (narrow-to-region 10 65)
    (goto-char 65)
    (delete-char 0)
    (goto-char 11)
    (insert "...............")
    (goto-char 77)
    (delete-char 0)
    (goto-char 51)
    (insert "...")
    (goto-char 75)
    (insert ".....")
    (goto-char 77)
    (delete-char 11)
    (goto-char 45)
    (delete-char 0)
    (goto-char 24)
    (widen)
    (narrow-to-region 33 52)
    (goto-char 46)
    (insert "..............")
    (goto-char 46)
    (insert "..........")
    (goto-char 39)
    (widen)
    (narrow-to-region 46 77)
    (goto-char 77)
    (insert "..............")
    (goto-char 54)
    (insert ".......")
    (goto-char 87)
    (insert ".")
    (goto-char 70)
    (delete-char 16)
    (goto-char 79)
    (delete-char 0)
    (goto-char 73)
    (widen)
    (narrow-to-region 74 100)
    (goto-char 91)
    (insert ".............")
    (goto-char 80)
    (delete-char 11)
    (goto-char 82)
    (insert "......")
    (goto-char 108)
    (delete-char 0)
    (goto-char 104)
    (insert ".....")
    (goto-char 100)
    (delete-char 1)
    (goto-char 90)
    (insert ".............")
    (goto-char 99)
    (insert ".............")
    (goto-char 124)
    (insert "..............")
    (goto-char 114)
    (insert "....")
    (goto-char 134)
    (delete-char 0)
    (goto-char 89)
    (delete-char 65)
    (goto-char 75)
    (delete-char 16)
    (should
     (equal
      (test-overlay-regions)
      '((2 . 75)
        (8 . 75)
        (9 . 76)
        (9 . 82)
        (27 . 82)
        (38 . 76)
        (41 . 75)
        (43 . 82)
        (70 . 75))))))

(ert-deftest overlay-autogenerated-test-50 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 29 53 nil t t)
    (make-overlay 65 64 nil nil nil)
    (make-overlay 3 31 nil nil t)
    (make-overlay 45 59 nil t nil)
    (make-overlay 60 37 nil t t)
    (make-overlay 7 5 nil t t)
    (make-overlay 37 24 nil nil nil)
    (make-overlay 45 20 nil nil nil)
    (make-overlay 33 42 nil nil t)
    (make-overlay 47 57 nil t nil)
    (make-overlay 14 49 nil t t)
    (make-overlay 14 30 nil t nil)
    (make-overlay 21 40 nil t t)
    (make-overlay 5 45 nil t t)
    (make-overlay 59 40 nil t t)
    (make-overlay 37 52 nil nil nil)
    (goto-char 48)
    (insert "")
    (goto-char 7)
    (insert ".........")
    (goto-char 31)
    (insert "...........")
    (goto-char 41)
    (delete-char 7)
    (goto-char 21)
    (delete-char 11)
    (goto-char 41)
    (widen)
    (narrow-to-region 51 53)
    (goto-char 52)
    (insert ".....")
    (goto-char 55)
    (widen)
    (narrow-to-region 18 24)
    (goto-char 23)
    (widen)
    (narrow-to-region 39 38)
    (goto-char 38)
    (insert ".............")
    (goto-char 41)
    (insert "......")
    (goto-char 38)
    (insert "..............")
    (goto-char 52)
    (insert "...............")
    (goto-char 78)
    (delete-char 5)
    (goto-char 50)
    (insert "..........")
    (goto-char 50)
    (delete-char 3)
    (goto-char 85)
    (widen)
    (narrow-to-region 86 1)
    (goto-char 5)
    (insert "....")
    (goto-char 69)
    (insert "...........")
    (goto-char 94)
    (insert "......")
    (goto-char 98)
    (delete-char 7)
    (goto-char 46)
    (insert "...............")
    (goto-char 79)
    (insert "............")
    (goto-char 89)
    (insert "")
    (goto-char 14)
    (delete-char 63)
    (goto-char 20)
    (insert ".........")
    (goto-char 34)
    (insert "...")
    (goto-char 53)
    (delete-char 14)
    (goto-char 6)
    (widen)
    (narrow-to-region 6 52)
    (goto-char 42)
    (insert "...........")
    (goto-char 40)
    (insert ".......")
    (goto-char 46)
    (widen)
    (narrow-to-region 1 68)
    (should
     (equal
      (test-overlay-regions)
      '((3 . 14)
        (9 . 14)
        (9 . 91)
        (14 . 14)
        (14 . 83)
        (14 . 86)
        (14 . 88)
        (14 . 91)
        (14 . 95)
        (14 . 104))))))

(ert-deftest overlay-autogenerated-test-51 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 14 5 nil t nil)
    (make-overlay 62 34 nil nil t)
    (make-overlay 7 62 nil nil t)
    (make-overlay 23 12 nil t t)
    (make-overlay 16 4 nil nil nil)
    (make-overlay 24 15 nil nil nil)
    (make-overlay 6 6 nil t t)
    (make-overlay 25 64 nil t t)
    (make-overlay 23 6 nil t t)
    (make-overlay 55 64 nil nil nil)
    (make-overlay 8 62 nil nil t)
    (make-overlay 65 65 nil nil nil)
    (make-overlay 57 51 nil t t)
    (make-overlay 35 8 nil t nil)
    (make-overlay 55 13 nil nil t)
    (make-overlay 60 62 nil nil t)
    (goto-char 12)
    (insert "..")
    (goto-char 66)
    (insert "............")
    (goto-char 32)
    (insert "..")
    (goto-char 27)
    (insert ".........")
    (goto-char 8)
    (insert ".............")
    (goto-char 79)
    (insert ".")
    (goto-char 47)
    (insert "....")
    (goto-char 49)
    (insert "...")
    (goto-char 81)
    (insert "....")
    (goto-char 112)
    (delete-char 0)
    (goto-char 97)
    (insert ".....")
    (goto-char 109)
    (delete-char 5)
    (goto-char 20)
    (insert ".....")
    (goto-char 59)
    (delete-char 33)
    (goto-char 87)
    (insert ".............")
    (goto-char 98)
    (insert "....")
    (goto-char 22)
    (delete-char 36)
    (goto-char 45)
    (insert "..............")
    (goto-char 42)
    (delete-char 29)
    (goto-char 51)
    (widen)
    (narrow-to-region 39 41)
    (goto-char 39)
    (delete-char 2)
    (goto-char 39)
    (insert ".............")
    (goto-char 51)
    (insert "......")
    (goto-char 52)
    (insert "...............")
    (goto-char 56)
    (widen)
    (narrow-to-region 59 20)
    (goto-char 56)
    (insert "............")
    (goto-char 57)
    (insert ".")
    (goto-char 37)
    (delete-char 12)
    (goto-char 39)
    (delete-char 11)
    (goto-char 38)
    (delete-char 8)
    (goto-char 36)
    (widen)
    (narrow-to-region 65 26)
    (goto-char 40)
    (widen)
    (narrow-to-region 27 55)
    (should
     (equal
      (test-overlay-regions)
      '((7 . 55)
        (8 . 55)
        (22 . 29)
        (23 . 55)
        (23 . 56)
        (24 . 31)
        (29 . 56)
        (37 . 55))))))

(ert-deftest overlay-autogenerated-test-52 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 58 32 nil nil nil)
    (make-overlay 44 54 nil nil t)
    (make-overlay 27 50 nil nil nil)
    (make-overlay 55 35 nil nil t)
    (make-overlay 40 46 nil nil t)
    (make-overlay 56 63 nil t nil)
    (make-overlay 29 48 nil nil nil)
    (make-overlay 45 24 nil t nil)
    (make-overlay 60 25 nil t nil)
    (make-overlay 55 41 nil t nil)
    (make-overlay 55 1 nil nil t)
    (make-overlay 30 45 nil t t)
    (make-overlay 26 19 nil nil t)
    (make-overlay 61 5 nil nil nil)
    (make-overlay 33 5 nil nil nil)
    (make-overlay 42 18 nil t nil)
    (goto-char 55)
    (insert ".")
    (goto-char 49)
    (delete-char 12)
    (goto-char 41)
    (insert "..........")
    (goto-char 27)
    (insert ".....")
    (goto-char 58)
    (insert "...........")
    (goto-char 24)
    (delete-char 23)
    (goto-char 47)
    (delete-char 9)
    (goto-char 4)
    (insert "...")
    (goto-char 10)
    (delete-char 32)
    (goto-char 4)
    (insert "..............")
    (goto-char 29)
    (insert "....")
    (goto-char 28)
    (delete-char 2)
    (goto-char 34)
    (insert "...........")
    (goto-char 9)
    (insert "......")
    (goto-char 5)
    (insert "")
    (goto-char 45)
    (delete-char 1)
    (goto-char 18)
    (insert ".........")
    (goto-char 36)
    (delete-char 5)
    (goto-char 15)
    (delete-char 27)
    (goto-char 15)
    (delete-char 10)
    (goto-char 16)
    (delete-char 2)
    (goto-char 16)
    (widen)
    (narrow-to-region 10 2)
    (goto-char 9)
    (delete-char 1)
    (goto-char 3)
    (delete-char 2)
    (goto-char 2)
    (widen)
    (narrow-to-region 9 10)
    (goto-char 9)
    (insert "...........")
    (goto-char 19)
    (delete-char 0)
    (goto-char 14)
    (delete-char 3)
    (goto-char 11)
    (delete-char 2)
    (goto-char 9)
    (delete-char 6)
    (goto-char 9)
    (delete-char 0)
    (goto-char 10)
    (insert "....")
    (should
     (equal
      (test-overlay-regions)
      '((1 . 17))))))

(ert-deftest overlay-autogenerated-test-53 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 10 30 nil nil nil)
    (make-overlay 11 57 nil t nil)
    (make-overlay 59 56 nil nil t)
    (make-overlay 20 37 nil nil t)
    (make-overlay 41 29 nil nil nil)
    (make-overlay 31 10 nil nil t)
    (make-overlay 6 36 nil nil nil)
    (make-overlay 12 54 nil nil nil)
    (make-overlay 25 26 nil t t)
    (make-overlay 21 19 nil nil t)
    (make-overlay 1 21 nil nil t)
    (make-overlay 48 51 nil nil nil)
    (make-overlay 54 55 nil t nil)
    (make-overlay 64 48 nil t t)
    (make-overlay 56 25 nil nil t)
    (make-overlay 12 60 nil t nil)
    (goto-char 41)
    (delete-char 1)
    (goto-char 63)
    (insert "")
    (goto-char 14)
    (delete-char 5)
    (goto-char 11)
    (insert "..............")
    (goto-char 41)
    (widen)
    (narrow-to-region 12 1)
    (goto-char 1)
    (delete-char 3)
    (goto-char 9)
    (delete-char 0)
    (goto-char 5)
    (insert "..............")
    (goto-char 1)
    (insert "..........")
    (goto-char 29)
    (insert "...............")
    (goto-char 4)
    (insert "..")
    (goto-char 31)
    (delete-char 15)
    (goto-char 31)
    (insert "")
    (goto-char 27)
    (insert "......")
    (goto-char 6)
    (insert "...")
    (goto-char 23)
    (widen)
    (narrow-to-region 23 47)
    (goto-char 37)
    (delete-char 2)
    (goto-char 35)
    (delete-char 5)
    (goto-char 38)
    (delete-char 2)
    (goto-char 30)
    (insert ".......")
    (goto-char 45)
    (widen)
    (narrow-to-region 13 2)
    (goto-char 9)
    (delete-char 1)
    (goto-char 3)
    (insert ".....")
    (goto-char 2)
    (insert "...............")
    (goto-char 16)
    (delete-char 5)
    (goto-char 20)
    (insert ".....")
    (goto-char 26)
    (delete-char 0)
    (goto-char 26)
    (widen)
    (narrow-to-region 76 98)
    (goto-char 88)
    (insert ".........")
    (goto-char 92)
    (insert ".")
    (goto-char 108)
    (delete-char 0)
    (goto-char 103)
    (delete-char 3)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 79)
        (37 . 103)
        (61 . 88)
        (61 . 99)
        (74 . 121)
        (75 . 118)
        (75 . 124)
        (77 . 79)
        (78 . 103)
        (83 . 84)
        (83 . 120)
        (87 . 106))))))

(ert-deftest overlay-autogenerated-test-54 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 58 36 nil t t)
    (make-overlay 55 49 nil nil t)
    (make-overlay 12 25 nil nil t)
    (make-overlay 16 37 nil t t)
    (make-overlay 42 25 nil t t)
    (make-overlay 8 41 nil t t)
    (make-overlay 13 27 nil nil t)
    (make-overlay 52 22 nil t nil)
    (make-overlay 36 17 nil t nil)
    (make-overlay 1 52 nil t nil)
    (make-overlay 55 5 nil nil t)
    (make-overlay 50 50 nil t nil)
    (make-overlay 32 15 nil t nil)
    (make-overlay 39 26 nil t nil)
    (make-overlay 26 4 nil nil nil)
    (make-overlay 38 47 nil t t)
    (goto-char 23)
    (insert ".")
    (goto-char 57)
    (delete-char 6)
    (goto-char 54)
    (insert "..............")
    (goto-char 46)
    (insert "...............")
    (goto-char 29)
    (insert ".......")
    (goto-char 58)
    (delete-char 21)
    (goto-char 45)
    (delete-char 4)
    (goto-char 50)
    (delete-char 4)
    (goto-char 20)
    (insert ".........")
    (goto-char 16)
    (insert "......")
    (goto-char 17)
    (insert ".....")
    (goto-char 63)
    (insert "........")
    (goto-char 83)
    (insert "....")
    (goto-char 73)
    (delete-char 8)
    (goto-char 69)
    (insert "...........")
    (goto-char 48)
    (widen)
    (narrow-to-region 19 31)
    (goto-char 22)
    (delete-char 3)
    (goto-char 23)
    (delete-char 5)
    (goto-char 20)
    (insert "............")
    (goto-char 23)
    (delete-char 11)
    (goto-char 19)
    (insert "..........")
    (goto-char 23)
    (insert "........")
    (goto-char 38)
    (delete-char 1)
    (goto-char 33)
    (delete-char 5)
    (goto-char 27)
    (insert "..........")
    (goto-char 35)
    (delete-char 8)
    (goto-char 35)
    (insert ".")
    (goto-char 20)
    (insert "......")
    (goto-char 22)
    (delete-char 22)
    (goto-char 23)
    (delete-char 0)
    (goto-char 22)
    (widen)
    (narrow-to-region 1 41)
    (goto-char 13)
    (insert ".......")
    (should
     (equal
      (test-overlay-regions)
      '((1 . 83)
        (4 . 46)
        (5 . 97)
        (8 . 83)
        (12 . 45)
        (13 . 47)
        (22 . 59)
        (30 . 82)
        (30 . 83)
        (41 . 83)
        (45 . 83)
        (46 . 83))))))

(ert-deftest overlay-autogenerated-test-55 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 58 20 nil nil nil)
    (make-overlay 60 33 nil t nil)
    (make-overlay 6 27 nil nil nil)
    (make-overlay 53 31 nil nil t)
    (make-overlay 30 55 nil t t)
    (make-overlay 4 64 nil t t)
    (make-overlay 51 31 nil nil t)
    (make-overlay 4 65 nil t t)
    (make-overlay 57 62 nil t t)
    (make-overlay 28 7 nil nil t)
    (make-overlay 61 48 nil t nil)
    (make-overlay 23 54 nil nil t)
    (make-overlay 47 49 nil nil nil)
    (make-overlay 12 52 nil t nil)
    (make-overlay 39 57 nil t t)
    (make-overlay 28 61 nil nil t)
    (goto-char 8)
    (insert "..............")
    (goto-char 63)
    (delete-char 3)
    (goto-char 67)
    (delete-char 6)
    (goto-char 3)
    (widen)
    (narrow-to-region 10 67)
    (goto-char 43)
    (insert ".............")
    (goto-char 20)
    (insert "...............")
    (goto-char 18)
    (insert "..")
    (goto-char 37)
    (delete-char 47)
    (goto-char 34)
    (insert "..............")
    (goto-char 31)
    (delete-char 2)
    (goto-char 16)
    (widen)
    (narrow-to-region 29 36)
    (goto-char 31)
    (delete-char 2)
    (goto-char 31)
    (insert ".......")
    (goto-char 40)
    (delete-char 0)
    (goto-char 32)
    (widen)
    (narrow-to-region 40 19)
    (goto-char 40)
    (insert "..")
    (goto-char 37)
    (delete-char 0)
    (goto-char 40)
    (delete-char 1)
    (goto-char 34)
    (delete-char 4)
    (goto-char 33)
    (insert "..............")
    (goto-char 19)
    (widen)
    (narrow-to-region 78 70)
    (goto-char 77)
    (insert ".........")
    (goto-char 80)
    (delete-char 1)
    (goto-char 73)
    (delete-char 3)
    (goto-char 70)
    (insert ".........")
    (goto-char 75)
    (delete-char 10)
    (goto-char 74)
    (delete-char 3)
    (goto-char 73)
    (insert "...............")
    (goto-char 90)
    (insert "......")
    (goto-char 94)
    (insert "..............")
    (goto-char 101)
    (insert "........")
    (goto-char 111)
    (insert "........")
    (should
     (equal
      (test-overlay-regions)
      '((4 . 132)
        (4 . 133)
        (65 . 89)
        (65 . 89)
        (65 . 89)
        (65 . 89)
        (65 . 129)
        (65 . 130)
        (65 . 130)
        (65 . 130)
        (65 . 130)
        (89 . 89)
        (89 . 130))))))

(ert-deftest overlay-autogenerated-test-56 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 7 14 nil nil t)
    (make-overlay 10 10 nil nil t)
    (make-overlay 21 23 nil nil t)
    (make-overlay 4 44 nil t nil)
    (make-overlay 42 16 nil t t)
    (make-overlay 1 57 nil t nil)
    (make-overlay 15 27 nil nil nil)
    (make-overlay 31 1 nil t nil)
    (make-overlay 56 45 nil t t)
    (make-overlay 46 19 nil t nil)
    (make-overlay 15 6 nil nil nil)
    (make-overlay 31 26 nil nil t)
    (make-overlay 39 41 nil t t)
    (make-overlay 52 48 nil nil t)
    (make-overlay 44 2 nil t nil)
    (make-overlay 60 7 nil nil t)
    (goto-char 49)
    (delete-char 11)
    (goto-char 43)
    (delete-char 9)
    (goto-char 42)
    (delete-char 2)
    (goto-char 12)
    (insert "...........")
    (goto-char 36)
    (insert ".........")
    (goto-char 1)
    (insert "......")
    (goto-char 67)
    (delete-char 0)
    (goto-char 47)
    (insert ".............")
    (goto-char 57)
    (insert "........")
    (goto-char 22)
    (widen)
    (narrow-to-region 75 33)
    (goto-char 41)
    (delete-char 28)
    (goto-char 43)
    (delete-char 0)
    (goto-char 33)
    (delete-char 5)
    (goto-char 38)
    (insert "..")
    (goto-char 42)
    (delete-char 0)
    (goto-char 38)
    (delete-char 0)
    (goto-char 38)
    (insert "............")
    (goto-char 51)
    (insert ".......")
    (goto-char 48)
    (insert "..")
    (goto-char 55)
    (insert ".")
    (goto-char 33)
    (delete-char 8)
    (goto-char 42)
    (insert "..")
    (goto-char 45)
    (insert "..")
    (goto-char 59)
    (insert ".............")
    (goto-char 53)
    (insert ".......")
    (goto-char 81)
    (delete-char 0)
    (goto-char 44)
    (delete-char 36)
    (goto-char 38)
    (delete-char 8)
    (goto-char 33)
    (insert ".............")
    (goto-char 41)
    (insert "..............")
    (goto-char 65)
    (insert "...............")
    (goto-char 61)
    (insert "...")
    (should
     (equal
      (test-overlay-regions)
      '((7 . 86)
        (7 . 97)
        (8 . 97)
        (10 . 97)
        (13 . 97)
        (32 . 68)
        (33 . 60)
        (60 . 97)
        (60 . 97)
        (68 . 86))))))

(ert-deftest overlay-autogenerated-test-57 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 52 31 nil t nil)
    (make-overlay 39 17 nil t nil)
    (make-overlay 19 20 nil t t)
    (make-overlay 18 3 nil nil t)
    (make-overlay 19 47 nil nil t)
    (make-overlay 38 54 nil nil nil)
    (make-overlay 30 51 nil nil t)
    (make-overlay 29 60 nil t t)
    (make-overlay 57 38 nil nil nil)
    (make-overlay 13 41 nil t nil)
    (make-overlay 9 44 nil t nil)
    (make-overlay 30 55 nil t nil)
    (make-overlay 33 10 nil nil nil)
    (make-overlay 14 35 nil nil t)
    (make-overlay 53 50 nil t nil)
    (make-overlay 25 28 nil nil t)
    (goto-char 40)
    (insert "..")
    (goto-char 64)
    (insert "........")
    (goto-char 47)
    (insert "............")
    (goto-char 65)
    (delete-char 0)
    (goto-char 86)
    (delete-char 1)
    (goto-char 59)
    (delete-char 11)
    (goto-char 64)
    (delete-char 8)
    (goto-char 53)
    (delete-char 0)
    (goto-char 28)
    (delete-char 8)
    (goto-char 6)
    (delete-char 33)
    (goto-char 14)
    (delete-char 2)
    (goto-char 2)
    (delete-char 10)
    (goto-char 3)
    (insert "..")
    (goto-char 5)
    (insert ".........")
    (goto-char 1)
    (insert "........")
    (goto-char 10)
    (delete-char 4)
    (goto-char 26)
    (insert "........")
    (goto-char 23)
    (insert "....")
    (goto-char 1)
    (widen)
    (narrow-to-region 15 23)
    (goto-char 19)
    (insert "...")
    (goto-char 24)
    (delete-char 0)
    (goto-char 19)
    (insert ".......")
    (goto-char 18)
    (insert "..")
    (goto-char 33)
    (insert "...")
    (goto-char 32)
    (insert "...............")
    (goto-char 29)
    (delete-char 10)
    (goto-char 29)
    (insert "..........")
    (goto-char 50)
    (insert "")
    (goto-char 16)
    (insert ".........")
    (goto-char 52)
    (widen)
    (narrow-to-region 59 15)
    (goto-char 35)
    (delete-char 4)
    (goto-char 18)
    (insert "....")
    (should
     (equal
      (test-overlay-regions)
      '((10 . 57)
        (10 . 57)
        (10 . 57)
        (10 . 60)
        (10 . 60)
        (10 . 61)
        (10 . 68)
        (57 . 57))))))

(ert-deftest overlay-autogenerated-test-58 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 8 16 nil t nil)
    (make-overlay 57 27 nil nil nil)
    (make-overlay 15 62 nil nil nil)
    (make-overlay 32 33 nil nil t)
    (make-overlay 47 27 nil nil t)
    (make-overlay 41 4 nil nil t)
    (make-overlay 57 61 nil t nil)
    (make-overlay 18 43 nil nil t)
    (make-overlay 64 51 nil t t)
    (make-overlay 44 26 nil nil nil)
    (make-overlay 9 13 nil nil t)
    (make-overlay 41 65 nil nil t)
    (make-overlay 23 13 nil t t)
    (make-overlay 26 59 nil t t)
    (make-overlay 65 65 nil t t)
    (make-overlay 15 7 nil nil nil)
    (goto-char 41)
    (insert "........")
    (goto-char 35)
    (delete-char 14)
    (goto-char 32)
    (widen)
    (narrow-to-region 23 46)
    (goto-char 41)
    (delete-char 5)
    (goto-char 29)
    (delete-char 10)
    (goto-char 31)
    (insert ".")
    (goto-char 29)
    (insert "........")
    (goto-char 27)
    (delete-char 7)
    (goto-char 29)
    (insert "")
    (goto-char 24)
    (insert "............")
    (goto-char 43)
    (delete-char 1)
    (goto-char 31)
    (delete-char 9)
    (goto-char 34)
    (widen)
    (narrow-to-region 20 14)
    (goto-char 20)
    (delete-char 0)
    (goto-char 17)
    (insert "...........")
    (goto-char 31)
    (delete-char 0)
    (goto-char 16)
    (insert "...........")
    (goto-char 17)
    (delete-char 8)
    (goto-char 23)
    (delete-char 5)
    (goto-char 20)
    (insert "..........")
    (goto-char 33)
    (widen)
    (narrow-to-region 16 29)
    (goto-char 24)
    (insert "...............")
    (goto-char 44)
    (delete-char 0)
    (goto-char 30)
    (insert "....")
    (goto-char 27)
    (widen)
    (narrow-to-region 4 22)
    (goto-char 10)
    (insert "..............")
    (goto-char 36)
    (insert "..")
    (goto-char 10)
    (delete-char 21)
    (goto-char 14)
    (delete-char 1)
    (goto-char 14)
    (insert "...........")
    (goto-char 12)
    (insert "........")
    (goto-char 32)
    (insert "........")
    (should
     (equal
      (test-overlay-regions)
      '((4 . 92)
        (7 . 10)
        (8 . 10)
        (9 . 10)
        (10 . 82)
        (10 . 104))))))

(ert-deftest overlay-autogenerated-test-59 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 46 30 nil t t)
    (make-overlay 3 26 nil nil nil)
    (make-overlay 36 28 nil t t)
    (make-overlay 49 49 nil t t)
    (make-overlay 27 61 nil t nil)
    (make-overlay 14 16 nil nil nil)
    (make-overlay 50 61 nil t nil)
    (make-overlay 59 63 nil nil nil)
    (make-overlay 36 34 nil t nil)
    (make-overlay 35 29 nil nil nil)
    (make-overlay 5 65 nil nil nil)
    (make-overlay 20 61 nil nil t)
    (make-overlay 10 42 nil nil nil)
    (make-overlay 47 49 nil nil t)
    (make-overlay 12 4 nil nil nil)
    (make-overlay 32 24 nil t t)
    (goto-char 11)
    (insert ".")
    (goto-char 32)
    (delete-char 2)
    (goto-char 61)
    (insert ".........")
    (goto-char 36)
    (insert "........")
    (goto-char 55)
    (widen)
    (narrow-to-region 8 55)
    (goto-char 21)
    (insert "....")
    (goto-char 32)
    (delete-char 15)
    (goto-char 30)
    (delete-char 5)
    (goto-char 31)
    (insert "......")
    (goto-char 18)
    (insert "..")
    (goto-char 14)
    (insert ".............")
    (goto-char 34)
    (insert "............")
    (goto-char 51)
    (widen)
    (narrow-to-region 58 31)
    (goto-char 50)
    (delete-char 5)
    (goto-char 53)
    (insert ".........")
    (goto-char 56)
    (insert "...............")
    (goto-char 45)
    (delete-char 1)
    (goto-char 67)
    (insert "............")
    (goto-char 84)
    (insert "")
    (goto-char 39)
    (delete-char 27)
    (goto-char 39)
    (delete-char 21)
    (goto-char 32)
    (insert "............")
    (goto-char 36)
    (widen)
    (narrow-to-region 7 37)
    (goto-char 11)
    (insert ".......")
    (goto-char 21)
    (delete-char 13)
    (goto-char 15)
    (insert "....")
    (goto-char 9)
    (insert ".............")
    (goto-char 13)
    (delete-char 21)
    (goto-char 21)
    (delete-char 6)
    (goto-char 16)
    (insert ".......")
    (goto-char 22)
    (insert "")
    (goto-char 27)
    (delete-char 0)
    (should
     (equal
      (test-overlay-regions)
      '((3 . 42)
        (4 . 16)
        (5 . 83)
        (13 . 51)
        (25 . 27))))))

(ert-deftest overlay-autogenerated-test-60 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 38 32 nil t nil)
    (make-overlay 32 42 nil t nil)
    (make-overlay 29 11 nil nil t)
    (make-overlay 52 22 nil t t)
    (make-overlay 39 59 nil t nil)
    (make-overlay 41 30 nil t t)
    (make-overlay 29 61 nil nil t)
    (make-overlay 11 45 nil nil nil)
    (make-overlay 46 17 nil nil t)
    (make-overlay 35 51 nil t t)
    (make-overlay 22 13 nil nil t)
    (make-overlay 52 34 nil nil t)
    (make-overlay 59 4 nil nil t)
    (make-overlay 8 22 nil nil nil)
    (make-overlay 4 49 nil nil nil)
    (make-overlay 52 45 nil t t)
    (goto-char 48)
    (delete-char 16)
    (goto-char 37)
    (delete-char 8)
    (goto-char 14)
    (insert "...............")
    (goto-char 40)
    (delete-char 16)
    (goto-char 19)
    (insert ".........")
    (goto-char 16)
    (insert "......")
    (goto-char 10)
    (insert "........")
    (goto-char 11)
    (insert "...............")
    (goto-char 22)
    (insert ".")
    (goto-char 62)
    (delete-char 16)
    (goto-char 14)
    (delete-char 11)
    (goto-char 47)
    (insert "....")
    (goto-char 33)
    (insert ".............")
    (goto-char 49)
    (delete-char 13)
    (goto-char 28)
    (insert "..")
    (goto-char 35)
    (delete-char 13)
    (goto-char 44)
    (insert "....")
    (goto-char 34)
    (delete-char 14)
    (goto-char 23)
    (insert ".....")
    (goto-char 25)
    (delete-char 4)
    (goto-char 33)
    (insert ".....")
    (goto-char 27)
    (delete-char 3)
    (goto-char 16)
    (widen)
    (narrow-to-region 36 37)
    (goto-char 36)
    (delete-char 1)
    (goto-char 36)
    (insert ".......")
    (goto-char 37)
    (widen)
    (narrow-to-region 35 31)
    (goto-char 34)
    (delete-char 0)
    (goto-char 31)
    (delete-char 2)
    (goto-char 31)
    (widen)
    (narrow-to-region 24 3)
    (goto-char 22)
    (delete-char 2)
    (goto-char 22)
    (insert ".............")
    (goto-char 4)
    (insert ".")
    (should
     (equal
      (test-overlay-regions)
      '((4 . 54)
        (4 . 54)
        (9 . 46))))))

(ert-deftest overlay-autogenerated-test-61 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 45 56 nil t nil)
    (make-overlay 60 45 nil nil nil)
    (make-overlay 26 8 nil t t)
    (make-overlay 63 39 nil nil nil)
    (make-overlay 18 11 nil t nil)
    (make-overlay 22 64 nil nil t)
    (make-overlay 8 41 nil nil t)
    (make-overlay 6 51 nil t t)
    (make-overlay 38 26 nil t t)
    (make-overlay 7 46 nil t nil)
    (make-overlay 2 42 nil nil t)
    (make-overlay 44 64 nil nil nil)
    (make-overlay 7 62 nil t nil)
    (make-overlay 8 40 nil nil t)
    (make-overlay 62 36 nil t t)
    (make-overlay 61 27 nil nil nil)
    (goto-char 21)
    (delete-char 0)
    (goto-char 8)
    (insert "")
    (goto-char 55)
    (insert "......")
    (goto-char 38)
    (delete-char 25)
    (goto-char 37)
    (delete-char 4)
    (goto-char 12)
    (delete-char 4)
    (goto-char 3)
    (delete-char 26)
    (goto-char 10)
    (insert ".......")
    (goto-char 18)
    (delete-char 0)
    (goto-char 16)
    (insert ".............")
    (goto-char 18)
    (delete-char 3)
    (goto-char 7)
    (insert "...")
    (goto-char 20)
    (insert "........")
    (goto-char 38)
    (delete-char 0)
    (goto-char 1)
    (delete-char 36)
    (goto-char 3)
    (delete-char 1)
    (goto-char 2)
    (insert "......")
    (goto-char 4)
    (insert ".......")
    (goto-char 2)
    (insert "...........")
    (goto-char 27)
    (insert ".....")
    (goto-char 15)
    (insert "...............")
    (goto-char 2)
    (insert "......")
    (goto-char 17)
    (delete-char 8)
    (goto-char 15)
    (delete-char 7)
    (goto-char 33)
    (delete-char 5)
    (goto-char 13)
    (insert "...........")
    (goto-char 34)
    (insert "...............")
    (goto-char 33)
    (insert "")
    (goto-char 51)
    (insert "....")
    (goto-char 14)
    (delete-char 36)
    (goto-char 16)
    (delete-char 1)
    (goto-char 14)
    (delete-char 8)
    (should
     (equal
      (test-overlay-regions)
      '((1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 1)
        (1 . 18)
        (1 . 18))))))

(ert-deftest overlay-autogenerated-test-62 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 25 36 nil t nil)
    (make-overlay 38 6 nil t nil)
    (make-overlay 40 63 nil nil t)
    (make-overlay 34 23 nil nil nil)
    (make-overlay 48 46 nil nil nil)
    (make-overlay 43 57 nil t t)
    (make-overlay 6 53 nil t t)
    (make-overlay 37 27 nil t t)
    (make-overlay 8 39 nil t nil)
    (make-overlay 62 6 nil nil nil)
    (make-overlay 51 6 nil t t)
    (make-overlay 58 11 nil nil t)
    (make-overlay 19 25 nil t nil)
    (make-overlay 13 8 nil nil nil)
    (make-overlay 19 8 nil nil t)
    (make-overlay 39 5 nil t t)
    (goto-char 51)
    (delete-char 5)
    (goto-char 16)
    (delete-char 9)
    (goto-char 18)
    (insert "")
    (goto-char 47)
    (delete-char 4)
    (goto-char 24)
    (insert ".........")
    (goto-char 24)
    (insert ".....")
    (goto-char 18)
    (insert "...........")
    (goto-char 5)
    (delete-char 6)
    (goto-char 30)
    (insert "...........")
    (goto-char 8)
    (insert ".............")
    (goto-char 78)
    (insert "............")
    (goto-char 67)
    (insert "")
    (goto-char 58)
    (insert "")
    (goto-char 5)
    (insert ".")
    (goto-char 79)
    (widen)
    (narrow-to-region 51 55)
    (goto-char 51)
    (insert "....")
    (goto-char 58)
    (widen)
    (narrow-to-region 36 37)
    (goto-char 37)
    (insert "....")
    (goto-char 40)
    (insert ".......")
    (goto-char 47)
    (delete-char 1)
    (goto-char 43)
    (delete-char 4)
    (goto-char 37)
    (insert "........")
    (goto-char 49)
    (insert "............")
    (goto-char 42)
    (widen)
    (narrow-to-region 75 111)
    (goto-char 104)
    (widen)
    (narrow-to-region 21 95)
    (goto-char 22)
    (widen)
    (narrow-to-region 64 79)
    (goto-char 64)
    (delete-char 0)
    (goto-char 68)
    (insert "........")
    (goto-char 82)
    (insert "")
    (goto-char 81)
    (insert "........")
    (goto-char 92)
    (delete-char 2)
    (goto-char 87)
    (insert ".")
    (should
     (equal
      (test-overlay-regions)
      '((5 . 145)
        (5 . 148)
        (6 . 118)
        (6 . 119)
        (6 . 119)
        (6 . 143)
        (6 . 143)
        (24 . 114)
        (24 . 116)
        (63 . 117))))))

(ert-deftest overlay-autogenerated-test-63 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 9 49 nil t nil)
    (make-overlay 9 16 nil nil nil)
    (make-overlay 64 2 nil t t)
    (make-overlay 17 31 nil nil t)
    (make-overlay 24 51 nil nil nil)
    (make-overlay 27 56 nil t t)
    (make-overlay 21 4 nil nil nil)
    (make-overlay 24 29 nil t t)
    (make-overlay 4 63 nil nil t)
    (make-overlay 34 49 nil t nil)
    (make-overlay 19 47 nil nil t)
    (make-overlay 8 50 nil t nil)
    (make-overlay 49 61 nil t nil)
    (make-overlay 52 10 nil t t)
    (make-overlay 64 30 nil t nil)
    (make-overlay 5 13 nil t nil)
    (goto-char 27)
    (insert "........")
    (goto-char 42)
    (insert "......")
    (goto-char 48)
    (insert "....")
    (goto-char 55)
    (widen)
    (narrow-to-region 10 5)
    (goto-char 8)
    (insert ".............")
    (goto-char 19)
    (insert "......")
    (goto-char 19)
    (delete-char 3)
    (goto-char 8)
    (delete-char 3)
    (goto-char 9)
    (insert ".......")
    (goto-char 29)
    (insert "...............")
    (goto-char 38)
    (insert ".......")
    (goto-char 34)
    (insert "......")
    (goto-char 28)
    (delete-char 20)
    (goto-char 22)
    (insert "............")
    (goto-char 21)
    (delete-char 23)
    (goto-char 25)
    (delete-char 2)
    (goto-char 19)
    (delete-char 2)
    (goto-char 12)
    (delete-char 6)
    (goto-char 12)
    (delete-char 0)
    (goto-char 13)
    (delete-char 0)
    (goto-char 12)
    (insert "........")
    (goto-char 23)
    (delete-char 2)
    (goto-char 5)
    (insert "...............")
    (goto-char 28)
    (delete-char 0)
    (goto-char 16)
    (insert "..........")
    (goto-char 8)
    (delete-char 17)
    (goto-char 27)
    (delete-char 0)
    (goto-char 12)
    (insert ".")
    (goto-char 14)
    (delete-char 12)
    (goto-char 11)
    (insert "..............")
    (goto-char 34)
    (insert "")
    (goto-char 25)
    (delete-char 8)
    (should
     (equal
      (test-overlay-regions)
      '((2 . 98)
        (4 . 37)
        (4 . 97)
        (25 . 29)
        (25 . 32)
        (25 . 84))))))

(ert-deftest overlay-autogenerated-test-64 nil
  (with-temp-buffer
    (insert "................................................................")
    (make-overlay 31 10 nil nil nil)
    (make-overlay 17 58 nil nil t)
    (make-overlay 20 21 nil t nil)
    (make-overlay 3 47 nil t t)
    (make-overlay 47 43 nil t t)
    (make-overlay 54 8 nil nil t)
    (make-overlay 51 26 nil t nil)
    (make-overlay 60 14 nil t nil)
    (make-overlay 38 6 nil nil t)
    (make-overlay 41 9 nil nil nil)
    (make-overlay 44 38 nil nil t)
    (make-overlay 55 48 nil nil t)
    (make-overlay 10 41 nil nil t)
    (make-overlay 35 49 nil t nil)
    (make-overlay 50 46 nil nil nil)
    (make-overlay 28 28 nil t nil)
    (goto-char 59)
    (delete-char 3)
    (goto-char 28)
    (widen)
    (narrow-to-region 13 7)
    (goto-char 11)
    (insert ".")
    (goto-char 9)
    (delete-char 3)
    (goto-char 8)
    (delete-char 0)
    (goto-char 7)
    (insert ".............")
    (goto-char 9)
    (insert "..........")
    (goto-char 22)
    (delete-char 1)
    (goto-char 31)
    (delete-char 2)
    (goto-char 22)
    (insert ".........")
    (goto-char 33)
    (delete-char 1)
    (goto-char 29)
    (widen)
    (narrow-to-region 59 51)
    (goto-char 52)
    (insert ".........")
    (goto-char 53)
    (insert "........")
    (goto-char 53)
    (delete-char 4)
    (goto-char 54)
    (insert "........")
    (goto-char 53)
    (insert "....")
    (goto-char 75)
    (widen)
    (goto-char 70)
    (delete-char 2)
    (goto-char 108)
    (delete-char 1)
    (goto-char 80)
    (widen)
    (goto-char 70)
    (widen)
    (narrow-to-region 49 63)
    (goto-char 49)
    (insert "...")
    (goto-char 66)
    (delete-char 0)
    (goto-char 63)
    (delete-char 3)
    (goto-char 59)
    (insert "..........")
    (goto-char 56)
    (delete-char 6)
    (goto-char 60)
    (insert ".........")
    (goto-char 62)
    (widen)
    (goto-char 58)
    (insert ".............")
    (goto-char 105)
    (widen)
    (narrow-to-region 94 109)
    (goto-char 103)
    (insert "............")
    (should
     (equal
      (test-overlay-regions)
      '((3 . 134)
        (6 . 125)
        (38 . 141)
        (39 . 118)
        (39 . 128)
        (39 . 128)
        (40 . 146)
        (43 . 145)
        (101 . 138)
        (103 . 103))))))

) ;; End of `when nil' for autogenerated insert/delete/narrow tests.

(ert-deftest buffer-multibyte-overlong-sequences ()
  (dolist (uni '("\xE0\x80\x80"
                 "\xF0\x80\x80\x80"
                 "\xF8\x8F\xBF\xBF\x80"))
    (let ((multi (string-to-multibyte uni)))
      (should
       (string-equal
        multi
        (with-temp-buffer
          (set-buffer-multibyte nil)
          (insert uni)
          (set-buffer-multibyte t)
          (buffer-string)))))))

;; https://debbugs.gnu.org/33492
(ert-deftest buffer-tests-buffer-local-variables-undo ()
  "Test that `buffer-undo-list' appears in `buffer-local-variables'."
  (with-temp-buffer
    (should (assq 'buffer-undo-list (buffer-local-variables)))))

(ert-deftest buffer-tests-inhibit-buffer-hooks ()
  "Test `get-buffer-create' argument INHIBIT-BUFFER-HOOKS."
  (let* (run-bluh (bluh (lambda () (setq run-bluh t))))
    (unwind-protect
        (let* ( run-kbh  (kbh  (lambda () (setq run-kbh  t)))
                run-kbqf (kbqf (lambda () (setq run-kbqf t))) )

          ;; Inhibited.
          (add-hook 'buffer-list-update-hook bluh)
          (with-current-buffer (generate-new-buffer " foo" t)
            (add-hook 'kill-buffer-hook kbh nil t)
            (add-hook 'kill-buffer-query-functions kbqf nil t)
            (kill-buffer))
          (with-temp-buffer (ignore))
          (with-output-to-string (ignore))
          (should-not run-bluh)
          (should-not run-kbh)
          (should-not run-kbqf)

          ;; Not inhibited.
          (with-current-buffer (generate-new-buffer " foo")
            (should run-bluh)
            (add-hook 'kill-buffer-hook kbh nil t)
            (add-hook 'kill-buffer-query-functions kbqf nil t)
            (kill-buffer))
          (should run-kbh)
          (should run-kbqf))
      (remove-hook 'buffer-list-update-hook bluh))))

(ert-deftest buffer-tests-inhibit-buffer-hooks-indirect ()
  "Test `make-indirect-buffer' argument INHIBIT-BUFFER-HOOKS."
  (let* ( base run-bluh run-kbh run-kbqf
          (bluh (lambda () (setq run-bluh t)))
          (kbh  (lambda () (setq run-kbh  t)))
          (kbqf (lambda () (setq run-kbqf t))))
    (dolist (inhibit-base '(nil t))
      (unwind-protect
          (let (indirect)
            (setq base (generate-new-buffer " base" inhibit-base))
            (dolist (inhibit-indirect '(nil t))
              (dotimes (_ 11)
                (unwind-protect
                    (let ((name (generate-new-buffer-name " indirect")))
                      (setq run-bluh nil run-kbh nil run-kbqf nil)
                      (add-hook 'buffer-list-update-hook bluh)
                      (with-current-buffer
                          (setq indirect (make-indirect-buffer
                                          base name nil inhibit-indirect))
                        (add-hook 'kill-buffer-hook kbh nil t)
                        (add-hook 'kill-buffer-query-functions kbqf nil t)
                        (kill-buffer))
                      (should (xor inhibit-indirect run-bluh))
                      (should (xor inhibit-indirect run-kbh))
                      (should (xor inhibit-indirect run-kbqf)))
                  (remove-hook 'buffer-list-update-hook bluh)
                  (when (buffer-live-p indirect)
                    (kill-buffer indirect))))))
        (when (buffer-live-p base)
          (kill-buffer base))))))

(ert-deftest zero-length-overlays-and-not ()
  (with-temp-buffer
    (insert "hello")
    (let ((long-overlay (make-overlay 2 4))
          (zero-overlay (make-overlay 3 3)))
      ;; Exclude.
      (should (= (length (overlays-at 3)) 1))
      (should (eq (car (overlays-at 3)) long-overlay))
      ;; Include.
      (should (= (length (overlays-in 3 3)) 2))
      (should (memq long-overlay (overlays-in 3 3)))
      (should (memq zero-overlay (overlays-in 3 3))))))

(ert-deftest test-remove-overlays ()
  (with-temp-buffer
    (insert "foo")
    (make-overlay (point) (point))
    (should (= (length (overlays-in (point-min) (point-max))) 1))
    (remove-overlays)
    (should (= (length (overlays-in (point-min) (point-max))) 0)))

  (with-temp-buffer
    (insert "foo")
    (goto-char 2)
    (make-overlay (point) (point))
    ;; We only count zero-length overlays at the end of the buffer.
    (should (= (length (overlays-in 1 2)) 0))
    (narrow-to-region 1 2)
    ;; We've now narrowed, so the zero-length overlay is at the end of
    ;; the (accessible part of the) buffer.
    (should (= (length (overlays-in 1 2)) 1))
    (remove-overlays)
    (should (= (length (overlays-in (point-min) (point-max))) 0))))

(defun test-kill-buffer-auto-save (auto-save-answer body-func)
  "Test helper for `kill-buffer-delete-auto-save' tests.

Call BODY-FUNC with the current buffer set to a buffer visiting a
temporary file.  Around the call, mock the \"Buffer modified;
kill anyway?\" and \"Delete auto-save file?\" prompts, answering
\"yes\" for the former and AUTO-SAVE-ANSWER for the latter.  The
expectation should be the characters `?y' or `?n', or `nil' if no
prompt is expected.  The test fails if the \"Delete auto-save
file?\" prompt does not either prompt is not issued as expected.
Finally, kill the buffer and its temporary file."
  (ert-with-temp-file file
    (should (file-exists-p file))
    (save-excursion
      (find-file file)
      (should (equal file (buffer-file-name)))
      (let ((buffer (current-buffer))
            (auto-save-prompt-happened nil))
        (cl-letf (((symbol-function #'read-multiple-choice)
                   (lambda (prompt choices &rest _)
                     (should (string-search "modified; kill anyway?" prompt))
                     (let ((answer (assq ?y choices)))
                       (should answer)
                       answer)))
                  ((symbol-function #'yes-or-no-p)
                   (lambda (prompt)
                     (should (string-search "Delete auto-save file?" prompt))
                     (setq auto-save-prompt-happened t)
                     (pcase-exhaustive auto-save-answer
                       (?y t)
                       (?n nil)))))
          (funcall body-func)
          (should (equal (null auto-save-prompt-happened)
                         (null auto-save-answer))))
        (when (buffer-live-p buffer)
          (with-current-buffer buffer
            (set-buffer-modified-p nil)
            (kill-buffer)))))))

(ert-deftest test-kill-buffer-auto-save-default ()
  (let ((kill-buffer-delete-auto-save-files nil))
    (test-kill-buffer-auto-save
     nil
     (lambda ()
       (let (auto-save)
         (auto-save-mode t)
         (insert "foo\n")
         (should buffer-auto-save-file-name)
         (setq auto-save buffer-auto-save-file-name)
         (do-auto-save t)
         (should (file-exists-p auto-save))
         (kill-buffer (current-buffer))
         (should (file-exists-p auto-save)))))))

(ert-deftest test-kill-buffer-auto-save-delete-yes ()
  (let ((kill-buffer-delete-auto-save-files t))
    (test-kill-buffer-auto-save
     ?y
     (lambda ()
       (let (auto-save)
         (auto-save-mode t)
         (insert "foo\n")
         (should buffer-auto-save-file-name)
         (setq auto-save buffer-auto-save-file-name)
         (do-auto-save t)
         (should (file-exists-p auto-save))
         ;; This should delete the auto-save file.
         (kill-buffer (current-buffer))
         (should-not (file-exists-p auto-save)))))))

(ert-deftest test-kill-buffer-auto-save-delete-no ()
  (let ((kill-buffer-delete-auto-save-files t))
    (test-kill-buffer-auto-save
     ?n
     (lambda ()
       (let (auto-save)
         (auto-save-mode t)
         (insert "foo\n")
         (should buffer-auto-save-file-name)
         (setq auto-save buffer-auto-save-file-name)
         (do-auto-save t)
         (should (file-exists-p auto-save))
         ;; This should not delete the auto-save file.
         (kill-buffer (current-buffer))
         (should (file-exists-p auto-save))
         (delete-file auto-save))))))

(ert-deftest test-buffer-modifications ()
  (ert-with-temp-file file
    (with-current-buffer (find-file file)
      (auto-save-mode 1)
      (should-not (buffer-modified-p))
      (insert "foo")
      (should (buffer-modified-p))
      (should-not (eq (buffer-modified-p) 'autosaved))
      (do-auto-save t t)
      (should (eq (buffer-modified-p) 'autosaved))
      (with-silent-modifications
        (put-text-property 1 3 'face 'bold))
      (should (eq (buffer-modified-p) 'autosaved))
      (save-buffer)
      (should-not (buffer-modified-p))
      (with-silent-modifications
        (put-text-property 1 3 'face 'italic))
      (should-not (buffer-modified-p)))))

(ert-deftest test-restore-buffer-modified-p ()
  (ert-with-temp-file file
    ;; This avoids the annoying "foo and bar are the same file" on
    ;; MS-Windows.
    (setq file (file-truename file))
    (with-current-buffer (find-file file)
      (auto-save-mode 1)
      (should-not (eq (buffer-modified-p) t))
      (insert "foo")
      (should (buffer-modified-p))
      (restore-buffer-modified-p nil)
      (should-not (buffer-modified-p))
      (insert "bar")
      (do-auto-save t t)
      (should (eq (buffer-modified-p) 'autosaved))
      (insert "zot")
      (restore-buffer-modified-p 'autosaved)
      (should (eq (buffer-modified-p) 'autosaved))

      ;; Clean up.
      (when (file-exists-p buffer-auto-save-file-name)
        (delete-file buffer-auto-save-file-name))))

  (ert-with-temp-file file
    (setq file (file-truename file))
    (with-current-buffer (find-file file)
      (auto-save-mode 1)
      (should-not (eq (buffer-modified-p) t))
      (insert "foo")
      (should (buffer-modified-p))
      (should-not (eq (buffer-modified-p) 'autosaved))
      (restore-buffer-modified-p 'autosaved)
      (should (eq (buffer-modified-p) 'autosaved)))))

(ert-deftest test-buffer-chars-modified-ticks ()
  "Test `buffer-chars-modified-tick'."
  (setq temporary-file-directory (file-truename temporary-file-directory))
  (let ((text "foobar")
        f1 f2)
    (unwind-protect
        (progn
          (setq f1 (make-temp-file "buf-modiff-tests")
                f2 (make-temp-file "buf-modiff-tests"))
          (with-current-buffer (find-file f1)
            (should (= (buffer-chars-modified-tick) 1))
            (should (= (buffer-chars-modified-tick) (buffer-modified-tick)))
            (write-region text nil f2 nil 'silent)
            (insert-file-contents f2)
            (should (= (buffer-chars-modified-tick) (buffer-modified-tick)))
            (should (> (buffer-chars-modified-tick) 1))))
      (if f1 (delete-file f1))
      (if f2 (delete-file f2))
      )))

(ert-deftest test-labeled-narrowing ()
  "Test `with-restriction' and `without-restriction'."
  (with-current-buffer (generate-new-buffer " foo" t)
    (insert (make-string 5000 ?a))
    (should (= (point-min) 1))
    (should (= (point-max) 5001))
    (with-restriction
     100 500 :label 'foo
     (should (= (point-min) 100))
     (should (= (point-max) 500))
     (widen)
     (should (= (point-min) 100))
     (should (= (point-max) 500))
     (narrow-to-region 1 5000)
     (should (= (point-min) 100))
     (should (= (point-max) 500))
     (narrow-to-region 50 150)
     (should (= (point-min) 100))
     (should (= (point-max) 150))
     (widen)
     (should (= (point-min) 100))
     (should (= (point-max) 500))
     (narrow-to-region 400 1000)
     (should (= (point-min) 400))
     (should (= (point-max) 500))
     (without-restriction
      :label 'bar
      (should (= (point-min) 100))
      (should (= (point-max) 500)))
     (without-restriction
      :label 'foo
      (should (= (point-min) 1))
      (should (= (point-max) 5001)))
     (should (= (point-min) 400))
     (should (= (point-max) 500))
     (widen)
     (should (= (point-min) 100))
     (should (= (point-max) 500))
     (with-restriction
      50 250 :label 'bar
      (should (= (point-min) 100))
      (should (= (point-max) 250))
      (widen)
      (should (= (point-min) 100))
      (should (= (point-max) 250))
      (without-restriction
       :label 'bar
       (should (= (point-min) 100))
       (should (= (point-max) 500))
       (without-restriction
        :label 'foo
        (should (= (point-min) 1))
        (should (= (point-max) 5001)))
       (should (= (point-min) 100))
       (should (= (point-max) 500)))
      (should (= (point-min) 100))
      (should (= (point-max) 250)))
     (should (= (point-min) 100))
     (should (= (point-max) 500))
     (with-restriction
      50 250 :label 'bar
      (should (= (point-min) 100))
      (should (= (point-max) 250))
      (with-restriction
       150 500 :label 'baz
       (should (= (point-min) 150))
       (should (= (point-max) 250))
       (without-restriction
        :label 'bar
        (should (= (point-min) 150))
        (should (= (point-max) 250)))
       (without-restriction
        :label 'foo
        (should (= (point-min) 150))
        (should (= (point-max) 250)))
       (without-restriction
        :label 'baz
        (should (= (point-min) 100))
        (should (= (point-max) 250))
        (without-restriction
         :label 'foo
         (should (= (point-min) 100))
         (should (= (point-max) 250)))
        (without-restriction
         :label 'bar
         (should (= (point-min) 100))
         (should (= (point-max) 500))
         (without-restriction
          :label 'foobar
          (should (= (point-min) 100))
          (should (= (point-max) 500)))
         (without-restriction
          :label 'foo
          (should (= (point-min) 1))
          (should (= (point-max) 5001)))
         (should (= (point-min) 100))
         (should (= (point-max) 500)))
        (should (= (point-min) 100))
        (should (= (point-max) 250)))
       (should (= (point-min) 150))
       (should (= (point-max) 250)))
      (should (= (point-min) 100))
      (should (= (point-max) 250))))
    (should (= (point-min) 1))
    (should (= (point-max) 5001))))

;;; buffer-tests.el ends here
