/*
 *  GNU Kart
 *
 *  Copyright (C) 2010 Eric P. Hutchins
 *
 *  GNU Kart is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  GNU Kart is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Kart.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "SDL.h"
#include "SDL_image.h"
#include "SDL_mixer.h"

#include <GL/gl.h>
#include <GL/glu.h>

#include "config.h"
#include "kart.h"
#include "model.h"

#define PI 3.1415926535

long start_time;

int frames;

int fps = 50;

long frame_time = 1000 / 50;

double drag = 0.0005;

int running = 0;

long last_time;

int screen_width = 320;
int screen_height = 240;

int sight_distance = 500;

int quality = 0;

int smooth = 0;

GLuint textures[10];
int texture_count = 0;

/* A point of the track has an X, Y, Z location and a radius for the sphere
 * around it */
typedef struct
{
  double x;
  double y;
  double z;
  double r;
} track_point;

/* A track edge connects two track points. */
typedef struct
{
  int v1;
  int v2;
} track_edge;

int start_edge = 0;

int finish_edge = 0;

int n_points = 4;
int n_edges = 4;

Model *track_model = NULL;

track_point **points;

track_edge *edges;

typedef struct
{
  double size;
  double x;
  double y;
  double z;
  double direction;
  double r;
  double g;
  double b;
} box;

box boxes[400];

/* All karts */
Kart *gnu = NULL;
Kart *guile = NULL;
Kart *babygnu = NULL;
Kart *wilber = NULL;
Kart *icecat = NULL;
Kart *ghost = NULL;

// An array of karts
Kart **karts = NULL;
int n_karts = 0;

/* Should always be updated to the current player */
Kart *player = NULL;

Mix_Music *music = NULL;

double turn_speed = 2;

int turning = 0;
int forward = 0;

double acceleration = 0.0075;
double brake = 0.05;
double max_speed = 5;
double max_reverse = -0.5;

double camx = 0;
double camy = 8;
double camz = 30;
double camdir = 0;

int fullscreen = 0;

int
parse_args (int argc, char *argv[])
{
  int i;
  for (i = 0; i < argc; i++)
    {
      if (strcmp (argv[i], "--help") == 0 || strcmp (argv[i], "-?") == 0)
        {
          printf ("usage: gnukart [OPTIONS]\n");
          printf ("  -f, --fullscreen           set fullscreen\n");
          printf ("  -q, --quality=QUALITY      set quality\n");
          printf ("  -r, --frame-rate=RATE      set framerate\n");
          printf ("  -d, --sight-distance=DIST  set the distance that you\n");
          printf ("                             can see forward\n");
          printf ("  -w, --width=WIDTH          set screen width\n");
          printf ("  -h, --height=HEIGHT        set screen height\n");
          printf ("  -v, --version              display version information and exit\n");
          printf ("  -?, --help                 display this help and exit\n");
          return -1;
        }
      if (strcmp (argv[i], "--version") == 0 || strcmp (argv[i], "-v") == 0)
        {
          printf ("GNU %s\n", PACKAGE_STRING);
          printf ("Copyright (C) 2009 Free Software Foundation\n");
          return -1;
        }
      if (strcmp (argv[i], "-f") == 0 || strcmp (argv[i], "--fullscreen") == 0)
        fullscreen = 1;
      if (strcmp (argv[i], "--smooth") == 0)
        {
          smooth = 1;
        }
      if (strcmp (argv[i], "--quality") == 0 || strcmp (argv[i], "-q") == 0)
        {
          if (argc < i + 1) return;
          else
            {
              i++;
              quality = atoi (argv[i]);
            }
        }
      if (strcmp (argv[i], "--frame-rate") == 0 || strcmp (argv[i], "-r") == 0)
        {
          if (argc < i + 1) return;
          else
            {
              i++;
              fps = atoi (argv[i]);
              frame_time = 1000 / fps;
            }
        }
      if (strcmp (argv[i], "--sight-distance") == 0 || strcmp (argv[i], "-d") == 0)
        {
          if (argc < i + 1) return;
          else
            {
              i++;
              sight_distance = atoi (argv[i]);
            }
        }
      if (strcmp (argv[i], "--width") == 0 || strcmp (argv[i], "-w") == 0)
        {
          if (argc < i + 1) return;
          else
            {
              i++;
              screen_width = atoi (argv[i]);
            }
        }
      if (strcmp (argv[i], "--height") == 0 || strcmp (argv[i], "-h") == 0)
        {
          if (argc < i + 1) return;
          else
            {
              i++;
              screen_height = atoi (argv[i]);
            }
        }
    }
  return 0;
}

void
gl_init ()
{
  /* Make the properties of a light */
  GLfloat light_ambient[] = {0.3f, 0.3f, 0.3f, 0.3f};
  GLfloat light_diffuse[] = {1.0f, 1.0f, 1.0f, 0.0f};
  GLfloat light_specular[] = {0.5f, 0.5f, 0.5f, 0.0f};
  GLfloat light_position[] = {0.0f, 10.0f, 0.f, 1.0f};

  /* Make a light using the properties */
  glLightfv (GL_LIGHT1, GL_AMBIENT, light_ambient);
  glLightfv (GL_LIGHT1, GL_DIFFUSE, light_diffuse);
  glLightfv (GL_LIGHT1, GL_SPECULAR, light_specular);
  glLightfv (GL_LIGHT1, GL_POSITION, light_position);

  /* Enable the light and enable lighting in OpenGL */
  glEnable (GL_LIGHT1);
  glEnable (GL_LIGHTING);

  glEnable (GL_TEXTURE_2D);
  glShadeModel (GL_SMOOTH);
  if (smooth) glShadeModel (GL_SMOOTH);
  glClearColor (0.2f, 0.2f, 0.5f, 0.0f);
  glClearDepth (1.0f);
  glEnable (GL_DEPTH_TEST);
  glDepthFunc (GL_LEQUAL);
  switch (quality)
    {
      case 0:
        glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
        break;
      default:
        glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
        break;
    }

  GLfloat ratio = (GLfloat)screen_width / (GLfloat)screen_height;
  glViewport (0, 0, (GLsizei)screen_width, (GLsizei)screen_height);
  glMatrixMode( GL_PROJECTION );
  glLoadIdentity( );
  gluPerspective( 45.0f, ratio, 0.1f, (float)sight_distance );
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity( );
}

void
draw_box (box a)
{
  glLoadIdentity ();

  glRotatef (-camdir, 0.0f, 1.0f, 0.0f);

  glTranslatef (a.x, a.y, a.z);
  glTranslatef (-camx, -camy, -camz);

  glRotatef (a.direction, 0.0f, 1.0f, 0.0f);

  glScalef (1, 1, a.size);

  float box_color[] = {a.r, a.g, a.b, 1.0f};
  glBegin(GL_QUADS);
    glMaterialfv (GL_FRONT, GL_AMBIENT_AND_DIFFUSE, box_color);
    glNormal3f (0.0f, 0.0f, -1.0f);
    glVertex3f (-0.5, 0, -0.5f);
    glVertex3f (-0.5, 1.0, -0.5f);
    glVertex3f (0.5, 1.0, -0.5f);
    glVertex3f (0.5, 0, -0.5f);

    glNormal3f (0.0f, 0.0f, 1.0f);
    glVertex3f (-0.5, 0, 0.5f);
    glVertex3f (0.5, 0, 0.5f);
    glVertex3f (0.5, 1.0, 0.5f);
    glVertex3f (-0.5, 1.0, 0.5f);

    glNormal3f (-1.0f, 0.0f, 0.0f);
    glVertex3f (-0.5, 0, 0.5f);
    glVertex3f (-0.5, 1.0, 0.5f);
    glVertex3f (-0.5, 1.0, -0.5f);
    glVertex3f (-0.5, 0, -0.5f);

    glNormal3f (0.0f, 1.0f, 0.0f);
    glVertex3f (-0.5, 1.0, 0.5f);
    glVertex3f (0.5, 1.0, 0.5f);
    glVertex3f (0.5, 1.0, -0.5f);
    glVertex3f (-0.5, 1.0, -0.5f);

    glNormal3f (1.0f, 0.0f, 0.0f);
    glVertex3f (0.5, 0, 0.5f);
    glVertex3f (0.5, 0, -0.5f);
    glVertex3f (0.5, 1.0, -0.5f);
    glVertex3f (0.5, 1.0, 0.5f);
  glEnd();
}

void
render ()
{
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glLoadIdentity ();

  int i;

  /* Draw the track */
  if (track_model != NULL)
    {
      /*                       x  y  z  direction r  g  b */
      model_draw (track_model, 0, 0, 0, 0,        1, 1, 1);
    }

  /* Draw all karts that exist */
  if (gnu != NULL)
    kart_draw (gnu);
  if (guile != NULL)
    kart_draw (guile);
  if (babygnu != NULL)
    kart_draw (babygnu);
  if (wilber != NULL)
    kart_draw (wilber);
  if (icecat != NULL)
    kart_draw (icecat);
  if (ghost != NULL)
    kart_draw (ghost);

  /* Draw track vertex markers */
/*
  glLoadIdentity ();

  glRotatef (-camdir, 0.0f, 1.0f, 0.0f);

  glTranslatef (0, 0, 0);
  glTranslatef (-camx, -camy, -camz);

  glRotatef (0, 0.0f, 1.0f, 0.0f);

  float white[] = {1.0f, 1.0f, 1.0f, 1.0f};
  for (i = 0; i < n_points; i++)
    {
      glBegin(GL_POLYGON);
        glMaterialfv (GL_FRONT, GL_AMBIENT_AND_DIFFUSE, white);
        glNormal3f(0.0f, 0.5f, 0.0f);
        glVertex3f (points[i]->x - points[i]->r, points[i]->y+1, points[i]->z);
        glVertex3f (points[i]->x, points[i]->y+1, points[i]->z - points[i]->r);
        glVertex3f (points[i]->x + points[i]->r, points[i]->y+1, points[i]->z);
        glVertex3f (points[i]->x, points[i]->y+1, points[i]->z + points[i]->r);
      glEnd();
    }

  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, white);
*/


  /* Draw the ground (this should be temporary too, since it should be 
   * either a 3D model itself or at least textured */
/*
  float ground_color[] = {0.9f, 1.0f, 0.2f, 1.0f};
  glBegin(GL_POLYGON);
    glMaterialfv (GL_FRONT, GL_AMBIENT_AND_DIFFUSE, ground_color);
    glNormal3f(0.0f, 1.0f, 0.0f);
    glVertex3f (-50000.0, 0.0, 50000.0f);
    glVertex3f (-50000.0, 0.0, -50000.0f);
    glVertex3f (50000.0, 0.0, -50000.0f);
    glVertex3f (50000.0, 0.0, 50000.0f);
  glEnd();
*/

  SDL_GL_SwapBuffers ();

  frames ++;
  long t = SDL_GetTicks ();
  if (t - start_time >= 2000)
    {
      double seconds = (t - start_time) / 1000.0;
      double fps = frames / seconds;
//      printf ("Frames per second: %lf\n", fps);
      start_time = t;
      frames = 0;
    }
}

void
kart_slow_down (Kart *player, double amount)
{
  player->speed -= 0.0001 + amount * player->speed;
}

void
physics (Kart *player)
{
  /* Slow down on turns */
  if (abs (player->turning) > 0)
    kart_slow_down (player, 0.005);

  /* Change direction if turning is true */
  player->direction -= player->turning * turn_speed;

  /* Normalize direction */
  if (player->direction >= 360.0) player->direction -= 360.0;
  if (player->direction < 0) player->direction += 360.0;

  /* Adjust speed based on drag */
  if (player->speed > 0)
    player->speed -= drag;
  else if (player->speed < 0)
    player->speed += drag;

  /* Make sure it can't go higher or lower than hard set max or min */
  if (player->speed > max_speed) player->speed = max_speed;
  if (player->speed < max_reverse) player->speed = max_reverse;

  /* Change the player's location based on current speed and direction */
  player->x -= player->speed * sin (PI * player->direction / 180.0);
  player->z -= player->speed * cos (PI * player->direction / 180.0);
}

/* A temporary autopilot function that serves as "AI" */
void
autopilot (Kart *kart)
{
  int i;
  int newest_edge = kart->current_edges[kart->n_current_edges - 1];
  // The target point is the endpoint of the newest edge
  track_point *target = points[edges[newest_edge].v2];
  // Get the angle to get to the next target point
  double angle;
  if (target->x == kart->x)
    {
      if (target->z < kart->z)
        {
          angle = 0;
        }
      else
        {
          angle = 180;
        }
    }
  else
    {
      double the_sin = (kart->x - target->x) / sqrt ((target->x - kart->x) * (target->x - kart->x) + (target->z - kart->z) * (target->z - kart->z));
      angle = asin (the_sin) * 180.0 / PI;
      if (target->z > kart->z)
        {
          angle = 180 - angle;
        }
    }
  /* Normalize direction */
  if (angle >= 360.0) angle -= 360.0;
  if (angle < 0) angle += 360.0;
  double diff_theta = angle - kart->direction;
  /* Normalize the difference in angle */
  if (diff_theta >= 360.0) diff_theta -= 360.0;
  if (diff_theta < 0) diff_theta += 360.0;
  // Decide which way to turn
  if (diff_theta < 180.0)
    {
      kart->turning = -1;
    }
  else
    {
      kart->turning = 1;
    }
  // Don't go too fast
  kart->forward = (SDL_GetTicks () % 8 == 0)?0:1;

  return;
}

void
drive (Kart *kart)
{
  if (kart->forward == 1)
    kart->speed += acceleration;
  else if (kart->forward == -1)
    {
      if (kart->speed > 0)
        {
          kart_slow_down (kart, brake);
        }
      else kart->speed -= acceleration;
    }
}

double
dist_from_line (double v_0x, double v_0y, double v_0z,
                double vx, double vy, double vz,
                double px, double py, double pz)
{
  /* Get the k for the equation for the plane that has normal equal to the
   * line */
  double k = vx * px + vy * py + vz * pz;
  /* Get the value for t that the intersection between this plane and the
   * line will fall on */
  double t = (k - (vx * v_0x + vy * v_0y + vz * v_0z))
             / (vx * vx + vy * vy + vz * vz);
  /* Get the actual point from the line that intersects this plane */
  double x = v_0x + t * vx;
  double y = v_0y + t * vy;
  double z = v_0z + t * vz;

  return sqrt ((x - px) * (x - px) + (y - py) * (y - py) + (z - pz) * (z - pz));
}

double
get_pos_on_line_segment (double x1, double y1, double z1,
                        double x2, double y2, double z2,
                        double px, double py, double pz)
{
  /* Get the minimums and maximums for later */
  double min_x = x1; if (x2 < min_x) min_x = x2;
  double min_y = y1; if (y2 < min_y) min_y = y2;
  double min_z = z1; if (z2 < min_z) min_z = z2;
  double max_x = x1; if (x2 > max_x) max_x = x2;
  double max_y = y1; if (y2 > max_y) max_y = y2;
  double max_z = z1; if (z2 > max_z) max_z = z2;
  /* Set the vector direction to the difference of the endpoints */
  double vx = (x2 - x1);
  double vy = (y2 - y1);
  double vz = (z2 - z1);
  /* Get the k for the equation for the plane that has normal equal to the
   * line segment */
  double k = vx * px + vy * py + vz * pz;
  /* Get the value for t that the intersection between this plane and the
   * line will fall on */
  return (k - (vx * x1 + vy * y1 + vz * z1))
           / (vx * vx + vy * vy + vz * vz);
}

double
dist_from_line_segment (double x1, double y1, double z1,
                        double x2, double y2, double z2,
                        double px, double py, double pz)
{
  /* Get the minimums and maximums for later */
  double min_x = x1; if (x2 < min_x) min_x = x2;
  double min_y = y1; if (y2 < min_y) min_y = y2;
  double min_z = z1; if (z2 < min_z) min_z = z2;
  double max_x = x1; if (x2 > max_x) max_x = x2;
  double max_y = y1; if (y2 > max_y) max_y = y2;
  double max_z = z1; if (z2 > max_z) max_z = z2;
  /* Set the vector direction to the difference of the endpoints */
  double vx = (x2 - x1);
  double vy = (y2 - y1);
  double vz = (z2 - z1);
  /* Get the k for the equation for the plane that has normal equal to the
   * line segment */
  double k = vx * px + vy * py + vz * pz;
  /* Get the value for t that the intersection between this plane and the
   * line will fall on */
  double t = (k - (vx * x1 + vy * y1 + vz * z1))
             / (vx * vx + vy * vy + vz * vz);
  /* Get the actual point from the line that intersects this plane */
  double x = x1 + t * vx;
  double y = y1 + t * vy;
  double z = z1 + t * vz;

  double dist1 = sqrt ((x1-px)*(x1-px)+(y1-py)*(y1-py)+(z1-pz)*(z1-pz));
  double dist2 = sqrt ((x2-px)*(x2-px)+(y2-py)*(y2-py)+(z2-pz)*(z2-pz));

  double min_dist;

  if (dist1 < dist2) min_dist = dist1;
  else min_dist = dist2;

  /* If the intersection between the point and the plane with normal equal
   * to the line segment is outside the line segment */
  if (x < min_x || y < min_y || z < min_z
      || x > max_x || y > max_y || z > max_z)
    {
      /* Just return the shortest distance to one of the endpoints */
      return min_dist;
    }

  /* Otherwise, it's the distance between the line segment point (x,y,z) and
   * the given point (px,py,pz) */
  return sqrt ((x-px)*(x-px)+(y-py)*(y-py)+(z-pz)*(z-pz));
}

void
update_edges (Kart *kart)
{
  int i;
  /* Look at all edges */
  for (i = 0; i < n_edges; i++)
    {
      int close_enough = 0;
      /* Get the distance the player is from this edge */
      double d = dist_from_line_segment (points[edges[i].v1]->x,
                                         points[edges[i].v1]->y,
                                         points[edges[i].v1]->z,
                                         points[edges[i].v2]->x,
                                         points[edges[i].v2]->y,
                                         points[edges[i].v2]->z,
                                         kart->x, kart->y, kart->z);
      /* Get the position the player is when projected onto this edge */
      /* How far along is the player? */
      double t = get_pos_on_line_segment (points[edges[i].v1]->x,
                                          points[edges[i].v1]->y,
                                          points[edges[i].v1]->z,
                                          points[edges[i].v2]->x,
                                          points[edges[i].v2]->y,
                                          points[edges[i].v2]->z,
                                          kart->x, kart->y, kart->z);
      /* If we're not on the edge, then the only way to be close enough is
       * to be within the spherical end-points to the edge */
      /* Check if we are within the radius of the end-points */
      if (t < 0 && d < points[edges[i].v1]->r
          || t > 1 && d < points[edges[i].v2]->r)
        {
          close_enough = 1;
        }
      /* Otherwise, we find out how close the player is to the "size" of the
       * edge at this point */
      else
        {
          /* If it isn't close enough to points on the line segment, move on */
          if (t < 0 || t > 1) close_enough = 0;
          /* Get the radius of the edge at this point along the edge */
          double r = points[edges[i].v1]->r * (1-t) + points[edges[i].v2]->r * t;
          /* If we are within that distance, then we are close enough */
          if (d < r) close_enough = 1;
        }

      /* If it's "close enough" to this edge */
      if (close_enough)
        {
          int already_current = 0;
          int j;
          /* See if it's already a current edge */
          for (j = 0;j < kart->n_current_edges; j++)
            {
              if (kart->current_edges[j] == i) already_current = 1;
            }
          
          /* If it's already a current edge, we're done */
          if (!already_current)
            {
              /* For every current edge */
              for (j = 0; j < kart->n_current_edges; j++)
                {
                  /* If the edge we're close to starts where a current edge 
                   * finishes, then it will become a current edge */
                  if (edges[i].v1 == edges[kart->current_edges[j]].v2)
                    {
                      kart->n_current_edges++;
                      kart->current_edges = realloc (kart->current_edges,
                                          sizeof (int) * kart->n_current_edges);
                      kart->current_edges[kart->n_current_edges-1] = i;
                      /* If we just added the finish edge, then we're done */
                      if (i == finish_edge)
                        {
                          running = 0;
                        }
                    }
                }
            }
        }
      /* If it's not close enough to this edge */
      else
        {
          int j;
          /* For every current edge */
          for (j = 0; j < kart->n_current_edges; j++)
            {
              /* If the edge we're looking at is a current edge, remove it */
              if (kart->current_edges[j] == i)
                {
                  int k;
                  /* Slide the ones on top down */
                  for (k = j + 1; k < kart->n_current_edges; k++)
                    {
                      kart->current_edges[k-1] = kart->current_edges[k];
                    }
                  /* Decrement the number of current edges */
                  kart->n_current_edges--;
                  /* Recheck this index, since there's something new here now */
                  j--;
                }
            }
        }
    }
}

void
update ()
{
  int i;
  for (i = 0; i < n_karts; ++i)
    {
      physics (karts[i]);
      if (player != karts[i]) autopilot (karts[i]);
      drive (karts[i]);
      update_edges (karts[i]);
    }

  camdir = player->direction;
  
  camx = player->x + 5.0 * sin (PI * camdir / 180.0);
  camy = player->y + 4.0;
  camz = player->z + 5.0 * cos (PI * camdir / 180.0);

  int prev_n_current_edges = player->n_current_edges;

  if (prev_n_current_edges > 0 && player->n_current_edges == 0)
    {
      printf ("You've driven off the road! Get back on!\n");
    }
}

/* Loads the track */
void
load_track ()
{
  int i;
  char track_path[1024];
  FILE *file;

  /* Get the path of the track */
  strcpy (track_path, DATADIR);
  strcat (track_path, "/");
  strcpy (track_path, "savannah.gkt");

  /* Open the file */
  file = fopen (track_path, "r");

  /* Get the number of points */
  fscanf (file, "%d\n", &n_points);

  points = (track_point**)malloc (sizeof (track_point*) * n_points);

  /* Define the points */
  for (i = 0; i < 4; i++)
    {
      points[i] = (track_point*)malloc (sizeof (track_point));
      fscanf (file, "%lf %lf %lf %lf\n", &(points[i]->x), &(points[i]->y), &(points[i]->z), &(points[i]->r));
    }

  /* The track has 4 edges (point to point) */
  fscanf (file, "%d\n", &n_edges);

  edges = (track_edge*)malloc (sizeof (track_edge) * n_edges);

  /* Get which points the edges connect */
  for (i = 0; i < n_edges; i++)
    {
      fscanf (file, "%d %d\n", &(edges[i].v1), &(edges[i].v2));
    }

  /* Get the start and finish edge */
  fscanf (file, "%d\n", &start_edge);
  fscanf (file, "%d\n", &finish_edge);

  char model_filename[256];

  /* Get the filename of the model */
  fscanf (file, "%s\n", model_filename);

  /* Load track model */
  char model_path[1024];
  strcpy (model_path, DATADIR);
  strcat (model_path, "/");
  strcat (model_path, model_filename);
  printf ("Loading model from \"%s\"\n", model_path);
  track_model = model_new_from_obj (model_path);

  /* Close the track file */
  fclose (file);
}

int
main (int argc, char *argv[])
{
  int i;

  /* Parse arguments */
  if (parse_args (argc, argv) < 0) return 0;

  /* Initialize SDL */
  SDL_Init (SDL_INIT_VIDEO | SDL_INIT_AUDIO);

  /* Initialize SDL_mixer */
  Mix_OpenAudio (MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 2, 2048);

  /* Set flags */
  int flags = SDL_OPENGL | SDL_GL_DOUBLEBUFFER;
  
  /* Check for fullscreen */
  if (fullscreen)
    flags |= SDL_FULLSCREEN;

  /* Set the video mode */
  SDL_Surface *screen;
  screen = SDL_SetVideoMode (screen_width, screen_height, 32, flags);

  /* Hide the cursor if going fullscreen */
  if (fullscreen)
    {
      SDL_ShowCursor (0);
    }

  /* Tell OpenGL to use a double buffer */
  SDL_GL_SetAttribute (SDL_GL_DOUBLEBUFFER, 1);

  /* Initialize OpenGL */
  gl_init ();

  /* Load music */
  char music_path[1024];
  strcpy (music_path, DATADIR);
  strcat (music_path, "/");
  strcat (music_path, "haven_forest.ogg");
  music = Mix_LoadMUS (music_path);
  if (music == NULL)
    {
      printf ("Failed to load music from file \"%s\"\n", music_path);
      exit (1);
    }

  /* Load the track */
  load_track ();

  /* Load the gnu kart */
  char model_path[1024];
  strcpy (model_path, DATADIR);
  strcat (model_path, "/");
  if (quality == 0)
    strcat (model_path, "kart_gnu_low_quality.obj");
  else
    strcat (model_path, "kart_gnu.obj");
  printf ("Loading model from \"%s\"\n", model_path);
  Model *gnu_kart = model_new_from_obj (model_path);

  /* Load the guile kart */
  strcpy (model_path, DATADIR);
  strcat (model_path, "/");
  if (quality == 0)
    strcat (model_path, "kart_guile_low_quality.obj");
  else
    strcat (model_path, "kart_guile.obj");
  printf ("Loading model from \"%s\"\n", model_path);
  Model *guile_kart = model_new_from_obj (model_path);

  /* Load the icecat kart */
  strcpy (model_path, DATADIR);
  strcat (model_path, "/");
  if (quality == 0)
    strcat (model_path, "kart_icecat_low_quality.obj");
  else
    strcat (model_path, "kart_icecat.obj");
  printf ("Loading model from \"%s\"\n", model_path);
  Model *icecat_kart = model_new_from_obj (model_path);

  printf ("loaded models\n");

  gnu = kart_new (2, 0, 0, -10, 0.7, 0.3, 0.2, 0, gnu_kart);
  guile = kart_new (2.5, 1, 0, -8, 0.8, 0.6, 0.4, 0, guile_kart);
  babygnu = kart_new (2.5, 2, 0, -6, 0.9, 0.45, 0.2, 0, gnu_kart);
  wilber = kart_new (1.5, 3, 0, -5, 0.8, 0.8, 0.8, 0, gnu_kart);
  icecat = kart_new (2.5, 4, 0, -2.5, 0.4, 0.3, 0.9, 0, icecat_kart);
  ghost = kart_new (1, 5, 0, -1.5, 0.9, 0.9, 0.9, 0, gnu_kart);

  // Make an array of karts
  n_karts = 6;
  karts = (Kart**)realloc (karts, sizeof (Kart*) * n_karts);
  karts[0] = gnu;
  karts[1] = guile;
  karts[2] = babygnu;
  karts[3] = wilber;
  karts[4] = icecat;
  karts[5] = ghost;

  // For each kart
  for (i = 0; i < n_karts; ++i)
    {
      /* Initialize the current_edges array and set the first one to the start
       * edge */
      if (karts[i]->current_edges == NULL)
        {
          karts[i]->current_edges = malloc (sizeof (int) * 1);
          karts[i]->current_edges[0] = start_edge;
          karts[i]->n_current_edges = 1;
        }
    }

  /* Set the player to the Gnu character */
  player = gnu;
//  player = icecat;

  srand (time (NULL));

  /* Start playing the music */
  Mix_PlayMusic (music, -1);

  last_time = SDL_GetTicks ();
  running = 1;
  while (running)
    {
      long frame_start = SDL_GetTicks ();
      SDL_Event event;
      while (SDL_PollEvent (&event))
        {
          switch (event.type)
            {
              case SDL_KEYUP:
                if (event.key.keysym.sym == SDLK_LEFT)
                  player->turning++;
                if (event.key.keysym.sym == SDLK_RIGHT)
                  player->turning--;
                if (event.key.keysym.sym == SDLK_UP)
                  {
                    player->forward--;
                  }
                if (event.key.keysym.sym == SDLK_DOWN)
                  {
                    player->forward++;
                  }
                break;
              case SDL_KEYDOWN:
                if (event.key.keysym.sym == SDLK_ESCAPE)
                  running = 0;
                if (event.key.keysym.sym == SDLK_LEFT)
                  player->turning--;
                if (event.key.keysym.sym == SDLK_RIGHT)
                  player->turning++;
                if (event.key.keysym.sym == SDLK_UP)
                  player->forward++;
                if (event.key.keysym.sym == SDLK_DOWN)
                  player->forward--;
                if (event.key.keysym.sym == SDLK_1)
                  player = gnu;
                if (event.key.keysym.sym == SDLK_2)
                  player = guile;
                if (event.key.keysym.sym == SDLK_3)
                  player = babygnu;
                if (event.key.keysym.sym == SDLK_4)
                  player = wilber;
                if (event.key.keysym.sym == SDLK_5)
                  player = icecat;
                if (event.key.keysym.sym == SDLK_6)
                  player = ghost;
                break;
              case SDL_QUIT:
                running = 0;
                break;
            }
        }
      int i;
      for (i = 0; i < 50 / fps; i++)
        update ();
      render ();
      long elapsed = SDL_GetTicks () - frame_start;
      if (elapsed < frame_time)
        SDL_Delay (frame_time - elapsed);
    }
  /* Destroy all the models */
  model_destroy (gnu_kart);
  model_destroy (guile_kart);
  model_destroy (icecat_kart);

  /* Destroy all the karts */
  kart_destroy (gnu);
  kart_destroy (guile);
  kart_destroy (babygnu);
  kart_destroy (wilber);
  kart_destroy (icecat);
  kart_destroy (ghost);

  /* Shutdown SDL_mixer */
  Mix_CloseAudio ();

  if (fullscreen)
    {
      SDL_ShowCursor (1);
    }
  SDL_Quit ();
}

