/*
 *  GNU Kart
 *
 *  Copyright (C) 2010 Eric P. Hutchins
 *
 *  GNU Kart is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  GNU Kart is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Kart.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "model.h"

Model
*model_new_from_obj (char *filename)
{
  Model *model = (Model*)malloc (sizeof (Model));
  model->objects = NULL;
  model->materials = NULL;
  FILE* file = fopen (filename, "r");
  model->material_count = 0;
  model->object_count = 0;
  model->vert_count = 0;
  model->normal_count = 0;
  char buffer[512];
  while (fgets (buffer, 512, file))
    {
      char code[512];
      sscanf (buffer, "%s", code);
      buffer[strlen (buffer) - 1] = '\0';
      if (buffer[0] == '#')
        {
          printf ("%s\n", buffer);
          continue;
        }
      
      else if (strcmp (code, "mtllib") == 0)
        {
          char material_lib_filename[512];
          sscanf (buffer, "%s %s", code, material_lib_filename);
          char *material_lib_filename_full = (char*)malloc (sizeof (char) * (strlen (DATADIR) + strlen (material_lib_filename) + 2));
          strcpy (material_lib_filename_full, DATADIR);
          strcat (material_lib_filename_full, "/");
          strcat (material_lib_filename_full, material_lib_filename);
          printf ("opening %s\n", material_lib_filename_full);
          FILE *mtlfile = fopen (material_lib_filename_full, "r");
          while (fgets (buffer, 512, mtlfile))
            {
              sscanf (buffer, "%s", code);
              if (strcmp (code, "newmtl") == 0)
                {
                  model->material_count++;
                  if (model->material_count == 1)
                    model->materials = (Material*)malloc (sizeof(Material));
                  else
                    model->materials = (Material*)realloc (model->materials, sizeof(Material) * model->material_count);
                  char name[512];
                  sscanf (buffer, "%s %s", code, name);
                  model->materials[model->material_count-1].name = (char*)malloc (strlen (name) * sizeof(char));
                  strcpy (model->materials[model->material_count-1].name, name);
                  printf ("Found material %s\n", name);
                }
              if (strcmp (code, "Kd") == 0)
                {
                  float r, g, b;
                  sscanf (buffer, "%s %f %f %f", code, &r, &g, &b);
                  model->materials[model->material_count-1].r = r;
                  model->materials[model->material_count-1].g = g;
                  model->materials[model->material_count-1].b = b;
                  printf ("  %f, %f, %f\n", r, g, b);
                }
            }
          fclose (mtlfile);
          printf ("Got materials from %s\n", material_lib_filename_full);
        }

      else if (strcmp (code, "vn") == 0)
        {
          float x;
          float y;
          float z;
          sscanf (buffer, "%s %f %f %f", code, &x, &y, &z);
          model->normal_count++;
          if (model->normal_count == 1)
            model->normals = (Vert*)malloc (sizeof (Vert));
          else
            model->normals = (Vert*)realloc (model->normals, sizeof (Vert) * model->normal_count);
          model->normals[model->normal_count-1].x = x;
          model->normals[model->normal_count-1].y = y;
          model->normals[model->normal_count-1].z = z;
          printf ("normal: %f, %f, %f\n", x, y, z);
        }

      else if (strcmp (code, "v") == 0)
        {
          float x;
          float y;
          float z;
          sscanf (buffer, "%s %f %f %f", code, &x, &y, &z);
          model->vert_count++;
          if (model->vert_count == 1)
            model->verts = (Vert*)malloc (sizeof (Vert));
          else
            model->verts = (Vert*)realloc (model->verts, sizeof (Vert) * model->vert_count);
          model->verts[model->vert_count-1].x = x;
          model->verts[model->vert_count-1].y = y;
          model->verts[model->vert_count-1].z = z;
          printf ("vertex: %f, %f, %f\n", x, y, z);
        }

      else if (strcmp (code, "f") == 0)
        {
          char code[2];
          int v1;
          int v2;
          int v3;
          int vn1;
          int vn2;
          int vn3;
          sscanf (buffer, "%s %d//%d %d//%d %d//%d", code, &v1, &vn1, &v2, &vn2, &v3, &vn3);

          model->objects[model->object_count-1].face_count++;
          if (model->objects[model->object_count-1].face_count == 1)
            model->objects[model->object_count-1].faces = (Face*)malloc (sizeof(Face));
          else
            model->objects[model->object_count-1].faces = (Face*)realloc (model->objects[model->object_count-1].faces, sizeof(Face) * model->objects[model->object_count-1].face_count);

          model->objects[model->object_count-1].faces[model->objects[model->object_count-1].face_count-1].v1 = v1;
          model->objects[model->object_count-1].faces[model->objects[model->object_count-1].face_count-1].vn1 = vn1;
          model->objects[model->object_count-1].faces[model->objects[model->object_count-1].face_count-1].v2 = v2;
          model->objects[model->object_count-1].faces[model->objects[model->object_count-1].face_count-1].vn2 = vn2;
          model->objects[model->object_count-1].faces[model->objects[model->object_count-1].face_count-1].v3 = v3;
          model->objects[model->object_count-1].faces[model->objects[model->object_count-1].face_count-1].vn3 = vn3;
        }
      else if (strcmp (code, "usemtl") == 0)
        {
          char name[512];
          sscanf (buffer, "%s %s", code, name);
          printf ("looking through the %d materials for %s\n", model->material_count, name);
          int i;
          for (i = 0; i < model->material_count; i++)
            {
              printf ("Looking at material %s\n", model->materials[i].name);
              if (strcmp (name, model->materials[i].name) == 0)
                {
                  printf ("match\n");
                  printf ("settings %s, %d\n", name, i);
                  model->objects[model->object_count-1].material_index = i;
                }
            }
          printf ("Using material %s\n", model->materials[model->objects[model->object_count-1].material_index].name);
        }
      else if (strcmp (code, "o") == 0)
        {
          printf ("object count: %d\n", model->object_count);
          model->object_count++;
          if (model->object_count == 1)
            model->objects = (Object*)malloc (sizeof(Object));
          else
            model->objects = (Object*)realloc (model->objects, sizeof(Object) * model->object_count);
          char name[512];
          sscanf (buffer, "%s %s", code, name);
          model->objects[model->object_count-1].name = (char*)malloc ((strlen (name) + 1) * sizeof(char));
          strcpy (model->objects[model->object_count-1].name, name);
          printf ("Found object %s\n", model->objects[model->object_count-1].name);
          model->objects[model->object_count-1].face_count = 0;
        }
    }
  fclose (file);
  return model;
}

void
model_draw (Model *model, float x, float y, float z, float direction, float r, float g, float b)
{
  int j;
  for (j = 0; j < model->object_count; j++)
    {
      Object *a = &model->objects[j];
      r = model->materials[a->material_index].r;
      g = model->materials[a->material_index].g;
      b = model->materials[a->material_index].b;
      glLoadIdentity ();

      glRotatef (-camdir, 0.0f, 1.0f, 0.0f);

      glTranslatef (x, y, z);
      glTranslatef (-camx, -camy, -camz);

      glRotatef (direction, 0.0f, 1.0f, 0.0f);

      int i = 0;
      glBegin(GL_TRIANGLES);
        for (i = 0; i < a->face_count; i++)
          {
            int v1 = a->faces[i].v1 - 1;
            int v2 = a->faces[i].v2 - 1;
            int v3 = a->faces[i].v3 - 1;
            int vn1 = a->faces[i].vn1 - 1;
            int vn2 = a->faces[i].vn2 - 1;
            int vn3 = a->faces[i].vn3 - 1;
            float v1x = model->verts[v1].x;
            float v1y = model->verts[v1].y;
            float v1z = model->verts[v1].z;
            float v2x = model->verts[v2].x;
            float v2y = model->verts[v2].y;
            float v2z = model->verts[v2].z;
            float v3x = model->verts[v3].x;
            float v3y = model->verts[v3].y;
            float v3z = model->verts[v3].z;
            float vn1x = model->normals[vn1].x;
            float vn1y = model->normals[vn1].y;
            float vn1z = model->normals[vn1].z;
            float vn2x = model->normals[vn2].x;
            float vn2y = model->normals[vn2].y;
            float vn2z = model->normals[vn2].z;
            float vn3x = model->normals[vn3].x;
            float vn3y = model->normals[vn3].y;
            float vn3z = model->normals[vn3].z;
            float color[] = {r, g, b, 1.0f};
            glMaterialfv (GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color);
            glNormal3f (vn1x, vn1y, vn1z);
            glVertex3f (v1x, v1y, v1z);
            glNormal3f (vn2x, vn2y, vn2z);
            glVertex3f (v2x, v2y, v2z);
            glNormal3f (vn3x, vn3y, vn3z);
            glVertex3f (v3x, v3y, v3z);
          }
      glEnd();
    }
}

void
model_destroy (Model *model)
{
  free (model->normals);
  free (model->verts);
  free (model->objects);
  free (model->materials);
  free (model);
}

