#!/usr/bin/env python

from gnuradio import gr
from gnuradio import usrp

class siggen (object):
    __slots__ = ['dac_freq', 'interp', 'waveform_type', 'waveform_ampl',
                 'waveform_freq', 'waveform_offset', 'fg', 'usrp',
                 'siggen', 'noisegen', 'src' ]

    def __init__ (self):
        # constants
        self.dac_freq = 128e6
        # controllable values
        self.interp = 64
        self.waveform_type = gr.GR_SIN_WAVE
        self.waveform_ampl = 16000
        self.waveform_freq = 250e3
        self.waveform_offset = 0
        self._instantiate_blocks ()
        self.set_waveform_type (self.waveform_type)

    def start (self):
        self.fg.start ()

    def stop (self):
        self.fg.stop ()
        
    def usb_freq (self):
        return self.dac_freq / self.interp

    def usb_throughput (self):
        return self.usb_freq () * 4
        
    def set_waveform_type (self, type):
        '''
        valid waveform types are: gr.GR_SIN_WAVE, gr.GR_CONST_WAVE,
        gr.GR_UNIFORM and gr.GR_GAUSSIAN
        '''
        self._configure_graph (type)
        self.waveform_type = type

    def set_waveform_ampl (self, ampl):
        self.waveform_ampl = ampl
        self.siggen.set_amplitude (ampl)
        self.noisegen.set_amplitude (ampl)

    def set_waveform_freq (self, freq):
        self.waveform_freq = freq
        self.siggen.set_frequency (freq)
        
    def set_waveform_offset (self, offset):
        self.waveform_offset = offset
        self.siggen.set_offset (offset)

    def set_interpolator (self, interp):
        self.interp = interp
        self.siggen.set_sampling_freq (self.usb_freq ())
        self.usrp.set_interp_rate (interp)

    def set_cordic_freq (self, freq):
        self.usrp.set_tx_freq (0, freq)
        
    def _instantiate_blocks (self):
        self.fg = gr.flow_graph ()
        self.src = None
        self.siggen = gr.sig_source_c (self.usb_freq (),
                                       gr.GR_SIN_WAVE,
                                       self.waveform_freq,
                                       self.waveform_ampl,
                                       self.waveform_offset)
        self.noisegen = gr.noise_source_c (gr.GR_UNIFORM,
                                           self.waveform_ampl)
        self.usrp = usrp.sink_c (0, self.interp)

    def _configure_graph (self, type):
        was_running = self.fg.is_running ()
        if was_running:
            self.fg.stop ()
        self.fg.disconnect_all ()
        if type == gr.GR_SIN_WAVE or type == gr.GR_CONST_WAVE:
            self.fg.connect (self.siggen, self.usrp)
            self.siggen.set_waveform (type)
            self.src = self.siggen
        elif type == gr.GR_UNIFORM or type == gr.GR_GAUSSIAN:
            self.fg.connect (self.noisegen, self.usrp)
            self.noisegen.set_type (type)
            self.src = self.noisegen
        else:
            raise ValueError, type
        if was_running:
            self.fg.start ()


def main ():
    sg = siggen ()
    sg.set_cordic_freq (0e6)
    
    sg.start ()
    raw_input ('Press Enter to quit: ')
    sg.stop ()

if __name__ == '__main__':
    main ()
