/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <usrp_source_c.h>
#include <gr_io_signature.h>
#include <usrp_standard.h>


usrp_source_c_sptr
usrp_make_source_c (int which_board, unsigned int decim_rate) throw (std::runtime_error)
{
  return usrp_source_c_sptr (new usrp_source_c (which_board, decim_rate));
}


usrp_source_c::usrp_source_c (int which_board, unsigned int decim_rate) throw (std::runtime_error)
  : gr_sync_block ("usrp_source_c",
		   gr_make_io_signature (0, 0, 0),
		   gr_make_io_signature (1, 1, sizeof (gr_complex))),
    d_noverruns (0)
{
  d_usrp = usrp_standard_rx::make (which_board,  decim_rate);
  if (d_usrp == 0)
    throw std::runtime_error ("can't open usrp");

  // All calls to d_usrp->read must be multiples of 512 bytes.
  // We jack this up to 4k to reduce overhead.

  set_output_multiple (4 * 1024 / sizeof (gr_complex));
}

usrp_source_c::~usrp_source_c ()
{
  delete d_usrp;
}

int
usrp_source_c::work (int noutput_items,
		     gr_vector_const_void_star &input_items,
		     gr_vector_void_star &output_items)
{
  assert (output_items.size () == 1);

  gr_complex	*out = (gr_complex *) output_items[0];

  static const int shorts_per_item = 2;		// I & Q
  static const int bytes_per_item  = shorts_per_item * sizeof (short);
  
  int		base  = 0;
  int		ntogo = noutput_items;

  while (ntogo > 0){
    static const int BUFSIZE = 8 * 1024;
    short buf[BUFSIZE];				// 16KB reads
    bool  overrun;
    int max_items = BUFSIZE / shorts_per_item;
    int nitems = std::min (ntogo, max_items);	// number of items to read this time through
    int nbytes = nitems * bytes_per_item;	// corresponding number of bytes

    int result_nbytes = d_usrp->read (buf, nbytes, &overrun);
    if (overrun){
      fprintf (stderr, "usrp_source_c: overrun\n");
      d_noverruns++;
    }

    if (result_nbytes < 0){	// got a problem (usually board unplugged or powered down)
      return -1;		// indicate we're done
    }

    if (result_nbytes != nbytes){	// not really an error, but unexpected
      fprintf (stderr, "usrp_source_c: short read.  Expected %d, got %d\n",
	       nbytes, result_nbytes);
    }

    int result_nitems = result_nbytes / bytes_per_item;

    // now unpack the raw data into the gr_complex output stream

    for (int i = 0, j = 0; i < result_nitems; i++){
      out[base+i] = gr_complex ((float) buf[j+0], (float) buf[j+1]);
      j += 2;
    }

    ntogo -= result_nitems;
    base += result_nitems;
  }

  return noutput_items - ntogo;
}

bool
usrp_source_c::set_decim_rate (unsigned int rate)
{
  return d_usrp->set_decim_rate (rate);
}

bool
usrp_source_c::set_rx_freq (int channel, double freq)
{
  return d_usrp->set_rx_freq (channel, freq);
}

long
usrp_source_c::adc_freq () const
{
  return d_usrp->adc_freq ();
}

unsigned int
usrp_source_c::decim_rate () const
{
  return d_usrp->decim_rate ();
}

double
usrp_source_c::rx_freq (int channel) const
{
  return d_usrp->rx_freq (channel);
}
