/* -*- c++ -*- */
/*
 * Copyright 2003,2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <usb.h>			/* needed for usb functions */
#include <getopt.h>
#include <assert.h>
#include <math.h>
#include "time_stuff.h"
#include "usrp_standard.h"

char *prog_name;

static bool test_output (usrp_standard_tx *utx, bool forever_p, double ampl, bool dc_p);

static void
set_progname (char *path)
{
  char *p = strrchr (path, '/');
  if (p != 0)
    prog_name = p+1;
  else
    prog_name = path;
}

static void
usage ()
{
  fprintf (stderr, 
	   "usage: %s [-f] [-v] [-d] [-a <ampl>][-I <interp>] [-F freq] [-D]\n", prog_name);
  fprintf (stderr, "  [-f] loop forever\n");
  fprintf (stderr, "  [-v] verbose\n");
  fprintf (stderr, "  [-d] dump registers\n");
  fprintf (stderr, "  [-D] DC output\n");
  exit (1);
}

static void
die (const char *msg)
{
  fprintf (stderr, "die: %s: %s\n", prog_name, msg);
  exit (1);
}

static void
dump_codec_regs (usrp_basic *u, int which_codec, FILE *fp)
{
  for (int i = 0; i < 64; i++){
    unsigned char v;
    u->_read_9862 (which_codec, i, &v);
    fprintf (fp, "%2d:  0x%02x\n", i, v); 
  }
  fflush (fp);
}

static void
do_dump_codec_regs (usrp_basic *u)
{
  char name[100];
  strcpy (name, "regsXXXXXX");
  int fd = mkstemp (name);
  if (fd == -1){
    perror (name);
  }
  else {
    FILE *fp = fdopen (fd, "w");
    dump_codec_regs (u, 0, fp);
    fclose (fp);
  }
}

int
main (int argc, char **argv)
{
  bool 	verbose_p = false;
  bool	forever_p = false;
  bool  dump_regs_p = false;
  bool	dc_p = false;
  int	ch;
  int	which_board = 0;
  int	interp = 8;			// 32.0 MB/sec 
  double	center_freq = 0;
  double	ampl = 10000;

  set_progname (argv[0]);

  while ((ch = getopt (argc, argv, "vfdI:F:a:D")) != EOF){
    switch (ch){
    case 'f':
      forever_p = true;
      break;

    case 'v':
      verbose_p = true;
      break;

    case 'd':
      dump_regs_p = true;
      break;
      
    case 'D':
      dc_p = true;
      break;
      
    case 'I':
      interp = strtol (optarg, 0, 0);
      break;
		      
    case 'F':
      center_freq = strtod (optarg, 0);
      break;
      
    case 'a':
      ampl = strtod (optarg, 0);
      break;

    default:
      usage ();
    }
  }

  usrp_standard_tx *utx;

  utx = usrp_standard_tx::make (which_board, interp);

  if (utx == 0)
    die ("usrp_standard_tx::make");
    
  if (!utx->set_tx_freq (0, center_freq))
    die ("utx->set_tx_freq");
    
  if (dump_regs_p)
    do_dump_codec_regs (utx);
  
  
  fflush (stdout);
  fflush (stderr);

  test_output (utx, forever_p, ampl, dc_p);

  delete utx;

  return 0;
}


static bool
test_output (usrp_standard_tx *utx, bool forever_p, double ampl, bool dc_p)
{
  static const int BUFSIZE = 16 * 1024;
  static const int N = BUFSIZE/sizeof (short);
  short 	   buf[N];
  int		   nbytes = 0;
  int		   max_bytes = 512 * (1L << 20);

  static const int    PERIOD = 65;		// any value is valid
  //static const int    PERIOD = 8;		// any value is valid
  static const int    PATLEN = 2 * PERIOD;	
  // static const double AMPL = 10000;
  short		      pattern[PATLEN];

  for (int i = 0; i < PERIOD; i++){
    if (dc_p){
      pattern[2*i+0] = (short) ampl;
      pattern[2*i+1] = (short) 0;
    }
    else {
      pattern[2*i+0] = (short) (ampl * cos (2*M_PI * i / PERIOD));
      pattern[2*i+1] = (short) (ampl * sin (2*M_PI * i / PERIOD));
    }
  }
  

  double	   start_wall_time = get_elapsed_time ();
  double	   start_cpu_time  = get_cpu_usage ();

  bool 	underrun;
  int 	nunderruns = 0;
  int 	pi = 0;

  for (nbytes = 0; forever_p || nbytes < max_bytes; nbytes += BUFSIZE){

    for (int i = 0; i < N; i++){
      buf[i] = pattern[pi];
      pi++;
      if (pi >= PATLEN)
	pi = 0;
    }

    int	ret = utx->write (buf, sizeof (buf), &underrun);
    if (ret != sizeof (buf)){
      fprintf (stderr, "test_output: error, ret = %d\n", ret);
    }

    if (underrun){
      nunderruns++;
      printf ("tx_underrun\n");
    }
  }

  utx->wait_for_completion ();

  double stop_wall_time = get_elapsed_time ();
  double stop_cpu_time  = get_cpu_usage ();

  double delta_wall = stop_wall_time - start_wall_time;
  double delta_cpu  = stop_cpu_time  - start_cpu_time;

  printf ("xfered %.3g bytes in %.3g seconds.  %.4g bytes/sec.  cpu time = %.3g\n",
	  (double) max_bytes, delta_wall, max_bytes / delta_wall, delta_cpu);

  printf ("%d underruns\n", nunderruns);

  return true;
}
