/* hfsplus.c - HFS+ Filesystem.  */
/*
 *  GRUB  --  GRand Unified Bootloader
 *  Copyright (C) 2005  Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <grub/err.h>
#include <grub/file.h>
#include <grub/mm.h>
#include <grub/misc.h>
#include <grub/disk.h>
#include <grub/dl.h>
#include <grub/types.h>
#include <grub/fshelp.h>

/* A HFS+ extent.  */
struct grub_hfsplus_extent
{
  /* The first block of a file on disk.  */
  grub_uint32_t start;
  /* The amount of blocks described by this extent.  */
  grub_uint32_t count;
} __attribute__ ((packed));

/* The descriptor of a fork.  */
struct grub_hfsplus_forkdata
{
  grub_uint64_t size;
  grub_uint32_t clumpsize;
  grub_uint32_t blocks;
  struct grub_hfsplus_extent extents[8];
} __attribute__ ((packed));

/* The HFS+ Volume Header.  */
struct grub_hfsplus_volheader
{
  grub_uint8_t magic[2];
  grub_uint16_t version;
  grub_uint32_t attributes;
  grub_uint8_t unused[32];
  grub_uint32_t blksize;
  grub_uint8_t unused2[68];
  struct grub_hfsplus_forkdata allocations_file;
  struct grub_hfsplus_forkdata extents_file;
  struct grub_hfsplus_forkdata catalog_file;
  struct grub_hfsplus_forkdata attrib_file;
  struct grub_hfsplus_forkdata startup_file;
} __attribute__ ((packed));

/* The type of node.  */
enum grub_hfsplus_btnode_type
  {
    GRUB_HFSPLUS_BTNODE_TYPE_LEAF = -1,
    GRUB_HFSPLUS_BTNODE_TYPE_INDEX = 0,
    GRUB_HFSPLUS_BTNODE_TYPE_HEADER = 1,
    GRUB_HFSPLUS_BTNODE_TYPE_MAP = 2,
  };

struct grub_hfsplus_btnode
{
  grub_uint32_t next;
  grub_uint32_t prev;
  grub_int8_t type;
  grub_uint8_t height;
  grub_uint16_t count;
  grub_uint16_t unused;
} __attribute__ ((packed));

/* The header of a HFS+ B+ Tree.  */
struct grub_hfsplus_btheader
{
  grub_uint16_t depth;
  grub_uint32_t root;
  grub_uint32_t leaf_records;
  grub_uint32_t first_leaf_node;
  grub_uint32_t last_leaf_node;
  grub_uint16_t nodesize;
  grub_uint16_t keysize;
} __attribute__ ((packed));

/* The on disk layout of a catalog key.  */
struct grub_hfsplus_catkey
{
  grub_uint16_t keylen;
  grub_uint32_t parent;
  grub_uint16_t namelen;
  grub_uint16_t name[30];
} __attribute__ ((packed));

/* The on disk layout of an extent overflow file key.  */
struct grub_hfsplus_extkey
{
  grub_uint16_t keylen;
  grub_uint8_t type;
  grub_uint8_t unused;
  grub_uint32_t fileid;
  grub_uint32_t start;
} __attribute__ ((packed));

struct grub_hfsplus_key
{
  union
  {
    struct grub_hfsplus_extkey extkey;
    struct grub_hfsplus_catkey catkey;
    grub_uint16_t keylen;
  };
} __attribute__ ((packed));

struct grub_hfsplus_catfile
{
  grub_uint16_t type;
  grub_uint16_t flags;
  grub_uint32_t reserved;
  grub_uint32_t fileid;
  grub_uint8_t unused1[30];
  grub_uint16_t mode;
  grub_uint8_t unused2[44];
  struct grub_hfsplus_forkdata data;
  struct grub_hfsplus_forkdata resource;
} __attribute__ ((packed));

/* Filetype information as used in inodes.  */
#define GRUB_HFSPLUS_FILEMODE_MASK	0170000
#define GRUB_HFSPLUS_FILEMODE_REG	0100000
#define GRUB_HFSPLUS_FILEMODE_DIRECTORY	0040000
#define GRUB_HFSPLUS_FILEMODE_SYMLINK	0120000

/* Some pre-defined file IDs.  */
#define GRUB_HFSPLUS_FILEID_ROOTDIR	2
#define GRUB_HFSPLUS_FILEID_OVERFLOW	3
#define GRUB_HFSPLUS_FILEID_CATALOG	4

enum grub_hfsplus_filetype
  {
    GRUB_HFSPLUS_FILETYPE_DIR = 1,
    GRUB_HFSPLUS_FILETYPE_REG = 2,
    GRUB_HFSPLUS_FILETYPE_DIR_THREAD = 3,
    GRUB_HFSPLUS_FILETYPE_REG_THREAD = 4
  };

/* Internal representation of a catalog key.  */
struct grub_hfsplus_catkey_internal
{
  int parent;
  char *name;
};

/* Internal representation of an extent overflow key.  */
struct grub_hfsplus_extkey_internal
{
  grub_uint32_t fileid;
  grub_uint32_t start;
};

struct grub_hfsplus_key_internal
{
  union
  {
    struct grub_hfsplus_extkey_internal extkey;
    struct grub_hfsplus_catkey_internal catkey;
  };
};



struct grub_fshelp_node
{
  struct grub_hfsplus_data *data;
  struct grub_hfsplus_extent extents[8];
  grub_uint64_t size;
  grub_uint32_t fileid;
};

struct grub_hfsplus_btree
{
  grub_uint32_t root;
  int nodesize;

  /* Catalog file node.  */
  struct grub_fshelp_node file;
};

/* Information about a "mounted" HFS+ filesystem.  */
struct grub_hfsplus_data
{
  struct grub_hfsplus_volheader volheader;
  grub_disk_t disk;

  unsigned int log2blksize;

  struct grub_hfsplus_btree catalog_tree;
  struct grub_hfsplus_btree extoverflow_tree;

  struct grub_fshelp_node dirroot;
  struct grub_fshelp_node opened_file;
};

#ifndef GRUB_UTIL
static grub_dl_t my_mod;
#endif


/* Find the extent that points to FILEBLOCK.  If it is not in one of
   the 8 extents described by EXTENT, return -1.  In that case set
   RETRY to the last block that was found in the last extent.  */
static int
grub_hfsplus_find_block (struct grub_hfsplus_extent *extent,
			 int fileblock, int *retry)
{
  int i;
  grub_size_t blksleft = fileblock;

  /* First lookup the file in the given extents.  */
  for (i = 0; i < 8; i++)
    {
      if (blksleft < grub_be_to_cpu32 (extent[i].count))
	return grub_be_to_cpu32 (extent[i].start) + blksleft;
      blksleft -= grub_be_to_cpu32 (extent[i].count);
    }

  *retry = fileblock - blksleft;
  return -1;
}

static grub_err_t
grub_hfsplus_btree_search (struct grub_hfsplus_btree *btree,
			   struct grub_hfsplus_key_internal *key,
			   int (*compare_keys) (struct grub_hfsplus_key *keya,
						struct grub_hfsplus_key_internal *keyb),
			   struct grub_hfsplus_btnode **matchnode, int *keyoffset);

static int grub_hfsplus_cmp_extkey (struct grub_hfsplus_key *keya,
				    struct grub_hfsplus_key_internal *keyb);

/* Search for the block FILEBLOCK inside the file NODE.  Return the
   blocknumber of this block on disk.  */
static int
grub_hfsplus_read_block (grub_fshelp_node_t node, int fileblock)
{
  struct grub_hfsplus_extkey_internal extoverflow;
  struct grub_hfsplus_extkey *keyfound;
  int blk;
  struct grub_hfsplus_btnode *nnode = 0;
  int ptr;
  int retry = 0;

  struct grub_hfsplus_extent *extents = &node->extents[0];

  while (1)
    {
      char *cnode;

      /* Try to find this block in the current set of extents.  */
      blk = grub_hfsplus_find_block (extents, fileblock, &retry);
      if (blk != -1)
	return blk;

      /* The previous iteration of this loop allocated memory.  The
	 code above used this memory, it can be free'ed now.  */
      if (nnode)
	grub_free (nnode);

      /* For the extent overflow file, extra extents can't be found in
	 the extent overflow file.  If this happens, you found a
	 bug...  */
      if (node->fileid == GRUB_HFSPLUS_FILEID_OVERFLOW)
	return -1;

      /* Set up the key to look for in the extent overflow file.  */
      extoverflow.fileid = node->fileid;
      extoverflow.start = retry;

      if (grub_hfsplus_btree_search (&node->data->extoverflow_tree,
				     (struct grub_hfsplus_key_internal *) &extoverflow,
				     grub_hfsplus_cmp_extkey, &nnode, &ptr))
	return -1;

      cnode = (char *) nnode;

      /* The extent overflow file has a 8 extents right after the key.  */
      keyfound = (struct grub_hfsplus_extkey *) 
	&cnode[(int) cnode[node->data->extoverflow_tree.nodesize - ptr - 1]];
      extents = (struct grub_hfsplus_extent *) 
	((char *) keyfound + sizeof (struct grub_hfsplus_extkey));

      /* The block wasn't found.  Perhaps the next iteration will find
	 it.  The last block we found is stored in FILEBLOCK now.  */
      /* XXX: Multiple iterations for locating the right extent was
	 not tested enough... */
      fileblock = retry;
    }

  if (nnode)
    grub_free (nnode);

  /* Too bad, you lose.  */
  return -1;
}


/* Read LEN bytes from the file described by DATA starting with byte
   POS.  Return the amount of read bytes in READ.  */
static grub_ssize_t
grub_hfsplus_read_file (grub_fshelp_node_t node,
			void (*read_hook) (unsigned long sector,
					   unsigned offset, unsigned length),
			int pos, unsigned int len, char *buf)
{
  return grub_fshelp_read_file (node->data->disk, node, read_hook,
				pos, len, buf, grub_hfsplus_read_block,
				node->size,
				node->data->log2blksize - GRUB_DISK_SECTOR_BITS);
}

static struct grub_hfsplus_data *
grub_hfsplus_mount (grub_disk_t disk)
{
  struct grub_hfsplus_data *data;
  struct grub_hfsplus_btheader header;
  struct grub_hfsplus_btnode node;

  data = grub_malloc (sizeof (*data));
  if (!data)
    return 0;

  data->disk = disk;

  /* Read the bootblock.  */
  grub_disk_read (disk, 2, 0, sizeof (struct grub_hfsplus_volheader),
		  (char *) &data->volheader);
  if (grub_errno)
    goto fail;

  /* Make sure this is an hfs+ filesystem.  XXX: Do we really support
     HFX?  */
  if (grub_strncmp (data->volheader.magic, "H+", 2)
      && grub_strncmp (data->volheader.magic, "HX", 2))
    {
      grub_error (GRUB_ERR_BAD_FS, "not a HFS+ filesystem");
      goto fail;
    }

  if (grub_fshelp_log2blksize (grub_be_to_cpu32 (data->volheader.blksize),
			       &data->log2blksize))
    goto fail;

  /* Make a new node for the catalog tree.  */
  data->catalog_tree.file.data = data;
  data->catalog_tree.file.fileid = GRUB_HFSPLUS_FILEID_CATALOG;
  grub_memcpy (&data->catalog_tree.file.extents,
	       data->volheader.catalog_file.extents,
	       sizeof data->volheader.catalog_file.extents);
  data->catalog_tree.file.size = 
    grub_be_to_cpu64 (data->volheader.catalog_file.size);

  /* Make a new node for the extent overflow file.  */
  data->extoverflow_tree.file.data = data;
  data->extoverflow_tree.file.fileid = GRUB_HFSPLUS_FILEID_OVERFLOW;
  grub_memcpy (&data->extoverflow_tree.file.extents,
	       data->volheader.extents_file.extents,
	       sizeof data->volheader.catalog_file.extents);

  data->extoverflow_tree.file.size = 
    grub_be_to_cpu64 (data->volheader.extents_file.size);

  /* Read the essential information about the trees.  */
  if (! grub_hfsplus_read_file (&data->catalog_tree.file, 0,
				sizeof (struct grub_hfsplus_btnode),
				sizeof (header), (char *) &header))
    goto fail;

  data->catalog_tree.root = grub_be_to_cpu32 (header.root);
  data->catalog_tree.nodesize = grub_be_to_cpu16 (header.nodesize);

  if (! grub_hfsplus_read_file (&data->extoverflow_tree.file, 0,
				sizeof (struct grub_hfsplus_btnode),
				sizeof (header), (char *) &header))
    goto fail;

  data->extoverflow_tree.root = grub_be_to_cpu32 (header.root);

  if (! grub_hfsplus_read_file (&data->extoverflow_tree.file, 0, 0,
				sizeof (node), (char *) &node))
    goto fail;

  data->extoverflow_tree.root = grub_be_to_cpu32 (header.root);
  data->extoverflow_tree.nodesize = grub_be_to_cpu16 (header.nodesize);

  data->dirroot.data = data;
  data->dirroot.fileid = GRUB_HFSPLUS_FILEID_ROOTDIR;

  return data;

 fail:
  grub_free (data);
  return 0;
}

/* Compare the on disk catalog key KEYA with the catalog key we are
   looking for (KEYB).  */
static int
grub_hfsplus_cmp_catkey (struct grub_hfsplus_key *keya,
			 struct grub_hfsplus_key_internal *keyb)
{
  struct grub_hfsplus_catkey *catkey_a = &keya->catkey;
  struct grub_hfsplus_catkey_internal *catkey_b = &keyb->catkey;
  char *filename;
  int i;
  int diff;
  
  diff = grub_be_to_cpu32 (catkey_a->parent) - catkey_b->parent;
  if (diff)
    return diff; 

  /* Change the filename in keya so the endianess is correct.  */
  for (i = 0; i < grub_be_to_cpu16 (catkey_a->namelen); i++)
    catkey_a->name[i] = grub_be_to_cpu16 (catkey_a->name[i]);

  filename = grub_malloc (grub_be_to_cpu16 (catkey_a->namelen) + 1);

  if (! grub_utf16_to_utf8 (filename, catkey_a->name,
			    grub_be_to_cpu16 (catkey_a->namelen)))
    return -1; /* XXX: This error never occurs, but in case it happens
		  just skip this entry.  */

  diff = grub_strncmp (filename, catkey_b->name,
		       grub_be_to_cpu16 (catkey_a->namelen));

  grub_free (filename);

  /* The endianess was changed to host format, change it back to
     whatever it was.  */
  for (i = 0; i < grub_be_to_cpu16 (catkey_a->namelen); i++)
    catkey_a->name[i] = grub_cpu_to_be16 (catkey_a->name[i]);
  return diff;
}

/* Compare the on disk extent overflow key KEYA with the extent
   overflow key we are looking for (KEYB).  */
static int
grub_hfsplus_cmp_extkey (struct grub_hfsplus_key *keya,
			 struct grub_hfsplus_key_internal *keyb)
{
  struct grub_hfsplus_extkey *extkey_a = &keya->extkey;
  struct grub_hfsplus_extkey_internal *extkey_b = &keyb->extkey;
  int diff;

  diff = grub_be_to_cpu32 (extkey_a->fileid) - extkey_b->fileid;

  if (diff)
    return diff;

  diff = grub_be_to_cpu32 (extkey_a->start) - extkey_b->start;
  return diff;
}

/* Return the offset of the record with the index INDEX, in the node
   NODE which is part of the B+ tree BTREE.  */
static inline unsigned int
grub_hfsplus_btree_recoffset (struct grub_hfsplus_btree *btree,
			   struct grub_hfsplus_btnode *node, int index)
{
  char *cnode = (char *) node;
  grub_uint16_t *recptr;
  recptr = (grub_uint16_t *) (&cnode[btree->nodesize
				     - index * sizeof (grub_uint16_t) - 2]);
  return grub_be_to_cpu16 (*recptr);
}

/* Return a pointer to the record with the index INDEX, in the node
   NODE which is part of the B+ tree BTREE.  */
static inline struct grub_hfsplus_key *
grub_hfsplus_btree_recptr (struct grub_hfsplus_btree *btree,
			   struct grub_hfsplus_btnode *node, int index)
{
  char *cnode = (char *) node;
  unsigned int offset;
  offset = grub_hfsplus_btree_recoffset (btree, node, index);
  return (struct grub_hfsplus_key *) &cnode[offset];
}


static char *
grub_hfsplus_read_symlink (grub_fshelp_node_t node)
{
  char *symlink;
  grub_ssize_t numread;

  symlink = grub_malloc (node->size + 1);
  if (!symlink)
    return 0;

  numread = grub_hfsplus_read_file (node, 0, 0, node->size, symlink);
  if (numread != (grub_ssize_t) node->size)
    {
      grub_free (symlink);
      return 0;
    }
  symlink[node->size] = '\0';

  return symlink;
}

int
grub_hfsplus_btree_iterate_node (struct grub_hfsplus_btree *btree,
				 struct grub_hfsplus_btnode *first_node,
				 int first_rec,
				 int (*hook) (void *record))
{
  int rec;  

  for (;;)
    {
      char *cnode = (char *) first_node;

      /* Iterate over all records in this node.  */
      for (rec = first_rec; rec < grub_be_to_cpu16 (first_node->count); rec++)
	{
	  if (hook (grub_hfsplus_btree_recptr (btree, first_node, rec)))
	    return 1;
	}

      if (! first_node->next)
	break;

      if (! grub_hfsplus_read_file (&btree->file, 0,
				    (grub_be_to_cpu32 (first_node->next)
				     * btree->nodesize),
				    btree->nodesize, cnode))
	return 1;

      /* Don't skip any record in the next iteration.  */
      first_rec = 0;
    }

  return 0;
}

/* Lookup the node described by KEY in the B+ Tree BTREE.  Compare
   keys using the function COMPARE_KEYS.  When a match is found,
   return the node in MATCHNODE and a pointer to the data in this node
   in KEYOFFSET.  MATCHNODE should be free'ed by the caller.  */
static grub_err_t
grub_hfsplus_btree_search (struct grub_hfsplus_btree *btree,
			   struct grub_hfsplus_key_internal *key,
			   int (*compare_keys) (struct grub_hfsplus_key *keya,
						struct grub_hfsplus_key_internal *keyb),
			   struct grub_hfsplus_btnode **matchnode, int *keyoffset)
{
  grub_uint64_t currnode;
  char *node;
  struct grub_hfsplus_btnode *nodedesc;
  int rec;

  node = grub_malloc (btree->nodesize);
  if (! node)
    return grub_errno;

  currnode = btree->root;
  while (1)
    {
      int match = 0;

      /* Read a node.  */
      if (! grub_hfsplus_read_file (&btree->file, 0,
				    (long)currnode * (long)btree->nodesize,
				    btree->nodesize, (char *) node))
	{
	  grub_free (node);
	  return grub_errno;
	}

      nodedesc = (struct grub_hfsplus_btnode *) node;

      /* Find the record in this tree.  */
      for (rec = 0; rec < grub_be_to_cpu16 (nodedesc->count); rec++)
	{
	  struct grub_hfsplus_key *currkey;
	  currkey = grub_hfsplus_btree_recptr (btree, nodedesc, rec);
	  
	  /* The action that has to be taken depend on the type of
	     record.  */
	  if (nodedesc->type == GRUB_HFSPLUS_BTNODE_TYPE_LEAF
	      && compare_keys (currkey, key) == 0)
	    {
	      /* An exact match was found!  */

	      *matchnode = nodedesc;
	      *keyoffset = rec;

	      return 0;
	    }
	  else if (nodedesc->type == GRUB_HFSPLUS_BTNODE_TYPE_INDEX)
	    {
	      grub_uint32_t *pointer;

	      /* The place where the key could've been found didn't
		 contain the key.  This means that the previous match
		 is the one that should be followed.  */
	      if (compare_keys (currkey, key) > 0)
		break;

	      /* Mark the last key which is lower or equal to the key
		 that we are looking for.  The last match that is
		 found will be used to locate the child which can
		 contain the record.  */
	      pointer = (grub_uint32_t *) ((char *) currkey
					   + grub_be_to_cpu16 (currkey->keylen)
					   + 2);
	      currnode = grub_be_to_cpu32 (*pointer);
	      match = 1;
	    }
	}

      /* No match was found, no record with this key exists in the
	 tree.  */
      if (! match)
	{
	  *matchnode = 0;
	  grub_free (node);
	  return 0;
	}
    }
}

static int
grub_hfsplus_iterate_dir (grub_fshelp_node_t dir,
			  int NESTED_FUNC_ATTR
			  (*hook) (const char *filename,
				   enum grub_fshelp_filetype filetype,
				   grub_fshelp_node_t node))
{
  auto int list_nodes (void *record);
  int list_nodes (void *record)
    {
      struct grub_hfsplus_catkey *catkey;
      char *filename;
      int i;
      int ret = 0;
      struct grub_fshelp_node *node;
      struct grub_hfsplus_catfile *fileinfo;
      enum grub_fshelp_filetype type = GRUB_FSHELP_UNKNOWN;

      catkey = (struct grub_hfsplus_catkey *) record;

      fileinfo = (record + grub_be_to_cpu16 (catkey->keylen)
		  + 2 + grub_be_to_cpu16(catkey->keylen) % 2);

      /* Stop iterating when the last directory entry was found.  */
      if (grub_be_to_cpu32 (catkey->parent) != dir->fileid)
	return 0;

      filename = grub_malloc (grub_be_to_cpu16 (catkey->namelen) + 1);
      if (! filename)
	return 0;

      /* Make sure the byte order of the UTF16 string is correct.  */
      for (i = 0; i < grub_be_to_cpu16 (catkey->namelen); i++)
	catkey->name[i] = grub_be_to_cpu16 (catkey->name[i]);

      if (! grub_utf16_to_utf8 (filename, catkey->name,
				grub_be_to_cpu16 (catkey->namelen)))
	{
	  grub_free (filename);
	  return 0;
	}

      filename[grub_be_to_cpu16 (catkey->namelen)] = '\0';

      /* Restore the byte order to what it was previously.  */
      for (i = 0; i < grub_be_to_cpu16 (catkey->namelen); i++)
	catkey->name[i] = grub_be_to_cpu16 (catkey->name[i]);

      /* Determine the type of the node that was found.  */
      if (grub_be_to_cpu16 (fileinfo->type) == GRUB_HFSPLUS_FILETYPE_REG)
	{
	  int mode = (grub_be_to_cpu16 (fileinfo->mode)
		      & GRUB_HFSPLUS_FILEMODE_MASK);

	  if (mode == GRUB_HFSPLUS_FILEMODE_REG)
	    type = GRUB_FSHELP_REG;
	  else if (mode == GRUB_HFSPLUS_FILEMODE_SYMLINK)
	    type = GRUB_FSHELP_SYMLINK;
	  else
	    type = GRUB_FSHELP_UNKNOWN;
	}
      else if (grub_be_to_cpu16 (fileinfo->type) == GRUB_HFSPLUS_FILETYPE_DIR)
	type = GRUB_FSHELP_DIR;

      /* Only accept valid nodes.  */
      if (type && grub_strlen (filename) == grub_be_to_cpu16 (catkey->namelen))
	{
	  /* A valid node was found; setup the node and call the
	     callback function.  */
	  node = grub_malloc (sizeof (*node));
	  node->data = dir->data;
	  
	  grub_memcpy (node->extents, fileinfo->data.extents,
		       sizeof (*node->extents));
	  node->size = grub_be_to_cpu64 (fileinfo->data.size);
	  node->fileid = grub_be_to_cpu32 (fileinfo->fileid);

	  ret = hook (filename, type, node);
	}

      grub_free (filename);

      return ret;
    }

  struct grub_hfsplus_key_internal intern;
  struct grub_hfsplus_btnode *node;
  int ptr;
  int ret;

  /* Create a key that points to the first entry in the directory.  */
  intern.catkey.parent = dir->fileid;
  intern.catkey.name = "";

  /* First lookup the first entry.  */
  if (grub_hfsplus_btree_search (&dir->data->catalog_tree, &intern,
				 grub_hfsplus_cmp_catkey, &node, &ptr))
    return 0;

  /* Iterate over all entries in this directory.  */
  ret = grub_hfsplus_btree_iterate_node (&dir->data->catalog_tree, node, ptr,
					 list_nodes);

  grub_free (node);

  return ret;
}

/* Open a file named NAME and initialize FILE.  */
static grub_err_t
grub_hfsplus_open (struct grub_file *file, const char *name)
{
  struct grub_hfsplus_data *data;
  struct grub_fshelp_node *fdiro = 0;
  
#ifndef GRUB_UTIL
  grub_dl_ref (my_mod);
#endif
  
  data = grub_hfsplus_mount (file->device->disk);
  if (!data)
    goto fail;

  grub_fshelp_find_file (name, &data->dirroot, &fdiro,
			 grub_hfsplus_iterate_dir,
			 grub_hfsplus_read_symlink, GRUB_FSHELP_REG);
  if (grub_errno)
    goto fail;

  file->size = fdiro->size;
  data->opened_file = *fdiro;
  grub_free (fdiro);

  file->data = data;
  file->offset = 0;

  return 0;

 fail:
  if (data && fdiro != &data->dirroot)
    grub_free (fdiro);
  grub_free (data);
  
#ifndef GRUB_UTIL
  grub_dl_unref (my_mod);
#endif

  return grub_errno;
}


static grub_err_t
grub_hfsplus_close (grub_file_t file)
{
  grub_free (file->data);

#ifndef GRUB_UTIL
  grub_dl_unref (my_mod);
#endif

  return GRUB_ERR_NONE;
}


/* Read LEN bytes data from FILE into BUF.  */
static grub_ssize_t
grub_hfsplus_read (grub_file_t file, char *buf, grub_ssize_t len)
{
  struct grub_hfsplus_data *data = 
    (struct grub_hfsplus_data *) file->data;

  int size = grub_hfsplus_read_file (&data->opened_file, file->read_hook,
				     file->offset, len, buf);

  return size;
}


static grub_err_t
grub_hfsplus_dir (grub_device_t device, const char *path, 
		  int (*hook) (const char *filename, int dir))
{
  struct grub_hfsplus_data *data = 0;
  struct grub_fshelp_node *fdiro = 0;
  
  auto int NESTED_FUNC_ATTR iterate (const char *filename,
				     enum grub_fshelp_filetype filetype,
				     grub_fshelp_node_t node);

  int NESTED_FUNC_ATTR iterate (const char *filename,
				enum grub_fshelp_filetype filetype,
				grub_fshelp_node_t node)
    {
      grub_free (node);
      
      if (filetype == GRUB_FSHELP_DIR)
	return hook (filename, 1);
      else 
	return hook (filename, 0);
      
      return 0;
    }

#ifndef GRUB_UTIL
  grub_dl_ref (my_mod);
#endif
  
  data = grub_hfsplus_mount (device->disk);
  if (!data)
    goto fail;

  /* Find the directory that should be opened.  */
  grub_fshelp_find_file (path, &data->dirroot, &fdiro,
			 grub_hfsplus_iterate_dir,
			 grub_hfsplus_read_symlink, GRUB_FSHELP_DIR);
  if (grub_errno)
    goto fail;

  /* Iterate over all entries in this directory.  */
  grub_hfsplus_iterate_dir (fdiro, iterate);
  
 fail:
  if (data && fdiro != &data->dirroot)
    grub_free (fdiro);
  grub_free (data);

#ifndef GRUB_UTIL
  grub_dl_unref (my_mod);
#endif

  return grub_errno;
}


static grub_err_t
grub_hfsplus_label (grub_device_t device __attribute__((unused))
		    , char **label __attribute__((unused)))
{
  /* XXX: It's not documented how to read a label.  */
  return grub_error (GRUB_ERR_NOT_IMPLEMENTED_YET,
		     "reading the label of a HFS+ "
		     "partiton is not implemented");
}


static struct grub_fs grub_hfsplus_fs =
  {
    .name = "hfsplus",
    .dir = grub_hfsplus_dir,
    .open = grub_hfsplus_open,
    .read = grub_hfsplus_read,
    .close = grub_hfsplus_close,
    .label = grub_hfsplus_label,
    .next = 0
  };

GRUB_MOD_INIT(hfsplus)
{
  grub_fs_register (&grub_hfsplus_fs);
#ifndef GRUB_UTIL
  my_mod = mod;
#endif
}

GRUB_MOD_FINI(hfsplus)
{
  grub_fs_unregister (&grub_hfsplus_fs);
}
