/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <stdlib.h>
#include "common.h"
#include "arena.h"
#include "misc.h"
#include "str.h"

#if __CHECKER__
#define DISCRETE_BLOCKS 1
#endif

/* PORTME: This should be the alignment used by malloc.  The default
   here should work in most cases; it was suggested by an author of
   alloca.c. */
#ifndef ALIGN_SIZE
#if defined(i386) || defined(__i386__)
#define ALIGN_SIZE 4		/* save some extra memory */
#else /* not i386 */
#define ALIGN_SIZE sizeof(double)
#endif /* not i386 */
#endif /* ALIGN_SIZE already defined */

#define HEADER_SIZE ROUND_UP (sizeof (arena), ALIGN_SIZE)

/* Creates an `arena', which can be used to allocate tiny blocks in a
   manner that is more time- and space- efficient than malloc: (1) it
   only has one header for every so many blocks, and (2) it's simpler,
   and thus faster, to allocate blocks.  The disadvantage is that all
   the blocks are freed at once, and that the blocks need to be small
   for greatest efficiency.

   Stores a pointer to the created arena in *A. */
void
arena_create (arena **a)
{
  arena *p = xmalloc (1024);
  p->prev = p->next = NULL;
  p->size = 1024;
  p->used = HEADER_SIZE;
  *a = p;
}

/* Really just a malloc wrapper that sticks the allocated block into
   the arena chain.  No gain in speed or in memory-utilization
   efficiency with this routine, but it may make it easier to clean up
   since the block is disposed of along with the rest of the arena.

   size and used fields are not important in arena_malloc blocks,
   since they're never accessed by arena_alloc.

   Returns a pointer to an allocated block of at least AMT bytes in
   arena *A. */
void *
arena_malloc (arena **a, size_t amt)
{
  arena *n;

  if (*a == NULL)
    arena_create (a);
  amt += HEADER_SIZE;
  n = xmalloc (amt);
  n->prev = (*a)->prev;
  n->next = *a;
  if ((*a)->prev)
    (*a)->prev->next = n;
  (*a)->prev = n;
  return &((char *) n)[HEADER_SIZE];
}

/* Resizes a block originally allocated with arena_malloc. */
void *
arena_realloc (arena **a, void *p, size_t amt)
{
  if (p)
    {
      arena *n;

      n = xrealloc ((arena *) & ((char *) p)[-HEADER_SIZE], HEADER_SIZE + amt);
      n->next->prev = n;
      if (n->prev)
	n->prev->next = n;
      return &((char *) n)[HEADER_SIZE];
    }
  else
    return arena_malloc (a, amt);
}

/* Returns a pointer to an allocated block of at least AMT bytes from
   *arena *A.  Best for use with small blocks (64 bytes or less).  If
   *A is NULL, a new arena is created before allocation proceeds. */
void *
arena_alloc (arena **a, size_t amt)
{
#if !DISCRETE_BLOCKS
  void *p;
  arena *n;

  if (*a == NULL)
    arena_create (a);
  if (amt > 64)
    return arena_malloc (a, amt);
  (*a)->used = ROUND_UP ((*a)->used, ALIGN_SIZE);
  if ((*a)->used + amt <= (*a)->size)
    {
      p = &((char *) (*a))[(*a)->used];
      (*a)->used += amt;
      return p;
    }

  n = xmalloc (1024);
  n->prev = *a;
  n->next = NULL;
  n->size = 1024;
  n->used = HEADER_SIZE + amt;
  (*a)->next = n;
  *a = n;
  return &((char *) (*a))[HEADER_SIZE];
#else /* Helps identify source of bugs for Checker users. */
  if (*a == NULL)
    arena_create (a);
  return arena_malloc (a, amt);
#endif
}

/* Removes a single malloc'd block from its arena and free()'s it. */
void
arena_free (void *p)
{
  arena *a = (arena *) & ((char *) p)[-HEADER_SIZE];
  assert (p);
  if (a->prev)
    a->prev->next = a->next;
  if (a->next)
    a->next->prev = a->prev;
  free (a);
}

/* Destroys arena A, if non-NULL. */
void
arena_destroy (arena **a)
{
  arena *cur, *prev;

  /* Note that *a might be within the arena!  So we've got to set it
     to NULL _before_ freeing any of the blocks, since otherwise we
     might write in a free block. */
  cur = *a;
  *a = NULL;
  for (; cur; cur = prev)
    {
      prev = cur->prev;
      free (cur);
    }
}

/* Deallocates all the sub-blocks with an arena.  Deletes any blocks
   in the arena other than the first. */
void
arena_clear (arena **a)
{
  arena *cur, *prev;

  (*a)->used = HEADER_SIZE;
  for (cur = (*a)->prev; cur; cur = prev)
    {
      prev = cur->prev;
      free (cur);
    }
  (*a)->prev = NULL;
}

/* Allocates space for string STRING in arena A. */
char *
arena_strdup (arena **a, const char *string)
{
  int amt = strlen (string) + 1;
  char *p;

  /* Note that strings need not be aligned on a boundary of ALIGN_SIZE
     bytes. */
  if ((*a)->used + amt <= (*a)->size)
    {
      p = &((char *) (*a))[(*a)->used];
      (*a)->used += amt;
    }
  else
    p = arena_alloc (a, amt);
  memcpy (p, string, amt);
  return p;
}

/* Allocates space for an s-string with the value of d-string S1,S2 in
   arena A. */
char *
arena_sd_strdup (arena **a, const char *s1, const char *s2)
{
  int amt = s2 - s1 + 1;
  char *p;

  /* Note that strings need not be aligned on a boundary of ALIGN_SIZE
     bytes. */
  if ((*a)->used + amt <= (*a)->size)
    {
      p = &((char *) (*a))[(*a)->used];
      (*a)->used += amt;
    }
  else
    p = arena_alloc (a, amt);
  memcpy (p, s1, s2 - s1);
  p[s2 - s1] = 0;
  return p;
}

/* Creates a c_string in arena A and stores a_string S in it. */
c_string *
arena_ca_strdup (arena **a, const a_string *s)
{
  int amt = sizeof (c_string) + s->l - 1;
  c_string *p;

  assert (s->l <= 65535);
#if !DISCRETE_BLOCKS
  if ((*a)->used + amt <= (*a)->size)
    {
      p = (c_string *) & ((char *) *a)[(*a)->used];
      (*a)->used += amt;
    }
  else
#endif
    p = arena_alloc (a, amt);
  STORE_2 (&p->l1, s->l);
  memcpy (p->s, s->s, s->l);
  return p;
}

/* Creates a c_string in arena A and stores null-terminated STRING in
   it. */
c_string *
arena_cs_strdup (arena ** a, const char *string)
{
  c_string *p;
  int amt;

  if (!string)
    return NULL;
  amt = sizeof (c_string) + strlen (string) - 1;

#if !DISCRETE_BLOCKS
  if ((*a)->used + amt <= (*a)->size)
    {
      p = (c_string *) & ((char *) (*a))[(*a)->used];
      (*a)->used += amt;
    }
  else
#endif
    p = arena_alloc (a, amt);

  amt += -sizeof (c_string) + 1;
  STORE_2 (&p->l1, amt);
  memcpy (p->s, string, amt);
  return p;
}

#if GLOBAL_DEBUGGING
/* Displays the contents of arena A on stdout. */
void
arena_dump (arena **a, const char *title)
{
  arena *iter;

  fputs (title, stdout);
  for (iter = *a; iter; iter = iter->prev)
    printf (" %p", iter);
  fputc ('\n', stdout);
}
#endif
