# GNU Solfege - ear training for GNOME
# Copyright (C) 2000, 2001, 2002, 2003, 2004, 2005  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from __future__ import division

__exercise_data__ = {
    'exercise_name': 'harmonic-interval',
    'stock-icon': 'solfege-harmonic-interval',
    'menupath': '/Menubar/FileMenu/IntervalsMenu',
    'name': _('_Harmonic interval'),
}
import exercise_setup
exercise_setup.register_exercise(__exercise_data__)

import os
import pickle
import time
import gtk, gnome, gobject
import gu, widgets, inputwidgets
import soundcard, mpd
import abstract, const, utils
import statistics, statisticsviewer
import htmlwidget
import cfg
import random
import gethomedir
import intervaltest

class Teacher(abstract.Teacher):
    exercise_data = __exercise_data__
    OK = 0
    ERR_PICKY = 1
    ERR_NO_INTERVALLS = 2
    ERR_NOTERANGE = 3
    def __init__(self, exname, app, config):
        abstract.Teacher.__init__(self, exname, app, config)
        self.m_statistics = statistics.HarmonicIntervalStatistics(self)
        self.m_tonika = None
        self.m_interval = None
    def new_question(self):
        """
        Return values:
        OK: new question created, all ok
        ERR_NO_INTERVALLS: no new question because no intervals are selected
        ERR_PICKY: you have to solve this question before you are allowed to create new
        """
        if self.m_timeout_handle:
            gobject.source_remove(self.m_timeout_handle)
            self.m_timeout_handle = None

        if self.get_bool('config/picky_on_new_question') \
                 and self.q_status in [const.QSTATUS_NEW, const.QSTATUS_WRONG]:
            return Teacher.ERR_PICKY

        if self.get_list('intervals') == []:
            self.q_status = const.QSTATUS_NO
            return Teacher.ERR_NO_INTERVALLS
        last_question = self.m_interval
        last_tonika = self.m_tonika
        L, H = utils.adjust_low_high_to_irange(self.get_string('lowest_tone'),
           self.get_string('highest_tone'), self.get_list('intervals'))
        while 1:
            # in this loop we will try to make a question that is not the
            # same that the last one.
            self.m_tonika, self.m_interval = \
                  utils.random_tonika_and_interval(L, H,
                    self.get_list('intervals'))
            if last_question is None:
                break
            if (self.m_interval == last_question
                and self.m_tonika == last_tonika) \
                and (len(self.get_list('intervals')) > 1 or (H - L) > 1):
                continue
            break
        assert self.m_tonika
        self.q_status = const.QSTATUS_NEW
        return Teacher.OK
    def guess_answer(self, answer):
        """
        Return: 1 if correct, None if wrong
        """
        assert self.q_status not in (const.QSTATUS_NO, const.QSTATUS_GIVE_UP)
        if self.m_interval == answer:
            if self.q_status == const.QSTATUS_NEW:
                self.m_statistics.add_correct(self.m_interval)
            self.maybe_auto_new_question()
            self.q_status = const.QSTATUS_SOLVED
            return 1
        else:
            if self.q_status == const.QSTATUS_NEW:
                self.m_statistics.add_wrong(self.m_interval, answer)
                self.q_status = const.QSTATUS_WRONG
    def give_up(self):
        """This function is only called *after* the user already has
        answered wrong once, so the statistics are already updated.
        """
        self.q_status = const.QSTATUS_GIVE_UP
    def play_question(self):
        if self.q_status == const.QSTATUS_NO:
            return
        instr_low, instr_low_vel, instr_high, instr_high_vel = self.get_instrument_config(2)
        low_tone = self.m_tonika.semitone_pitch()
        high_tone = (self.m_tonika+self.m_interval).semitone_pitch()

        m = soundcard.Track()
        m.set_bpm(self.get_int('config/default_bpm'))
        m.set_patch(0, instr_low)
        m.set_patch(1, instr_high)
        m.start_note(0, low_tone, instr_low_vel)
        m.start_note(1, high_tone, instr_high_vel)
        m.notelen_time(4)
        m.stop_note(0, low_tone, instr_low_vel)
        m.stop_note(1, high_tone, instr_high_vel)
        soundcard.synth.play_track(m)
        return 1
    def play_melodic(self):
        if self.q_status == const.QSTATUS_NO:
            return
        if self.get_bool('override_default_instrument'):
            instr_low = self.get_int('lowest_instrument')
            instr_low_vel = self.get_int('lowest_instrument_velocity')
            instr_high = self.get_int('highest_instrument')
            instr_high_vel = self.get_int('highest_instrument_velocity')
        else:
            instr_low = instr_high = self.get_int('config/preferred_instrument')
            instr_low_vel = instr_high_vel \
                    = self.get_int('config/preferred_instrument_velocity')
        low_tone = self.m_tonika.semitone_pitch()
        high_tone = (self.m_tonika+self.m_interval).semitone_pitch()
        m = soundcard.Track()
        m.set_bpm(self.get_int('config/default_bpm'))
        m.set_patch(0, instr_low)
        m.set_patch(1, instr_high)
        m.note(4, 0, low_tone, instr_low_vel)
        m.note(4, 1, high_tone, instr_high_vel)
        soundcard.synth.play_track(m)

class Gui(abstract.IntervalGui):
    def __init__(self, teacher, window):
        abstract.IntervalGui.__init__(self, teacher, window)
        self.g_test_stat_dlg = None
        ################
        # practice_box #
        ################
        self.g_merge_id = None
        self.g_repeat_melodic = gu.bButton(self.action_area,
            _("Repeat _melodic"), self.repeat_melodic)

        self.g_repeat_melodic.set_sensitive(False)

        self.g_give_up = gu.bButton(self.action_area, _("_Give up"), self.give_up)
        self.g_give_up.set_sensitive(False)
        ##############
        # config_box #        
        ##############
        frame = gtk.Frame(_("Ask for these intervals"))
        self.config_box.pack_start(frame, False)

        self.g_interval_selector = widgets.IntervalCheckBox(
                                          self.m_exname, 'intervals')

        def _ff(var):
            self.on_end_practise()
            self.on_start_practise()
        self.add_watch('intervals', _ff)
        self.g_interval_selector.set_border_width(gnome.ui.PAD)
        frame.add(self.g_interval_selector)
        #------we need some space
        self.config_box.pack_start(gtk.HBox(), False,
                                   padding=gnome.ui.PAD_SMALL)
        # -------------------------------------------------
        hbox = gu.bHBox(self.config_box, False)
        hbox.set_spacing(gnome.ui.PAD)
        hbox.pack_start(gtk.Label(_("Try to keep question with range from")),
                        False)
        lowspin = widgets.NotenameSpinButton(self.get_string('lowest_tone'))
        hbox.pack_start(lowspin, False)
        gu.bLabel(hbox, _("to"), False)
        highspin = widgets.NotenameSpinButton(self.get_string('highest_tone'))
        hbox.pack_start(highspin, False)
        self.m_notenamerange = widgets.nNotenameRangeController(lowspin, highspin,
                           mpd.LOWEST_NOTENAME, mpd.HIGHEST_NOTENAME,
                           self.m_t.m_exname, 'lowest_tone', 'highest_tone')

        # ------------------------------------------
        self._add_auto_new_question_gui(self.config_box)
        # ----------------------------------------------
        hbox = gu.bHBox(self.config_box, False)
        hbox.set_spacing(gnome.ui.PAD_SMALL)
        gu.bLabel(hbox, _("Input interface:"), False)

        combo = gtk.combo_box_new_text()
        for i in range(len(inputwidgets.inputwidget_names)):
            combo.append_text(inputwidgets.inputwidget_names[i])
        combo.set_active(self.get_int('inputwidget'))
        combo.connect('changed', lambda w: self.use_inputwidget(w.get_active()))
        hbox.pack_start(combo, False)

        self.g_disable_unused_buttons = gu.nCheckButton(self.m_exname,
                    'disable_unused_intervals', _("_Disable unused buttons"))
        hbox.pack_start(self.g_disable_unused_buttons)
        # ------------ frame -------------------
        self.g_instrument_configurator = widgets.InstrumentConfigurator(
                                                             self.m_exname, 2)
        self.config_box.pack_start(self.g_instrument_configurator, False)
        self.config_box.set_spacing(0)
        self.config_box.show_all()
        ##############
        # statistics #
        ##############
        self.setup_statisticsviewer(statisticsviewer.StatisticsViewer,
                                   _("Harmonic interval"))
        ########
        # Test #
        ########
        self.g_action_group = gtk.ActionGroup('HarmonicIntervalActions')
        self.g_action_group.add_actions([
            ('ExerciseMenu', None, _('_Exercise')),
            ('TestEx', None, _('_Run test'), None, None, self.run_test),
            ('HarmTestStatistics', None, _('_Test statistics'), None, None,
             self.show_test_statistics),
            ])
        self.g_win.g_ui_manager.insert_action_group(self.g_action_group, 0)
        self.select_inputwidget()
    def run_test(self, *v):
        tw = intervaltest.HarmonicIntervalTestDialog()
        tw.set_modal(True)
        tw.show_all()
        gtk.main()
        self.g_test_stat_dlg = intervaltest.TestStatisticsDialog(self, _("Harmonic interval test results"), tw.m_test.get_test_results_dict(), False)
        self.g_test_stat_dlg.show_all()
        tw.destroy()
    def give_up(self, _o=None):
        if self.m_t.q_status == const.QSTATUS_WRONG:
            self.g_flashbar.push(_("The answer is: %s")
                 % const.int_interval[self.m_t.m_interval])
            self.m_t.give_up()
            self.g_new_interval.set_sensitive(True)
            self.g_new_interval.grab_focus()
            self.g_give_up.set_sensitive(False)
    def get_interval_input_list(self):
        return self.get_list('intervals')
    def click_on_interval(self, mouse_button, interval, midi_int):
        if mouse_button == 1:
            # we have to ignore intervals greater than decime, because
            # the statistics viewer can't handle it. It is no big deal
            # to fix it, but then the statistics can become almost unreadable
            # because of so many intervals.
            if not -17 < interval < 17:
                self.g_flashbar.flash(_("Ignoring intervals greater than decim."))
                self.g_input.forget_last_tone()
                return
            if self.m_t.q_status == const.QSTATUS_NO:
                self.g_flashbar.flash(_("Click 'New interval' to begin."))
                return
            if self.m_t.q_status == const.QSTATUS_SOLVED:
                if self.m_t.guess_answer(interval):
                    self.g_flashbar.flash(_("Correct, but you have already solved this question"))
                else:
                    self.g_flashbar.flash(_("Wrong, but you have already solved this question"))
            elif self.m_t.q_status in (const.QSTATUS_NEW, const.QSTATUS_WRONG):
                if self.m_t.guess_answer(interval):
                    self.g_flashbar.flash(_("Correct"))
                    self.g_new_interval.set_sensitive(True)
                    self.g_new_interval.grab_focus()
                    self.g_give_up.set_sensitive(False)
                else:
                    self.g_flashbar.flash(_("Wrong"))
                    self.g_give_up.set_sensitive(True)
                    if self.get_bool("config/auto_repeat_question_if_wrong_answer"):
                        self.m_t.play_question()
            self.g_input.set_first_note(self.m_t.m_tonika)
        elif mouse_button == 2:
            # we only get here if you use one if the instrument widgets, not
            # with the buttons interface.
            soundcard.play_note(self.get_int('config/preferred_instrument'),
                          4, 0, midi_int,
                          self.get_int('config/preferred_instrument_velocity'))
        elif mouse_button == 3 and self.m_t.q_status != const.QSTATUS_NO:
            instr_low, instr_low_vel, instr_high, instr_high_vel = self.m_t.get_instrument_config(2)
            n2 = self.m_t.m_tonika + interval
            m = soundcard.Track()
            m.set_patch(0, instr_low)
            m.set_patch(1, instr_high)
            m.start_note(0, self.m_t.m_tonika.semitone_pitch(), instr_low_vel)
            m.start_note(1, n2.semitone_pitch(), instr_low_vel)
            m.notelen_time(4)
            m.stop_note(0, self.m_t.m_tonika.semitone_pitch(), instr_high_vel)
            m.stop_note(1, n2.semitone_pitch(), instr_high_vel)
            soundcard.synth.play_track(m)
    def new_question(self, _o=None):
        """This function is called when you click on the 'New interval'
        button, or when you use the key bindings. So it can be called even
        if the 'New interval' button is insensitive.
        """
        g = self.m_t.new_question()
        if g == Teacher.OK: # new question, everything is OK
            self.g_repeat.set_sensitive(True)
            self.g_repeat_melodic.set_sensitive(True)
            self.g_input.set_first_note(self.m_t.m_tonika)
            self.g_new_interval.set_sensitive(
              not self.get_bool('config/picky_on_new_question'))
            self.m_t.play_question()
            self.g_flashbar.clear()
            #inputwidget 0 is always the buttons.
            if self.get_int('inputwidget') == 0:
                self.g_input.grab_focus_first_sensitive_button()
        elif g == Teacher.ERR_PICKY:
            self.g_flashbar.flash(_("You have to solve this question first."))
        elif g == Teacher.ERR_NO_INTERVALLS:
            self.g_repeat.set_sensitive(False)
            self.g_repeat_melodic.set_sensitive(False)
            self.g_flashbar.clear()
            self.g_flashbar.flash(
                    _("You have to select some intervals to practise."))
    def repeat_melodic(self, *w):
        self.m_t.play_melodic()
        self.g_input.grab_focus_first_sensitive_button()
    def on_start_practise(self):
        self.m_t.m_statistics.reset_session()
        self.g_action_group.set_sensitive(True)
        self.g_new_interval.grab_focus()
        gobject.timeout_add(const.SHORT_WAIT, lambda self=self:
            self.g_flashbar.flash(_("Click 'New interval' to begin.")))
        self.g_merge_id = self.g_win.g_ui_manager.add_ui_from_string("""
<menubar name='Menubar'>
  <menu action='ExerciseMenu'>
    <menuitem action='TestEx'/>
    <menuitem action='HarmTestStatistics'/>
  </menu>
</menubar>""")
    def on_end_practise(self):
        if self.g_merge_id:
            self.g_win.g_ui_manager.remove_ui(self.g_merge_id)
        self.m_t.end_practise()
        self.g_new_interval.set_sensitive(True)
        self.g_repeat.set_sensitive(False)
        self.g_repeat_melodic.set_sensitive(False)
        self.g_give_up.set_sensitive(False)
        self.g_flashbar.clear()
        self.g_input.clear()
