// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.
//
// This exception applies only to the code released under the name GNU
// ccScript.  If you copy code from other releases into a copy of GNU
// ccScript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for GNU ccScript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.
//

#ifndef	CCXX_BAYONNE_H_
#define	CCXX_BAYONNE_H_

#ifndef	CCXX_SCRIPT3_H_
#include <cc++/script3.h>
#endif

#ifndef	CCXX_CCAUDIO2_H_
#include <cc++/audio2.h>
#endif

#ifndef	CCXX_SOCKET_H_
#include <cc++/socket.h>
#endif

#define	NO_TIMESLOT	0xffff
#define	MAX_DTMF	32
#define	MAX_LIST	256

namespace ost {

class __EXPORT BayonneMsgport;
class __EXPORT BayonneDriver;
class __EXPORT BayonneSession;
class __EXPORT BayonneSpan;
class __EXPORT BayonneService;

/**
 * Generic Bayonne master class to reference various useful data types
 * and core static members used for locating resources found in libbayonne.
 *
 * @author David Sugar <dyfet@gnutelephony.org>
 * @short Master bayonne library class.
 */
class __EXPORT Bayonne : public Script
{
public:
#ifdef	WIN32
	typedef	WORD	timeslot_t;
#else
	typedef uint16_t timeslot_t;
#endif

protected:
	static BayonneSession **timeslots;
	static char *status;
	static ScriptCommand *server;
	static timeslot_t ts_count;
	static timeslot_t ts_used;
	static std::ostream *logging;
	static const char *path_prompts;
	static const char *path_tmpfs;
	static const char *path_tmp;
	static unsigned idle_count;
	static unsigned idle_limit;
	static bool shutdown_flag;
	static char sla[];
	static time_t start_time;

public:
	static timeout_t step_timer;
	static timeout_t reset_timer;
	static timeout_t exec_timer;

	static BayonneDriver *start_driver;
	static Audio::Encoding peer_encoding;
	static Audio::timeout_t peer_framing;

	/**
	 * Telephony endpoint interface identifiers.
	 */ 
	typedef enum
	{
		IF_PSTN,
		IF_SPAN,
		IF_ISDN,
		IF_SS7,
		IF_INET,
		IF_NONE,
		IF_POTS=IF_PSTN
	}	interface_t;

	/**
	 * Type of call session being processed.
	 */
	typedef enum
	{
		NONE,		/* idle */
		INCOMING,	/* inbound call */
		OUTGOING,	/* outbound call */
		PICKUP,		/* station pickup */
		FORWARDED,	/* inbound forwarded call */
		RECALL		/* inbound hold recall */
	}	calltype_t;

	/**
	 * Type of bridge used for joining ports.
	 */
	typedef	enum
	{
		BR_TDM,
		BR_INET,
		BR_SOFT,
		BR_GATE,
		BR_NONE
	}	bridge_t;

	/**
	 * Call processing states offered in core library.  This list
	 * must be ordered to match the entries in the state table
	 * (statetab).
	 */ 
	typedef enum
	{
		STATE_INITIAL = 0,
		STATE_IDLE,
		STATE_RESET,
		STATE_RELEASE,
		STATE_BUSY,
		STATE_DOWN,
		STATE_RING,
		STATE_PICKUP,
		STATE_SEIZE,
		STATE_STEP,
		STATE_EXEC,
		STATE_THREAD,
		STATE_CLEAR,
		STATE_INKEY,
		STATE_INPUT,
		STATE_READ,
		STATE_COLLECT,
		STATE_DIAL,
		STATE_TONE,
		STATE_PLAY,
		STATE_RECORD,
		STATE_JOIN,
		STATE_WAIT,
		STATE_CONNECT,
		STATE_SLEEP,
		STATE_START,
		STATE_HANGUP,
		STATE_FINAL,

		STATE_STANDBY = STATE_DOWN,
		STATE_LIBEXEC = STATE_EXEC,
		STATE_RINGING = STATE_RING,
		STATE_RUNNING = STATE_STEP,
		STATE_THREADING = STATE_THREAD
	} state_t;

	/**
	 * Signaled interpreter events.  These can be masked and accessed
	 * through ^xxx handlers in the scripting language.
	 */
	typedef enum
	{
		SIGNAL_EXIT = 0,
		SIGNAL_ERROR,
		SIGNAL_TIMEOUT,
		SIGNAL_DTMF,
	
		SIGNAL_0,
		SIGNAL_1,
		SIGNAL_2,
		SIGNAL_3,
		SIGNAL_4,
		SIGNAL_5,
		SIGNAL_6,
		SIGNAL_7,
		SIGNAL_8,
		SIGNAL_9,
		SIGNAL_STAR,
		SIGNAL_POUND,
		SIGNAL_A,
		SIGNAL_OVERRIDE = SIGNAL_A,
		SIGNAL_B,
		SIGNAL_FLASH = SIGNAL_B,
		SIGNAL_C,
		SIGNAL_IMMEDIATE = SIGNAL_C,
		SIGNAL_D,
		SIGNAL_PRIORITY = SIGNAL_D,

		SIGNAL_RING,
		SIGNAL_TONE,
		SIGNAL_EVENT,
		SIGNAL_WINK,

		SIGNAL_CHILD,
		SIGNAL_FAIL,
		SIGNAL_PICKUP,
		SIGNAL_PART,

		SIGNAL_INVALID,
		SIGNAL_PARENT,
		SIGNAL_WAIT,

		SIGNAL_HANGUP = SIGNAL_EXIT
	} signal_t;

	/**
	 * Primary event identifiers.  These are the events that can
	 * be passed into the Bayonne state machine.  They are broken
	 * into categories.
	 */
	typedef enum 
	{
		// msgport management events, server control...

		MSGPORT_WAKEUP = 0,	// wakeup a waiting message port
		MSGPORT_SHUTDOWN,	// notify msgport of server shutdown
		MSGPORT_LOGGING,	// enable driver event logging
		MSGPORT_REGISTER,	// update registrations

		// primary event identifiers

		ENTER_STATE = 100,	// newly entered state handler
		EXIT_STATE,
		EXIT_THREAD,		// thread death notify
		EXIT_TIMER,		// timer notified exit
		NULL_EVENT,		// generic null event pusher

		// primary session control events

		START_SCRIPT = 200,	// start a script, incoming
		START_SELECTED,		// start selected script, outbound
		START_RECALL,		// start a recalled call
		START_FORWARDED,	// start a forwarded call
		STOP_SCRIPT,
		STOP_DISCONNECT,
		STOP_PARENT,
		CANCEL_CHILD,
		DETACH_CHILD,
		CHILD_RUNNING,
		CHILD_FAILED,
		CHILD_INVALID,
		CHILD_EXPIRED,
		CHILD_BUSY,
		CHILD_FAX,

		START_INCOMING = START_SCRIPT,
		START_OUTGOING = START_SELECTED,

		// libexec specific events

		ENTER_LIBEXEC = 300,
		EXIT_LIBEXEC,
		HEAD_LIBEXEC,
		ARGS_LIBEXEC,
		VAR_LIBEXEC,
		GET_LIBEXEC,
		DROP_LIBEXEC,
		STAT_LIBEXEC,

		// primary driver events

		TIMER_EXPIRED = 400,	// trunk timer expired
		LINE_WINK,
		LINE_PICKUP,
		LINE_HANGUP,
		LINE_DISCONNECT,
		LINE_ON_HOOK,
		LINE_OFF_HOOK,
		RING_ON,
		RING_OFF,
		RING_STOP,
		LINE_CALLER_ID,
		RINGING_DID,
		DEVICE_BLOCKED,
		DEVICE_UNBLOCKED,
		DEVICE_OPEN,
		DEVICE_CLOSE,
		DSP_READY,

		// primary call processing events

		CALL_DETECT = 500,
		CALL_CONNECTED,
		CALL_RELEASED,
		CALL_ACCEPTED,
		CALL_ANSWERED,
		CALL_HOLD,
		CALL_HOLDING=CALL_HOLD,
		CALL_NOHOLD,
		CALL_DIGITS,
		CALL_OFFERED,
		CALL_ANI,
		CALL_ACTIVE,
		CALL_NOACTIVE,
		CALL_BILLING,
		CALL_RESTART,
		CALL_SETSTATE,
		CALL_FAILURE,
		CALL_ALERTING,
		CALL_INFO,
		CALL_BUSY,
		CALL_DIVERT,
		CALL_FACILITY,
		CALL_FRAME,
		CALL_NOTIFY,
		CALL_NSI,
		CALL_RINGING,
		CALL_DISCONNECT,
		CALL_CLEARED,
		RESTART_FAILED,
		RELEASE_FAILED,

		// some timeslot specific control and dial events

		START_RING = 600,
		STOP_RING,
		CLEAR_TIMESLOT,	// garbage collect
		START_FLASH,
		STOP_FLASH,
		DIAL_CONNECT,
		DIAL_TIMEOUT,
		DIAL_FAILED,
		DIAL_INVALID,
		DIAL_BUSY,
		DIAL_FAX,
		DIAL_PAM,

		DIAL_MACHINE = DIAL_PAM,

		// basic audio stuff

		AUDIO_IDLE = 700,
		AUDIO_ACTIVE,
		AUDIO_EXPIRED,
		INPUT_PENDING,
		OUTPUT_PENDING,
		AUDIO_BUFFER,
		TONE_IDLE,
		DTMF_KEYDOWN,
		DTMF_KEYUP,
		TONE_START,
		TONE_STOP,
		AUDIO_START,
		AUDIO_STOP,
		DTMF_START,
		DTMF_STOP,

		// make modes and special timeslot stuff

		MAKE_TEST = 800,
		MAKE_BUSY,
		MAKE_IDLE,
		MAKE_DOWN,
		MAKE_UP,
		MAKE_EXPIRED,
		ENABLE_LOGGING,
		DISABLE_LOGGING,
		PART_USER,
		PART_TIMER,
		PART_DIGIT,
		PART_EXIT,
		JOIN_PEER,
		PEER_WAITING,

		// master control events

		SYSTEM_DOWN = 900,	

		// driver specific events and anomalies

		DRIVER_SPECIFIC = 1000	// oddball events

	} event_t;

	/**
	 * The event data structure includes the event identifier and
	 * any paramaters.  Additional information is attached both to
	 * assist in debugging, and to track which timeslot a given
	 * event is being issued against when queued through a master
	 * msgport.
	 */
	typedef struct
	{
		event_t id;
		timeslot_t timeslot;
		uint16 seq;

		union
		{
			// used to invoke attach.  The image refcount
			// is assumed to already be inc'd!  If no
			// scr is passed, then uses default or search
			// vars passed in sym, value pairs
			struct
			{
				ScriptImage *img;
				Script::Name *scr;
				BayonneSession *parent;
				const char *dialing;
			}	start;

			struct
			{
				const char *tid;
				const char *fname;
				int pid, result;
			}	libexec;

			struct
			{
				const char *tid;
				const char *name;
				const char *value;
				int size;
			}	varexec;

			struct
			{
				const char *tid;
				timeout_t timeout;
				unsigned count;
				const char *exit;
			}	libinput;

			struct
			{
				timeout_t duration;
				int digit;
			}	dtmf;

			struct
			{
				const char *err;
				const char *msg;
			}	cpa;

			struct
			{
				const char *name;
				bool exit;
			}	tone;

			struct
			{
				std::ostream *output;
				const char *logstate;
			}	debug;

			const char *errmsg;
			BayonneSession *pid;
			BayonneSession *peer;
			void *data;
		};

	}	Event;

	/**
	 * The current state handler in effect for a given channel
	 * to receive events.  This is done by a direct method pointer
	 * for fast processing.
	 */	
	typedef bool (BayonneSession::*Handler)(Event *event);

	/**
	 * A list of each state and a description.  This is used so that
	 * named states can be presented when debugging posted events.
	 */
	typedef struct
	{
		const char *name;
		Handler handler;
		char flag;
	}	statetab;

	/**
	 * The primary state data structure.   This holds data that is setup
	 * by the interpreter and which must remain persistent for
	 * the execution of a given state.  This is composed of some
	 * common elements which exist in a session for all states, and
	 * a union of state specific data elements, all packed together.
	 */
	typedef struct
	{
		Handler handler, logstate;
		const char *name;
		timeout_t timeout;
		Name *menu;
		unsigned stack;
		Line *lib;
		int result;
		int pid;
#ifndef	WIN32
		int pfd;
#endif

		union
		{
			struct
			{
				Audio::Mode mode;
				Audio::Level level;
				timeout_t total, silence, intersilence;
				long lastnum;
				bool exitkey;
				bool compress;
				bool trigger;
				const char *pos;
				const char *exit;
				const char *menu;
				const char *note;
				const char *list[MAX_LIST];
			} 	audio;
				
			struct
			{
				timeout_t interdigit;
				const char *var;
				const char *exit;
				const char *format;
				const char *ignore;
				const char *route;
				unsigned count, size, required;
			}	input;

			struct
			{
				const char *var;
				const char *menu;
			}	inkey;

			struct
			{
				const char *sequence;
				bool flashing;
				bool dialing;
				bool exiting;
				timeout_t duration;
				char digits[64];
				char sessionid[16];
			}	tone;

			struct
			{
				timeout_t on, off, interdigit;
				unsigned pos;
				bool flashing;
				bool dialing;
				unsigned char digits[64];
			}	pulse;	

			struct
			{
				const char *dial;
				const char *exit;
				BayonneSession *peer;
				timeout_t answer_timer;
				char digits[64];
			}	join;
		};
		
	}	State;
	
	/**
	 * Table of states ordered by id.
	 */
	static statetab states[];

	/**
	 * A mutex to serialize any direct console I/O operations.  Sometimes
	 * used to serialize other kinds of time insensitive requests.
	 */
	static Mutex serialize;

	/**
	 * Allocates the maximum number of timeslots the server will use as
	 * a whole and attaches a given server to the library.
	 *
	 * @param timeslots to allocate.
	 * @param pointer to server shell.
	 */	
	static void allocate(timeslot_t timeslots, ScriptCommand *server = NULL);

	/**
	 * Get server uptime.
	 *
	 * @return uptime in seconds.
	 */
	static unsigned long uptime(void);

	/**
	 * Request active scripts to be recompiled from the library.
	 *
	 * @return script image that was created, or NULL.
	 */
	static ScriptCompiler *reload(void);

	/**
	 * Used to down the server from the library.
	 */
	static void down(void);

	/**
	 * Sets server service level from the library.
	 *
	 * @param service level or NULL to clear.
	 * @return true if set.
	 */
	static bool service(const char *svc);

	/**
	 * Used internally to compile a directory of script files.
	 *
	 * @param directory prefix to compile.
	 * @param script image to build into.
	 */
	static void compileDir(const char *prefix, ScriptCompiler *img);

	/**
	 * Returns a session pointer for a server timeslot.  Each
	 * server timeslot can map to a single session object.
	 *
	 * @param server timeslot number.
	 * @return session object or NULL if timeslot empty/invalid.
	 */
	static BayonneSession *getSession(timeslot_t timeslot);

	/**
	 * Returns a session pointer for a string identifier.  This can
	 * be for a transaction id, a call id, or other unique identifiers
	 * which can be mapped into a single timeslot.
	 *
	 * @param string session identifier.
	 * @return session object or NULL if not found.
	 */
	static BayonneSession *getSid(const char *id);

	/**
	 * Returns a server timeslot number for a string identifier.
	 *
	 * @return timeslot number or invalid value.
	 * @param string identifier for a session.
	 * @see #getSid
	 */
	static timeslot_t toTimeslot(const char *id);

	/**
	 * Return total library timeslots used (highest used).
	 *
	 * @return highest server timeslot in use.
	 */
	static inline timeslot_t getTimeslotsUsed(void)
		{return Bayonne::ts_used;};

	/**
	 * Return total timeslots allocated for the server.
	 *
	 * @return total number of timeslots, max + 1.
	 */
	static inline timeslot_t getTimeslotCount(void)
		{return Bayonne::ts_count;};	

	/**
	 * Return remaining timeslots available to allocate driver
	 * ports into.
	 *
	 * @return remaining timeslots.
	 */
	static inline timeslot_t getAvailTimeslots(void)
		{return Bayonne::ts_count - Bayonne::ts_used;};

	/**
	 * Map a state name into a handler.  Used for logging requests.  Uses
	 * the statetab.
	 *
	 * @param name of state to lookup.
	 * @return handler method for state if found.
	 */
	static Handler getState(const char *name);

	/**
	 * Convert a dtmf character into a 0-15 number reference.
	 *
	 * @param dtmf digit as ascii
	 * @return dtmf digit number.
	 */
	static int getDigit(char dig);

	/**
	 * Convert a dtmf digit number into it's ascii code.
	 *
	 * @param dtmf digit number.
	 * @return dtmf character code.
	 */
	static char getChar(int dtmf);

	/**
	 * A function to support pattern matching and templates for digit
	 * strings.  This is used for digit @xxx:... entries and the route
	 * command.
	 *
	 * @param digits to use.
	 * @param digit pattern to match against.
	 * @param accept partial match.
	 * @return true if digits match to pattern.
	 */
	static bool matchDigits(const char *digits, const char *match, bool partial = false);

	/**
	 * Use the current compiled script image; mark as in use.
	 *
	 * @return current script image to pass to new calls.
	 */
	static ScriptImage *useImage(void);

	/**
	 * Release a script image in use.  If no active calls are using it
	 * and it's no longer the top active image, purge from memory.
	 *
	 * @param script image from useImage.
	 */
	static void endImage(ScriptImage *img);

	/**
	 * Load a plugin module.
	 *
	 * @param prefix path for plugin.
	 * @param id of plugin.
	 * @return true if successful.
	 */
	static bool loadPlugin(const char *prefix, const char *path);
};

/**
 * A core class to support language translation services in Bayonne
 * phrasebook.
 *
 * @author David Sugar <dyfet@gnutelephony.org>
 * @short Phrasebook translation base class.
 */
class __EXPORT BayonneTranslator : public Bayonne
{
protected:
	static BayonneTranslator *first;
	BayonneTranslator *next;
	const char *id;

public:
	/**
	 * Create a translator instance for a specific language identifier.
	 * Generally iso names will be sued.  Sometimes iso sub-identifiers 
	 * may be used for nation specific versions of phrasebook.
	 *
	 * @param iso name of language/locale.
	 */
	BayonneTranslator(const char *id);

	/**
	 * Find a translator for a given name/location.
	 *
	 * @param iso language/locale name of translator to find.
	 * @return derived translator class for translator.
	 */
	static BayonneTranslator *get(const char *name);

	/**
	 * Load a named translator into memory for use.  This is used
	 * by the fifo/script language command.
	 *
	 * @param iso module name to load.
	 * @return true if successful.
	 */
	static bool load(const char *prefix, const char *name);

	/**
	 * Translate a simple set of digits to spoken speech.
	 *
	 * @param timeslot session to save list of prompts.
	 * @param current count of prompts used.
	 * @param string to be processed.
	 * @return new count of prompts used.
	 */
	virtual unsigned digits(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Spell out the string as individual letters.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned spell(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate an ordinal number (xxnth) to prompts.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned order(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate a number to spoken speech.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned number(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate a counting number (integer) to spoken speech.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned counting(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate a string for time into speech.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned saytime(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate a string with a date into the spoken weekday.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned weekday(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate a string with a date into a spoken date.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned saydate(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate a logical value and speak as yes/no.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned saybool(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate and speak a phone number.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned phone(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translate and speak a phone extension.
         *
         * @param timeslot session to save list of prompts.
         * @param current count of prompts used.
         * @param string to be processed.
         * @return new count of prompts used.
         */
	virtual unsigned extension(BayonneSession *s, unsigned count, const char *cp);

        /**
         * Translation dispatch, processes script and invokes other methods.
         *
         * @return error message or NULL if okay
	 * @param timeslot session to process a command from.
         */
	virtual const char *speak(BayonneSession *session);
};

/**
 * Offers core Bayonne audio processing in a self contained class.  The
 * BayonneAudio class is used with each session object.
 *
 * @short self contained Bayonne audio processing. 
 * @author David Sugar <dyfet@gnutelephony.org>
 */
class __EXPORT BayonneAudio : public AudioStream, public Bayonne
{
protected:
	char filename[256];
	const char **list;
	char *getContinuation(void);

public:
	/**
	 * Current tone object to use for generation of audio tones,
	 * dtmf dialing sequences, etc.
	 */
	AudioTone *tone;

	/**
	 * Current language translator in effect for the current set of
	 * autio prompts.
	 */
	BayonneTranslator *translator;

	const char *extension, *voicelib, *libext, *prefixdir, *offset;
	Encoding encoding;
	timeout_t framing;
	char var_position[14];

	/**
	 * Initialize instance of audio.
	 */
	BayonneAudio();

	/**
	 * Convert a prompt identifier into a complete audio file pathname.
	 *
	 * @return pointer to fully qualified file path or NULL if invalid.
	 * @param prompt requested.
	 * @param true if request is for file write.
	 */
        const char *getFilename(const char *name, bool write = false);

	/**
	 * Clear open files and other data structures from previous
	 * audio processing operations.
	 */
	void cleanup(void);
	
	/**
	 * Open a sequence of audio prompts for playback.	
	 *
	 * @param list of prompts to open.
	 * @param playback mode for file processing of list.
	 */
	void play(const char **list, Mode m = modeRead);

	/**
	 * Open an audio prompt for recording.
	 *
	 * @param name of prompt to open.
	 * @param mode, whether to create or use pre-existing recording.
	 * @param annotation to save in file if supported by format used.
	 */
	void record(const char *name, Mode m = modeCreate, const char *note = NULL);

	/**
	 * Check if a voice library is available.
	 *
	 * @return true if voice library can be used.
	 * @param iso name of library to request.
	 */
	bool isVoicelib(const char *cp);

	/**
	 * Get audio codec used.
	 *
	 * @return audio codec.
	 */
	inline AudioCodec *getCodec(void)
		{return codec;};
};		

/**
 * Bayonne Msgports are used to queue and post session events which
 * normally have to be passed through another thread context.  This
 * can happen for a thread termination event, for example, since the
 * thread terminating must be joined from another thread.  Some drivers
 * use session specific msgports to process all channel events.
 *
 * @author David Sugar <dyfet@gnutelephony.org>
 * @short Msgport event queing and dispatch.
 */
class __EXPORT BayonneMsgport : public Thread, public Buffer, public Bayonne
{
public:
	/**
	 * Destroy a msgport.
	 */
	virtual ~BayonneMsgport();

	/**
	 * Create a message port and optionally bind it to a given driver.
	 *
	 * @param driver to bind msgport to.
	 */
	BayonneMsgport(BayonneDriver *driver);

	/**
	 * Request retiming.  This is used for msgports that are per
	 * session to get the session to be retimed after an event has
	 * been directly posted outside the msgport.
	 */
        void update(void); 

	/**
	 * Initialize msgport, determine which sessions it will perform
	 * timing for based on the driver it is bound to.
	 */
	void initial(void);

protected:
	BayonneDriver *msgdriver;
	Event *msglist;
	unsigned msgsize, msghead, msgtail;
	timeslot_t tsfirst, tscount;
	char msgname[16];

	/**
	 * Send shutdown event to the msgport.
	 */
	void shutdown(void);

	/**
	 * Determine sleep time to schedule for waiting, unless an
	 * update occurs to force rescheduling.
	 *
	 * @return shortest timeout based on session timers.
	 * @param event to pass when timeout occurs.
	 */
	virtual timeout_t getTimeout(Event *event);
	
	void run(void);

        size_t onWait(void *buf);
        size_t onPost(void *buf);
        size_t onPeek(void *buf);
};

/**
 * The principle driver node for a given collection of spans and sessions
 * of a given Bayonne driver family type.
 *
 * @author David Sugar <dyfet@gnutelephony.org>
 * @short Bayonne driver node class.
 */
class __EXPORT BayonneDriver : public Bayonne, public Keydata, public Mutex
{
protected:
	friend class __EXPORT BayonneSession;
	friend class __EXPORT BayonneMsgport;
	static BayonneDriver *firstDriver;
	static BayonneDriver *lastDriver;
	static Semaphore oink;	// the pig has been kicked!

	BayonneSession *firstIdle, *lastIdle;
	BayonneMsgport *msgport;
	BayonneDriver *nextDriver;
	const char *name;
	timeslot_t timeslot, count;
	unsigned avail;
	unsigned span, spans;	
	bool running;
	std::ostream *logevents;

	int audio_priority;
	size_t audio_stack;
	Audio::Level audio_level;

	timeout_t pickup_timer, hangup_timer, seize_timer, ring_timer;
	timeout_t reset_timer, release_timer, flash_timer, interdigit_timer;
	unsigned answer_count;

	/**
	 * Virtual to override method for activating the driver and
	 * creating all session and span objects associated with it.
	 */
	virtual void startDriver(void);

	/**
	 * Virtual to override method for clean shutdown of the driver.
	 */
	virtual void stopDriver(void);

public:
	/**
	 * Create a driver instance.
	 *
	 * @param default keyword entries for config.
	 * @param name of config key.
	 * @param driver id string.
	 * @param whether front load or use all timeslots.
	 */
	BayonneDriver(Keydata::Define *pairs, const char *cfg, const char *id, bool pri);

	/**
	 * Destroy driver instance.
	 */
	~BayonneDriver();

	/**
	 * Get longest idle session to active for call processing.
	 *
	 * @return handle to longest idle session, if none idle, NULL.
	 */
	BayonneSession *getIdle(void);

	/**
	 * Process driver protocol specific proxy registration requests.
	 *
	 * @return error message if invalid request, NULL if ok.
	 * @param script image being compiled.
	 * @param line record of "register" command.
	 */
        virtual const char *registerScript(ScriptImage *img, Line *line);

	/**
	 * Process driver specific assign requests.
	 *
	 * @return error message if invalid request, NULL if ok.
	 * @param script image being compiled.
	 * @param line record of "assign" command.
	 */
	virtual const char *assignScript(ScriptImage *img, Line *line);

	/**
	 * Find and return driver object from id name.
	 *
	 * @param driver name.
	 * @return associated driver node.
	 */
	static BayonneDriver *get(const char *id);

	/**
	 * Load a bayonne driver into memory.
	 *
	 * @param driver name to load.
	 * @return NULL or pointer to loaded driver.
	 */
	static BayonneDriver *load(const char *prefix, const char *id);

	/**
	 * Get list of driver names into string array.
	 *
	 * @param array to save in.
	 * @param count of elements available.
	 * @return number of drivers.
	 */
	static unsigned BayonneDriver::list(char **items, unsigned max);

	/**
	 * Start all loaded drivers.
	 */
	static void start(void);
	
	/**
	 * Stop all loaded drivers.
	 */
	static void stop(void);

	/**
	 * Add session to driver idle list for getIdle, usually during
	 * stateIdle.
	 *
	 * @param session being added.
	 */ 
	static void add(BayonneSession *session);

	/**
	 * Remove session from driver idle list if still present.  Usually
	 * when changing from idle to an active state.
	 *
	 * @param session being removed.
	 */
	static void del(BayonneSession *session);

	/**
	 * Get first server timeslot this driver uses.
	 *
	 * @return first server timeslot for driver.
	 */
	inline timeslot_t getFirst(void)
		{return timeslot;};

	/**
	 * Get the total number of timeslots this driver uses.
	 *
	 * @return total timeslots for driver.
	 */
	inline timeslot_t getCount(void)
		{return count;};

	/**
	 * Get the first span id used.
	 *
	 * @return span id.
	 */
	inline unsigned getSpanFirst(void)
		{return span;};

	/**
	 * Get the number of span objects used by driver.
	 *
	 * @return span count.
	 */
	inline unsigned getSpansUsed(void)
		{return spans;};

	/**
	 * Get the name of the driver.
	 *
	 * @return name of driver.
	 */
	inline const char *getName(void)
		{return name;};

	/**
	 * Get the reset timer for this driver when resetting a thread in
	 * the step state.
	 *
	 * @return reset timer in milliseconds.
	 */
	inline timeout_t getResetTimer(void)
		{return reset_timer;};

	/**
	 * Get the release timer when releasing a trunk.
 	 *
	 * @return release timer in milliseconds.
	 */
	inline timeout_t getReleaseTimer(void)
		{return release_timer;};

	/**
	 * Get the hangup timer for hang time before going idle.
	 *
	 * @return hangup timer in milliseconds.
	 */
	inline timeout_t getHangupTimer(void)
		{return hangup_timer;};

	/**
	 * Get the pickup timer to wait for channel pickup.
	 *
	 * @return pickup timer in milliseconds.
	 */
	inline timeout_t getPickupTimer(void)
		{return pickup_timer;};

	/**
	 * Get the sieze time to wait for dialtone on outbound call.
	 *
	 * @return seize timer in milliseconds.
	 */
	inline timeout_t getSeizeTimer(void)
		{return seize_timer;};

	/**
	 * Get the programmed flash timer to signal trunk flash.
	 *
	 * @return flash timer in milliseconds.
	 */
	inline timeout_t getFlashTimer(void)
		{return flash_timer;};

	/**
	 * Get default dtmf interdigit timer to use.
	 *
	 * @return interdigit timer in milliseconds.
	 */
	inline timeout_t getInterdigit(void)
		{return interdigit_timer;};

	/**
	 * Get the timer to wait for next ring before deciding a call
	 * has dissapeared.  Used when set to answer on nth ring.
	 *
	 * @return ring timer in milliseconds.
	 * #see getAnswerCount.
	 */
	inline timeout_t getRingTimer(void)
		{return ring_timer;};

	/**
	 * Get the number of rings to wait before answering.
	 *
	 * @return number of rings before answer.
	 */
	inline unsigned getAnswerCount(void)
		{return answer_count;};

	/**
	 * Get the nth span object associated with this driver.
	 *
	 * @param nth span to return.
	 * @return span object or NULL if past limit/no spans.
	 */
	BayonneSpan *getSpan(unsigned id);

	/**
	 * Get the session associated with the nth timeslot for this
	 * driver.
	 *
	 * @param nth timeslot of driver.
	 * @return session object.
	 */
	BayonneSession *getTimeslot(timeslot_t id);

	/**
	 * Return the message port bound with this driver.
	 *
	 * @return bound msgport for driver.
	 */
	inline BayonneMsgport *getMsgport(void)
		{return msgport;};

	/**
	 * Get the size of the stack for audio threads.
	 *
	 * @return stack size in bytes.
	 */
	inline size_t getAudioStack(void)
		{return audio_stack;};

	/**
	 * Get the thread priority to use for audio threads for this driver.
	 *
	 * @return thread priority.
	 */
	inline int getAudioPriority(void)
		{return audio_priority;};

	/**
	 * Get the audio level for silence detection.
	 *
	 * @return audio threashold for silence.
	 */
	inline Audio::Level getAudioLevel(void)
		{return audio_level;};

	/**
	 * Set driver logging.
	 *
	 * @param stream to log driver.
	 */
	inline void setLogging(std::ostream *output)
		{logevents = output;};
};	

/**
 * A span is a collection of ports under a single control interface or
 * communication channel, such as a T1/E1/PRI/BRI span.
 *
 * @author David Sugar <dyfet@gnutelephony.org>
 * @short Span management object.
 */
class __EXPORT BayonneSpan : public Bayonne, public Keydata
{
protected:
	friend class __EXPORT BayonneSession;

	static BayonneSpan *first;
	static BayonneSpan *last;
	static unsigned spans;
	static BayonneSpan **index;
	
	unsigned id;
	BayonneDriver *driver;
	BayonneSpan *next;
	timeslot_t timeslot, count;		// timeslots

public:
	/**
	 * Create a span for a specified number of timeslots.
	 *
	 * @param driver associated with span.
	 * @param timeslots this span covers.
	 */
	BayonneSpan(BayonneDriver *d, timeslot_t count);

	/**
	 * Get a span by a global span id.
	 *
	 * @param numeric span id.
	 * @return span object associated with id.
	 */
	static BayonneSpan *get(unsigned id);
	
	/**
	 * Get the session associated with the nth timeslot of the span.
	 *
	 * @param nth timeslot of span.
	 * @return associated session object.
	 */
	BayonneSession *getTimeslot(timeslot_t id);

	/**
	 * Allocate the total number of spans this server will support.
	 *
	 * @param total span count.
	 */
	static void allocate(unsigned count = 0);

	/**
	 * Return total spans in use.
	 *
	 * @return total spans in use.
	 */
	static inline unsigned getSpans(void)
		{return spans;};

	/**
	 * Get the first server timeslot of this span.
	 *
	 * @return first server timeslot.
	 */
	inline timeslot_t getFirst(void)
		{return timeslot;};

	/**
	 * Return total number of server timeslots in this span.
	 *
	 * @return server timeslot count.
	 */
	inline timeslot_t getCount(void)
		{return count;};

	/**
	 * Get the id associated with this span.
	 *
	 * @return global id of this span object.
	 */
	inline unsigned getId(void)
		{return id;};

	/**
	 * Get driver associated with this span.
	 *
	 * @return driver object for span.
	 */
	inline BayonneDriver *getDriver(void)
		{return driver;};
};

/**
 * The primary session object representing a server timeslot and active
 * communication endpoint in Bayonne.
 *
 * @author David Sugar <dyfet@gnutelephony.org>
 * @short Session timeslot object.
 */
class __EXPORT BayonneSession : public ScriptInterp, public Bayonne
{
private:
	friend class __EXPORT BayonneMsgport;
	friend class __EXPORT BayonneTranslator;
	friend class __EXPORT BayonneDriver;
	friend class __EXPORT Bayonne;

	BayonneSession() {};

	BayonneSession *nextIdle, *prevIdle;
	bool isAvail;

protected:
	static BayonneTranslator langNone;
	static ScriptSymbols *globalSyms;
	static Mutex globalLock;	

	std::ostream *logevents, *logtrace;
	BayonneDriver *driver;
	BayonneMsgport *msgport;
	BayonneSession *peer;
	BayonneSpan *span;
	timeslot_t timeslot;
	uint8 seq;
	uint16 evseq;
	uint32 tseq;
	volatile unsigned children;
	bool offhook, dtmf, answered, starting;
	time_t exittimer, starttime;
	interface_t iface;
	bridge_t bridge;
	calltype_t type;
	event_t seizure;
	/**
	 * Used to indicate commands which require dtmf handling to be
	 * enabled.
	 *
	 * @return true if dtmf was enabled.  If false, error processing
	 * occured for the interpreter.
	 */
	bool requiresDTMF(void);

	/**
	 * Enable dtmf detection for this channel.
	 *
	 * @return true if successful.
	 */
	virtual bool enableDTMF(void);

	/**
	 * Disable dtmf detection for this channel.
	 */
	virtual void disableDTMF(void);

	/**
	 * Check audio properties for file and channel audio processing
	 * based on the driver specific capabilities of this channel
	 * through it's virtual.
	 *
	 * @return error message if audio format unacceptable, NULL if ok.
	 * @param true if for live audio, false if for file only.
	 */
	virtual const char *checkAudio(bool live);

	/**
	 * virtual to filter incoming events.
	 *
	 * @param event being sent to channel.
	 * @return true if accepting event.
	 */
	virtual bool filterPosting(Event *event);

	virtual bool enterCommon(Event *event);
	virtual bool enterInitial(Event *event);
	virtual bool enterFinal(Event *event);
	virtual bool enterIdle(Event *event);
	virtual bool enterReset(Event *event);
	virtual bool enterRelease(Event *event);
	virtual bool enterRinging(Event *event);
	virtual bool enterPickup(Event *event);
	virtual bool enterSeize(Event *event);
	virtual bool enterHangup(Event *event);
	virtual bool enterTone(Event *event);
	virtual bool enterPlay(Event *event);
	virtual bool enterRecord(Event *event);
	virtual bool enterJoin(Event *event);
	virtual bool enterWait(Event *event);
	virtual bool enterDial(Event *event);
	virtual bool enterBusy(Event *event);
	virtual bool enterStandby(Event *event);

	/**
	 * Check dtmf handling and other nessisities for the interpreter
	 * after an event has changed interpreter state.
	 */
	void check(void);

	bool stateInitial(Event *event);
	bool stateFinal(Event *event);
	bool stateIdle(Event *event);
	bool stateReset(Event *event);
	bool stateRelease(Event *event);
	bool stateBusy(Event *event);
	bool stateStandby(Event *event);
	bool stateRinging(Event *event);
	bool statePickup(Event *event);
	bool stateSeize(Event *event);
	bool stateRunning(Event *event);
	bool stateLibexec(Event *event);
	bool stateThreading(Event *event);
	bool stateHangup(Event *event);
	bool stateCollect(Event *event);
	bool stateSleep(Event *event);
	bool stateStart(Event *event);
	bool stateClear(Event *event);
	bool stateInkey(Event *event);
	bool stateInput(Event *event);
	bool stateRead(Event *event);
	bool stateDial(Event *event);
	bool stateTone(Event *event);
	bool statePlay(Event *event);
	bool stateRecord(Event *event);
	bool stateJoin(Event *event);
	bool stateWait(Event *event);
	bool stateConnect(Event *event);

	/**
	 * Direct method to post an event to a channel.
	 *
	 * @return true if event is claimed by channel.
	 * @param event being posted.
	 */
	bool putEvent(Event *event);

public:
	/**
	 * Process interpreter session symbols.
	 *
	 * @param symbol being requested.
	 * @return NULL if not external, else value.
	 */
	const char *getExternal(const char *opt);

protected:
	/**
	 * Return session id for interpreter session command.
	 *
	 * @param id of session request.
	 * @return ccengine object to return for the id.
	 */
	ScriptInterp *getInterp(const char *id);

	/**
	 * Return ccengine symbol page map.  Gives access to globals.
	 *
	 * @param table symbol id.
	 * @return table map to use.
	 */
	ScriptSymbols *getSymbols(const char *id);

	/**
	 * Translator in effect for this session.
	 */
	BayonneTranslator *translator;

	/**
	 * Start a script from idle or ringing.  This may use the
	 * assign statements to find the script name if none is passed.
	 *
	 * @param event passed to kick off the script.
	 * @return script to run or NULL.
	 */
	Name *attachStart(Event *event);

	/**
	 * Used by ccengine.
	 */
	unsigned getId(void);

	/**
	 * Compute a uneque call session id for the current call on the
	 * current session object.  Also a great key for cdr records.
	 */
	void setSid(void);

	/**
	 * Set the session to a new state.
	 */
	void setState(state_t);

	/**
	 * Set the session to the running state, resume interpreter or
	 * libexec.
	 */
	void setRunning(void);

	/**
	 * ccengine.
	 */
	void finalize(void);

	/**
	 * Exit processing for interpreter.
	 *
	 * @return true of exiting.
	 */
	bool exit(void);

	char var_date[12];
	char var_time[12];
	char var_duration[12];
	char var_callid[12];
	char var_tid[14];
	char var_sid[16];
	char var_pid[16];
	char var_rings[4];
	char var_timeslot[8];
	char var_spanid[8];
	char var_bankid[4];
	char var_spantsid[12];
	const char *voicelib;
	char *dtmf_digits;		// dtmf sym space;
	unsigned digit_count, ring_count;

	State state;

public:
	/**
	 * Create a new session
	 *
	 * @param bind to driver.
	 * @param bind to timeslot.
	 * @param bind to span, or NULL if no span associated.
	 */
	BayonneSession(BayonneDriver *d, timeslot_t timeslot, BayonneSpan *span = NULL);

	/**
	 * Destroy a session.
	 */
	virtual ~BayonneSession();

	/**
	 * Initial kickoff event.
	 */
	void initialevent(void);

	/**
	 * Initialine ccengine script environment.
	 */
	void initialize(void);

	/**
	 * Detach interpreter.
	 */
	void detach(void);

	/**
	 * Return driver associated with this session.
	 *
	 * @return driver object.
	 */
	inline BayonneDriver *getDriver(void)
		{return driver;};

	/**
	 * Return time remaining until timer expires.  Commonly used for
	 * msgport scheduling.
	 *
	 * @param time remaining in milliseconds.
	 */
	virtual timeout_t getRemaining(void) = 0;

	/**
	 * Start a timer on the session.  Used extensivily in state
	 * handler code.
	 *
	 * @param start timer for specified milliseconds.
	 */
	virtual void startTimer(timeout_t timer) = 0;

	/**
	 * Stop the timer for the session.  Used extensivily in state
	 * handler code.
	 */
	virtual void stopTimer(void) = 0;

	/**
	 * Set the port hook state.  Mostly for analog devices.
	 *
	 * @param true to set offhook, false onhook.
	 */
	virtual void setOffhook(bool state);

	/**
	 * Handles driver specific stuff when going idle.
	 */
	virtual void makeIdle(void);

	/**
	 * Disconnect notify peer...
	 *
	 * @param event id to pass
	 */
	void part(event_t reason);

	/**
	 * Post an event to the session state engine.
	 *
	 * @return true if event claimed.
	 * @param event being posted.
	 */
	virtual bool postEvent(Event *event);


	bool matchLine(Line *line);

	/**
	 * queue an event through the msgport.
	 *
	 * @param event to queue.
	 */
        virtual void queEvent(Event *event);

	/**
	 * ccengine thread handling.
	 */
	virtual void startThread(void);

	/**
	 * ccengine thread handling.
	 */
	virtual void enterThread(ScriptThread *thr);

	/**
	 * ccengine thread handling.
	 */
	virtual void exitThread(const char *msg);

	/**
	 * Clear/cleanup audio processing for the session.
	 */
	virtual void clrAudio(void);

	/**
	 * Get frame rate used for creating tone generators.
	 */
	virtual timeout_t getToneFraming(void);

	/**
	 * Check script keywords for audio processing.
	 *
	 * @return NULL if ok, else error message.
	 * @param true if for live, else for file processing.
	 */
	const char *getAudio(bool live = true);

	/**
	 * ccengine branch event notification.  Used for menudef
	 * processing.
	 */
	void branching(void);

	/**
	 * Return hook state.
	 *
	 * @return true if offhook.
	 */
	inline bool isOffhook(void)
		{return offhook;};

	/**
	 * Return interface type of this session.
	 *
	 * @return interface type.
	 */
	inline interface_t getInterface(void)
		{return iface;};

	/**
	 * Return bridge type for joins.
	 *
	 * @return bridge type.
	 */
	inline bridge_t getBridge(void)
		{return bridge;};

	/**
	 * Return call type on session.
	 *
	 * @return call type.
	 */
	inline calltype_t getType(void)
		{return type;};

	/**
	 * Return server timeslot this session uses.
	 *
	 * @return server timeslot.
	 */
	inline timeslot_t getSlot(void)
		{return timeslot;};

	/**
	 * Return if the session is currently idle.
	 *
	 * @return true if currently idle.
	 */
	inline bool isIdle(void)
		{return isAvail;};

	/**
	 * Return state of join.
	 *
	 * @return true if currently joined.
	 */
	bool isJoined(void);

	/**
	 * Return parent answer timer, if joining.
	 *
	 * @return timeout for joining.
	 */
	timeout_t getJoinTimer(void);

	/**
	 * Signal notification to script.
	 *
	 * @param signal to send to script engine.
	 * @return true if signal claimed.
	 */
	inline bool signalScript(signal_t sig)
		{return ScriptInterp::signal(sig);};

	virtual bool peerAudio(Audio::Encoded encoded);

	const char *getKeyString(const char *id);
	bool getKeyBool(const char *id);
	long getKeyValue(const char *id);
	timeout_t getSecTimeout(const char *id);
	timeout_t getMSecTimeout(const char *id);
	timeout_t getTimeoutValue(const char *opt = NULL);
	timeout_t getTimeoutKeyword(const char *kw);
	const char *getExitKeyword(const char *def);
	const char *getMenuKeyword(const char *def);

	unsigned getInputCount(const char *digits, unsigned max);

	/**
	 * Compute a new unique transaction id.  These are like pids and
	 * are often used to assure transaction coherence, such as in
	 * the libexec system.
	 *
	 * @return generated integer transaction identifier.
	 */
	uint32 newTid(void);

	/**
	 * Get the current transaction identifier string for the session.
	 *
	 * @return transaction identifier.
	 */
	inline const char *getTid(void)
		{return var_tid;};

	/**
	 * Throw a digit pattern matching event message to the interprer.
	 *
	 * @return true if throw caught.
	 * @param event message.
	 */
	bool digitEvent(const char *evt);

	/**
	 * Get the next pending digit in the DTMF input buffer.
	 *
	 * @return digit.
	 */
	char getDigit(void);

	BayonneAudio audio;
};

/**
 * Bayonne services are used for threaded modules which may be
 * installed at runtime.  These exist to integrate plugins with
 * server managed startup and shutdown.
 *
 * @short threaded server service.
 * @author David Sugar <dyfet@gnutelephony.org>
 */ 
class __EXPORT BayonneService : public Thread                
{          
private:            
        static BayonneService *first;
	static BayonneService *last;
        BayonneService *next;
        friend void startServices(void);        
        friend void stopServices(void);

protected:
        BayonneService(int pri, size_t stack);

        /**
         * Used for stop call interface.             
         */
        virtual void stopService(void);

        /**
         * Used for start call interface.
         */
        virtual void startService(void);

public:
	static void start(void);
	static void stop(void); 
};

} // namespace

#endif
