// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

namespace sipdriver {
using namespace ost;
using namespace std;

Session::Session(timeslot_t ts) :
BayonneSession(&Driver::sip, ts),
TimerPort()
{
#ifndef	WIN32
	if(getppid() > 1)
		logevents = &cout;
#endif
	iface = IF_INET;
	bridge = BR_GATE;
	cid = 0;
	did = 0;
	rtp = NULL;

	update_pos = false;

	buffer = new unsigned char[Driver::sip.info.framesize];
	lbuffer = new Sample[Driver::sip.info.framecount];
	peer_buffer = NULL;
	peer_codec = NULL;

	if(Driver::peer_encoding == Audio::pcm16Mono)
	{
		peer_buffer = new Sample[Driver::sip.info.framecount];
		peer_codec = AudioCodec::getCodec(Driver::sip.info);
	} 

	codec = AudioCodec::getCodec(Driver::sip.info);
}

Session::~Session()
{
}

timeout_t Session::getRemaining(void)
{
	return TimerPort::getTimer();
}

void Session::startTimer(timeout_t timer)
{
	TimerPort::setTimer(timer);
	msgport->update();
}

void Session::stopTimer(void)
{
	TimerPort::endTimer();
	msgport->update();
}

tpport_t Session::getLocalPort(void)
{
	Driver *d = (Driver *)(driver);

	return d->rtp_port + (4 * timeslot);
}	

void Session::startRTP(void)
{
	if(rtp)
	{
		slog.error("%s: rtp already started", logname);
		return;
	}

	rtp = new RTPStream(this);

	if(!rtp->addDestination(remote_address, remote_port))
		slog.error("%s: destination not available");

	rtp->start();
}		

void Session::stopRTP(void)
{
	if(!rtp)
		return;

	rtp->setSource(NULL);
	rtp->setSink(NULL);
	rtp->setTone(NULL);
	delete rtp;
	rtp = NULL;
}

void Session::makeIdle(void)
{
	update_pos = false;
	sipHangup();
	stopRTP();
	BayonneSession::makeIdle();
}

bool Session::enterHangup(Event *event)
{
	if(event->id == TIMER_EXPIRED)
	{
		offhook = true;
		sipHangup();
		stopRTP();
	}

	update_pos = false;
	return false;
}

bool Session::enterRecord(Event *event)
{
	switch(event->id)
	{
	case ENTER_STATE:
		audio.record(state.audio.list[0], state.audio.mode, state.audio.note);
		update_pos = true;
		if(!audio.isOpen())
		{
			slog.error("%s: audio file access error", logname);
                        error("no-files");
                        setRunning();
			return true;
		}
		rtp->setSink(&audio, state.audio.total);
		return false;
	}
	return false;
}

bool Session::enterPlay(Event *event)
{
	switch(event->id)
	{
	case AUDIO_IDLE:
		if(!Driver::sip.audio_timer)
			return false;
		startTimer(Driver::sip.audio_timer);
		return true;
	case ENTER_STATE:
		if(state.audio.mode == Audio::modeReadAny)
			update_pos = false;
		else
			update_pos = true;
		audio.play(state.audio.list, state.audio.mode);
		if(!audio.isOpen())
		{
                	slog.error("%s: audio file access error", logname);
                	error("no-files");
			setRunning();
			return true;
		}
		rtp->setSource(&audio);		
		return false;
	}
	return false;
}

bool Session::enterPickup(Event *event)
{
	if(event->id == ENTER_STATE)
	{
		sipAnswer(200);
		startRTP();
		startTimer(driver->getPickupTimer());
		return true;
	}
	else if(event->id == CALL_ACCEPTED)
	{
		startTimer(Driver::sip.accept_timer);
		return true;
	}

	return false;
}

bool Session::enterTone(Event *event)
{
	if(event->id == ENTER_STATE && audio.tone)
		rtp->setTone(audio.tone);

	return false;
}

bool Session::enterXfer(Event *event)
{
	if(event->id == ENTER_STATE)
	{
		if(!eXosip_transfer_call(did, (char *)state.url.ref))
		{
			setState(STATE_HANGUP);
			return true;
		}
		event->errmsg = "transfer-invalid";
		event->id = ERROR_STATE;
	}
	return false;
}

void Session::clrAudio(void)
{
	if(rtp)
	{
		rtp->setSource(NULL);
		rtp->setTone(NULL);
	}

	if(audio.isOpen() && update_pos)
	{
		audio.getPosition(audio.var_position, 12);
		update_pos = false;
	}
	audio.cleanup();
}

void Session::sipHangup(void)
{
	if(!offhook)
		return;

	eXosip_lock();
	eXosip_terminate_call(cid, did);
	eXosip_unlock();
	offhook = false;
}

void Session::sipAnswer(int code)
{
	char buf[16];

	if(offhook)
		return;

	snprintf(buf, sizeof(buf), "%d", getLocalPort());
	eXosip_lock();
	eXosip_answer_call(did, code, buf);
	eXosip_unlock();
	offhook = true;
}

timeout_t Session::getToneFraming(void)
{
	return Driver::sip.info.framing;
}

const char *Session::checkAudio(bool live)
{
	audio.libext = ".au";

	switch(Driver::sip.info.encoding)
	{
	case alawAudio:
		audio.libext = ".al";
		break;
	case gsmVoice:
		if(!audio.extension)
			audio.extension = ".gsm";
		audio.libext = ".gsm";
		break;
	}

	if(!audio.extension)
		audio.extension = ".au";

	if(audio.encoding == unknownEncoding)
		audio.encoding = Driver::sip.info.encoding;
	
	if(!live)
	{
		if(!audio.framing)
			audio.framing = 10;
		return NULL;
	}

	audio.framing = Driver::sip.info.framing;
	if(audio.encoding != Driver::sip.info.encoding)
		return "unsupported audio format";

	return NULL;
}

void Session::postAudio(Encoded encoded)
{
	if(!peer)
		return;

	if(Driver::peer_encoding == pcm16Mono)
	{
		peer_codec->decode(peer_buffer, encoded, Driver::sip.info.framecount);
		encoded = (Encoded)peer_buffer;
	}

	peer->peerAudio(encoded);
}
	
bool Session::peerAudio(Encoded encoded)
{
	if(Driver::peer_encoding == pcm16Mono)
	{
		codec->encode((Linear)encoded, buffer, Driver::sip.info.framecount);
		encoded = buffer;
	}

	enter();
	if(!peer || !rtp || rtp->source)
	{
		leave();
		return false;
	}

	rtp->peerAudio(encoded);
	leave();
	return true;
}
	
		
} // namespace
