// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/process.h>
#include <cc++/slog.h>
#include "server.h"

namespace server {
using namespace ost;
using namespace std;

PersistProperty typePersist;
TimeslotProperty typeTimeslot;
PositionProperty typePosition;

ThreadLock PersistProperty::lock;
ScriptSymbols PersistProperty::syms;
bool PersistProperty::loaded = false;

PersistProperty::PersistProperty() :
ScriptProperty("persist")
{
}

void PersistProperty::set(const char *data, char *save, unsigned size) 
{
	persist_t *p = (persist_t *)save;
	Symbol *sym = p->sym;
	snprintf(p->cache, PERSIST_CACHE_SIZE, "%s", data);
	lock.writeLock();
	setString(sym->data, PERSIST_CACHE_SIZE, data);
	lock.unlock();
}

void PersistProperty::load(void)
{
	const char *fname = keypaths.getLast("keysave");
	FILE *fp = fopen(fname, "r");
	char buf[128];
	char *cp;
	char *ep;
	Symbol *sym;

	loaded = true;
	if(!fp)
		return;

	for(;;)
	{
		fgets(buf, sizeof(buf), fp);
		if(feof(fp))
			break;
		
		if(!isalnum(buf[0]))
			continue;

		cp = strchr(buf, ' ');
		if(!cp)
			continue;
		*(cp++) = 0;
		ep = strchr(cp, '\r');
		if(!ep)
			ep = strchr(cp, '\n');
		if(ep)
			*ep = 0;
		sym = syms.find(buf, PERSIST_CACHE_SIZE);
		if(!sym)
			continue;
		sym->type = ScriptProperty::symNORMAL;
		setString(sym->data, PERSIST_CACHE_SIZE, cp);
	}

	fclose(fp);
}

void PersistProperty::save(void)
{
	if(!loaded)
		return;

	Symbol **ind = new Symbol *[4096];
	Symbol **mem = ind;

	const char *tmp = keypaths.getLast("keytemp");
	const char *sav = keypaths.getLast("keysave");
	FILE *fp;

	
	ind[0] = NULL;
	remove(tmp);
	fp = fopen(tmp, "w");
	lock.writeLock();
	syms.gather(ind, 4095, "", NULL);
	while(*ind)
	{
		if((*ind)->data[0])
			fprintf(fp, "%s %s\n", (*ind)->id, (*ind)->data);
		++ind;
	}
	lock.unlock();
	fclose(fp);
	rename(tmp, sav);
	delete[] mem;
}

bool PersistProperty::test(const char *id)
{
	lock.readLock();
	Symbol *sym = syms.find(id);
	lock.unlock();
	if(!sym)
		return false;
	if(!sym->data[0])
		return false;
	return true;
}

bool PersistProperty::remap(const char *id, char *save, const char *val)
{
	persist_t *p = (persist_t *)save;

	if(!val)
		val = "";

	lock.writeLock();
	Symbol *sym = syms.find(id, PERSIST_CACHE_SIZE);
	if(!sym)
	{
		lock.unlock();
		return false;
	}

	if(sym->type == ScriptProperty::symINITIAL)
	{
		sym->type = ScriptProperty::symNORMAL;
		sym->data[0] = 0;
		setString(p->cache, PERSIST_CACHE_SIZE, val);		
	}	
	else
		setString(p->cache, PERSIST_CACHE_SIZE, sym->data);
	lock.unlock();
	p->sym = sym;
	return true;
}	

bool PersistProperty::refresh(Symbol *var, const char *ind, const char *val)
{
	ScriptProperty *p = &typePersist;
	persist_t *pv;
	char *dp = var->data + sizeof(p);
	char databuf[128];

	if(!val)
		val = "";

	if(var->type == ScriptProperty::symINITIAL && var->size == getSize())
	{
		memcpy(&var->data, &p, sizeof(p));
		var->type = ScriptProperty::symPROPERTY;
		if(!ind && !strchr(var->id, '.'))
			ind = "sys";
		if(ind)
		{
			snprintf(databuf, sizeof(databuf), "%s.%s", ind, var->id); 
			ind = databuf;
		}
		else
			ind = var->id;	
		return remap(ind, dp, val);
	}

	if(var->type != ScriptProperty::symPROPERTY)
		return false;

        memcpy(&p, &var->data, sizeof(p)); 
	if(p != &typePersist)
		return false;

	pv = (persist_t *)dp;
	lock.readLock();
	setString(pv->cache, PERSIST_CACHE_SIZE, pv->sym->data);
	lock.unlock();
	if(!pv->cache[0])
		setString(pv->cache, PERSIST_CACHE_SIZE, val);
	return true;
}

PositionProperty::PositionProperty() :
ScriptProperty("position")
{
}

unsigned PositionProperty::prec(void)
{
	return 3;
}

char PositionProperty::token(void)
{
	return ':';
}

void PositionProperty::clear(char *data, unsigned size)
{
	if(!size)
		size = 13;

	if(size >= 13)
		strcpy(data, "00:00:00.000");
	else
		*data = 0;
}

void PositionProperty::set(const char *data, char *save, unsigned size)
{
	char msec[4] = "000";
	unsigned pos = 0;
	long secs = getValue(data);

	data = strchr(data, '.');
	if(data)
	{
		while(*(++data) && pos < 3)
			msec[pos++] = *data;
	}
	
	if(size < 13)
	{
		snprintf(save, size, "%d", secs);
		return;
	}

	snprintf(save, size, "%02d:%02d:%02d.%s",
		secs / 3600l, (secs / 60l) % 60l, secs % 60, msec);
}

long PositionProperty::getValue(const char *data)
{
	long offset = 0;

	const char *hp = data;
	const char *sp = strrchr(data, ':');
	const char *mp = strchr(data, ':');
	
	if(mp == sp)
		mp = NULL;

	if(!mp)
		hp = NULL;

	if(!sp)
		sp = data;
	else
		++sp;

	offset = atol(sp);
	if(mp)
		offset += 60l * atol(++mp);
	if(hp)
		offset += 3600l * atol(hp);

	return offset;
}
	
TimeslotProperty::TimeslotProperty() :
ScriptProperty("timeslot")
{
}

void TimeslotProperty::set(const char *data, char *save, unsigned size)
{
	timeslot_t ts = toTimeslot(data);
	
	if(ts == NO_TIMESLOT)
	{
		*save = 0;
		return;
	}

	if(size > 14)
		setString(save, size, data);
	else
		snprintf(save, size, "%d", ts);
}

long TimeslotProperty::getValue(const char *data)
{
	timeslot_t ts = toTimeslot(data);

	if(ts == NO_TIMESLOT)
		return -1l;

	return ts;
}

long TimeslotProperty::adjustValue(long val)
{
	if(val < 0)
		return -1;

	if(val > ts_used)
		return -1;

	return val;
}

} // end namespace
