// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/slog.h>
#include <cc++/process.h>
#include "server.h"

using namespace server;
using namespace ost;
using namespace std;

#ifdef	WIN32

static bool test = true;

static SERVICE_STATUS_HANDLE hStatus = 0;
static SERVICE_STATUS sStatus;
static HANDLE hDown;

static BOOL WINAPI stop(DWORD code)
{
        if(code == CTRL_LOGOFF_EVENT)
                return TRUE;

	BayonneService::stop();
	BayonneDriver::stop();
	ScriptBinder::shutdown();

	if(code)
		Bayonne::errlog("failed", "server exiting; reason=%d", code);
	else
		Bayonne::errlog("notice", "server exiting; normal termination");

	return TRUE;
}

#else
static int mainpid;

static RETSIGTYPE stop(int signo)
{
	if(signo > 0 && getpid() != mainpid)
	{
		kill(mainpid, signo);
		return;
	}

        signal(SIGINT, SIG_IGN);
        signal(SIGABRT, SIG_IGN);
        signal(SIGTERM, SIG_IGN);
        signal(SIGQUIT, SIG_IGN);

	BayonneService::stop();
	BayonneDriver::stop();
	ScriptBinder::shutdown();

	purgedir(keypaths.getLast("tmp"));
	purgedir(keypaths.getLast("tmpfs"));

        if(signo == -2)
                Bayonne::errlog("failed", "server exiting; no timeslots allocated");
        else if(signo)
                Bayonne::errlog("failed", "server exiting; reason=%d", signo);
        else
                Bayonne::errlog("notice", "server exiting; normal termination");

#ifdef	HAVE_LIBEXEC
	BayonneSysexec::cleanup();
#endif

	Thread::sleep(100);

	::exit(signo);
}
#endif

static void version(void)
{
        cout << VERSION << endl;
        exit(0);
}

static void loading(void)
{
	char buffer[256];
	char lang[32];
	char ** keys;
	char *plist[65];
	const char *cp, *proto;
	char *sp, *tok;
	unsigned pcount = keyswitch.getIndex(plist, sizeof(plist) / sizeof(char **));
	unsigned count = keyoptions.getCount("modules");
	BayonneDriver *driver;
	BayonneTranslator *translator = NULL;
	static char vlib[64];
	bool pflag = true;
#ifdef	HAVE_LIBEXEC
	size_t bs = 0;
	int pri = 0;
#endif

	if(!Process::getEnv("SERVER_SYSEXEC"))
		Process::setEnv("SERVER_SYSEXEC", keypaths.getLast("scripts"), true);
	
	cp = keypaths.getLast("btsexec");
	if(cp && *cp)
		Process::setEnv("SHELL_BTSEXEC", cp, true);

	Process::setEnv("SERVER_PREFIX", keypaths.getLast("datafiles"), true);
       	Process::setEnv("SERVER_PROMPTS", keypaths.getLast("prompts"), true);
       	Process::setEnv("SERVER_SCRIPTS", keypaths.getLast("scripts"), true);
       	Process::setEnv("SERVER_LIBEXEC", keypaths.getLast("libexec"), true);     
       	Process::setEnv("SERVER_SOFTWARE", "bayonne", true);
       	Process::setEnv("SERVER_VERSION", VERSION, true);
       	Process::setEnv("SERVER_TOKEN", " ", true);

#ifdef	HAVE_LIBEXEC
	cp = keyengine.getLast("gateways");
	if(cp)
		pri = atoi(cp);

	cp = keyengine.getLast("buffers");
	if(cp)
		bs = atoi(cp) * 1024;

	getcwd(buffer, sizeof(buffer));

	BayonneSysexec::allocate(keypaths.getLast("libexec"), bs, pri,
		keypaths.getLast("modexec"));
#endif
        cp = keyserver.getLast("language");
        if(cp && (strlen(cp) == 2 || cp[2] == '_'))  
		BayonneTranslator::loadTranslator(cp);  

        cp = keyserver.getLast("voice");
        if(cp && strchr(cp, '/'))
        {
                setString(lang, sizeof(lang), cp);
                sp = strchr(lang, '/');
                if(sp)
                        *sp = 0;
                if(strlen(lang) == 2 || sp[2] == '_')
                        translator = BayonneTranslator::loadTranslator(lang);

	}

	if(cp && translator)
	{
		snprintf(buffer, sizeof(buffer), "%s/%s", keypaths.getLast("prompts"), cp);
		if(isDir(buffer))
		{
			slog.debug("using %s as default voice library", cp);
			Bayonne::init_translator = translator;
			Bayonne::init_voicelib = cp;
		}
		else if(cp[2] == '_')
		{
			vlib[0] = cp[0];
			vlib[1] = cp[1];
			cp = strchr(cp, '/');
			if(!cp)
				cp = "/default";
			snprintf(vlib + 2, sizeof(vlib) - 2, "%s", cp);
			snprintf(buffer, sizeof(buffer), "%s/%s",
				keypaths.getLast("prompts"), vlib);
			if(isDir(buffer))
			{
                        	slog.debug("using %s as default voice library", cp);
                        	Bayonne::init_translator = translator;
                        	Bayonne::init_voicelib = vlib; 
			}
		}
        } 

	
	cp = keyoptions.getLast("driver");
	if(cp && *cp)
	{
		driver = BayonneDriver::loadDriver(cp);

		if(!driver)
			stop(-1);
	}
	else
	{
		proto = keyswitch.getLast("protocol");
		if(proto && !strnicmp(proto, "no", 2))
		{
			pflag = false;
			proto = NULL;
		}

		if(proto && !stricmp(proto, "yes"))
			proto = NULL;

                if(proto && !stricmp(proto, "use"))
                        proto = NULL; 

                if(proto && !stricmp(proto, "any"))
                        proto = NULL; 

                if(proto && !stricmp(proto, "all"))
                        proto = NULL; 

		if(proto && !*proto)
			proto = NULL;

		cp = keyswitch.getLast("trunking");
		if(cp && *cp && strnicmp(cp, "no", 2))
			BayonneDriver::loadTrunking(cp);
		keys = plist;
		if(pflag)
			pflag = BayonneDriver::useProtocols();

		while(*keys && BayonneDriver::useProtocols())
		{
			cp = keyswitch.getLast(*keys);
			if(!cp)
				cp = "";
			pcount = atoi(cp);
			cp = *keys;
			if(proto && !stricmp(cp, proto))
				pcount = 0;
			else if(!stricmp(cp, "timeslots"))
				pcount = 0;
			else if(!stricmp(cp, "framing"))
				pcount = 0;
			else if(!stricmp(cp, "trunking"))
				pcount = 0;
			else if(!stricmp(cp, "protocol"))
				pcount = 0;
			else if(!stricmp(cp, "encoding"))
				pcount = 0;
			++keys;
			if(!pcount)
				continue;
			BayonneDriver::loadProtocol(cp, pcount);
		}
		if(proto)
			BayonneDriver::loadProtocol(proto, 0);
	}	

	sp = (char *)keyserver.getLast("monitoring");
	if(sp && !strnicmp(sp, "no", 2))
		sp = NULL;

	if(sp && !*sp)
		sp = NULL;

	if(sp)
		cp = strtok_r(sp, " ,;:\t", &tok);
	else
		cp = NULL;

	while(cp)
	{
		Bayonne::loadMonitor(cp);
		cp = strtok_r(NULL, " ,;:\t", &tok);
	}

	if(!count)
		goto start;

	keys = (char **)keyoptions.getList("modules");

	while(*keys)
		Bayonne::loadPlugin(*keys++);

start:
	if(!BayonneDriver::getPrimary())
	{
		slog.critical("bayonne: no drivers loaded");
		stop(-1);
	}

	BayonneDriver::start();
	count = Bayonne::getTimeslotsUsed();
	if(!count)
		stop(-2);

	count = Bayonne::getTimeslotsUsed();

#ifdef	HAVE_LIBEXEC
	BayonneSysexec::startup();
#endif
// reload
	loadBinder(keyserver.getLast("binding"));
	Bayonne::reload();
	if(!Bayonne::compile_count)
	{
		Bayonne::errlog("critical", "no applications defined");
		stop(-1);
	}
	else if(Bayonne::provision_ripple)
		Bayonne::errlog("notice", "%d configurations compiled", Bayonne::compile_count);
	else
		Bayonne::errlog("notice", "%d applications compiled", Bayonne::compile_count); 

	BayonneService::start();
	Bayonne::errlog("notice", "driver(s) started; %d timeslot(s) used", count);
#ifndef	WIN32
        Process::setPosixSignal(SIGPIPE, SIG_IGN);
        Process::setPosixSignal(SIGINT, &stop);
        Process::setPosixSignal(SIGHUP, &stop);
        Process::setPosixSignal(SIGTERM, &stop);
        Process::setPosixSignal(SIGABRT, &stop);
#endif
}

static void logging(void)
{
	slog.open("bayonne", Slog::classDaemon);
	const char *level = keyserver.getLast("logging");

	if(!stricmp(level, "notice"))
		slog.level(Slog::levelNotice);
	else if(!stricmp(level, "info"))
		slog.level(Slog::levelInfo);
	else if(!stricmp(level, "error"))
		slog.level(Slog::levelError);
	else if(!stricmp(level, "debug"))
		slog.level(Slog::levelDebug);
}

static void checking(void)
{
	keyserver.setValue("logging", "debug");
	logging();
	if(!Bayonne::getTimeslotCount())
		Bayonne::allocate(1, dynamic_cast<ScriptCommand *>(&runtime));
	Bayonne::reload();
        if(!Bayonne::compile_count)
        {
                Bayonne::errlog("critical", "no applications defined");
                stop(-1);
        }                                                                     
	stop(0);
}

#ifdef	WIN32

static void control(DWORD request)
{
	switch(request)
	{
	case SERVICE_CONTROL_SHUTDOWN:
	case SERVICE_CONTROL_STOP:
		BayonneService::stop();
		BayonneDriver::stop();
		Bayonne::errlog("notice", "server exiting; service shutdown");
		sStatus.dwWin32ExitCode = 0;
		sStatus.dwCurrentState = SERVICE_STOPPED;
		SetServiceStatus(hStatus, &sStatus);
		SetEvent(hDown);
		return;
	}
	SetServiceStatus(hStatus, &sStatus);
}

static VOID service(DWORD argc, LPTSTR *argv)
{
	hStatus = RegisterServiceCtrlHandler("Bayonne", (LPHANDLER_FUNCTION)control);
	hDown = CreateEvent(0, TRUE, FALSE, 0);

	if(!hStatus || !hDown)
		return;

	sStatus.dwServiceType = SERVICE_WIN32;
	sStatus.dwCurrentState = SERVICE_START_PENDING;
	sStatus.dwControlsAccepted = 0;
	sStatus.dwWin32ExitCode = 0;
	sStatus.dwServiceSpecificExitCode = 0;
	sStatus.dwCheckPoint = 0;
	sStatus.dwWaitHint = 500;

	SetServiceStatus(hStatus, &sStatus);

	sStatus.dwCurrentState = SERVICE_RUNNING;
	sStatus.dwControlsAccepted = SERVICE_ACCEPT_STOP | SERVICE_ACCEPT_SHUTDOWN;
	sStatus.dwWin32ExitCode = 0;
	sStatus.dwWaitHint = 0;
	SetServiceStatus(hStatus, &sStatus);

	WaitForSingleObject(hDown, INFINITE);
	return;
};

static SERVICE_TABLE_ENTRY services[] =
{
	{"Bayonne", (LPSERVICE_MAIN_FUNCTION)service},
	{NULL, NULL}
};

#endif

static void banner(void)
{
	cout << "SERVER VERSION " << VERSION << "; ";
	cout << keyserver.getLast("node") << " ";
	cout << keyserver.getLast("platform") << " ";
	cout << keyserver.getLast("cpu") << " ";
	cout << endl;
}

int main(int argc, char **argv)
{
	const char *argv0;
	const char *cp;

	Script::use_macros = true;
	Script::use_prefix = true;

#ifdef	WIN32
	argv0 = "bayonne";
#else
	argv0 = strrchr(argv[0], '/');

	if(argv0)
		++argv0;
	else
		argv0 = argv[0];
#endif

	keyserver.setValue("argv0", argv0);

	if(argc == 2)
	{
		cp = argv[1];
		if(!strncmp(cp, "--", 2))
			++cp;
		
		if(!stricmp(cp, "-version"))
			version();
		else if(!stricmp(cp, "-check"))
		{
                	loadConfig();
                	liveConfig();
                	runtime.setValue("scripts", keypaths.getLast("scripts"));
                	if(keypaths.getLast("addons"))
                    		runtime.setValue("addons", keypaths.getLast("addons"));    
                	runtime.setValue("include", keypaths.getLast("macros")); 
			loadBinder(keyserver.getLast("binding"));
			checking();
		}
#ifdef	HAVE_TESTING
        	else if(!stricmp(cp, "-dump-install"))
        	{
                	loadConfig();
                	liveConfig();
                	banner();
                	dumpConfig(keypaths);
                	exit(0); 
        	} 
       		else if(!stricmp(cp, "-dump-testing"))
        	{
                	testConfig(*argv);
                	liveConfig();    
                	banner();
                	dumpConfig(keypaths);
                	exit(0);
        	}
#ifdef  WIN32
        	else if(!stricmp(cp, "-dump-registry"))
        	{
                	// we assume these objects are registry initialized
                	// and dump them without further processing.
                	cout << "[paths]" << endl;   
                	dumpConfig(keypaths);
			cout << "[switch]" << endl;
			dumpConfig(keyswitch);
                	cout << "[server]" << endl;
                	dumpConfig(keyserver);
                	cout << "[engine]" << endl;
                	dumpConfig(keyengine);
                	exit(0);
        	}
#endif  
#endif
	}

#ifdef	WIN32
	SetConsoleTitle("Bayonne");
        SetConsoleCtrlHandler((PHANDLER_ROUTINE)stop, TRUE);
#else
	mainpid = getpid();
	Process::setPosixSignal(SIGPIPE, SIG_IGN);
        Process::setPosixSignal(SIGINT, &stop);
        Process::setPosixSignal(SIGHUP, &stop);
        Process::setPosixSignal(SIGTERM, &stop);
        Process::setPosixSignal(SIGABRT, &stop);

	if(!getuid())
		Bayonne::provision_system = true;
	else if(Process::getEnv("HOME"))
		Bayonne::provision_user = true;

#endif

	Bayonne::provision_daemon = true;
	loadConfig();
	parseConfig(argv);
	liveConfig();
	if(Bayonne::provision_check)
		checking();

	if(stricmp(argv0, "bayonne"))
	{
		cp = keygateway.getLast("timeslots");
		if(cp && *cp)
			keyswitch.setValue("timeslots", cp);

		cp = keygateway.getLast("trunking");
		if(cp && *cp)
			keyswitch.setValue("trunking", cp);
	
		cp = keygateway.getLast("binding");
		if(cp && *cp)
			keyserver.setValue("binding", cp);
	}
		
	Process::setScheduler(keyengine.getLast("scheduler"));
        Process::setPriority(atoi(keyengine.getLast("priority")));   
#ifndef WIN32
        if(!getuid())
        {
                if(!Process::setUser(keyserver.getLast("user")))
                {
                        Bayonne::errlog("fatal", "%s: cannot set user",
                                keyserver.getLast("user"));
                        stop(-1);
                }
        }
	if(Bayonne::provision_daemon)
		Process::detach();

        mainpid = getpid();
#endif
	logging();
        Bayonne::errlog("notice", "starting %s on %s %s; timeslots=%s",
                VERSION,
                keyserver.getLast("cpu"), keyserver.getLast("platform"),
                keyswitch.getLast("timeslots"));   

	loading();

	Runtime::process();
	return 0;
}
