// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"


#ifdef  WIN32
#define CONFIG_FILES    "C:/Program Files/GNU Telephony/Bayonne Config"
#define	strdup	ost::strdup
#else
#include <private.h>
#endif

namespace sipdriver {
using namespace ost;
using namespace std;

static Keydata::Define driver[] = {
	{"type", "proto"},
	{"proto", "sip"},
	{"driver", "exosip2"},
	{"stack", "0"},
	{"events", "128"},
	{"priority", "0"},
	{"encoding", "mulaw"},
	{"framing", "20"},
	{"timer", "50"},
	{"payload", "0"},
	{"dtmf", "101"},
	{"inband", "false"},
	{"silence", "500"},
	{"duration", "10m"},
	{"invite", "60000"},
	{"pickup", "2000"},
	{"hangup", "250"},
	{"accept", "320"},
	{"audio", "60"},
	{"jitter", "2"},
	{"filler", "false"},
	{"urlprefix", "sip:"},
	{"peering", "friend"},	// default registration type
#ifdef	WIN32		// use localhost default if not in conf for w32
	{"interface", "127.0.0.1:5070"},
#endif
        {NULL, NULL}};

#ifdef	WIN32
#define	KEYS	"/bayonne/sip"
#else
#define	KEYS	"/bayonne/driver/sip"
#endif

Driver Driver::sip;

bool Registry::isActive(void)
{
	if(!proxy)
		return active;

	if(!active)
		return false;

	if(getTimer() > 0)
		return true;

	return false;
}

Driver::Driver() :
BayonneDriver(driver, KEYS, "sip", false),
Thread(atoi(getLast("priority")), atoi(getLast("stack")) * 1024)
{
	const char *cp, *p;
	char buf[1024];
	AudioCodec *codec;

#ifdef	WIN32
	const char *env = Process::getEnv("DRIVERCONFIG");

	if(env)
		loadFile(env, "sip");
#else
	if(Bayonne::provision_system)
		load("/bayonne/provision/sip");
	else if(Bayonne::provision_user)
		load("~bayonne/sip");
#endif
	sip_port = 5060;


	cp = getLast("ack");
	if(!cp)
		cp = getLast("acktimer");

	if(!cp)
		cp = getLast("pickup");

	if(cp)
		pickup_timer = atol(cp);

	if(pickup_timer < 10)
		pickup_timer *= 1000;

	cp = getLast("hangup");
	if(cp)
		hangup_timer = atol(cp);

	cp = getLast("bind");
	if(!cp)
		cp = getLast("interface");

	if(!cp)
	{
		gethostname(buf, sizeof(buf) - 1);
		InetAddress host(buf);
		snprintf(buf, sizeof(buf), "%s:5060", inet_ntoa(host.getAddress()));
		setValue("interface", buf);
		cp = getLast("interface");
	}
	else
		setValue("interface", cp);

	p = strchr(cp, ':');
	if(p)
		sip_port = atoi(++p);
	else
		sip_port = atoi(cp);		

	rtp_port = sip_port + 2;

	cp = getLast("rtp");
	if(cp)
		rtp_port = atoi(cp);

	silence = atoi(getLast("silence"));
	cp = getLast("inband");

	dtmf_inband = true;
	switch(*cp)
	{
	case 'n':
	case 'N':
	case 'f':
	case 'F':
	case '0':
	case 0:
		dtmf_inband = false;
		break;
	}

        cp = getLast("filler");

        data_filler = true;
        switch(*cp)
        {
        case 'n':
        case 'N':
        case 'f':
        case 'F':
        case '0':
        case 0:
                data_filler = false;
                break;
        }


	audio_timer = atol(getLast("audio"));
	hangup_timer -= audio_timer;
	accept_timer = atol(getLast("accept"));
	jitter = atoi(getLast("jitter"));
	data_negotiate = atoi(getLast("payload"));
	dtmf_negotiate = 0;

	cp = getLast("dtmf");
	if(cp)
		dtmf_negotiate = atoi(cp);

	memset(&info, 0, sizeof(info));
	cp = getLast("encoding");
	info.encoding = getEncoding(cp);
	info.rate = rate8khz;
	cp = getLast("rate");
	if(cp)
		info.rate = atol(cp);
	if(peer_encoding != Audio::unknownEncoding)
	{
		if(peer_encoding != Audio::pcm16Mono)	
			info.encoding = peer_encoding;
		info.setFraming(peer_framing);
	}
	else
		info.setFraming(atol(getLast("framing")));
	registry = false;
	silent_frame = new Sample[info.framecount];
	memset(silent_frame, 0, info.framecount * 2);
	silent_encoded = new unsigned char[info.framesize];
	if(info.encoding == Audio::pcm16Mono)
		memset(silent_encoded, 0, info.framesize);
	else
	{
		codec = AudioCodec::getCodec(info);
		codec->encode(silent_frame, silent_encoded, info.framecount);
		AudioCodec::endCodec(codec);
	}	

	server->setValue("sip.conf", CONFIG_FILES "/sip.conf");
	server->setValue("uri.conf", CONFIG_FILES "/uri.conf");
}

bool Driver::getAuthentication(Session *s)
{
	char buf[256];
	ScriptImage *img = s->getImage();
	const char *cp = s->getSymbol("session.registry");
	char *p;
	Registry *reg;
	bool rtn = false;

	if(!cp)
		goto done;

	reg = (Registry *)img->getPointer(buf);
	if(!reg)
		goto done;

	if(!reg->userid)
		goto done;

	setString(buf, sizeof(buf), reg->userid);
	p = strchr(buf, '@');
	if(p)
		*(p++) = 0;

	eXosip_lock();
	if(eXosip_add_authentication_info(buf, buf, reg->secret, NULL, reg->realm))
		slog.error("sip: authentication failed; host=%s", reg->proxy);
	else
	{
		rtn = true;
	        slog.debug("sip: authenticating to %s using %s", reg->proxy, reg->userid);
	}
	eXosip_unlock();
done:
	return rtn;
}

void Driver::setAuthentication(Registry *reg)
{
	char uname[65];
	char *p;

	setString(uname, sizeof(uname), reg->userid);
	p = strchr(uname, '@');
	if(p)
		*p = 0;

	eXosip_lock();
	int rtn = eXosip_add_authentication_info(uname, uname, reg->secret, NULL, reg->realm);
	eXosip_unlock();
}
		
void Driver::startDriver(void)
{
	char cbuf[65];
	unsigned tries = 0;
	char *p;
	const char *cp;
	unsigned max_count = ts_count;
	cp = server->getLast("slots.sip");
	int rtn;

	exiting = false;
	timeslot = ts_used;
	msgport = new BayonneMsgport(this);
	
	if(cp)
		max_count = atoi(cp);

	while(ts_used < ts_count && max_count--)
		new Session(ts_used);

	count = ts_used - timeslot;
	if(!count)
		return;

	eXosip_init();

	for(;;)
	{
		cp = getLast("transport");
		if(!cp)
			cp = "udp";

		if(!stricmp(cp, "tcp"))
			rtn = eXosip_listen_addr(IPPROTO_TCP, NULL, sip_port, AF_INET, 0);
		else if(!stricmp(cp, "tls"))
			rtn = eXosip_listen_addr(IPPROTO_TCP, NULL, sip_port, AF_INET, 1);
		else
			rtn = eXosip_listen_addr(IPPROTO_UDP, NULL, sip_port, AF_INET, 0);

		if(!rtn)
			break;

		slog.error("sip: failed to bind %d", sip_port);
		if(tries++ > 3)
		{
			eXosip_quit();
			return;
		}
		sip_port += 10;
		if(!getLast("rtp"))
			rtp_port += 10;
	}

	cp = getLast("interface");
	if(!cp)
		cp = "all";

	setString(cbuf, sizeof(cbuf) - 8, getLast("interface"));
	p = strrchr(cbuf, ':');
	snprintf(p + 1, 8, "%d", sip_port);
	setValue("interface", cbuf);
	*p = 0;
	setValue("localip", cbuf);
	
        osip_trace_initialize_syslog(TRACE_LEVEL0, "bayonne");
/*        osip_trace_enable_level(TRACE_LEVEL1);
        osip_trace_enable_level(TRACE_LEVEL2);
        osip_trace_enable_level(TRACE_LEVEL3);
        osip_trace_enable_level(TRACE_LEVEL4);
        osip_trace_enable_level(TRACE_LEVEL5);
        osip_trace_enable_level(TRACE_LEVEL6);
        osip_trace_enable_level(TRACE_LEVEL7);
*/
	switch(info.encoding)
	{
	case mulawAudio:
		if(!data_negotiate)
			setValue("payload", "0");
		snprintf(cbuf, sizeof(cbuf), "%d PCMU/%d", 
			data_negotiate, info.rate);
		break;
	case alawAudio:
		if(!data_negotiate)
		{
			data_negotiate = 8;
			setValue("payload", "8");
		}
		snprintf(cbuf, sizeof(cbuf), "%d PCMA/%d",
			data_negotiate, info.rate);
		break;
	case pcm16Mono:
		snprintf(cbuf, sizeof(cbuf), "%d L16_1CH/%d",
			data_negotiate, info.rate);
		break;
	case pcm16Stereo:
		snprintf(cbuf, sizeof(cbuf), "%d L16_2CH/%d",
			data_negotiate, info.rate);
		break;
	case gsmVoice:
		if(!data_negotiate)
		{
			data_negotiate = 3;
			setValue("payload", "3");
		}
		snprintf(cbuf, sizeof(cbuf), "%d GSM/%d",
			data_negotiate, info.rate);
		break;
	default:
		slog.error("sip: unsupported sdp audio encoding");
		return;
	}

	slog.debug("sip: adding sdp encoding %s", cbuf);
	info.annotation = strdup(cbuf);
	eXosip_set_user_agent("GNU Bayonne");

	slog.debug("sip: bound to port %d", sip_port);
	msgport->start();
	Thread::start();
	Thread::sleep(100);
	BayonneDriver::startDriver();	
}

void Driver::stopDriver(void)
{
	if(running)
	{
		exiting = true;
		__eXosip_wakeup_event();
		terminate();
		eXosip_quit();
		BayonneDriver::stopDriver();
	}
}

const char *Driver::assignScript(ScriptImage *img, Line *line)
{
	Name *scr = img->getCurrent();
	char buffer[1024];
	char cbuf[65];
	char uname[65];
	char rname[76];
	unsigned idx = 0;
	const char *cp;
	char *ep, *contact;
	Registry *reg;
	const char *host, *port;

	for(;;)
	{
		cp = line->args[idx++];
		if(!cp && idx > 1)
			break;

		Registry *reg = (Registry *)img->getMemory(sizeof(Registry));
		reg->proxy = NULL;
		reg->active = true;
		reg->protocol = "sip";
		reg->iface = "none";
		reg->uri = cp;
		reg->hostid = "none";
		reg->userid = "none";
		reg->realm = NULL;
		reg->type = "none";
		reg->dtmf = NULL;
		reg->scr = scr;
		reg->secret = NULL;
		reg->line = line;
		reg->duration = 0;		

		setString(cbuf, sizeof(cbuf), scr->name);
		ep = strchr(cbuf, ':');
		if(ep)
		{
			*(ep++) = 0;
			*ep = '.';
		}
		else
			ep = "";

                host = strchr(cp, '@');
                if(host)
                {
                        ++host;
                        ep = strrchr((char *)host, ':');
                        if(ep)
                                *ep = 0;
                        reg->address = host;
                        setString(uname, sizeof(uname), cp);
                        ep = strchr(uname, '@');
                        if(ep)
                                *ep = 0;
                }
                else if(cp)
		{
			reg->address = NULL;
			setString(uname, sizeof(uname), cp);
		}
		else
                {
                        reg->address = NULL;
                        snprintf(uname, sizeof(uname), "%s%s", cbuf, ep);
                }

		snprintf(rname, sizeof(rname), "uri.%s", uname);
                host = getLast("interface");
                if(reg->address)
                {
                        port = strrchr(host, ':');
                        host = reg->address;
                }
                else
                        port = "";

                snprintf(buffer, sizeof(buffer), "sip:%s@%s%s",
                        uname, host, port);

		ep = strrchr(buffer, ':');
		if(ep && !stricmp(ep, ":5060"))
			*ep = 0;	
		contact = (char *)img->getMemory(strlen(buffer) + 1);
		strcpy(contact, buffer);
		reg->contact = (const char *)contact;
		if(!reg->uri)
			reg->uri = reg->contact;
                reg->localid = (char *)img->getMemory(strlen(uname) + 1);
                strcpy((char *)reg->localid, uname);
		slog.debug("sip: assigning %s to %s", scr->name, buffer);
		img->setPointer(rname, reg);
		if(!cp)
			break;
	}
	return "";
}

const char *Driver::registerScript(ScriptImage *img, Line *line)
{
	Name *scr = img->getCurrent();
	Registry *reg = (Registry *)img->getMemory(sizeof(Registry));
        const char *cp = ScriptChecks::findKeyword(line, "timeout");
	const char *lcp;
	char buf[1024];
	char uname[65];
	unsigned len, pos;
	char *p;
	char *uri;
	long mult = 60000;
	osip_message_t *msg = NULL; 

	if(!ScriptChecks::useKeywords(line, "=uri=timeout=server=proxy=userid=secret=type=realm=dtmf=public"))
		return "invalid keywords for sip, use uri, timeout, proxy, userid, secret, port";

	reg->active = false;
	reg->protocol = "sip";
	reg->iface = getLast("interface");
	reg->uri = ScriptChecks::findKeyword(line, "uri");
	reg->address = ScriptChecks::findKeyword(line, "public");
	reg->hostid = ScriptChecks::findKeyword(line, "proxy");
	reg->userid = ScriptChecks::findKeyword(line, "userid");
	reg->secret = ScriptChecks::findKeyword(line, "secret");
	reg->realm = ScriptChecks::findKeyword(line, "realm");
	reg->type = ScriptChecks::findKeyword(line, "type");
	reg->dtmf = ScriptChecks::findKeyword(line, "dtmf");

	if(!reg->dtmf)
		reg->dtmf = getLast("dtmf");

	if(!reg->dtmf)
		reg->dtmf = "auto";

	if(!reg->type)
		reg->type = getLast("peering");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = ScriptChecks::findKeyword(line, "server");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = getLast("proxy");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = server->getLast("proxy");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = server->getLast("server");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = "localhost";

	if(!reg->userid)
		reg->userid = getLast("userid");

	if(!reg->userid)
		reg->userid = server->getLast("userid");

	if(!*reg->userid)
		reg->userid = NULL;

	if(!reg->secret)
		reg->secret = getLast("secret");

	if(!reg->secret)
		reg->secret = server->getLast("secret");

	if(!*reg->secret)
		reg->secret = NULL;	

	if(!cp)
		cp = getLast("duration");

	lcp = cp + strlen(cp) - 1;
	switch(*lcp)
	{
	case 'm':
	case 'M':
		mult = 60000;
		break;
	case 's':
	case 'S':
		mult = 1000;
		break;
	case 'h':
	case 'H':
		mult = 360000;
		break;
	}

	reg->duration = mult * atol(cp);
	reg->line = line;
	reg->scr = scr;

	// min 3 minutes, as automatic resend is minute prior

	if(reg->duration < 180000)
		reg->duration = 180000;

        cp = scr->name;
        if(!strnicmp(cp, "sip::", 5))
                cp += 5; 

	snprintf(buf, sizeof(buf) - 2, "sip:%s", cp);
	p = strchr(buf + 4, ':');
	if(p)
	{
		*(p++) = '.';
		cp = strrchr(scr->name, ':');
		strcpy(p, cp + 1);
	}
	
	len = strlen(buf) - 3;
	reg->localid = (const char *)img->getMemory(len);
	strcpy((char *)reg->localid, buf + 4);
        if(!reg->userid)
                reg->userid = reg->localid; 

	len = strlen(buf);
	buf[len++] = '@';
	buf[len] = 0;
	pos = len;

        if(reg->address)
        {
                cp = strrchr(reg->iface, ':');
                if(!cp)
                        cp = "";
                snprintf(buf + pos, sizeof(buf) - pos, "%s%s",
                        reg->address, cp);
        }
        else
		setString(buf + pos, sizeof(buf) - pos, reg->iface);

	len = strlen(buf) + 1;
	reg->contact = (const char *)img->getMemory(len);
	strcpy((char *)reg->contact, buf);

        setString(buf + pos, sizeof(buf) - pos, reg->hostid);
	p = strrchr(buf + 5, ':');
	if(p)
		*p = 0;
	len = 8 + strlen(reg->userid) + strlen(reg->hostid);
        uri = (char *)img->getMemory(len);
	snprintf(uri, len, "sip:%s@%s", reg->userid, reg->hostid);

	snprintf(buf, sizeof(buf), "sip:%s", reg->hostid);

	len = strlen(buf) + 1;
	reg->proxy = (const char *)img->getMemory(len);
	strcpy((char *)reg->proxy, buf);

	if(!reg->uri)
		reg->uri = uri;

	slog.debug("register %s as %s on %s", reg->localid, reg->userid, reg->proxy);

	if(!registry)
		registry = true;

	eXosip_lock();
	reg->regid = eXosip_register_build_initial_register((char *)reg->uri, (char *)reg->proxy, (char *)reg->contact, reg->duration / 1000, &msg);
	eXosip_unlock();

	if(reg->regid < 0)
		return "cannot create sip registration";

	if(reg->secret)
		setAuthentication(reg);

//	osip_message_set_supported(msg, "path");   
	osip_message_set_header(msg, "Event", "Registration");
	osip_message_set_header(msg, "Allow-Events", "presence");

/*	cp = getLast("transport");
	if(cp && !stricmp(cp, "tcp"))
		eXosip_transport_set(msg, "TCP"); 
	else if(cp && !stricmp(cp, "tls"))
		eXosip_transport_set(msg, "TLS");


	reg->route = ScriptChecks::findKeyword(line, "route");
	if(reg->route && reg->route[0])
	{
		char *hdr = osip_strdup(reg->route);
		osip_message_set_multiple_header(msg, hdr, reg->route);
		osip_free(hdr);
	}
*/
	eXosip_lock();
	eXosip_register_send_register(reg->regid, msg);
	eXosip_unlock();
	
	if(!strnicmp(reg->proxy, "sip:", 4))
		snprintf(buf, sizeof(buf), "sip.%s", reg->proxy + 4);
	else
		snprintf(buf, sizeof(buf), "sip.%s", reg->proxy);
	p = strrchr(buf, ':');
	if(p && !stricmp(p, ":5060"))
		*p = 0;
	img->setPointer(buf, reg);

	snprintf(buf, sizeof(buf), "sipreg.%d", reg->regid);
	img->setPointer(buf, reg);
	snprintf(buf, sizeof(buf), "uri.%s", reg->localid);
	img->setPointer(buf, reg);
	img->addRegistration(line);
	line->scr.registry = reg;

	return "";
}

Session *Driver::getCall(int callid)
{
	timeslot_t ts = 0;
	Session *session;
	BayonneSession *bs;

	while(ts < count)
	{
		bs = getTimeslot(ts++);
		if(!bs)
			continue;
		session = (Session *)bs;
		if(session->cid == callid && session->offhook)
			return session;
	}
	return NULL;
}

void Driver::run(void)
{
        eXosip_event_t *sevent;
        Event event;
	char buf[1024];
	char cbuf[256];
	Registry *reg, *peer, *proxy;
	Name *scr;
	ScriptImage *img;
	const char *cp, *lcp;
	const char *caller = NULL;
	const char *dialed = NULL;
	const char *display = NULL;
	BayonneSession *s;
	Session *session;
	char *k, *v, *l, *p, *line;
	osip_uri_t *url;
	osip_from_t *from;
	osip_to_t *to;
	char digit[10];
	char duration[10];
	osip_body_t *mbody;
	const char *auth = "anon";
	timeout_t event_timer = atoi(getLast("timer"));
	osip_message_t *msg = NULL;
	char *remote_uri = NULL;
	char *local_uri = NULL;
	sdp_message_t *remote_sdp, *local_sdp;
	sdp_connection_t *conn;
	osip_content_type_t *ct;
	sdp_media_t *remote_media, *local_media, *audio_media;
	char *tmp, *audio_payload;
	const char *data_rtpmap;
	int pos, mpos, media_payload, rtn;
	char *sdp;
	char sdp_payload[128];
	bool reg_inband;
	uint8 reg_payload;

	Bayonne::waitLoaded();

	for(;;)
        {
		sevent = eXosip_event_wait(0, event_timer);
		eXosip_lock();
		eXosip_automatic_action();
		eXosip_unlock();

		Thread::yield();

		if(exiting)
			Thread::sync();

		if(!sevent)
			continue;

		slog.debug("sip: event %04x; cid=%d, did=%d, rid=%d",
			sevent->type, sevent->cid, sevent->did, sevent->rid);

		switch(sevent->type)
		{
		case EXOSIP_REGISTRATION_NEW:
			slog.info("sip: new registration");
			break;
		case EXOSIP_REGISTRATION_SUCCESS:
			snprintf(buf, sizeof(buf), "sipreg.%d", sevent->rid);
			server->enter();
			img = server->getActive();
			reg = (Registry *)img->getPointer(buf);	
			if(reg)
			{
				reg->active = true;
				reg->setTimer(reg->duration);
				slog.debug("registration for %s confirmed", reg->contact);
			}
			else
				slog.warn("unknown sip registration confirmed; rid=%s", sevent->rid);
			server->leave();
			break;
		case EXOSIP_REGISTRATION_FAILURE:
			if(sevent->response && sevent->response->status_code == 401)
				break;				
			snprintf(buf, sizeof(buf), "sipreg.%d", sevent->rid);
			server->enter();
			img = server->getActive();
			reg = (Registry *)img->getPointer(buf);	
			if(reg)
			{
				reg->setTimer(0);
				reg->active = false;
				slog.info("registration for %s failed", reg->contact);
			}
			else
				slog.warn("unknown sip registration failed; rid=%s", sevent->rid);
			server->leave();
			break;
		case EXOSIP_REGISTRATION_REFRESHED:
			slog.info("sip registration refreshed");
			break;
		case EXOSIP_REGISTRATION_TERMINATED:
			slog.info("sip registration terminated");
			break;	
		case EXOSIP_CALL_RELEASED:
                        session = getCall(sevent->cid);
                        if(session)
				goto exitcall;
			break;
		case EXOSIP_CALL_MESSAGE_NEW:
			if(!sevent->request)
				break;
			if(!MSG_IS_INFO(sevent->request))
				break;
			session = getCall(sevent->cid);
			if(!session)
			{
				slog.warn("sip: got info for non-existant call");
				goto killit;
			}
			ct = sevent->request->content_type;
			if(!ct || !ct->type || !ct->subtype)
				break;

			if(stricmp(ct->type, "application") || stricmp(ct->subtype, "dtmf-relay"))
				break;

			mbody = NULL;
			osip_message_get_body(sevent->request, 0, &mbody);
			line=mbody->body;
			digit[0] = 0;
			duration[0] = 0;
			while(line && *line)
			{
				k = v = line;
				while(*v && *v != '=' && *v != '\r' && *v != '\n')
					++v;

				if(*v != '=')
					break;

				++v;

				while(*v == ' ' || *v == '\b')
					++v;

				if(!*v || *v == '\r' || *v == '\n')
					break;

				l = v;
				while(*l && *l != ' ' && *l != '\r' && *l != '\n')
					++l;

				if(!*l)
					break;

				if(!strnicmp(k, "signal", 6))
					strncpy(digit, v, l-v);
				else if(!strnicmp(k, "duration", 8))
					strncpy(duration, v, l-v);

				++l;
				while(*l == ' ' || *l == '\n' || *l == '\r')
					++l;

				line = l;
				if(!*line)
					break;
			}

			msg = NULL;
			eXosip_lock();
			eXosip_call_build_answer(sevent->tid, 200, &msg);
			eXosip_call_send_answer(sevent->tid, 200, msg);
			eXosip_unlock();

			if(!digit[0] || !duration[0])
				break;

			memset(&event, 0, sizeof(event));
			event.id = DTMF_KEYUP;
			session->dtmf_inband = false;
			switch(*digit)
			{
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                                event.dtmf.digit = *digit - '0';
                                event.dtmf.duration = atoi(duration);
				session->postEvent(&event);
				break;
			case '*':
                                event.dtmf.digit = 10;
                                event.dtmf.duration = atoi(duration);
                                session->postEvent(&event);
                                break;
                        case '#':
                                event.dtmf.digit = 11;
                                event.dtmf.duration = atoi(duration);
                                session->postEvent(&event);
                                break;
			}
			break;				
		case EXOSIP_CALL_CLOSED:
		case EXOSIP_CALL_CANCELLED:
			session = getCall(sevent->cid);
			if(!session)
			{
				slog.warn("sip: got bye for non-existant call");
				break;
			}
exitcall:
			memset(&event, 0, sizeof(event));
			event.id = STOP_DISCONNECT;	
			session->did = 0;
			session->postEvent(&event);
			break;
		case EXOSIP_CALL_PROCEEDING:
			session = getCall(sevent->cid);
			if(!session)
				break;
			memset(&event, 0, sizeof(event));
			event.id = CALL_PROCEEDING;
			session->postEvent(&event);
			break;
                case EXOSIP_CALL_RINGING:
                        session = getCall(sevent->cid);
                        if(!session)
                                break;
                        memset(&event, 0, sizeof(event));
                        event.id = CALL_RINGING;
                        session->postEvent(&event);
                        break;
                case EXOSIP_CALL_ANSWERED:
                        session = getCall(sevent->cid);
                        if(!session)
                                break;
			msg = NULL;
			sdp = NULL;
			local_sdp = NULL;
			remote_sdp = NULL;
			conn = NULL;
			eXosip_lock();
			eXosip_call_build_ack(sevent->did, &msg);
			if(sevent->request && sevent->response)
			{
				local_sdp = eXosip_get_sdp_info(sevent->request);
				remote_sdp = eXosip_get_sdp_info(sevent->response);
			}
			eXosip_unlock();
			data_rtpmap = session->getSymbol("session.rtpmap");
                        cp = session->getSymbol("session.ip_public");
			if(local_sdp && remote_sdp && session->dtmf_payload)
                        	snprintf(buf, sizeof(buf),
                                	"v=0\r\n"
                                	"o=bayonne 0 0 IN IP4 %s\r\n"
                                	"s=call\r\n"
                                	"c=IN IP4 %s\r\n"
                                	"t=0 0\r\n"
                                	"m=audio %d RTP/AVP %d %d\r\n"
                                	"%s\r\n"
                                	"a=rtpmap:%d telephone-events/8000\r\n",
                                	cp, cp,
                                	session->getLocalPort(), 
					session->data_payload, 
					session->dtmf_payload,
                                	data_rtpmap, 
					session->dtmf_payload);
			else if(local_sdp && remote_sdp)
                        	snprintf(buf, sizeof(buf),
                                	"v=0\r\n"
                                	"o=bayonne 0 0 IN IP4 %s\r\n"
                                	"s=call\r\n"
                                	"c=IN IP4 %s\r\n"
                                	"t=0 0\r\n"
                                	"m=audio %d RTP/AVP %d\r\n"
                                	"%s\r\n",
                                	cp, cp,
                                	session->getLocalPort(), 
					session->data_payload, 
                                	data_rtpmap);

			// we should verify sdp stuff...

			if(remote_sdp)
			{
				conn = eXosip_get_audio_connection(remote_sdp);
				remote_media = eXosip_get_audio_media(remote_sdp);
			}
			if(conn && remote_media)
			{
				session->remote_address = conn->c_addr;
				session->remote_port = atoi(remote_media->m_port);
				snprintf(cbuf, sizeof(cbuf), "%s:%d",
					inet_ntoa(session->remote_address.getAddress()),
					session->remote_port);
				session->setConst("session.rtp_remote", cbuf);
				snprintf(cbuf, sizeof(cbuf), "%s:%d",
					getLast("localip"),
					session->getLocalPort());
				session->setConst("session.rtp_local", cbuf);
			}

			eXosip_lock();
			osip_message_set_body(msg, buf, strlen(buf));
			osip_message_set_content_type(msg, "application/sdp");
			eXosip_unlock();

                        memset(&event, 0, sizeof(event));
                        event.id = CALL_ANSWERED;
                        if(session->postEvent(&event))
			{
				eXosip_lock();
				rtn = eXosip_call_send_ack(sevent->did, msg);
				eXosip_unlock();
				msg = NULL;	
				memset(&event, 0, sizeof(event));
				if(rtn)
					event.id = CALL_FAILURE;
				else
					event.id = CALL_ACCEPTED;
				session->queEvent(&event);
			}
			if(sdp)
				delString(sdp);

			if(msg)
				osip_message_free(msg);
			if(local_sdp)
				sdp_message_free(local_sdp);
			if(remote_sdp)
				sdp_message_free(remote_sdp);
                        break;
		case EXOSIP_CALL_GLOBALFAILURE:
		case EXOSIP_CALL_SERVERFAILURE:
                        session = getCall(sevent->cid);
                        if(!session)
                                break;
                        memset(&event, 0, sizeof(event));
                        event.id = CALL_FAILURE;
                        session->postEvent(&event);
                        break;
		case EXOSIP_CALL_ACK:
			session = getCall(sevent->cid);
			if(!session)
			{
				slog.warn("sip: call ack for non-existant call");
killit:
				eXosip_lock();
				eXosip_call_terminate(sevent->cid, sevent->did);
				eXosip_unlock();
				break;
			}
			memset(&event, 0, sizeof(event));
			event.id = CALL_ACCEPTED;
			session->postEvent(&event);
			break;

		case EXOSIP_CALL_REQUESTFAILURE:
			session = getCall(sevent->cid);
			if(!session)
				break;

			memset(&event, 0, sizeof(event));
			switch(sevent->response->status_code)
			{
			case AUTH_REQUIRED:
				getAuthentication(session);
				eXosip_lock();
				eXosip_automatic_action();
				eXosip_unlock();
				break;
			case UNAUTHORIZED:
				getAuthentication(session);
				break;
			case BUSY_HERE:
				event.id = DIAL_BUSY;
				break;
			case REQ_TIMEOUT:
				event.id = DIAL_TIMEOUT;
				break;
			case NOT_FOUND:
			case NOT_ALLOWED:
			case NOT_ACCEPTABLE:
			case FORBIDDEN:
			case BAD_REQ:
			case TEMP_UNAVAILABLE:
				event.id = DIAL_INVALID;
			default:
				event.id = DIAL_FAILED;
			}	
			if(event.id)
				session->postEvent(&event);
			break;
		case EXOSIP_CALL_INVITE:
			if(!sevent->request)
				break;
			if(sevent->cid < 1 && sevent->did < 1)
				break;

			remote_media = NULL;
			local_media = NULL;
			conn = NULL;
			remote_sdp = NULL;
			local_sdp = NULL;
			display = NULL;
			caller = NULL;
			dialed = NULL;
			local_uri = NULL;
			remote_uri = NULL;
			from = NULL;
			to = NULL;
			sdp = NULL;
			msg = NULL;
			session = NULL;

			osip_to_to_str(sevent->request->to, &local_uri);

			if(!local_uri)
				goto clear;

			osip_from_to_str(sevent->request->from, &remote_uri);
			if(!remote_uri)
				goto clear;

			osip_to_init(&to);
			osip_to_parse(to, local_uri);
			if(!to)
				goto clear;

			if(to->url != NULL && to->url->username != NULL)
				dialed = to->url->username;

			osip_from_init(&from);
			osip_from_parse(from, remote_uri);		
			if(!from)
				goto clear;

			if(from->url != NULL && from->url->username != NULL)
				caller = from->url->username;					
			
                        display = osip_from_get_displayname(from);  
                        if(display && *display == '\"')
                        {
                                ++display;
                                p = strrchr(display, '\"');
                                if(p)
                                        *p = 0; 
                        }
			else if(!display)
				display = caller;

			img = useImage();

                        if(to->url->port && stricmp(to->url->port, "5060"))                                     snprintf(buf, sizeof(buf), "sip.%s:%s",
                                        to->url->host, to->url->port);
                        else
                                snprintf(buf, sizeof(buf), "sip.%s",
                                        to->url->host); 
                        proxy = (Registry *)img->getPointer(buf);
                        if(proxy && !proxy->isActive())
                                goto sip404;

			if(from->url->port && stricmp(from->url->port, "5060"))
				snprintf(buf, sizeof(buf), "sip.%s:%s",
					from->url->host, from->url->port);
			else
				snprintf(buf, sizeof(buf), "sip.%s", from->url->host);
			peer = (Registry *)img->getPointer(buf);
			
			if(peer && !peer->isActive())
				goto sip404;

			snprintf(buf, sizeof(buf), "uri.%s", dialed);
			reg = (Registry *)img->getPointer(buf);
			reg_inband = dtmf_inband;
			reg_payload = dtmf_negotiate;
			auth = "none";
			if(peer && !reg)
				auth = peer->type;
			else if(reg)
				auth = reg->type;
			if(!strnicmp(auth, "ext", 3))
				auth = "peer";
			if(!stricmp(auth, "friend") && peer)
				auth = "peer";
			else if(!stricmp(auth, "friend") && proxy)
				auth = "proxy";
			else if(!stricmp(auth, "friend"))
				reg = NULL;
			if(proxy && !reg && !stricmp(auth, "proxy"))
				reg = proxy;
			if(peer && !reg && !stricmp(auth, "peer"))
				reg = peer;
			if(reg && !peer && !stricmp(auth, "peer"))
				reg = NULL;
			if(reg && !peer && !stricmp(auth, "user"))
				reg = NULL;
                        if(reg && !proxy && !stricmp(auth, "proxy"))
                                reg = NULL; 
			if(reg)
			{
				if(!reg->isActive())
					goto sip404;

				scr = img->getScript(reg->scr->name);
				cp = reg->dtmf;
				if(cp && *cp)
				{
					if(!stricmp(cp, "info") || !stricmp(cp, "sipinfo"))
						reg_inband = false;
					else if(atoi(cp) < 255 && atoi(cp) > 10)
						reg_payload = atoi(cp);
					else if(stricmp(cp, "2833") && stricmp(cp, "rfc2833"))
						reg_payload = 0;
				}		
			}
			else
				goto sip404;

			if(!scr)
			{
				slog.warn("sip: unknown uri %s dialed", dialed);
sip404:
				endImage(img);
				eXosip_lock();
				eXosip_call_send_answer(sevent->tid, 404, NULL);
				eXosip_unlock();				
				goto clear;
			}

			s = getIdle();
			if(!s)
			{
				slog.warn("sip: no timeslots available");
				endImage(img);
				eXosip_lock();
				eXosip_call_send_answer(sevent->tid, 480, NULL);
				eXosip_unlock();
				goto clear;
			}
		
			session = (Session *)(s);
                        cp = getLast("localip");
                        if(reg->address)
                        {
                                InetAddress host(reg->address);
                                snprintf(cbuf, sizeof(cbuf), "%s",
                                        inet_ntoa(host.getAddress()));
                                session->setConst("session.ip_public", cbuf);
                        }
                        else
                                session->setConst("session.ip_public", cp);

                        session->setConst("session.ip_local", cp);

			session->cid = sevent->cid;
			session->did = sevent->did;
			session->setConst("session.info", auth);
			session->setConst("session.dialed", dialed);
			session->setConst("session.caller", caller);
			session->setConst("session.display", display);

			if(from->url->port && stricmp(from->url->port, "5060"))
				snprintf(buf, sizeof(buf), "sip:%s@%s:%s",
					from->url->username,
					from->url->host,
					from->url->port);
			else
				snprintf(buf, sizeof(buf), "sip:%s@%s",
					from->url->username,
					from->url->host);
					
			session->setConst("session.uri_remote", buf);

			if(reg)
			{
				snprintf(cbuf, sizeof(cbuf), "sipreg.%d", reg->regid);
				session->setConst("session.registry", cbuf);
				session->setConst("session.uri_local", reg->contact);
			}
			else
			{
				auth = "anon";
				snprintf(buf, sizeof(buf), "sip:anon@%s", getLast("interface"));
				p = strrchr(buf, ':');
				if(p && !stricmp(p, ":5060"))
					*p = 0;
				session->setConst("session.uri_local", buf);
			}
				
                        session->local_address = getLast("localip");
                        snprintf(buf, sizeof(buf), "%s:%d",
                                inet_ntoa(session->local_address.getAddress()),
                                session->getLocalPort());
                        session->setConst("session.rtp_local", buf);    

			memset(&event, 0, sizeof(event));
			
			if(!stricmp(auth, "peer"))
				event.id = START_INCOMING;
			else
				event.id = START_DIRECT;

			event.start.img = img;
			event.start.scr = scr;

			eXosip_lock();
			remote_sdp = eXosip_get_sdp_info(sevent->request);
			eXosip_unlock();
			if(!remote_sdp)
			{
				eXosip_unlock();
				eXosip_call_send_answer(sevent->tid, 400, NULL);
				eXosip_unlock();
				goto clear;
			}		
			eXosip_lock();
			conn = eXosip_get_audio_connection(remote_sdp);
			remote_media = eXosip_get_audio_media(remote_sdp);	
			eXosip_unlock();
			if(!remote_media || !remote_media->m_port)
			{
sip415:
				eXosip_lock();
				eXosip_call_send_answer(sevent->tid, 415, NULL);
				eXosip_unlock();
				goto clear;
			}
			session->remote_address = conn->c_addr;			
			session->remote_port = atoi(remote_media->m_port);

                        snprintf(buf, sizeof(buf), "%s:%d",
                                inet_ntoa(session->remote_address.getAddress()),                                session->getRemotePort());
                        session->setConst("session.rtp_remote", buf);  

                        cp = session->getSymbol("session.ip_public");
			sdp = newString("", 4096);
			snprintf(sdp, 4096, 
				"v=0\r\n"
				"o=bayonne 0 0 IN IP4 %s\r\n"
				"s=conversation\r\n"
				"c=IN IP4 %s\r\n"
				"t=0 0\r\n", cp, cp);

			session->data_payload = data_negotiate;
			session->dtmf_payload = 0;
			mpos = 0;
			switch(info.encoding)
			{
			case mulawAudio:
				session->data_payload = 0;
				data_rtpmap = "a=rtpmap:0 PCMU/8000/1";
				break;
			case alawAudio:
				session->data_payload = 8;
				data_rtpmap = "a=rtpmap:8 PCMA/8000/1";
				break;
			case gsmVoice:
				session->data_payload = 3;
				data_rtpmap = "a=rtpmap:3 GSM/8000/1";
				break;
			}
			audio_payload = NULL;
			while(!osip_list_eol(remote_sdp->m_medias, mpos))
			{
				remote_media = (sdp_media_t *)osip_list_get(remote_sdp->m_medias, mpos++);
				if(!remote_media)
					continue;

				if(!stricmp(remote_media->m_media, "audio") && !audio_payload)
					audio_media = remote_media;
				else
					audio_media = NULL;

				audio_payload = NULL;
				tmp = NULL;
				pos = 0;
				while(!osip_list_eol(remote_media->m_payloads, pos))
				{
					tmp = (char *)osip_list_get(remote_media->m_payloads, pos++);
					if(!tmp || !*tmp)
						continue;

					if(reg_payload && atoi(tmp) == reg_payload)
					{
						session->dtmf_payload = dtmf_negotiate;
						reg_inband = false;
					}

					if(audio_media && atoi(tmp) == session->data_payload)
						audio_payload = tmp;
				}
			}
			
			session->dtmf_inband = reg_inband;		

			if(!audio_payload)
				goto sip415;

			if(session->dtmf_payload)
				snprintf(cbuf, sizeof(cbuf),
					"m=audio %d RTP/AVP %d %d\r\n"
					"%s\r\n",
						session->getLocalPort(),
						session->data_payload, 
						session->dtmf_payload,
						data_rtpmap);
			else
				snprintf(cbuf, sizeof(cbuf),
					"m=audio %d RTP/AVP %d\r\n"
					"%s\r\n",
						session->getLocalPort(),
						session->data_payload,
						data_rtpmap);
				
			addString(sdp, 4096, cbuf);

			session->setConst("session.rtpmap", data_rtpmap);

			if(session->dtmf_payload)
			{
				snprintf(cbuf, sizeof(cbuf),
					"a=rtpmap:%d telephone-events/8000\r\n",
					session->dtmf_payload);
				addString(sdp, 4096, cbuf);
			}

                        if(!s->postEvent(&event))
                                goto sip404;

			eXosip_lock();
			eXosip_call_send_answer(sevent->tid, 180, NULL);
			eXosip_unlock();

			eXosip_lock();
			eXosip_call_build_answer(sevent->tid, 200, &msg);
			if(!msg)
				goto sip404;
			
			osip_message_set_require(msg, "100rel");
			osip_message_set_header(msg, "RSeq", "1");
			osip_message_set_body(msg, sdp, strlen(sdp));
			osip_message_set_content_type(msg, "application/sdp");

			eXosip_call_send_answer(sevent->tid, 183, msg); 
			
			eXosip_unlock();
			session = NULL;


clear:
			if(session)
				add(session);

			eXosip_lock();
//			if(msg)
//				osip_message_free(msg);

			if(sdp)
				delString(sdp);

			if(remote_sdp)
				sdp_message_free(remote_sdp);

			if(local_sdp)
				sdp_message_free(local_sdp);

			if(from) 	                                                             
				osip_from_free(from);

			if(to)
				osip_to_free(to);

			if(remote_uri)
				osip_free(remote_uri);

			if(local_uri)
				osip_free(local_uri);

			eXosip_unlock();
			break;
		}			
		eXosip_event_free(sevent);
	}
}

} // namespace
