;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: LAPIDARY; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;         The Garnet User Interface Development Environment.      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code was written as part of the Garnet project at          ;;;
;;; Carnegie Mellon University, and has been placed in the public   ;;;
;;; domain.  If you are using this code or any part of Garnet,      ;;;
;;; please contact garnet@cs.cmu.edu to be put on the mailing list. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Lapidary:Defs.Lisp
;;;
;;; This file contains many of the schemas, defconstants, defvars, and 
;;;  defstructs which are used by Lapidary.  This does not contain any 
;;;  defmacros, however.
;;;

(in-package "LAPIDARY" :use '("LISP" "KR"))


;;;=========================================================
;;;
;;; structure for holding the left, top, width, and height
;;; of a bounding box
;;;
;;;=========================================================

(defstruct bbox-wh
   left 
   top
   width
   height
)

;;;=========================================================
;;;
;;; set up a global selection schema that keeps track of the 
;;; selected objects in all windows and the selection-type 
;;; of objects in all windows (e.g., one-one--one primary 
;;; and one secondary selection, zero-one--no primary and 
;;; one secondary selections)
;;;
;;;=========================================================

(create-schema '*selection-info*
     (:p-selected nil)
     (:s-selected nil)
     (:selected (o-formula (union (gvl :p-selected) (gvl :s-selected))))
     (:feedback nil) ; list of selection feedback objects currently displayed
     (:selection-type (o-formula (classify-selections))))

;;;====================================================================
;;;
;;; set up the schema that keeps track of event information
;;; for the start, stop, and abort events in interactor dialog
;;; boxes. event-cards contains a card for each event, event-type
;;; indicates whether the event is a start-, stop, or abort-event
;;; and queue indicates on which interactor's queue the information
;;; should be stored
;;;
;;;====================================================================

(create-schema '*event-info*
     (:event-cards nil)
     (:event-type nil)
     (:queue nil))

;;; put names into opal objects, filling styles, and line styles so
;;; that meaningful names can be constructed for objects

;;; opal objects

(s-value opal:rectangle :name "rectangle")
(s-value opal:line :name "line")
(s-value opal:circle :name "circle")
(s-value opal:text :name "text")
(s-value opal:roundtangle :name "roundtangle")

;;; opal filling styles

(s-value opal:white-fill :name "white")
(s-value opal:light-gray-fill :name "light-gray")
(s-value opal:gray-fill :name "gray")
(s-value opal:dark-gray-fill :name "dark-gray")
(s-value opal:black-fill :name "black")

;;; opal line styles

(s-value opal:thin-line :name "thin-line")
(s-value opal:dotted-line :name "dotted-line")
(s-value opal:dashed-line :name "dashed-line")
(s-value opal:line-2 :name "line2")
(s-value opal:line-4 :name "line4")
(s-value opal:line-8 :name "line8")

;;;=========================================================
;;;
;;; map names of menu selections to positions in a menu 
;;; array
;;;
;;;=========================================================

;;; indicates if dialog boxes should be loaded

(defparameter *load-db* nil)

;;; indicates if the constraints between two objects should be highlighted

(defparameter *show-constraints* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Set standard fonts
;;;;;;;;;;;;;;;;;;;;;;;;;;

(create-instance '*fnt* opal:font-from-file
		 (:font-name "timr12"))
(create-instance '*title-font* opal:font-from-file
		 (:font-name "timbi18"))
(create-instance '*label-font* opal:font-from-file
		 (:font-name "timb12"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Useful constants for attaching constraints to lines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconstant 45deg (+ 1 (/ (sqrt 2) 2)))
(defconstant 135deg (- 1 (/ (sqrt 2) 2)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Size of the selection boxes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconstant sel-box-size 7)
(defconstant sel-box-sizeD2 3)
(defconstant *agg-sel-circle-size* 17)
(defconstant *min-agg-size* (+ (* 2 *agg-sel-circle-size*) 4))
(defconstant *min-leaf-size* (+ (* 2 sel-box-size) 4))

;;; =============================================================
;;; prototype formulas for the constraints shown in lapidary's
;;; constraint menus
;;; =============================================================

(defvar *left-outside-formula* 
  (o-formula (- (lapidary::gv-right-is-left-of (gvl :left-over))
					  (gvl :left-offset))))

(defvar *left-inside-formula* (o-formula (+ (gvl :left-over :left)
				    (gvl :left-offset))))
(defvar *left-center-formula* 
  (o-formula (- (+ (gvl :left-over :left)
		   (truncate (* (gvl :left-over :width)
				(gvl :left-offset))))
		(truncate (gvl :width) 2))))
(defvar *right-inside-formula* 
  (o-formula (1+ (- (lapidary::gv-right (gvl :left-over))
		    (gvl :width)
		    (gvl :left-offset)))))
(defvar *right-outside-formula* (o-formula (+ (lapidary::gv-right (gvl :left-over))
				      (gvl :left-offset))))

(defvar *left-constraint-vector* 
  (make-array 5 :initial-contents (list *left-outside-formula*
					*left-inside-formula*
					*left-center-formula*
					*right-inside-formula*
					*right-outside-formula*)))

(defvar *top-outside-formula* 
  (o-formula (- (lapidary::gv-bottom-is-top-of (gvl :top-over))
		(gvl :top-offset))))

(defvar *top-inside-formula* (o-formula (+ (gvl :top-over :top)
				   (gvl :top-offset))))
(defvar *top-center-formula* 
  (o-formula (- (+ (gvl :top-over :top)
		   (truncate (* (gvl :top-over :height)
				(gvl :top-offset))))
		(truncate (gvl :height) 2))))
(defvar *bottom-inside-formula* 
  (o-formula (1+ (- (lapidary::gv-bottom (gvl :top-over))
		    (gvl :height)
		    (gvl :top-offset)))))

(defvar *bottom-outside-formula* (o-formula (+ (lapidary::gv-bottom (gvl :top-over))
				       (gvl :top-offset))))

(defvar *top-constraint-vector* 
  (make-array 5 :initial-contents (list *top-outside-formula*
					*top-inside-formula*
					*top-center-formula*
					*bottom-inside-formula*
					*bottom-outside-formula*)))

(defvar *width-formula* (o-formula (round (+ (* (gvl :width-scale) 
						(gvl :width-over :width))
					     (gvl :width-difference)))))

(defvar *width-constraint-vector* 
  (make-array 1 :initial-contents (list *width-formula*)))

(defvar *height-formula* (o-formula (round (+ (* (gvl :height-scale) 
					 (gvl :height-over :height))
				      (gvl :height-difference)))))

(defvar *height-constraint-vector* 
  (make-array 1 :initial-contents (list *height-formula*)))
(defvar *custom-constraint* nil)

;; error gadget to display error messages
(create-instance '*lapidary-error-window* garnet-gadgets:error-gadget)

;; create a general purpose query gadget
(create-instance 'lapidary-query-gadget GARNET-GADGETS:query-gadget
      (:SELECTION-FUNCTION 'by-demo-OKCANCEL-FUNCTION)
      (:modal-p nil)
      (:function-for-ok 'by-demo-ok-function))

;; definitions for lists that contain free feedback objects
(defvar *undersized-feedback-list* nil)
(defvar *leaf-feedback-list* nil)
(defvar *agg-feedback-list* nil)
(defvar *line-feedback-list* nil)

;; definitions for variables that contain cursor information
(defvar copy-cursor-pair nil)
(defvar instance-cursor-pair nil)
(defvar load-cursor-pair nil)
(defvar move-cursor-pair nil)
(defvar delete-cursor-pair nil)

(defvar *CREATED-INSTANCES* NIL
  "list of objects created by loading a file")

(defvar *window-count* 0
  "number of windows created")
;; lists of offset slots used by objects
(defvar *list-offsets* (list :x1-offset :x2-offset :y1-offset :y2-offset))
(defvar *box-offsets* (list :left-offset :top-offset 
			    :width-difference :height-difference))


(defvar *prop-sheet*
  (create-instance NIL garnet-gadgets:prop-sheet-for-obj-with-OK
		   (:OK-Function 'Aggrelist-Prop-Sheet-Finish)
		   (:Apply-Function 'Aggrelist-Prop-Sheet-Finish)))

(defparameter *aggrelist-slots* (List '(:direction (:Vertical :Horizontal))
				      (list :v-spacing 'gilt::num-only)
				      (list :h-spacing 'gilt::num-only)
				      '(:fixed-width-p (T NIL))
				      '(:fixed-height-p (T NIL))
				      '(:h-align (:left :center :right))
				      '(:v-align (:top :center :bottom))
				      (list :rank-margin 'gilt::nil-or-num)
				      (list :pixel-margin 'gilt::nil-or-num)
				      (list :indent 'gilt::num-only)))

;;; used for creating feedback when resizing an aggrelist
(defvar *aggrelist-feedback-slots* 
  '(:left :top :direction :v-spacing :h-spacing 
		:fixed-width-size :fixed-height-size
		:fixed-width-p :fixed-height-p
		:h-align :v-align
		:rank-margin :pixel-margin
		:indent :items))

;;; font used for slot names in constraint menus
(defvar *slot-font* (create-instance NIL opal:font
		       (:size :large) (:family :serif) (:face :bold-italic)))

;;; used so that objects will appear with a white border if they are 
;;; inverted

(defvar *white-line-style* (create-instance NIL opal:line-style
			      (:stipple opal::white-fill-bitmap)))

(defvar *large-bold-italic-serif-font* (create-instance NIL opal:font
				         (:size :large) (:face :bold-italic)
					 (:family :serif)))

;; some slots should not be altered by the user. These are stored in
;; an object's :do-not-alter-slots

(s-value opal:text :do-not-alter-slots '(:width :height))
(s-value opal:bitmap :do-not-alter-slots '(:width :height))

;; when a constraint is destroyed in a slot, some slots, such as
;; :filling-style and :line-style, should inherit from their prototype;
;; others, such as :left and :top, should assume their current values.
;; the following list records those slots which should not inherit

(defvar *do-not-inherit-list* '(:left :top :active))


;; add slots to list of slots that c32 should show for lapidary interactors
(s-value lapidary:directional-move-grow-interactor :slots-to-show
	 (append (g-value lapidary:directional-move-grow-interactor 
			  :slots-to-show)
		 '(:grow-box-parms :move-box-parms)))
