/* Test mpz_import.

Copyright 2002, 2004 Free Software Foundation, Inc.

This file is part of the GNU MP Library.

The GNU MP Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The GNU MP Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the GNU MP Library; see the file COPYING.LIB.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gmp.h"
#include "gmp-impl.h"
#include "tests.h"


void
check_data (void)
{
  static const struct {
    const char  *want;
    size_t      count;
    int         order;
    size_t      size;
    int         endian;
    int         nail;
    char        src[64];

  } data[] = {

    { "0", 0,1, 1,1, 0 },
    { "0", 1,1, 0,1, 0 },

    { "0x12345678", 4,1,  1,1, 0, { 0x12, 0x34, 0x56, 0x78 } },
    { "0x12345678", 1,1,  4,1, 0, { 0x12, 0x34, 0x56, 0x78 } },
    { "0x12345678", 1,-1, 4,1, 0, { 0x12, 0x34, 0x56, 0x78 } },

    { "0x12345678", 4,-1, 1,-1, 0, { 0x78, 0x56, 0x34, 0x12 } },
    { "0x12345678", 1,1,  4,-1, 0, { 0x78, 0x56, 0x34, 0x12 } },
    { "0x12345678", 1,-1, 4,-1, 0, { 0x78, 0x56, 0x34, 0x12 } },

    { "0",    5,1,  1,1, 7, { 0xFE, 0xFE, 0xFE, 0xFE, 0xFE } },
    { "0",    5,-1, 1,1, 7, { 0xFE, 0xFE, 0xFE, 0xFE, 0xFE } },
    { "0x15", 5,1,  1,1, 7, { 0xFF, 0xFE, 0xFF, 0xFE, 0xFF } },

    { "0",    3,1,  2,1,   1, { 0x80,0x00, 0x80,0x00, 0x80,0x00 } },
    { "0",    3,1,  2,-1,  1, { 0x00,0x80, 0x00,0x80, 0x00,0x80 } },
    { "0",    3,1,  2,1,  15, { 0xFF,0xFE, 0xFF,0xFE, 0xFF,0xFE } },

    { "0x2A", 3,1,  2,1, 14, { 0xFF,0xFE, 0xFF,0xFE, 0xFF,0xFE } },
    { "0x06", 3,1,  2,1, 14, { 0xFF,0xFC, 0xFF,0xFD, 0xFF,0xFE } },
    { "0x24", 3,-1, 2,1, 14, { 0xFF,0xFC, 0xFF,0xFD, 0xFF,0xFE } },

    { "0x123456789ABC", 3,1,  2,1,  0, { 0x12,0x34, 0x56,0x78, 0x9A,0xBC } },
    { "0x123456789ABC", 3,-1, 2,1,  0, { 0x9A,0xBC, 0x56,0x78, 0x12,0x34 } },
    { "0x123456789ABC", 3,1,  2,-1, 0, { 0x34,0x12, 0x78,0x56, 0xBC,0x9A } },
    { "0x123456789ABC", 3,-1, 2,-1, 0, { 0xBC,0x9A, 0x78,0x56, 0x34,0x12 } },

    { "0x112233445566778899AABBCC", 3,1,  4,1,  0,
      { 0x11,0x22,0x33,0x44, 0x55,0x66,0x77,0x88, 0x99,0xAA,0xBB,0xCC } },
    { "0x112233445566778899AABBCC", 3,-1, 4,1,  0,
      { 0x99,0xAA,0xBB,0xCC, 0x55,0x66,0x77,0x88, 0x11,0x22,0x33,0x44 } },
    { "0x112233445566778899AABBCC", 3,1,  4,-1, 0,
      { 0x44,0x33,0x22,0x11, 0x88,0x77,0x66,0x55, 0xCC,0xBB,0xAA,0x99 } },
    { "0x112233445566778899AABBCC", 3,-1, 4,-1, 0,
      { 0xCC,0xBB,0xAA,0x99, 0x88,0x77,0x66,0x55, 0x44,0x33,0x22,0x11 } },

    { "0x100120023003400450056006700780089009A00AB00BC00C", 3,1,  8,1,  0,
      { 0x10,0x01,0x20,0x02,0x30,0x03,0x40,0x04,
        0x50,0x05,0x60,0x06,0x70,0x07,0x80,0x08,
        0x90,0x09,0xA0,0x0A,0xB0,0x0B,0xC0,0x0C } },
    { "0x100120023003400450056006700780089009A00AB00BC00C", 3,-1, 8,1,  0,
      { 0x90,0x09,0xA0,0x0A,0xB0,0x0B,0xC0,0x0C,
        0x50,0x05,0x60,0x06,0x70,0x07,0x80,0x08,
        0x10,0x01,0x20,0x02,0x30,0x03,0x40,0x04 } },
    { "0x100120023003400450056006700780089009A00AB00BC00C", 3,1,  8,-1, 0,
      { 0x04,0x40,0x03,0x30,0x02,0x20,0x01,0x10,
        0x08,0x80,0x07,0x70,0x06,0x60,0x05,0x50,
        0x0C,0xC0,0x0B,0xB0,0x0A,0xA0,0x09,0x90 } },
    { "0x100120023003400450056006700780089009A00AB00BC00C", 3,-1, 8,-1, 0,
      { 0x0C,0xC0,0x0B,0xB0,0x0A,0xA0,0x09,0x90,
        0x08,0x80,0x07,0x70,0x06,0x60,0x05,0x50,
        0x04,0x40,0x03,0x30,0x02,0x20,0x01,0x10 } },

    { "0x155555555555555555555555", 3,1,  4,1,  1,
      { 0xD5,0x55,0x55,0x55, 0xAA,0xAA,0xAA,0xAA, 0xD5,0x55,0x55,0x55 } },
    { "0x155555555555555555555555", 3,-1,  4,1,  1,
      { 0xD5,0x55,0x55,0x55, 0xAA,0xAA,0xAA,0xAA, 0xD5,0x55,0x55,0x55 } },
    { "0x155555555555555555555555", 3,1,  4,-1,  1,
      { 0x55,0x55,0x55,0xD5, 0xAA,0xAA,0xAA,0xAA, 0x55,0x55,0x55,0xD5 } },
    { "0x155555555555555555555555", 3,-1,  4,-1,  1,
      { 0x55,0x55,0x55,0xD5, 0xAA,0xAA,0xAA,0xAA, 0x55,0x55,0x55,0xD5 } },
  };

  char    buf[sizeof(data[0].src) + sizeof (mp_limb_t)];
  char    *src;
  size_t  align;
  int     i;
  mpz_t   got, want;

  mpz_init (got);
  mpz_init (want);

  for (i = 0; i < numberof (data); i++)
    {
      for (align = 0; align < sizeof (mp_limb_t); align++)
        {
          mpz_set_str_or_abort (want, data[i].want, 0);
          src = buf + align;
          memcpy (src, data[i].src, data[i].count * data[i].size);

          mpz_set_ui (got, 0L);
          mpz_import (got, data[i].count, data[i].order,
                      data[i].size, data[i].endian, data[i].nail, src);

          MPZ_CHECK_FORMAT (got);
          if (mpz_cmp (got, want) != 0)
            {
              printf ("wrong at data[%d]\n", i);
              printf ("    count=%u order=%d  size=%u endian=%d nail=%u  align=%u\n",
                      data[i].count, data[i].order,
                      data[i].size, data[i].endian, data[i].nail,
                      align);
              mpz_trace ("    got ", got);
              mpz_trace ("    want", want);
              abort ();
            }
        }
    }
  mpz_clear (got);
  mpz_clear (want);
}


int
main (void)
{
  tests_start ();

  mp_trace_base = -16;
  check_data ();

  tests_end ();
  exit (0);
}
