/*
    GNUbik -- A 3 dimensional magic cube game.
    Copyright (C) 2003  John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
static const char RCSID[]="$Id: menus-gtk.c,v 1.16 2004/08/09 10:54:39 jmd Exp $";

#include <gtk/gtk.h>
#include "menus-gtk.h"

#include "gnubik.h"
#include "cube.h"
#include "glarea.h"
#include "version.h"
#include "ui.h"
#include "drwBlock.h"
#include "widget-set.h"

#include <memory.h>
#include <glib.h>
#include <assert.h>

extern int frameQty;

static int new_frameQty;
static int new_dim ;

static void
confirm_preferences(GtkWidget *w , gpointer data);

static void
set_lighting(GtkToggleButton *b , gpointer user_data);


static void        
value_changed(GtkAdjustment *adj,gpointer user_data);




#if DEBUG
static void manual_move(GtkWidget *w , Move_Data * data);

static void        
fill_value(GtkAdjustment *adj, int *p)
{
  *p = (int) adj->value;
}

#endif


static void
set_lighting(GtkToggleButton *b , gpointer user_data)
{

  if (   gtk_toggle_button_get_active(b) ) 
    glEnable(GL_LIGHTING);
  else
    glDisable(GL_LIGHTING);
    
  postRedisplay();
}



extern Move_Queue *move_queue;

gboolean new_game(gpointer p);

gboolean 
new_game(gpointer p)
{
  int i;

  int reallocate = GPOINTER_TO_INT(p);

  if ( is_animating() ) 
    return TRUE;

  if ( reallocate ) 
    {

      re_initialize_glarea();


      destroy_the_cube ();
      /* create the cube */
      number_of_blocks = create_the_cube (cube_dimension);
      if ( 0 > number_of_blocks ) {
	print_cube_error (the_cube, "Error creating cube");
	exit(-1);
      }

    }

  for( i = 0; i < 8*cube_dimension; i++ )
  {
      Slice_Blocks *const slice =
          identify_blocks (the_cube, rand () % number_of_blocks, rand () % 3);
      
      rotate_slice(the_cube, rand()%2 +1, slice);

      free_slice_blocks (slice);
  }
  
  postRedisplay();

  if (move_queue != NULL) free_move_queue (move_queue);
  move_queue = new_move_queue ();
  
  set_toolbar_state (0);
  update_statusbar();


  return FALSE;
}



/* Request that the game be restarted
   If data is non zero, then all it's data will be reallocated
*/
void 
request_new_game( GtkWidget *w, gpointer   reallocate )
{

  request_stop();
  if (is_animating() ) 
    abort_animation();
  gtk_idle_add(new_game,reallocate);

}


static void  size_changed  (GtkEditable *editable,gpointer data);

#define BOX_PADDING 10 

static GtkWidget *
create_dimension_widget(void)
{
  GtkWidget *vbox; 
  GtkWidget *entry;
  GtkWidget *label;
  GtkWidget *hbox;
  GtkObject *adj;
  GtkTooltips *tips;


  tips = gtk_tooltips_new();

  vbox = gtk_vbox_new(TRUE,BOX_PADDING);
  hbox = gtk_hbox_new(FALSE,BOX_PADDING);
  
  adj = gtk_adjustment_new(cube_dimension,1,G_MAXFLOAT,1,1,10);

  entry = gtk_spin_button_new(GTK_ADJUSTMENT(adj),0,0);
  gtk_tooltips_set_tip(tips,entry,_("Sets the number of blocks in each side"),
		       0);



  label  = gtk_label_new(_("Size of cube:"));


  gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(hbox),entry,FALSE,FALSE,0);

  gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,TRUE,0);

  new_dim = cube_dimension;

  g_signal_connect(GTK_OBJECT(entry),"changed",
		     GTK_SIGNAL_FUNC(size_changed),0);


  gtk_widget_show(label);
  gtk_widget_show(entry);
  gtk_widget_show(hbox);



  return vbox;
}




static GtkWidget *
create_animation_widget(void)
{
  GtkWidget *vboxOuter; 
  GtkWidget *label1;
  GtkWidget *label2;
  GtkWidget *vbox;
  GtkObject *adjustment;
  GtkWidget *scrollbar;
  GtkWidget *hbox;

  GtkTooltips *tips;


  tips = gtk_tooltips_new();
  vboxOuter = gtk_vbox_new(TRUE,BOX_PADDING);
  vbox = gtk_vbox_new(FALSE,BOX_PADDING);

  adjustment = gtk_adjustment_new(frameQty,0,99,1,1,10);
  

  hbox = gtk_hbox_new(FALSE,0);
  label1  = gtk_label_new(_("Faster"));
  gtk_label_set_justify(GTK_LABEL(label1),GTK_JUSTIFY_LEFT);

  label2  = gtk_label_new(_("Slower"));
  gtk_label_set_justify(GTK_LABEL(label2),GTK_JUSTIFY_RIGHT);

  scrollbar = gtk_hscrollbar_new(GTK_ADJUSTMENT(adjustment));
  gtk_tooltips_set_tip(tips,scrollbar,
		       _("Controls the speed with which slices rotate"),
		       0);


  gtk_box_pack_start(GTK_BOX(hbox),label1,TRUE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(hbox),label2,TRUE,FALSE,0);

  
  gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(vbox),scrollbar,TRUE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(vboxOuter),vbox,TRUE,TRUE,0);


  /* Add any necessary callbacks */

  g_signal_connect(adjustment,"value-changed",
		     GTK_SIGNAL_FUNC(value_changed),0);

  /* Show the widgets */

  gtk_widget_show_all(vbox);



  return vboxOuter;
}



void 
preferences( GtkWidget *w,
	       gpointer   data )
{


  GtkWidget *button_cancel;
  GtkWidget *button_ok;

  GtkWidget *dialog ;

  GtkWidget *vbox;

  GtkWidget *frame_dimensions;
  GtkWidget *frame_animation;
  GtkWidget *dimensions;
  GtkWidget *animations;

  GtkWidget *frame_lighting;
  GtkWidget *button_lighting;

  GtkWindow *toplevel  = GTK_WINDOW(data);

  GtkTooltips *tips;

  tips = gtk_tooltips_new();

  dialog = gtk_dialog_new();
  gtk_window_set_modal(GTK_WINDOW(dialog),TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(dialog),toplevel);

  gtk_window_set_title(GTK_WINDOW(dialog),_("Preferences"));


  vbox = gtk_vbox_new(FALSE,BOX_PADDING);

  frame_dimensions = gtk_frame_new(_("Dimensions"));
  frame_animation = gtk_frame_new(_("Animation"));
  frame_lighting = gtk_frame_new(_("Lighting"));


  dimensions = create_dimension_widget();
  gtk_container_add(GTK_CONTAINER(frame_dimensions),dimensions);

  animations = create_animation_widget();
  gtk_container_add(GTK_CONTAINER(frame_animation),animations);

  button_lighting = gtk_check_button_new_with_label(_("Enable lighting"));
  gtk_tooltips_set_tip(tips,button_lighting,
		       _("Makes the cube appear illuminated"),
		       0);

  gtk_container_add(GTK_CONTAINER(frame_lighting),button_lighting);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button_lighting),
			       glIsEnabled(GL_LIGHTING));
                                          
  button_cancel = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  button_ok = gtk_button_new_from_stock(GTK_STOCK_OK);

  
  gtk_box_pack_start(GTK_BOX(vbox),frame_dimensions,FALSE,0,0);
  gtk_box_pack_start(GTK_BOX(vbox),frame_animation,FALSE,0,0);
  gtk_box_pack_start(GTK_BOX(vbox),frame_lighting,FALSE,0,0);


  gtk_container_set_border_width(GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
				 BOX_PADDING);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
                      vbox);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                      button_ok);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                      button_cancel);


  g_signal_connect_swapped(GTK_OBJECT(button_cancel),"clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) dialog);

  g_signal_connect_swapped(GTK_OBJECT(button_ok),"clicked",
		     GTK_SIGNAL_FUNC(confirm_preferences),
		     (gpointer) dialog);


  g_signal_connect(GTK_OBJECT(button_lighting),"toggled",
		     GTK_SIGNAL_FUNC(set_lighting),
		     0);
  

  /* Show all the widgets */

  gtk_widget_show_all(dialog);
}

extern GtkWidget *main_application_window;

static gboolean new_values=FALSE;

static void 
yesnoresp(GtkWidget *w, gint resp, gpointer data)
{
  switch( resp ) { 
  case GTK_RESPONSE_YES:

    cube_dimension = new_dim ;

    request_new_game(w,GINT_TO_POINTER(1));

    break ;

  case GTK_RESPONSE_NO:
    break ;

  default:
    g_assert_not_reached();
  }

  gtk_widget_destroy(w);

}

/* Close the preferences window, and update all the necessary values */
static void
confirm_preferences(GtkWidget *w , gpointer data)
{

  if ( new_values) { 
    frameQty = new_frameQty;
  }


  gtk_widget_destroy(w);


  if ( new_dim != cube_dimension) { 
    /* Popup dialog asking whether to restart the cube */
    GtkWidget *dialog ;
   
    dialog = gtk_message_dialog_new( GTK_WINDOW(main_application_window),
				     GTK_DIALOG_MODAL 
				     | GTK_DIALOG_DESTROY_WITH_PARENT,
				     GTK_MESSAGE_QUESTION,
				     GTK_BUTTONS_YES_NO,
				     _("Start cube with new settings?"));

    g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
                           G_CALLBACK (yesnoresp),
                           GTK_OBJECT (dialog));

    gtk_widget_show_all (dialog);

  }
  
}




static void        
value_changed(GtkAdjustment *adj,gpointer user_data)
{

  new_values = TRUE;
  new_frameQty = adj->value;
}




static void  
size_changed  (GtkEditable *editable,gpointer data)
{
  gchar *str; 

  str = gtk_editable_get_chars(GTK_EDITABLE(editable),0,-1);

  new_dim = g_strtod(str,0);

  g_free(str);
  
}




#if DEBUG
void 
move( GtkWidget *w,      gpointer   data )
{
  GtkWidget *dialog; 

  GtkWidget *hbox1;
  GtkWidget *hbox2;
  GtkWidget *hbox4;
  GtkWidget *button_turn;
  GtkWidget *button_close;
  GtkObject *adj1,   *adj2,     *adj4;
  GtkWidget *label1, *label2,   *label4;
  GtkWidget *sb1,    *sb2,      *sb4;

  static Move_Data md;

  dialog = gtk_dialog_new();
   

  button_turn = gtk_button_new_with_label("Turn");
  button_close = gtk_button_new_with_label("Close");


  hbox1 = gtk_hbox_new(TRUE,0);
  adj1 = gtk_adjustment_new(0,0,number_of_blocks-1,1,1,10);
  
  g_signal_connect(adj1,"value-changed",GTK_SIGNAL_FUNC(fill_value),&md.slice);

  sb1 = gtk_spin_button_new(GTK_ADJUSTMENT(adj1),0,0);
  label1  = gtk_label_new("Block to move:");


  hbox2 = gtk_hbox_new(TRUE,0);
  adj2 = gtk_adjustment_new(0,0,2,1,1,10);
  g_signal_connect(adj2,"value-changed",GTK_SIGNAL_FUNC(fill_value),&md.axis);
  sb2 = gtk_spin_button_new(GTK_ADJUSTMENT(adj2),0,0);
  label2  = gtk_label_new("Axis:");


  hbox4 = gtk_hbox_new(TRUE,0);
  adj4 = gtk_adjustment_new(0,0,1,1,1,1);
  g_signal_connect(adj4,"value-changed",GTK_SIGNAL_FUNC(fill_value),&md.dir);
  sb4 = gtk_spin_button_new(GTK_ADJUSTMENT(adj4),0,0);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON(sb4),TRUE);
  label4  = gtk_label_new("Direction:");



  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                      button_turn);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                      button_close);


  gtk_box_pack_start(GTK_BOX(hbox1),label1,TRUE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(hbox1),sb1,TRUE,TRUE,0);

  gtk_box_pack_start(GTK_BOX(hbox2),label2,TRUE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(hbox2),sb2,TRUE,TRUE,0);


  gtk_box_pack_start(GTK_BOX(hbox4),label4,TRUE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(hbox4),sb4,TRUE,TRUE,0);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
                      hbox1);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
                      hbox2);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
                      hbox4);

  /* Ensure that the dialog box is destroyed when the user responds. */


  g_signal_connect(GTK_OBJECT(button_turn),
		     "clicked",
		     GTK_SIGNAL_FUNC(manual_move),&md);

  g_signal_connect_swapped(GTK_OBJECT(button_close),
		     "clicked",
		     GTK_SIGNAL_FUNC(gtk_widget_destroy),(gpointer) dialog);


   
  gtk_widget_show_all (dialog);

 
}






/* Manually make a move on the cube */
static void
manual_move(GtkWidget *w , Move_Data * data)
{

  /* Just loose the first argument and call the next function */

  request_rotation(data);

  postRedisplay();


}


#endif

void 
about( GtkWidget *w,
	gpointer   data )
{

  GtkWidget *dialog;
  GtkWidget *text;
  GtkWidget *okay_button;
  GtkTextBuffer *buffer;
  GtkWindow *toplevel=GTK_WINDOW(data);
   
  /* Create the widgets */
   
  dialog = gtk_dialog_new();
  gtk_window_set_default_size(GTK_WINDOW(dialog),650,370);
  gtk_window_set_modal(GTK_WINDOW(dialog),TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(dialog),toplevel);
  


  okay_button = gtk_button_new_from_stock(GTK_STOCK_OK);

  buffer = gtk_text_buffer_new(0);
  text = gtk_text_view_new_with_buffer(buffer);


  gtk_text_view_set_editable(GTK_TEXT_VIEW(text),FALSE);
  gtk_text_buffer_insert_at_cursor(buffer,copyleft_notice,
				   strlen(copyleft_notice));
   
  /* Ensure that the dialog box is destroyed when the user clicks ok. */
   
  g_signal_connect_swapped (GTK_OBJECT (okay_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), 
			     (gpointer) dialog);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     okay_button);

  /* Add the text, and show everything we've added to the dialog. */

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
		     text);

  gtk_widget_show_all (dialog);

}
