/*
   This file is part of GNUnet.
   Copyright (C) 2020 GNUnet e.V.

   GNUnet is free software: you can redistribute it and/or modify it
   under the terms of the GNU Affero General Public License as published
   by the Free Software Foundation, either version 3 of the License,
   or (at your option) any later version.

   GNUnet is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Affero General Public License for more details.

   You should have received a copy of the GNU Affero General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   SPDX-License-Identifier: AGPL3.0-or-later
 */
/**
 * @author Tobias Frisch
 * @file src/messenger/gnunet-service-messenger_message_recv.h
 * @brief GNUnet MESSENGER service
 */

#ifndef GNUNET_SERVICE_MESSENGER_MESSAGE_RECV_H
#define GNUNET_SERVICE_MESSENGER_MESSAGE_RECV_H

#include "platform.h"
#include "gnunet_crypto_lib.h"

#include "gnunet-service-messenger_tunnel.h"
#include "messenger_api_message.h"

/**
 * Handles a received info message to change the current member id to the one generated by
 * the host connected to. (all current tunnels will be informed about the id change)
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message INFO-Message
 * @param hash Hash of the message
 */
void
recv_message_info (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                   struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received join message to forward all member information to the new member if the message was
 * the direct reaction to a previous info message from this peer.
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message JOIN-Message
 * @param hash Hash of the message
 */
void
recv_message_join (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                   struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received leave message.
 * @see handle_message_leave()
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message LEAVE-Message
 * @param hash Hash of the message
 */
void
recv_message_leave (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                    struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received name message.
 * @see handle_message_name()
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message NAME-Message
 * @param hash Hash of the message
 */
void
recv_message_name (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                   struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received key message.
 * @see handle_message_key()
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message KEY-Message
 * @param hash Hash of the message
 */
void
recv_message_key (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                  struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received peer message to link it to its origin tunnel if the peer identity matches.
 * (the peer message and the member id can potentially be linked to the tunnel)
 *
 * TODO: This handling will only check the one given tunnel!
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message PEER-Message
 * @param hash Hash of the message
 */
void
recv_message_peer (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                   struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received id message to change the tunnels linked member id if necessary.
 * (the tunnels linked member id will be changed if the sender id is matching)
 *
 * TODO: This handling will only check the one given tunnel!
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message ID-Message
 * @param hash Hash of the message
 */
void
recv_message_id (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                 struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received miss message.
 * @see handle_message_miss()
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message MISS-Message
 * @param hash Hash of the message
 */
void
recv_message_miss (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                   struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

/**
 * Handles a received request message by checking for the requested message and forwarding it back
 * if the message was found.
 * (this can also cause this peer to send a new request instead of only forwarding the received one)
 *
 * TODO: Requests can cause exponentially more requests!
 *
 * @param room Room of the message
 * @param tunnel Receiving connection
 * @param message REQUEST-Message
 * @param hash Hash of the message
 */
void
recv_message_request (struct GNUNET_MESSENGER_SrvRoom *room, struct GNUNET_MESSENGER_SrvTunnel *tunnel,
                      struct GNUNET_MESSENGER_Message *message, const struct GNUNET_HashCode *hash);

#endif //GNUNET_SERVICE_MESSENGER_MESSAGE_RECV_H
