/* -*- Mode: c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GRTCPSINK_H_
#define _GRTCPSINK_H_

#include <vector>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>

using std::vector;

/*
 * versions prior to gcc-3.x croak on volatile asm in templates,
 * hence the following workaround
 */
static inline unsigned long int _local_htonl (unsigned long int hostlong)
{
  return htonl (hostlong);
}

static inline unsigned short int _local_htons (unsigned short int hostshort)
{
  return htons (hostshort);
}

template <class iType>
class GrTCPSink : public VrSink<iType>
{
public:
  GrTCPSink (int port);
  virtual ~GrTCPSink ();

  
  virtual const char *name () { return "GrTCPSink"; }
  virtual int work3(VrSampleRange output, 
		    VrSampleRange inputs[], void *i[]);

protected:
  int			listen_socket;
  std::vector<int>  	connection;	// open sockets

};

template <class iType>
GrTCPSink<iType>::GrTCPSink (int port)
{
  struct sockaddr_in server_addr;

  
  listen_socket = socket (AF_INET, SOCK_STREAM, 0);
  if (listen_socket == -1){
    perror ("GrTPCSink: socket");
    exit (1);
  }

  memset (&server_addr, 0, sizeof (server_addr));
  server_addr.sin_family = AF_INET;
  server_addr.sin_addr.s_addr = _local_htonl (INADDR_ANY);
  server_addr.sin_port = _local_htons (port);

  if (bind (listen_socket, (struct sockaddr *) &server_addr,
	    sizeof (server_addr)) == -1){
    perror ("GrTCPSink: bind");
    exit (1);
  }

  if (listen (listen_socket, 5) == -1){
    perror ("GrTCPSink: listen");
    exit (1);
  }

  long file_flags = fcntl (listen_socket, F_GETFL);
  if (fcntl (listen_socket, F_SETFL, file_flags | O_NONBLOCK) == -1){
    perror ("GrTCPSink: fcntl / O_NONBLOCK");
    exit (1);
  }
}

template<class iType> int
GrTCPSink<iType>::work3(VrSampleRange output, 
			 VrSampleRange inputs[], void *ai[]) 
{
  iType *in = ((iType **)ai)[0];

  /*
   * We changed this so that it HANGS when there are no TCP
   * connections, rather than throwing away samples.  This will
   * hang the entire GNU Radio pipeline (in a tight loop
   * doing accept() calls).  We should add a significant
   * delay here, perhaps a select, which would stop burning
   * CPU time while waiting.  FIXME.  
   */
  do {
    // 
    // check for new connections
    // (FIXME, do this only every now and then, say 100 ms,
    //  to reduce number of system calls.)
    //
    int	new_socket = accept (listen_socket, 0, 0);
    if (new_socket == -1){
      if (errno == EAGAIN){
	// no new connection, this is OK
      }
      else {
	perror ("GrTCPSink: accept");
	exit (1);
      }
    }
    else {	// got a new connection
      fprintf (stderr, "GrTCPSink: new connection\n");
      connection.push_back (new_socket);
    }

    //
    // handle normal output
    //
    
  } while (connection.size () == 0);

  /*
   * Before the above "while loop" was added, this test would discard
   * any samples that arrived when no TCP connections are currently
   * connected.  Now it's a no-op.
   */
  if (connection.size () == 0)
  	// nothing to do, return success
    return output.size;


  long	nbytes_to_write = output.size * sizeof (iType);

  // iterate over all current connections

  vector<int>::iterator p = connection.begin ();

  while (p != connection.end ()){
    //
    // write whole mess
    //
    long	 r;
    int		flags = 0;
#ifdef MSG_NOSIGNAL
    flags = MSG_NOSIGNAL;
#endif
    if ((r = send (*p, in, nbytes_to_write, flags)) != nbytes_to_write){
      if (r == -1 && errno == EPIPE){
	// other end was closed...
	fprintf (stderr, "GrTCPSink: closing connection\n");
	close (*p);
	p = connection.erase (p);
      }
      else {
	perror ("GrTCPSink: send");
      }
    }
    else
      p++;
  }

  return output.size;
}


template <class iType>
GrTCPSink<iType>::~GrTCPSink ()
{
  vector<int>::iterator p = connection.begin ();
  while (p != connection.end ()){
    close (*p);
    p++;
  }

  close (listen_socket);
}


#endif // _GRTCPSINK_H_
