/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <mc4020_source.h>
#include <gr_io_signature.h>
#include <mc4020.h>
#include <gr_pagesize.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>
#include <stdexcept>
#include <algorithm>
#include <gri_add_const_ss.h>


// FIXME ought to be configurable
#define	MC4020_BUFFER_SIZE	(16L << 20)	// 16 MB

static const bool VERBOSE_DEBUG = false;

static bool
double_mmap (int fd, unsigned long bufsize, void **bufptr)
{
  int	fdz;
  char	*start = 0;

  fdz = open ("/dev/zero", O_RDONLY);
  if (fdz < 0){
    perror ("/dev/zero");
    return false;
  }

  start = (char *) mmap (0, bufsize * 2, PROT_NONE, MAP_SHARED, fdz, 0);
  if (start == MAP_FAILED || munmap (start, bufsize * 2) == -1){
    perror ("mc4020_source: could not allocate mmap buffer");
    return false;
  }
  close (fdz);

  if ((mmap (start, bufsize, PROT_READ,
	     MAP_FIXED | MAP_SHARED, fd, 0)) == MAP_FAILED
      || (mmap (start + bufsize, bufsize, PROT_READ,
		MAP_FIXED | MAP_SHARED, fd, 0)) == MAP_FAILED){
    perror ("mc4020_source: could not mmap ADC buffer");
    return false;
  }
    
  *bufptr = start;
  return true;
}

mc4020_source_sptr
mc4020_make_source (double sampling_freq,
		    unsigned long config_bitmask,
		    const std::string dev)
{
  return mc4020_source_sptr (new mc4020_source (sampling_freq, config_bitmask, dev));
}

mc4020_source::mc4020_source (double sampling_freq,
			      unsigned long config_bitmask, const std::string dev)
  : gr_sync_block ("mc4020_source",
		   gr_make_io_signature (0, 0, 0),
		   gr_make_io_signature (1, 1, sizeof (short))),
    d_samples_per_page (gr_pagesize () / sizeof (short)),
    d_device_fd (-1), d_total_lost (0), d_buf (0)
{
  memset (&d_status, 0, sizeof (d_status));

  struct mc4020_config	c;

  d_buffersize_pages = MC4020_BUFFER_SIZE / gr_pagesize ();

  if ((d_device_fd = open (dev.c_str (), O_RDONLY)) < 0) {
    perror (dev.c_str ());
    throw std::runtime_error ("mc4020_source");
  }

  if ((config_bitmask & MCC_CLK_MASK) == MCC_CLK_INTERNAL)
    c.scan_rate = (unsigned long) sampling_freq;
  else
    c.scan_rate = 2;		// minimum divisor

  config_bitmask = (config_bitmask & ~MCC_ASRC_MASK) | MCC_ASRC_BNC;	// ensure some sanity

  if ((config_bitmask & (MCC_CH0_EN | MCC_CH1_EN | MCC_CH2_EN | MCC_CH3_EN)) == 0){
    fprintf (stderr, "mc4020_source: you must enable at least one channel (MCC_CH?_EN)\n");
    goto bail;
  }

  c.bitmask = config_bitmask;

  if (ioctl (d_device_fd, GIOCSETCONFIG, &c) < 0){
    perror ("mc4020_source: failed to set configuration (GIOCSETCONFIG)");
    goto bail;
  }

  if (ioctl (d_device_fd, GIOCSETBUFSIZE, d_buffersize_pages * gr_pagesize ()) < 0) {
    fprintf (stderr, "buffersize = %ld (%#lx)\n", MC4020_BUFFER_SIZE, MC4020_BUFFER_SIZE);
    perror("mc4020_source: failed to set buffersize (GIOCSETBUFSIZE)");
    goto bail;
  }

  if (!double_mmap (d_device_fd, d_buffersize_pages * gr_pagesize (), (void **) &d_buf))
    goto bail;

  if (ioctl (d_device_fd, GIOCSTART) < 0){
    perror ("mc4020_source: GIOCSTART failed");
    goto bail;
  }

  // we set the output multiple high to reduce the number of system calls (ioctls)
  set_output_multiple (2 * (1L << 20) / sizeof (short));
  return;

 bail:
  close (d_device_fd);
  throw std::runtime_error ("mc4020_source");
}


mc4020_source::~mc4020_source ()
{
  if (ioctl (d_device_fd, GIOCSTOP) < 0)
    perror ("mc4020_source: GIOCSTOP failed");

  if (munmap (d_buf, d_buffersize_pages * gr_pagesize ()) < 0)
    perror ("mc4020_source: munmap failed");
  
  close (d_device_fd);
}

int
mc4020_source::work (int noutput_items,
		     gr_vector_const_void_star &input_items,
		     gr_vector_void_star &output_items)
{
  int	npages = noutput_items / d_samples_per_page;
  int	page_size = gr_pagesize ();
  short	*p;
  short	*out = (short *) output_items[0];
  
  static bool first = true;
  if (VERBOSE_DEBUG && first){
    printf ("\n index             num  lost\n");
    first = false;
  }

  int ndone = 0;
  while (ndone < npages){

    // ask driver how many pages are available, and where to they start

    if (ioctl (d_device_fd, GIOCSETGETSTATUS, &d_status) < 0){
      perror ("mc4020_source: GIOCSETGETSTATUS");
      return -1;	// trouble
    }

    if (VERBOSE_DEBUG)
      printf ("a %6d\t%6d\t%d  ", d_status.index, d_status.num, d_status.lost);

    if (d_status.lost){
      d_total_lost++;
      if (1)
	fprintf (stderr, "O");
    }

    if (d_status.num){					// num pages are available

      p = (short *) &d_buf[d_status.index * page_size];	// they start here

      int n = std::min ((int) d_status.num, npages - ndone); // # of pages we're using

      if (VERBOSE_DEBUG)
	printf ("n = %4d\n", n);
      
      // next call to giocsetgetstatus tells driver we've used this many pages
      d_status.num = n;	

      int nshorts = n * page_size / sizeof (short);
      
      // copy to output buffer, converting from straight binary to 2's complement
      gri_add_const_ss (out, p, nshorts, -0x0800);

      out += nshorts;

      ndone += n;
    }
    else {	// d_status.num == 0
      /* fprintf (stderr, d_status.lost ? "z" : "Z"); */
    }
  }

  return noutput_items;
}
