/*
 * Copyright (C) 2000-2011 Free Software Foundation, Inc.
 *
 * Author: Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * The GnuTLS is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 3 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 *
 */

#ifndef GNUTLS_INT_H
#define GNUTLS_INT_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stddef.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <limits.h>
#include <stdint.h>

#ifdef NO_SSIZE_T
#define HAVE_SSIZE_T
typedef int ssize_t;
#endif

#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <time.h>
#include <u64.h> /* gnulib for uint64_t */

#ifdef HAVE_LIBNETTLE
# include <nettle/memxor.h>
#else
# include <gl/memxor.h>
# define memxor gl_memxor
#endif

/* some systems had problems with long long int, thus,
 * it is not used.
 */
typedef struct
{
  unsigned char i[8];
} uint64;

#include <gnutls/gnutls.h>
#include <gnutls/abstract.h>
#include <system.h>

/*
 * They are not needed any more. You can simply enable
 * the gnutls_log callback to get error descriptions.

#define BUFFERS_DEBUG
#define WRITE_DEBUG
#define READ_DEBUG
#define HANDSHAKE_DEBUG // Prints some information on handshake 
#define COMPRESSION_DEBUG
#define DEBUG
*/

/* The size of a handshake message should not
 * be larger than this value.
 */
#define MAX_HANDSHAKE_PACKET_SIZE 48*1024

#define TLS_MAX_SESSION_ID_SIZE 32

/* The maximum digest size of hash algorithms. 
 */
#define MAX_HASH_SIZE 64
#define MAX_CIPHER_BLOCK_SIZE 16
#define MAX_CIPHER_KEY_SIZE 32

#define MAX_USERNAME_SIZE 128
#define MAX_SERVER_NAME_SIZE 128

#define SESSION_TICKET_KEY_NAME_SIZE 16
#define SESSION_TICKET_KEY_SIZE 16
#define SESSION_TICKET_IV_SIZE 16
#define SESSION_TICKET_MAC_SECRET_SIZE 32

#define AEAD_EXPLICIT_DATA_SIZE 8
#define AEAD_IMPLICIT_DATA_SIZE 4

#define GNUTLS_MASTER_SIZE 48
#define GNUTLS_RANDOM_SIZE 32

/* TLS Extensions */
/* we can receive up to MAX_EXT_TYPES extensions.
 */
#define MAX_EXT_TYPES 32

  /**
   * gnutls_ext_parse_type_t:
   * @GNUTLS_EXT_NONE: Never parsed
   * @GNUTLS_EXT_ANY: Any extension type.
   * @GNUTLS_EXT_APPLICATION: Application extension.
   * @GNUTLS_EXT_TLS: TLS-internal extension.
   * @GNUTLS_EXT_MANDATORY: Extension parsed even if resuming (or extensions are disabled).
   *
   * Enumeration of different TLS extension types.  This flag
   * indicates for an extension whether it is useful to application
   * level or TLS level only.  This is (only) used to parse the
   * application level extensions before the "client_hello" callback
   * is called.
   */
  typedef enum
  {
    GNUTLS_EXT_ANY = 0,
    GNUTLS_EXT_APPLICATION = 1,
    GNUTLS_EXT_TLS = 2,
    GNUTLS_EXT_MANDATORY = 3,
    GNUTLS_EXT_NONE = 4
  } gnutls_ext_parse_type_t;


/* expire time for resuming sessions */
#define DEFAULT_EXPIRE_TIME 3600

typedef enum transport_t
{
  GNUTLS_STREAM,
  GNUTLS_DGRAM
} transport_t;

/* the maximum size of encrypted packets */
#define IS_DTLS(session) (session->internals.transport == GNUTLS_DGRAM)

#define DEFAULT_MAX_RECORD_SIZE 16384
#define TLS_RECORD_HEADER_SIZE 5
#define DTLS_RECORD_HEADER_SIZE (TLS_RECORD_HEADER_SIZE+8)
#define RECORD_HEADER_SIZE(session) (IS_DTLS(session) ? DTLS_RECORD_HEADER_SIZE : TLS_RECORD_HEADER_SIZE)
#define MAX_RECORD_HEADER_SIZE DTLS_RECORD_HEADER_SIZE

#define MAX_RECORD_SEND_SIZE(session) (IS_DTLS(session)?((size_t)session->internals.dtls.mtu-DTLS_RECORD_HEADER_SIZE):(size_t)session->security_parameters.max_record_send_size)
#define MAX_RECORD_RECV_SIZE(session) ((size_t)session->security_parameters.max_record_recv_size)
#define MAX_PAD_SIZE 255
#define EXTRA_COMP_SIZE 2048
#define MAX_RECORD_OVERHEAD (MAX_CIPHER_BLOCK_SIZE/*iv*/+MAX_PAD_SIZE+EXTRA_COMP_SIZE+MAX_HASH_SIZE/*MAC*/)
#define MAX_RECV_SIZE(session) (MAX_RECORD_OVERHEAD+MAX_RECORD_RECV_SIZE(session)+RECORD_HEADER_SIZE(session))

#define TLS_HANDSHAKE_HEADER_SIZE 4
#define DTLS_HANDSHAKE_HEADER_SIZE (TLS_HANDSHAKE_HEADER_SIZE+8)
#define HANDSHAKE_HEADER_SIZE(session) (IS_DTLS(session) ? DTLS_HANDSHAKE_HEADER_SIZE : TLS_HANDSHAKE_HEADER_SIZE)
#define MAX_HANDSHAKE_HEADER_SIZE DTLS_HANDSHAKE_HEADER_SIZE

/* This is the maximum handshake message size we send without
   fragmentation. This currently ignores record layer overhead. */
#define DTLS_DEFAULT_MTU 1200

/* the maximum size of the DTLS cookie */
#define DTLS_MAX_COOKIE_SIZE 32

/* The maximum number of HELLO_VERIFY_REQUEST messages the client
   processes before aborting. */
#define MAX_HANDSHAKE_HELLO_VERIFY_REQUESTS 5

/* defaults for verification functions
 */
#define DEFAULT_VERIFY_DEPTH 32
#define DEFAULT_VERIFY_BITS 16*1024

#include <gnutls_mem.h>

#define MEMSUB(x,y) ((ssize_t)((ptrdiff_t)x-(ptrdiff_t)y))

#define DECR_LEN(len, x) do { len-=x; if (len<0) {gnutls_assert(); return GNUTLS_E_UNEXPECTED_PACKET_LENGTH;} } while (0)
#define DECR_LENGTH_RET(len, x, RET) do { len-=x; if (len<0) {gnutls_assert(); return RET;} } while (0)
#define DECR_LENGTH_COM(len, x, COM) do { len-=x; if (len<0) {gnutls_assert(); COM;} } while (0)

#define HASH2MAC(x) ((gnutls_mac_algorithm_t)x)

#define GNUTLS_POINTER_TO_INT(_) ((int) GNUTLS_POINTER_TO_INT_CAST (_))
#define GNUTLS_INT_TO_POINTER(_) ((void*) GNUTLS_POINTER_TO_INT_CAST (_))

typedef unsigned char opaque;
typedef struct
{
  opaque pint[3];
} uint24;

#include <gnutls_mpi.h>

typedef enum handshake_state_t
{ STATE0 = 0, STATE1, STATE2,
  STATE3, STATE4, STATE5,
  STATE6, STATE7, STATE8, STATE9, STATE11 = 11,
  STATE20 = 20, STATE21, STATE22,
  STATE30 = 30, STATE31, STATE40 = 40, STATE41, STATE50 = 50,
  STATE60 = 60, STATE61, STATE62, STATE70, STATE71
} handshake_state_t;

#include <gnutls_str.h>

/* This is the maximum number of algorithms (ciphers or macs etc).
 * keep it synced with GNUTLS_MAX_ALGORITHM_NUM in gnutls.h
 */
#define MAX_ALGOS GNUTLS_MAX_ALGORITHM_NUM

typedef enum extensions_t
{
  GNUTLS_EXTENSION_SERVER_NAME = 0,
  GNUTLS_EXTENSION_MAX_RECORD_SIZE = 1,
  GNUTLS_EXTENSION_CERT_TYPE = 9,
  GNUTLS_EXTENSION_SUPPORTED_ECC = 10,
  GNUTLS_EXTENSION_SUPPORTED_ECC_PF = 11,
  GNUTLS_EXTENSION_SRP = 12,
  GNUTLS_EXTENSION_SIGNATURE_ALGORITHMS = 13,
  GNUTLS_EXTENSION_SESSION_TICKET = 35,
  GNUTLS_EXTENSION_SAFE_RENEGOTIATION = 65281   /* aka: 0xff01 */
} extensions_t;

typedef enum
{ CIPHER_STREAM, CIPHER_BLOCK } cipher_type_t;

#define RESUME_TRUE 0
#define RESUME_FALSE -1

/* Record Protocol */
typedef enum content_type_t
{
  GNUTLS_CHANGE_CIPHER_SPEC = 20, GNUTLS_ALERT,
  GNUTLS_HANDSHAKE, GNUTLS_APPLICATION_DATA,
} content_type_t;


#define GNUTLS_PK_ANY (gnutls_pk_algorithm_t)-1
#define GNUTLS_PK_NONE (gnutls_pk_algorithm_t)-2

/* Message buffers (mbuffers) structures */

/* this is actually the maximum number of distinct handshake
 * messages that can arrive in a single flight
 */
#define MAX_HANDSHAKE_MSGS 6
typedef struct
{
  /* Handshake layer type and sequence of message */
  gnutls_handshake_description_t htype;
  uint32_t length;

  /* valid in DTLS */
  uint16_t sequence;

  /* indicate whether that message is complete.
   * complete means start_offset == 0 and end_offset == length
   */
  uint32_t start_offset;
  uint32_t end_offset;
  
  opaque header[MAX_HANDSHAKE_HEADER_SIZE];
  int header_size;

  gnutls_buffer_st data;
} handshake_buffer_st;

typedef struct mbuffer_st
{
  /* when used in mbuffer_head_st */
  struct mbuffer_st *next;
  struct mbuffer_st *prev;

  /* msg->size - mark = number of bytes left to process in this
     message. Mark should only be non-zero when this buffer is the
     head of the queue. */
  size_t mark;


  /* the data */
  gnutls_datum_t msg;
  size_t maximum_size;

  /* used during fill in, to separate header from data
   * body. */
  unsigned int user_mark;

  /* Filled in by record layer on recv:
   * type, record_sequence
   */

  /* record layer content type */
  content_type_t type;

  /* record layer sequence */
  uint64 record_sequence;

  /* Filled in by handshake layer on send:
   * type, epoch, htype, handshake_sequence
   */

  /* Record layer epoch of message */
  uint16_t epoch;

  /* Handshake layer type and sequence of message */
  gnutls_handshake_description_t htype;
  uint16_t handshake_sequence;
} mbuffer_st;

typedef struct mbuffer_head_st
{
  mbuffer_st *head;
  mbuffer_st *tail;

  unsigned int length;
  size_t byte_length;
} mbuffer_head_st;

/* Store & Retrieve functions defines: 
 */

typedef struct auth_cred_st
{
  gnutls_credentials_type_t algorithm;

  /* the type of credentials depends on algorithm 
   */
  void *credentials;
  struct auth_cred_st *next;
} auth_cred_st;

struct gnutls_key_st
{
  /* For ECDH KX */
  gnutls_pk_params_st ecdh_params;
  bigint_t ecdh_x;
  bigint_t ecdh_y;

  /* For DH KX */
  gnutls_datum_t key;
  bigint_t KEY;
  bigint_t client_Y;
  bigint_t client_g;
  bigint_t client_p;
  bigint_t dh_secret;
  /* for SRP */
  bigint_t A;
  bigint_t B;
  bigint_t u;
  bigint_t b;
  bigint_t a;
  bigint_t x;
  /* RSA: e, m
   */
  bigint_t rsa[2];

  /* this is used to hold the peers authentication data 
   */
  /* auth_info_t structures SHOULD NOT contain malloced 
   * elements. Check gnutls_session_pack.c, and gnutls_auth.c.
   * Rememember that this should be calloced!
   */
  void *auth_info;
  gnutls_credentials_type_t auth_info_type;
  int auth_info_size;           /* needed in order to store to db for restoring 
                                 */
  uint8_t crypt_algo;

  auth_cred_st *cred;           /* used to specify keys/certificates etc */

  int certificate_requested;
  /* some ciphersuites use this
   * to provide client authentication.
   * 1 if client auth was requested
   * by the peer, 0 otherwise
   *** In case of a server this
   * holds 1 if we should wait
   * for a client certificate verify
   */
};
typedef struct gnutls_key_st *gnutls_key_st;


struct record_state_st;
typedef struct record_state_st record_state_st;

struct record_parameters_st;
typedef struct record_parameters_st record_parameters_st;

/* STATE (cont) */

#include <gnutls_hash_int.h>
#include <gnutls_cipher_int.h>
#include <gnutls_compress.h>

typedef struct
{
  uint8_t suite[2];
} cipher_suite_st;

typedef struct
{
  uint8_t hash_algorithm;
  uint8_t sign_algorithm;       /* pk algorithm actually */
} sign_algorithm_st;

/* This structure holds parameters got from TLS extension
 * mechanism. (some extensions may hold parameters in auth_info_t
 * structures also - see SRP).
 */

#define MAX_SIGNATURE_ALGORITHMS 16
#define MAX_SIGN_ALGO_SIZE (2 + MAX_SIGNATURE_ALGORITHMS * 2)

#define MAX_VERIFY_DATA_SIZE 36 /* in SSL 3.0, 12 in TLS 1.0 */

/* auth_info_t structures now MAY contain malloced 
 * elements.
 */

/* This structure and auth_info_t, are stored in the resume database,
 * and are restored, in case of resume.
 * Holds all the required parameters to resume the current 
 * session.
 */

/* if you add anything in Security_Parameters struct, then
 * also modify CPY_COMMON in gnutls_constate.c. 
 */

/* Note that the security parameters structure is set up after the
 * handshake has finished. The only value you may depend on while
 * the handshake is in progress is the cipher suite value.
 */
typedef struct
{
  int entity; /* GNUTLS_SERVER or GNUTLS_CLIENT */
  gnutls_kx_algorithm_t kx_algorithm;

  /* The epoch used to read and write */
  uint16_t epoch_read;
  uint16_t epoch_write;

  /* The epoch that the next handshake will initialize. */
  uint16_t epoch_next;

  /* The epoch at index 0 of record_parameters. */
  uint16_t epoch_min;

  /* this is the ciphersuite we are going to use 
   * moved here from internals in order to be restored
   * on resume;
   */
  cipher_suite_st current_cipher_suite;
  gnutls_compression_method_t compression_method;
  opaque master_secret[GNUTLS_MASTER_SIZE];
  opaque client_random[GNUTLS_RANDOM_SIZE];
  opaque server_random[GNUTLS_RANDOM_SIZE];
  opaque session_id[TLS_MAX_SESSION_ID_SIZE];
  uint8_t session_id_size;
  time_t timestamp;

  /* The send size is the one requested by the programmer.
   * The recv size is the one negotiated with the peer.
   */
  uint16_t max_record_send_size;
  uint16_t max_record_recv_size;
  /* holds the negotiated certificate type */
  gnutls_certificate_type_t cert_type;
  gnutls_ecc_curve_t ecc_curve; /* holds the first supported ECC curve requested by client */
  gnutls_protocol_t version;    /* moved here */

  /* FIXME: The following are not saved in the session storage
   * for session resumption.
   */

  /* Used by extensions that enable supplemental data: Which ones
   * do that? Do they belong in security parameters?
   */
  int do_recv_supplemental, do_send_supplemental;
} security_parameters_st;

struct record_state_st
{
  gnutls_datum_t mac_secret;
  gnutls_datum_t IV;
  gnutls_datum_t key;
  auth_cipher_hd_st cipher_state;
  comp_hd_t compression_state;
  uint64 sequence_number;
};

/* These are used to resolve relative epochs. These values are just
   outside the 16 bit range to prevent off-by-one errors. An absolute
   epoch may be referred to by its numeric id in the range
   0x0000-0xffff. */
#define EPOCH_READ_CURRENT  70000
#define EPOCH_WRITE_CURRENT 70001
#define EPOCH_NEXT          70002

struct record_parameters_st
{
  uint16_t epoch;
  int initialized;

  gnutls_cipher_algorithm_t cipher_algorithm;
  gnutls_mac_algorithm_t mac_algorithm;
  gnutls_compression_method_t compression_algorithm;

  record_state_st read;
  record_state_st write;
  
  /* Whether this state is in use, i.e., if there is
     a pending handshake message waiting to be encrypted
     under this epoch's parameters.
   */
  int usage_cnt;
};

typedef struct
{
  unsigned int priority[MAX_ALGOS];
  unsigned int algorithms;
} priority_st;

typedef enum
{
  SR_DISABLED,
  SR_UNSAFE,
  SR_PARTIAL,
  SR_SAFE
} safe_renegotiation_t;

/* For the external api */
struct gnutls_priority_st
{
  priority_st cipher;
  priority_st mac;
  priority_st kx;
  priority_st compression;
  priority_st protocol;
  priority_st cert_type;
  priority_st sign_algo;
  priority_st supported_ecc;

  /* to disable record padding */
  int no_padding:1;
  int allow_large_records:1;
  safe_renegotiation_t sr;
  int ssl3_record_version:1;
  int additional_verify_flags;
};


/* DH and RSA parameters types.
 */
typedef struct gnutls_dh_params_int
{
  /* [0] is the prime, [1] is the generator.
   */
  bigint_t params[2];
} dh_params_st;

typedef struct
{
  gnutls_dh_params_t dh_params;
  int free_dh_params;
  gnutls_rsa_params_t rsa_params;
  int free_rsa_params;
} internal_params_st;

#define DTLS_RECORD_WINDOW_SIZE 64

/* DTLS session state
 */
typedef struct
{
  /* HelloVerifyRequest DOS prevention cookie */
  opaque  cookie[DTLS_MAX_COOKIE_SIZE];
  uint8_t cookie_len;

  /* For DTLS handshake fragmentation and reassembly. */
  uint16_t hsk_write_seq;
  unsigned int hsk_read_seq;
  uint16_t mtu;

  /* a flight transmission is in process */
  unsigned int flight_init:1;
  /* whether this is the last flight in the protocol  */
  unsigned int last_flight:1;
  unsigned int retrans_timeout;
  unsigned int total_timeout;

  unsigned int hsk_hello_verify_requests;
  
  uint64_t record_sw[DTLS_RECORD_WINDOW_SIZE];
  unsigned int record_sw_size;

  /* non blocking stuff variables */
  unsigned int blocking:1;
  /* starting time of current handshake */
  time_t handshake_start_time;
  /* time in seconds of the last handshake call */
  time_t handshake_last_call;
  /* The actual retrans_timeout for the next message (e.g. doubled or so) */
  unsigned int actual_retrans_timeout;

  /* timers to handle async handshake after gnutls_handshake()
   * has terminated. Required to handle retransmissions.
   */
  time_t async_term;
  
  /* last retransmission triggered by record layer */
  time_t last_retransmit;
} dtls_st;


typedef union
{
  void *ptr;
  uint32_t num;
} extension_priv_data_t;

typedef struct
{
  /* holds all the data received by the record layer */
  mbuffer_head_st record_buffer; 

  int handshake_hash_buffer_prev_len;           /* keeps the length of handshake_hash_buffer, excluding
                                                 * the last received message */
  gnutls_buffer_st handshake_hash_buffer;       /* used to keep the last received handshake 
                                                 * message */
  int resumable:1;              /* TRUE or FALSE - if we can resume that session */
  handshake_state_t handshake_state;    /* holds
                                         * a number which indicates where
                                         * the handshake procedure has been
                                         * interrupted. If it is 0 then
                                         * no interruption has happened.
                                         */

  int invalid_connection:1;     /* true or FALSE - if this session is valid */

  int may_not_read:1;           /* if it's 0 then we can read/write, otherwise it's forbiden to read/write
                                 */
  int may_not_write:1;
  int read_eof:1;               /* non-zero if we have received a closure alert. */

  int last_alert;               /* last alert received */

  /* The last handshake messages sent or received.
   */
  int last_handshake_in;
  int last_handshake_out;

  /* priorities */
  struct gnutls_priority_st priorities;

  /* resumed session */
  int resumed:1;                /* RESUME_TRUE or FALSE - if we are resuming a session */
  security_parameters_st resumed_security_parameters;

  /* These buffers are used in the handshake
   * protocol only. freed using _gnutls_handshake_io_buffer_clear();
   */
  mbuffer_head_st handshake_send_buffer;
  handshake_buffer_st handshake_recv_buffer[MAX_HANDSHAKE_MSGS];
  int handshake_recv_buffer_size;

  /* this buffer holds a record packet -mostly used for
   * non blocking IO.
   */
  mbuffer_head_st record_recv_buffer;   /* buffer holding the record that is currently being received */
  mbuffer_head_st record_send_buffer;   /* holds cached data
                                         * for the gnutls_io_write_buffered()
                                         * function.
                                         */
  size_t record_send_buffer_user_size;  /* holds the
                                         * size of the user specified data to
                                         * send.
                                         */

  int expire_time;              /* after expire_time seconds this session will expire */
  struct mod_auth_st_int *auth_struct;  /* used in handshake packets and KX algorithms */

  /* this is the highest version available
   * to the peer. (advertized version).
   * This is obtained by the Handshake Client Hello 
   * message. (some implementations read the Record version)
   */
  uint8_t adv_version_major;
  uint8_t adv_version_minor;

  /* if this is non zero a certificate request message
   * will be sent to the client. - only if the ciphersuite
   * supports it.
   */
  int send_cert_req;

  /* bits to use for DHE and DHA 
   * use _gnutls_dh_get_prime_bits() and gnutls_dh_set_prime_bits() 
   * to access it.
   */
  uint16_t dh_prime_bits;

  size_t max_handshake_data_buffer_size;

  /* PUSH & PULL functions.
   */
  gnutls_pull_timeout_func pull_timeout_func;
  gnutls_pull_func pull_func;
  gnutls_push_func push_func;
  gnutls_vec_push_func vec_push_func;
  gnutls_errno_func errno_func;
  /* Holds the first argument of PUSH and PULL
   * functions;
   */
  gnutls_transport_ptr_t transport_recv_ptr;
  gnutls_transport_ptr_t transport_send_ptr;

  /* STORE & RETRIEVE functions. Only used if other
   * backend than gdbm is used.
   */
  gnutls_db_store_func db_store_func;
  gnutls_db_retr_func db_retrieve_func;
  gnutls_db_remove_func db_remove_func;
  void *db_ptr;

  /* post client hello callback (server side only)
   */
  gnutls_handshake_post_client_hello_func user_hello_func;

  /* holds the selected certificate and key.
   * use _gnutls_selected_certs_deinit() and _gnutls_selected_certs_set()
   * to change them.
   */
  gnutls_pcert_st *selected_cert_list;
  int selected_cert_list_length;
  struct gnutls_privkey_st *selected_key;
  int selected_need_free:1;

  /* holds the extensions we sent to the peer
   * (in case of a client)
   */
  uint16_t extensions_sent[MAX_EXT_TYPES];
  uint16_t extensions_sent_size;

  /* is 0 if we are to send the whole PGP key, or non zero
   * if the fingerprint is to be sent.
   */
  int pgp_fingerprint;

  /* This holds the default version that our first
   * record packet will have. */
  opaque default_record_version[2];

  void *user_ptr;

  int enable_private;           /* non zero to
                                 * enable cipher suites
                                 * which have 0xFF status.
                                 */

  /* Holds 0 if the last called function was interrupted while
   * receiving, and non zero otherwise.
   */
  int direction;

  /* This callback will be used (if set) to receive an
   * openpgp key. (if the peer sends a fingerprint)
   */
  gnutls_openpgp_recv_key_func openpgp_recv_key_func;

  /* If non zero the server will not advertize the CA's he
   * trusts (do not send an RDN sequence).
   */
  int ignore_rdn_sequence;

  /* This is used to set an arbitary version in the RSA
   * PMS secret. Can be used by clients to test whether the
   * server checks that version. (** only used in gnutls-cli-debug)
   */
  opaque rsa_pms_version[2];

  /* Here we cache the DH or RSA parameters got from the
   * credentials structure, or from a callback. That is to
   * minimize external calls.
   */
  internal_params_st params;

  /* To avoid using global variables, and especially on Windows where
   * the application may use a different errno variable than GnuTLS,
   * it is possible to use gnutls_transport_set_errno to set a
   * session-specific errno variable in the user-replaceable push/pull
   * functions.  This value is used by the send/recv functions.  (The
   * strange name of this variable is because 'errno' is typically
   * #define'd.)
   */
  int errnum;

  /* Function used to perform public-key signing operation during
     handshake.  Used by gnutls_sig.c:_gnutls_tls_sign(), see also
     gnutls_sign_callback_set(). */
  gnutls_sign_func sign_func;
  void *sign_func_userdata;

  /* minimum bits to allow for SRP
   * use gnutls_srp_set_prime_bits() to adjust it.
   */
  uint16_t srp_prime_bits;

  /* A handshake process has been completed */
  int initial_negotiation_completed:1;

  struct
  {
    uint16_t type;
    extension_priv_data_t priv;
    int set:1;
  } extension_int_data[MAX_EXT_TYPES];

  struct
  {
    uint16_t type;
    extension_priv_data_t priv;
    int set:1;
  } resumed_extension_int_data[MAX_EXT_TYPES];
  /* The type of transport protocol; stream or datagram */
  transport_t transport;

  /* DTLS session state */
  dtls_st dtls;

  unsigned int cb_tls_unique_len;
  unsigned char cb_tls_unique[MAX_VERIFY_DATA_SIZE];

  /* If you add anything here, check _gnutls_handshake_internal_state_clear().
   */
} internals_st;

/* Maximum number of epochs we keep around. */
#define MAX_EPOCH_INDEX 16

struct gnutls_session_int
{
  security_parameters_st security_parameters;
  record_parameters_st *record_parameters[MAX_EPOCH_INDEX];
  internals_st internals;
  gnutls_key_st key;
};


/* functions 
 */
void _gnutls_set_current_version (gnutls_session_t session,
                                  gnutls_protocol_t version);
void _gnutls_free_auth_info (gnutls_session_t session);

/* These two macros return the advertized TLS version of
 * the peer.
 */
#define _gnutls_get_adv_version_major( session) \
	session->internals.adv_version_major

#define _gnutls_get_adv_version_minor( session) \
	session->internals.adv_version_minor

#define set_adv_version( session, major, minor) \
	session->internals.adv_version_major = major; \
	session->internals.adv_version_minor = minor

void _gnutls_set_adv_version (gnutls_session_t, gnutls_protocol_t);
gnutls_protocol_t _gnutls_get_adv_version (gnutls_session_t);

int _gnutls_is_secure_mem_null (const void *);

#endif /* GNUTLS_INT_H */
