/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright(C) 2000, 2001 Edgar Denny(e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <mysql.h>
#include "wrap_mysql.h"

static MYSQL *mysql = NULL;

/* connect to the database. */
gboolean
connect_db( char *user, 
            char *password)
{
	g_return_val_if_fail( user, FALSE);
	g_return_val_if_fail( password, FALSE);

	mysql = mysql_init( NULL);
	mysql_options( mysql, MYSQL_OPT_CONNECT_TIMEOUT, "2");
	if (!mysql_real_connect( mysql, "localhost", user, password,
		NULL, 0, NULL, 0))
	{
		fprintf( stderr, "failed to connect to database: Error: %s\n",
			mysql_error( mysql));
		return FALSE;
	}
	return TRUE;
}

gboolean
change_user_db( char *user, 
                char *password, 
                char *db)
{       
	g_return_val_if_fail( user, FALSE);
	g_return_val_if_fail( password, FALSE);
	g_return_val_if_fail( db, FALSE);

	if ( mysql_change_user( mysql, user, password, db)) {
		fprintf(stderr, "Failed to change user, Error: %s\n",
			mysql_error( mysql));
		return FALSE;
	}
	return TRUE;
}

/* an SQL query where the result consists of several rows as
 * a GList. */
GList *
rows_glist_ret_val_query_db( char *query)
{
	GList *ret_list = NULL;
	MYSQL_RES *result = NULL;
	MYSQL_ROW data;
	int num_fields;
	int i;
	char **elm;

	g_return_val_if_fail( query, NULL);

	if ( mysql_real_query( mysql, query, strlen( query))) {
		fprintf( stderr, "For: %s, Error: %s\n", query, 
			mysql_error(mysql));
		return NULL;
	}

	result = mysql_use_result( mysql);

	if ( !result) {
/*		fprintf(stderr, "For: %s, No result. Error: %s\n", 
			query, mysql_error( mysql)); */
		return NULL;
	}

	num_fields = mysql_num_fields( result);

	if ( num_fields == 0) {
/*		fprintf( stderr, "No Fields. Error: %s\n", query); */
		return NULL;
	}

	while (( data = mysql_fetch_row( result))) {
		elm = (char **)g_malloc( num_fields * sizeof(char*));
		for ( i=0; i<num_fields; i++) {
			elm[i] = g_strdup( data[i]);
		}
		ret_list = g_list_append( ret_list, (gpointer)elm);
	}
	mysql_free_result( result);

	return ret_list;
}

/* an SQL query which returns a row from a table. */
char **
row_ret_val_query_db( char *query) 
{
	char **ret = NULL;
	MYSQL_RES *result = NULL;
	MYSQL_ROW data;
	guint num_fields;
	guint i;

	g_return_val_if_fail( query, NULL);

	if ( mysql_real_query( mysql, query, strlen( query))) {
		fprintf(stderr, "For: %s, Error: %s\n", query, 
			mysql_error( mysql));
		return NULL;
	}

	result = mysql_use_result( mysql);

	if ( !result) {
/*		fprintf( stderr, "For: %s, No Result. Error: %s\n", 
			query, mysql_error(mysql)); */
		return NULL;
	}

	num_fields = mysql_num_fields( result);

	if ( num_fields == 0) {
/*		fprintf( stderr, "No Fields. Error: %s\n", query); */
		return NULL;
	}

	ret = (char **)g_malloc(num_fields * sizeof(char *));
	/* there should only be one row. */
	data = mysql_fetch_row( result);
	for ( i=0; i<num_fields; i++) {
		ret[i] = g_strdup( data[i]);
	}

	mysql_free_result( result);

	return ret;
}

/* an SQL query with no result and no return value. */
gboolean
no_ret_val_query_db( char *query) 
{
	gboolean ret = TRUE;
	g_return_val_if_fail( query, FALSE);

	if ( mysql_real_query( mysql, query, strlen(query))) {
		fprintf( stderr, "For: %s, Error: %s\n", query, 
			mysql_error(mysql));
		ret = FALSE;
	}
	g_assert( ret);
	return ret;
}

/* an SQL query where the return value is a single char pointer. */
char *
single_ret_val_query_db( char *query) 
{
	char *ret = NULL;
	MYSQL_RES *result = NULL;
	MYSQL_ROW data;

	g_return_val_if_fail( query, NULL);

	if ( mysql_real_query( mysql, query, strlen(query))) {
		fprintf( stderr, "For: %s, Error: %s\n", query, 
			mysql_error(mysql));
		return NULL;
	}

	result = mysql_use_result( mysql);

	if ( !result) {
/*		fprintf( stderr, "For: %s, No Result. Error: %s\n", query, 
			mysql_error(mysql)); */
		return NULL;
	}

	data = mysql_fetch_row( result);
	if ( !data) {
/*		fprintf( stderr, "For: %s, No Data. Error: %s\n", query, 
			mysql_error(mysql)); */
		return NULL;
	}
	ret = g_strdup( data[0]);

	if ( result) mysql_free_result( result);

	return ret;
}

/* an SQL query where the return value is a GList pointer 
 * of char pointers. */
GList *
get_glist_fields_query_db( char *query)
{
	GList *ret_list = NULL;
	MYSQL_RES *result = NULL;
	MYSQL_ROW data;
	guint num_fields;
	char *elm;

	g_return_val_if_fail( query, NULL);

	if ( mysql_real_query( mysql, query, strlen( query))) {
		fprintf( stderr, "For: %s, Error: %s\n", query, 
			mysql_error( mysql));
		return NULL;
	}

	result = mysql_use_result( mysql);

	if ( !result) {
/*		fprintf( stderr, "For: %s, No Result. Error: %s\n", query, 
			mysql_error(mysql)); */
		return NULL;
	}

	num_fields = mysql_num_fields( result);

	/* there should only be one field, and one or more rows. */
	if ( num_fields != 1 ) {
/*		fprintf( stderr, "Not a single field.  Error: %s\n", query); */
		return NULL;
	}

	while ( ( data = mysql_fetch_row( result))) {
		elm = g_strdup( data[0]);
		ret_list = g_list_append( ret_list, (gpointer)elm);
	}

	mysql_free_result( result);

	return ret_list;
}

void
gnutr_free_row_list( GList *list, 
                     int n)
{
	GList *ptr;
	char **elm;
	int i;

	g_return_if_fail( list);
	g_return_if_fail( n > 0);
	
	for ( ptr = list; ptr; ptr = ptr->next) {
		 elm = (char **)ptr->data;
		 if (elm) {
		 	for ( i=0; i<n; i++) {
				if ( elm[i]) g_free( elm[i]);
			}
		 	g_free( elm);
		 }
	 }
	 g_list_free( list);
}

void
gnutr_free_field_list( GList *list)
{
	GList *ptr;
	char *elm;

	g_return_if_fail( list);
	
	for ( ptr = list; ptr; ptr = ptr->next) {
		 elm = (char *)ptr->data;
		 if (elm) g_free( elm);
	 }
	 g_list_free( list);
}
