/* simo.c

   written by Don Maszle
   22 November 1991

   Copyright (c) 1993.  Don Maszle, Frederic Bois.  All rights reserved.

   -- Revisions -----
     Logfile:  SCCS/s.simo.c
    Revision:  1.14
        Date:  13 Nov 1995
     Modtime:  15:27:43
      Author:  @a
   -- SCCS  ---------

   Output routines for the simulation
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lexerr.h"
#include "simo.h"
#include "modelu.h"

static char vszDefOutFilename[] = "sim.out";
static char vszDefMCOutFilename[] = "simmc.out";


/* ----------------------------------------------------------------------------
   SaveOutputs
*/

void SaveOutputs (PEXPERIMENT pexp, PDOUBLE pdTout)
{
  #define SO_EPSILON (1e-100) /* Smaller values are zeroed  */

  matType1 rgdInterpStates, rgdInterpDeriv;
  int i, j, index;
  PMODELINFO pmod = pexp->pmodelinfo;
  POUTSPEC pos = &pexp->os;
  extern IFN vrgInputs[];    /* Input Function records */

  memcpy (rgdInterpStates, pmod->pdModelVars,
          pmod->nModelVars*sizeof(double));

  CalcDeriv(rgdInterpStates, rgdInterpDeriv, pdTout);

  /* Update output scaling */
  CalcOutputs (rgdInterpStates, rgdInterpDeriv, pdTout);

  for (i = 0; i < pos->nOutputs; i++) {

    /* Save interpolated value if there are still times to output
       for this variable, and if this time is scheduled */

    if (pos->piCurrentOut[i] < pos->pcOutputTimes[i]
        && *pdTout == pos->prgdOutputTimes[i][pos->piCurrentOut[i]]) {
      double dTmp;

      if (IsModelVar(pos->phvar[i]))  /* Use interp'd model value */
        dTmp = rgdInterpStates[ ModelIndex(pos->phvar[i])];

      else { /* Use current parm/input value */
        index = HINDEX(pos->phvar[i]);
        if (IsInput(pos->phvar[i]) && (vrgInputs[index].iType == IFN_SPIKES)) {

#ifdef ndef
        if (pexp->dTime < 1.5) {
          printf("iDosecur = %d\n", vrgInputs[index].iDoseCur);
          printf("nDoses   = %d\n", vrgInputs[index].nDoses);
          printf("time     = %g\n", pexp->dTime);
          printf("out time = %g\n", vrgInputs[index].rgT0s[vrgInputs[index].iDoseCur]);
          printf("magnitude= %g\n", vrgInputs[index].rgMags[vrgInputs[index].iDoseCur]);
        }
#endif
          
          for (j = 0; j < vrgInputs[index].iDoseCur; j++) {
            if (vrgInputs[index].rgT0s[j] == pexp->dTime) {
              dTmp = vrgInputs[index].rgMags[j];
              break;
            }
          }

          if (j == vrgInputs[index].iDoseCur) 
            dTmp = 0.0;
        }
        else
          dTmp = GetVarValue (pos->phvar[i]);
      }

      if (fabs(dTmp) < SO_EPSILON) /* Avoid silly little numbers  */
        dTmp = 0.0;

      pos->prgdOutputVals[i][pos->piCurrentOut[i]++] = dTmp;

    } /* if */
  } /* for */

} /* SaveOutputs */


/* -----------------------------------------------------------------------------
   NextOutputTime

   Returns in pdTout,the next time, > pdTout, at which an variable is
   to be output.
*/

void NextOutputTime (PEXPERIMENT pexp, PDOUBLE pdTout, PINT piOut)
{
  if (pexp->dTime < pexp->dTfinal)
    if (++*piOut < pexp->os.cDistinctTimes)
      *pdTout = pexp->os.rgdDistinctTimes[*piOut];
    else
      *pdTout = pexp->dTfinal;

} /* NextOutputTime */


/* -----------------------------------------------------------------------------
   WriteOneMod

   writes one parameter modification from the list.   Inputs are *not*
   written.
*/

int WriteOneMod (PVOID pData, PVOID pInfo)
{
  PMCVAR pmcvar = (PMCVAR) pData;
  PFILE pfile = (PFILE) pInfo;

  if (!IsInput (pmcvar->hvar))
    fprintf(pfile, "%g\t", pmcvar->dVal);

  return 0;

} /* WriteOneMod */


/* -----------------------------------------------------------------------------
   WriteMCHeader

   Write a tabulated text header with the run number, the list of parameters 
   and outputs.
*/

void WriteMCHeader (PFILE pfileOut, PANALYSIS panal)
{
  long i, j, k;
  PMONTECARLO pmc = &panal->mc;
  OUTSPEC *pos;

  fprintf (pfileOut, "Iter");

  for (i = 0; i < pmc->nParms; i++)
   fprintf (pfileOut, "\t%s", GetVarName(pmc->rgpMCVar[i]->hvar));
  
  /* print the outputs as they come with experiment and time code */
  for (i = 0; i < panal->expGlobal.iExp; i++) {
    pos = &panal->rgpExps[i]->os;
    for (j = 0; j < pos->nOutputs; j++) {
      for (k = 0; k < pos->pcOutputTimes[j]; k++)
         fprintf (pfileOut, "\t%s_%ld.%ld", pos->pszOutputNames[j], i+1, k+1);
    } /* for j */
  } /* for i */

  fprintf (pfileOut, "\n");

  fflush (pfileOut);

} /* WriteMCHeader */


/* -----------------------------------------------------------------------------
   OpenMCFiles

   Open all the files written to be WriteMCOutput()

   Return non-NULL on error;
*/

int OpenMCFiles (PANALYSIS panal)
{
  int iErr = 0;
  PMONTECARLO pmc = &panal->mc;

  if (!pmc->szMCPassFilename) {
    /* If distributed output files not specified, use one file */

    /* Use command line spec if given */
    if (panal->expGlobal.os.bCommandLineSpec)
      pmc->szMCOutfilename = panal->expGlobal.os.szOutfilename;

    else if (!(pmc->szMCOutfilename))  /* Default if none given */
      pmc->szMCOutfilename = vszDefMCOutFilename;

    if (!pmc->pfileMCOut
        && !(pmc->pfileMCOut = fopen (pmc->szMCOutfilename, "w"))) {
      iErr++;
      ReportError (NULL, RE_FATAL | RE_CANNOTOPEN, pmc->szMCOutfilename,
                   "OpenMCFiles()");
    }

    WriteMCHeader (pmc->pfileMCOut, panal);

  } /* if */

  else{
    /* Otherwise, Use 2 output files */
    if (pmc->szMCPassFilename
        && !pmc->pfileMCPass
        && !(pmc->pfileMCPass = fopen (pmc->szMCPassFilename, "w"))) {
      iErr++;
      ReportError (NULL, RE_FATAL | RE_CANNOTOPEN,
                   pmc->szMCPassFilename, "OpenMCFiles()");
    } /* if */

    if (pmc->szMCFailFilename
        && !pmc->pfileMCFail
        && !(pmc->pfileMCFail = fopen (panal->mc.szMCFailFilename, "w"))) {
      iErr++;
      ReportError (NULL, RE_FATAL | RE_CANNOTOPEN,
                   pmc->szMCFailFilename, "OpenMCFiles()");
    } /* if */

    WriteMCHeader (pmc->pfileMCPass, panal);
    WriteMCHeader (pmc->pfileMCFail, panal);

  } /* else */

  return (iErr);

} /* OpenMCFiles */


/* -----------------------------------------------------------------------------
   CloseMCFiles

   Closes output files associated with Monte Carlo and set points runs
*/

void CloseMCFiles (PANALYSIS panal)
{
  if (panal->mc.pfileMCOut) {
    fclose (panal->mc.pfileMCOut);
    printf ("\nWrote results to \"%s\"\n", panal->mc.szMCOutfilename);
  }
  else {
    fclose (panal->mc.pfileMCPass);
    fclose (panal->mc.pfileMCFail);
    printf ("\nWrote results to 2 outputs files\n");
  }

} /* CloseMCFiles */


/* -----------------------------------------------------------------------------
   WriteMCOutput

   Output the parameters for this run and the results of the
   simulation.
*/

void WriteMCOutput (PANALYSIS panal, PMCPREDOUT pmcpredout)
{
  PFILE pfileMC;
  PMONTECARLO pmc = &panal->mc;

  /* Choose Output File.  If separate pass and fail files are
     specified, use them depending on the pass criteria
     else pfileMC is normal MC output and contains both passes and fails
  */
  if (pmc->pfileMCPass && pmc->pfileMCFail) { /* Pass/Fail criteria */
    if (pmcpredout->passflag == 1) {
      panal->mc.nPasses++;
      pfileMC = pmc->pfileMCPass; /* Use pass file */
    }
    else {
      panal->mc.nFails++;
      pfileMC = pmc->pfileMCFail; /* Use fail file */
    }
  } /* if */
  else
    pfileMC = pmc->pfileMCOut;

  /* Write variations, the actual outputs, and the results of the
     simulations (passed through TransformPred) depending on how the
     fOptions flags are set in panal.
  */

  fprintf (pfileMC, "%ld\t", panal->mc.lRun);

  /* Include parameter values for that run */
  if (panal->fOptions & OF_PARMLIST) {
    WriteArray (pfileMC, panal->mc.nParms, panal->mc.rgdParms);
    fprintf (pfileMC, "\t");
  }

  if (panal->fOptions & OF_MCRESULT) {
    /* write the flattened and eventually transformed predictions */
    WriteArray (pfileMC, pmcpredout->nbrdy, pmcpredout->pred);
    fprintf (pfileMC, "\n");
  }

  fflush (pfileMC);

} /* WriteMCOutput */


/* -----------------------------------------------------------------------------
   WriteNormalOutput

   Write the results in the output file. This procedure is
   called only from time to time in order to save storage space
*/

void WriteNormalOutput (PANALYSIS panal, PEXPERIMENT pexp)
{
  int i, j;
  PFILE pfile;
  POUTSPEC posGlo, pos;

  if (!panal) return;

  posGlo = &panal->expGlobal.os;
  pos = &pexp->os;

  if (!posGlo->szOutfilename)
    posGlo->szOutfilename = vszDefOutFilename;

  if (!(posGlo->pfileOut))
    if (!(posGlo->pfileOut = fopen (posGlo->szOutfilename, "w")))
      ReportError (NULL, RE_CANNOTOPEN | RE_FATAL,
               posGlo->szOutfilename, NULL);

  pfile = posGlo->pfileOut;
  fprintf (pfile, "Results of Experiment %d\n\n", pexp->iExp);

  /* Vertical output:  Formatted  Time1    Out_Var1  Out_Var2 ... */

  fprintf (pfile, "Time");

  for (i = 0; i < pos->nOutputs; i++)
    fprintf (pfile, "\t%s", pos->pszOutputNames[i]);
  fprintf (pfile, "\n");

  for (j = 0; j < pos->nOutputs; j++)
    pos->piCurrentOut[j] = 0;

  for (i = 0; i < pos->cDistinctTimes; i++) {
    fprintf (pfile, "%g", pos->rgdDistinctTimes[i]);
    for (j = 0; j < pos->nOutputs; j++) {

      if (pos->piCurrentOut[j] < pos->pcOutputTimes[j]
          && pos->rgdDistinctTimes[i]
          == pos->prgdOutputTimes[j][pos->piCurrentOut[j]])
        fprintf (pfile, "\t%g",
                 pos->prgdOutputVals[j][pos->piCurrentOut[j]++]);

      else
        fprintf (pfile, "\t");

    } /* for */

    fprintf (pfile, "\n");

  } /* for */

  fprintf (pfile, "\n");

} /* WriteNormalOutput */
