package main_test

import (
	"bytes"
	"crypto/ed25519"
	"crypto/rand"
	"crypto/sha512"
	"encoding/binary"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"os"
	"strings"
	"testing"

	"github.com/schanzen/taler-go/pkg/merchant"
	talerutil "github.com/schanzen/taler-go/pkg/util"
	"gopkg.in/ini.v1"
	"gorm.io/driver/sqlite"
	"taler.net/taler-mailbox/internal/gana"
	"taler.net/taler-mailbox/internal/util"
	"taler.net/taler-mailbox/pkg/rest"
)

var a mailbox.Mailbox

var testWalletAlicePriv ed25519.PrivateKey
var testWalletAlice ed25519.PublicKey
var testWalletAliceString string

const merchantConfigResponse = `{
  "currency": "KUDOS",
  "currencies": {
    "KUDOS": {
      "name": "Kudos (Taler Demonstrator)",
      "currency": "KUDOS",
      "num_fractional_input_digits": 2,
      "num_fractional_normal_digits": 2,
      "num_fractional_trailing_zero_digits": 2,
      "alt_unit_names": {
        "0": "ク"
      }
    }
  },
  "exchanges": [
    {
      "master_pub": "F80MFRG8HVH6R9CQ47KRFQSJP3T6DBJ4K1D9B703RJY3Z39TBMJ0",
      "currency": "KUDOS",
      "base_url": "https://exchange.demo.taler.net/"
    }
  ],
  "implementation": "urn:net:taler:specs:taler-merchant:c-reference",
  "name": "taler-merchant",
  "version": "18:0:15"
}`

func executeRequest(req *http.Request) *httptest.ResponseRecorder {
	rr := httptest.NewRecorder()
	a.Router.ServeHTTP(rr, req)
	return rr
}

func checkResponseCode(t *testing.T, expected, actual int) {
	if expected != actual {
		t.Errorf("Expected response code %d, Got %d\n", expected, actual)
	}
}

func TestMain(m *testing.M) {
	cfg, err := ini.Load("test-mailbox.conf")
	if err != nil {
		fmt.Printf("Failed to read config: %v", err)
		os.Exit(1)
	}
	db := sqlite.Open("file::memory:?cache=shared")
	merchServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		var orderResp merchant.PostOrderRequest
		if r.URL.Path == "/config" {
			w.WriteHeader(http.StatusOK)
			w.Write([]byte(merchantConfigResponse))
			return
		}
		if !strings.HasPrefix(r.URL.Path, "/private/orders") {
			fmt.Printf("Expected to request '/private/orders', got: %s\n", r.URL.Path)
			return
		}
		if r.Method == http.MethodPost {
			err := json.NewDecoder(r.Body).Decode(&orderResp)
			if err != nil {
				fmt.Printf("Error %s\n", err)
			}
			jsonResp := fmt.Sprintf("{\"order_id\":\"%s\"}", orderResp.Order.OrderId)
			w.WriteHeader(http.StatusOK)
			w.Write([]byte(jsonResp))
		} else {
			if r.Header.Get("PaidIndicator") == "yes" {
				jsonResp := "{\"order_status\":\"paid\"}"
				w.WriteHeader(http.StatusOK)
				w.Write([]byte(jsonResp))
			} else {
				jsonResp := "{\"order_status\":\"unpaid\", \"taler_pay_uri\": \"somepaytouri\"}"
				w.WriteHeader(http.StatusOK)
				w.Write([]byte(jsonResp))
			}
		}
	}))
	defer merchServer.Close()
	merch := merchant.NewMerchant(merchServer.URL, "supersecret")
	a.Initialize(mailbox.MailboxConfig{
		Version:  "testing",
		Db:       db,
		Merchant: merch,
		Ini:      cfg})
	testWalletAlice, testWalletAlicePriv, _ = ed25519.GenerateKey(nil)
	h := sha512.New()
	h.Write(testWalletAlice)
	testWalletAliceString = util.Base32CrockfordEncode(h.Sum(nil))

	a.Merchant = merchant.NewMerchant(merchServer.URL, "")

	code := m.Run()
	// Purge DB
	a.Db.Where("1 = 1").Delete(&mailbox.InboxEntry{})
	os.Exit(code)
}

func TestEmptyMailbox(t *testing.T) {
	a.Db.Where("1 = 1").Delete(&mailbox.InboxEntry{})
	req, _ := http.NewRequest("GET", "/"+testWalletAliceString, nil)
	response := executeRequest(req)

	checkResponseCode(t, http.StatusNoContent, response.Code)

	body := response.Body.String()
	if body != "" {
		t.Errorf("Expected empty response, Got %s", body)
	}
}

func TestPostMessage(t *testing.T) {
	testMessage := make([]byte, 256)
	a.Db.Where("1 = 1").Delete(&mailbox.InboxEntry{})
	req, _ := http.NewRequest("POST", "/"+testWalletAliceString, bytes.NewReader(testMessage))
	response := executeRequest(req)

	checkResponseCode(t, http.StatusNoContent, response.Code)

	body := response.Body.String()
	if body != "" {
		t.Errorf("Expected empty response, Got %s", body)
	}
}

func TestPostMessagePaid(t *testing.T) {
	testMessage := make([]byte, 256)

	// Make paid
	fee, err := talerutil.ParseAmount("KUDOS:1")
	if err != nil {
		t.Errorf("%v", err)
	}
	a.MessageFee = fee
	a.Db.Where("1 = 1").Delete(&mailbox.InboxEntry{})
	req, _ := http.NewRequest("POST", "/"+testWalletAliceString, bytes.NewReader(testMessage))
	response := executeRequest(req)

	checkResponseCode(t, http.StatusPaymentRequired, response.Code)

	// TODO check QR / payto response

	req, _ = http.NewRequest("POST", "/"+testWalletAliceString, bytes.NewReader(testMessage))
	req.Header.Add("PaidIndicator", "yes")
	response = executeRequest(req)

	checkResponseCode(t, http.StatusPaymentRequired, response.Code)

	body := response.Body.String()
	if body != "" {
		t.Errorf("Expected empty response, Got %s", body)
	}
	a.MessageFee, _ = talerutil.ParseAmount("KUDOS:0")
}

func TestPostThenDeleteMessage(t *testing.T) {
	// testMessage := make([]byte, 256)
	var deletionReq mailbox.MessageDeletionRequest
	numMessagesToPost := (a.MessageResponseLimit + 7)
	testMessages := make([]byte, 256*numMessagesToPost)
	_, _ = rand.Read(testMessages)
	a.Db.Where("1 = 1").Delete(&mailbox.InboxEntry{})

	for i := 0; i < int(numMessagesToPost); i++ {
		testMessage := testMessages[i*256 : (i+1)*256]
		req, _ := http.NewRequest("POST", "/"+testWalletAliceString, bytes.NewReader(testMessage))
		response := executeRequest(req)

		checkResponseCode(t, http.StatusNoContent, response.Code)

		body := response.Body.String()
		if body != "" {
			t.Errorf("Expected empty response, Got %s", body)
		}
	}

	req, _ := http.NewRequest("GET", "/"+testWalletAliceString, nil)
	response := executeRequest(req)

	checkResponseCode(t, http.StatusOK, response.Code)

	if response.Body.Len() != int(256*a.MessageResponseLimit) {
		t.Errorf("Expected response of 25600 bytes, Got %d", response.Body.Len())
	}

	etag := response.Result().Header.Get("ETag")

	if etag == "" {
		t.Errorf("ETag missing!\n")
	}

	// Now  delete 10 messages
	h := sha512.New()
	for i := 0; i < int(a.MessageResponseLimit); i++ {
		h.Write(testMessages[i*256 : (i+1)*256])
	}
	var signed_msg [64 + 4 + 4]byte
	size := signed_msg[0:4]
	binary.BigEndian.PutUint32(size, 64+4+4)
	purp := signed_msg[4:8]
	binary.BigEndian.PutUint32(purp, gana.TALER_SIGNATURE_PURPOSE_MAILBOX_MESSAGES_DELETE)
	checksum := h.Sum(nil)
	copy(signed_msg[8:], checksum)
	sig := ed25519.Sign(testWalletAlicePriv, signed_msg[0:])
	if !ed25519.Verify(testWalletAlice, signed_msg[0:], sig) {
		t.Errorf("Signature invalid!")
	}
	deletionReq.WalletSig = util.Base32CrockfordEncode(sig)
	deletionReq.Count = int(a.MessageResponseLimit)
	deletionReq.Checksum = util.Base32CrockfordEncode(checksum)
	reqJson, _ := json.Marshal(deletionReq)

	hAddress := util.Base32CrockfordEncode(testWalletAlice)
	req, _ = http.NewRequest("DELETE", "/"+hAddress, bytes.NewBuffer(reqJson))
	req.Header.Add("If-Match", etag)
	response = executeRequest(req)

	checkResponseCode(t, http.StatusNoContent, response.Code)

	body := response.Body.String()
	if body != "" {
		t.Errorf("Expected empty response, Got %s", body)
	}

	req, _ = http.NewRequest("GET", "/"+testWalletAliceString, nil)
	response = executeRequest(req)

	checkResponseCode(t, http.StatusOK, response.Code)

	if response.Body.Len() != int(256*7) {
		t.Errorf("Expected response of 256*7 bytes, Got %d", response.Body.Len())
	}
}
