/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  j2s,
  TalerCorebankApiClient,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import {
  createSyncCryptoApi,
  EddsaKeypair,
  WalletApiOperation,
} from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  DbInfo,
  ExchangeService,
  generateRandomPayto,
  GlobalTestState,
  HarnessExchangeBankAccount,
  setupDb,
  WalletClient,
  WalletService,
} from "../harness/harness.js";
import {
  createWalletDaemonWithClient,
  EnvOptions,
  postAmlDecisionNoRules,
  withdrawViaBankV3,
} from "../harness/helpers.js";

interface KycTestEnv {
  commonDb: DbInfo;
  bankClient: TalerCorebankApiClient;
  exchange: ExchangeService;
  exchangeBankAccount: HarnessExchangeBankAccount;
  walletClient: WalletClient;
  walletService: WalletService;
  amlKeypair: EddsaKeypair;
}

async function createKycTestkudosEnvironment(
  t: GlobalTestState,
  coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS")),
  opts: EnvOptions = {},
): Promise<KycTestEnv> {
  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw";
  let exchangePaytoUri = generateRandomPayto(exchangeBankUsername);

  await exchange.addBankAccount("1", {
    accountName: exchangeBankUsername,
    accountPassword: exchangeBankPassword,
    wireGatewayApiBaseUrl: new URL(
      "accounts/exchange/taler-wire-gateway/",
      bank.baseUrl,
    ).href,
    accountPaytoUri: exchangePaytoUri,
  });

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "adminpw",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  const ageMaskSpec = opts.ageMaskSpec;

  if (ageMaskSpec) {
    exchange.enableAgeRestrictions(ageMaskSpec);
    // Enable age restriction for all coins.
    exchange.addCoinConfigList(
      coinConfig.map((x) => ({
        ...x,
        name: `${x.name}-age`,
        ageRestricted: true,
      })),
    );
    // For mixed age restrictions, we also offer coins without age restrictions
    if (opts.mixedAgeRestriction) {
      exchange.addCoinConfigList(
        coinConfig.map((x) => ({ ...x, ageRestricted: false })),
      );
    }
  } else {
    exchange.addCoinConfigList(coinConfig);
  }

  await exchange.modifyConfig(async (config) => {
    config.setString("exchange", "enable_kyc", "yes");

    config.setString("KYC-RULE-R1", "operation_type", "merge");
    config.setString("KYC-RULE-R1", "enabled", "yes");
    config.setString("KYC-RULE-R1", "exposed", "yes");
    config.setString("KYC-RULE-R1", "is_and_combinator", "yes");
    config.setString("KYC-RULE-R1", "threshold", "TESTKUDOS:5");
    config.setString("KYC-RULE-R1", "timeframe", "1d");
    config.setString("KYC-RULE-R1", "next_measures", "M1");

    config.setString("KYC-MEASURE-M1", "check_name", "C1");
    config.setString("KYC-MEASURE-M1", "context", "{}");
    config.setString("KYC-MEASURE-M1", "program", "P1");

    config.setString("AML-PROGRAM-P1", "command", "/bin/true");
    config.setString("AML-PROGRAM-P1", "enabled", "true");
    config.setString("AML-PROGRAM-P1", "description", "this does nothing");
    config.setString("AML-PROGRAM-P1", "fallback", "M1");

    config.setString("KYC-CHECK-C1", "type", "INFO");
    config.setString("KYC-CHECK-C1", "description", "my check!");
    config.setString("KYC-CHECK-C1", "fallback", "M1");
  });

  await exchange.start();

  const cryptoApi = createSyncCryptoApi();
  const amlKeypair = await cryptoApi.createEddsaKeypair({});

  await exchange.enableAmlAccount(amlKeypair.pub, "Alice");

  const walletService = new WalletService(t, {
    name: "wallet",
    useInMemoryDb: true,
  });
  await walletService.start();
  await walletService.pingUntilAvailable();

  const walletClient = new WalletClient({
    name: "wallet",
    unixPath: walletService.socketPath,
    onNotification(n) {
      console.log("got notification", n);
    },
  });
  await walletClient.connect();
  await walletClient.client.call(WalletApiOperation.InitWallet, {
    config: {
      testing: {
        skipDefaults: true,
      },
    },
  });

  console.log("setup done!");

  return {
    commonDb: db,
    exchange,
    amlKeypair,
    walletClient,
    walletService,
    bankClient,
    exchangeBankAccount: {
      accountName: "",
      accountPassword: "",
      accountPaytoUri: "",
      wireGatewayApiBaseUrl: "",
    },
  };
}

export async function runKycPeerPushTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, amlKeypair } =
    await createKycTestkudosEnvironment(t);

  // Origin wallet for the p2p transaction.
  const w0 = await createWalletDaemonWithClient(t, {
    name: "w0",
  });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    bankClient,
    amount: "TESTKUDOS:20",
    exchange: exchange,
    walletClient: w0.walletClient,
  });

  await wres.withdrawalFinishedCond;

  const pushDebitRes = await doPeerPushDebit(t, {
    walletClient: w0.walletClient,
    amount: "TESTKUDOS:10",
    summary: "Test1",
  });

  const prepRes = await walletClient.call(
    WalletApiOperation.PreparePeerPushCredit,
    {
      talerUri: pushDebitRes.talerUri,
    },
  );

  console.log("prepRes", j2s(prepRes));

  await walletClient.call(WalletApiOperation.ConfirmPeerPushCredit, {
    transactionId: prepRes.transactionId,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: prepRes.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.MergeKycRequired,
    },
  });

  const txDet = await walletClient.call(WalletApiOperation.GetTransactionById, {
    transactionId: prepRes.transactionId,
  });

  console.log("tx details", j2s(txDet));

  const kycPaytoHash = txDet.kycPaytoHash;

  t.assertTrue(!!kycPaytoHash);

  await postAmlDecisionNoRules(t, {
    amlPriv: amlKeypair.priv,
    amlPub: amlKeypair.pub,
    exchangeBaseUrl: exchange.baseUrl,
    paytoHash: kycPaytoHash,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: prepRes.transactionId,
    txState: {
      major: TransactionMajorState.Done,
    },
  });
}

/**
 * Initiate a push debit transaction, wait until the transaction
 * is ready.
 */
async function doPeerPushDebit(
  t: GlobalTestState,
  args: {
    walletClient: WalletClient;
    amount: AmountString;
    summary?: string;
  },
): Promise<{
  transactionId: string;
  talerUri: string;
}> {
  const purse_expiration = AbsoluteTime.toProtocolTimestamp(
    AbsoluteTime.addDuration(
      AbsoluteTime.now(),
      Duration.fromSpec({ days: 2 }),
    ),
  );
  const initRet = await args.walletClient.call(
    WalletApiOperation.InitiatePeerPushDebit,
    {
      partialContractTerms: {
        amount: args.amount,
        summary: args.summary ?? "Test P2P Payment",
        purse_expiration,
      },
    },
  );

  await args.walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: initRet.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.Ready,
    },
  });

  const txDet = await args.walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: initRet.transactionId,
    },
  );

  t.assertTrue(txDet.type === TransactionType.PeerPushDebit);
  const talerUri = txDet.talerUri;
  t.assertTrue(!!talerUri);

  return {
    transactionId: initRet.transactionId,
    talerUri,
  };
}

runKycPeerPushTest.suites = ["wallet"];
