/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Codec,
  TranslatedString,
  buildCodecForObject,
  codecForBoolean,
  codecOptionalDefault,
} from "@gnu-taler/taler-util";
import {
  buildStorageKey,
  useLocalStorage,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";

interface Preferences {
  showDebugInfo: boolean;
  allowInsecurePassword: boolean;
  keepSessionAfterReload: boolean;
  testingDialect: boolean;
  preventCompression: boolean;
}

export const codecForPreferences = (): Codec<Preferences> =>
  buildCodecForObject<Preferences>()
    .property(
      "allowInsecurePassword",
      codecOptionalDefault(codecForBoolean(), false),
    )
    .property("showDebugInfo", codecOptionalDefault(codecForBoolean(), false))
    .property("testingDialect", codecOptionalDefault(codecForBoolean(), false))
    .property(
      "keepSessionAfterReload",
      codecOptionalDefault(codecForBoolean(), false),
    )
    .property(
      "preventCompression",
      codecOptionalDefault(codecForBoolean(), false),
    )
    .build("Preferences");

const defaultPreferences: Preferences = {
  allowInsecurePassword: false,
  showDebugInfo: false,
  testingDialect: false,
  keepSessionAfterReload: false,
  preventCompression: false,
};

const PREFERENCES_KEY = buildStorageKey(
  "aml-preferences",
  codecForPreferences(),
);
/**
 * User preferences.
 *
 * @returns tuple of [state, update()]
 */
export function usePreferences(): [
  Readonly<Preferences>,
  <T extends keyof Preferences>(key: T, value: Preferences[T]) => void,
] {
  const { value, update } = useLocalStorage(
    PREFERENCES_KEY,
    defaultPreferences,
  );

  function updateField<T extends keyof Preferences>(k: T, v: Preferences[T]) {
    const newValue = { ...value, [k]: v };
    update(newValue);
  }
  return [value, updateField];
}

export function getAllBooleanPreferences(): Array<keyof Preferences> {
  return [
    "showDebugInfo",
    "allowInsecurePassword",
    "keepSessionAfterReload",
    "testingDialect",
    "preventCompression",
  ];
}

const TALER_SCREEN_ID = 118;

export function getLabelForPreferences(
  k: keyof Preferences,
  i18n: ReturnType<typeof useTranslationContext>["i18n"],
): TranslatedString {
  switch (k) {
    case "showDebugInfo":
      return i18n.str`Show debug info`;
    case "testingDialect":
      return i18n.str`Use testing dialect`;
    case "allowInsecurePassword":
      return i18n.str`Allow Insecure password`;
    case "keepSessionAfterReload":
      return i18n.str`Keep session after reload`;
    case "preventCompression":
      return i18n.str`Don't compress request`;
  }
}
