/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import {
  AccessToken,
  TalerHttpError,
  TalerMerchantManagementResultByMethod,
} from "@gnu-taler/taler-util";
import { buildPaginatedResult } from "@gnu-taler/web-util/browser";
import { useState } from "preact/hooks";
import _useSWR, { SWRHook, mutate } from "swr";
import { useSessionContext } from "../context/session.js";
import { PAGINATED_LIST_REQUEST } from "../utils/constants.js";
const useSWR = _useSWR as unknown as SWRHook;

export interface InstanceBankAccountFilter {}

export function revalidateInstanceAccessTokens() {
  return mutate(
    (key) =>
      Array.isArray(key) && key[key.length - 1] === "useInstanceAccessTokens",
    undefined,
    { revalidate: true },
  );
}
export function useInstanceAccessTokens() {
  const { state, lib } = useSessionContext();

  const [offset, setOffset] = useState<number | undefined>();

  async function fetcher([token, tid]: [AccessToken, number]) {
    return await lib.instance.listAccessTokens(token, {
      limit: PAGINATED_LIST_REQUEST,
      offset: !tid? undefined :String(tid),
      order: "dec",
    });
  }

  const { data, error } = useSWR<
    TalerMerchantManagementResultByMethod<"listAccessTokens">,
    TalerHttpError
  >([state.token, offset, "useInstanceAccessTokens"], fetcher);

  if (error) return error;
  if (data === undefined) return undefined;
  if (data.type !== "ok") return data;

  return buildPaginatedResult(
    data.body.tokens,
    offset,
    setOffset,
    (d) => d.serial,
    PAGINATED_LIST_REQUEST,
  );
}
