/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Duration,
  HttpStatusCode,
  MerchantAuthMethod,
  TanChannel,
} from "@gnu-taler/taler-util";
import {
  undefinedIfEmpty,
  useChallengeHandler,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { AsyncButton } from "../../components/exception/AsyncButton.js";
import {
  FormErrors,
  FormProvider,
} from "../../components/form/FormProvider.js";
import { Input } from "../../components/form/Input.js";
import { InputWithAddon } from "../../components/form/InputWithAddon.js";
import { NotificationCard } from "../../components/menu/index.js";
import { SolveMFAChallenges } from "../../components/SolveMFA.js";
import { useSessionContext } from "../../context/session.js";
import {
  EMAIL_REGEX,
  INSTANCE_ID_REGEX,
  PHONE_JUST_NUMBERS_REGEX,
} from "../../utils/constants.js";
import { Notification } from "../../utils/types.js";

export interface Account {
  id: string;
  name: string;
  password: string;
  repeat: string;
  email: string;
  phone: string;
}

const twoHours = Duration.fromSpec({ hours: 2 });
const twoDays = Duration.fromSpec({ days: 2 });

interface Props {
  onCancel: () => void;
  onCreated: () => void;
}
export function NewAccount({ onCancel, onCreated }: Props): VNode {
  const { i18n } = useTranslationContext();
  const { state: session, lib, logIn, config } = useSessionContext();
  const [notif, setNotif] = useState<Notification | undefined>(undefined);

  const [value, setValue] = useState<Partial<Account>>({});

  const serverRequiresEmail =
    config.mandatory_tan_channels?.indexOf(TanChannel.EMAIL) !== -1;
  const serverRequiresSms =
    config.mandatory_tan_channels?.indexOf(TanChannel.SMS) !== -1;

  const errors = undefinedIfEmpty<FormErrors<Account>>({
    id: !value.id
      ? i18n.str`Required`
      : !INSTANCE_ID_REGEX.test(value.id)
        ? i18n.str`Invalid`
        : undefined,
    name: !value.name ? i18n.str`Required` : undefined,
    password: !value.password ? i18n.str`Required` : undefined,
    repeat: !value.repeat
      ? i18n.str`Required`
      : value.password !== value.repeat
        ? i18n.str`Doesn't match`
        : undefined,
    email: !serverRequiresEmail
      ? undefined
      : !value.email
        ? i18n.str`Required`
        : !EMAIL_REGEX.test(value.email)
          ? i18n.str`Doesn't have the pattern of an email`
          : undefined,
    phone: !serverRequiresSms
      ? undefined
      : !value.phone
        ? i18n.str`Required`
        : !value.phone.startsWith("+")
          ? i18n.str`Should start with +`
          : !PHONE_JUST_NUMBERS_REGEX.test(value.phone)
            ? i18n.str`A phone number consists of numbers only`
            : undefined,
  });

  function valueHandler(s: (d: Partial<Account>) => Partial<Account>): void {
    const next = s(value);
    const v: Account = {
      id: next.id ?? "",
      name: next.name ?? "",
      password: next.password ?? "",
      repeat: next.repeat ?? "",
      email: next.email ?? "",
      phone: next.phone ?? "",
    };
    setValue(v);
  }

  const mfa = useChallengeHandler();
  const [doCreate, repeatCreate] = mfa.withMfaHandler(
    ({ challengeIds, onChallengeRequired }) =>
      async function doCreateImpl() {
        try {
          const id = value.id!;
          const resp = await lib.instance.createInstanceSelfProvision(
            {
              address: {},
              auth: {
                method: MerchantAuthMethod.TOKEN,
                password: value.password!,
              },
              default_pay_delay: Duration.toTalerProtocolDuration(twoHours),
              default_wire_transfer_delay:
                Duration.toTalerProtocolDuration(twoDays),
              id,
              jurisdiction: {},
              name: value.name!,
              use_stefan: true,
              email: value.email,
              phone_number: value.phone,
            },
            {
              tokenValidity: Duration.fromSpec({ months: 6 }),
              challengeIds,
            },
          );

          if (resp.type === "fail") {
            if (resp.case === HttpStatusCode.Accepted) {
              onChallengeRequired(resp.body);
            } else {
              setNotif({
                message: i18n.str`Failed to create account`,
                type: "ERROR",
                description: resp.detail?.hint,
              });
            }
            return;
          }
          if (resp.body) {
            logIn(id, resp.body.access_token);
          }
          onCreated();
        } catch (error) {
          setNotif({
            message: i18n.str`Failed to create account`,
            type: "ERROR",
            description: error instanceof Error ? error.message : String(error),
          });
        }
      },
  );

  if (mfa.pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={mfa.pendingChallenge}
        onCompleted={repeatCreate}
        onCancel={mfa.doCancelChallenge}
      />
    );
  }

  return (
    <Fragment>
      <NotificationCard notification={notif} />

      <div class="columns is-centered" style={{ margin: "auto" }}>
        <div class="column is-two-thirds ">
          <div class="modal-card" style={{ width: "100%", margin: 0 }}>
            <header
              class="modal-card-head"
              style={{ border: "1px solid", borderBottom: 0 }}
            >
              <p class="modal-card-title">
                <i18n.Translate>Self provision</i18n.Translate>
              </p>
            </header>
            <section
              class="modal-card-body"
              style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
            >
              <FormProvider<Account>
                name="settings"
                errors={errors}
                object={value}
                valueHandler={valueHandler}
              >
                <InputWithAddon<Account>
                  name="id"
                  addonBefore={
                    new URL("instances/", session.backendUrl.href).href
                  }
                  label={i18n.str`Identifier`}
                  tooltip={i18n.str`Name of the instance in URLs. The 'admin' instance is special in that it is used to administer other instances.`}
                />

                <Input<Account>
                  name="name"
                  label={i18n.str`Business name`}
                  tooltip={i18n.str`Legal name of the business represented by this instance.`}
                />
                <Input<Account>
                  label={i18n.str`New password`}
                  tooltip={i18n.str`Next password to be used`}
                  inputType="password"
                  name="password"
                />
                <Input<Account>
                  label={i18n.str`Repeat password`}
                  tooltip={i18n.str`Confirm the same password`}
                  inputType="password"
                  name="repeat"
                />
                {serverRequiresEmail ? (
                  <Input<Account>
                    label={i18n.str`Email`}
                    tooltip={i18n.str`Contact email`}
                    name="email"
                  />
                ) : undefined}
                {serverRequiresSms ? (
                  <Input<Account>
                    label={i18n.str`Phone`}
                    tooltip={i18n.str`Contact phone number`}
                    name="phone"
                  />
                ) : undefined}
              </FormProvider>
            </section>
            <footer
              class="modal-card-foot "
              style={{
                justifyContent: "space-between",
                border: "1px solid",
                borderTop: 0,
              }}
            >
              <button class="button" onClick={onCancel}>
                <i18n.Translate>Cancel</i18n.Translate>
              </button>
              <AsyncButton
                type="is-info"
                disabled={!!errors}
                onClick={doCreate}
              >
                <i18n.Translate>Create</i18n.Translate>
              </AsyncButton>
            </footer>
          </div>
        </div>
      </div>
    </Fragment>
  );
}
